<?php

/**
 * vim:set softtabstop=4 shiftwidth=4 expandtab:
 *
 * LICENSE: GNU Affero General Public License, version 3 (AGPL-3.0-or-later)
 * Copyright Ampache.org, 2001-2024
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */

namespace Ampache\Module\Artist\Deletion;

use Ampache\Repository\LabelRepositoryInterface;
use Ampache\Module\Art\ArtCleanupInterface;
use Ampache\Repository\Model\Artist;
use Ampache\Repository\Model\ModelFactoryInterface;
use Ampache\Repository\Model\Rating;
use Ampache\Repository\Model\Userflag;
use Ampache\Module\Album\Deletion\AlbumDeleterInterface;
use Ampache\Module\Album\Deletion\Exception\AlbumDeletionException;
use Ampache\Module\Artist\Deletion\Exception\ArtistDeletionException;
use Ampache\Module\System\LegacyLogger;
use Ampache\Repository\AlbumRepositoryInterface;
use Ampache\Repository\ArtistRepositoryInterface;
use Ampache\Repository\ShoutRepositoryInterface;
use Ampache\Repository\UserActivityRepositoryInterface;
use Psr\Log\LoggerInterface;

final class ArtistDeleter implements ArtistDeleterInterface
{
    private AlbumDeleterInterface $albumDeleter;

    private ArtistRepositoryInterface $artistRepository;

    private AlbumRepositoryInterface $albumRepository;

    private ModelFactoryInterface $modelFactory;

    private LoggerInterface $logger;

    private ShoutRepositoryInterface $shoutRepository;

    private UserActivityRepositoryInterface $useractivityRepository;

    private LabelRepositoryInterface $labelRepository;
    private ArtCleanupInterface $artCleanup;

    public function __construct(
        AlbumDeleterInterface $albumDeleter,
        ArtistRepositoryInterface $artistRepository,
        AlbumRepositoryInterface $albumRepository,
        ModelFactoryInterface $modelFactory,
        LoggerInterface $logger,
        ShoutRepositoryInterface $shoutRepository,
        UserActivityRepositoryInterface $useractivityRepository,
        LabelRepositoryInterface $labelRepository,
        ArtCleanupInterface $artCleanup
    ) {
        $this->albumDeleter           = $albumDeleter;
        $this->artistRepository       = $artistRepository;
        $this->albumRepository        = $albumRepository;
        $this->modelFactory           = $modelFactory;
        $this->logger                 = $logger;
        $this->shoutRepository        = $shoutRepository;
        $this->useractivityRepository = $useractivityRepository;
        $this->labelRepository        = $labelRepository;
        $this->artCleanup             = $artCleanup;
    }

    /**
     * @throws ArtistDeletionException
     */
    public function remove(
        Artist $artist
    ): void {
        $album_ids = $this->albumRepository->getAlbumByArtist($artist->id);

        foreach ($album_ids as $albumId) {
            $album = $this->modelFactory->createAlbum($albumId);

            try {
                $this->albumDeleter->delete($album);
            } catch (AlbumDeletionException) {
                $this->logger->critical(
                    sprintf(
                        "Error when deleting the album `%d`.",
                        $albumId
                    ),
                    [LegacyLogger::CONTEXT_TYPE => self::class]
                );

                throw new ArtistDeletionException();
            }
        }

        $artistId = $artist->getId();

        $this->artistRepository->delete($artist);

        $this->artCleanup->collectGarbageForObject('artist', $artistId);
        Userflag::garbage_collection('artist', $artistId);
        Rating::garbage_collection('artist', $artistId);
        $this->labelRepository->collectGarbage();
        $this->shoutRepository->collectGarbage('artist', $artistId);
        $this->useractivityRepository->collectGarbage('artist', $artistId);
    }
}
