<?php

declare(strict_types=0);

/**
 * vim:set softtabstop=4 shiftwidth=4 expandtab:
 *
 * LICENSE: GNU Affero General Public License, version 3 (AGPL-3.0-or-later)
 * Copyright Ampache.org, 2001-2024
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */

namespace Ampache\Module\Application\Admin\Mail;

use Ampache\Config\ConfigContainerInterface;
use Ampache\Config\ConfigurationKeyEnum;
use Ampache\Module\Application\ApplicationActionInterface;
use Ampache\Module\Application\Exception\AccessDeniedException;
use Ampache\Module\Authorization\AccessLevelEnum;
use Ampache\Module\Authorization\AccessTypeEnum;
use Ampache\Module\Authorization\GuiGatekeeperInterface;
use Ampache\Module\System\Core;
use Ampache\Module\Util\Mailer;
use Ampache\Module\Util\RequestParserInterface;
use Ampache\Module\Util\UiInterface;
use Ampache\Repository\Model\User;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

final class SendMailAction implements ApplicationActionInterface
{
    public const REQUEST_KEY = 'send_mail';

    private RequestParserInterface $requestParser;

    private UiInterface $ui;

    private ConfigContainerInterface $configContainer;

    public function __construct(
        RequestParserInterface $requestParser,
        UiInterface $ui,
        ConfigContainerInterface $configContainer
    ) {
        $this->requestParser   = $requestParser;
        $this->ui              = $ui;
        $this->configContainer = $configContainer;
    }

    public function run(ServerRequestInterface $request, GuiGatekeeperInterface $gatekeeper): ?ResponseInterface
    {
        if (
            $gatekeeper->mayAccess(AccessTypeEnum::INTERFACE, AccessLevelEnum::MANAGER) === false ||
            $this->configContainer->isFeatureEnabled(ConfigurationKeyEnum::DEMO_MODE) === true
        ) {
            throw new AccessDeniedException();
        }

        $this->ui->showHeader();

        // Multi-byte Character Mail
        if (function_exists('mb_language')) {
            $ini_default_charset = 'default_charset';
            if (ini_get($ini_default_charset)) {
                ini_set($ini_default_charset, "UTF-8");
            }
            mb_language("uni");
        }

        if (Mailer::is_mail_enabled()) {
            $mailer = new Mailer();

            // Set the vars on the object
            $mailer->setSubject($this->requestParser->getFromRequest('subject'));
            $mailer->setMessage($this->requestParser->getFromRequest('message'));

            if ($this->requestParser->getFromRequest('from') == 'system') {
                $mailer->set_default_sender();
            } else {
                $user = Core::get_global('user');
                if ($user instanceof User) {
                    $mailer->setSender((string)$user->email, (string)$user->get_fullname());
                } else {
                    $mailer->set_default_sender();
                }
            }

            if ($mailer->send_to_group($this->requestParser->getFromRequest('to'))) {
                $title = T_('No Problem');
                $body  = T_('Your e-mail has been sent');
            } else {
                $title = T_('There Was a Problem');
                $body  = T_('Your e-mail has not been sent');
            }

            $url = sprintf(
                '%s/mail.php',
                $this->configContainer->getWebPath('/admin')
            );
            $this->ui->showConfirmation($title, $body, $url);
        }

        $this->ui->showQueryStats();
        $this->ui->showFooter();

        return null;
    }
}
