#  This file is part of Lazylibrarian.
#
# Purpose:
#    Contain all the config settings and defaults used across LazyLibrarian

from copy import deepcopy
from typing import List, Dict, Tuple

from lazylibrarian.configtypes import ConfigItem, ConfigStr, ConfigBool, ConfigInt, ConfigEmail, \
    ConfigPerm, ConfigCSV, ConfigURL, ConfigRangedInt, ConfigFloat, ConfigFolder, \
    ConfigScheduler, ConfigDownloadTypes, ConfigConnection, TimeUnit
from lazylibrarian.formatter import ImportPrefs
from lazylibrarian.logconfig import LogConfig


BASE_DEFAULTS: List[ConfigItem] = [
    ConfigURL('General', 'OL_URL', 'https://www.openlibrary.org'),
    ConfigURL('General', 'GR_URL', 'https://www.goodreads.com'),
    ConfigURL('General', 'GB_URL', 'https://www.googleapis.com'),
    ConfigURL('General', 'LT_URL', 'https://www.librarything.com'),
    ConfigURL('General', 'CV_URL', 'https://www.comicvine.gamespot.com'),
    ConfigURL('General', 'CX_URL', 'https://www.comixology.com'),
    ConfigBool('General', 'SHOW_NEWZ_PROV', 1),
    ConfigBool('General', 'SHOW_TORZ_PROV', 1),
    ConfigBool('General', 'SHOW_TOR_PROV', 1),
    ConfigBool('General', 'SHOW_RSS_PROV', 1),
    ConfigBool('General', 'SHOW_IRC_PROV', 1),
    ConfigBool('General', 'SHOW_GEN_PROV', False),  # Unused
    ConfigBool('General', 'SHOW_DIRECT_PROV', 1),
    ConfigBool('General', 'USER_ACCOUNTS', 0),
    ConfigBool('General', 'SINGLE_USER', 0),
    ConfigEmail('General', 'ADMIN_EMAIL', ''),
    ConfigCSV('General', 'WHITELIST', ''),
    ConfigStr('General', 'SYS_ENCODING', ''),
    ConfigStr('General', 'HOMEPAGE', ''),
    ConfigStr('General', 'AUTH_TYPE', "BASIC"),
    ConfigInt('General', 'WALL_COLUMNS', 6),
    ConfigPerm('General', 'FILE_PERM', '0o644'),
    ConfigPerm('General', 'DIR_PERM', '0o755'),
    ConfigInt('General', 'BLOCKLIST_TIMER', 3600),
    ConfigInt('General', 'MAX_PAGES', 0),
    ConfigInt('General', 'MAX_BOOKPAGES', 0),
    ConfigInt('General', 'MAX_WALL', 0),
    ConfigInt('General', 'MATCH_RATIO', 80),
    ConfigInt('General', 'DLOAD_RATIO', 90),
    ConfigInt('General', 'NAME_RATIO', 90),
    ConfigInt('General', 'NAME_PARTIAL', 95),
    ConfigInt('General', 'NAME_PARTNAME', 98),
    ConfigInt('General', 'DISPLAYLENGTH', 10),
    ConfigInt('General', 'HIST_REFRESH', 1000),
    ConfigBool('General', 'NO_IPV6', 0),
    ConfigStr('General', 'BOOKSTRAP_THEME', 'slate'),
    ConfigBool('General', 'SERIES_TAB', 1),
    ConfigBool('General', 'AUDIO_TAB', 1),
    ConfigBool('General', 'EBOOK_TAB', 1),
    ConfigRangedInt('General', 'CONFIG_TAB_NUM', 1, 1, 10),
    ConfigBool('General', 'TOGGLES', 1),
    ConfigBool('General', 'SORT_DEFINITE', 0),
    ConfigBool('General', 'SORT_SURNAME', 0),
    ConfigBool('General', 'SHOW_GENRES', 0),
    ConfigBool('General', 'IGNORE_PAUSED', 0),
    ConfigBool('General', 'LAUNCH_BROWSER', 1),
    ConfigCSV('General', 'NAME_POSTFIX', 'snr, jnr, jr, sr, phd', force_lower=True),
    ConfigCSV('General', 'NAME_DEFINITE', 'the, a', force_lower=True),

    ConfigBool('API', 'API_ENABLED', 0),
    ConfigStr('API', 'API_KEY', ''),
    ConfigStr('API', 'API_RO_KEY', ''),

    ConfigStr('Proxy', 'PROXY_HOST', ''),
    ConfigStr('Proxy', 'PROXY_TYPE', ''),
    ConfigStr('Proxy', 'PROXY_LOCAL', ''),
    ConfigBool('Proxy', 'PROXY_AUTH', 0),
    ConfigBool('Proxy', 'PROXY_REGISTER', 0),
    ConfigStr('Proxy', 'PROXY_AUTH_USER', 'X-WEBAUTH-USER'),
    ConfigStr('Proxy', 'PROXY_AUTH_EMAIL', 'X-WEBAUTH-EMAIL'),
    ConfigStr('Proxy', 'PROXY_AUTH_NAME', 'X-WEBAUTH-FULLNAME'),

    ConfigCSV('Postprocess', 'SKIPPED_EXT', 'fail, part, bts, !ut, torrent, magnet, nzb, unpack'),
    ConfigCSV('Postprocess', 'BANNED_EXT', 'avi, mp4, mov, iso, m4v', force_lower=True),
    ConfigCSV('General', 'IMP_PREFLANG', 'en, eng, en-US, en-GB, English', onchange=ImportPrefs.lang_changed),
    ConfigStr('General', 'ISS_FORMAT', '$Y-$m-$d'),
    ConfigStr('General', 'DATE_FORMAT', '$Y-$m-$d'),
    ConfigStr('General', 'AUTHOR_DATE_FORMAT', '$d-$m-$Y'),
    ConfigCSV('General', 'ISSUE_NOUNS', 'issue, iss, no, nr, #, n', force_lower=True),
    ConfigCSV('General', 'VOLUME_NOUNS', "vol, volume", force_lower=True),
    ConfigCSV('General', 'IMP_MONTHLANG', ''),
    ConfigStr('General', 'IMP_AUTOADD', ''),
    ConfigBool('General', 'IMP_AUTOADD_COPY', 1),
    ConfigBool('General', 'IMP_AUTOADD_BOOKONLY', 0),
    ConfigBool('General', 'IMP_AUTOSEARCH', 0),
    ConfigBool('General', 'BLACKLIST_FAILED', 1),
    ConfigBool('General', 'BLACKLIST_PROCESSED', 0),
    ConfigStr('General', 'SSL_CERTS', ''),
    ConfigBool('General', 'SSL_VERIFY', 1),
    ConfigRangedInt('General', 'HTTP_TIMEOUT', 30, 30, 100000),
    ConfigInt('General', 'HTTP_EXT_TIMEOUT', 90),

    ConfigRangedInt('WebServer', 'HTTP_PORT', 5299, 21, 65535),
    ConfigStr('WebServer', 'HTTP_HOST', '0.0.0.0'),
    ConfigStr('WebServer', 'HTTP_USER', ''),
    ConfigStr('WebServer', 'HTTP_PASS', ''),
    ConfigBool('WebServer', 'HTTP_PROXY', 0),
    ConfigStr('WebServer', 'HTTP_ROOT', ''),
    ConfigStr('WebServer', 'HTTP_LOOK', 'bookstrap'),
    ConfigBool('WebServer', 'HTTPS_ENABLED', 0),
    ConfigStr('WebServer', 'HTTPS_CERT', ''),
    ConfigStr('WebServer', 'HTTPS_KEY', ''),

    ConfigFolder('Logging', 'LOGDIR', ''),
    ConfigInt('Logging', 'LOGLIMIT', 500, onchange=LogConfig.change_memory_limit),
    ConfigInt('Logging', 'LOGFILES', 10),
    ConfigInt('Logging', 'LOGSIZE', 204800),
    ConfigBool('Logging', 'DETAILEDUILOG', 0),
    ConfigBool('Logging', 'LOGREDACT', 0, onchange=LogConfig.change_loguiredact),
    ConfigBool('Logging', 'HOSTREDACT', 0),
    ConfigBool('Logging', 'LOGFILEREDACT', 0, onchange=LogConfig.change_logfileredact),
    ConfigInt('Logging', 'LOGLEVEL', 20, onchange=LogConfig.change_root_loglevel),
    ConfigCSV('Logging', 'LOGSPECIALDEBUG', ''),

    ConfigBool('Importer', 'MULTI_SOURCE', 0),

    ConfigBool('Calibre', 'CALIBRE_USE_SERVER', 0),
    ConfigStr('Calibre', 'CALIBRE_SERVER', ''),
    ConfigStr('Calibre', 'CALIBRE_USER', ''),
    ConfigStr('Calibre', 'CALIBRE_PASS', ''),
    ConfigBool('Calibre', 'CALIBRE_RENAME', 0),
    ConfigStr('Calibre', 'IMP_CALIBREDB', ''),
    ConfigBool('Calibre', 'IMP_CALIBREOVERWRITE', 0),
    ConfigBool('Calibre', 'IMP_CALIBRE_EBOOK', 0),
    ConfigBool('Calibre', 'IMP_CALIBRE_COMIC', 0),
    ConfigBool('Calibre', 'IMP_CALIBRE_MAGAZINE', 0),
    ConfigBool('Calibre', 'IMP_CALIBRE_MAGTITLE', 1),
    ConfigBool('Calibre', 'IMP_CALIBRE_MAGISSUE', 0),

    ConfigBool('General', 'IMP_SINGLEBOOK', 0),
    ConfigBool('General', 'IMP_RENAME', 0),
    ConfigBool('General', 'MAG_RENAME', 0),
    ConfigBool('General', 'IMP_COMICOPF', 0),
    ConfigBool('General', 'IMP_COMICCOVER', 1),
    ConfigStr('General', 'IMP_CONVERT', ''),
    ConfigCSV('General', 'IMP_NOSPLIT', '', force_lower=True, onchange=ImportPrefs.nosplit_changed),
    ConfigStr('General', 'EXT_PREPROCESS', ''),
    ConfigStr('General', 'GIT_PROGRAM', ''),
    ConfigInt('General', 'CACHE_AGE', 30),
    ConfigInt('General', 'BACKUP_DB', 30),
    ConfigInt('General', 'TASK_AGE', 2),
    ConfigBool('General', 'OPF_TAGS', 1),
    ConfigBool('General', 'GENRE_TAGS', 0),
    ConfigBool('General', 'WISHLIST_TAGS', 1),
    ConfigBool('General', 'WISHLIST_GENRES', 1),
    ConfigBool('General', 'NOTIFY_WITH_TITLE', 0),
    ConfigBool('General', 'NOTIFY_WITH_URL', 0),
    ConfigStr('Telemetry', 'SERVER_ID', ''),
    ConfigURL('Telemetry', 'TELEMETRY_SERVER', ''),
    ConfigScheduler('Telemetry', 'TELEMETRY_INTERVAL', 'telemetry_send', 6, TimeUnit.HOUR,
                    'TELEMETRYSEND', 'lazylibrarian.telemetry.telemetry_send', 'Send telemetry', needs_provider=False,
                    persist=True),
    ConfigBool('Telemetry', 'TELEMETRY_ENABLE', False),
    ConfigBool('Telemetry', 'TELEMETRY_SEND_CONFIG', True),
    ConfigBool('Telemetry', 'TELEMETRY_SEND_USAGE', True),
    ConfigStr('Git', 'GIT_HOST', 'gitlab.com'),
    ConfigStr('Git', 'GIT_USER', 'LazyLibrarian'),
    ConfigStr('Git', 'GIT_REPO', 'lazylibrarian'),
    ConfigStr('Git', 'GIT_PROJECT', '9317860'),
    ConfigStr('Git', 'GIT_BRANCH', 'master'),
    ConfigInt('Git', 'GIT_UPDATED', 0),
    ConfigStr('Git', 'INSTALL_TYPE', ''),
    ConfigStr('Git', 'CURRENT_VERSION', ''),
    ConfigStr('Git', 'LATEST_VERSION', ''),
    ConfigInt('Git', 'COMMITS_BEHIND', 0),
    ConfigBool('Git', 'AUTO_UPDATE', False),
    ConfigStr('SABnzbd', 'SAB_HOST', ''),
    ConfigRangedInt('SABnzbd', 'SAB_PORT', 0, 0, 65535),
    ConfigStr('SABnzbd', 'SAB_SUBDIR', ''),
    ConfigStr('SABnzbd', 'SAB_USER', ''),
    ConfigStr('SABnzbd', 'SAB_PASS', ''),
    ConfigStr('SABnzbd', 'SAB_API', ''),
    ConfigStr('SABnzbd', 'SAB_CAT', ''),
    ConfigBool('SABnzbd', 'SAB_DELETE', 1),
    ConfigStr('SABnzbd', 'SAB_EXTERNAL_HOST', ''),
    ConfigStr('NZBGet', 'NZBGET_HOST', ''),
    ConfigRangedInt('NZBGet', 'NZBGET_PORT', 0, 0, 65535),
    ConfigStr('NZBGet', 'NZBGET_USER', ''),
    ConfigStr('NZBGet', 'NZBGET_PASS', ''),
    ConfigStr('NZBGet', 'NZBGET_CATEGORY', ''),
    ConfigInt('NZBGet', 'NZBGET_PRIORITY', 0),
    ConfigBool('General', 'DESTINATION_COPY', 0),
    ConfigStr('General', 'EBOOK_DIR', ''),
    ConfigStr('General', 'AUDIO_DIR', ''),
    ConfigStr('General', 'ALTERNATE_DIR', ''),
    ConfigStr('General', 'TESTDATA_DIR', ''),
    ConfigBool('General', 'DELETE_CSV', 0),
    ConfigStr('General', 'DOWNLOAD_DIR', ''),
    ConfigBool('USENET', 'NZB_DOWNLOADER_SABNZBD', 0),
    ConfigBool('USENET', 'NZB_DOWNLOADER_NZBGET', 0),
    ConfigBool('USENET', 'NZB_DOWNLOADER_SYNOLOGY', 0),
    ConfigBool('USENET', 'NZB_DOWNLOADER_BLACKHOLE', 0),
    ConfigStr('USENET', 'NZB_BLACKHOLEDIR', ''),
    ConfigBool('USENET', 'NZB_PAUSED', 0),
    ConfigInt('USENET', 'USENET_RETENTION', 0),
    ConfigStr('NZBMatrix', 'NZBMATRIX_USER', ''),
    ConfigStr('NZBMatrix', 'NZBMATRIX_API', ''),
    ConfigBool('NZBMatrix', 'NZBMATRIX', 0),  # Unused
    ConfigBool('TORRENT', 'TOR_DOWNLOADER_BLACKHOLE', 0),
    ConfigBool('TORRENT', 'TOR_CONVERT_MAGNET', 0),
    ConfigBool('TORRENT', 'TOR_DOWNLOADER_UTORRENT', 0),
    ConfigBool('TORRENT', 'TOR_DOWNLOADER_RTORRENT', 0),
    ConfigBool('TORRENT', 'TOR_DOWNLOADER_QBITTORRENT', 0),
    ConfigBool('TORRENT', 'TOR_DOWNLOADER_TRANSMISSION', 0),
    ConfigBool('TORRENT', 'TOR_DOWNLOADER_SYNOLOGY', 0),
    ConfigBool('TORRENT', 'TOR_DOWNLOADER_DELUGE', 0),
    ConfigBool('TORRENT', 'TORRENT_PAUSED', 0),
    ConfigInt('TORRENT', 'NUMBEROFSEEDERS', 10),
    ConfigBool('TORRENT', 'KEEP_SEEDING', 1),
    ConfigBool('TORRENT', 'SEED_WAIT', 1),
    ConfigBool('TORRENT', 'PREFER_MAGNET', 1),
    ConfigStr('TORRENT', 'TORRENT_DIR', ''),
    ConfigStr('RTORRENT', 'RTORRENT_HOST', ''),
    ConfigStr('RTORRENT', 'RTORRENT_USER', ''),
    ConfigStr('RTORRENT', 'RTORRENT_PASS', ''),
    ConfigStr('RTORRENT', 'RTORRENT_LABEL', ''),
    ConfigStr('RTORRENT', 'RTORRENT_DIR', ''),
    ConfigStr('UTORRENT', 'UTORRENT_HOST', ''),
    ConfigRangedInt('UTORRENT', 'UTORRENT_PORT', 0, 0, 65535),
    ConfigStr('UTORRENT', 'UTORRENT_BASE', ''),
    ConfigStr('UTORRENT', 'UTORRENT_USER', ''),
    ConfigStr('UTORRENT', 'UTORRENT_PASS', ''),
    ConfigStr('UTORRENT', 'UTORRENT_LABEL', ''),
    ConfigStr('QBITTORRENT', 'QBITTORRENT_HOST', ''),
    ConfigRangedInt('QBITTORRENT', 'QBITTORRENT_PORT', 0, 0, 65535),
    ConfigStr('QBITTORRENT', 'QBITTORRENT_BASE', ''),
    ConfigStr('QBITTORRENT', 'QBITTORRENT_USER', ''),
    ConfigStr('QBITTORRENT', 'QBITTORRENT_PASS', ''),
    ConfigStr('QBITTORRENT', 'QBITTORRENT_LABEL', ''),
    ConfigStr('QBITTORRENT', 'QBITTORRENT_DIR', ''),
    ConfigStr('TRANSMISSION', 'TRANSMISSION_HOST', ''),
    ConfigStr('TRANSMISSION', 'TRANSMISSION_BASE', ''),
    ConfigRangedInt('TRANSMISSION', 'TRANSMISSION_PORT', 0, 0, 65535),
    ConfigStr('TRANSMISSION', 'TRANSMISSION_USER', ''),
    ConfigStr('TRANSMISSION', 'TRANSMISSION_PASS', ''),
    ConfigStr('TRANSMISSION', 'TRANSMISSION_DIR', ''),
    ConfigStr('TRANSMISSION', 'TRANSMISSION_LABEL', ''),
    ConfigStr('DELUGE', 'DELUGE_CERT', ''),
    ConfigStr('DELUGE', 'DELUGE_HOST', ''),
    ConfigStr('DELUGE', 'DELUGE_BASE', ''),
    ConfigRangedInt('DELUGE', 'DELUGE_PORT', 0, 0, 65535),
    ConfigStr('DELUGE', 'DELUGE_USER', ''),
    ConfigStr('DELUGE', 'DELUGE_PASS', ''),
    ConfigStr('DELUGE', 'DELUGE_LABEL', ''),
    ConfigStr('DELUGE', 'DELUGE_DIR', ''),
    ConfigInt('DELUGE', 'DELUGE_TIMEOUT', 3600),
    ConfigStr('SYNOLOGY', 'SYNOLOGY_HOST', ''),
    ConfigRangedInt('SYNOLOGY', 'SYNOLOGY_PORT', 0, 0, 65535),
    ConfigStr('SYNOLOGY', 'SYNOLOGY_USER', ''),
    ConfigStr('SYNOLOGY', 'SYNOLOGY_PASS', ''),
    ConfigStr('SYNOLOGY', 'SYNOLOGY_DIR', 'Multimedia/Download'),
    ConfigBool('SYNOLOGY', 'USE_SYNOLOGY', 0),
    ConfigStr('KAT', 'KAT_HOST', 'kickass.cd'),
    ConfigBool('KAT', 'KAT', 0),
    ConfigInt('KAT', 'KAT_DLPRIORITY', 0),
    ConfigDownloadTypes('KAT', 'KAT_DLTYPES', 'A,E,M'),
    ConfigInt('KAT', 'KAT_SEEDERS', 0),
    ConfigStr('TPB', 'TPB_HOST', 'https://pirateproxy.cc'),
    ConfigBool('TPB', 'TPB', 0),
    ConfigInt('TPB', 'TPB_DLPRIORITY', 0),
    ConfigDownloadTypes('TPB', 'TPB_DLTYPES', 'A,C,E,M'),
    ConfigInt('TPB', 'TPB_SEEDERS', 0),
    ConfigStr('TDL', 'TDL_HOST', 'torrentdownloads.me'),
    ConfigBool('TDL', 'TDL', 0),
    ConfigInt('TDL', 'TDL_DLPRIORITY', 0),
    ConfigDownloadTypes('TDL', 'TDL_DLTYPES', 'A,E,M'),
    ConfigInt('TDL', 'TDL_SEEDERS', 0),
    ConfigStr('BOK', 'BOK_HOST', 'b-ok.cc'),
    ConfigStr('BOK', 'BOK_PASS', ''),
    ConfigBool('BOK', 'BOK', 0),
    ConfigInt('BOK', 'BOK_DLPRIORITY', 0),
    ConfigInt('BOK', 'BOK_DLLIMIT', 5),
    ConfigDownloadTypes('BOK', 'BOK_DLTYPES', 'E'),
    ConfigStr('BFI', 'BFI_HOST', 'en.bookfi.net'),
    ConfigBool('BFI', 'BFI', 0),
    ConfigInt('BFI', 'BFI_DLPRIORITY', 0),
    ConfigDownloadTypes('BFI', 'BFI_DLTYPES', 'E'),
    ConfigStr('LIME', 'LIME_HOST', 'https://www.limetorrents.cc'),
    ConfigBool('LIME', 'LIME', 0),
    ConfigInt('LIME', 'LIME_DLPRIORITY', 0),
    ConfigDownloadTypes('LIME', 'LIME_DLTYPES', 'A,E,M'),
    ConfigInt('LIME', 'LIME_SEEDERS', 0),
    ConfigStr('Newzbin', 'NEWZBIN_UID', ''),
    ConfigStr('Newzbin', 'NEWZBIN_PASS', ''),
    ConfigBool('Newzbin', 'NEWZBIN', 0),  # Unused
    ConfigCSV('General', 'EBOOK_TYPE', 'epub, mobi, pdf', force_lower=True),
    ConfigCSV('General', 'AUDIOBOOK_TYPE', 'mp3, m4b', force_lower=True),
    ConfigCSV('General', 'MAG_TYPE', 'pdf', force_lower=True),
    ConfigCSV('General', 'REJECT_PUBLISHER', '', force_lower=True),
    ConfigCSV('General', 'REJECT_WORDS', 'audiobook, mp3', force_lower=True),
    ConfigCSV('General', 'PREFER_WORDS', 'retail', force_lower=True),
    ConfigCSV('General', 'REJECT_AUDIO', 'epub, mobi', force_lower=True),
    ConfigInt('General', 'MAG_AGE', 31),
    ConfigScheduler('SearchScan', 'SEARCH_BOOKINTERVAL', 'search_book', 360, TimeUnit.MIN,
                    'SEARCHALLBOOKS', 'lazylibrarian.searchbook.cron_search_book', 'Book search', True),
    ConfigScheduler('SearchScan', 'SEARCH_MAGINTERVAL', 'search_magazines', 360, TimeUnit.MIN,
                    'SEARCHALLMAG', 'lazylibrarian.searchmag.cron_search_magazines', 'Magazines search', True),
    ConfigScheduler('SearchScan', 'SCAN_INTERVAL', 'PostProcessor', 10, TimeUnit.MIN,
                    'POSTPROCESSOR', 'lazylibrarian.postprocess.PostProcessor', 'PostProcessor', False),
    ConfigScheduler('SearchScan', 'SEARCHRSS_INTERVAL', 'search_rss_book', 20, TimeUnit.MIN,
                    'SEARCHALLRSS', 'lazylibrarian.searchrss.cron_search_rss_book', 'RSS book search', False),
    ConfigScheduler('SearchScan', 'WISHLIST_INTERVAL', 'search_wishlist', 2, TimeUnit.HOUR,
                    'SEARCHWISHLIST', 'lazylibrarian.searchrss.cron_search_wishlist', 'Wishlist search', False),
    ConfigScheduler('SearchScan', 'SEARCH_COMICINTERVAL', 'search_comics', 24, TimeUnit.HOUR,
                    'SEARCHALLCOMICS', 'lazylibrarian.comicsearch.cron_search_comics', 'Comic search', True),
    ConfigScheduler('SearchScan', 'VERSIONCHECK_INTERVAL', 'check_for_updates', 24, TimeUnit.HOUR,
                    'VERSIONCHECK', 'lazylibrarian.versioncheck.check_for_updates', 'Check for updates', False),
    ConfigScheduler('SearchScan', 'GOODREADS_INTERVAL', 'sync_to_gr', 48, TimeUnit.HOUR,
                    'GRSYNC', 'lazylibrarian.grsync.cron_sync_to_gr', 'GoodReads sync', False),
    ConfigScheduler('SearchScan', 'HARDCOVER_INTERVAL', 'hc_sync', 48, TimeUnit.HOUR,
                    'HCSYNC', 'lazylibrarian.hc.hc_sync', 'HardCover sync', False),
    ConfigScheduler('SearchScan', 'CLEAN_CACHE_INTERVAL', 'clean_cache', 30, TimeUnit.DAY,
                    'CLEANCACHE', 'lazylibrarian.cache.clean_cache', 'Clean cache', False, persist=False),
    ConfigScheduler('SearchScan', 'BACKUP_INTERVAL', 'backup', 30, TimeUnit.DAY,
                    'BACKUP', 'lazylibrarian.common.cron_dbbackup', 'Backup', False, persist=False),
    # Interval is not used
    ConfigScheduler('SearchScan', 'AUTHORUPDATE_INTERVAL', 'author_update', 1, TimeUnit.MIN,
                    'AUTHORUPDATE', 'lazylibrarian.scheduling.author_update', 'Update authors', False, persist=False),
    # Interval is not used
    ConfigScheduler('SearchScan', 'SERIESUPDATE_INTERVAL', 'series_update', 1, TimeUnit.MIN,
                    'SERIESUPDATE', 'lazylibrarian.scheduling.series_update', 'Update series', False, persist=False),
    # Interval is not used
    ConfigBool('SearchScan', 'DELAYSEARCH', 0),
    ConfigInt('SearchScan', 'SEARCH_RATELIMIT', 0),
    ConfigBool('LibraryScan', 'FULL_SCAN', 0),
    ConfigBool('LibraryScan', 'ADD_AUTHOR', 1),
    ConfigBool('LibraryScan', 'ADD_SERIES', 1),
    ConfigStr('LibraryScan', 'NOTFOUND_STATUS', 'Skipped'),
    ConfigStr('LibraryScan', 'FOUND_STATUS', 'Open'),
    ConfigBool('LibraryScan', 'NO_SINGLE_BOOK_SERIES', 0),
    ConfigBool('LibraryScan', 'NO_NONINTEGER_SERIES', 0),
    ConfigStr('LibraryScan', 'NEWSERIES_STATUS', 'Paused'),
    ConfigStr('LibraryScan', 'NEWBOOK_STATUS', 'Skipped'),
    ConfigStr('LibraryScan', 'NEWAUDIO_STATUS', 'Skipped'),
    ConfigStr('LibraryScan', 'NEWAUTHOR_STATUS', 'Skipped'),
    ConfigStr('LibraryScan', 'NEWAUTHOR_AUDIO', 'Skipped'),
    ConfigBool('LibraryScan', 'NEWAUTHOR_BOOKS', 0),
    ConfigBool('LibraryScan', 'NO_FUTURE', 0),
    ConfigBool('LibraryScan', 'NO_PUBDATE', 0),
    ConfigBool('LibraryScan', 'NO_ISBN', 0),
    ConfigBool('LibraryScan', 'NO_SETS', 0),
    ConfigBool('LibraryScan', 'NO_LANG', 0),
    ConfigBool('LibraryScan', 'ISBN_LOOKUP', 1),
    ConfigBool('LibraryScan', 'IMP_IGNORE', 0),
    
    ConfigStr('PostProcess', 'CREATE_LINK', ''),
    ConfigFolder('PostProcess', 'EBOOK_DEST_FOLDER', '$Author/$Title'),
    ConfigStr('PostProcess', 'EBOOK_DEST_FILE', '$Title - $Author'),
    ConfigStr('PostProcess', 'AUDIOBOOK_DEST_FILE', '$Author - $Title Part $Part of $Total'),
    ConfigStr('PostProcess', 'AUDIOBOOK_SINGLE_FILE', ''),
    ConfigFolder('PostProcess', 'AUDIOBOOK_DEST_FOLDER', 'None'),
    ConfigBool('PostProcess', 'ONE_FORMAT', 0),
    ConfigBool('Postprocess', 'DEL_DOWNLOADFAILED', 0),
    ConfigInt('PostProcess', 'PP_DELAY', 0),
    ConfigBool('PostProcess', 'DEL_FAILED', 1),
    ConfigBool('PostProcess', 'DEL_COMPLETED', 1),

    ConfigFolder('Comics', 'COMIC_DEST_FOLDER', '_Comics/$Title/$Issue'),
    ConfigBool('Comics', 'COMIC_TAB', 0),
    ConfigBool('Comics', 'COMIC_RELATIVE', 1),
    ConfigBool('Comics', 'COMIC_DELFOLDER', 1),
    ConfigCSV('Comics', 'COMIC_TYPE', 'cbr, cbz', force_lower=True),
    ConfigBool('Comics', 'COMIC_SINGLE', 1),
    ConfigCSV('Comics', 'REJECT_COMIC', 'epub, mobi', force_lower=True),
    ConfigInt('Comics', 'REJECT_MAXCOMIC', 0),
    ConfigInt('Comics', 'REJECT_MINCOMIC', 0),
    ConfigStr('Comics', 'CV_APIKEY', ''),  # ComicVine API key

    ConfigBool('Magazines', 'MAG_TAB', 1),
    ConfigStr('Magazines', 'MAG_COVERSWAP', ''),
    ConfigFolder('Magazines', 'MAG_DEST_FOLDER', '_Magazines/$Title/$IssueDate'),
    ConfigStr('Magazines', 'MAG_DEST_FILE', '$IssueDate - $Title'),
    ConfigBool('Magazines', 'MAG_RELATIVE', 1),
    ConfigBool('Magazines', 'MAG_DELFOLDER', 1),
    ConfigCSV('Magazines', 'REJECT_MAGS', '', force_lower=True),
    ConfigInt('Magazines', 'REJECT_MAXSIZE', 0),
    ConfigInt('Magazines', 'REJECT_MINSIZE', 0),
    ConfigInt('Magazines', 'REJECT_MAXAUDIO', 0),
    ConfigInt('Magazines', 'REJECT_MINAUDIO', 0),
    ConfigInt('Magazines', 'REJECT_MAGSIZE', 0),
    ConfigInt('Magazines', 'REJECT_MAGMIN', 0),
    ConfigBool('Magazines', 'IMP_MAGOPF', 1),
    ConfigBool('Magazines', 'IMP_MAGCOVER', 1),
    ConfigBool('Magazines', 'MAG_SINGLE', 1),
    ConfigCSV('Magazines', 'MAG_NOUNS', "winter, spring, summer, fall, autumn, christmas, edition, special",
              force_lower=True),
    ConfigStr('Magazines', 'IMP_AUTOADDMAG', ''),
    ConfigBool('Magazines', 'IMP_AUTOADDMAG_COPY', 1),
    ConfigBool('Magazines', 'IMP_AUTOADD_MAGONLY', 0),

    ConfigBool('General', 'HIDE_OLD_NOTIFIERS', 0),
    ConfigBool('Twitter', 'USE_TWITTER', 0),
    ConfigBool('Twitter', 'TWITTER_NOTIFY_ONSNATCH', 0),
    ConfigBool('Twitter', 'TWITTER_NOTIFY_ONDOWNLOAD', 0),
    ConfigStr('Twitter', 'TWITTER_USERNAME', ''),
    ConfigStr('Twitter', 'TWITTER_PASSWORD', ''),
    ConfigStr('Twitter', 'TWITTER_PREFIX', 'LazyLibrarian'),
    ConfigBool('Boxcar', 'USE_BOXCAR', 0),
    ConfigBool('Boxcar', 'BOXCAR_NOTIFY_ONSNATCH', 0),
    ConfigBool('Boxcar', 'BOXCAR_NOTIFY_ONDOWNLOAD', 0),
    ConfigStr('Boxcar', 'BOXCAR_TOKEN', ''),
    ConfigBool('Pushbullet', 'USE_PUSHBULLET', 0),
    ConfigBool('Pushbullet', 'PUSHBULLET_NOTIFY_ONSNATCH', 0),
    ConfigBool('Pushbullet', 'PUSHBULLET_NOTIFY_ONDOWNLOAD', 0),
    ConfigStr('Pushbullet', 'PUSHBULLET_TOKEN', ''),
    ConfigStr('Pushbullet', 'PUSHBULLET_DEVICEID', ''),
    ConfigBool('Pushover', 'USE_PUSHOVER', 0),
    ConfigBool('Pushover', 'PUSHOVER_ONSNATCH', 0),
    ConfigBool('Pushover', 'PUSHOVER_ONDOWNLOAD', 0),
    ConfigStr('Pushover', 'PUSHOVER_KEYS', ''),
    ConfigStr('Pushover', 'PUSHOVER_APITOKEN', ''),
    ConfigInt('Pushover', 'PUSHOVER_PRIORITY', 0),
    ConfigStr('Pushover', 'PUSHOVER_DEVICE', ''),
    ConfigBool('AndroidPN', 'USE_ANDROIDPN', 0),
    ConfigBool('AndroidPN', 'ANDROIDPN_NOTIFY_ONSNATCH', 0),
    ConfigBool('AndroidPN', 'ANDROIDPN_NOTIFY_ONDOWNLOAD', 0),
    ConfigURL('AndroidPN', 'ANDROIDPN_URL', ''),
    ConfigStr('AndroidPN', 'ANDROIDPN_USERNAME', ''),
    ConfigBool('AndroidPN', 'ANDROIDPN_BROADCAST', 0),
    ConfigBool('Telegram', 'USE_TELEGRAM', 0),
    ConfigStr('Telegram', 'TELEGRAM_TOKEN', ''),
    ConfigStr('Telegram', 'TELEGRAM_USERID', ''),
    ConfigBool('Telegram', 'TELEGRAM_ONSNATCH', 0),
    ConfigBool('Telegram', 'TELEGRAM_ONDOWNLOAD', 0),
    ConfigBool('Prowl', 'USE_PROWL', 0),
    ConfigStr('Prowl', 'PROWL_APIKEY', ''),
    ConfigInt('Prowl', 'PROWL_PRIORITY', 0),
    ConfigBool('Prowl', 'PROWL_ONSNATCH', 0),
    ConfigBool('Prowl', 'PROWL_ONDOWNLOAD', 0),
    ConfigBool('Growl', 'USE_GROWL', 0),
    ConfigStr('Growl', 'GROWL_HOST', ''),
    ConfigStr('Growl', 'GROWL_PASSWORD', ''),
    ConfigBool('Growl', 'GROWL_ONSNATCH', 0),
    ConfigBool('Growl', 'GROWL_ONDOWNLOAD', 0),
    ConfigBool('Slack', 'USE_SLACK', 0),
    ConfigBool('Slack', 'SLACK_NOTIFY_ONSNATCH', 0),
    ConfigBool('Slack', 'SLACK_NOTIFY_ONDOWNLOAD', 0),
    ConfigStr('Slack', 'SLACK_TOKEN', ''),
    ConfigURL('Slack', 'SLACK_URL', "https://hooks.slack.com/services"),
    ConfigBool('Custom', 'USE_CUSTOM', 0),
    ConfigBool('Custom', 'CUSTOM_NOTIFY_ONSNATCH', 0),
    ConfigBool('Custom', 'CUSTOM_NOTIFY_ONDOWNLOAD', 0),
    ConfigStr('Custom', 'CUSTOM_SCRIPT', ''),
    ConfigBool('Email', 'USE_EMAIL', 0),
    ConfigBool('Email', 'EMAIL_NOTIFY_ONSNATCH', 0),
    ConfigBool('Email', 'EMAIL_NOTIFY_ONDOWNLOAD', 0),
    ConfigBool('Email', 'EMAIL_SENDFILE_ONDOWNLOAD', 0),
    ConfigEmail('Email', 'EMAIL_FROM', ''),
    ConfigEmail('Email', 'EMAIL_TO', ''),
    ConfigBool('Email', 'EMAIL_SSL', 0),
    ConfigStr('Email', 'EMAIL_SMTP_SERVER', ''),
    ConfigRangedInt('Email', 'EMAIL_SMTP_PORT', 25, 21, 65535),
    ConfigBool('Email', 'EMAIL_TLS', 0),
    ConfigStr('Email', 'EMAIL_SMTP_USER', ''),
    ConfigStr('Email', 'EMAIL_SMTP_PASSWORD', ''),
    ConfigInt('Email', 'EMAIL_LIMIT', 20),
    ConfigBool('Email', 'USE_EMAIL_CUSTOM_FORMAT', 0),  # Unused
    ConfigCSV('Email', 'EMAIL_CONVERT_FROM', '', force_lower=True),
    ConfigCSV('Email', 'EMAIL_SEND_TYPE', '', force_lower=True),
    ConfigStr('API', 'BOOK_API', 'OpenLibrary'),
    ConfigStr('API', 'LT_DEVKEY', ''),
    ConfigBool('API', 'CV_WEBSEARCH', 0),
    ConfigBool('API', 'OL_API', 1),
    ConfigStr('API', 'HC_API', ''),
    ConfigBool('API', 'HC_SYNC', 0),
    ConfigStr('API', 'GR_API', 'ckvsiSDsuqh7omh74ZZ6Q'),
    ConfigBool('API', 'GR_SYNC', 0),
    ConfigBool('API', 'GR_SYNCUSER', 0),
    ConfigStr('API', 'GR_USER', ''),
    ConfigBool('API', 'GR_SYNCREADONLY', 0),
    ConfigStr('API', 'GR_SECRET', ''),  # tied to users own api key
    ConfigStr('API', 'GR_OAUTH_TOKEN', ''),  # gives access to users bookshelves
    ConfigStr('API', 'GR_OAUTH_SECRET', ''),  # gives access to users bookshelves
    ConfigStr('API', 'GR_WANTED', ''),  # sync wanted to this shelf
    ConfigStr('API', 'GR_OWNED', ''),  # sync open/have to this shelf
    ConfigStr('API', 'GR_AWANTED', ''),  # sync wanted to this shelf
    ConfigStr('API', 'GR_AOWNED', ''),  # sync open/have to this shelf
    ConfigBool('API', 'GR_UNIQUE', 0),  # delete from wanted if already owned
    ConfigBool('API', 'GR_FOLLOW', 0),  # follow authors on goodreads
    ConfigBool('API', 'GR_FOLLOWNEW', 0),  # follow new authors on goodreads
    ConfigStr('API', 'GB_API', ''),  # API key has daily limits, each user needs their own
    ConfigStr('API', 'GB_COUNTRY', ''),  # optional two-letter country code for geographically restricted results
    ConfigStr('FMT', 'FMT_SERNAME', '$SerName'),
    ConfigStr('FMT', 'FMT_SERNUM', 'Book #$SerNum -$$'),
    ConfigStr('FMT', 'FMT_SERIES', '( $FmtName $FmtNum )'),
    ConfigBool('OPDS', 'OPDS_ENABLED', 0),
    ConfigBool('OPDS', 'OPDS_AUTHENTICATION', 0),
    ConfigStr('OPDS', 'OPDS_USERNAME', ''),
    ConfigStr('OPDS', 'OPDS_PASSWORD', ''),
    ConfigBool('OPDS', 'OPDS_METAINFO', 0),
    ConfigInt('OPDS', 'OPDS_PAGE', 30),
    ConfigBool('rss', 'RSS_ENABLED', 1),
    ConfigBool('rss', 'RSS_PODCAST', 1),
    ConfigStr('rss', 'RSS_HOST', ''),
    ConfigInt('General', 'PREF_UNRARLIB', 1),
    ConfigStr('General', 'USER_AGENT', ''),
    ConfigBool('General', 'RATESTARS', 1),

    ConfigStr('Preprocess', 'EBOOK_WANTED_FORMATS', ''),
    ConfigBool('Preprocess', 'DELETE_OTHER_FORMATS', 0),
    ConfigStr('Preprocess', 'EBOOK_CONVERT', 'ebook-convert'),
    ConfigBool('Preprocess', 'KEEP_OPF', 1),
    ConfigBool('Preprocess', 'KEEP_JPG', 1),
    ConfigStr('Preprocess', 'FFMPEG', 'ffmpeg'),
    ConfigStr('Preprocess', 'FFMPEG_OUT', ''),
    ConfigStr('Preprocess', 'AUDIO_OPTIONS', '-vn -b:a 128k -f mp3'),
    ConfigBool('Preprocess', 'CREATE_SINGLEAUDIO', 0),
    ConfigBool('Preprocess', 'KEEP_SEPARATEAUDIO', 0),
    ConfigBool('Preprocess', 'WRITE_AUDIOTAGS', 0),
    ConfigBool('Preprocess', 'ZIP_AUDIOPARTS', 0),
    ConfigBool('Preprocess', 'SWAP_COVERPAGE', 0),
    ConfigInt('Preprocess', 'SHRINK_MAG', 0),
    ConfigInt('Testing', 'Unpersisted_test', 2, persist=False)
]

# (Sectionname, SectionNameTemplate, [Items])
DefaultArrayDef = Tuple[str, str, List[ConfigItem]]

ARRAY_DEFS: Dict[str, DefaultArrayDef] = {
    'NEWZNAB': ('HOST', 'Newznab_%i', [
        ConfigStr('', 'NAME', 'Newznab', persist=False),
        ConfigStr('', "DISPNAME", ''),
        ConfigBool('', "ENABLED", False),
        ConfigStr('', "HOST", ''),
        ConfigStr('', "API", ''),
        ConfigStr('', "GENERALSEARCH", ''),
        ConfigStr('', "BOOKSEARCH", ''),
        ConfigStr('', "MAGSEARCH", ''),
        ConfigStr('', "AUDIOSEARCH", ''),
        ConfigStr('', "COMICSEARCH", ''),
        ConfigCSV('', "BOOKCAT", '7000,7020'),
        ConfigCSV('', "MAGCAT", '7010'),
        ConfigCSV('', "AUDIOCAT", '3030'),
        ConfigCSV('', "COMICCAT", '7030'),
        ConfigInt('', "EXTENDED", 1),
        ConfigStr('', "UPDATED", ''),
        ConfigBool('', "MANUAL", False),
        ConfigInt('', "APILIMIT", 0),
        ConfigInt('', "APICOUNT", 0, persist=False),
        ConfigInt('', "RATELIMIT", 0),
        ConfigInt('', "DLPRIORITY", 0),
        ConfigDownloadTypes('', "DLTYPES", 'A,E,M'),
        ConfigInt('', "LASTUSED", 0),
    ]),
    'TORZNAB': ('HOST', 'Torznab_%i', [
        ConfigStr('', 'NAME', 'Torznab', persist=False),
        ConfigStr('', "DISPNAME", ''),
        ConfigBool('', "ENABLED", False),
        ConfigStr('', "HOST", ''),
        ConfigStr('', "API", ''),
        ConfigStr('', "GENERALSEARCH", ''),
        ConfigStr('', "BOOKSEARCH", ''),
        ConfigStr('', "MAGSEARCH", ''),
        ConfigStr('', "AUDIOSEARCH", ''),
        ConfigStr('', "COMICSEARCH", ''),
        ConfigCSV('', "BOOKCAT", '8000,8010'),
        ConfigCSV('', "MAGCAT", '8030'),
        ConfigCSV('', "AUDIOCAT", '3030'),
        ConfigCSV('', "COMICCAT", '8020'),
        ConfigInt('', "EXTENDED", 1),
        ConfigStr('', "UPDATED", ''),
        ConfigBool('', "MANUAL", False),
        ConfigInt('', "APILIMIT", 0),
        ConfigInt('', "APICOUNT", 0, persist=False),
        ConfigInt('', "RATELIMIT", 0),
        ConfigInt('', "DLPRIORITY", 0),
        ConfigDownloadTypes('', "DLTYPES", 'A,E,M'),
        ConfigInt('', "SEEDERS", 0),
        ConfigFloat('', "SEED_RATIO", 0),
        ConfigInt('', "SEED_DURATION", 0),
        ConfigInt('', "LASTUSED", 0),
    ]),
    'RSS': ('HOST', 'RSS_%i', [
        ConfigStr('', 'NAME', 'RSS', persist=False),
        ConfigStr('', "DISPNAME", ''),
        ConfigBool('', "ENABLED", False),
        ConfigStr('', "HOST", ''),
        ConfigInt('', "DLPRIORITY", 0),
        ConfigDownloadTypes('', "DLTYPES", 'E'),
        ConfigStr('', "LABEL", ''),
    ]),
    'IRC': ('SERVER', 'IRC_%i', [
        ConfigStr('', 'NAME', 'IRC', persist=False),
        ConfigStr('', "DISPNAME", ''),
        ConfigBool('', "ENABLED", False),
        ConfigStr('', "SERVER", ''),
        ConfigStr('', "CHANNEL", ''),
        ConfigStr('', "BOTNICK", ''),
        ConfigStr('', "SEARCH", '@search'),
        ConfigInt('', "DLPRIORITY", 0),
        ConfigDownloadTypes('', "DLTYPES", 'E'),
        ConfigConnection('', 'CONNECTION')
    ]),
    'GEN': ('HOST', 'GEN_%i', [
        ConfigStr('', 'NAME', 'GEN', persist=False),
        ConfigStr('', "DISPNAME", ''),
        ConfigBool('', "ENABLED", False),
        ConfigStr('', "HOST", ''),
        ConfigStr('', "SEARCH", ''),
        ConfigInt('', "DLPRIORITY", 0),
        ConfigDownloadTypes('', "DLTYPES", 'E'),
    ]),
    'APPRISE': ('URL', 'APPRISE_%i', [
        ConfigStr('', 'NAME', 'APPRISE', persist=False),
        ConfigStr('', 'DISPNAME', 'Apprise'),
        ConfigBool('', 'SNATCH', False),
        ConfigBool('', 'DOWNLOAD', False),
        ConfigStr('', 'URL', ''),
    ]),
}

CONFIG_GIT = ['GIT_REPO', 'GIT_USER', 'GIT_BRANCH', 'LATEST_VERSION', 'GIT_UPDATED', 'CURRENT_VERSION',
              'GIT_HOST', 'COMMITS_BEHIND', 'INSTALL_TYPE', 'AUTO_UPDATE']


def get_default(setting: str):
    """ Look up setting in base defaults, return the default value if it exists, or None """
    for default in BASE_DEFAULTS:
        if default.is_key(setting):
            return default.get_default()
    return None


def configitem_from_default(item: ConfigItem) -> ConfigItem:
    """ Create a ConfigItem from a DEFAULT item """
    rc = deepcopy(item)  # Deepcopy to get all attributes copied and retain the class
    rc.section = rc.section.upper()  # To have consistency across implementation
    rc.key = rc.key.upper()
    return rc
