/**
 * This file is part of the "libunicode" project
 *   Copyright (c) 2020-2021 Christian Parpart <christian@parpart.family>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <libunicode/ucd.h>
#include <libunicode/ucd_private.h>

#include <array>

namespace unicode
{

namespace tables
{
    // clang-format off
    auto static const Plane = std::array<Prop<::unicode::Plane>, 17>{ // {{{
        Prop<::unicode::Plane> { { 0x0000, 0xFFFF }, unicode::Plane::Basic_Multilingual_Plane }, // Plane 0 BMP
        Prop<::unicode::Plane> { { 0x10000, 0x1FFFF }, unicode::Plane::Supplementary_Multilingual_Plane }, // Plane 1 SMP
        Prop<::unicode::Plane> { { 0x20000, 0x2FFFF }, unicode::Plane::Supplementary_Ideographic_Plane }, // Plane 2 SIP
        Prop<::unicode::Plane> { { 0x30000, 0x3FFFF }, unicode::Plane::Tertiary_Ideographic_Plane }, // Plane 3 TIP
        Prop<::unicode::Plane> { { 0x40000, 0x4FFFF }, unicode::Plane::Unassigned }, // Plane 4
        Prop<::unicode::Plane> { { 0x50000, 0x5FFFF }, unicode::Plane::Unassigned }, // Plane 5
        Prop<::unicode::Plane> { { 0x60000, 0x6FFFF }, unicode::Plane::Unassigned }, // Plane 6
        Prop<::unicode::Plane> { { 0x70000, 0x7FFFF }, unicode::Plane::Unassigned }, // Plane 7
        Prop<::unicode::Plane> { { 0x80000, 0x8FFFF }, unicode::Plane::Unassigned }, // Plane 8
        Prop<::unicode::Plane> { { 0x90000, 0x9FFFF }, unicode::Plane::Unassigned }, // Plane 9
        Prop<::unicode::Plane> { { 0xA0000, 0xAFFFF }, unicode::Plane::Unassigned }, // Plane 10
        Prop<::unicode::Plane> { { 0xB0000, 0xBFFFF }, unicode::Plane::Unassigned }, // Plane 11
        Prop<::unicode::Plane> { { 0xC0000, 0xCFFFF }, unicode::Plane::Unassigned }, // Plane 12
        Prop<::unicode::Plane> { { 0xD0000, 0xDFFFF }, unicode::Plane::Unassigned }, // Plane 13
        Prop<::unicode::Plane> { { 0xE0000, 0xEFFFF }, unicode::Plane::Supplementary_Special_purpose_Plane }, // Plane 14 SSP
        Prop<::unicode::Plane> { { 0xF0000, 0xFFFFF }, unicode::Plane::Supplementary_Private_Use_Area_Plane }, // Plane 15 SPUA-A
        Prop<::unicode::Plane> { { 0x100000, 0x10FFFF }, unicode::Plane::Supplementary_Private_Use_Area_Plane }, // Plane 16 SPUA-B
    }; // }}}
    // clang-format off
} // namespace tables

Plane plane(char32_t codepoint) noexcept {
    return search(tables::Plane, codepoint).value_or(Plane::Unassigned);
}

namespace tables {
auto static const Alphabetic = std::array<Interval, 1140>{ // {{{
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00AA, 0x00AA }, // Lo       FEMININE ORDINAL INDICATOR
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00BA, 0x00BA }, // Lo       MASCULINE ORDINAL INDICATOR
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00F6 }, // L&  [31] LATIN CAPITAL LETTER O WITH STROKE..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x01BA }, // L& [195] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER EZH WITH TAIL
    Interval{ 0x01BB, 0x01BB }, // Lo       LATIN LETTER TWO WITH STROKE
    Interval{ 0x01BC, 0x01BF }, // L&   [4] LATIN CAPITAL LETTER TONE FIVE..LATIN LETTER WYNN
    Interval{ 0x01C0, 0x01C3 }, // Lo   [4] LATIN LETTER DENTAL CLICK..LATIN LETTER RETROFLEX CLICK
    Interval{ 0x01C4, 0x0293 }, // L& [208] LATIN CAPITAL LETTER DZ WITH CARON..LATIN SMALL LETTER EZH WITH CURL
    Interval{ 0x0294, 0x0294 }, // Lo       LATIN LETTER GLOTTAL STOP
    Interval{ 0x0295, 0x02AF }, // L&  [27] LATIN LETTER PHARYNGEAL VOICED FRICATIVE..LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    Interval{ 0x02B0, 0x02C1 }, // Lm  [18] MODIFIER LETTER SMALL H..MODIFIER LETTER REVERSED GLOTTAL STOP
    Interval{ 0x02C6, 0x02D1 }, // Lm  [12] MODIFIER LETTER CIRCUMFLEX ACCENT..MODIFIER LETTER HALF TRIANGULAR COLON
    Interval{ 0x02E0, 0x02E4 }, // Lm   [5] MODIFIER LETTER SMALL GAMMA..MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    Interval{ 0x02EC, 0x02EC }, // Lm       MODIFIER LETTER VOICING
    Interval{ 0x02EE, 0x02EE }, // Lm       MODIFIER LETTER DOUBLE APOSTROPHE
    Interval{ 0x0345, 0x0345 }, // Mn       COMBINING GREEK YPOGEGRAMMENI
    Interval{ 0x0370, 0x0373 }, // L&   [4] GREEK CAPITAL LETTER HETA..GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0374, 0x0374 }, // Lm       GREEK NUMERAL SIGN
    Interval{ 0x0376, 0x0377 }, // L&   [2] GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA..GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037A, 0x037A }, // Lm       GREEK YPOGEGRAMMENI
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x03A1 }, // L&  [20] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03F5 }, // L&  [83] GREEK CAPITAL LETTER SIGMA..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F7, 0x0481 }, // L& [139] GREEK CAPITAL LETTER SHO..CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x048A, 0x052F }, // L& [166] CYRILLIC CAPITAL LETTER SHORT I WITH TAIL..CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x0559, 0x0559 }, // Lm       ARMENIAN MODIFIER LETTER LEFT HALF RING
    Interval{ 0x0560, 0x0588 }, // L&  [41] ARMENIAN SMALL LETTER TURNED AYB..ARMENIAN SMALL LETTER YI WITH STROKE
    Interval{ 0x05B0, 0x05BD }, // Mn  [14] HEBREW POINT SHEVA..HEBREW POINT METEG
    Interval{ 0x05BF, 0x05BF }, // Mn       HEBREW POINT RAFE
    Interval{ 0x05C1, 0x05C2 }, // Mn   [2] HEBREW POINT SHIN DOT..HEBREW POINT SIN DOT
    Interval{ 0x05C4, 0x05C5 }, // Mn   [2] HEBREW MARK UPPER DOT..HEBREW MARK LOWER DOT
    Interval{ 0x05C7, 0x05C7 }, // Mn       HEBREW POINT QAMATS QATAN
    Interval{ 0x05D0, 0x05EA }, // Lo  [27] HEBREW LETTER ALEF..HEBREW LETTER TAV
    Interval{ 0x05EF, 0x05F2 }, // Lo   [4] HEBREW YOD TRIANGLE..HEBREW LIGATURE YIDDISH DOUBLE YOD
    Interval{ 0x0610, 0x061A }, // Mn  [11] ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM..ARABIC SMALL KASRA
    Interval{ 0x0620, 0x063F }, // Lo  [32] ARABIC LETTER KASHMIRI YEH..ARABIC LETTER FARSI YEH WITH THREE DOTS ABOVE
    Interval{ 0x0640, 0x0640 }, // Lm       ARABIC TATWEEL
    Interval{ 0x0641, 0x064A }, // Lo  [10] ARABIC LETTER FEH..ARABIC LETTER YEH
    Interval{ 0x064B, 0x0657 }, // Mn  [13] ARABIC FATHATAN..ARABIC INVERTED DAMMA
    Interval{ 0x0659, 0x065F }, // Mn   [7] ARABIC ZWARAKAY..ARABIC WAVY HAMZA BELOW
    Interval{ 0x066E, 0x066F }, // Lo   [2] ARABIC LETTER DOTLESS BEH..ARABIC LETTER DOTLESS QAF
    Interval{ 0x0670, 0x0670 }, // Mn       ARABIC LETTER SUPERSCRIPT ALEF
    Interval{ 0x0671, 0x06D3 }, // Lo  [99] ARABIC LETTER ALEF WASLA..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
    Interval{ 0x06D5, 0x06D5 }, // Lo       ARABIC LETTER AE
    Interval{ 0x06D6, 0x06DC }, // Mn   [7] ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA..ARABIC SMALL HIGH SEEN
    Interval{ 0x06E1, 0x06E4 }, // Mn   [4] ARABIC SMALL HIGH DOTLESS HEAD OF KHAH..ARABIC SMALL HIGH MADDA
    Interval{ 0x06E5, 0x06E6 }, // Lm   [2] ARABIC SMALL WAW..ARABIC SMALL YEH
    Interval{ 0x06E7, 0x06E8 }, // Mn   [2] ARABIC SMALL HIGH YEH..ARABIC SMALL HIGH NOON
    Interval{ 0x06ED, 0x06ED }, // Mn       ARABIC SMALL LOW MEEM
    Interval{ 0x06EE, 0x06EF }, // Lo   [2] ARABIC LETTER DAL WITH INVERTED V..ARABIC LETTER REH WITH INVERTED V
    Interval{ 0x06FA, 0x06FC }, // Lo   [3] ARABIC LETTER SHEEN WITH DOT BELOW..ARABIC LETTER GHAIN WITH DOT BELOW
    Interval{ 0x06FF, 0x06FF }, // Lo       ARABIC LETTER HEH WITH INVERTED V
    Interval{ 0x0710, 0x0710 }, // Lo       SYRIAC LETTER ALAPH
    Interval{ 0x0711, 0x0711 }, // Mn       SYRIAC LETTER SUPERSCRIPT ALAPH
    Interval{ 0x0712, 0x072F }, // Lo  [30] SYRIAC LETTER BETH..SYRIAC LETTER PERSIAN DHALATH
    Interval{ 0x0730, 0x073F }, // Mn  [16] SYRIAC PTHAHA ABOVE..SYRIAC RWAHA
    Interval{ 0x074D, 0x07A5 }, // Lo  [89] SYRIAC LETTER SOGDIAN ZHAIN..THAANA LETTER WAAVU
    Interval{ 0x07A6, 0x07B0 }, // Mn  [11] THAANA ABAFILI..THAANA SUKUN
    Interval{ 0x07B1, 0x07B1 }, // Lo       THAANA LETTER NAA
    Interval{ 0x07CA, 0x07EA }, // Lo  [33] NKO LETTER A..NKO LETTER JONA RA
    Interval{ 0x07F4, 0x07F5 }, // Lm   [2] NKO HIGH TONE APOSTROPHE..NKO LOW TONE APOSTROPHE
    Interval{ 0x07FA, 0x07FA }, // Lm       NKO LAJANYALAN
    Interval{ 0x0800, 0x0815 }, // Lo  [22] SAMARITAN LETTER ALAF..SAMARITAN LETTER TAAF
    Interval{ 0x0816, 0x0817 }, // Mn   [2] SAMARITAN MARK IN..SAMARITAN MARK IN-ALAF
    Interval{ 0x081A, 0x081A }, // Lm       SAMARITAN MODIFIER LETTER EPENTHETIC YUT
    Interval{ 0x081B, 0x0823 }, // Mn   [9] SAMARITAN MARK EPENTHETIC YUT..SAMARITAN VOWEL SIGN A
    Interval{ 0x0824, 0x0824 }, // Lm       SAMARITAN MODIFIER LETTER SHORT A
    Interval{ 0x0825, 0x0827 }, // Mn   [3] SAMARITAN VOWEL SIGN SHORT A..SAMARITAN VOWEL SIGN U
    Interval{ 0x0828, 0x0828 }, // Lm       SAMARITAN MODIFIER LETTER I
    Interval{ 0x0829, 0x082C }, // Mn   [4] SAMARITAN VOWEL SIGN LONG I..SAMARITAN VOWEL SIGN SUKUN
    Interval{ 0x0840, 0x0858 }, // Lo  [25] MANDAIC LETTER HALQA..MANDAIC LETTER AIN
    Interval{ 0x0860, 0x086A }, // Lo  [11] SYRIAC LETTER MALAYALAM NGA..SYRIAC LETTER MALAYALAM SSA
    Interval{ 0x0870, 0x0887 }, // Lo  [24] ARABIC LETTER ALEF WITH ATTACHED FATHA..ARABIC BASELINE ROUND DOT
    Interval{ 0x0889, 0x088E }, // Lo   [6] ARABIC LETTER NOON WITH INVERTED SMALL V..ARABIC VERTICAL TAIL
    Interval{ 0x08A0, 0x08C8 }, // Lo  [41] ARABIC LETTER BEH WITH SMALL V BELOW..ARABIC LETTER GRAF
    Interval{ 0x08C9, 0x08C9 }, // Lm       ARABIC SMALL FARSI YEH
    Interval{ 0x08D4, 0x08DF }, // Mn  [12] ARABIC SMALL HIGH WORD AR-RUB..ARABIC SMALL HIGH WORD WAQFA
    Interval{ 0x08E3, 0x08E9 }, // Mn   [7] ARABIC TURNED DAMMA BELOW..ARABIC CURLY KASRATAN
    Interval{ 0x08F0, 0x0902 }, // Mn  [19] ARABIC OPEN FATHATAN..DEVANAGARI SIGN ANUSVARA
    Interval{ 0x0903, 0x0903 }, // Mc       DEVANAGARI SIGN VISARGA
    Interval{ 0x0904, 0x0939 }, // Lo  [54] DEVANAGARI LETTER SHORT A..DEVANAGARI LETTER HA
    Interval{ 0x093A, 0x093A }, // Mn       DEVANAGARI VOWEL SIGN OE
    Interval{ 0x093B, 0x093B }, // Mc       DEVANAGARI VOWEL SIGN OOE
    Interval{ 0x093D, 0x093D }, // Lo       DEVANAGARI SIGN AVAGRAHA
    Interval{ 0x093E, 0x0940 }, // Mc   [3] DEVANAGARI VOWEL SIGN AA..DEVANAGARI VOWEL SIGN II
    Interval{ 0x0941, 0x0948 }, // Mn   [8] DEVANAGARI VOWEL SIGN U..DEVANAGARI VOWEL SIGN AI
    Interval{ 0x0949, 0x094C }, // Mc   [4] DEVANAGARI VOWEL SIGN CANDRA O..DEVANAGARI VOWEL SIGN AU
    Interval{ 0x094E, 0x094F }, // Mc   [2] DEVANAGARI VOWEL SIGN PRISHTHAMATRA E..DEVANAGARI VOWEL SIGN AW
    Interval{ 0x0950, 0x0950 }, // Lo       DEVANAGARI OM
    Interval{ 0x0955, 0x0957 }, // Mn   [3] DEVANAGARI VOWEL SIGN CANDRA LONG E..DEVANAGARI VOWEL SIGN UUE
    Interval{ 0x0958, 0x0961 }, // Lo  [10] DEVANAGARI LETTER QA..DEVANAGARI LETTER VOCALIC LL
    Interval{ 0x0962, 0x0963 }, // Mn   [2] DEVANAGARI VOWEL SIGN VOCALIC L..DEVANAGARI VOWEL SIGN VOCALIC LL
    Interval{ 0x0971, 0x0971 }, // Lm       DEVANAGARI SIGN HIGH SPACING DOT
    Interval{ 0x0972, 0x0980 }, // Lo  [15] DEVANAGARI LETTER CANDRA A..BENGALI ANJI
    Interval{ 0x0981, 0x0981 }, // Mn       BENGALI SIGN CANDRABINDU
    Interval{ 0x0982, 0x0983 }, // Mc   [2] BENGALI SIGN ANUSVARA..BENGALI SIGN VISARGA
    Interval{ 0x0985, 0x098C }, // Lo   [8] BENGALI LETTER A..BENGALI LETTER VOCALIC L
    Interval{ 0x098F, 0x0990 }, // Lo   [2] BENGALI LETTER E..BENGALI LETTER AI
    Interval{ 0x0993, 0x09A8 }, // Lo  [22] BENGALI LETTER O..BENGALI LETTER NA
    Interval{ 0x09AA, 0x09B0 }, // Lo   [7] BENGALI LETTER PA..BENGALI LETTER RA
    Interval{ 0x09B2, 0x09B2 }, // Lo       BENGALI LETTER LA
    Interval{ 0x09B6, 0x09B9 }, // Lo   [4] BENGALI LETTER SHA..BENGALI LETTER HA
    Interval{ 0x09BD, 0x09BD }, // Lo       BENGALI SIGN AVAGRAHA
    Interval{ 0x09BE, 0x09C0 }, // Mc   [3] BENGALI VOWEL SIGN AA..BENGALI VOWEL SIGN II
    Interval{ 0x09C1, 0x09C4 }, // Mn   [4] BENGALI VOWEL SIGN U..BENGALI VOWEL SIGN VOCALIC RR
    Interval{ 0x09C7, 0x09C8 }, // Mc   [2] BENGALI VOWEL SIGN E..BENGALI VOWEL SIGN AI
    Interval{ 0x09CB, 0x09CC }, // Mc   [2] BENGALI VOWEL SIGN O..BENGALI VOWEL SIGN AU
    Interval{ 0x09CE, 0x09CE }, // Lo       BENGALI LETTER KHANDA TA
    Interval{ 0x09D7, 0x09D7 }, // Mc       BENGALI AU LENGTH MARK
    Interval{ 0x09DC, 0x09DD }, // Lo   [2] BENGALI LETTER RRA..BENGALI LETTER RHA
    Interval{ 0x09DF, 0x09E1 }, // Lo   [3] BENGALI LETTER YYA..BENGALI LETTER VOCALIC LL
    Interval{ 0x09E2, 0x09E3 }, // Mn   [2] BENGALI VOWEL SIGN VOCALIC L..BENGALI VOWEL SIGN VOCALIC LL
    Interval{ 0x09F0, 0x09F1 }, // Lo   [2] BENGALI LETTER RA WITH MIDDLE DIAGONAL..BENGALI LETTER RA WITH LOWER DIAGONAL
    Interval{ 0x09FC, 0x09FC }, // Lo       BENGALI LETTER VEDIC ANUSVARA
    Interval{ 0x0A01, 0x0A02 }, // Mn   [2] GURMUKHI SIGN ADAK BINDI..GURMUKHI SIGN BINDI
    Interval{ 0x0A03, 0x0A03 }, // Mc       GURMUKHI SIGN VISARGA
    Interval{ 0x0A05, 0x0A0A }, // Lo   [6] GURMUKHI LETTER A..GURMUKHI LETTER UU
    Interval{ 0x0A0F, 0x0A10 }, // Lo   [2] GURMUKHI LETTER EE..GURMUKHI LETTER AI
    Interval{ 0x0A13, 0x0A28 }, // Lo  [22] GURMUKHI LETTER OO..GURMUKHI LETTER NA
    Interval{ 0x0A2A, 0x0A30 }, // Lo   [7] GURMUKHI LETTER PA..GURMUKHI LETTER RA
    Interval{ 0x0A32, 0x0A33 }, // Lo   [2] GURMUKHI LETTER LA..GURMUKHI LETTER LLA
    Interval{ 0x0A35, 0x0A36 }, // Lo   [2] GURMUKHI LETTER VA..GURMUKHI LETTER SHA
    Interval{ 0x0A38, 0x0A39 }, // Lo   [2] GURMUKHI LETTER SA..GURMUKHI LETTER HA
    Interval{ 0x0A3E, 0x0A40 }, // Mc   [3] GURMUKHI VOWEL SIGN AA..GURMUKHI VOWEL SIGN II
    Interval{ 0x0A41, 0x0A42 }, // Mn   [2] GURMUKHI VOWEL SIGN U..GURMUKHI VOWEL SIGN UU
    Interval{ 0x0A47, 0x0A48 }, // Mn   [2] GURMUKHI VOWEL SIGN EE..GURMUKHI VOWEL SIGN AI
    Interval{ 0x0A4B, 0x0A4C }, // Mn   [2] GURMUKHI VOWEL SIGN OO..GURMUKHI VOWEL SIGN AU
    Interval{ 0x0A51, 0x0A51 }, // Mn       GURMUKHI SIGN UDAAT
    Interval{ 0x0A59, 0x0A5C }, // Lo   [4] GURMUKHI LETTER KHHA..GURMUKHI LETTER RRA
    Interval{ 0x0A5E, 0x0A5E }, // Lo       GURMUKHI LETTER FA
    Interval{ 0x0A70, 0x0A71 }, // Mn   [2] GURMUKHI TIPPI..GURMUKHI ADDAK
    Interval{ 0x0A72, 0x0A74 }, // Lo   [3] GURMUKHI IRI..GURMUKHI EK ONKAR
    Interval{ 0x0A75, 0x0A75 }, // Mn       GURMUKHI SIGN YAKASH
    Interval{ 0x0A81, 0x0A82 }, // Mn   [2] GUJARATI SIGN CANDRABINDU..GUJARATI SIGN ANUSVARA
    Interval{ 0x0A83, 0x0A83 }, // Mc       GUJARATI SIGN VISARGA
    Interval{ 0x0A85, 0x0A8D }, // Lo   [9] GUJARATI LETTER A..GUJARATI VOWEL CANDRA E
    Interval{ 0x0A8F, 0x0A91 }, // Lo   [3] GUJARATI LETTER E..GUJARATI VOWEL CANDRA O
    Interval{ 0x0A93, 0x0AA8 }, // Lo  [22] GUJARATI LETTER O..GUJARATI LETTER NA
    Interval{ 0x0AAA, 0x0AB0 }, // Lo   [7] GUJARATI LETTER PA..GUJARATI LETTER RA
    Interval{ 0x0AB2, 0x0AB3 }, // Lo   [2] GUJARATI LETTER LA..GUJARATI LETTER LLA
    Interval{ 0x0AB5, 0x0AB9 }, // Lo   [5] GUJARATI LETTER VA..GUJARATI LETTER HA
    Interval{ 0x0ABD, 0x0ABD }, // Lo       GUJARATI SIGN AVAGRAHA
    Interval{ 0x0ABE, 0x0AC0 }, // Mc   [3] GUJARATI VOWEL SIGN AA..GUJARATI VOWEL SIGN II
    Interval{ 0x0AC1, 0x0AC5 }, // Mn   [5] GUJARATI VOWEL SIGN U..GUJARATI VOWEL SIGN CANDRA E
    Interval{ 0x0AC7, 0x0AC8 }, // Mn   [2] GUJARATI VOWEL SIGN E..GUJARATI VOWEL SIGN AI
    Interval{ 0x0AC9, 0x0AC9 }, // Mc       GUJARATI VOWEL SIGN CANDRA O
    Interval{ 0x0ACB, 0x0ACC }, // Mc   [2] GUJARATI VOWEL SIGN O..GUJARATI VOWEL SIGN AU
    Interval{ 0x0AD0, 0x0AD0 }, // Lo       GUJARATI OM
    Interval{ 0x0AE0, 0x0AE1 }, // Lo   [2] GUJARATI LETTER VOCALIC RR..GUJARATI LETTER VOCALIC LL
    Interval{ 0x0AE2, 0x0AE3 }, // Mn   [2] GUJARATI VOWEL SIGN VOCALIC L..GUJARATI VOWEL SIGN VOCALIC LL
    Interval{ 0x0AF9, 0x0AF9 }, // Lo       GUJARATI LETTER ZHA
    Interval{ 0x0AFA, 0x0AFC }, // Mn   [3] GUJARATI SIGN SUKUN..GUJARATI SIGN MADDAH
    Interval{ 0x0B01, 0x0B01 }, // Mn       ORIYA SIGN CANDRABINDU
    Interval{ 0x0B02, 0x0B03 }, // Mc   [2] ORIYA SIGN ANUSVARA..ORIYA SIGN VISARGA
    Interval{ 0x0B05, 0x0B0C }, // Lo   [8] ORIYA LETTER A..ORIYA LETTER VOCALIC L
    Interval{ 0x0B0F, 0x0B10 }, // Lo   [2] ORIYA LETTER E..ORIYA LETTER AI
    Interval{ 0x0B13, 0x0B28 }, // Lo  [22] ORIYA LETTER O..ORIYA LETTER NA
    Interval{ 0x0B2A, 0x0B30 }, // Lo   [7] ORIYA LETTER PA..ORIYA LETTER RA
    Interval{ 0x0B32, 0x0B33 }, // Lo   [2] ORIYA LETTER LA..ORIYA LETTER LLA
    Interval{ 0x0B35, 0x0B39 }, // Lo   [5] ORIYA LETTER VA..ORIYA LETTER HA
    Interval{ 0x0B3D, 0x0B3D }, // Lo       ORIYA SIGN AVAGRAHA
    Interval{ 0x0B3E, 0x0B3E }, // Mc       ORIYA VOWEL SIGN AA
    Interval{ 0x0B3F, 0x0B3F }, // Mn       ORIYA VOWEL SIGN I
    Interval{ 0x0B40, 0x0B40 }, // Mc       ORIYA VOWEL SIGN II
    Interval{ 0x0B41, 0x0B44 }, // Mn   [4] ORIYA VOWEL SIGN U..ORIYA VOWEL SIGN VOCALIC RR
    Interval{ 0x0B47, 0x0B48 }, // Mc   [2] ORIYA VOWEL SIGN E..ORIYA VOWEL SIGN AI
    Interval{ 0x0B4B, 0x0B4C }, // Mc   [2] ORIYA VOWEL SIGN O..ORIYA VOWEL SIGN AU
    Interval{ 0x0B56, 0x0B56 }, // Mn       ORIYA AI LENGTH MARK
    Interval{ 0x0B57, 0x0B57 }, // Mc       ORIYA AU LENGTH MARK
    Interval{ 0x0B5C, 0x0B5D }, // Lo   [2] ORIYA LETTER RRA..ORIYA LETTER RHA
    Interval{ 0x0B5F, 0x0B61 }, // Lo   [3] ORIYA LETTER YYA..ORIYA LETTER VOCALIC LL
    Interval{ 0x0B62, 0x0B63 }, // Mn   [2] ORIYA VOWEL SIGN VOCALIC L..ORIYA VOWEL SIGN VOCALIC LL
    Interval{ 0x0B71, 0x0B71 }, // Lo       ORIYA LETTER WA
    Interval{ 0x0B82, 0x0B82 }, // Mn       TAMIL SIGN ANUSVARA
    Interval{ 0x0B83, 0x0B83 }, // Lo       TAMIL SIGN VISARGA
    Interval{ 0x0B85, 0x0B8A }, // Lo   [6] TAMIL LETTER A..TAMIL LETTER UU
    Interval{ 0x0B8E, 0x0B90 }, // Lo   [3] TAMIL LETTER E..TAMIL LETTER AI
    Interval{ 0x0B92, 0x0B95 }, // Lo   [4] TAMIL LETTER O..TAMIL LETTER KA
    Interval{ 0x0B99, 0x0B9A }, // Lo   [2] TAMIL LETTER NGA..TAMIL LETTER CA
    Interval{ 0x0B9C, 0x0B9C }, // Lo       TAMIL LETTER JA
    Interval{ 0x0B9E, 0x0B9F }, // Lo   [2] TAMIL LETTER NYA..TAMIL LETTER TTA
    Interval{ 0x0BA3, 0x0BA4 }, // Lo   [2] TAMIL LETTER NNA..TAMIL LETTER TA
    Interval{ 0x0BA8, 0x0BAA }, // Lo   [3] TAMIL LETTER NA..TAMIL LETTER PA
    Interval{ 0x0BAE, 0x0BB9 }, // Lo  [12] TAMIL LETTER MA..TAMIL LETTER HA
    Interval{ 0x0BBE, 0x0BBF }, // Mc   [2] TAMIL VOWEL SIGN AA..TAMIL VOWEL SIGN I
    Interval{ 0x0BC0, 0x0BC0 }, // Mn       TAMIL VOWEL SIGN II
    Interval{ 0x0BC1, 0x0BC2 }, // Mc   [2] TAMIL VOWEL SIGN U..TAMIL VOWEL SIGN UU
    Interval{ 0x0BC6, 0x0BC8 }, // Mc   [3] TAMIL VOWEL SIGN E..TAMIL VOWEL SIGN AI
    Interval{ 0x0BCA, 0x0BCC }, // Mc   [3] TAMIL VOWEL SIGN O..TAMIL VOWEL SIGN AU
    Interval{ 0x0BD0, 0x0BD0 }, // Lo       TAMIL OM
    Interval{ 0x0BD7, 0x0BD7 }, // Mc       TAMIL AU LENGTH MARK
    Interval{ 0x0C00, 0x0C00 }, // Mn       TELUGU SIGN COMBINING CANDRABINDU ABOVE
    Interval{ 0x0C01, 0x0C03 }, // Mc   [3] TELUGU SIGN CANDRABINDU..TELUGU SIGN VISARGA
    Interval{ 0x0C04, 0x0C04 }, // Mn       TELUGU SIGN COMBINING ANUSVARA ABOVE
    Interval{ 0x0C05, 0x0C0C }, // Lo   [8] TELUGU LETTER A..TELUGU LETTER VOCALIC L
    Interval{ 0x0C0E, 0x0C10 }, // Lo   [3] TELUGU LETTER E..TELUGU LETTER AI
    Interval{ 0x0C12, 0x0C28 }, // Lo  [23] TELUGU LETTER O..TELUGU LETTER NA
    Interval{ 0x0C2A, 0x0C39 }, // Lo  [16] TELUGU LETTER PA..TELUGU LETTER HA
    Interval{ 0x0C3D, 0x0C3D }, // Lo       TELUGU SIGN AVAGRAHA
    Interval{ 0x0C3E, 0x0C40 }, // Mn   [3] TELUGU VOWEL SIGN AA..TELUGU VOWEL SIGN II
    Interval{ 0x0C41, 0x0C44 }, // Mc   [4] TELUGU VOWEL SIGN U..TELUGU VOWEL SIGN VOCALIC RR
    Interval{ 0x0C46, 0x0C48 }, // Mn   [3] TELUGU VOWEL SIGN E..TELUGU VOWEL SIGN AI
    Interval{ 0x0C4A, 0x0C4C }, // Mn   [3] TELUGU VOWEL SIGN O..TELUGU VOWEL SIGN AU
    Interval{ 0x0C55, 0x0C56 }, // Mn   [2] TELUGU LENGTH MARK..TELUGU AI LENGTH MARK
    Interval{ 0x0C58, 0x0C5A }, // Lo   [3] TELUGU LETTER TSA..TELUGU LETTER RRRA
    Interval{ 0x0C5D, 0x0C5D }, // Lo       TELUGU LETTER NAKAARA POLLU
    Interval{ 0x0C60, 0x0C61 }, // Lo   [2] TELUGU LETTER VOCALIC RR..TELUGU LETTER VOCALIC LL
    Interval{ 0x0C62, 0x0C63 }, // Mn   [2] TELUGU VOWEL SIGN VOCALIC L..TELUGU VOWEL SIGN VOCALIC LL
    Interval{ 0x0C80, 0x0C80 }, // Lo       KANNADA SIGN SPACING CANDRABINDU
    Interval{ 0x0C81, 0x0C81 }, // Mn       KANNADA SIGN CANDRABINDU
    Interval{ 0x0C82, 0x0C83 }, // Mc   [2] KANNADA SIGN ANUSVARA..KANNADA SIGN VISARGA
    Interval{ 0x0C85, 0x0C8C }, // Lo   [8] KANNADA LETTER A..KANNADA LETTER VOCALIC L
    Interval{ 0x0C8E, 0x0C90 }, // Lo   [3] KANNADA LETTER E..KANNADA LETTER AI
    Interval{ 0x0C92, 0x0CA8 }, // Lo  [23] KANNADA LETTER O..KANNADA LETTER NA
    Interval{ 0x0CAA, 0x0CB3 }, // Lo  [10] KANNADA LETTER PA..KANNADA LETTER LLA
    Interval{ 0x0CB5, 0x0CB9 }, // Lo   [5] KANNADA LETTER VA..KANNADA LETTER HA
    Interval{ 0x0CBD, 0x0CBD }, // Lo       KANNADA SIGN AVAGRAHA
    Interval{ 0x0CBE, 0x0CBE }, // Mc       KANNADA VOWEL SIGN AA
    Interval{ 0x0CBF, 0x0CBF }, // Mn       KANNADA VOWEL SIGN I
    Interval{ 0x0CC0, 0x0CC4 }, // Mc   [5] KANNADA VOWEL SIGN II..KANNADA VOWEL SIGN VOCALIC RR
    Interval{ 0x0CC6, 0x0CC6 }, // Mn       KANNADA VOWEL SIGN E
    Interval{ 0x0CC7, 0x0CC8 }, // Mc   [2] KANNADA VOWEL SIGN EE..KANNADA VOWEL SIGN AI
    Interval{ 0x0CCA, 0x0CCB }, // Mc   [2] KANNADA VOWEL SIGN O..KANNADA VOWEL SIGN OO
    Interval{ 0x0CCC, 0x0CCC }, // Mn       KANNADA VOWEL SIGN AU
    Interval{ 0x0CD5, 0x0CD6 }, // Mc   [2] KANNADA LENGTH MARK..KANNADA AI LENGTH MARK
    Interval{ 0x0CDD, 0x0CDE }, // Lo   [2] KANNADA LETTER NAKAARA POLLU..KANNADA LETTER FA
    Interval{ 0x0CE0, 0x0CE1 }, // Lo   [2] KANNADA LETTER VOCALIC RR..KANNADA LETTER VOCALIC LL
    Interval{ 0x0CE2, 0x0CE3 }, // Mn   [2] KANNADA VOWEL SIGN VOCALIC L..KANNADA VOWEL SIGN VOCALIC LL
    Interval{ 0x0CF1, 0x0CF2 }, // Lo   [2] KANNADA SIGN JIHVAMULIYA..KANNADA SIGN UPADHMANIYA
    Interval{ 0x0CF3, 0x0CF3 }, // Mc       KANNADA SIGN COMBINING ANUSVARA ABOVE RIGHT
    Interval{ 0x0D00, 0x0D01 }, // Mn   [2] MALAYALAM SIGN COMBINING ANUSVARA ABOVE..MALAYALAM SIGN CANDRABINDU
    Interval{ 0x0D02, 0x0D03 }, // Mc   [2] MALAYALAM SIGN ANUSVARA..MALAYALAM SIGN VISARGA
    Interval{ 0x0D04, 0x0D0C }, // Lo   [9] MALAYALAM LETTER VEDIC ANUSVARA..MALAYALAM LETTER VOCALIC L
    Interval{ 0x0D0E, 0x0D10 }, // Lo   [3] MALAYALAM LETTER E..MALAYALAM LETTER AI
    Interval{ 0x0D12, 0x0D3A }, // Lo  [41] MALAYALAM LETTER O..MALAYALAM LETTER TTTA
    Interval{ 0x0D3D, 0x0D3D }, // Lo       MALAYALAM SIGN AVAGRAHA
    Interval{ 0x0D3E, 0x0D40 }, // Mc   [3] MALAYALAM VOWEL SIGN AA..MALAYALAM VOWEL SIGN II
    Interval{ 0x0D41, 0x0D44 }, // Mn   [4] MALAYALAM VOWEL SIGN U..MALAYALAM VOWEL SIGN VOCALIC RR
    Interval{ 0x0D46, 0x0D48 }, // Mc   [3] MALAYALAM VOWEL SIGN E..MALAYALAM VOWEL SIGN AI
    Interval{ 0x0D4A, 0x0D4C }, // Mc   [3] MALAYALAM VOWEL SIGN O..MALAYALAM VOWEL SIGN AU
    Interval{ 0x0D4E, 0x0D4E }, // Lo       MALAYALAM LETTER DOT REPH
    Interval{ 0x0D54, 0x0D56 }, // Lo   [3] MALAYALAM LETTER CHILLU M..MALAYALAM LETTER CHILLU LLL
    Interval{ 0x0D57, 0x0D57 }, // Mc       MALAYALAM AU LENGTH MARK
    Interval{ 0x0D5F, 0x0D61 }, // Lo   [3] MALAYALAM LETTER ARCHAIC II..MALAYALAM LETTER VOCALIC LL
    Interval{ 0x0D62, 0x0D63 }, // Mn   [2] MALAYALAM VOWEL SIGN VOCALIC L..MALAYALAM VOWEL SIGN VOCALIC LL
    Interval{ 0x0D7A, 0x0D7F }, // Lo   [6] MALAYALAM LETTER CHILLU NN..MALAYALAM LETTER CHILLU K
    Interval{ 0x0D81, 0x0D81 }, // Mn       SINHALA SIGN CANDRABINDU
    Interval{ 0x0D82, 0x0D83 }, // Mc   [2] SINHALA SIGN ANUSVARAYA..SINHALA SIGN VISARGAYA
    Interval{ 0x0D85, 0x0D96 }, // Lo  [18] SINHALA LETTER AYANNA..SINHALA LETTER AUYANNA
    Interval{ 0x0D9A, 0x0DB1 }, // Lo  [24] SINHALA LETTER ALPAPRAANA KAYANNA..SINHALA LETTER DANTAJA NAYANNA
    Interval{ 0x0DB3, 0x0DBB }, // Lo   [9] SINHALA LETTER SANYAKA DAYANNA..SINHALA LETTER RAYANNA
    Interval{ 0x0DBD, 0x0DBD }, // Lo       SINHALA LETTER DANTAJA LAYANNA
    Interval{ 0x0DC0, 0x0DC6 }, // Lo   [7] SINHALA LETTER VAYANNA..SINHALA LETTER FAYANNA
    Interval{ 0x0DCF, 0x0DD1 }, // Mc   [3] SINHALA VOWEL SIGN AELA-PILLA..SINHALA VOWEL SIGN DIGA AEDA-PILLA
    Interval{ 0x0DD2, 0x0DD4 }, // Mn   [3] SINHALA VOWEL SIGN KETTI IS-PILLA..SINHALA VOWEL SIGN KETTI PAA-PILLA
    Interval{ 0x0DD6, 0x0DD6 }, // Mn       SINHALA VOWEL SIGN DIGA PAA-PILLA
    Interval{ 0x0DD8, 0x0DDF }, // Mc   [8] SINHALA VOWEL SIGN GAETTA-PILLA..SINHALA VOWEL SIGN GAYANUKITTA
    Interval{ 0x0DF2, 0x0DF3 }, // Mc   [2] SINHALA VOWEL SIGN DIGA GAETTA-PILLA..SINHALA VOWEL SIGN DIGA GAYANUKITTA
    Interval{ 0x0E01, 0x0E30 }, // Lo  [48] THAI CHARACTER KO KAI..THAI CHARACTER SARA A
    Interval{ 0x0E31, 0x0E31 }, // Mn       THAI CHARACTER MAI HAN-AKAT
    Interval{ 0x0E32, 0x0E33 }, // Lo   [2] THAI CHARACTER SARA AA..THAI CHARACTER SARA AM
    Interval{ 0x0E34, 0x0E3A }, // Mn   [7] THAI CHARACTER SARA I..THAI CHARACTER PHINTHU
    Interval{ 0x0E40, 0x0E45 }, // Lo   [6] THAI CHARACTER SARA E..THAI CHARACTER LAKKHANGYAO
    Interval{ 0x0E46, 0x0E46 }, // Lm       THAI CHARACTER MAIYAMOK
    Interval{ 0x0E4D, 0x0E4D }, // Mn       THAI CHARACTER NIKHAHIT
    Interval{ 0x0E81, 0x0E82 }, // Lo   [2] LAO LETTER KO..LAO LETTER KHO SUNG
    Interval{ 0x0E84, 0x0E84 }, // Lo       LAO LETTER KHO TAM
    Interval{ 0x0E86, 0x0E8A }, // Lo   [5] LAO LETTER PALI GHA..LAO LETTER SO TAM
    Interval{ 0x0E8C, 0x0EA3 }, // Lo  [24] LAO LETTER PALI JHA..LAO LETTER LO LING
    Interval{ 0x0EA5, 0x0EA5 }, // Lo       LAO LETTER LO LOOT
    Interval{ 0x0EA7, 0x0EB0 }, // Lo  [10] LAO LETTER WO..LAO VOWEL SIGN A
    Interval{ 0x0EB1, 0x0EB1 }, // Mn       LAO VOWEL SIGN MAI KAN
    Interval{ 0x0EB2, 0x0EB3 }, // Lo   [2] LAO VOWEL SIGN AA..LAO VOWEL SIGN AM
    Interval{ 0x0EB4, 0x0EB9 }, // Mn   [6] LAO VOWEL SIGN I..LAO VOWEL SIGN UU
    Interval{ 0x0EBB, 0x0EBC }, // Mn   [2] LAO VOWEL SIGN MAI KON..LAO SEMIVOWEL SIGN LO
    Interval{ 0x0EBD, 0x0EBD }, // Lo       LAO SEMIVOWEL SIGN NYO
    Interval{ 0x0EC0, 0x0EC4 }, // Lo   [5] LAO VOWEL SIGN E..LAO VOWEL SIGN AI
    Interval{ 0x0EC6, 0x0EC6 }, // Lm       LAO KO LA
    Interval{ 0x0ECD, 0x0ECD }, // Mn       LAO NIGGAHITA
    Interval{ 0x0EDC, 0x0EDF }, // Lo   [4] LAO HO NO..LAO LETTER KHMU NYO
    Interval{ 0x0F00, 0x0F00 }, // Lo       TIBETAN SYLLABLE OM
    Interval{ 0x0F40, 0x0F47 }, // Lo   [8] TIBETAN LETTER KA..TIBETAN LETTER JA
    Interval{ 0x0F49, 0x0F6C }, // Lo  [36] TIBETAN LETTER NYA..TIBETAN LETTER RRA
    Interval{ 0x0F71, 0x0F7E }, // Mn  [14] TIBETAN VOWEL SIGN AA..TIBETAN SIGN RJES SU NGA RO
    Interval{ 0x0F7F, 0x0F7F }, // Mc       TIBETAN SIGN RNAM BCAD
    Interval{ 0x0F80, 0x0F83 }, // Mn   [4] TIBETAN VOWEL SIGN REVERSED I..TIBETAN SIGN SNA LDAN
    Interval{ 0x0F88, 0x0F8C }, // Lo   [5] TIBETAN SIGN LCE TSA CAN..TIBETAN SIGN INVERTED MCHU CAN
    Interval{ 0x0F8D, 0x0F97 }, // Mn  [11] TIBETAN SUBJOINED SIGN LCE TSA CAN..TIBETAN SUBJOINED LETTER JA
    Interval{ 0x0F99, 0x0FBC }, // Mn  [36] TIBETAN SUBJOINED LETTER NYA..TIBETAN SUBJOINED LETTER FIXED-FORM RA
    Interval{ 0x1000, 0x102A }, // Lo  [43] MYANMAR LETTER KA..MYANMAR LETTER AU
    Interval{ 0x102B, 0x102C }, // Mc   [2] MYANMAR VOWEL SIGN TALL AA..MYANMAR VOWEL SIGN AA
    Interval{ 0x102D, 0x1030 }, // Mn   [4] MYANMAR VOWEL SIGN I..MYANMAR VOWEL SIGN UU
    Interval{ 0x1031, 0x1031 }, // Mc       MYANMAR VOWEL SIGN E
    Interval{ 0x1032, 0x1036 }, // Mn   [5] MYANMAR VOWEL SIGN AI..MYANMAR SIGN ANUSVARA
    Interval{ 0x1038, 0x1038 }, // Mc       MYANMAR SIGN VISARGA
    Interval{ 0x103B, 0x103C }, // Mc   [2] MYANMAR CONSONANT SIGN MEDIAL YA..MYANMAR CONSONANT SIGN MEDIAL RA
    Interval{ 0x103D, 0x103E }, // Mn   [2] MYANMAR CONSONANT SIGN MEDIAL WA..MYANMAR CONSONANT SIGN MEDIAL HA
    Interval{ 0x103F, 0x103F }, // Lo       MYANMAR LETTER GREAT SA
    Interval{ 0x1050, 0x1055 }, // Lo   [6] MYANMAR LETTER SHA..MYANMAR LETTER VOCALIC LL
    Interval{ 0x1056, 0x1057 }, // Mc   [2] MYANMAR VOWEL SIGN VOCALIC R..MYANMAR VOWEL SIGN VOCALIC RR
    Interval{ 0x1058, 0x1059 }, // Mn   [2] MYANMAR VOWEL SIGN VOCALIC L..MYANMAR VOWEL SIGN VOCALIC LL
    Interval{ 0x105A, 0x105D }, // Lo   [4] MYANMAR LETTER MON NGA..MYANMAR LETTER MON BBE
    Interval{ 0x105E, 0x1060 }, // Mn   [3] MYANMAR CONSONANT SIGN MON MEDIAL NA..MYANMAR CONSONANT SIGN MON MEDIAL LA
    Interval{ 0x1061, 0x1061 }, // Lo       MYANMAR LETTER SGAW KAREN SHA
    Interval{ 0x1062, 0x1064 }, // Mc   [3] MYANMAR VOWEL SIGN SGAW KAREN EU..MYANMAR TONE MARK SGAW KAREN KE PHO
    Interval{ 0x1065, 0x1066 }, // Lo   [2] MYANMAR LETTER WESTERN PWO KAREN THA..MYANMAR LETTER WESTERN PWO KAREN PWA
    Interval{ 0x1067, 0x106D }, // Mc   [7] MYANMAR VOWEL SIGN WESTERN PWO KAREN EU..MYANMAR SIGN WESTERN PWO KAREN TONE-5
    Interval{ 0x106E, 0x1070 }, // Lo   [3] MYANMAR LETTER EASTERN PWO KAREN NNA..MYANMAR LETTER EASTERN PWO KAREN GHWA
    Interval{ 0x1071, 0x1074 }, // Mn   [4] MYANMAR VOWEL SIGN GEBA KAREN I..MYANMAR VOWEL SIGN KAYAH EE
    Interval{ 0x1075, 0x1081 }, // Lo  [13] MYANMAR LETTER SHAN KA..MYANMAR LETTER SHAN HA
    Interval{ 0x1082, 0x1082 }, // Mn       MYANMAR CONSONANT SIGN SHAN MEDIAL WA
    Interval{ 0x1083, 0x1084 }, // Mc   [2] MYANMAR VOWEL SIGN SHAN AA..MYANMAR VOWEL SIGN SHAN E
    Interval{ 0x1085, 0x1086 }, // Mn   [2] MYANMAR VOWEL SIGN SHAN E ABOVE..MYANMAR VOWEL SIGN SHAN FINAL Y
    Interval{ 0x1087, 0x108C }, // Mc   [6] MYANMAR SIGN SHAN TONE-2..MYANMAR SIGN SHAN COUNCIL TONE-3
    Interval{ 0x108D, 0x108D }, // Mn       MYANMAR SIGN SHAN COUNCIL EMPHATIC TONE
    Interval{ 0x108E, 0x108E }, // Lo       MYANMAR LETTER RUMAI PALAUNG FA
    Interval{ 0x108F, 0x108F }, // Mc       MYANMAR SIGN RUMAI PALAUNG TONE-5
    Interval{ 0x109A, 0x109C }, // Mc   [3] MYANMAR SIGN KHAMTI TONE-1..MYANMAR VOWEL SIGN AITON A
    Interval{ 0x109D, 0x109D }, // Mn       MYANMAR VOWEL SIGN AITON AI
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FC, 0x10FC }, // Lm       MODIFIER LETTER GEORGIAN NAR
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x1100, 0x1248 }, // Lo [329] HANGUL CHOSEONG KIYEOK..ETHIOPIC SYLLABLE QWA
    Interval{ 0x124A, 0x124D }, // Lo   [4] ETHIOPIC SYLLABLE QWI..ETHIOPIC SYLLABLE QWE
    Interval{ 0x1250, 0x1256 }, // Lo   [7] ETHIOPIC SYLLABLE QHA..ETHIOPIC SYLLABLE QHO
    Interval{ 0x1258, 0x1258 }, // Lo       ETHIOPIC SYLLABLE QHWA
    Interval{ 0x125A, 0x125D }, // Lo   [4] ETHIOPIC SYLLABLE QHWI..ETHIOPIC SYLLABLE QHWE
    Interval{ 0x1260, 0x1288 }, // Lo  [41] ETHIOPIC SYLLABLE BA..ETHIOPIC SYLLABLE XWA
    Interval{ 0x128A, 0x128D }, // Lo   [4] ETHIOPIC SYLLABLE XWI..ETHIOPIC SYLLABLE XWE
    Interval{ 0x1290, 0x12B0 }, // Lo  [33] ETHIOPIC SYLLABLE NA..ETHIOPIC SYLLABLE KWA
    Interval{ 0x12B2, 0x12B5 }, // Lo   [4] ETHIOPIC SYLLABLE KWI..ETHIOPIC SYLLABLE KWE
    Interval{ 0x12B8, 0x12BE }, // Lo   [7] ETHIOPIC SYLLABLE KXA..ETHIOPIC SYLLABLE KXO
    Interval{ 0x12C0, 0x12C0 }, // Lo       ETHIOPIC SYLLABLE KXWA
    Interval{ 0x12C2, 0x12C5 }, // Lo   [4] ETHIOPIC SYLLABLE KXWI..ETHIOPIC SYLLABLE KXWE
    Interval{ 0x12C8, 0x12D6 }, // Lo  [15] ETHIOPIC SYLLABLE WA..ETHIOPIC SYLLABLE PHARYNGEAL O
    Interval{ 0x12D8, 0x1310 }, // Lo  [57] ETHIOPIC SYLLABLE ZA..ETHIOPIC SYLLABLE GWA
    Interval{ 0x1312, 0x1315 }, // Lo   [4] ETHIOPIC SYLLABLE GWI..ETHIOPIC SYLLABLE GWE
    Interval{ 0x1318, 0x135A }, // Lo  [67] ETHIOPIC SYLLABLE GGA..ETHIOPIC SYLLABLE FYA
    Interval{ 0x1380, 0x138F }, // Lo  [16] ETHIOPIC SYLLABLE SEBATBEIT MWA..ETHIOPIC SYLLABLE PWE
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1401, 0x166C }, // Lo [620] CANADIAN SYLLABICS E..CANADIAN SYLLABICS CARRIER TTSA
    Interval{ 0x166F, 0x167F }, // Lo  [17] CANADIAN SYLLABICS QAI..CANADIAN SYLLABICS BLACKFOOT W
    Interval{ 0x1681, 0x169A }, // Lo  [26] OGHAM LETTER BEITH..OGHAM LETTER PEITH
    Interval{ 0x16A0, 0x16EA }, // Lo  [75] RUNIC LETTER FEHU FEOH FE F..RUNIC LETTER X
    Interval{ 0x16EE, 0x16F0 }, // Nl   [3] RUNIC ARLAUG SYMBOL..RUNIC BELGTHOR SYMBOL
    Interval{ 0x16F1, 0x16F8 }, // Lo   [8] RUNIC LETTER K..RUNIC LETTER FRANKS CASKET AESC
    Interval{ 0x1700, 0x1711 }, // Lo  [18] TAGALOG LETTER A..TAGALOG LETTER HA
    Interval{ 0x1712, 0x1713 }, // Mn   [2] TAGALOG VOWEL SIGN I..TAGALOG VOWEL SIGN U
    Interval{ 0x171F, 0x1731 }, // Lo  [19] TAGALOG LETTER ARCHAIC RA..HANUNOO LETTER HA
    Interval{ 0x1732, 0x1733 }, // Mn   [2] HANUNOO VOWEL SIGN I..HANUNOO VOWEL SIGN U
    Interval{ 0x1740, 0x1751 }, // Lo  [18] BUHID LETTER A..BUHID LETTER HA
    Interval{ 0x1752, 0x1753 }, // Mn   [2] BUHID VOWEL SIGN I..BUHID VOWEL SIGN U
    Interval{ 0x1760, 0x176C }, // Lo  [13] TAGBANWA LETTER A..TAGBANWA LETTER YA
    Interval{ 0x176E, 0x1770 }, // Lo   [3] TAGBANWA LETTER LA..TAGBANWA LETTER SA
    Interval{ 0x1772, 0x1773 }, // Mn   [2] TAGBANWA VOWEL SIGN I..TAGBANWA VOWEL SIGN U
    Interval{ 0x1780, 0x17B3 }, // Lo  [52] KHMER LETTER KA..KHMER INDEPENDENT VOWEL QAU
    Interval{ 0x17B6, 0x17B6 }, // Mc       KHMER VOWEL SIGN AA
    Interval{ 0x17B7, 0x17BD }, // Mn   [7] KHMER VOWEL SIGN I..KHMER VOWEL SIGN UA
    Interval{ 0x17BE, 0x17C5 }, // Mc   [8] KHMER VOWEL SIGN OE..KHMER VOWEL SIGN AU
    Interval{ 0x17C6, 0x17C6 }, // Mn       KHMER SIGN NIKAHIT
    Interval{ 0x17C7, 0x17C8 }, // Mc   [2] KHMER SIGN REAHMUK..KHMER SIGN YUUKALEAPINTU
    Interval{ 0x17D7, 0x17D7 }, // Lm       KHMER SIGN LEK TOO
    Interval{ 0x17DC, 0x17DC }, // Lo       KHMER SIGN AVAKRAHASANYA
    Interval{ 0x1820, 0x1842 }, // Lo  [35] MONGOLIAN LETTER A..MONGOLIAN LETTER CHI
    Interval{ 0x1843, 0x1843 }, // Lm       MONGOLIAN LETTER TODO LONG VOWEL SIGN
    Interval{ 0x1844, 0x1878 }, // Lo  [53] MONGOLIAN LETTER TODO E..MONGOLIAN LETTER CHA WITH TWO DOTS
    Interval{ 0x1880, 0x1884 }, // Lo   [5] MONGOLIAN LETTER ALI GALI ANUSVARA ONE..MONGOLIAN LETTER ALI GALI INVERTED UBADAMA
    Interval{ 0x1885, 0x1886 }, // Mn   [2] MONGOLIAN LETTER ALI GALI BALUDA..MONGOLIAN LETTER ALI GALI THREE BALUDA
    Interval{ 0x1887, 0x18A8 }, // Lo  [34] MONGOLIAN LETTER ALI GALI A..MONGOLIAN LETTER MANCHU ALI GALI BHA
    Interval{ 0x18A9, 0x18A9 }, // Mn       MONGOLIAN LETTER ALI GALI DAGALGA
    Interval{ 0x18AA, 0x18AA }, // Lo       MONGOLIAN LETTER MANCHU ALI GALI LHA
    Interval{ 0x18B0, 0x18F5 }, // Lo  [70] CANADIAN SYLLABICS OY..CANADIAN SYLLABICS CARRIER DENTAL S
    Interval{ 0x1900, 0x191E }, // Lo  [31] LIMBU VOWEL-CARRIER LETTER..LIMBU LETTER TRA
    Interval{ 0x1920, 0x1922 }, // Mn   [3] LIMBU VOWEL SIGN A..LIMBU VOWEL SIGN U
    Interval{ 0x1923, 0x1926 }, // Mc   [4] LIMBU VOWEL SIGN EE..LIMBU VOWEL SIGN AU
    Interval{ 0x1927, 0x1928 }, // Mn   [2] LIMBU VOWEL SIGN E..LIMBU VOWEL SIGN O
    Interval{ 0x1929, 0x192B }, // Mc   [3] LIMBU SUBJOINED LETTER YA..LIMBU SUBJOINED LETTER WA
    Interval{ 0x1930, 0x1931 }, // Mc   [2] LIMBU SMALL LETTER KA..LIMBU SMALL LETTER NGA
    Interval{ 0x1932, 0x1932 }, // Mn       LIMBU SMALL LETTER ANUSVARA
    Interval{ 0x1933, 0x1938 }, // Mc   [6] LIMBU SMALL LETTER TA..LIMBU SMALL LETTER LA
    Interval{ 0x1950, 0x196D }, // Lo  [30] TAI LE LETTER KA..TAI LE LETTER AI
    Interval{ 0x1970, 0x1974 }, // Lo   [5] TAI LE LETTER TONE-2..TAI LE LETTER TONE-6
    Interval{ 0x1980, 0x19AB }, // Lo  [44] NEW TAI LUE LETTER HIGH QA..NEW TAI LUE LETTER LOW SUA
    Interval{ 0x19B0, 0x19C9 }, // Lo  [26] NEW TAI LUE VOWEL SIGN VOWEL SHORTENER..NEW TAI LUE TONE MARK-2
    Interval{ 0x1A00, 0x1A16 }, // Lo  [23] BUGINESE LETTER KA..BUGINESE LETTER HA
    Interval{ 0x1A17, 0x1A18 }, // Mn   [2] BUGINESE VOWEL SIGN I..BUGINESE VOWEL SIGN U
    Interval{ 0x1A19, 0x1A1A }, // Mc   [2] BUGINESE VOWEL SIGN E..BUGINESE VOWEL SIGN O
    Interval{ 0x1A1B, 0x1A1B }, // Mn       BUGINESE VOWEL SIGN AE
    Interval{ 0x1A20, 0x1A54 }, // Lo  [53] TAI THAM LETTER HIGH KA..TAI THAM LETTER GREAT SA
    Interval{ 0x1A55, 0x1A55 }, // Mc       TAI THAM CONSONANT SIGN MEDIAL RA
    Interval{ 0x1A56, 0x1A56 }, // Mn       TAI THAM CONSONANT SIGN MEDIAL LA
    Interval{ 0x1A57, 0x1A57 }, // Mc       TAI THAM CONSONANT SIGN LA TANG LAI
    Interval{ 0x1A58, 0x1A5E }, // Mn   [7] TAI THAM SIGN MAI KANG LAI..TAI THAM CONSONANT SIGN SA
    Interval{ 0x1A61, 0x1A61 }, // Mc       TAI THAM VOWEL SIGN A
    Interval{ 0x1A62, 0x1A62 }, // Mn       TAI THAM VOWEL SIGN MAI SAT
    Interval{ 0x1A63, 0x1A64 }, // Mc   [2] TAI THAM VOWEL SIGN AA..TAI THAM VOWEL SIGN TALL AA
    Interval{ 0x1A65, 0x1A6C }, // Mn   [8] TAI THAM VOWEL SIGN I..TAI THAM VOWEL SIGN OA BELOW
    Interval{ 0x1A6D, 0x1A72 }, // Mc   [6] TAI THAM VOWEL SIGN OY..TAI THAM VOWEL SIGN THAM AI
    Interval{ 0x1A73, 0x1A74 }, // Mn   [2] TAI THAM VOWEL SIGN OA ABOVE..TAI THAM SIGN MAI KANG
    Interval{ 0x1AA7, 0x1AA7 }, // Lm       TAI THAM SIGN MAI YAMOK
    Interval{ 0x1ABF, 0x1AC0 }, // Mn   [2] COMBINING LATIN SMALL LETTER W BELOW..COMBINING LATIN SMALL LETTER TURNED W BELOW
    Interval{ 0x1ACC, 0x1ACE }, // Mn   [3] COMBINING LATIN SMALL LETTER INSULAR G..COMBINING LATIN SMALL LETTER INSULAR T
    Interval{ 0x1B00, 0x1B03 }, // Mn   [4] BALINESE SIGN ULU RICEM..BALINESE SIGN SURANG
    Interval{ 0x1B04, 0x1B04 }, // Mc       BALINESE SIGN BISAH
    Interval{ 0x1B05, 0x1B33 }, // Lo  [47] BALINESE LETTER AKARA..BALINESE LETTER HA
    Interval{ 0x1B35, 0x1B35 }, // Mc       BALINESE VOWEL SIGN TEDUNG
    Interval{ 0x1B36, 0x1B3A }, // Mn   [5] BALINESE VOWEL SIGN ULU..BALINESE VOWEL SIGN RA REPA
    Interval{ 0x1B3B, 0x1B3B }, // Mc       BALINESE VOWEL SIGN RA REPA TEDUNG
    Interval{ 0x1B3C, 0x1B3C }, // Mn       BALINESE VOWEL SIGN LA LENGA
    Interval{ 0x1B3D, 0x1B41 }, // Mc   [5] BALINESE VOWEL SIGN LA LENGA TEDUNG..BALINESE VOWEL SIGN TALING REPA TEDUNG
    Interval{ 0x1B42, 0x1B42 }, // Mn       BALINESE VOWEL SIGN PEPET
    Interval{ 0x1B43, 0x1B43 }, // Mc       BALINESE VOWEL SIGN PEPET TEDUNG
    Interval{ 0x1B45, 0x1B4C }, // Lo   [8] BALINESE LETTER KAF SASAK..BALINESE LETTER ARCHAIC JNYA
    Interval{ 0x1B80, 0x1B81 }, // Mn   [2] SUNDANESE SIGN PANYECEK..SUNDANESE SIGN PANGLAYAR
    Interval{ 0x1B82, 0x1B82 }, // Mc       SUNDANESE SIGN PANGWISAD
    Interval{ 0x1B83, 0x1BA0 }, // Lo  [30] SUNDANESE LETTER A..SUNDANESE LETTER HA
    Interval{ 0x1BA1, 0x1BA1 }, // Mc       SUNDANESE CONSONANT SIGN PAMINGKAL
    Interval{ 0x1BA2, 0x1BA5 }, // Mn   [4] SUNDANESE CONSONANT SIGN PANYAKRA..SUNDANESE VOWEL SIGN PANYUKU
    Interval{ 0x1BA6, 0x1BA7 }, // Mc   [2] SUNDANESE VOWEL SIGN PANAELAENG..SUNDANESE VOWEL SIGN PANOLONG
    Interval{ 0x1BA8, 0x1BA9 }, // Mn   [2] SUNDANESE VOWEL SIGN PAMEPET..SUNDANESE VOWEL SIGN PANEULEUNG
    Interval{ 0x1BAC, 0x1BAD }, // Mn   [2] SUNDANESE CONSONANT SIGN PASANGAN MA..SUNDANESE CONSONANT SIGN PASANGAN WA
    Interval{ 0x1BAE, 0x1BAF }, // Lo   [2] SUNDANESE LETTER KHA..SUNDANESE LETTER SYA
    Interval{ 0x1BBA, 0x1BE5 }, // Lo  [44] SUNDANESE AVAGRAHA..BATAK LETTER U
    Interval{ 0x1BE7, 0x1BE7 }, // Mc       BATAK VOWEL SIGN E
    Interval{ 0x1BE8, 0x1BE9 }, // Mn   [2] BATAK VOWEL SIGN PAKPAK E..BATAK VOWEL SIGN EE
    Interval{ 0x1BEA, 0x1BEC }, // Mc   [3] BATAK VOWEL SIGN I..BATAK VOWEL SIGN O
    Interval{ 0x1BED, 0x1BED }, // Mn       BATAK VOWEL SIGN KARO O
    Interval{ 0x1BEE, 0x1BEE }, // Mc       BATAK VOWEL SIGN U
    Interval{ 0x1BEF, 0x1BF1 }, // Mn   [3] BATAK VOWEL SIGN U FOR SIMALUNGUN SA..BATAK CONSONANT SIGN H
    Interval{ 0x1C00, 0x1C23 }, // Lo  [36] LEPCHA LETTER KA..LEPCHA LETTER A
    Interval{ 0x1C24, 0x1C2B }, // Mc   [8] LEPCHA SUBJOINED LETTER YA..LEPCHA VOWEL SIGN UU
    Interval{ 0x1C2C, 0x1C33 }, // Mn   [8] LEPCHA VOWEL SIGN E..LEPCHA CONSONANT SIGN T
    Interval{ 0x1C34, 0x1C35 }, // Mc   [2] LEPCHA CONSONANT SIGN NYIN-DO..LEPCHA CONSONANT SIGN KANG
    Interval{ 0x1C36, 0x1C36 }, // Mn       LEPCHA SIGN RAN
    Interval{ 0x1C4D, 0x1C4F }, // Lo   [3] LEPCHA LETTER TTA..LEPCHA LETTER DDA
    Interval{ 0x1C5A, 0x1C77 }, // Lo  [30] OL CHIKI LETTER LA..OL CHIKI LETTER OH
    Interval{ 0x1C78, 0x1C7D }, // Lm   [6] OL CHIKI MU TTUDDAG..OL CHIKI AHAD
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1CE9, 0x1CEC }, // Lo   [4] VEDIC SIGN ANUSVARA ANTARGOMUKHA..VEDIC SIGN ANUSVARA VAMAGOMUKHA WITH TAIL
    Interval{ 0x1CEE, 0x1CF3 }, // Lo   [6] VEDIC SIGN HEXIFORM LONG ANUSVARA..VEDIC SIGN ROTATED ARDHAVISARGA
    Interval{ 0x1CF5, 0x1CF6 }, // Lo   [2] VEDIC SIGN JIHVAMULIYA..VEDIC SIGN UPADHMANIYA
    Interval{ 0x1CFA, 0x1CFA }, // Lo       VEDIC SIGN DOUBLE ANUSVARA ANTARGOMUKHA
    Interval{ 0x1D00, 0x1D2B }, // L&  [44] LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTER SMALL CAPITAL EL
    Interval{ 0x1D2C, 0x1D6A }, // Lm  [63] MODIFIER LETTER CAPITAL A..GREEK SUBSCRIPT SMALL LETTER CHI
    Interval{ 0x1D6B, 0x1D77 }, // L&  [13] LATIN SMALL LETTER UE..LATIN SMALL LETTER TURNED G
    Interval{ 0x1D78, 0x1D78 }, // Lm       MODIFIER LETTER CYRILLIC EN
    Interval{ 0x1D79, 0x1D9A }, // L&  [34] LATIN SMALL LETTER INSULAR G..LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
    Interval{ 0x1D9B, 0x1DBF }, // Lm  [37] MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER LETTER SMALL THETA
    Interval{ 0x1DE7, 0x1DF4 }, // Mn  [14] COMBINING LATIN SMALL LETTER ALPHA..COMBINING LATIN SMALL LETTER U WITH DIAERESIS
    Interval{ 0x1E00, 0x1F15 }, // L& [278] LATIN CAPITAL LETTER A WITH RING BELOW..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F45 }, // L&  [38] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F7D }, // L&  [31] GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1FB4 }, // L&  [53] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FBC }, // L&   [7] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FCC }, // L&   [7] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FDB }, // L&   [6] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE0, 0x1FEC }, // L&  [13] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FFC }, // L&   [7] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x2071, 0x2071 }, // Lm       SUPERSCRIPT LATIN SMALL LETTER I
    Interval{ 0x207F, 0x207F }, // Lm       SUPERSCRIPT LATIN SMALL LETTER N
    Interval{ 0x2090, 0x209C }, // Lm  [13] LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCRIPT SMALL LETTER T
    Interval{ 0x2102, 0x2102 }, // L&       DOUBLE-STRUCK CAPITAL C
    Interval{ 0x2107, 0x2107 }, // L&       EULER CONSTANT
    Interval{ 0x210A, 0x2113 }, // L&  [10] SCRIPT SMALL G..SCRIPT SMALL L
    Interval{ 0x2115, 0x2115 }, // L&       DOUBLE-STRUCK CAPITAL N
    Interval{ 0x2119, 0x211D }, // L&   [5] DOUBLE-STRUCK CAPITAL P..DOUBLE-STRUCK CAPITAL R
    Interval{ 0x2124, 0x2124 }, // L&       DOUBLE-STRUCK CAPITAL Z
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x2128, 0x2128 }, // L&       BLACK-LETTER CAPITAL Z
    Interval{ 0x212A, 0x212D }, // L&   [4] KELVIN SIGN..BLACK-LETTER CAPITAL C
    Interval{ 0x212F, 0x2134 }, // L&   [6] SCRIPT SMALL E..SCRIPT SMALL O
    Interval{ 0x2135, 0x2138 }, // Lo   [4] ALEF SYMBOL..DALET SYMBOL
    Interval{ 0x2139, 0x2139 }, // L&       INFORMATION SOURCE
    Interval{ 0x213C, 0x213F }, // L&   [4] DOUBLE-STRUCK SMALL PI..DOUBLE-STRUCK CAPITAL PI
    Interval{ 0x2145, 0x2149 }, // L&   [5] DOUBLE-STRUCK ITALIC CAPITAL D..DOUBLE-STRUCK ITALIC SMALL J
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2160, 0x2182 }, // Nl  [35] ROMAN NUMERAL ONE..ROMAN NUMERAL TEN THOUSAND
    Interval{ 0x2183, 0x2184 }, // L&   [2] ROMAN NUMERAL REVERSED ONE HUNDRED..LATIN SMALL LETTER REVERSED C
    Interval{ 0x2185, 0x2188 }, // Nl   [4] ROMAN NUMERAL SIX LATE FORM..ROMAN NUMERAL ONE HUNDRED THOUSAND
    Interval{ 0x24B6, 0x24E9 }, // So  [52] CIRCLED LATIN CAPITAL LETTER A..CIRCLED LATIN SMALL LETTER Z
    Interval{ 0x2C00, 0x2C7B }, // L& [124] GLAGOLITIC CAPITAL LETTER AZU..LATIN LETTER SMALL CAPITAL TURNED E
    Interval{ 0x2C7C, 0x2C7D }, // Lm   [2] LATIN SUBSCRIPT SMALL LETTER J..MODIFIER LETTER CAPITAL V
    Interval{ 0x2C7E, 0x2CE4 }, // L& [103] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC SYMBOL KAI
    Interval{ 0x2CEB, 0x2CEE }, // L&   [4] COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI..COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF2, 0x2CF3 }, // L&   [2] COPTIC CAPITAL LETTER BOHAIRIC KHEI..COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0x2D30, 0x2D67 }, // Lo  [56] TIFINAGH LETTER YA..TIFINAGH LETTER YO
    Interval{ 0x2D6F, 0x2D6F }, // Lm       TIFINAGH MODIFIER LETTER LABIALIZATION MARK
    Interval{ 0x2D80, 0x2D96 }, // Lo  [23] ETHIOPIC SYLLABLE LOA..ETHIOPIC SYLLABLE GGWE
    Interval{ 0x2DA0, 0x2DA6 }, // Lo   [7] ETHIOPIC SYLLABLE SSA..ETHIOPIC SYLLABLE SSO
    Interval{ 0x2DA8, 0x2DAE }, // Lo   [7] ETHIOPIC SYLLABLE CCA..ETHIOPIC SYLLABLE CCO
    Interval{ 0x2DB0, 0x2DB6 }, // Lo   [7] ETHIOPIC SYLLABLE ZZA..ETHIOPIC SYLLABLE ZZO
    Interval{ 0x2DB8, 0x2DBE }, // Lo   [7] ETHIOPIC SYLLABLE CCHA..ETHIOPIC SYLLABLE CCHO
    Interval{ 0x2DC0, 0x2DC6 }, // Lo   [7] ETHIOPIC SYLLABLE QYA..ETHIOPIC SYLLABLE QYO
    Interval{ 0x2DC8, 0x2DCE }, // Lo   [7] ETHIOPIC SYLLABLE KYA..ETHIOPIC SYLLABLE KYO
    Interval{ 0x2DD0, 0x2DD6 }, // Lo   [7] ETHIOPIC SYLLABLE XYA..ETHIOPIC SYLLABLE XYO
    Interval{ 0x2DD8, 0x2DDE }, // Lo   [7] ETHIOPIC SYLLABLE GYA..ETHIOPIC SYLLABLE GYO
    Interval{ 0x2DE0, 0x2DFF }, // Mn  [32] COMBINING CYRILLIC LETTER BE..COMBINING CYRILLIC LETTER IOTIFIED BIG YUS
    Interval{ 0x2E2F, 0x2E2F }, // Lm       VERTICAL TILDE
    Interval{ 0x3005, 0x3005 }, // Lm       IDEOGRAPHIC ITERATION MARK
    Interval{ 0x3006, 0x3006 }, // Lo       IDEOGRAPHIC CLOSING MARK
    Interval{ 0x3007, 0x3007 }, // Nl       IDEOGRAPHIC NUMBER ZERO
    Interval{ 0x3021, 0x3029 }, // Nl   [9] HANGZHOU NUMERAL ONE..HANGZHOU NUMERAL NINE
    Interval{ 0x3031, 0x3035 }, // Lm   [5] VERTICAL KANA REPEAT MARK..VERTICAL KANA REPEAT MARK LOWER HALF
    Interval{ 0x3038, 0x303A }, // Nl   [3] HANGZHOU NUMERAL TEN..HANGZHOU NUMERAL THIRTY
    Interval{ 0x303B, 0x303B }, // Lm       VERTICAL IDEOGRAPHIC ITERATION MARK
    Interval{ 0x303C, 0x303C }, // Lo       MASU MARK
    Interval{ 0x3041, 0x3096 }, // Lo  [86] HIRAGANA LETTER SMALL A..HIRAGANA LETTER SMALL KE
    Interval{ 0x309D, 0x309E }, // Lm   [2] HIRAGANA ITERATION MARK..HIRAGANA VOICED ITERATION MARK
    Interval{ 0x309F, 0x309F }, // Lo       HIRAGANA DIGRAPH YORI
    Interval{ 0x30A1, 0x30FA }, // Lo  [90] KATAKANA LETTER SMALL A..KATAKANA LETTER VO
    Interval{ 0x30FC, 0x30FE }, // Lm   [3] KATAKANA-HIRAGANA PROLONGED SOUND MARK..KATAKANA VOICED ITERATION MARK
    Interval{ 0x30FF, 0x30FF }, // Lo       KATAKANA DIGRAPH KOTO
    Interval{ 0x3105, 0x312F }, // Lo  [43] BOPOMOFO LETTER B..BOPOMOFO LETTER NN
    Interval{ 0x3131, 0x318E }, // Lo  [94] HANGUL LETTER KIYEOK..HANGUL LETTER ARAEAE
    Interval{ 0x31A0, 0x31BF }, // Lo  [32] BOPOMOFO LETTER BU..BOPOMOFO LETTER AH
    Interval{ 0x31F0, 0x31FF }, // Lo  [16] KATAKANA LETTER SMALL KU..KATAKANA LETTER SMALL RO
    Interval{ 0x3400, 0x4DBF }, // Lo [6592] CJK UNIFIED IDEOGRAPH-3400..CJK UNIFIED IDEOGRAPH-4DBF
    Interval{ 0x4E00, 0xA014 }, // Lo [21013] CJK UNIFIED IDEOGRAPH-4E00..YI SYLLABLE E
    Interval{ 0xA015, 0xA015 }, // Lm       YI SYLLABLE WU
    Interval{ 0xA016, 0xA48C }, // Lo [1143] YI SYLLABLE BIT..YI SYLLABLE YYR
    Interval{ 0xA4D0, 0xA4F7 }, // Lo  [40] LISU LETTER BA..LISU LETTER OE
    Interval{ 0xA4F8, 0xA4FD }, // Lm   [6] LISU LETTER TONE MYA TI..LISU LETTER TONE MYA JEU
    Interval{ 0xA500, 0xA60B }, // Lo [268] VAI SYLLABLE EE..VAI SYLLABLE NG
    Interval{ 0xA60C, 0xA60C }, // Lm       VAI SYLLABLE LENGTHENER
    Interval{ 0xA610, 0xA61F }, // Lo  [16] VAI SYLLABLE NDOLE FA..VAI SYMBOL JONG
    Interval{ 0xA62A, 0xA62B }, // Lo   [2] VAI SYLLABLE NDOLE MA..VAI SYLLABLE NDOLE DO
    Interval{ 0xA640, 0xA66D }, // L&  [46] CYRILLIC CAPITAL LETTER ZEMLYA..CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA66E, 0xA66E }, // Lo       CYRILLIC LETTER MULTIOCULAR O
    Interval{ 0xA674, 0xA67B }, // Mn   [8] COMBINING CYRILLIC LETTER UKRAINIAN IE..COMBINING CYRILLIC LETTER OMEGA
    Interval{ 0xA67F, 0xA67F }, // Lm       CYRILLIC PAYEROK
    Interval{ 0xA680, 0xA69B }, // L&  [28] CYRILLIC CAPITAL LETTER DWE..CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA69C, 0xA69D }, // Lm   [2] MODIFIER LETTER CYRILLIC HARD SIGN..MODIFIER LETTER CYRILLIC SOFT SIGN
    Interval{ 0xA69E, 0xA69F }, // Mn   [2] COMBINING CYRILLIC LETTER EF..COMBINING CYRILLIC LETTER IOTIFIED E
    Interval{ 0xA6A0, 0xA6E5 }, // Lo  [70] BAMUM LETTER A..BAMUM LETTER KI
    Interval{ 0xA6E6, 0xA6EF }, // Nl  [10] BAMUM LETTER MO..BAMUM LETTER KOGHOM
    Interval{ 0xA717, 0xA71F }, // Lm   [9] MODIFIER LETTER DOT VERTICAL BAR..MODIFIER LETTER LOW INVERTED EXCLAMATION MARK
    Interval{ 0xA722, 0xA76F }, // L&  [78] LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF..LATIN SMALL LETTER CON
    Interval{ 0xA770, 0xA770 }, // Lm       MODIFIER LETTER US
    Interval{ 0xA771, 0xA787 }, // L&  [23] LATIN SMALL LETTER DUM..LATIN SMALL LETTER INSULAR T
    Interval{ 0xA788, 0xA788 }, // Lm       MODIFIER LETTER LOW CIRCUMFLEX ACCENT
    Interval{ 0xA78B, 0xA78E }, // L&   [4] LATIN CAPITAL LETTER SALTILLO..LATIN SMALL LETTER L WITH RETROFLEX HOOK AND BELT
    Interval{ 0xA78F, 0xA78F }, // Lo       LATIN LETTER SINOLOGICAL DOT
    Interval{ 0xA790, 0xA7CA }, // L&  [59] LATIN CAPITAL LETTER N WITH DESCENDER..LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D1 }, // L&   [2] LATIN CAPITAL LETTER CLOSED INSULAR G..LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D3, 0xA7D3 }, // L&       LATIN SMALL LETTER DOUBLE THORN
    Interval{ 0xA7D5, 0xA7D9 }, // L&   [5] LATIN SMALL LETTER DOUBLE WYNN..LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F2, 0xA7F4 }, // Lm   [3] MODIFIER LETTER CAPITAL C..MODIFIER LETTER CAPITAL Q
    Interval{ 0xA7F5, 0xA7F6 }, // L&   [2] LATIN CAPITAL LETTER REVERSED HALF H..LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xA7F7, 0xA7F7 }, // Lo       LATIN EPIGRAPHIC LETTER SIDEWAYS I
    Interval{ 0xA7F8, 0xA7F9 }, // Lm   [2] MODIFIER LETTER CAPITAL H WITH STROKE..MODIFIER LETTER SMALL LIGATURE OE
    Interval{ 0xA7FA, 0xA7FA }, // L&       LATIN LETTER SMALL CAPITAL TURNED M
    Interval{ 0xA7FB, 0xA801 }, // Lo   [7] LATIN EPIGRAPHIC LETTER REVERSED F..SYLOTI NAGRI LETTER I
    Interval{ 0xA802, 0xA802 }, // Mn       SYLOTI NAGRI SIGN DVISVARA
    Interval{ 0xA803, 0xA805 }, // Lo   [3] SYLOTI NAGRI LETTER U..SYLOTI NAGRI LETTER O
    Interval{ 0xA807, 0xA80A }, // Lo   [4] SYLOTI NAGRI LETTER KO..SYLOTI NAGRI LETTER GHO
    Interval{ 0xA80B, 0xA80B }, // Mn       SYLOTI NAGRI SIGN ANUSVARA
    Interval{ 0xA80C, 0xA822 }, // Lo  [23] SYLOTI NAGRI LETTER CO..SYLOTI NAGRI LETTER HO
    Interval{ 0xA823, 0xA824 }, // Mc   [2] SYLOTI NAGRI VOWEL SIGN A..SYLOTI NAGRI VOWEL SIGN I
    Interval{ 0xA825, 0xA826 }, // Mn   [2] SYLOTI NAGRI VOWEL SIGN U..SYLOTI NAGRI VOWEL SIGN E
    Interval{ 0xA827, 0xA827 }, // Mc       SYLOTI NAGRI VOWEL SIGN OO
    Interval{ 0xA840, 0xA873 }, // Lo  [52] PHAGS-PA LETTER KA..PHAGS-PA LETTER CANDRABINDU
    Interval{ 0xA880, 0xA881 }, // Mc   [2] SAURASHTRA SIGN ANUSVARA..SAURASHTRA SIGN VISARGA
    Interval{ 0xA882, 0xA8B3 }, // Lo  [50] SAURASHTRA LETTER A..SAURASHTRA LETTER LLA
    Interval{ 0xA8B4, 0xA8C3 }, // Mc  [16] SAURASHTRA CONSONANT SIGN HAARU..SAURASHTRA VOWEL SIGN AU
    Interval{ 0xA8C5, 0xA8C5 }, // Mn       SAURASHTRA SIGN CANDRABINDU
    Interval{ 0xA8F2, 0xA8F7 }, // Lo   [6] DEVANAGARI SIGN SPACING CANDRABINDU..DEVANAGARI SIGN CANDRABINDU AVAGRAHA
    Interval{ 0xA8FB, 0xA8FB }, // Lo       DEVANAGARI HEADSTROKE
    Interval{ 0xA8FD, 0xA8FE }, // Lo   [2] DEVANAGARI JAIN OM..DEVANAGARI LETTER AY
    Interval{ 0xA8FF, 0xA8FF }, // Mn       DEVANAGARI VOWEL SIGN AY
    Interval{ 0xA90A, 0xA925 }, // Lo  [28] KAYAH LI LETTER KA..KAYAH LI LETTER OO
    Interval{ 0xA926, 0xA92A }, // Mn   [5] KAYAH LI VOWEL UE..KAYAH LI VOWEL O
    Interval{ 0xA930, 0xA946 }, // Lo  [23] REJANG LETTER KA..REJANG LETTER A
    Interval{ 0xA947, 0xA951 }, // Mn  [11] REJANG VOWEL SIGN I..REJANG CONSONANT SIGN R
    Interval{ 0xA952, 0xA952 }, // Mc       REJANG CONSONANT SIGN H
    Interval{ 0xA960, 0xA97C }, // Lo  [29] HANGUL CHOSEONG TIKEUT-MIEUM..HANGUL CHOSEONG SSANGYEORINHIEUH
    Interval{ 0xA980, 0xA982 }, // Mn   [3] JAVANESE SIGN PANYANGGA..JAVANESE SIGN LAYAR
    Interval{ 0xA983, 0xA983 }, // Mc       JAVANESE SIGN WIGNYAN
    Interval{ 0xA984, 0xA9B2 }, // Lo  [47] JAVANESE LETTER A..JAVANESE LETTER HA
    Interval{ 0xA9B4, 0xA9B5 }, // Mc   [2] JAVANESE VOWEL SIGN TARUNG..JAVANESE VOWEL SIGN TOLONG
    Interval{ 0xA9B6, 0xA9B9 }, // Mn   [4] JAVANESE VOWEL SIGN WULU..JAVANESE VOWEL SIGN SUKU MENDUT
    Interval{ 0xA9BA, 0xA9BB }, // Mc   [2] JAVANESE VOWEL SIGN TALING..JAVANESE VOWEL SIGN DIRGA MURE
    Interval{ 0xA9BC, 0xA9BD }, // Mn   [2] JAVANESE VOWEL SIGN PEPET..JAVANESE CONSONANT SIGN KERET
    Interval{ 0xA9BE, 0xA9BF }, // Mc   [2] JAVANESE CONSONANT SIGN PENGKAL..JAVANESE CONSONANT SIGN CAKRA
    Interval{ 0xA9CF, 0xA9CF }, // Lm       JAVANESE PANGRANGKEP
    Interval{ 0xA9E0, 0xA9E4 }, // Lo   [5] MYANMAR LETTER SHAN GHA..MYANMAR LETTER SHAN BHA
    Interval{ 0xA9E5, 0xA9E5 }, // Mn       MYANMAR SIGN SHAN SAW
    Interval{ 0xA9E6, 0xA9E6 }, // Lm       MYANMAR MODIFIER LETTER SHAN REDUPLICATION
    Interval{ 0xA9E7, 0xA9EF }, // Lo   [9] MYANMAR LETTER TAI LAING NYA..MYANMAR LETTER TAI LAING NNA
    Interval{ 0xA9FA, 0xA9FE }, // Lo   [5] MYANMAR LETTER TAI LAING LLA..MYANMAR LETTER TAI LAING BHA
    Interval{ 0xAA00, 0xAA28 }, // Lo  [41] CHAM LETTER A..CHAM LETTER HA
    Interval{ 0xAA29, 0xAA2E }, // Mn   [6] CHAM VOWEL SIGN AA..CHAM VOWEL SIGN OE
    Interval{ 0xAA2F, 0xAA30 }, // Mc   [2] CHAM VOWEL SIGN O..CHAM VOWEL SIGN AI
    Interval{ 0xAA31, 0xAA32 }, // Mn   [2] CHAM VOWEL SIGN AU..CHAM VOWEL SIGN UE
    Interval{ 0xAA33, 0xAA34 }, // Mc   [2] CHAM CONSONANT SIGN YA..CHAM CONSONANT SIGN RA
    Interval{ 0xAA35, 0xAA36 }, // Mn   [2] CHAM CONSONANT SIGN LA..CHAM CONSONANT SIGN WA
    Interval{ 0xAA40, 0xAA42 }, // Lo   [3] CHAM LETTER FINAL K..CHAM LETTER FINAL NG
    Interval{ 0xAA43, 0xAA43 }, // Mn       CHAM CONSONANT SIGN FINAL NG
    Interval{ 0xAA44, 0xAA4B }, // Lo   [8] CHAM LETTER FINAL CH..CHAM LETTER FINAL SS
    Interval{ 0xAA4C, 0xAA4C }, // Mn       CHAM CONSONANT SIGN FINAL M
    Interval{ 0xAA4D, 0xAA4D }, // Mc       CHAM CONSONANT SIGN FINAL H
    Interval{ 0xAA60, 0xAA6F }, // Lo  [16] MYANMAR LETTER KHAMTI GA..MYANMAR LETTER KHAMTI FA
    Interval{ 0xAA70, 0xAA70 }, // Lm       MYANMAR MODIFIER LETTER KHAMTI REDUPLICATION
    Interval{ 0xAA71, 0xAA76 }, // Lo   [6] MYANMAR LETTER KHAMTI XA..MYANMAR LOGOGRAM KHAMTI HM
    Interval{ 0xAA7A, 0xAA7A }, // Lo       MYANMAR LETTER AITON RA
    Interval{ 0xAA7B, 0xAA7B }, // Mc       MYANMAR SIGN PAO KAREN TONE
    Interval{ 0xAA7C, 0xAA7C }, // Mn       MYANMAR SIGN TAI LAING TONE-2
    Interval{ 0xAA7D, 0xAA7D }, // Mc       MYANMAR SIGN TAI LAING TONE-5
    Interval{ 0xAA7E, 0xAAAF }, // Lo  [50] MYANMAR LETTER SHWE PALAUNG CHA..TAI VIET LETTER HIGH O
    Interval{ 0xAAB0, 0xAAB0 }, // Mn       TAI VIET MAI KANG
    Interval{ 0xAAB1, 0xAAB1 }, // Lo       TAI VIET VOWEL AA
    Interval{ 0xAAB2, 0xAAB4 }, // Mn   [3] TAI VIET VOWEL I..TAI VIET VOWEL U
    Interval{ 0xAAB5, 0xAAB6 }, // Lo   [2] TAI VIET VOWEL E..TAI VIET VOWEL O
    Interval{ 0xAAB7, 0xAAB8 }, // Mn   [2] TAI VIET MAI KHIT..TAI VIET VOWEL IA
    Interval{ 0xAAB9, 0xAABD }, // Lo   [5] TAI VIET VOWEL UEA..TAI VIET VOWEL AN
    Interval{ 0xAABE, 0xAABE }, // Mn       TAI VIET VOWEL AM
    Interval{ 0xAAC0, 0xAAC0 }, // Lo       TAI VIET TONE MAI NUENG
    Interval{ 0xAAC2, 0xAAC2 }, // Lo       TAI VIET TONE MAI SONG
    Interval{ 0xAADB, 0xAADC }, // Lo   [2] TAI VIET SYMBOL KON..TAI VIET SYMBOL NUENG
    Interval{ 0xAADD, 0xAADD }, // Lm       TAI VIET SYMBOL SAM
    Interval{ 0xAAE0, 0xAAEA }, // Lo  [11] MEETEI MAYEK LETTER E..MEETEI MAYEK LETTER SSA
    Interval{ 0xAAEB, 0xAAEB }, // Mc       MEETEI MAYEK VOWEL SIGN II
    Interval{ 0xAAEC, 0xAAED }, // Mn   [2] MEETEI MAYEK VOWEL SIGN UU..MEETEI MAYEK VOWEL SIGN AAI
    Interval{ 0xAAEE, 0xAAEF }, // Mc   [2] MEETEI MAYEK VOWEL SIGN AU..MEETEI MAYEK VOWEL SIGN AAU
    Interval{ 0xAAF2, 0xAAF2 }, // Lo       MEETEI MAYEK ANJI
    Interval{ 0xAAF3, 0xAAF4 }, // Lm   [2] MEETEI MAYEK SYLLABLE REPETITION MARK..MEETEI MAYEK WORD REPETITION MARK
    Interval{ 0xAAF5, 0xAAF5 }, // Mc       MEETEI MAYEK VOWEL SIGN VISARGA
    Interval{ 0xAB01, 0xAB06 }, // Lo   [6] ETHIOPIC SYLLABLE TTHU..ETHIOPIC SYLLABLE TTHO
    Interval{ 0xAB09, 0xAB0E }, // Lo   [6] ETHIOPIC SYLLABLE DDHU..ETHIOPIC SYLLABLE DDHO
    Interval{ 0xAB11, 0xAB16 }, // Lo   [6] ETHIOPIC SYLLABLE DZU..ETHIOPIC SYLLABLE DZO
    Interval{ 0xAB20, 0xAB26 }, // Lo   [7] ETHIOPIC SYLLABLE CCHHA..ETHIOPIC SYLLABLE CCHHO
    Interval{ 0xAB28, 0xAB2E }, // Lo   [7] ETHIOPIC SYLLABLE BBA..ETHIOPIC SYLLABLE BBO
    Interval{ 0xAB30, 0xAB5A }, // L&  [43] LATIN SMALL LETTER BARRED ALPHA..LATIN SMALL LETTER Y WITH SHORT RIGHT LEG
    Interval{ 0xAB5C, 0xAB5F }, // Lm   [4] MODIFIER LETTER SMALL HENG..MODIFIER LETTER SMALL U WITH LEFT HOOK
    Interval{ 0xAB60, 0xAB68 }, // L&   [9] LATIN SMALL LETTER SAKHA YAT..LATIN SMALL LETTER TURNED R WITH MIDDLE TILDE
    Interval{ 0xAB69, 0xAB69 }, // Lm       MODIFIER LETTER SMALL TURNED W
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xABC0, 0xABE2 }, // Lo  [35] MEETEI MAYEK LETTER KOK..MEETEI MAYEK LETTER I LONSUM
    Interval{ 0xABE3, 0xABE4 }, // Mc   [2] MEETEI MAYEK VOWEL SIGN ONAP..MEETEI MAYEK VOWEL SIGN INAP
    Interval{ 0xABE5, 0xABE5 }, // Mn       MEETEI MAYEK VOWEL SIGN ANAP
    Interval{ 0xABE6, 0xABE7 }, // Mc   [2] MEETEI MAYEK VOWEL SIGN YENAP..MEETEI MAYEK VOWEL SIGN SOUNAP
    Interval{ 0xABE8, 0xABE8 }, // Mn       MEETEI MAYEK VOWEL SIGN UNAP
    Interval{ 0xABE9, 0xABEA }, // Mc   [2] MEETEI MAYEK VOWEL SIGN CHEINAP..MEETEI MAYEK VOWEL SIGN NUNG
    Interval{ 0xAC00, 0xD7A3 }, // Lo [11172] HANGUL SYLLABLE GA..HANGUL SYLLABLE HIH
    Interval{ 0xD7B0, 0xD7C6 }, // Lo  [23] HANGUL JUNGSEONG O-YEO..HANGUL JUNGSEONG ARAEA-E
    Interval{ 0xD7CB, 0xD7FB }, // Lo  [49] HANGUL JONGSEONG NIEUN-RIEUL..HANGUL JONGSEONG PHIEUPH-THIEUTH
    Interval{ 0xF900, 0xFA6D }, // Lo [366] CJK COMPATIBILITY IDEOGRAPH-F900..CJK COMPATIBILITY IDEOGRAPH-FA6D
    Interval{ 0xFA70, 0xFAD9 }, // Lo [106] CJK COMPATIBILITY IDEOGRAPH-FA70..CJK COMPATIBILITY IDEOGRAPH-FAD9
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFB1D, 0xFB1D }, // Lo       HEBREW LETTER YOD WITH HIRIQ
    Interval{ 0xFB1E, 0xFB1E }, // Mn       HEBREW POINT JUDEO-SPANISH VARIKA
    Interval{ 0xFB1F, 0xFB28 }, // Lo  [10] HEBREW LIGATURE YIDDISH YOD YOD PATAH..HEBREW LETTER WIDE TAV
    Interval{ 0xFB2A, 0xFB36 }, // Lo  [13] HEBREW LETTER SHIN WITH SHIN DOT..HEBREW LETTER ZAYIN WITH DAGESH
    Interval{ 0xFB38, 0xFB3C }, // Lo   [5] HEBREW LETTER TET WITH DAGESH..HEBREW LETTER LAMED WITH DAGESH
    Interval{ 0xFB3E, 0xFB3E }, // Lo       HEBREW LETTER MEM WITH DAGESH
    Interval{ 0xFB40, 0xFB41 }, // Lo   [2] HEBREW LETTER NUN WITH DAGESH..HEBREW LETTER SAMEKH WITH DAGESH
    Interval{ 0xFB43, 0xFB44 }, // Lo   [2] HEBREW LETTER FINAL PE WITH DAGESH..HEBREW LETTER PE WITH DAGESH
    Interval{ 0xFB46, 0xFBB1 }, // Lo [108] HEBREW LETTER TSADI WITH DAGESH..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
    Interval{ 0xFBD3, 0xFD3D }, // Lo [363] ARABIC LETTER NG ISOLATED FORM..ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
    Interval{ 0xFD50, 0xFD8F }, // Lo  [64] ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM..ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
    Interval{ 0xFD92, 0xFDC7 }, // Lo  [54] ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM..ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
    Interval{ 0xFDF0, 0xFDFB }, // Lo  [12] ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM..ARABIC LIGATURE JALLAJALALOUHOU
    Interval{ 0xFE70, 0xFE74 }, // Lo   [5] ARABIC FATHATAN ISOLATED FORM..ARABIC KASRATAN ISOLATED FORM
    Interval{ 0xFE76, 0xFEFC }, // Lo [135] ARABIC FATHA ISOLATED FORM..ARABIC LIGATURE LAM WITH ALEF FINAL FORM
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0xFF66, 0xFF6F }, // Lo  [10] HALFWIDTH KATAKANA LETTER WO..HALFWIDTH KATAKANA LETTER SMALL TU
    Interval{ 0xFF70, 0xFF70 }, // Lm       HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
    Interval{ 0xFF71, 0xFF9D }, // Lo  [45] HALFWIDTH KATAKANA LETTER A..HALFWIDTH KATAKANA LETTER N
    Interval{ 0xFF9E, 0xFF9F }, // Lm   [2] HALFWIDTH KATAKANA VOICED SOUND MARK..HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
    Interval{ 0xFFA0, 0xFFBE }, // Lo  [31] HALFWIDTH HANGUL FILLER..HALFWIDTH HANGUL LETTER HIEUH
    Interval{ 0xFFC2, 0xFFC7 }, // Lo   [6] HALFWIDTH HANGUL LETTER A..HALFWIDTH HANGUL LETTER E
    Interval{ 0xFFCA, 0xFFCF }, // Lo   [6] HALFWIDTH HANGUL LETTER YEO..HALFWIDTH HANGUL LETTER OE
    Interval{ 0xFFD2, 0xFFD7 }, // Lo   [6] HALFWIDTH HANGUL LETTER YO..HALFWIDTH HANGUL LETTER YU
    Interval{ 0xFFDA, 0xFFDC }, // Lo   [3] HALFWIDTH HANGUL LETTER EU..HALFWIDTH HANGUL LETTER I
    Interval{ 0x10000, 0x1000B }, // Lo  [12] LINEAR B SYLLABLE B008 A..LINEAR B SYLLABLE B046 JE
    Interval{ 0x1000D, 0x10026 }, // Lo  [26] LINEAR B SYLLABLE B036 JO..LINEAR B SYLLABLE B032 QO
    Interval{ 0x10028, 0x1003A }, // Lo  [19] LINEAR B SYLLABLE B060 RA..LINEAR B SYLLABLE B042 WO
    Interval{ 0x1003C, 0x1003D }, // Lo   [2] LINEAR B SYLLABLE B017 ZA..LINEAR B SYLLABLE B074 ZE
    Interval{ 0x1003F, 0x1004D }, // Lo  [15] LINEAR B SYLLABLE B020 ZO..LINEAR B SYLLABLE B091 TWO
    Interval{ 0x10050, 0x1005D }, // Lo  [14] LINEAR B SYMBOL B018..LINEAR B SYMBOL B089
    Interval{ 0x10080, 0x100FA }, // Lo [123] LINEAR B IDEOGRAM B100 MAN..LINEAR B IDEOGRAM VESSEL B305
    Interval{ 0x10140, 0x10174 }, // Nl  [53] GREEK ACROPHONIC ATTIC ONE QUARTER..GREEK ACROPHONIC STRATIAN FIFTY MNAS
    Interval{ 0x10280, 0x1029C }, // Lo  [29] LYCIAN LETTER A..LYCIAN LETTER X
    Interval{ 0x102A0, 0x102D0 }, // Lo  [49] CARIAN LETTER A..CARIAN LETTER UUU3
    Interval{ 0x10300, 0x1031F }, // Lo  [32] OLD ITALIC LETTER A..OLD ITALIC LETTER ESS
    Interval{ 0x1032D, 0x10340 }, // Lo  [20] OLD ITALIC LETTER YE..GOTHIC LETTER PAIRTHRA
    Interval{ 0x10341, 0x10341 }, // Nl       GOTHIC LETTER NINETY
    Interval{ 0x10342, 0x10349 }, // Lo   [8] GOTHIC LETTER RAIDA..GOTHIC LETTER OTHAL
    Interval{ 0x1034A, 0x1034A }, // Nl       GOTHIC LETTER NINE HUNDRED
    Interval{ 0x10350, 0x10375 }, // Lo  [38] OLD PERMIC LETTER AN..OLD PERMIC LETTER IA
    Interval{ 0x10376, 0x1037A }, // Mn   [5] COMBINING OLD PERMIC LETTER AN..COMBINING OLD PERMIC LETTER SII
    Interval{ 0x10380, 0x1039D }, // Lo  [30] UGARITIC LETTER ALPA..UGARITIC LETTER SSU
    Interval{ 0x103A0, 0x103C3 }, // Lo  [36] OLD PERSIAN SIGN A..OLD PERSIAN SIGN HA
    Interval{ 0x103C8, 0x103CF }, // Lo   [8] OLD PERSIAN SIGN AURAMAZDAA..OLD PERSIAN SIGN BUUMISH
    Interval{ 0x103D1, 0x103D5 }, // Nl   [5] OLD PERSIAN NUMBER ONE..OLD PERSIAN NUMBER HUNDRED
    Interval{ 0x10400, 0x1044F }, // L&  [80] DESERET CAPITAL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x10450, 0x1049D }, // Lo  [78] SHAVIAN LETTER PEEP..OSMANYA LETTER OO
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10500, 0x10527 }, // Lo  [40] ELBASAN LETTER A..ELBASAN LETTER KHE
    Interval{ 0x10530, 0x10563 }, // Lo  [52] CAUCASIAN ALBANIAN LETTER ALT..CAUCASIAN ALBANIAN LETTER KIW
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10600, 0x10736 }, // Lo [311] LINEAR A SIGN AB001..LINEAR A SIGN A664
    Interval{ 0x10740, 0x10755 }, // Lo  [22] LINEAR A SIGN A701 A..LINEAR A SIGN A732 JE
    Interval{ 0x10760, 0x10767 }, // Lo   [8] LINEAR A SIGN A800..LINEAR A SIGN A807
    Interval{ 0x10780, 0x10785 }, // Lm   [6] MODIFIER LETTER SMALL CAPITAL AA..MODIFIER LETTER SMALL B WITH HOOK
    Interval{ 0x10787, 0x107B0 }, // Lm  [42] MODIFIER LETTER SMALL DZ DIGRAPH..MODIFIER LETTER SMALL V WITH RIGHT HOOK
    Interval{ 0x107B2, 0x107BA }, // Lm   [9] MODIFIER LETTER SMALL CAPITAL Y..MODIFIER LETTER SMALL S WITH CURL
    Interval{ 0x10800, 0x10805 }, // Lo   [6] CYPRIOT SYLLABLE A..CYPRIOT SYLLABLE JA
    Interval{ 0x10808, 0x10808 }, // Lo       CYPRIOT SYLLABLE JO
    Interval{ 0x1080A, 0x10835 }, // Lo  [44] CYPRIOT SYLLABLE KA..CYPRIOT SYLLABLE WO
    Interval{ 0x10837, 0x10838 }, // Lo   [2] CYPRIOT SYLLABLE XA..CYPRIOT SYLLABLE XE
    Interval{ 0x1083C, 0x1083C }, // Lo       CYPRIOT SYLLABLE ZA
    Interval{ 0x1083F, 0x10855 }, // Lo  [23] CYPRIOT SYLLABLE ZO..IMPERIAL ARAMAIC LETTER TAW
    Interval{ 0x10860, 0x10876 }, // Lo  [23] PALMYRENE LETTER ALEPH..PALMYRENE LETTER TAW
    Interval{ 0x10880, 0x1089E }, // Lo  [31] NABATAEAN LETTER FINAL ALEPH..NABATAEAN LETTER TAW
    Interval{ 0x108E0, 0x108F2 }, // Lo  [19] HATRAN LETTER ALEPH..HATRAN LETTER QOPH
    Interval{ 0x108F4, 0x108F5 }, // Lo   [2] HATRAN LETTER SHIN..HATRAN LETTER TAW
    Interval{ 0x10900, 0x10915 }, // Lo  [22] PHOENICIAN LETTER ALF..PHOENICIAN LETTER TAU
    Interval{ 0x10920, 0x10939 }, // Lo  [26] LYDIAN LETTER A..LYDIAN LETTER C
    Interval{ 0x10980, 0x109B7 }, // Lo  [56] MEROITIC HIEROGLYPHIC LETTER A..MEROITIC CURSIVE LETTER DA
    Interval{ 0x109BE, 0x109BF }, // Lo   [2] MEROITIC CURSIVE LOGOGRAM RMT..MEROITIC CURSIVE LOGOGRAM IMN
    Interval{ 0x10A00, 0x10A00 }, // Lo       KHAROSHTHI LETTER A
    Interval{ 0x10A01, 0x10A03 }, // Mn   [3] KHAROSHTHI VOWEL SIGN I..KHAROSHTHI VOWEL SIGN VOCALIC R
    Interval{ 0x10A05, 0x10A06 }, // Mn   [2] KHAROSHTHI VOWEL SIGN E..KHAROSHTHI VOWEL SIGN O
    Interval{ 0x10A0C, 0x10A0F }, // Mn   [4] KHAROSHTHI VOWEL LENGTH MARK..KHAROSHTHI SIGN VISARGA
    Interval{ 0x10A10, 0x10A13 }, // Lo   [4] KHAROSHTHI LETTER KA..KHAROSHTHI LETTER GHA
    Interval{ 0x10A15, 0x10A17 }, // Lo   [3] KHAROSHTHI LETTER CA..KHAROSHTHI LETTER JA
    Interval{ 0x10A19, 0x10A35 }, // Lo  [29] KHAROSHTHI LETTER NYA..KHAROSHTHI LETTER VHA
    Interval{ 0x10A60, 0x10A7C }, // Lo  [29] OLD SOUTH ARABIAN LETTER HE..OLD SOUTH ARABIAN LETTER THETH
    Interval{ 0x10A80, 0x10A9C }, // Lo  [29] OLD NORTH ARABIAN LETTER HEH..OLD NORTH ARABIAN LETTER ZAH
    Interval{ 0x10AC0, 0x10AC7 }, // Lo   [8] MANICHAEAN LETTER ALEPH..MANICHAEAN LETTER WAW
    Interval{ 0x10AC9, 0x10AE4 }, // Lo  [28] MANICHAEAN LETTER ZAYIN..MANICHAEAN LETTER TAW
    Interval{ 0x10B00, 0x10B35 }, // Lo  [54] AVESTAN LETTER A..AVESTAN LETTER HE
    Interval{ 0x10B40, 0x10B55 }, // Lo  [22] INSCRIPTIONAL PARTHIAN LETTER ALEPH..INSCRIPTIONAL PARTHIAN LETTER TAW
    Interval{ 0x10B60, 0x10B72 }, // Lo  [19] INSCRIPTIONAL PAHLAVI LETTER ALEPH..INSCRIPTIONAL PAHLAVI LETTER TAW
    Interval{ 0x10B80, 0x10B91 }, // Lo  [18] PSALTER PAHLAVI LETTER ALEPH..PSALTER PAHLAVI LETTER TAW
    Interval{ 0x10C00, 0x10C48 }, // Lo  [73] OLD TURKIC LETTER ORKHON A..OLD TURKIC LETTER ORKHON BASH
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x10D00, 0x10D23 }, // Lo  [36] HANIFI ROHINGYA LETTER A..HANIFI ROHINGYA MARK NA KHONNA
    Interval{ 0x10D24, 0x10D27 }, // Mn   [4] HANIFI ROHINGYA SIGN HARBAHAY..HANIFI ROHINGYA SIGN TASSI
    Interval{ 0x10E80, 0x10EA9 }, // Lo  [42] YEZIDI LETTER ELIF..YEZIDI LETTER ET
    Interval{ 0x10EAB, 0x10EAC }, // Mn   [2] YEZIDI COMBINING HAMZA MARK..YEZIDI COMBINING MADDA MARK
    Interval{ 0x10EB0, 0x10EB1 }, // Lo   [2] YEZIDI LETTER LAM WITH DOT ABOVE..YEZIDI LETTER YOT WITH CIRCUMFLEX ABOVE
    Interval{ 0x10F00, 0x10F1C }, // Lo  [29] OLD SOGDIAN LETTER ALEPH..OLD SOGDIAN LETTER FINAL TAW WITH VERTICAL TAIL
    Interval{ 0x10F27, 0x10F27 }, // Lo       OLD SOGDIAN LIGATURE AYIN-DALETH
    Interval{ 0x10F30, 0x10F45 }, // Lo  [22] SOGDIAN LETTER ALEPH..SOGDIAN INDEPENDENT SHIN
    Interval{ 0x10F70, 0x10F81 }, // Lo  [18] OLD UYGHUR LETTER ALEPH..OLD UYGHUR LETTER LESH
    Interval{ 0x10FB0, 0x10FC4 }, // Lo  [21] CHORASMIAN LETTER ALEPH..CHORASMIAN LETTER TAW
    Interval{ 0x10FE0, 0x10FF6 }, // Lo  [23] ELYMAIC LETTER ALEPH..ELYMAIC LIGATURE ZAYIN-YODH
    Interval{ 0x11000, 0x11000 }, // Mc       BRAHMI SIGN CANDRABINDU
    Interval{ 0x11001, 0x11001 }, // Mn       BRAHMI SIGN ANUSVARA
    Interval{ 0x11002, 0x11002 }, // Mc       BRAHMI SIGN VISARGA
    Interval{ 0x11003, 0x11037 }, // Lo  [53] BRAHMI SIGN JIHVAMULIYA..BRAHMI LETTER OLD TAMIL NNNA
    Interval{ 0x11038, 0x11045 }, // Mn  [14] BRAHMI VOWEL SIGN AA..BRAHMI VOWEL SIGN AU
    Interval{ 0x11071, 0x11072 }, // Lo   [2] BRAHMI LETTER OLD TAMIL SHORT E..BRAHMI LETTER OLD TAMIL SHORT O
    Interval{ 0x11073, 0x11074 }, // Mn   [2] BRAHMI VOWEL SIGN OLD TAMIL SHORT E..BRAHMI VOWEL SIGN OLD TAMIL SHORT O
    Interval{ 0x11075, 0x11075 }, // Lo       BRAHMI LETTER OLD TAMIL LLA
    Interval{ 0x11080, 0x11081 }, // Mn   [2] KAITHI SIGN CANDRABINDU..KAITHI SIGN ANUSVARA
    Interval{ 0x11082, 0x11082 }, // Mc       KAITHI SIGN VISARGA
    Interval{ 0x11083, 0x110AF }, // Lo  [45] KAITHI LETTER A..KAITHI LETTER HA
    Interval{ 0x110B0, 0x110B2 }, // Mc   [3] KAITHI VOWEL SIGN AA..KAITHI VOWEL SIGN II
    Interval{ 0x110B3, 0x110B6 }, // Mn   [4] KAITHI VOWEL SIGN U..KAITHI VOWEL SIGN AI
    Interval{ 0x110B7, 0x110B8 }, // Mc   [2] KAITHI VOWEL SIGN O..KAITHI VOWEL SIGN AU
    Interval{ 0x110C2, 0x110C2 }, // Mn       KAITHI VOWEL SIGN VOCALIC R
    Interval{ 0x110D0, 0x110E8 }, // Lo  [25] SORA SOMPENG LETTER SAH..SORA SOMPENG LETTER MAE
    Interval{ 0x11100, 0x11102 }, // Mn   [3] CHAKMA SIGN CANDRABINDU..CHAKMA SIGN VISARGA
    Interval{ 0x11103, 0x11126 }, // Lo  [36] CHAKMA LETTER AA..CHAKMA LETTER HAA
    Interval{ 0x11127, 0x1112B }, // Mn   [5] CHAKMA VOWEL SIGN A..CHAKMA VOWEL SIGN UU
    Interval{ 0x1112C, 0x1112C }, // Mc       CHAKMA VOWEL SIGN E
    Interval{ 0x1112D, 0x11132 }, // Mn   [6] CHAKMA VOWEL SIGN AI..CHAKMA AU MARK
    Interval{ 0x11144, 0x11144 }, // Lo       CHAKMA LETTER LHAA
    Interval{ 0x11145, 0x11146 }, // Mc   [2] CHAKMA VOWEL SIGN AA..CHAKMA VOWEL SIGN EI
    Interval{ 0x11147, 0x11147 }, // Lo       CHAKMA LETTER VAA
    Interval{ 0x11150, 0x11172 }, // Lo  [35] MAHAJANI LETTER A..MAHAJANI LETTER RRA
    Interval{ 0x11176, 0x11176 }, // Lo       MAHAJANI LIGATURE SHRI
    Interval{ 0x11180, 0x11181 }, // Mn   [2] SHARADA SIGN CANDRABINDU..SHARADA SIGN ANUSVARA
    Interval{ 0x11182, 0x11182 }, // Mc       SHARADA SIGN VISARGA
    Interval{ 0x11183, 0x111B2 }, // Lo  [48] SHARADA LETTER A..SHARADA LETTER HA
    Interval{ 0x111B3, 0x111B5 }, // Mc   [3] SHARADA VOWEL SIGN AA..SHARADA VOWEL SIGN II
    Interval{ 0x111B6, 0x111BE }, // Mn   [9] SHARADA VOWEL SIGN U..SHARADA VOWEL SIGN O
    Interval{ 0x111BF, 0x111BF }, // Mc       SHARADA VOWEL SIGN AU
    Interval{ 0x111C1, 0x111C4 }, // Lo   [4] SHARADA SIGN AVAGRAHA..SHARADA OM
    Interval{ 0x111CE, 0x111CE }, // Mc       SHARADA VOWEL SIGN PRISHTHAMATRA E
    Interval{ 0x111CF, 0x111CF }, // Mn       SHARADA SIGN INVERTED CANDRABINDU
    Interval{ 0x111DA, 0x111DA }, // Lo       SHARADA EKAM
    Interval{ 0x111DC, 0x111DC }, // Lo       SHARADA HEADSTROKE
    Interval{ 0x11200, 0x11211 }, // Lo  [18] KHOJKI LETTER A..KHOJKI LETTER JJA
    Interval{ 0x11213, 0x1122B }, // Lo  [25] KHOJKI LETTER NYA..KHOJKI LETTER LLA
    Interval{ 0x1122C, 0x1122E }, // Mc   [3] KHOJKI VOWEL SIGN AA..KHOJKI VOWEL SIGN II
    Interval{ 0x1122F, 0x11231 }, // Mn   [3] KHOJKI VOWEL SIGN U..KHOJKI VOWEL SIGN AI
    Interval{ 0x11232, 0x11233 }, // Mc   [2] KHOJKI VOWEL SIGN O..KHOJKI VOWEL SIGN AU
    Interval{ 0x11234, 0x11234 }, // Mn       KHOJKI SIGN ANUSVARA
    Interval{ 0x11237, 0x11237 }, // Mn       KHOJKI SIGN SHADDA
    Interval{ 0x1123E, 0x1123E }, // Mn       KHOJKI SIGN SUKUN
    Interval{ 0x1123F, 0x11240 }, // Lo   [2] KHOJKI LETTER QA..KHOJKI LETTER SHORT I
    Interval{ 0x11241, 0x11241 }, // Mn       KHOJKI VOWEL SIGN VOCALIC R
    Interval{ 0x11280, 0x11286 }, // Lo   [7] MULTANI LETTER A..MULTANI LETTER GA
    Interval{ 0x11288, 0x11288 }, // Lo       MULTANI LETTER GHA
    Interval{ 0x1128A, 0x1128D }, // Lo   [4] MULTANI LETTER CA..MULTANI LETTER JJA
    Interval{ 0x1128F, 0x1129D }, // Lo  [15] MULTANI LETTER NYA..MULTANI LETTER BA
    Interval{ 0x1129F, 0x112A8 }, // Lo  [10] MULTANI LETTER BHA..MULTANI LETTER RHA
    Interval{ 0x112B0, 0x112DE }, // Lo  [47] KHUDAWADI LETTER A..KHUDAWADI LETTER HA
    Interval{ 0x112DF, 0x112DF }, // Mn       KHUDAWADI SIGN ANUSVARA
    Interval{ 0x112E0, 0x112E2 }, // Mc   [3] KHUDAWADI VOWEL SIGN AA..KHUDAWADI VOWEL SIGN II
    Interval{ 0x112E3, 0x112E8 }, // Mn   [6] KHUDAWADI VOWEL SIGN U..KHUDAWADI VOWEL SIGN AU
    Interval{ 0x11300, 0x11301 }, // Mn   [2] GRANTHA SIGN COMBINING ANUSVARA ABOVE..GRANTHA SIGN CANDRABINDU
    Interval{ 0x11302, 0x11303 }, // Mc   [2] GRANTHA SIGN ANUSVARA..GRANTHA SIGN VISARGA
    Interval{ 0x11305, 0x1130C }, // Lo   [8] GRANTHA LETTER A..GRANTHA LETTER VOCALIC L
    Interval{ 0x1130F, 0x11310 }, // Lo   [2] GRANTHA LETTER EE..GRANTHA LETTER AI
    Interval{ 0x11313, 0x11328 }, // Lo  [22] GRANTHA LETTER OO..GRANTHA LETTER NA
    Interval{ 0x1132A, 0x11330 }, // Lo   [7] GRANTHA LETTER PA..GRANTHA LETTER RA
    Interval{ 0x11332, 0x11333 }, // Lo   [2] GRANTHA LETTER LA..GRANTHA LETTER LLA
    Interval{ 0x11335, 0x11339 }, // Lo   [5] GRANTHA LETTER VA..GRANTHA LETTER HA
    Interval{ 0x1133D, 0x1133D }, // Lo       GRANTHA SIGN AVAGRAHA
    Interval{ 0x1133E, 0x1133F }, // Mc   [2] GRANTHA VOWEL SIGN AA..GRANTHA VOWEL SIGN I
    Interval{ 0x11340, 0x11340 }, // Mn       GRANTHA VOWEL SIGN II
    Interval{ 0x11341, 0x11344 }, // Mc   [4] GRANTHA VOWEL SIGN U..GRANTHA VOWEL SIGN VOCALIC RR
    Interval{ 0x11347, 0x11348 }, // Mc   [2] GRANTHA VOWEL SIGN EE..GRANTHA VOWEL SIGN AI
    Interval{ 0x1134B, 0x1134C }, // Mc   [2] GRANTHA VOWEL SIGN OO..GRANTHA VOWEL SIGN AU
    Interval{ 0x11350, 0x11350 }, // Lo       GRANTHA OM
    Interval{ 0x11357, 0x11357 }, // Mc       GRANTHA AU LENGTH MARK
    Interval{ 0x1135D, 0x11361 }, // Lo   [5] GRANTHA SIGN PLUTA..GRANTHA LETTER VOCALIC LL
    Interval{ 0x11362, 0x11363 }, // Mc   [2] GRANTHA VOWEL SIGN VOCALIC L..GRANTHA VOWEL SIGN VOCALIC LL
    Interval{ 0x11400, 0x11434 }, // Lo  [53] NEWA LETTER A..NEWA LETTER HA
    Interval{ 0x11435, 0x11437 }, // Mc   [3] NEWA VOWEL SIGN AA..NEWA VOWEL SIGN II
    Interval{ 0x11438, 0x1143F }, // Mn   [8] NEWA VOWEL SIGN U..NEWA VOWEL SIGN AI
    Interval{ 0x11440, 0x11441 }, // Mc   [2] NEWA VOWEL SIGN O..NEWA VOWEL SIGN AU
    Interval{ 0x11443, 0x11444 }, // Mn   [2] NEWA SIGN CANDRABINDU..NEWA SIGN ANUSVARA
    Interval{ 0x11445, 0x11445 }, // Mc       NEWA SIGN VISARGA
    Interval{ 0x11447, 0x1144A }, // Lo   [4] NEWA SIGN AVAGRAHA..NEWA SIDDHI
    Interval{ 0x1145F, 0x11461 }, // Lo   [3] NEWA LETTER VEDIC ANUSVARA..NEWA SIGN UPADHMANIYA
    Interval{ 0x11480, 0x114AF }, // Lo  [48] TIRHUTA ANJI..TIRHUTA LETTER HA
    Interval{ 0x114B0, 0x114B2 }, // Mc   [3] TIRHUTA VOWEL SIGN AA..TIRHUTA VOWEL SIGN II
    Interval{ 0x114B3, 0x114B8 }, // Mn   [6] TIRHUTA VOWEL SIGN U..TIRHUTA VOWEL SIGN VOCALIC LL
    Interval{ 0x114B9, 0x114B9 }, // Mc       TIRHUTA VOWEL SIGN E
    Interval{ 0x114BA, 0x114BA }, // Mn       TIRHUTA VOWEL SIGN SHORT E
    Interval{ 0x114BB, 0x114BE }, // Mc   [4] TIRHUTA VOWEL SIGN AI..TIRHUTA VOWEL SIGN AU
    Interval{ 0x114BF, 0x114C0 }, // Mn   [2] TIRHUTA SIGN CANDRABINDU..TIRHUTA SIGN ANUSVARA
    Interval{ 0x114C1, 0x114C1 }, // Mc       TIRHUTA SIGN VISARGA
    Interval{ 0x114C4, 0x114C5 }, // Lo   [2] TIRHUTA SIGN AVAGRAHA..TIRHUTA GVANG
    Interval{ 0x114C7, 0x114C7 }, // Lo       TIRHUTA OM
    Interval{ 0x11580, 0x115AE }, // Lo  [47] SIDDHAM LETTER A..SIDDHAM LETTER HA
    Interval{ 0x115AF, 0x115B1 }, // Mc   [3] SIDDHAM VOWEL SIGN AA..SIDDHAM VOWEL SIGN II
    Interval{ 0x115B2, 0x115B5 }, // Mn   [4] SIDDHAM VOWEL SIGN U..SIDDHAM VOWEL SIGN VOCALIC RR
    Interval{ 0x115B8, 0x115BB }, // Mc   [4] SIDDHAM VOWEL SIGN E..SIDDHAM VOWEL SIGN AU
    Interval{ 0x115BC, 0x115BD }, // Mn   [2] SIDDHAM SIGN CANDRABINDU..SIDDHAM SIGN ANUSVARA
    Interval{ 0x115BE, 0x115BE }, // Mc       SIDDHAM SIGN VISARGA
    Interval{ 0x115D8, 0x115DB }, // Lo   [4] SIDDHAM LETTER THREE-CIRCLE ALTERNATE I..SIDDHAM LETTER ALTERNATE U
    Interval{ 0x115DC, 0x115DD }, // Mn   [2] SIDDHAM VOWEL SIGN ALTERNATE U..SIDDHAM VOWEL SIGN ALTERNATE UU
    Interval{ 0x11600, 0x1162F }, // Lo  [48] MODI LETTER A..MODI LETTER LLA
    Interval{ 0x11630, 0x11632 }, // Mc   [3] MODI VOWEL SIGN AA..MODI VOWEL SIGN II
    Interval{ 0x11633, 0x1163A }, // Mn   [8] MODI VOWEL SIGN U..MODI VOWEL SIGN AI
    Interval{ 0x1163B, 0x1163C }, // Mc   [2] MODI VOWEL SIGN O..MODI VOWEL SIGN AU
    Interval{ 0x1163D, 0x1163D }, // Mn       MODI SIGN ANUSVARA
    Interval{ 0x1163E, 0x1163E }, // Mc       MODI SIGN VISARGA
    Interval{ 0x11640, 0x11640 }, // Mn       MODI SIGN ARDHACANDRA
    Interval{ 0x11644, 0x11644 }, // Lo       MODI SIGN HUVA
    Interval{ 0x11680, 0x116AA }, // Lo  [43] TAKRI LETTER A..TAKRI LETTER RRA
    Interval{ 0x116AB, 0x116AB }, // Mn       TAKRI SIGN ANUSVARA
    Interval{ 0x116AC, 0x116AC }, // Mc       TAKRI SIGN VISARGA
    Interval{ 0x116AD, 0x116AD }, // Mn       TAKRI VOWEL SIGN AA
    Interval{ 0x116AE, 0x116AF }, // Mc   [2] TAKRI VOWEL SIGN I..TAKRI VOWEL SIGN II
    Interval{ 0x116B0, 0x116B5 }, // Mn   [6] TAKRI VOWEL SIGN U..TAKRI VOWEL SIGN AU
    Interval{ 0x116B8, 0x116B8 }, // Lo       TAKRI LETTER ARCHAIC KHA
    Interval{ 0x11700, 0x1171A }, // Lo  [27] AHOM LETTER KA..AHOM LETTER ALTERNATE BA
    Interval{ 0x1171D, 0x1171F }, // Mn   [3] AHOM CONSONANT SIGN MEDIAL LA..AHOM CONSONANT SIGN MEDIAL LIGATING RA
    Interval{ 0x11720, 0x11721 }, // Mc   [2] AHOM VOWEL SIGN A..AHOM VOWEL SIGN AA
    Interval{ 0x11722, 0x11725 }, // Mn   [4] AHOM VOWEL SIGN I..AHOM VOWEL SIGN UU
    Interval{ 0x11726, 0x11726 }, // Mc       AHOM VOWEL SIGN E
    Interval{ 0x11727, 0x1172A }, // Mn   [4] AHOM VOWEL SIGN AW..AHOM VOWEL SIGN AM
    Interval{ 0x11740, 0x11746 }, // Lo   [7] AHOM LETTER CA..AHOM LETTER LLA
    Interval{ 0x11800, 0x1182B }, // Lo  [44] DOGRA LETTER A..DOGRA LETTER RRA
    Interval{ 0x1182C, 0x1182E }, // Mc   [3] DOGRA VOWEL SIGN AA..DOGRA VOWEL SIGN II
    Interval{ 0x1182F, 0x11837 }, // Mn   [9] DOGRA VOWEL SIGN U..DOGRA SIGN ANUSVARA
    Interval{ 0x11838, 0x11838 }, // Mc       DOGRA SIGN VISARGA
    Interval{ 0x118A0, 0x118DF }, // L&  [64] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x118FF, 0x11906 }, // Lo   [8] WARANG CITI OM..DIVES AKURU LETTER E
    Interval{ 0x11909, 0x11909 }, // Lo       DIVES AKURU LETTER O
    Interval{ 0x1190C, 0x11913 }, // Lo   [8] DIVES AKURU LETTER KA..DIVES AKURU LETTER JA
    Interval{ 0x11915, 0x11916 }, // Lo   [2] DIVES AKURU LETTER NYA..DIVES AKURU LETTER TTA
    Interval{ 0x11918, 0x1192F }, // Lo  [24] DIVES AKURU LETTER DDA..DIVES AKURU LETTER ZA
    Interval{ 0x11930, 0x11935 }, // Mc   [6] DIVES AKURU VOWEL SIGN AA..DIVES AKURU VOWEL SIGN E
    Interval{ 0x11937, 0x11938 }, // Mc   [2] DIVES AKURU VOWEL SIGN AI..DIVES AKURU VOWEL SIGN O
    Interval{ 0x1193B, 0x1193C }, // Mn   [2] DIVES AKURU SIGN ANUSVARA..DIVES AKURU SIGN CANDRABINDU
    Interval{ 0x1193F, 0x1193F }, // Lo       DIVES AKURU PREFIXED NASAL SIGN
    Interval{ 0x11940, 0x11940 }, // Mc       DIVES AKURU MEDIAL YA
    Interval{ 0x11941, 0x11941 }, // Lo       DIVES AKURU INITIAL RA
    Interval{ 0x11942, 0x11942 }, // Mc       DIVES AKURU MEDIAL RA
    Interval{ 0x119A0, 0x119A7 }, // Lo   [8] NANDINAGARI LETTER A..NANDINAGARI LETTER VOCALIC RR
    Interval{ 0x119AA, 0x119D0 }, // Lo  [39] NANDINAGARI LETTER E..NANDINAGARI LETTER RRA
    Interval{ 0x119D1, 0x119D3 }, // Mc   [3] NANDINAGARI VOWEL SIGN AA..NANDINAGARI VOWEL SIGN II
    Interval{ 0x119D4, 0x119D7 }, // Mn   [4] NANDINAGARI VOWEL SIGN U..NANDINAGARI VOWEL SIGN VOCALIC RR
    Interval{ 0x119DA, 0x119DB }, // Mn   [2] NANDINAGARI VOWEL SIGN E..NANDINAGARI VOWEL SIGN AI
    Interval{ 0x119DC, 0x119DF }, // Mc   [4] NANDINAGARI VOWEL SIGN O..NANDINAGARI SIGN VISARGA
    Interval{ 0x119E1, 0x119E1 }, // Lo       NANDINAGARI SIGN AVAGRAHA
    Interval{ 0x119E3, 0x119E3 }, // Lo       NANDINAGARI HEADSTROKE
    Interval{ 0x119E4, 0x119E4 }, // Mc       NANDINAGARI VOWEL SIGN PRISHTHAMATRA E
    Interval{ 0x11A00, 0x11A00 }, // Lo       ZANABAZAR SQUARE LETTER A
    Interval{ 0x11A01, 0x11A0A }, // Mn  [10] ZANABAZAR SQUARE VOWEL SIGN I..ZANABAZAR SQUARE VOWEL LENGTH MARK
    Interval{ 0x11A0B, 0x11A32 }, // Lo  [40] ZANABAZAR SQUARE LETTER KA..ZANABAZAR SQUARE LETTER KSSA
    Interval{ 0x11A35, 0x11A38 }, // Mn   [4] ZANABAZAR SQUARE SIGN CANDRABINDU..ZANABAZAR SQUARE SIGN ANUSVARA
    Interval{ 0x11A39, 0x11A39 }, // Mc       ZANABAZAR SQUARE SIGN VISARGA
    Interval{ 0x11A3A, 0x11A3A }, // Lo       ZANABAZAR SQUARE CLUSTER-INITIAL LETTER RA
    Interval{ 0x11A3B, 0x11A3E }, // Mn   [4] ZANABAZAR SQUARE CLUSTER-FINAL LETTER YA..ZANABAZAR SQUARE CLUSTER-FINAL LETTER VA
    Interval{ 0x11A50, 0x11A50 }, // Lo       SOYOMBO LETTER A
    Interval{ 0x11A51, 0x11A56 }, // Mn   [6] SOYOMBO VOWEL SIGN I..SOYOMBO VOWEL SIGN OE
    Interval{ 0x11A57, 0x11A58 }, // Mc   [2] SOYOMBO VOWEL SIGN AI..SOYOMBO VOWEL SIGN AU
    Interval{ 0x11A59, 0x11A5B }, // Mn   [3] SOYOMBO VOWEL SIGN VOCALIC R..SOYOMBO VOWEL LENGTH MARK
    Interval{ 0x11A5C, 0x11A89 }, // Lo  [46] SOYOMBO LETTER KA..SOYOMBO CLUSTER-INITIAL LETTER SA
    Interval{ 0x11A8A, 0x11A96 }, // Mn  [13] SOYOMBO FINAL CONSONANT SIGN G..SOYOMBO SIGN ANUSVARA
    Interval{ 0x11A97, 0x11A97 }, // Mc       SOYOMBO SIGN VISARGA
    Interval{ 0x11A9D, 0x11A9D }, // Lo       SOYOMBO MARK PLUTA
    Interval{ 0x11AB0, 0x11AF8 }, // Lo  [73] CANADIAN SYLLABICS NATTILIK HI..PAU CIN HAU GLOTTAL STOP FINAL
    Interval{ 0x11C00, 0x11C08 }, // Lo   [9] BHAIKSUKI LETTER A..BHAIKSUKI LETTER VOCALIC L
    Interval{ 0x11C0A, 0x11C2E }, // Lo  [37] BHAIKSUKI LETTER E..BHAIKSUKI LETTER HA
    Interval{ 0x11C2F, 0x11C2F }, // Mc       BHAIKSUKI VOWEL SIGN AA
    Interval{ 0x11C30, 0x11C36 }, // Mn   [7] BHAIKSUKI VOWEL SIGN I..BHAIKSUKI VOWEL SIGN VOCALIC L
    Interval{ 0x11C38, 0x11C3D }, // Mn   [6] BHAIKSUKI VOWEL SIGN E..BHAIKSUKI SIGN ANUSVARA
    Interval{ 0x11C3E, 0x11C3E }, // Mc       BHAIKSUKI SIGN VISARGA
    Interval{ 0x11C40, 0x11C40 }, // Lo       BHAIKSUKI SIGN AVAGRAHA
    Interval{ 0x11C72, 0x11C8F }, // Lo  [30] MARCHEN LETTER KA..MARCHEN LETTER A
    Interval{ 0x11C92, 0x11CA7 }, // Mn  [22] MARCHEN SUBJOINED LETTER KA..MARCHEN SUBJOINED LETTER ZA
    Interval{ 0x11CA9, 0x11CA9 }, // Mc       MARCHEN SUBJOINED LETTER YA
    Interval{ 0x11CAA, 0x11CB0 }, // Mn   [7] MARCHEN SUBJOINED LETTER RA..MARCHEN VOWEL SIGN AA
    Interval{ 0x11CB1, 0x11CB1 }, // Mc       MARCHEN VOWEL SIGN I
    Interval{ 0x11CB2, 0x11CB3 }, // Mn   [2] MARCHEN VOWEL SIGN U..MARCHEN VOWEL SIGN E
    Interval{ 0x11CB4, 0x11CB4 }, // Mc       MARCHEN VOWEL SIGN O
    Interval{ 0x11CB5, 0x11CB6 }, // Mn   [2] MARCHEN SIGN ANUSVARA..MARCHEN SIGN CANDRABINDU
    Interval{ 0x11D00, 0x11D06 }, // Lo   [7] MASARAM GONDI LETTER A..MASARAM GONDI LETTER E
    Interval{ 0x11D08, 0x11D09 }, // Lo   [2] MASARAM GONDI LETTER AI..MASARAM GONDI LETTER O
    Interval{ 0x11D0B, 0x11D30 }, // Lo  [38] MASARAM GONDI LETTER AU..MASARAM GONDI LETTER TRA
    Interval{ 0x11D31, 0x11D36 }, // Mn   [6] MASARAM GONDI VOWEL SIGN AA..MASARAM GONDI VOWEL SIGN VOCALIC R
    Interval{ 0x11D3A, 0x11D3A }, // Mn       MASARAM GONDI VOWEL SIGN E
    Interval{ 0x11D3C, 0x11D3D }, // Mn   [2] MASARAM GONDI VOWEL SIGN AI..MASARAM GONDI VOWEL SIGN O
    Interval{ 0x11D3F, 0x11D41 }, // Mn   [3] MASARAM GONDI VOWEL SIGN AU..MASARAM GONDI SIGN VISARGA
    Interval{ 0x11D43, 0x11D43 }, // Mn       MASARAM GONDI SIGN CANDRA
    Interval{ 0x11D46, 0x11D46 }, // Lo       MASARAM GONDI REPHA
    Interval{ 0x11D47, 0x11D47 }, // Mn       MASARAM GONDI RA-KARA
    Interval{ 0x11D60, 0x11D65 }, // Lo   [6] GUNJALA GONDI LETTER A..GUNJALA GONDI LETTER UU
    Interval{ 0x11D67, 0x11D68 }, // Lo   [2] GUNJALA GONDI LETTER EE..GUNJALA GONDI LETTER AI
    Interval{ 0x11D6A, 0x11D89 }, // Lo  [32] GUNJALA GONDI LETTER OO..GUNJALA GONDI LETTER SA
    Interval{ 0x11D8A, 0x11D8E }, // Mc   [5] GUNJALA GONDI VOWEL SIGN AA..GUNJALA GONDI VOWEL SIGN UU
    Interval{ 0x11D90, 0x11D91 }, // Mn   [2] GUNJALA GONDI VOWEL SIGN EE..GUNJALA GONDI VOWEL SIGN AI
    Interval{ 0x11D93, 0x11D94 }, // Mc   [2] GUNJALA GONDI VOWEL SIGN OO..GUNJALA GONDI VOWEL SIGN AU
    Interval{ 0x11D95, 0x11D95 }, // Mn       GUNJALA GONDI SIGN ANUSVARA
    Interval{ 0x11D96, 0x11D96 }, // Mc       GUNJALA GONDI SIGN VISARGA
    Interval{ 0x11D98, 0x11D98 }, // Lo       GUNJALA GONDI OM
    Interval{ 0x11EE0, 0x11EF2 }, // Lo  [19] MAKASAR LETTER KA..MAKASAR ANGKA
    Interval{ 0x11EF3, 0x11EF4 }, // Mn   [2] MAKASAR VOWEL SIGN I..MAKASAR VOWEL SIGN U
    Interval{ 0x11EF5, 0x11EF6 }, // Mc   [2] MAKASAR VOWEL SIGN E..MAKASAR VOWEL SIGN O
    Interval{ 0x11F00, 0x11F01 }, // Mn   [2] KAWI SIGN CANDRABINDU..KAWI SIGN ANUSVARA
    Interval{ 0x11F02, 0x11F02 }, // Lo       KAWI SIGN REPHA
    Interval{ 0x11F03, 0x11F03 }, // Mc       KAWI SIGN VISARGA
    Interval{ 0x11F04, 0x11F10 }, // Lo  [13] KAWI LETTER A..KAWI LETTER O
    Interval{ 0x11F12, 0x11F33 }, // Lo  [34] KAWI LETTER KA..KAWI LETTER JNYA
    Interval{ 0x11F34, 0x11F35 }, // Mc   [2] KAWI VOWEL SIGN AA..KAWI VOWEL SIGN ALTERNATE AA
    Interval{ 0x11F36, 0x11F3A }, // Mn   [5] KAWI VOWEL SIGN I..KAWI VOWEL SIGN VOCALIC R
    Interval{ 0x11F3E, 0x11F3F }, // Mc   [2] KAWI VOWEL SIGN E..KAWI VOWEL SIGN AI
    Interval{ 0x11F40, 0x11F40 }, // Mn       KAWI VOWEL SIGN EU
    Interval{ 0x11FB0, 0x11FB0 }, // Lo       LISU LETTER YHA
    Interval{ 0x12000, 0x12399 }, // Lo [922] CUNEIFORM SIGN A..CUNEIFORM SIGN U U
    Interval{ 0x12400, 0x1246E }, // Nl [111] CUNEIFORM NUMERIC SIGN TWO ASH..CUNEIFORM NUMERIC SIGN NINE U VARIANT FORM
    Interval{ 0x12480, 0x12543 }, // Lo [196] CUNEIFORM SIGN AB TIMES NUN TENU..CUNEIFORM SIGN ZU5 TIMES THREE DISH TENU
    Interval{ 0x12F90, 0x12FF0 }, // Lo  [97] CYPRO-MINOAN SIGN CM001..CYPRO-MINOAN SIGN CM114
    Interval{ 0x13000, 0x1342F }, // Lo [1072] EGYPTIAN HIEROGLYPH A001..EGYPTIAN HIEROGLYPH V011D
    Interval{ 0x13441, 0x13446 }, // Lo   [6] EGYPTIAN HIEROGLYPH FULL BLANK..EGYPTIAN HIEROGLYPH WIDE LOST SIGN
    Interval{ 0x14400, 0x14646 }, // Lo [583] ANATOLIAN HIEROGLYPH A001..ANATOLIAN HIEROGLYPH A530
    Interval{ 0x16800, 0x16A38 }, // Lo [569] BAMUM LETTER PHASE-A NGKUE MFON..BAMUM LETTER PHASE-F VUEQ
    Interval{ 0x16A40, 0x16A5E }, // Lo  [31] MRO LETTER TA..MRO LETTER TEK
    Interval{ 0x16A70, 0x16ABE }, // Lo  [79] TANGSA LETTER OZ..TANGSA LETTER ZA
    Interval{ 0x16AD0, 0x16AED }, // Lo  [30] BASSA VAH LETTER ENNI..BASSA VAH LETTER I
    Interval{ 0x16B00, 0x16B2F }, // Lo  [48] PAHAWH HMONG VOWEL KEEB..PAHAWH HMONG CONSONANT CAU
    Interval{ 0x16B40, 0x16B43 }, // Lm   [4] PAHAWH HMONG SIGN VOS SEEV..PAHAWH HMONG SIGN IB YAM
    Interval{ 0x16B63, 0x16B77 }, // Lo  [21] PAHAWH HMONG SIGN VOS LUB..PAHAWH HMONG SIGN CIM NRES TOS
    Interval{ 0x16B7D, 0x16B8F }, // Lo  [19] PAHAWH HMONG CLAN SIGN TSHEEJ..PAHAWH HMONG CLAN SIGN VWJ
    Interval{ 0x16E40, 0x16E7F }, // L&  [64] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x16F00, 0x16F4A }, // Lo  [75] MIAO LETTER PA..MIAO LETTER RTE
    Interval{ 0x16F4F, 0x16F4F }, // Mn       MIAO SIGN CONSONANT MODIFIER BAR
    Interval{ 0x16F50, 0x16F50 }, // Lo       MIAO LETTER NASALIZATION
    Interval{ 0x16F51, 0x16F87 }, // Mc  [55] MIAO SIGN ASPIRATION..MIAO VOWEL SIGN UI
    Interval{ 0x16F8F, 0x16F92 }, // Mn   [4] MIAO TONE RIGHT..MIAO TONE BELOW
    Interval{ 0x16F93, 0x16F9F }, // Lm  [13] MIAO LETTER TONE-2..MIAO LETTER REFORMED TONE-8
    Interval{ 0x16FE0, 0x16FE1 }, // Lm   [2] TANGUT ITERATION MARK..NUSHU ITERATION MARK
    Interval{ 0x16FE3, 0x16FE3 }, // Lm       OLD CHINESE ITERATION MARK
    Interval{ 0x16FF0, 0x16FF1 }, // Mc   [2] VIETNAMESE ALTERNATE READING MARK CA..VIETNAMESE ALTERNATE READING MARK NHAY
    Interval{ 0x17000, 0x187F7 }, // Lo [6136] TANGUT IDEOGRAPH-17000..TANGUT IDEOGRAPH-187F7
    Interval{ 0x18800, 0x18CD5 }, // Lo [1238] TANGUT COMPONENT-001..KHITAN SMALL SCRIPT CHARACTER-18CD5
    Interval{ 0x18D00, 0x18D08 }, // Lo   [9] TANGUT IDEOGRAPH-18D00..TANGUT IDEOGRAPH-18D08
    Interval{ 0x1AFF0, 0x1AFF3 }, // Lm   [4] KATAKANA LETTER MINNAN TONE-2..KATAKANA LETTER MINNAN TONE-5
    Interval{ 0x1AFF5, 0x1AFFB }, // Lm   [7] KATAKANA LETTER MINNAN TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-5
    Interval{ 0x1AFFD, 0x1AFFE }, // Lm   [2] KATAKANA LETTER MINNAN NASALIZED TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-8
    Interval{ 0x1B000, 0x1B122 }, // Lo [291] KATAKANA LETTER ARCHAIC E..KATAKANA LETTER ARCHAIC WU
    Interval{ 0x1B132, 0x1B132 }, // Lo       HIRAGANA LETTER SMALL KO
    Interval{ 0x1B150, 0x1B152 }, // Lo   [3] HIRAGANA LETTER SMALL WI..HIRAGANA LETTER SMALL WO
    Interval{ 0x1B155, 0x1B155 }, // Lo       KATAKANA LETTER SMALL KO
    Interval{ 0x1B164, 0x1B167 }, // Lo   [4] KATAKANA LETTER SMALL WI..KATAKANA LETTER SMALL N
    Interval{ 0x1B170, 0x1B2FB }, // Lo [396] NUSHU CHARACTER-1B170..NUSHU CHARACTER-1B2FB
    Interval{ 0x1BC00, 0x1BC6A }, // Lo [107] DUPLOYAN LETTER H..DUPLOYAN LETTER VOCALIC M
    Interval{ 0x1BC70, 0x1BC7C }, // Lo  [13] DUPLOYAN AFFIX LEFT HORIZONTAL SECANT..DUPLOYAN AFFIX ATTACHED TANGENT HOOK
    Interval{ 0x1BC80, 0x1BC88 }, // Lo   [9] DUPLOYAN AFFIX HIGH ACUTE..DUPLOYAN AFFIX HIGH VERTICAL
    Interval{ 0x1BC90, 0x1BC99 }, // Lo  [10] DUPLOYAN AFFIX LOW ACUTE..DUPLOYAN AFFIX LOW ARROW
    Interval{ 0x1BC9E, 0x1BC9E }, // Mn       DUPLOYAN DOUBLE MARK
    Interval{ 0x1D400, 0x1D454 }, // L&  [85] MATHEMATICAL BOLD CAPITAL A..MATHEMATICAL ITALIC SMALL G
    Interval{ 0x1D456, 0x1D49C }, // L&  [71] MATHEMATICAL ITALIC SMALL I..MATHEMATICAL SCRIPT CAPITAL A
    Interval{ 0x1D49E, 0x1D49F }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL C..MATHEMATICAL SCRIPT CAPITAL D
    Interval{ 0x1D4A2, 0x1D4A2 }, // L&       MATHEMATICAL SCRIPT CAPITAL G
    Interval{ 0x1D4A5, 0x1D4A6 }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL J..MATHEMATICAL SCRIPT CAPITAL K
    Interval{ 0x1D4A9, 0x1D4AC }, // L&   [4] MATHEMATICAL SCRIPT CAPITAL N..MATHEMATICAL SCRIPT CAPITAL Q
    Interval{ 0x1D4AE, 0x1D4B9 }, // L&  [12] MATHEMATICAL SCRIPT CAPITAL S..MATHEMATICAL SCRIPT SMALL D
    Interval{ 0x1D4BB, 0x1D4BB }, // L&       MATHEMATICAL SCRIPT SMALL F
    Interval{ 0x1D4BD, 0x1D4C3 }, // L&   [7] MATHEMATICAL SCRIPT SMALL H..MATHEMATICAL SCRIPT SMALL N
    Interval{ 0x1D4C5, 0x1D505 }, // L&  [65] MATHEMATICAL SCRIPT SMALL P..MATHEMATICAL FRAKTUR CAPITAL B
    Interval{ 0x1D507, 0x1D50A }, // L&   [4] MATHEMATICAL FRAKTUR CAPITAL D..MATHEMATICAL FRAKTUR CAPITAL G
    Interval{ 0x1D50D, 0x1D514 }, // L&   [8] MATHEMATICAL FRAKTUR CAPITAL J..MATHEMATICAL FRAKTUR CAPITAL Q
    Interval{ 0x1D516, 0x1D51C }, // L&   [7] MATHEMATICAL FRAKTUR CAPITAL S..MATHEMATICAL FRAKTUR CAPITAL Y
    Interval{ 0x1D51E, 0x1D539 }, // L&  [28] MATHEMATICAL FRAKTUR SMALL A..MATHEMATICAL DOUBLE-STRUCK CAPITAL B
    Interval{ 0x1D53B, 0x1D53E }, // L&   [4] MATHEMATICAL DOUBLE-STRUCK CAPITAL D..MATHEMATICAL DOUBLE-STRUCK CAPITAL G
    Interval{ 0x1D540, 0x1D544 }, // L&   [5] MATHEMATICAL DOUBLE-STRUCK CAPITAL I..MATHEMATICAL DOUBLE-STRUCK CAPITAL M
    Interval{ 0x1D546, 0x1D546 }, // L&       MATHEMATICAL DOUBLE-STRUCK CAPITAL O
    Interval{ 0x1D54A, 0x1D550 }, // L&   [7] MATHEMATICAL DOUBLE-STRUCK CAPITAL S..MATHEMATICAL DOUBLE-STRUCK CAPITAL Y
    Interval{ 0x1D552, 0x1D6A5 }, // L& [340] MATHEMATICAL DOUBLE-STRUCK SMALL A..MATHEMATICAL ITALIC SMALL DOTLESS J
    Interval{ 0x1D6A8, 0x1D6C0 }, // L&  [25] MATHEMATICAL BOLD CAPITAL ALPHA..MATHEMATICAL BOLD CAPITAL OMEGA
    Interval{ 0x1D6C2, 0x1D6DA }, // L&  [25] MATHEMATICAL BOLD SMALL ALPHA..MATHEMATICAL BOLD SMALL OMEGA
    Interval{ 0x1D6DC, 0x1D6FA }, // L&  [31] MATHEMATICAL BOLD EPSILON SYMBOL..MATHEMATICAL ITALIC CAPITAL OMEGA
    Interval{ 0x1D6FC, 0x1D714 }, // L&  [25] MATHEMATICAL ITALIC SMALL ALPHA..MATHEMATICAL ITALIC SMALL OMEGA
    Interval{ 0x1D716, 0x1D734 }, // L&  [31] MATHEMATICAL ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D736, 0x1D74E }, // L&  [25] MATHEMATICAL BOLD ITALIC SMALL ALPHA..MATHEMATICAL BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D750, 0x1D76E }, // L&  [31] MATHEMATICAL BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGA
    Interval{ 0x1D770, 0x1D788 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD SMALL OMEGA
    Interval{ 0x1D78A, 0x1D7A8 }, // L&  [31] MATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D7AA, 0x1D7C2 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D7C4, 0x1D7CB }, // L&   [8] MATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD SMALL DIGAMMA
    Interval{ 0x1DF00, 0x1DF09 }, // L&  [10] LATIN SMALL LETTER FENG DIGRAPH WITH TRILL..LATIN SMALL LETTER T WITH HOOK AND RETROFLEX HOOK
    Interval{ 0x1DF0A, 0x1DF0A }, // Lo       LATIN LETTER RETROFLEX CLICK WITH RETROFLEX HOOK
    Interval{ 0x1DF0B, 0x1DF1E }, // L&  [20] LATIN SMALL LETTER ESH WITH DOUBLE BAR..LATIN SMALL LETTER S WITH CURL
    Interval{ 0x1DF25, 0x1DF2A }, // L&   [6] LATIN SMALL LETTER D WITH MID-HEIGHT LEFT HOOK..LATIN SMALL LETTER T WITH MID-HEIGHT LEFT HOOK
    Interval{ 0x1E000, 0x1E006 }, // Mn   [7] COMBINING GLAGOLITIC LETTER AZU..COMBINING GLAGOLITIC LETTER ZHIVETE
    Interval{ 0x1E008, 0x1E018 }, // Mn  [17] COMBINING GLAGOLITIC LETTER ZEMLJA..COMBINING GLAGOLITIC LETTER HERU
    Interval{ 0x1E01B, 0x1E021 }, // Mn   [7] COMBINING GLAGOLITIC LETTER SHTA..COMBINING GLAGOLITIC LETTER YATI
    Interval{ 0x1E023, 0x1E024 }, // Mn   [2] COMBINING GLAGOLITIC LETTER YU..COMBINING GLAGOLITIC LETTER SMALL YUS
    Interval{ 0x1E026, 0x1E02A }, // Mn   [5] COMBINING GLAGOLITIC LETTER YO..COMBINING GLAGOLITIC LETTER FITA
    Interval{ 0x1E030, 0x1E06D }, // Lm  [62] MODIFIER LETTER CYRILLIC SMALL A..MODIFIER LETTER CYRILLIC SMALL STRAIGHT U WITH STROKE
    Interval{ 0x1E08F, 0x1E08F }, // Mn       COMBINING CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
    Interval{ 0x1E100, 0x1E12C }, // Lo  [45] NYIAKENG PUACHUE HMONG LETTER MA..NYIAKENG PUACHUE HMONG LETTER W
    Interval{ 0x1E137, 0x1E13D }, // Lm   [7] NYIAKENG PUACHUE HMONG SIGN FOR PERSON..NYIAKENG PUACHUE HMONG SYLLABLE LENGTHENER
    Interval{ 0x1E14E, 0x1E14E }, // Lo       NYIAKENG PUACHUE HMONG LOGOGRAM NYAJ
    Interval{ 0x1E290, 0x1E2AD }, // Lo  [30] TOTO LETTER PA..TOTO LETTER A
    Interval{ 0x1E2C0, 0x1E2EB }, // Lo  [44] WANCHO LETTER AA..WANCHO LETTER YIH
    Interval{ 0x1E4D0, 0x1E4EA }, // Lo  [27] NAG MUNDARI LETTER O..NAG MUNDARI LETTER ELL
    Interval{ 0x1E4EB, 0x1E4EB }, // Lm       NAG MUNDARI SIGN OJOD
    Interval{ 0x1E7E0, 0x1E7E6 }, // Lo   [7] ETHIOPIC SYLLABLE HHYA..ETHIOPIC SYLLABLE HHYO
    Interval{ 0x1E7E8, 0x1E7EB }, // Lo   [4] ETHIOPIC SYLLABLE GURAGE HHWA..ETHIOPIC SYLLABLE HHWE
    Interval{ 0x1E7ED, 0x1E7EE }, // Lo   [2] ETHIOPIC SYLLABLE GURAGE MWI..ETHIOPIC SYLLABLE GURAGE MWEE
    Interval{ 0x1E7F0, 0x1E7FE }, // Lo  [15] ETHIOPIC SYLLABLE GURAGE QWI..ETHIOPIC SYLLABLE GURAGE PWEE
    Interval{ 0x1E800, 0x1E8C4 }, // Lo [197] MENDE KIKAKUI SYLLABLE M001 KI..MENDE KIKAKUI SYLLABLE M060 NYON
    Interval{ 0x1E900, 0x1E943 }, // L&  [68] ADLAM CAPITAL LETTER ALIF..ADLAM SMALL LETTER SHA
    Interval{ 0x1E947, 0x1E947 }, // Mn       ADLAM HAMZA
    Interval{ 0x1E94B, 0x1E94B }, // Lm       ADLAM NASALIZATION MARK
    Interval{ 0x1EE00, 0x1EE03 }, // Lo   [4] ARABIC MATHEMATICAL ALEF..ARABIC MATHEMATICAL DAL
    Interval{ 0x1EE05, 0x1EE1F }, // Lo  [27] ARABIC MATHEMATICAL WAW..ARABIC MATHEMATICAL DOTLESS QAF
    Interval{ 0x1EE21, 0x1EE22 }, // Lo   [2] ARABIC MATHEMATICAL INITIAL BEH..ARABIC MATHEMATICAL INITIAL JEEM
    Interval{ 0x1EE24, 0x1EE24 }, // Lo       ARABIC MATHEMATICAL INITIAL HEH
    Interval{ 0x1EE27, 0x1EE27 }, // Lo       ARABIC MATHEMATICAL INITIAL HAH
    Interval{ 0x1EE29, 0x1EE32 }, // Lo  [10] ARABIC MATHEMATICAL INITIAL YEH..ARABIC MATHEMATICAL INITIAL QAF
    Interval{ 0x1EE34, 0x1EE37 }, // Lo   [4] ARABIC MATHEMATICAL INITIAL SHEEN..ARABIC MATHEMATICAL INITIAL KHAH
    Interval{ 0x1EE39, 0x1EE39 }, // Lo       ARABIC MATHEMATICAL INITIAL DAD
    Interval{ 0x1EE3B, 0x1EE3B }, // Lo       ARABIC MATHEMATICAL INITIAL GHAIN
    Interval{ 0x1EE42, 0x1EE42 }, // Lo       ARABIC MATHEMATICAL TAILED JEEM
    Interval{ 0x1EE47, 0x1EE47 }, // Lo       ARABIC MATHEMATICAL TAILED HAH
    Interval{ 0x1EE49, 0x1EE49 }, // Lo       ARABIC MATHEMATICAL TAILED YEH
    Interval{ 0x1EE4B, 0x1EE4B }, // Lo       ARABIC MATHEMATICAL TAILED LAM
    Interval{ 0x1EE4D, 0x1EE4F }, // Lo   [3] ARABIC MATHEMATICAL TAILED NOON..ARABIC MATHEMATICAL TAILED AIN
    Interval{ 0x1EE51, 0x1EE52 }, // Lo   [2] ARABIC MATHEMATICAL TAILED SAD..ARABIC MATHEMATICAL TAILED QAF
    Interval{ 0x1EE54, 0x1EE54 }, // Lo       ARABIC MATHEMATICAL TAILED SHEEN
    Interval{ 0x1EE57, 0x1EE57 }, // Lo       ARABIC MATHEMATICAL TAILED KHAH
    Interval{ 0x1EE59, 0x1EE59 }, // Lo       ARABIC MATHEMATICAL TAILED DAD
    Interval{ 0x1EE5B, 0x1EE5B }, // Lo       ARABIC MATHEMATICAL TAILED GHAIN
    Interval{ 0x1EE5D, 0x1EE5D }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS NOON
    Interval{ 0x1EE5F, 0x1EE5F }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS QAF
    Interval{ 0x1EE61, 0x1EE62 }, // Lo   [2] ARABIC MATHEMATICAL STRETCHED BEH..ARABIC MATHEMATICAL STRETCHED JEEM
    Interval{ 0x1EE64, 0x1EE64 }, // Lo       ARABIC MATHEMATICAL STRETCHED HEH
    Interval{ 0x1EE67, 0x1EE6A }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED HAH..ARABIC MATHEMATICAL STRETCHED KAF
    Interval{ 0x1EE6C, 0x1EE72 }, // Lo   [7] ARABIC MATHEMATICAL STRETCHED MEEM..ARABIC MATHEMATICAL STRETCHED QAF
    Interval{ 0x1EE74, 0x1EE77 }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED SHEEN..ARABIC MATHEMATICAL STRETCHED KHAH
    Interval{ 0x1EE79, 0x1EE7C }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED DAD..ARABIC MATHEMATICAL STRETCHED DOTLESS BEH
    Interval{ 0x1EE7E, 0x1EE7E }, // Lo       ARABIC MATHEMATICAL STRETCHED DOTLESS FEH
    Interval{ 0x1EE80, 0x1EE89 }, // Lo  [10] ARABIC MATHEMATICAL LOOPED ALEF..ARABIC MATHEMATICAL LOOPED YEH
    Interval{ 0x1EE8B, 0x1EE9B }, // Lo  [17] ARABIC MATHEMATICAL LOOPED LAM..ARABIC MATHEMATICAL LOOPED GHAIN
    Interval{ 0x1EEA1, 0x1EEA3 }, // Lo   [3] ARABIC MATHEMATICAL DOUBLE-STRUCK BEH..ARABIC MATHEMATICAL DOUBLE-STRUCK DAL
    Interval{ 0x1EEA5, 0x1EEA9 }, // Lo   [5] ARABIC MATHEMATICAL DOUBLE-STRUCK WAW..ARABIC MATHEMATICAL DOUBLE-STRUCK YEH
    Interval{ 0x1EEAB, 0x1EEBB }, // Lo  [17] ARABIC MATHEMATICAL DOUBLE-STRUCK LAM..ARABIC MATHEMATICAL DOUBLE-STRUCK GHAIN
    Interval{ 0x1F130, 0x1F149 }, // So  [26] SQUARED LATIN CAPITAL LETTER A..SQUARED LATIN CAPITAL LETTER Z
    Interval{ 0x1F150, 0x1F169 }, // So  [26] NEGATIVE CIRCLED LATIN CAPITAL LETTER A..NEGATIVE CIRCLED LATIN CAPITAL LETTER Z
    Interval{ 0x1F170, 0x1F189 }, // So  [26] NEGATIVE SQUARED LATIN CAPITAL LETTER A..NEGATIVE SQUARED LATIN CAPITAL LETTER Z
    Interval{ 0x20000, 0x2A6DF }, // Lo [42720] CJK UNIFIED IDEOGRAPH-20000..CJK UNIFIED IDEOGRAPH-2A6DF
    Interval{ 0x2A700, 0x2B739 }, // Lo [4154] CJK UNIFIED IDEOGRAPH-2A700..CJK UNIFIED IDEOGRAPH-2B739
    Interval{ 0x2B740, 0x2B81D }, // Lo [222] CJK UNIFIED IDEOGRAPH-2B740..CJK UNIFIED IDEOGRAPH-2B81D
    Interval{ 0x2B820, 0x2CEA1 }, // Lo [5762] CJK UNIFIED IDEOGRAPH-2B820..CJK UNIFIED IDEOGRAPH-2CEA1
    Interval{ 0x2CEB0, 0x2EBE0 }, // Lo [7473] CJK UNIFIED IDEOGRAPH-2CEB0..CJK UNIFIED IDEOGRAPH-2EBE0
    Interval{ 0x2F800, 0x2FA1D }, // Lo [542] CJK COMPATIBILITY IDEOGRAPH-2F800..CJK COMPATIBILITY IDEOGRAPH-2FA1D
    Interval{ 0x30000, 0x3134A }, // Lo [4939] CJK UNIFIED IDEOGRAPH-30000..CJK UNIFIED IDEOGRAPH-3134A
    Interval{ 0x31350, 0x323AF }, // Lo [4192] CJK UNIFIED IDEOGRAPH-31350..CJK UNIFIED IDEOGRAPH-323AF
}; // }}}
auto static const Case_Ignorable = std::array<Interval, 491>{ // {{{
    Interval{ 0x0027, 0x0027 }, // Po       APOSTROPHE
    Interval{ 0x002E, 0x002E }, // Po       FULL STOP
    Interval{ 0x003A, 0x003A }, // Po       COLON
    Interval{ 0x005E, 0x005E }, // Sk       CIRCUMFLEX ACCENT
    Interval{ 0x0060, 0x0060 }, // Sk       GRAVE ACCENT
    Interval{ 0x00A8, 0x00A8 }, // Sk       DIAERESIS
    Interval{ 0x00AD, 0x00AD }, // Cf       SOFT HYPHEN
    Interval{ 0x00AF, 0x00AF }, // Sk       MACRON
    Interval{ 0x00B4, 0x00B4 }, // Sk       ACUTE ACCENT
    Interval{ 0x00B7, 0x00B7 }, // Po       MIDDLE DOT
    Interval{ 0x00B8, 0x00B8 }, // Sk       CEDILLA
    Interval{ 0x02B0, 0x02C1 }, // Lm  [18] MODIFIER LETTER SMALL H..MODIFIER LETTER REVERSED GLOTTAL STOP
    Interval{ 0x02C2, 0x02C5 }, // Sk   [4] MODIFIER LETTER LEFT ARROWHEAD..MODIFIER LETTER DOWN ARROWHEAD
    Interval{ 0x02C6, 0x02D1 }, // Lm  [12] MODIFIER LETTER CIRCUMFLEX ACCENT..MODIFIER LETTER HALF TRIANGULAR COLON
    Interval{ 0x02D2, 0x02DF }, // Sk  [14] MODIFIER LETTER CENTRED RIGHT HALF RING..MODIFIER LETTER CROSS ACCENT
    Interval{ 0x02E0, 0x02E4 }, // Lm   [5] MODIFIER LETTER SMALL GAMMA..MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    Interval{ 0x02E5, 0x02EB }, // Sk   [7] MODIFIER LETTER EXTRA-HIGH TONE BAR..MODIFIER LETTER YANG DEPARTING TONE MARK
    Interval{ 0x02EC, 0x02EC }, // Lm       MODIFIER LETTER VOICING
    Interval{ 0x02ED, 0x02ED }, // Sk       MODIFIER LETTER UNASPIRATED
    Interval{ 0x02EE, 0x02EE }, // Lm       MODIFIER LETTER DOUBLE APOSTROPHE
    Interval{ 0x02EF, 0x02FF }, // Sk  [17] MODIFIER LETTER LOW DOWN ARROWHEAD..MODIFIER LETTER LOW LEFT ARROW
    Interval{ 0x0300, 0x036F }, // Mn [112] COMBINING GRAVE ACCENT..COMBINING LATIN SMALL LETTER X
    Interval{ 0x0374, 0x0374 }, // Lm       GREEK NUMERAL SIGN
    Interval{ 0x0375, 0x0375 }, // Sk       GREEK LOWER NUMERAL SIGN
    Interval{ 0x037A, 0x037A }, // Lm       GREEK YPOGEGRAMMENI
    Interval{ 0x0384, 0x0385 }, // Sk   [2] GREEK TONOS..GREEK DIALYTIKA TONOS
    Interval{ 0x0387, 0x0387 }, // Po       GREEK ANO TELEIA
    Interval{ 0x0483, 0x0487 }, // Mn   [5] COMBINING CYRILLIC TITLO..COMBINING CYRILLIC POKRYTIE
    Interval{ 0x0488, 0x0489 }, // Me   [2] COMBINING CYRILLIC HUNDRED THOUSANDS SIGN..COMBINING CYRILLIC MILLIONS SIGN
    Interval{ 0x0559, 0x0559 }, // Lm       ARMENIAN MODIFIER LETTER LEFT HALF RING
    Interval{ 0x055F, 0x055F }, // Po       ARMENIAN ABBREVIATION MARK
    Interval{ 0x0591, 0x05BD }, // Mn  [45] HEBREW ACCENT ETNAHTA..HEBREW POINT METEG
    Interval{ 0x05BF, 0x05BF }, // Mn       HEBREW POINT RAFE
    Interval{ 0x05C1, 0x05C2 }, // Mn   [2] HEBREW POINT SHIN DOT..HEBREW POINT SIN DOT
    Interval{ 0x05C4, 0x05C5 }, // Mn   [2] HEBREW MARK UPPER DOT..HEBREW MARK LOWER DOT
    Interval{ 0x05C7, 0x05C7 }, // Mn       HEBREW POINT QAMATS QATAN
    Interval{ 0x05F4, 0x05F4 }, // Po       HEBREW PUNCTUATION GERSHAYIM
    Interval{ 0x0600, 0x0605 }, // Cf   [6] ARABIC NUMBER SIGN..ARABIC NUMBER MARK ABOVE
    Interval{ 0x0610, 0x061A }, // Mn  [11] ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM..ARABIC SMALL KASRA
    Interval{ 0x061C, 0x061C }, // Cf       ARABIC LETTER MARK
    Interval{ 0x0640, 0x0640 }, // Lm       ARABIC TATWEEL
    Interval{ 0x064B, 0x065F }, // Mn  [21] ARABIC FATHATAN..ARABIC WAVY HAMZA BELOW
    Interval{ 0x0670, 0x0670 }, // Mn       ARABIC LETTER SUPERSCRIPT ALEF
    Interval{ 0x06D6, 0x06DC }, // Mn   [7] ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA..ARABIC SMALL HIGH SEEN
    Interval{ 0x06DD, 0x06DD }, // Cf       ARABIC END OF AYAH
    Interval{ 0x06DF, 0x06E4 }, // Mn   [6] ARABIC SMALL HIGH ROUNDED ZERO..ARABIC SMALL HIGH MADDA
    Interval{ 0x06E5, 0x06E6 }, // Lm   [2] ARABIC SMALL WAW..ARABIC SMALL YEH
    Interval{ 0x06E7, 0x06E8 }, // Mn   [2] ARABIC SMALL HIGH YEH..ARABIC SMALL HIGH NOON
    Interval{ 0x06EA, 0x06ED }, // Mn   [4] ARABIC EMPTY CENTRE LOW STOP..ARABIC SMALL LOW MEEM
    Interval{ 0x070F, 0x070F }, // Cf       SYRIAC ABBREVIATION MARK
    Interval{ 0x0711, 0x0711 }, // Mn       SYRIAC LETTER SUPERSCRIPT ALAPH
    Interval{ 0x0730, 0x074A }, // Mn  [27] SYRIAC PTHAHA ABOVE..SYRIAC BARREKH
    Interval{ 0x07A6, 0x07B0 }, // Mn  [11] THAANA ABAFILI..THAANA SUKUN
    Interval{ 0x07EB, 0x07F3 }, // Mn   [9] NKO COMBINING SHORT HIGH TONE..NKO COMBINING DOUBLE DOT ABOVE
    Interval{ 0x07F4, 0x07F5 }, // Lm   [2] NKO HIGH TONE APOSTROPHE..NKO LOW TONE APOSTROPHE
    Interval{ 0x07FA, 0x07FA }, // Lm       NKO LAJANYALAN
    Interval{ 0x07FD, 0x07FD }, // Mn       NKO DANTAYALAN
    Interval{ 0x0816, 0x0819 }, // Mn   [4] SAMARITAN MARK IN..SAMARITAN MARK DAGESH
    Interval{ 0x081A, 0x081A }, // Lm       SAMARITAN MODIFIER LETTER EPENTHETIC YUT
    Interval{ 0x081B, 0x0823 }, // Mn   [9] SAMARITAN MARK EPENTHETIC YUT..SAMARITAN VOWEL SIGN A
    Interval{ 0x0824, 0x0824 }, // Lm       SAMARITAN MODIFIER LETTER SHORT A
    Interval{ 0x0825, 0x0827 }, // Mn   [3] SAMARITAN VOWEL SIGN SHORT A..SAMARITAN VOWEL SIGN U
    Interval{ 0x0828, 0x0828 }, // Lm       SAMARITAN MODIFIER LETTER I
    Interval{ 0x0829, 0x082D }, // Mn   [5] SAMARITAN VOWEL SIGN LONG I..SAMARITAN MARK NEQUDAA
    Interval{ 0x0859, 0x085B }, // Mn   [3] MANDAIC AFFRICATION MARK..MANDAIC GEMINATION MARK
    Interval{ 0x0888, 0x0888 }, // Sk       ARABIC RAISED ROUND DOT
    Interval{ 0x0890, 0x0891 }, // Cf   [2] ARABIC POUND MARK ABOVE..ARABIC PIASTRE MARK ABOVE
    Interval{ 0x0898, 0x089F }, // Mn   [8] ARABIC SMALL HIGH WORD AL-JUZ..ARABIC HALF MADDA OVER MADDA
    Interval{ 0x08C9, 0x08C9 }, // Lm       ARABIC SMALL FARSI YEH
    Interval{ 0x08CA, 0x08E1 }, // Mn  [24] ARABIC SMALL HIGH FARSI YEH..ARABIC SMALL HIGH SIGN SAFHA
    Interval{ 0x08E2, 0x08E2 }, // Cf       ARABIC DISPUTED END OF AYAH
    Interval{ 0x08E3, 0x0902 }, // Mn  [32] ARABIC TURNED DAMMA BELOW..DEVANAGARI SIGN ANUSVARA
    Interval{ 0x093A, 0x093A }, // Mn       DEVANAGARI VOWEL SIGN OE
    Interval{ 0x093C, 0x093C }, // Mn       DEVANAGARI SIGN NUKTA
    Interval{ 0x0941, 0x0948 }, // Mn   [8] DEVANAGARI VOWEL SIGN U..DEVANAGARI VOWEL SIGN AI
    Interval{ 0x094D, 0x094D }, // Mn       DEVANAGARI SIGN VIRAMA
    Interval{ 0x0951, 0x0957 }, // Mn   [7] DEVANAGARI STRESS SIGN UDATTA..DEVANAGARI VOWEL SIGN UUE
    Interval{ 0x0962, 0x0963 }, // Mn   [2] DEVANAGARI VOWEL SIGN VOCALIC L..DEVANAGARI VOWEL SIGN VOCALIC LL
    Interval{ 0x0971, 0x0971 }, // Lm       DEVANAGARI SIGN HIGH SPACING DOT
    Interval{ 0x0981, 0x0981 }, // Mn       BENGALI SIGN CANDRABINDU
    Interval{ 0x09BC, 0x09BC }, // Mn       BENGALI SIGN NUKTA
    Interval{ 0x09C1, 0x09C4 }, // Mn   [4] BENGALI VOWEL SIGN U..BENGALI VOWEL SIGN VOCALIC RR
    Interval{ 0x09CD, 0x09CD }, // Mn       BENGALI SIGN VIRAMA
    Interval{ 0x09E2, 0x09E3 }, // Mn   [2] BENGALI VOWEL SIGN VOCALIC L..BENGALI VOWEL SIGN VOCALIC LL
    Interval{ 0x09FE, 0x09FE }, // Mn       BENGALI SANDHI MARK
    Interval{ 0x0A01, 0x0A02 }, // Mn   [2] GURMUKHI SIGN ADAK BINDI..GURMUKHI SIGN BINDI
    Interval{ 0x0A3C, 0x0A3C }, // Mn       GURMUKHI SIGN NUKTA
    Interval{ 0x0A41, 0x0A42 }, // Mn   [2] GURMUKHI VOWEL SIGN U..GURMUKHI VOWEL SIGN UU
    Interval{ 0x0A47, 0x0A48 }, // Mn   [2] GURMUKHI VOWEL SIGN EE..GURMUKHI VOWEL SIGN AI
    Interval{ 0x0A4B, 0x0A4D }, // Mn   [3] GURMUKHI VOWEL SIGN OO..GURMUKHI SIGN VIRAMA
    Interval{ 0x0A51, 0x0A51 }, // Mn       GURMUKHI SIGN UDAAT
    Interval{ 0x0A70, 0x0A71 }, // Mn   [2] GURMUKHI TIPPI..GURMUKHI ADDAK
    Interval{ 0x0A75, 0x0A75 }, // Mn       GURMUKHI SIGN YAKASH
    Interval{ 0x0A81, 0x0A82 }, // Mn   [2] GUJARATI SIGN CANDRABINDU..GUJARATI SIGN ANUSVARA
    Interval{ 0x0ABC, 0x0ABC }, // Mn       GUJARATI SIGN NUKTA
    Interval{ 0x0AC1, 0x0AC5 }, // Mn   [5] GUJARATI VOWEL SIGN U..GUJARATI VOWEL SIGN CANDRA E
    Interval{ 0x0AC7, 0x0AC8 }, // Mn   [2] GUJARATI VOWEL SIGN E..GUJARATI VOWEL SIGN AI
    Interval{ 0x0ACD, 0x0ACD }, // Mn       GUJARATI SIGN VIRAMA
    Interval{ 0x0AE2, 0x0AE3 }, // Mn   [2] GUJARATI VOWEL SIGN VOCALIC L..GUJARATI VOWEL SIGN VOCALIC LL
    Interval{ 0x0AFA, 0x0AFF }, // Mn   [6] GUJARATI SIGN SUKUN..GUJARATI SIGN TWO-CIRCLE NUKTA ABOVE
    Interval{ 0x0B01, 0x0B01 }, // Mn       ORIYA SIGN CANDRABINDU
    Interval{ 0x0B3C, 0x0B3C }, // Mn       ORIYA SIGN NUKTA
    Interval{ 0x0B3F, 0x0B3F }, // Mn       ORIYA VOWEL SIGN I
    Interval{ 0x0B41, 0x0B44 }, // Mn   [4] ORIYA VOWEL SIGN U..ORIYA VOWEL SIGN VOCALIC RR
    Interval{ 0x0B4D, 0x0B4D }, // Mn       ORIYA SIGN VIRAMA
    Interval{ 0x0B55, 0x0B56 }, // Mn   [2] ORIYA SIGN OVERLINE..ORIYA AI LENGTH MARK
    Interval{ 0x0B62, 0x0B63 }, // Mn   [2] ORIYA VOWEL SIGN VOCALIC L..ORIYA VOWEL SIGN VOCALIC LL
    Interval{ 0x0B82, 0x0B82 }, // Mn       TAMIL SIGN ANUSVARA
    Interval{ 0x0BC0, 0x0BC0 }, // Mn       TAMIL VOWEL SIGN II
    Interval{ 0x0BCD, 0x0BCD }, // Mn       TAMIL SIGN VIRAMA
    Interval{ 0x0C00, 0x0C00 }, // Mn       TELUGU SIGN COMBINING CANDRABINDU ABOVE
    Interval{ 0x0C04, 0x0C04 }, // Mn       TELUGU SIGN COMBINING ANUSVARA ABOVE
    Interval{ 0x0C3C, 0x0C3C }, // Mn       TELUGU SIGN NUKTA
    Interval{ 0x0C3E, 0x0C40 }, // Mn   [3] TELUGU VOWEL SIGN AA..TELUGU VOWEL SIGN II
    Interval{ 0x0C46, 0x0C48 }, // Mn   [3] TELUGU VOWEL SIGN E..TELUGU VOWEL SIGN AI
    Interval{ 0x0C4A, 0x0C4D }, // Mn   [4] TELUGU VOWEL SIGN O..TELUGU SIGN VIRAMA
    Interval{ 0x0C55, 0x0C56 }, // Mn   [2] TELUGU LENGTH MARK..TELUGU AI LENGTH MARK
    Interval{ 0x0C62, 0x0C63 }, // Mn   [2] TELUGU VOWEL SIGN VOCALIC L..TELUGU VOWEL SIGN VOCALIC LL
    Interval{ 0x0C81, 0x0C81 }, // Mn       KANNADA SIGN CANDRABINDU
    Interval{ 0x0CBC, 0x0CBC }, // Mn       KANNADA SIGN NUKTA
    Interval{ 0x0CBF, 0x0CBF }, // Mn       KANNADA VOWEL SIGN I
    Interval{ 0x0CC6, 0x0CC6 }, // Mn       KANNADA VOWEL SIGN E
    Interval{ 0x0CCC, 0x0CCD }, // Mn   [2] KANNADA VOWEL SIGN AU..KANNADA SIGN VIRAMA
    Interval{ 0x0CE2, 0x0CE3 }, // Mn   [2] KANNADA VOWEL SIGN VOCALIC L..KANNADA VOWEL SIGN VOCALIC LL
    Interval{ 0x0D00, 0x0D01 }, // Mn   [2] MALAYALAM SIGN COMBINING ANUSVARA ABOVE..MALAYALAM SIGN CANDRABINDU
    Interval{ 0x0D3B, 0x0D3C }, // Mn   [2] MALAYALAM SIGN VERTICAL BAR VIRAMA..MALAYALAM SIGN CIRCULAR VIRAMA
    Interval{ 0x0D41, 0x0D44 }, // Mn   [4] MALAYALAM VOWEL SIGN U..MALAYALAM VOWEL SIGN VOCALIC RR
    Interval{ 0x0D4D, 0x0D4D }, // Mn       MALAYALAM SIGN VIRAMA
    Interval{ 0x0D62, 0x0D63 }, // Mn   [2] MALAYALAM VOWEL SIGN VOCALIC L..MALAYALAM VOWEL SIGN VOCALIC LL
    Interval{ 0x0D81, 0x0D81 }, // Mn       SINHALA SIGN CANDRABINDU
    Interval{ 0x0DCA, 0x0DCA }, // Mn       SINHALA SIGN AL-LAKUNA
    Interval{ 0x0DD2, 0x0DD4 }, // Mn   [3] SINHALA VOWEL SIGN KETTI IS-PILLA..SINHALA VOWEL SIGN KETTI PAA-PILLA
    Interval{ 0x0DD6, 0x0DD6 }, // Mn       SINHALA VOWEL SIGN DIGA PAA-PILLA
    Interval{ 0x0E31, 0x0E31 }, // Mn       THAI CHARACTER MAI HAN-AKAT
    Interval{ 0x0E34, 0x0E3A }, // Mn   [7] THAI CHARACTER SARA I..THAI CHARACTER PHINTHU
    Interval{ 0x0E46, 0x0E46 }, // Lm       THAI CHARACTER MAIYAMOK
    Interval{ 0x0E47, 0x0E4E }, // Mn   [8] THAI CHARACTER MAITAIKHU..THAI CHARACTER YAMAKKAN
    Interval{ 0x0EB1, 0x0EB1 }, // Mn       LAO VOWEL SIGN MAI KAN
    Interval{ 0x0EB4, 0x0EBC }, // Mn   [9] LAO VOWEL SIGN I..LAO SEMIVOWEL SIGN LO
    Interval{ 0x0EC6, 0x0EC6 }, // Lm       LAO KO LA
    Interval{ 0x0EC8, 0x0ECE }, // Mn   [7] LAO TONE MAI EK..LAO YAMAKKAN
    Interval{ 0x0F18, 0x0F19 }, // Mn   [2] TIBETAN ASTROLOGICAL SIGN -KHYUD PA..TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS
    Interval{ 0x0F35, 0x0F35 }, // Mn       TIBETAN MARK NGAS BZUNG NYI ZLA
    Interval{ 0x0F37, 0x0F37 }, // Mn       TIBETAN MARK NGAS BZUNG SGOR RTAGS
    Interval{ 0x0F39, 0x0F39 }, // Mn       TIBETAN MARK TSA -PHRU
    Interval{ 0x0F71, 0x0F7E }, // Mn  [14] TIBETAN VOWEL SIGN AA..TIBETAN SIGN RJES SU NGA RO
    Interval{ 0x0F80, 0x0F84 }, // Mn   [5] TIBETAN VOWEL SIGN REVERSED I..TIBETAN MARK HALANTA
    Interval{ 0x0F86, 0x0F87 }, // Mn   [2] TIBETAN SIGN LCI RTAGS..TIBETAN SIGN YANG RTAGS
    Interval{ 0x0F8D, 0x0F97 }, // Mn  [11] TIBETAN SUBJOINED SIGN LCE TSA CAN..TIBETAN SUBJOINED LETTER JA
    Interval{ 0x0F99, 0x0FBC }, // Mn  [36] TIBETAN SUBJOINED LETTER NYA..TIBETAN SUBJOINED LETTER FIXED-FORM RA
    Interval{ 0x0FC6, 0x0FC6 }, // Mn       TIBETAN SYMBOL PADMA GDAN
    Interval{ 0x102D, 0x1030 }, // Mn   [4] MYANMAR VOWEL SIGN I..MYANMAR VOWEL SIGN UU
    Interval{ 0x1032, 0x1037 }, // Mn   [6] MYANMAR VOWEL SIGN AI..MYANMAR SIGN DOT BELOW
    Interval{ 0x1039, 0x103A }, // Mn   [2] MYANMAR SIGN VIRAMA..MYANMAR SIGN ASAT
    Interval{ 0x103D, 0x103E }, // Mn   [2] MYANMAR CONSONANT SIGN MEDIAL WA..MYANMAR CONSONANT SIGN MEDIAL HA
    Interval{ 0x1058, 0x1059 }, // Mn   [2] MYANMAR VOWEL SIGN VOCALIC L..MYANMAR VOWEL SIGN VOCALIC LL
    Interval{ 0x105E, 0x1060 }, // Mn   [3] MYANMAR CONSONANT SIGN MON MEDIAL NA..MYANMAR CONSONANT SIGN MON MEDIAL LA
    Interval{ 0x1071, 0x1074 }, // Mn   [4] MYANMAR VOWEL SIGN GEBA KAREN I..MYANMAR VOWEL SIGN KAYAH EE
    Interval{ 0x1082, 0x1082 }, // Mn       MYANMAR CONSONANT SIGN SHAN MEDIAL WA
    Interval{ 0x1085, 0x1086 }, // Mn   [2] MYANMAR VOWEL SIGN SHAN E ABOVE..MYANMAR VOWEL SIGN SHAN FINAL Y
    Interval{ 0x108D, 0x108D }, // Mn       MYANMAR SIGN SHAN COUNCIL EMPHATIC TONE
    Interval{ 0x109D, 0x109D }, // Mn       MYANMAR VOWEL SIGN AITON AI
    Interval{ 0x10FC, 0x10FC }, // Lm       MODIFIER LETTER GEORGIAN NAR
    Interval{ 0x135D, 0x135F }, // Mn   [3] ETHIOPIC COMBINING GEMINATION AND VOWEL LENGTH MARK..ETHIOPIC COMBINING GEMINATION MARK
    Interval{ 0x1712, 0x1714 }, // Mn   [3] TAGALOG VOWEL SIGN I..TAGALOG SIGN VIRAMA
    Interval{ 0x1732, 0x1733 }, // Mn   [2] HANUNOO VOWEL SIGN I..HANUNOO VOWEL SIGN U
    Interval{ 0x1752, 0x1753 }, // Mn   [2] BUHID VOWEL SIGN I..BUHID VOWEL SIGN U
    Interval{ 0x1772, 0x1773 }, // Mn   [2] TAGBANWA VOWEL SIGN I..TAGBANWA VOWEL SIGN U
    Interval{ 0x17B4, 0x17B5 }, // Mn   [2] KHMER VOWEL INHERENT AQ..KHMER VOWEL INHERENT AA
    Interval{ 0x17B7, 0x17BD }, // Mn   [7] KHMER VOWEL SIGN I..KHMER VOWEL SIGN UA
    Interval{ 0x17C6, 0x17C6 }, // Mn       KHMER SIGN NIKAHIT
    Interval{ 0x17C9, 0x17D3 }, // Mn  [11] KHMER SIGN MUUSIKATOAN..KHMER SIGN BATHAMASAT
    Interval{ 0x17D7, 0x17D7 }, // Lm       KHMER SIGN LEK TOO
    Interval{ 0x17DD, 0x17DD }, // Mn       KHMER SIGN ATTHACAN
    Interval{ 0x180B, 0x180D }, // Mn   [3] MONGOLIAN FREE VARIATION SELECTOR ONE..MONGOLIAN FREE VARIATION SELECTOR THREE
    Interval{ 0x180E, 0x180E }, // Cf       MONGOLIAN VOWEL SEPARATOR
    Interval{ 0x180F, 0x180F }, // Mn       MONGOLIAN FREE VARIATION SELECTOR FOUR
    Interval{ 0x1843, 0x1843 }, // Lm       MONGOLIAN LETTER TODO LONG VOWEL SIGN
    Interval{ 0x1885, 0x1886 }, // Mn   [2] MONGOLIAN LETTER ALI GALI BALUDA..MONGOLIAN LETTER ALI GALI THREE BALUDA
    Interval{ 0x18A9, 0x18A9 }, // Mn       MONGOLIAN LETTER ALI GALI DAGALGA
    Interval{ 0x1920, 0x1922 }, // Mn   [3] LIMBU VOWEL SIGN A..LIMBU VOWEL SIGN U
    Interval{ 0x1927, 0x1928 }, // Mn   [2] LIMBU VOWEL SIGN E..LIMBU VOWEL SIGN O
    Interval{ 0x1932, 0x1932 }, // Mn       LIMBU SMALL LETTER ANUSVARA
    Interval{ 0x1939, 0x193B }, // Mn   [3] LIMBU SIGN MUKPHRENG..LIMBU SIGN SA-I
    Interval{ 0x1A17, 0x1A18 }, // Mn   [2] BUGINESE VOWEL SIGN I..BUGINESE VOWEL SIGN U
    Interval{ 0x1A1B, 0x1A1B }, // Mn       BUGINESE VOWEL SIGN AE
    Interval{ 0x1A56, 0x1A56 }, // Mn       TAI THAM CONSONANT SIGN MEDIAL LA
    Interval{ 0x1A58, 0x1A5E }, // Mn   [7] TAI THAM SIGN MAI KANG LAI..TAI THAM CONSONANT SIGN SA
    Interval{ 0x1A60, 0x1A60 }, // Mn       TAI THAM SIGN SAKOT
    Interval{ 0x1A62, 0x1A62 }, // Mn       TAI THAM VOWEL SIGN MAI SAT
    Interval{ 0x1A65, 0x1A6C }, // Mn   [8] TAI THAM VOWEL SIGN I..TAI THAM VOWEL SIGN OA BELOW
    Interval{ 0x1A73, 0x1A7C }, // Mn  [10] TAI THAM VOWEL SIGN OA ABOVE..TAI THAM SIGN KHUEN-LUE KARAN
    Interval{ 0x1A7F, 0x1A7F }, // Mn       TAI THAM COMBINING CRYPTOGRAMMIC DOT
    Interval{ 0x1AA7, 0x1AA7 }, // Lm       TAI THAM SIGN MAI YAMOK
    Interval{ 0x1AB0, 0x1ABD }, // Mn  [14] COMBINING DOUBLED CIRCUMFLEX ACCENT..COMBINING PARENTHESES BELOW
    Interval{ 0x1ABE, 0x1ABE }, // Me       COMBINING PARENTHESES OVERLAY
    Interval{ 0x1ABF, 0x1ACE }, // Mn  [16] COMBINING LATIN SMALL LETTER W BELOW..COMBINING LATIN SMALL LETTER INSULAR T
    Interval{ 0x1B00, 0x1B03 }, // Mn   [4] BALINESE SIGN ULU RICEM..BALINESE SIGN SURANG
    Interval{ 0x1B34, 0x1B34 }, // Mn       BALINESE SIGN REREKAN
    Interval{ 0x1B36, 0x1B3A }, // Mn   [5] BALINESE VOWEL SIGN ULU..BALINESE VOWEL SIGN RA REPA
    Interval{ 0x1B3C, 0x1B3C }, // Mn       BALINESE VOWEL SIGN LA LENGA
    Interval{ 0x1B42, 0x1B42 }, // Mn       BALINESE VOWEL SIGN PEPET
    Interval{ 0x1B6B, 0x1B73 }, // Mn   [9] BALINESE MUSICAL SYMBOL COMBINING TEGEH..BALINESE MUSICAL SYMBOL COMBINING GONG
    Interval{ 0x1B80, 0x1B81 }, // Mn   [2] SUNDANESE SIGN PANYECEK..SUNDANESE SIGN PANGLAYAR
    Interval{ 0x1BA2, 0x1BA5 }, // Mn   [4] SUNDANESE CONSONANT SIGN PANYAKRA..SUNDANESE VOWEL SIGN PANYUKU
    Interval{ 0x1BA8, 0x1BA9 }, // Mn   [2] SUNDANESE VOWEL SIGN PAMEPET..SUNDANESE VOWEL SIGN PANEULEUNG
    Interval{ 0x1BAB, 0x1BAD }, // Mn   [3] SUNDANESE SIGN VIRAMA..SUNDANESE CONSONANT SIGN PASANGAN WA
    Interval{ 0x1BE6, 0x1BE6 }, // Mn       BATAK SIGN TOMPI
    Interval{ 0x1BE8, 0x1BE9 }, // Mn   [2] BATAK VOWEL SIGN PAKPAK E..BATAK VOWEL SIGN EE
    Interval{ 0x1BED, 0x1BED }, // Mn       BATAK VOWEL SIGN KARO O
    Interval{ 0x1BEF, 0x1BF1 }, // Mn   [3] BATAK VOWEL SIGN U FOR SIMALUNGUN SA..BATAK CONSONANT SIGN H
    Interval{ 0x1C2C, 0x1C33 }, // Mn   [8] LEPCHA VOWEL SIGN E..LEPCHA CONSONANT SIGN T
    Interval{ 0x1C36, 0x1C37 }, // Mn   [2] LEPCHA SIGN RAN..LEPCHA SIGN NUKTA
    Interval{ 0x1C78, 0x1C7D }, // Lm   [6] OL CHIKI MU TTUDDAG..OL CHIKI AHAD
    Interval{ 0x1CD0, 0x1CD2 }, // Mn   [3] VEDIC TONE KARSHANA..VEDIC TONE PRENKHA
    Interval{ 0x1CD4, 0x1CE0 }, // Mn  [13] VEDIC SIGN YAJURVEDIC MIDLINE SVARITA..VEDIC TONE RIGVEDIC KASHMIRI INDEPENDENT SVARITA
    Interval{ 0x1CE2, 0x1CE8 }, // Mn   [7] VEDIC SIGN VISARGA SVARITA..VEDIC SIGN VISARGA ANUDATTA WITH TAIL
    Interval{ 0x1CED, 0x1CED }, // Mn       VEDIC SIGN TIRYAK
    Interval{ 0x1CF4, 0x1CF4 }, // Mn       VEDIC TONE CANDRA ABOVE
    Interval{ 0x1CF8, 0x1CF9 }, // Mn   [2] VEDIC TONE RING ABOVE..VEDIC TONE DOUBLE RING ABOVE
    Interval{ 0x1D2C, 0x1D6A }, // Lm  [63] MODIFIER LETTER CAPITAL A..GREEK SUBSCRIPT SMALL LETTER CHI
    Interval{ 0x1D78, 0x1D78 }, // Lm       MODIFIER LETTER CYRILLIC EN
    Interval{ 0x1D9B, 0x1DBF }, // Lm  [37] MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER LETTER SMALL THETA
    Interval{ 0x1DC0, 0x1DFF }, // Mn  [64] COMBINING DOTTED GRAVE ACCENT..COMBINING RIGHT ARROWHEAD AND DOWN ARROWHEAD BELOW
    Interval{ 0x1FBD, 0x1FBD }, // Sk       GREEK KORONIS
    Interval{ 0x1FBF, 0x1FC1 }, // Sk   [3] GREEK PSILI..GREEK DIALYTIKA AND PERISPOMENI
    Interval{ 0x1FCD, 0x1FCF }, // Sk   [3] GREEK PSILI AND VARIA..GREEK PSILI AND PERISPOMENI
    Interval{ 0x1FDD, 0x1FDF }, // Sk   [3] GREEK DASIA AND VARIA..GREEK DASIA AND PERISPOMENI
    Interval{ 0x1FED, 0x1FEF }, // Sk   [3] GREEK DIALYTIKA AND VARIA..GREEK VARIA
    Interval{ 0x1FFD, 0x1FFE }, // Sk   [2] GREEK OXIA..GREEK DASIA
    Interval{ 0x200B, 0x200F }, // Cf   [5] ZERO WIDTH SPACE..RIGHT-TO-LEFT MARK
    Interval{ 0x2018, 0x2018 }, // Pi       LEFT SINGLE QUOTATION MARK
    Interval{ 0x2019, 0x2019 }, // Pf       RIGHT SINGLE QUOTATION MARK
    Interval{ 0x2024, 0x2024 }, // Po       ONE DOT LEADER
    Interval{ 0x2027, 0x2027 }, // Po       HYPHENATION POINT
    Interval{ 0x202A, 0x202E }, // Cf   [5] LEFT-TO-RIGHT EMBEDDING..RIGHT-TO-LEFT OVERRIDE
    Interval{ 0x2060, 0x2064 }, // Cf   [5] WORD JOINER..INVISIBLE PLUS
    Interval{ 0x2066, 0x206F }, // Cf  [10] LEFT-TO-RIGHT ISOLATE..NOMINAL DIGIT SHAPES
    Interval{ 0x2071, 0x2071 }, // Lm       SUPERSCRIPT LATIN SMALL LETTER I
    Interval{ 0x207F, 0x207F }, // Lm       SUPERSCRIPT LATIN SMALL LETTER N
    Interval{ 0x2090, 0x209C }, // Lm  [13] LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCRIPT SMALL LETTER T
    Interval{ 0x20D0, 0x20DC }, // Mn  [13] COMBINING LEFT HARPOON ABOVE..COMBINING FOUR DOTS ABOVE
    Interval{ 0x20DD, 0x20E0 }, // Me   [4] COMBINING ENCLOSING CIRCLE..COMBINING ENCLOSING CIRCLE BACKSLASH
    Interval{ 0x20E1, 0x20E1 }, // Mn       COMBINING LEFT RIGHT ARROW ABOVE
    Interval{ 0x20E2, 0x20E4 }, // Me   [3] COMBINING ENCLOSING SCREEN..COMBINING ENCLOSING UPWARD POINTING TRIANGLE
    Interval{ 0x20E5, 0x20F0 }, // Mn  [12] COMBINING REVERSE SOLIDUS OVERLAY..COMBINING ASTERISK ABOVE
    Interval{ 0x2C7C, 0x2C7D }, // Lm   [2] LATIN SUBSCRIPT SMALL LETTER J..MODIFIER LETTER CAPITAL V
    Interval{ 0x2CEF, 0x2CF1 }, // Mn   [3] COPTIC COMBINING NI ABOVE..COPTIC COMBINING SPIRITUS LENIS
    Interval{ 0x2D6F, 0x2D6F }, // Lm       TIFINAGH MODIFIER LETTER LABIALIZATION MARK
    Interval{ 0x2D7F, 0x2D7F }, // Mn       TIFINAGH CONSONANT JOINER
    Interval{ 0x2DE0, 0x2DFF }, // Mn  [32] COMBINING CYRILLIC LETTER BE..COMBINING CYRILLIC LETTER IOTIFIED BIG YUS
    Interval{ 0x2E2F, 0x2E2F }, // Lm       VERTICAL TILDE
    Interval{ 0x3005, 0x3005 }, // Lm       IDEOGRAPHIC ITERATION MARK
    Interval{ 0x302A, 0x302D }, // Mn   [4] IDEOGRAPHIC LEVEL TONE MARK..IDEOGRAPHIC ENTERING TONE MARK
    Interval{ 0x3031, 0x3035 }, // Lm   [5] VERTICAL KANA REPEAT MARK..VERTICAL KANA REPEAT MARK LOWER HALF
    Interval{ 0x303B, 0x303B }, // Lm       VERTICAL IDEOGRAPHIC ITERATION MARK
    Interval{ 0x3099, 0x309A }, // Mn   [2] COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK..COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    Interval{ 0x309B, 0x309C }, // Sk   [2] KATAKANA-HIRAGANA VOICED SOUND MARK..KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    Interval{ 0x309D, 0x309E }, // Lm   [2] HIRAGANA ITERATION MARK..HIRAGANA VOICED ITERATION MARK
    Interval{ 0x30FC, 0x30FE }, // Lm   [3] KATAKANA-HIRAGANA PROLONGED SOUND MARK..KATAKANA VOICED ITERATION MARK
    Interval{ 0xA015, 0xA015 }, // Lm       YI SYLLABLE WU
    Interval{ 0xA4F8, 0xA4FD }, // Lm   [6] LISU LETTER TONE MYA TI..LISU LETTER TONE MYA JEU
    Interval{ 0xA60C, 0xA60C }, // Lm       VAI SYLLABLE LENGTHENER
    Interval{ 0xA66F, 0xA66F }, // Mn       COMBINING CYRILLIC VZMET
    Interval{ 0xA670, 0xA672 }, // Me   [3] COMBINING CYRILLIC TEN MILLIONS SIGN..COMBINING CYRILLIC THOUSAND MILLIONS SIGN
    Interval{ 0xA674, 0xA67D }, // Mn  [10] COMBINING CYRILLIC LETTER UKRAINIAN IE..COMBINING CYRILLIC PAYEROK
    Interval{ 0xA67F, 0xA67F }, // Lm       CYRILLIC PAYEROK
    Interval{ 0xA69C, 0xA69D }, // Lm   [2] MODIFIER LETTER CYRILLIC HARD SIGN..MODIFIER LETTER CYRILLIC SOFT SIGN
    Interval{ 0xA69E, 0xA69F }, // Mn   [2] COMBINING CYRILLIC LETTER EF..COMBINING CYRILLIC LETTER IOTIFIED E
    Interval{ 0xA6F0, 0xA6F1 }, // Mn   [2] BAMUM COMBINING MARK KOQNDON..BAMUM COMBINING MARK TUKWENTIS
    Interval{ 0xA700, 0xA716 }, // Sk  [23] MODIFIER LETTER CHINESE TONE YIN PING..MODIFIER LETTER EXTRA-LOW LEFT-STEM TONE BAR
    Interval{ 0xA717, 0xA71F }, // Lm   [9] MODIFIER LETTER DOT VERTICAL BAR..MODIFIER LETTER LOW INVERTED EXCLAMATION MARK
    Interval{ 0xA720, 0xA721 }, // Sk   [2] MODIFIER LETTER STRESS AND HIGH TONE..MODIFIER LETTER STRESS AND LOW TONE
    Interval{ 0xA770, 0xA770 }, // Lm       MODIFIER LETTER US
    Interval{ 0xA788, 0xA788 }, // Lm       MODIFIER LETTER LOW CIRCUMFLEX ACCENT
    Interval{ 0xA789, 0xA78A }, // Sk   [2] MODIFIER LETTER COLON..MODIFIER LETTER SHORT EQUALS SIGN
    Interval{ 0xA7F2, 0xA7F4 }, // Lm   [3] MODIFIER LETTER CAPITAL C..MODIFIER LETTER CAPITAL Q
    Interval{ 0xA7F8, 0xA7F9 }, // Lm   [2] MODIFIER LETTER CAPITAL H WITH STROKE..MODIFIER LETTER SMALL LIGATURE OE
    Interval{ 0xA802, 0xA802 }, // Mn       SYLOTI NAGRI SIGN DVISVARA
    Interval{ 0xA806, 0xA806 }, // Mn       SYLOTI NAGRI SIGN HASANTA
    Interval{ 0xA80B, 0xA80B }, // Mn       SYLOTI NAGRI SIGN ANUSVARA
    Interval{ 0xA825, 0xA826 }, // Mn   [2] SYLOTI NAGRI VOWEL SIGN U..SYLOTI NAGRI VOWEL SIGN E
    Interval{ 0xA82C, 0xA82C }, // Mn       SYLOTI NAGRI SIGN ALTERNATE HASANTA
    Interval{ 0xA8C4, 0xA8C5 }, // Mn   [2] SAURASHTRA SIGN VIRAMA..SAURASHTRA SIGN CANDRABINDU
    Interval{ 0xA8E0, 0xA8F1 }, // Mn  [18] COMBINING DEVANAGARI DIGIT ZERO..COMBINING DEVANAGARI SIGN AVAGRAHA
    Interval{ 0xA8FF, 0xA8FF }, // Mn       DEVANAGARI VOWEL SIGN AY
    Interval{ 0xA926, 0xA92D }, // Mn   [8] KAYAH LI VOWEL UE..KAYAH LI TONE CALYA PLOPHU
    Interval{ 0xA947, 0xA951 }, // Mn  [11] REJANG VOWEL SIGN I..REJANG CONSONANT SIGN R
    Interval{ 0xA980, 0xA982 }, // Mn   [3] JAVANESE SIGN PANYANGGA..JAVANESE SIGN LAYAR
    Interval{ 0xA9B3, 0xA9B3 }, // Mn       JAVANESE SIGN CECAK TELU
    Interval{ 0xA9B6, 0xA9B9 }, // Mn   [4] JAVANESE VOWEL SIGN WULU..JAVANESE VOWEL SIGN SUKU MENDUT
    Interval{ 0xA9BC, 0xA9BD }, // Mn   [2] JAVANESE VOWEL SIGN PEPET..JAVANESE CONSONANT SIGN KERET
    Interval{ 0xA9CF, 0xA9CF }, // Lm       JAVANESE PANGRANGKEP
    Interval{ 0xA9E5, 0xA9E5 }, // Mn       MYANMAR SIGN SHAN SAW
    Interval{ 0xA9E6, 0xA9E6 }, // Lm       MYANMAR MODIFIER LETTER SHAN REDUPLICATION
    Interval{ 0xAA29, 0xAA2E }, // Mn   [6] CHAM VOWEL SIGN AA..CHAM VOWEL SIGN OE
    Interval{ 0xAA31, 0xAA32 }, // Mn   [2] CHAM VOWEL SIGN AU..CHAM VOWEL SIGN UE
    Interval{ 0xAA35, 0xAA36 }, // Mn   [2] CHAM CONSONANT SIGN LA..CHAM CONSONANT SIGN WA
    Interval{ 0xAA43, 0xAA43 }, // Mn       CHAM CONSONANT SIGN FINAL NG
    Interval{ 0xAA4C, 0xAA4C }, // Mn       CHAM CONSONANT SIGN FINAL M
    Interval{ 0xAA70, 0xAA70 }, // Lm       MYANMAR MODIFIER LETTER KHAMTI REDUPLICATION
    Interval{ 0xAA7C, 0xAA7C }, // Mn       MYANMAR SIGN TAI LAING TONE-2
    Interval{ 0xAAB0, 0xAAB0 }, // Mn       TAI VIET MAI KANG
    Interval{ 0xAAB2, 0xAAB4 }, // Mn   [3] TAI VIET VOWEL I..TAI VIET VOWEL U
    Interval{ 0xAAB7, 0xAAB8 }, // Mn   [2] TAI VIET MAI KHIT..TAI VIET VOWEL IA
    Interval{ 0xAABE, 0xAABF }, // Mn   [2] TAI VIET VOWEL AM..TAI VIET TONE MAI EK
    Interval{ 0xAAC1, 0xAAC1 }, // Mn       TAI VIET TONE MAI THO
    Interval{ 0xAADD, 0xAADD }, // Lm       TAI VIET SYMBOL SAM
    Interval{ 0xAAEC, 0xAAED }, // Mn   [2] MEETEI MAYEK VOWEL SIGN UU..MEETEI MAYEK VOWEL SIGN AAI
    Interval{ 0xAAF3, 0xAAF4 }, // Lm   [2] MEETEI MAYEK SYLLABLE REPETITION MARK..MEETEI MAYEK WORD REPETITION MARK
    Interval{ 0xAAF6, 0xAAF6 }, // Mn       MEETEI MAYEK VIRAMA
    Interval{ 0xAB5B, 0xAB5B }, // Sk       MODIFIER BREVE WITH INVERTED BREVE
    Interval{ 0xAB5C, 0xAB5F }, // Lm   [4] MODIFIER LETTER SMALL HENG..MODIFIER LETTER SMALL U WITH LEFT HOOK
    Interval{ 0xAB69, 0xAB69 }, // Lm       MODIFIER LETTER SMALL TURNED W
    Interval{ 0xAB6A, 0xAB6B }, // Sk   [2] MODIFIER LETTER LEFT TACK..MODIFIER LETTER RIGHT TACK
    Interval{ 0xABE5, 0xABE5 }, // Mn       MEETEI MAYEK VOWEL SIGN ANAP
    Interval{ 0xABE8, 0xABE8 }, // Mn       MEETEI MAYEK VOWEL SIGN UNAP
    Interval{ 0xABED, 0xABED }, // Mn       MEETEI MAYEK APUN IYEK
    Interval{ 0xFB1E, 0xFB1E }, // Mn       HEBREW POINT JUDEO-SPANISH VARIKA
    Interval{ 0xFBB2, 0xFBC2 }, // Sk  [17] ARABIC SYMBOL DOT ABOVE..ARABIC SYMBOL WASLA ABOVE
    Interval{ 0xFE00, 0xFE0F }, // Mn  [16] VARIATION SELECTOR-1..VARIATION SELECTOR-16
    Interval{ 0xFE13, 0xFE13 }, // Po       PRESENTATION FORM FOR VERTICAL COLON
    Interval{ 0xFE20, 0xFE2F }, // Mn  [16] COMBINING LIGATURE LEFT HALF..COMBINING CYRILLIC TITLO RIGHT HALF
    Interval{ 0xFE52, 0xFE52 }, // Po       SMALL FULL STOP
    Interval{ 0xFE55, 0xFE55 }, // Po       SMALL COLON
    Interval{ 0xFEFF, 0xFEFF }, // Cf       ZERO WIDTH NO-BREAK SPACE
    Interval{ 0xFF07, 0xFF07 }, // Po       FULLWIDTH APOSTROPHE
    Interval{ 0xFF0E, 0xFF0E }, // Po       FULLWIDTH FULL STOP
    Interval{ 0xFF1A, 0xFF1A }, // Po       FULLWIDTH COLON
    Interval{ 0xFF3E, 0xFF3E }, // Sk       FULLWIDTH CIRCUMFLEX ACCENT
    Interval{ 0xFF40, 0xFF40 }, // Sk       FULLWIDTH GRAVE ACCENT
    Interval{ 0xFF70, 0xFF70 }, // Lm       HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
    Interval{ 0xFF9E, 0xFF9F }, // Lm   [2] HALFWIDTH KATAKANA VOICED SOUND MARK..HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
    Interval{ 0xFFE3, 0xFFE3 }, // Sk       FULLWIDTH MACRON
    Interval{ 0xFFF9, 0xFFFB }, // Cf   [3] INTERLINEAR ANNOTATION ANCHOR..INTERLINEAR ANNOTATION TERMINATOR
    Interval{ 0x101FD, 0x101FD }, // Mn       PHAISTOS DISC SIGN COMBINING OBLIQUE STROKE
    Interval{ 0x102E0, 0x102E0 }, // Mn       COPTIC EPACT THOUSANDS MARK
    Interval{ 0x10376, 0x1037A }, // Mn   [5] COMBINING OLD PERMIC LETTER AN..COMBINING OLD PERMIC LETTER SII
    Interval{ 0x10780, 0x10785 }, // Lm   [6] MODIFIER LETTER SMALL CAPITAL AA..MODIFIER LETTER SMALL B WITH HOOK
    Interval{ 0x10787, 0x107B0 }, // Lm  [42] MODIFIER LETTER SMALL DZ DIGRAPH..MODIFIER LETTER SMALL V WITH RIGHT HOOK
    Interval{ 0x107B2, 0x107BA }, // Lm   [9] MODIFIER LETTER SMALL CAPITAL Y..MODIFIER LETTER SMALL S WITH CURL
    Interval{ 0x10A01, 0x10A03 }, // Mn   [3] KHAROSHTHI VOWEL SIGN I..KHAROSHTHI VOWEL SIGN VOCALIC R
    Interval{ 0x10A05, 0x10A06 }, // Mn   [2] KHAROSHTHI VOWEL SIGN E..KHAROSHTHI VOWEL SIGN O
    Interval{ 0x10A0C, 0x10A0F }, // Mn   [4] KHAROSHTHI VOWEL LENGTH MARK..KHAROSHTHI SIGN VISARGA
    Interval{ 0x10A38, 0x10A3A }, // Mn   [3] KHAROSHTHI SIGN BAR ABOVE..KHAROSHTHI SIGN DOT BELOW
    Interval{ 0x10A3F, 0x10A3F }, // Mn       KHAROSHTHI VIRAMA
    Interval{ 0x10AE5, 0x10AE6 }, // Mn   [2] MANICHAEAN ABBREVIATION MARK ABOVE..MANICHAEAN ABBREVIATION MARK BELOW
    Interval{ 0x10D24, 0x10D27 }, // Mn   [4] HANIFI ROHINGYA SIGN HARBAHAY..HANIFI ROHINGYA SIGN TASSI
    Interval{ 0x10EAB, 0x10EAC }, // Mn   [2] YEZIDI COMBINING HAMZA MARK..YEZIDI COMBINING MADDA MARK
    Interval{ 0x10EFD, 0x10EFF }, // Mn   [3] ARABIC SMALL LOW WORD SAKTA..ARABIC SMALL LOW WORD MADDA
    Interval{ 0x10F46, 0x10F50 }, // Mn  [11] SOGDIAN COMBINING DOT BELOW..SOGDIAN COMBINING STROKE BELOW
    Interval{ 0x10F82, 0x10F85 }, // Mn   [4] OLD UYGHUR COMBINING DOT ABOVE..OLD UYGHUR COMBINING TWO DOTS BELOW
    Interval{ 0x11001, 0x11001 }, // Mn       BRAHMI SIGN ANUSVARA
    Interval{ 0x11038, 0x11046 }, // Mn  [15] BRAHMI VOWEL SIGN AA..BRAHMI VIRAMA
    Interval{ 0x11070, 0x11070 }, // Mn       BRAHMI SIGN OLD TAMIL VIRAMA
    Interval{ 0x11073, 0x11074 }, // Mn   [2] BRAHMI VOWEL SIGN OLD TAMIL SHORT E..BRAHMI VOWEL SIGN OLD TAMIL SHORT O
    Interval{ 0x1107F, 0x11081 }, // Mn   [3] BRAHMI NUMBER JOINER..KAITHI SIGN ANUSVARA
    Interval{ 0x110B3, 0x110B6 }, // Mn   [4] KAITHI VOWEL SIGN U..KAITHI VOWEL SIGN AI
    Interval{ 0x110B9, 0x110BA }, // Mn   [2] KAITHI SIGN VIRAMA..KAITHI SIGN NUKTA
    Interval{ 0x110BD, 0x110BD }, // Cf       KAITHI NUMBER SIGN
    Interval{ 0x110C2, 0x110C2 }, // Mn       KAITHI VOWEL SIGN VOCALIC R
    Interval{ 0x110CD, 0x110CD }, // Cf       KAITHI NUMBER SIGN ABOVE
    Interval{ 0x11100, 0x11102 }, // Mn   [3] CHAKMA SIGN CANDRABINDU..CHAKMA SIGN VISARGA
    Interval{ 0x11127, 0x1112B }, // Mn   [5] CHAKMA VOWEL SIGN A..CHAKMA VOWEL SIGN UU
    Interval{ 0x1112D, 0x11134 }, // Mn   [8] CHAKMA VOWEL SIGN AI..CHAKMA MAAYYAA
    Interval{ 0x11173, 0x11173 }, // Mn       MAHAJANI SIGN NUKTA
    Interval{ 0x11180, 0x11181 }, // Mn   [2] SHARADA SIGN CANDRABINDU..SHARADA SIGN ANUSVARA
    Interval{ 0x111B6, 0x111BE }, // Mn   [9] SHARADA VOWEL SIGN U..SHARADA VOWEL SIGN O
    Interval{ 0x111C9, 0x111CC }, // Mn   [4] SHARADA SANDHI MARK..SHARADA EXTRA SHORT VOWEL MARK
    Interval{ 0x111CF, 0x111CF }, // Mn       SHARADA SIGN INVERTED CANDRABINDU
    Interval{ 0x1122F, 0x11231 }, // Mn   [3] KHOJKI VOWEL SIGN U..KHOJKI VOWEL SIGN AI
    Interval{ 0x11234, 0x11234 }, // Mn       KHOJKI SIGN ANUSVARA
    Interval{ 0x11236, 0x11237 }, // Mn   [2] KHOJKI SIGN NUKTA..KHOJKI SIGN SHADDA
    Interval{ 0x1123E, 0x1123E }, // Mn       KHOJKI SIGN SUKUN
    Interval{ 0x11241, 0x11241 }, // Mn       KHOJKI VOWEL SIGN VOCALIC R
    Interval{ 0x112DF, 0x112DF }, // Mn       KHUDAWADI SIGN ANUSVARA
    Interval{ 0x112E3, 0x112EA }, // Mn   [8] KHUDAWADI VOWEL SIGN U..KHUDAWADI SIGN VIRAMA
    Interval{ 0x11300, 0x11301 }, // Mn   [2] GRANTHA SIGN COMBINING ANUSVARA ABOVE..GRANTHA SIGN CANDRABINDU
    Interval{ 0x1133B, 0x1133C }, // Mn   [2] COMBINING BINDU BELOW..GRANTHA SIGN NUKTA
    Interval{ 0x11340, 0x11340 }, // Mn       GRANTHA VOWEL SIGN II
    Interval{ 0x11366, 0x1136C }, // Mn   [7] COMBINING GRANTHA DIGIT ZERO..COMBINING GRANTHA DIGIT SIX
    Interval{ 0x11370, 0x11374 }, // Mn   [5] COMBINING GRANTHA LETTER A..COMBINING GRANTHA LETTER PA
    Interval{ 0x11438, 0x1143F }, // Mn   [8] NEWA VOWEL SIGN U..NEWA VOWEL SIGN AI
    Interval{ 0x11442, 0x11444 }, // Mn   [3] NEWA SIGN VIRAMA..NEWA SIGN ANUSVARA
    Interval{ 0x11446, 0x11446 }, // Mn       NEWA SIGN NUKTA
    Interval{ 0x1145E, 0x1145E }, // Mn       NEWA SANDHI MARK
    Interval{ 0x114B3, 0x114B8 }, // Mn   [6] TIRHUTA VOWEL SIGN U..TIRHUTA VOWEL SIGN VOCALIC LL
    Interval{ 0x114BA, 0x114BA }, // Mn       TIRHUTA VOWEL SIGN SHORT E
    Interval{ 0x114BF, 0x114C0 }, // Mn   [2] TIRHUTA SIGN CANDRABINDU..TIRHUTA SIGN ANUSVARA
    Interval{ 0x114C2, 0x114C3 }, // Mn   [2] TIRHUTA SIGN VIRAMA..TIRHUTA SIGN NUKTA
    Interval{ 0x115B2, 0x115B5 }, // Mn   [4] SIDDHAM VOWEL SIGN U..SIDDHAM VOWEL SIGN VOCALIC RR
    Interval{ 0x115BC, 0x115BD }, // Mn   [2] SIDDHAM SIGN CANDRABINDU..SIDDHAM SIGN ANUSVARA
    Interval{ 0x115BF, 0x115C0 }, // Mn   [2] SIDDHAM SIGN VIRAMA..SIDDHAM SIGN NUKTA
    Interval{ 0x115DC, 0x115DD }, // Mn   [2] SIDDHAM VOWEL SIGN ALTERNATE U..SIDDHAM VOWEL SIGN ALTERNATE UU
    Interval{ 0x11633, 0x1163A }, // Mn   [8] MODI VOWEL SIGN U..MODI VOWEL SIGN AI
    Interval{ 0x1163D, 0x1163D }, // Mn       MODI SIGN ANUSVARA
    Interval{ 0x1163F, 0x11640 }, // Mn   [2] MODI SIGN VIRAMA..MODI SIGN ARDHACANDRA
    Interval{ 0x116AB, 0x116AB }, // Mn       TAKRI SIGN ANUSVARA
    Interval{ 0x116AD, 0x116AD }, // Mn       TAKRI VOWEL SIGN AA
    Interval{ 0x116B0, 0x116B5 }, // Mn   [6] TAKRI VOWEL SIGN U..TAKRI VOWEL SIGN AU
    Interval{ 0x116B7, 0x116B7 }, // Mn       TAKRI SIGN NUKTA
    Interval{ 0x1171D, 0x1171F }, // Mn   [3] AHOM CONSONANT SIGN MEDIAL LA..AHOM CONSONANT SIGN MEDIAL LIGATING RA
    Interval{ 0x11722, 0x11725 }, // Mn   [4] AHOM VOWEL SIGN I..AHOM VOWEL SIGN UU
    Interval{ 0x11727, 0x1172B }, // Mn   [5] AHOM VOWEL SIGN AW..AHOM SIGN KILLER
    Interval{ 0x1182F, 0x11837 }, // Mn   [9] DOGRA VOWEL SIGN U..DOGRA SIGN ANUSVARA
    Interval{ 0x11839, 0x1183A }, // Mn   [2] DOGRA SIGN VIRAMA..DOGRA SIGN NUKTA
    Interval{ 0x1193B, 0x1193C }, // Mn   [2] DIVES AKURU SIGN ANUSVARA..DIVES AKURU SIGN CANDRABINDU
    Interval{ 0x1193E, 0x1193E }, // Mn       DIVES AKURU VIRAMA
    Interval{ 0x11943, 0x11943 }, // Mn       DIVES AKURU SIGN NUKTA
    Interval{ 0x119D4, 0x119D7 }, // Mn   [4] NANDINAGARI VOWEL SIGN U..NANDINAGARI VOWEL SIGN VOCALIC RR
    Interval{ 0x119DA, 0x119DB }, // Mn   [2] NANDINAGARI VOWEL SIGN E..NANDINAGARI VOWEL SIGN AI
    Interval{ 0x119E0, 0x119E0 }, // Mn       NANDINAGARI SIGN VIRAMA
    Interval{ 0x11A01, 0x11A0A }, // Mn  [10] ZANABAZAR SQUARE VOWEL SIGN I..ZANABAZAR SQUARE VOWEL LENGTH MARK
    Interval{ 0x11A33, 0x11A38 }, // Mn   [6] ZANABAZAR SQUARE FINAL CONSONANT MARK..ZANABAZAR SQUARE SIGN ANUSVARA
    Interval{ 0x11A3B, 0x11A3E }, // Mn   [4] ZANABAZAR SQUARE CLUSTER-FINAL LETTER YA..ZANABAZAR SQUARE CLUSTER-FINAL LETTER VA
    Interval{ 0x11A47, 0x11A47 }, // Mn       ZANABAZAR SQUARE SUBJOINER
    Interval{ 0x11A51, 0x11A56 }, // Mn   [6] SOYOMBO VOWEL SIGN I..SOYOMBO VOWEL SIGN OE
    Interval{ 0x11A59, 0x11A5B }, // Mn   [3] SOYOMBO VOWEL SIGN VOCALIC R..SOYOMBO VOWEL LENGTH MARK
    Interval{ 0x11A8A, 0x11A96 }, // Mn  [13] SOYOMBO FINAL CONSONANT SIGN G..SOYOMBO SIGN ANUSVARA
    Interval{ 0x11A98, 0x11A99 }, // Mn   [2] SOYOMBO GEMINATION MARK..SOYOMBO SUBJOINER
    Interval{ 0x11C30, 0x11C36 }, // Mn   [7] BHAIKSUKI VOWEL SIGN I..BHAIKSUKI VOWEL SIGN VOCALIC L
    Interval{ 0x11C38, 0x11C3D }, // Mn   [6] BHAIKSUKI VOWEL SIGN E..BHAIKSUKI SIGN ANUSVARA
    Interval{ 0x11C3F, 0x11C3F }, // Mn       BHAIKSUKI SIGN VIRAMA
    Interval{ 0x11C92, 0x11CA7 }, // Mn  [22] MARCHEN SUBJOINED LETTER KA..MARCHEN SUBJOINED LETTER ZA
    Interval{ 0x11CAA, 0x11CB0 }, // Mn   [7] MARCHEN SUBJOINED LETTER RA..MARCHEN VOWEL SIGN AA
    Interval{ 0x11CB2, 0x11CB3 }, // Mn   [2] MARCHEN VOWEL SIGN U..MARCHEN VOWEL SIGN E
    Interval{ 0x11CB5, 0x11CB6 }, // Mn   [2] MARCHEN SIGN ANUSVARA..MARCHEN SIGN CANDRABINDU
    Interval{ 0x11D31, 0x11D36 }, // Mn   [6] MASARAM GONDI VOWEL SIGN AA..MASARAM GONDI VOWEL SIGN VOCALIC R
    Interval{ 0x11D3A, 0x11D3A }, // Mn       MASARAM GONDI VOWEL SIGN E
    Interval{ 0x11D3C, 0x11D3D }, // Mn   [2] MASARAM GONDI VOWEL SIGN AI..MASARAM GONDI VOWEL SIGN O
    Interval{ 0x11D3F, 0x11D45 }, // Mn   [7] MASARAM GONDI VOWEL SIGN AU..MASARAM GONDI VIRAMA
    Interval{ 0x11D47, 0x11D47 }, // Mn       MASARAM GONDI RA-KARA
    Interval{ 0x11D90, 0x11D91 }, // Mn   [2] GUNJALA GONDI VOWEL SIGN EE..GUNJALA GONDI VOWEL SIGN AI
    Interval{ 0x11D95, 0x11D95 }, // Mn       GUNJALA GONDI SIGN ANUSVARA
    Interval{ 0x11D97, 0x11D97 }, // Mn       GUNJALA GONDI VIRAMA
    Interval{ 0x11EF3, 0x11EF4 }, // Mn   [2] MAKASAR VOWEL SIGN I..MAKASAR VOWEL SIGN U
    Interval{ 0x11F00, 0x11F01 }, // Mn   [2] KAWI SIGN CANDRABINDU..KAWI SIGN ANUSVARA
    Interval{ 0x11F36, 0x11F3A }, // Mn   [5] KAWI VOWEL SIGN I..KAWI VOWEL SIGN VOCALIC R
    Interval{ 0x11F40, 0x11F40 }, // Mn       KAWI VOWEL SIGN EU
    Interval{ 0x11F42, 0x11F42 }, // Mn       KAWI CONJOINER
    Interval{ 0x13430, 0x1343F }, // Cf  [16] EGYPTIAN HIEROGLYPH VERTICAL JOINER..EGYPTIAN HIEROGLYPH END WALLED ENCLOSURE
    Interval{ 0x13440, 0x13440 }, // Mn       EGYPTIAN HIEROGLYPH MIRROR HORIZONTALLY
    Interval{ 0x13447, 0x13455 }, // Mn  [15] EGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT TOP START..EGYPTIAN HIEROGLYPH MODIFIER DAMAGED
    Interval{ 0x16AF0, 0x16AF4 }, // Mn   [5] BASSA VAH COMBINING HIGH TONE..BASSA VAH COMBINING HIGH-LOW TONE
    Interval{ 0x16B30, 0x16B36 }, // Mn   [7] PAHAWH HMONG MARK CIM TUB..PAHAWH HMONG MARK CIM TAUM
    Interval{ 0x16B40, 0x16B43 }, // Lm   [4] PAHAWH HMONG SIGN VOS SEEV..PAHAWH HMONG SIGN IB YAM
    Interval{ 0x16F4F, 0x16F4F }, // Mn       MIAO SIGN CONSONANT MODIFIER BAR
    Interval{ 0x16F8F, 0x16F92 }, // Mn   [4] MIAO TONE RIGHT..MIAO TONE BELOW
    Interval{ 0x16F93, 0x16F9F }, // Lm  [13] MIAO LETTER TONE-2..MIAO LETTER REFORMED TONE-8
    Interval{ 0x16FE0, 0x16FE1 }, // Lm   [2] TANGUT ITERATION MARK..NUSHU ITERATION MARK
    Interval{ 0x16FE3, 0x16FE3 }, // Lm       OLD CHINESE ITERATION MARK
    Interval{ 0x16FE4, 0x16FE4 }, // Mn       KHITAN SMALL SCRIPT FILLER
    Interval{ 0x1AFF0, 0x1AFF3 }, // Lm   [4] KATAKANA LETTER MINNAN TONE-2..KATAKANA LETTER MINNAN TONE-5
    Interval{ 0x1AFF5, 0x1AFFB }, // Lm   [7] KATAKANA LETTER MINNAN TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-5
    Interval{ 0x1AFFD, 0x1AFFE }, // Lm   [2] KATAKANA LETTER MINNAN NASALIZED TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-8
    Interval{ 0x1BC9D, 0x1BC9E }, // Mn   [2] DUPLOYAN THICK LETTER SELECTOR..DUPLOYAN DOUBLE MARK
    Interval{ 0x1BCA0, 0x1BCA3 }, // Cf   [4] SHORTHAND FORMAT LETTER OVERLAP..SHORTHAND FORMAT UP STEP
    Interval{ 0x1CF00, 0x1CF2D }, // Mn  [46] ZNAMENNY COMBINING MARK GORAZDO NIZKO S KRYZHEM ON LEFT..ZNAMENNY COMBINING MARK KRYZH ON LEFT
    Interval{ 0x1CF30, 0x1CF46 }, // Mn  [23] ZNAMENNY COMBINING TONAL RANGE MARK MRACHNO..ZNAMENNY PRIZNAK MODIFIER ROG
    Interval{ 0x1D167, 0x1D169 }, // Mn   [3] MUSICAL SYMBOL COMBINING TREMOLO-1..MUSICAL SYMBOL COMBINING TREMOLO-3
    Interval{ 0x1D173, 0x1D17A }, // Cf   [8] MUSICAL SYMBOL BEGIN BEAM..MUSICAL SYMBOL END PHRASE
    Interval{ 0x1D17B, 0x1D182 }, // Mn   [8] MUSICAL SYMBOL COMBINING ACCENT..MUSICAL SYMBOL COMBINING LOURE
    Interval{ 0x1D185, 0x1D18B }, // Mn   [7] MUSICAL SYMBOL COMBINING DOIT..MUSICAL SYMBOL COMBINING TRIPLE TONGUE
    Interval{ 0x1D1AA, 0x1D1AD }, // Mn   [4] MUSICAL SYMBOL COMBINING DOWN BOW..MUSICAL SYMBOL COMBINING SNAP PIZZICATO
    Interval{ 0x1D242, 0x1D244 }, // Mn   [3] COMBINING GREEK MUSICAL TRISEME..COMBINING GREEK MUSICAL PENTASEME
    Interval{ 0x1DA00, 0x1DA36 }, // Mn  [55] SIGNWRITING HEAD RIM..SIGNWRITING AIR SUCKING IN
    Interval{ 0x1DA3B, 0x1DA6C }, // Mn  [50] SIGNWRITING MOUTH CLOSED NEUTRAL..SIGNWRITING EXCITEMENT
    Interval{ 0x1DA75, 0x1DA75 }, // Mn       SIGNWRITING UPPER BODY TILTING FROM HIP JOINTS
    Interval{ 0x1DA84, 0x1DA84 }, // Mn       SIGNWRITING LOCATION HEAD NECK
    Interval{ 0x1DA9B, 0x1DA9F }, // Mn   [5] SIGNWRITING FILL MODIFIER-2..SIGNWRITING FILL MODIFIER-6
    Interval{ 0x1DAA1, 0x1DAAF }, // Mn  [15] SIGNWRITING ROTATION MODIFIER-2..SIGNWRITING ROTATION MODIFIER-16
    Interval{ 0x1E000, 0x1E006 }, // Mn   [7] COMBINING GLAGOLITIC LETTER AZU..COMBINING GLAGOLITIC LETTER ZHIVETE
    Interval{ 0x1E008, 0x1E018 }, // Mn  [17] COMBINING GLAGOLITIC LETTER ZEMLJA..COMBINING GLAGOLITIC LETTER HERU
    Interval{ 0x1E01B, 0x1E021 }, // Mn   [7] COMBINING GLAGOLITIC LETTER SHTA..COMBINING GLAGOLITIC LETTER YATI
    Interval{ 0x1E023, 0x1E024 }, // Mn   [2] COMBINING GLAGOLITIC LETTER YU..COMBINING GLAGOLITIC LETTER SMALL YUS
    Interval{ 0x1E026, 0x1E02A }, // Mn   [5] COMBINING GLAGOLITIC LETTER YO..COMBINING GLAGOLITIC LETTER FITA
    Interval{ 0x1E030, 0x1E06D }, // Lm  [62] MODIFIER LETTER CYRILLIC SMALL A..MODIFIER LETTER CYRILLIC SMALL STRAIGHT U WITH STROKE
    Interval{ 0x1E08F, 0x1E08F }, // Mn       COMBINING CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
    Interval{ 0x1E130, 0x1E136 }, // Mn   [7] NYIAKENG PUACHUE HMONG TONE-B..NYIAKENG PUACHUE HMONG TONE-D
    Interval{ 0x1E137, 0x1E13D }, // Lm   [7] NYIAKENG PUACHUE HMONG SIGN FOR PERSON..NYIAKENG PUACHUE HMONG SYLLABLE LENGTHENER
    Interval{ 0x1E2AE, 0x1E2AE }, // Mn       TOTO SIGN RISING TONE
    Interval{ 0x1E2EC, 0x1E2EF }, // Mn   [4] WANCHO TONE TUP..WANCHO TONE KOINI
    Interval{ 0x1E4EB, 0x1E4EB }, // Lm       NAG MUNDARI SIGN OJOD
    Interval{ 0x1E4EC, 0x1E4EF }, // Mn   [4] NAG MUNDARI SIGN MUHOR..NAG MUNDARI SIGN SUTUH
    Interval{ 0x1E8D0, 0x1E8D6 }, // Mn   [7] MENDE KIKAKUI COMBINING NUMBER TEENS..MENDE KIKAKUI COMBINING NUMBER MILLIONS
    Interval{ 0x1E944, 0x1E94A }, // Mn   [7] ADLAM ALIF LENGTHENER..ADLAM NUKTA
    Interval{ 0x1E94B, 0x1E94B }, // Lm       ADLAM NASALIZATION MARK
    Interval{ 0x1F3FB, 0x1F3FF }, // Sk   [5] EMOJI MODIFIER FITZPATRICK TYPE-1-2..EMOJI MODIFIER FITZPATRICK TYPE-6
    Interval{ 0xE0001, 0xE0001 }, // Cf       LANGUAGE TAG
    Interval{ 0xE0020, 0xE007F }, // Cf  [96] TAG SPACE..CANCEL TAG
    Interval{ 0xE0100, 0xE01EF }, // Mn [240] VARIATION SELECTOR-17..VARIATION SELECTOR-256
}; // }}}
auto static const Cased = std::array<Interval, 174>{ // {{{
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00AA, 0x00AA }, // Lo       FEMININE ORDINAL INDICATOR
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00BA, 0x00BA }, // Lo       MASCULINE ORDINAL INDICATOR
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00F6 }, // L&  [31] LATIN CAPITAL LETTER O WITH STROKE..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x01BA }, // L& [195] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER EZH WITH TAIL
    Interval{ 0x01BC, 0x01BF }, // L&   [4] LATIN CAPITAL LETTER TONE FIVE..LATIN LETTER WYNN
    Interval{ 0x01C4, 0x0293 }, // L& [208] LATIN CAPITAL LETTER DZ WITH CARON..LATIN SMALL LETTER EZH WITH CURL
    Interval{ 0x0295, 0x02AF }, // L&  [27] LATIN LETTER PHARYNGEAL VOICED FRICATIVE..LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    Interval{ 0x02B0, 0x02B8 }, // Lm   [9] MODIFIER LETTER SMALL H..MODIFIER LETTER SMALL Y
    Interval{ 0x02C0, 0x02C1 }, // Lm   [2] MODIFIER LETTER GLOTTAL STOP..MODIFIER LETTER REVERSED GLOTTAL STOP
    Interval{ 0x02E0, 0x02E4 }, // Lm   [5] MODIFIER LETTER SMALL GAMMA..MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    Interval{ 0x0345, 0x0345 }, // Mn       COMBINING GREEK YPOGEGRAMMENI
    Interval{ 0x0370, 0x0373 }, // L&   [4] GREEK CAPITAL LETTER HETA..GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0376, 0x0377 }, // L&   [2] GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA..GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037A, 0x037A }, // Lm       GREEK YPOGEGRAMMENI
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x03A1 }, // L&  [20] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03F5 }, // L&  [83] GREEK CAPITAL LETTER SIGMA..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F7, 0x0481 }, // L& [139] GREEK CAPITAL LETTER SHO..CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x048A, 0x052F }, // L& [166] CYRILLIC CAPITAL LETTER SHORT I WITH TAIL..CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x0560, 0x0588 }, // L&  [41] ARMENIAN SMALL LETTER TURNED AYB..ARMENIAN SMALL LETTER YI WITH STROKE
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FC, 0x10FC }, // Lm       MODIFIER LETTER GEORGIAN NAR
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1D00, 0x1D2B }, // L&  [44] LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTER SMALL CAPITAL EL
    Interval{ 0x1D2C, 0x1D6A }, // Lm  [63] MODIFIER LETTER CAPITAL A..GREEK SUBSCRIPT SMALL LETTER CHI
    Interval{ 0x1D6B, 0x1D77 }, // L&  [13] LATIN SMALL LETTER UE..LATIN SMALL LETTER TURNED G
    Interval{ 0x1D78, 0x1D78 }, // Lm       MODIFIER LETTER CYRILLIC EN
    Interval{ 0x1D79, 0x1D9A }, // L&  [34] LATIN SMALL LETTER INSULAR G..LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
    Interval{ 0x1D9B, 0x1DBF }, // Lm  [37] MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER LETTER SMALL THETA
    Interval{ 0x1E00, 0x1F15 }, // L& [278] LATIN CAPITAL LETTER A WITH RING BELOW..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F45 }, // L&  [38] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F7D }, // L&  [31] GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1FB4 }, // L&  [53] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FBC }, // L&   [7] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FCC }, // L&   [7] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FDB }, // L&   [6] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE0, 0x1FEC }, // L&  [13] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FFC }, // L&   [7] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x2071, 0x2071 }, // Lm       SUPERSCRIPT LATIN SMALL LETTER I
    Interval{ 0x207F, 0x207F }, // Lm       SUPERSCRIPT LATIN SMALL LETTER N
    Interval{ 0x2090, 0x209C }, // Lm  [13] LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCRIPT SMALL LETTER T
    Interval{ 0x2102, 0x2102 }, // L&       DOUBLE-STRUCK CAPITAL C
    Interval{ 0x2107, 0x2107 }, // L&       EULER CONSTANT
    Interval{ 0x210A, 0x2113 }, // L&  [10] SCRIPT SMALL G..SCRIPT SMALL L
    Interval{ 0x2115, 0x2115 }, // L&       DOUBLE-STRUCK CAPITAL N
    Interval{ 0x2119, 0x211D }, // L&   [5] DOUBLE-STRUCK CAPITAL P..DOUBLE-STRUCK CAPITAL R
    Interval{ 0x2124, 0x2124 }, // L&       DOUBLE-STRUCK CAPITAL Z
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x2128, 0x2128 }, // L&       BLACK-LETTER CAPITAL Z
    Interval{ 0x212A, 0x212D }, // L&   [4] KELVIN SIGN..BLACK-LETTER CAPITAL C
    Interval{ 0x212F, 0x2134 }, // L&   [6] SCRIPT SMALL E..SCRIPT SMALL O
    Interval{ 0x2139, 0x2139 }, // L&       INFORMATION SOURCE
    Interval{ 0x213C, 0x213F }, // L&   [4] DOUBLE-STRUCK SMALL PI..DOUBLE-STRUCK CAPITAL PI
    Interval{ 0x2145, 0x2149 }, // L&   [5] DOUBLE-STRUCK ITALIC CAPITAL D..DOUBLE-STRUCK ITALIC SMALL J
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2160, 0x217F }, // Nl  [32] ROMAN NUMERAL ONE..SMALL ROMAN NUMERAL ONE THOUSAND
    Interval{ 0x2183, 0x2184 }, // L&   [2] ROMAN NUMERAL REVERSED ONE HUNDRED..LATIN SMALL LETTER REVERSED C
    Interval{ 0x24B6, 0x24E9 }, // So  [52] CIRCLED LATIN CAPITAL LETTER A..CIRCLED LATIN SMALL LETTER Z
    Interval{ 0x2C00, 0x2C7B }, // L& [124] GLAGOLITIC CAPITAL LETTER AZU..LATIN LETTER SMALL CAPITAL TURNED E
    Interval{ 0x2C7C, 0x2C7D }, // Lm   [2] LATIN SUBSCRIPT SMALL LETTER J..MODIFIER LETTER CAPITAL V
    Interval{ 0x2C7E, 0x2CE4 }, // L& [103] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC SYMBOL KAI
    Interval{ 0x2CEB, 0x2CEE }, // L&   [4] COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI..COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF2, 0x2CF3 }, // L&   [2] COPTIC CAPITAL LETTER BOHAIRIC KHEI..COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0xA640, 0xA66D }, // L&  [46] CYRILLIC CAPITAL LETTER ZEMLYA..CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA680, 0xA69B }, // L&  [28] CYRILLIC CAPITAL LETTER DWE..CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA69C, 0xA69D }, // Lm   [2] MODIFIER LETTER CYRILLIC HARD SIGN..MODIFIER LETTER CYRILLIC SOFT SIGN
    Interval{ 0xA722, 0xA76F }, // L&  [78] LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF..LATIN SMALL LETTER CON
    Interval{ 0xA770, 0xA770 }, // Lm       MODIFIER LETTER US
    Interval{ 0xA771, 0xA787 }, // L&  [23] LATIN SMALL LETTER DUM..LATIN SMALL LETTER INSULAR T
    Interval{ 0xA78B, 0xA78E }, // L&   [4] LATIN CAPITAL LETTER SALTILLO..LATIN SMALL LETTER L WITH RETROFLEX HOOK AND BELT
    Interval{ 0xA790, 0xA7CA }, // L&  [59] LATIN CAPITAL LETTER N WITH DESCENDER..LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D1 }, // L&   [2] LATIN CAPITAL LETTER CLOSED INSULAR G..LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D3, 0xA7D3 }, // L&       LATIN SMALL LETTER DOUBLE THORN
    Interval{ 0xA7D5, 0xA7D9 }, // L&   [5] LATIN SMALL LETTER DOUBLE WYNN..LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F2, 0xA7F4 }, // Lm   [3] MODIFIER LETTER CAPITAL C..MODIFIER LETTER CAPITAL Q
    Interval{ 0xA7F5, 0xA7F6 }, // L&   [2] LATIN CAPITAL LETTER REVERSED HALF H..LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xA7F8, 0xA7F9 }, // Lm   [2] MODIFIER LETTER CAPITAL H WITH STROKE..MODIFIER LETTER SMALL LIGATURE OE
    Interval{ 0xA7FA, 0xA7FA }, // L&       LATIN LETTER SMALL CAPITAL TURNED M
    Interval{ 0xAB30, 0xAB5A }, // L&  [43] LATIN SMALL LETTER BARRED ALPHA..LATIN SMALL LETTER Y WITH SHORT RIGHT LEG
    Interval{ 0xAB5C, 0xAB5F }, // Lm   [4] MODIFIER LETTER SMALL HENG..MODIFIER LETTER SMALL U WITH LEFT HOOK
    Interval{ 0xAB60, 0xAB68 }, // L&   [9] LATIN SMALL LETTER SAKHA YAT..LATIN SMALL LETTER TURNED R WITH MIDDLE TILDE
    Interval{ 0xAB69, 0xAB69 }, // Lm       MODIFIER LETTER SMALL TURNED W
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0x10400, 0x1044F }, // L&  [80] DESERET CAPITAL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10780, 0x10780 }, // Lm       MODIFIER LETTER SMALL CAPITAL AA
    Interval{ 0x10783, 0x10785 }, // Lm   [3] MODIFIER LETTER SMALL AE..MODIFIER LETTER SMALL B WITH HOOK
    Interval{ 0x10787, 0x107B0 }, // Lm  [42] MODIFIER LETTER SMALL DZ DIGRAPH..MODIFIER LETTER SMALL V WITH RIGHT HOOK
    Interval{ 0x107B2, 0x107BA }, // Lm   [9] MODIFIER LETTER SMALL CAPITAL Y..MODIFIER LETTER SMALL S WITH CURL
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x118A0, 0x118DF }, // L&  [64] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x16E40, 0x16E7F }, // L&  [64] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x1D400, 0x1D454 }, // L&  [85] MATHEMATICAL BOLD CAPITAL A..MATHEMATICAL ITALIC SMALL G
    Interval{ 0x1D456, 0x1D49C }, // L&  [71] MATHEMATICAL ITALIC SMALL I..MATHEMATICAL SCRIPT CAPITAL A
    Interval{ 0x1D49E, 0x1D49F }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL C..MATHEMATICAL SCRIPT CAPITAL D
    Interval{ 0x1D4A2, 0x1D4A2 }, // L&       MATHEMATICAL SCRIPT CAPITAL G
    Interval{ 0x1D4A5, 0x1D4A6 }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL J..MATHEMATICAL SCRIPT CAPITAL K
    Interval{ 0x1D4A9, 0x1D4AC }, // L&   [4] MATHEMATICAL SCRIPT CAPITAL N..MATHEMATICAL SCRIPT CAPITAL Q
    Interval{ 0x1D4AE, 0x1D4B9 }, // L&  [12] MATHEMATICAL SCRIPT CAPITAL S..MATHEMATICAL SCRIPT SMALL D
    Interval{ 0x1D4BB, 0x1D4BB }, // L&       MATHEMATICAL SCRIPT SMALL F
    Interval{ 0x1D4BD, 0x1D4C3 }, // L&   [7] MATHEMATICAL SCRIPT SMALL H..MATHEMATICAL SCRIPT SMALL N
    Interval{ 0x1D4C5, 0x1D505 }, // L&  [65] MATHEMATICAL SCRIPT SMALL P..MATHEMATICAL FRAKTUR CAPITAL B
    Interval{ 0x1D507, 0x1D50A }, // L&   [4] MATHEMATICAL FRAKTUR CAPITAL D..MATHEMATICAL FRAKTUR CAPITAL G
    Interval{ 0x1D50D, 0x1D514 }, // L&   [8] MATHEMATICAL FRAKTUR CAPITAL J..MATHEMATICAL FRAKTUR CAPITAL Q
    Interval{ 0x1D516, 0x1D51C }, // L&   [7] MATHEMATICAL FRAKTUR CAPITAL S..MATHEMATICAL FRAKTUR CAPITAL Y
    Interval{ 0x1D51E, 0x1D539 }, // L&  [28] MATHEMATICAL FRAKTUR SMALL A..MATHEMATICAL DOUBLE-STRUCK CAPITAL B
    Interval{ 0x1D53B, 0x1D53E }, // L&   [4] MATHEMATICAL DOUBLE-STRUCK CAPITAL D..MATHEMATICAL DOUBLE-STRUCK CAPITAL G
    Interval{ 0x1D540, 0x1D544 }, // L&   [5] MATHEMATICAL DOUBLE-STRUCK CAPITAL I..MATHEMATICAL DOUBLE-STRUCK CAPITAL M
    Interval{ 0x1D546, 0x1D546 }, // L&       MATHEMATICAL DOUBLE-STRUCK CAPITAL O
    Interval{ 0x1D54A, 0x1D550 }, // L&   [7] MATHEMATICAL DOUBLE-STRUCK CAPITAL S..MATHEMATICAL DOUBLE-STRUCK CAPITAL Y
    Interval{ 0x1D552, 0x1D6A5 }, // L& [340] MATHEMATICAL DOUBLE-STRUCK SMALL A..MATHEMATICAL ITALIC SMALL DOTLESS J
    Interval{ 0x1D6A8, 0x1D6C0 }, // L&  [25] MATHEMATICAL BOLD CAPITAL ALPHA..MATHEMATICAL BOLD CAPITAL OMEGA
    Interval{ 0x1D6C2, 0x1D6DA }, // L&  [25] MATHEMATICAL BOLD SMALL ALPHA..MATHEMATICAL BOLD SMALL OMEGA
    Interval{ 0x1D6DC, 0x1D6FA }, // L&  [31] MATHEMATICAL BOLD EPSILON SYMBOL..MATHEMATICAL ITALIC CAPITAL OMEGA
    Interval{ 0x1D6FC, 0x1D714 }, // L&  [25] MATHEMATICAL ITALIC SMALL ALPHA..MATHEMATICAL ITALIC SMALL OMEGA
    Interval{ 0x1D716, 0x1D734 }, // L&  [31] MATHEMATICAL ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D736, 0x1D74E }, // L&  [25] MATHEMATICAL BOLD ITALIC SMALL ALPHA..MATHEMATICAL BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D750, 0x1D76E }, // L&  [31] MATHEMATICAL BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGA
    Interval{ 0x1D770, 0x1D788 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD SMALL OMEGA
    Interval{ 0x1D78A, 0x1D7A8 }, // L&  [31] MATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D7AA, 0x1D7C2 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D7C4, 0x1D7CB }, // L&   [8] MATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD SMALL DIGAMMA
    Interval{ 0x1DF00, 0x1DF09 }, // L&  [10] LATIN SMALL LETTER FENG DIGRAPH WITH TRILL..LATIN SMALL LETTER T WITH HOOK AND RETROFLEX HOOK
    Interval{ 0x1DF0B, 0x1DF1E }, // L&  [20] LATIN SMALL LETTER ESH WITH DOUBLE BAR..LATIN SMALL LETTER S WITH CURL
    Interval{ 0x1DF25, 0x1DF2A }, // L&   [6] LATIN SMALL LETTER D WITH MID-HEIGHT LEFT HOOK..LATIN SMALL LETTER T WITH MID-HEIGHT LEFT HOOK
    Interval{ 0x1E030, 0x1E06D }, // Lm  [62] MODIFIER LETTER CYRILLIC SMALL A..MODIFIER LETTER CYRILLIC SMALL STRAIGHT U WITH STROKE
    Interval{ 0x1E900, 0x1E943 }, // L&  [68] ADLAM CAPITAL LETTER ALIF..ADLAM SMALL LETTER SHA
    Interval{ 0x1F130, 0x1F149 }, // So  [26] SQUARED LATIN CAPITAL LETTER A..SQUARED LATIN CAPITAL LETTER Z
    Interval{ 0x1F150, 0x1F169 }, // So  [26] NEGATIVE CIRCLED LATIN CAPITAL LETTER A..NEGATIVE CIRCLED LATIN CAPITAL LETTER Z
    Interval{ 0x1F170, 0x1F189 }, // So  [26] NEGATIVE SQUARED LATIN CAPITAL LETTER A..NEGATIVE SQUARED LATIN CAPITAL LETTER Z
}; // }}}
auto static const Changes_When_Casefolded = std::array<Interval, 622>{ // {{{
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00DF }, // L&   [8] LATIN CAPITAL LETTER O WITH STROKE..LATIN SMALL LETTER SHARP S
    Interval{ 0x0100, 0x0100 }, // L&       LATIN CAPITAL LETTER A WITH MACRON
    Interval{ 0x0102, 0x0102 }, // L&       LATIN CAPITAL LETTER A WITH BREVE
    Interval{ 0x0104, 0x0104 }, // L&       LATIN CAPITAL LETTER A WITH OGONEK
    Interval{ 0x0106, 0x0106 }, // L&       LATIN CAPITAL LETTER C WITH ACUTE
    Interval{ 0x0108, 0x0108 }, // L&       LATIN CAPITAL LETTER C WITH CIRCUMFLEX
    Interval{ 0x010A, 0x010A }, // L&       LATIN CAPITAL LETTER C WITH DOT ABOVE
    Interval{ 0x010C, 0x010C }, // L&       LATIN CAPITAL LETTER C WITH CARON
    Interval{ 0x010E, 0x010E }, // L&       LATIN CAPITAL LETTER D WITH CARON
    Interval{ 0x0110, 0x0110 }, // L&       LATIN CAPITAL LETTER D WITH STROKE
    Interval{ 0x0112, 0x0112 }, // L&       LATIN CAPITAL LETTER E WITH MACRON
    Interval{ 0x0114, 0x0114 }, // L&       LATIN CAPITAL LETTER E WITH BREVE
    Interval{ 0x0116, 0x0116 }, // L&       LATIN CAPITAL LETTER E WITH DOT ABOVE
    Interval{ 0x0118, 0x0118 }, // L&       LATIN CAPITAL LETTER E WITH OGONEK
    Interval{ 0x011A, 0x011A }, // L&       LATIN CAPITAL LETTER E WITH CARON
    Interval{ 0x011C, 0x011C }, // L&       LATIN CAPITAL LETTER G WITH CIRCUMFLEX
    Interval{ 0x011E, 0x011E }, // L&       LATIN CAPITAL LETTER G WITH BREVE
    Interval{ 0x0120, 0x0120 }, // L&       LATIN CAPITAL LETTER G WITH DOT ABOVE
    Interval{ 0x0122, 0x0122 }, // L&       LATIN CAPITAL LETTER G WITH CEDILLA
    Interval{ 0x0124, 0x0124 }, // L&       LATIN CAPITAL LETTER H WITH CIRCUMFLEX
    Interval{ 0x0126, 0x0126 }, // L&       LATIN CAPITAL LETTER H WITH STROKE
    Interval{ 0x0128, 0x0128 }, // L&       LATIN CAPITAL LETTER I WITH TILDE
    Interval{ 0x012A, 0x012A }, // L&       LATIN CAPITAL LETTER I WITH MACRON
    Interval{ 0x012C, 0x012C }, // L&       LATIN CAPITAL LETTER I WITH BREVE
    Interval{ 0x012E, 0x012E }, // L&       LATIN CAPITAL LETTER I WITH OGONEK
    Interval{ 0x0130, 0x0130 }, // L&       LATIN CAPITAL LETTER I WITH DOT ABOVE
    Interval{ 0x0132, 0x0132 }, // L&       LATIN CAPITAL LIGATURE IJ
    Interval{ 0x0134, 0x0134 }, // L&       LATIN CAPITAL LETTER J WITH CIRCUMFLEX
    Interval{ 0x0136, 0x0136 }, // L&       LATIN CAPITAL LETTER K WITH CEDILLA
    Interval{ 0x0139, 0x0139 }, // L&       LATIN CAPITAL LETTER L WITH ACUTE
    Interval{ 0x013B, 0x013B }, // L&       LATIN CAPITAL LETTER L WITH CEDILLA
    Interval{ 0x013D, 0x013D }, // L&       LATIN CAPITAL LETTER L WITH CARON
    Interval{ 0x013F, 0x013F }, // L&       LATIN CAPITAL LETTER L WITH MIDDLE DOT
    Interval{ 0x0141, 0x0141 }, // L&       LATIN CAPITAL LETTER L WITH STROKE
    Interval{ 0x0143, 0x0143 }, // L&       LATIN CAPITAL LETTER N WITH ACUTE
    Interval{ 0x0145, 0x0145 }, // L&       LATIN CAPITAL LETTER N WITH CEDILLA
    Interval{ 0x0147, 0x0147 }, // L&       LATIN CAPITAL LETTER N WITH CARON
    Interval{ 0x0149, 0x014A }, // L&   [2] LATIN SMALL LETTER N PRECEDED BY APOSTROPHE..LATIN CAPITAL LETTER ENG
    Interval{ 0x014C, 0x014C }, // L&       LATIN CAPITAL LETTER O WITH MACRON
    Interval{ 0x014E, 0x014E }, // L&       LATIN CAPITAL LETTER O WITH BREVE
    Interval{ 0x0150, 0x0150 }, // L&       LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
    Interval{ 0x0152, 0x0152 }, // L&       LATIN CAPITAL LIGATURE OE
    Interval{ 0x0154, 0x0154 }, // L&       LATIN CAPITAL LETTER R WITH ACUTE
    Interval{ 0x0156, 0x0156 }, // L&       LATIN CAPITAL LETTER R WITH CEDILLA
    Interval{ 0x0158, 0x0158 }, // L&       LATIN CAPITAL LETTER R WITH CARON
    Interval{ 0x015A, 0x015A }, // L&       LATIN CAPITAL LETTER S WITH ACUTE
    Interval{ 0x015C, 0x015C }, // L&       LATIN CAPITAL LETTER S WITH CIRCUMFLEX
    Interval{ 0x015E, 0x015E }, // L&       LATIN CAPITAL LETTER S WITH CEDILLA
    Interval{ 0x0160, 0x0160 }, // L&       LATIN CAPITAL LETTER S WITH CARON
    Interval{ 0x0162, 0x0162 }, // L&       LATIN CAPITAL LETTER T WITH CEDILLA
    Interval{ 0x0164, 0x0164 }, // L&       LATIN CAPITAL LETTER T WITH CARON
    Interval{ 0x0166, 0x0166 }, // L&       LATIN CAPITAL LETTER T WITH STROKE
    Interval{ 0x0168, 0x0168 }, // L&       LATIN CAPITAL LETTER U WITH TILDE
    Interval{ 0x016A, 0x016A }, // L&       LATIN CAPITAL LETTER U WITH MACRON
    Interval{ 0x016C, 0x016C }, // L&       LATIN CAPITAL LETTER U WITH BREVE
    Interval{ 0x016E, 0x016E }, // L&       LATIN CAPITAL LETTER U WITH RING ABOVE
    Interval{ 0x0170, 0x0170 }, // L&       LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x0172, 0x0172 }, // L&       LATIN CAPITAL LETTER U WITH OGONEK
    Interval{ 0x0174, 0x0174 }, // L&       LATIN CAPITAL LETTER W WITH CIRCUMFLEX
    Interval{ 0x0176, 0x0176 }, // L&       LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
    Interval{ 0x0178, 0x0179 }, // L&   [2] LATIN CAPITAL LETTER Y WITH DIAERESIS..LATIN CAPITAL LETTER Z WITH ACUTE
    Interval{ 0x017B, 0x017B }, // L&       LATIN CAPITAL LETTER Z WITH DOT ABOVE
    Interval{ 0x017D, 0x017D }, // L&       LATIN CAPITAL LETTER Z WITH CARON
    Interval{ 0x017F, 0x017F }, // L&       LATIN SMALL LETTER LONG S
    Interval{ 0x0181, 0x0182 }, // L&   [2] LATIN CAPITAL LETTER B WITH HOOK..LATIN CAPITAL LETTER B WITH TOPBAR
    Interval{ 0x0184, 0x0184 }, // L&       LATIN CAPITAL LETTER TONE SIX
    Interval{ 0x0186, 0x0187 }, // L&   [2] LATIN CAPITAL LETTER OPEN O..LATIN CAPITAL LETTER C WITH HOOK
    Interval{ 0x0189, 0x018B }, // L&   [3] LATIN CAPITAL LETTER AFRICAN D..LATIN CAPITAL LETTER D WITH TOPBAR
    Interval{ 0x018E, 0x0191 }, // L&   [4] LATIN CAPITAL LETTER REVERSED E..LATIN CAPITAL LETTER F WITH HOOK
    Interval{ 0x0193, 0x0194 }, // L&   [2] LATIN CAPITAL LETTER G WITH HOOK..LATIN CAPITAL LETTER GAMMA
    Interval{ 0x0196, 0x0198 }, // L&   [3] LATIN CAPITAL LETTER IOTA..LATIN CAPITAL LETTER K WITH HOOK
    Interval{ 0x019C, 0x019D }, // L&   [2] LATIN CAPITAL LETTER TURNED M..LATIN CAPITAL LETTER N WITH LEFT HOOK
    Interval{ 0x019F, 0x01A0 }, // L&   [2] LATIN CAPITAL LETTER O WITH MIDDLE TILDE..LATIN CAPITAL LETTER O WITH HORN
    Interval{ 0x01A2, 0x01A2 }, // L&       LATIN CAPITAL LETTER OI
    Interval{ 0x01A4, 0x01A4 }, // L&       LATIN CAPITAL LETTER P WITH HOOK
    Interval{ 0x01A6, 0x01A7 }, // L&   [2] LATIN LETTER YR..LATIN CAPITAL LETTER TONE TWO
    Interval{ 0x01A9, 0x01A9 }, // L&       LATIN CAPITAL LETTER ESH
    Interval{ 0x01AC, 0x01AC }, // L&       LATIN CAPITAL LETTER T WITH HOOK
    Interval{ 0x01AE, 0x01AF }, // L&   [2] LATIN CAPITAL LETTER T WITH RETROFLEX HOOK..LATIN CAPITAL LETTER U WITH HORN
    Interval{ 0x01B1, 0x01B3 }, // L&   [3] LATIN CAPITAL LETTER UPSILON..LATIN CAPITAL LETTER Y WITH HOOK
    Interval{ 0x01B5, 0x01B5 }, // L&       LATIN CAPITAL LETTER Z WITH STROKE
    Interval{ 0x01B7, 0x01B8 }, // L&   [2] LATIN CAPITAL LETTER EZH..LATIN CAPITAL LETTER EZH REVERSED
    Interval{ 0x01BC, 0x01BC }, // L&       LATIN CAPITAL LETTER TONE FIVE
    Interval{ 0x01C4, 0x01C5 }, // L&   [2] LATIN CAPITAL LETTER DZ WITH CARON..LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
    Interval{ 0x01C7, 0x01C8 }, // L&   [2] LATIN CAPITAL LETTER LJ..LATIN CAPITAL LETTER L WITH SMALL LETTER J
    Interval{ 0x01CA, 0x01CB }, // L&   [2] LATIN CAPITAL LETTER NJ..LATIN CAPITAL LETTER N WITH SMALL LETTER J
    Interval{ 0x01CD, 0x01CD }, // L&       LATIN CAPITAL LETTER A WITH CARON
    Interval{ 0x01CF, 0x01CF }, // L&       LATIN CAPITAL LETTER I WITH CARON
    Interval{ 0x01D1, 0x01D1 }, // L&       LATIN CAPITAL LETTER O WITH CARON
    Interval{ 0x01D3, 0x01D3 }, // L&       LATIN CAPITAL LETTER U WITH CARON
    Interval{ 0x01D5, 0x01D5 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
    Interval{ 0x01D7, 0x01D7 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
    Interval{ 0x01D9, 0x01D9 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
    Interval{ 0x01DB, 0x01DB }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
    Interval{ 0x01DE, 0x01DE }, // L&       LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
    Interval{ 0x01E0, 0x01E0 }, // L&       LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
    Interval{ 0x01E2, 0x01E2 }, // L&       LATIN CAPITAL LETTER AE WITH MACRON
    Interval{ 0x01E4, 0x01E4 }, // L&       LATIN CAPITAL LETTER G WITH STROKE
    Interval{ 0x01E6, 0x01E6 }, // L&       LATIN CAPITAL LETTER G WITH CARON
    Interval{ 0x01E8, 0x01E8 }, // L&       LATIN CAPITAL LETTER K WITH CARON
    Interval{ 0x01EA, 0x01EA }, // L&       LATIN CAPITAL LETTER O WITH OGONEK
    Interval{ 0x01EC, 0x01EC }, // L&       LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
    Interval{ 0x01EE, 0x01EE }, // L&       LATIN CAPITAL LETTER EZH WITH CARON
    Interval{ 0x01F1, 0x01F2 }, // L&   [2] LATIN CAPITAL LETTER DZ..LATIN CAPITAL LETTER D WITH SMALL LETTER Z
    Interval{ 0x01F4, 0x01F4 }, // L&       LATIN CAPITAL LETTER G WITH ACUTE
    Interval{ 0x01F6, 0x01F8 }, // L&   [3] LATIN CAPITAL LETTER HWAIR..LATIN CAPITAL LETTER N WITH GRAVE
    Interval{ 0x01FA, 0x01FA }, // L&       LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
    Interval{ 0x01FC, 0x01FC }, // L&       LATIN CAPITAL LETTER AE WITH ACUTE
    Interval{ 0x01FE, 0x01FE }, // L&       LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
    Interval{ 0x0200, 0x0200 }, // L&       LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
    Interval{ 0x0202, 0x0202 }, // L&       LATIN CAPITAL LETTER A WITH INVERTED BREVE
    Interval{ 0x0204, 0x0204 }, // L&       LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
    Interval{ 0x0206, 0x0206 }, // L&       LATIN CAPITAL LETTER E WITH INVERTED BREVE
    Interval{ 0x0208, 0x0208 }, // L&       LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
    Interval{ 0x020A, 0x020A }, // L&       LATIN CAPITAL LETTER I WITH INVERTED BREVE
    Interval{ 0x020C, 0x020C }, // L&       LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
    Interval{ 0x020E, 0x020E }, // L&       LATIN CAPITAL LETTER O WITH INVERTED BREVE
    Interval{ 0x0210, 0x0210 }, // L&       LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
    Interval{ 0x0212, 0x0212 }, // L&       LATIN CAPITAL LETTER R WITH INVERTED BREVE
    Interval{ 0x0214, 0x0214 }, // L&       LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
    Interval{ 0x0216, 0x0216 }, // L&       LATIN CAPITAL LETTER U WITH INVERTED BREVE
    Interval{ 0x0218, 0x0218 }, // L&       LATIN CAPITAL LETTER S WITH COMMA BELOW
    Interval{ 0x021A, 0x021A }, // L&       LATIN CAPITAL LETTER T WITH COMMA BELOW
    Interval{ 0x021C, 0x021C }, // L&       LATIN CAPITAL LETTER YOGH
    Interval{ 0x021E, 0x021E }, // L&       LATIN CAPITAL LETTER H WITH CARON
    Interval{ 0x0220, 0x0220 }, // L&       LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
    Interval{ 0x0222, 0x0222 }, // L&       LATIN CAPITAL LETTER OU
    Interval{ 0x0224, 0x0224 }, // L&       LATIN CAPITAL LETTER Z WITH HOOK
    Interval{ 0x0226, 0x0226 }, // L&       LATIN CAPITAL LETTER A WITH DOT ABOVE
    Interval{ 0x0228, 0x0228 }, // L&       LATIN CAPITAL LETTER E WITH CEDILLA
    Interval{ 0x022A, 0x022A }, // L&       LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
    Interval{ 0x022C, 0x022C }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND MACRON
    Interval{ 0x022E, 0x022E }, // L&       LATIN CAPITAL LETTER O WITH DOT ABOVE
    Interval{ 0x0230, 0x0230 }, // L&       LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
    Interval{ 0x0232, 0x0232 }, // L&       LATIN CAPITAL LETTER Y WITH MACRON
    Interval{ 0x023A, 0x023B }, // L&   [2] LATIN CAPITAL LETTER A WITH STROKE..LATIN CAPITAL LETTER C WITH STROKE
    Interval{ 0x023D, 0x023E }, // L&   [2] LATIN CAPITAL LETTER L WITH BAR..LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
    Interval{ 0x0241, 0x0241 }, // L&       LATIN CAPITAL LETTER GLOTTAL STOP
    Interval{ 0x0243, 0x0246 }, // L&   [4] LATIN CAPITAL LETTER B WITH STROKE..LATIN CAPITAL LETTER E WITH STROKE
    Interval{ 0x0248, 0x0248 }, // L&       LATIN CAPITAL LETTER J WITH STROKE
    Interval{ 0x024A, 0x024A }, // L&       LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
    Interval{ 0x024C, 0x024C }, // L&       LATIN CAPITAL LETTER R WITH STROKE
    Interval{ 0x024E, 0x024E }, // L&       LATIN CAPITAL LETTER Y WITH STROKE
    Interval{ 0x0345, 0x0345 }, // Mn       COMBINING GREEK YPOGEGRAMMENI
    Interval{ 0x0370, 0x0370 }, // L&       GREEK CAPITAL LETTER HETA
    Interval{ 0x0372, 0x0372 }, // L&       GREEK CAPITAL LETTER ARCHAIC SAMPI
    Interval{ 0x0376, 0x0376 }, // L&       GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x038F }, // L&   [2] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER OMEGA WITH TONOS
    Interval{ 0x0391, 0x03A1 }, // L&  [17] GREEK CAPITAL LETTER ALPHA..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03AB }, // L&   [9] GREEK CAPITAL LETTER SIGMA..GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
    Interval{ 0x03C2, 0x03C2 }, // L&       GREEK SMALL LETTER FINAL SIGMA
    Interval{ 0x03CF, 0x03D1 }, // L&   [3] GREEK CAPITAL KAI SYMBOL..GREEK THETA SYMBOL
    Interval{ 0x03D5, 0x03D6 }, // L&   [2] GREEK PHI SYMBOL..GREEK PI SYMBOL
    Interval{ 0x03D8, 0x03D8 }, // L&       GREEK LETTER ARCHAIC KOPPA
    Interval{ 0x03DA, 0x03DA }, // L&       GREEK LETTER STIGMA
    Interval{ 0x03DC, 0x03DC }, // L&       GREEK LETTER DIGAMMA
    Interval{ 0x03DE, 0x03DE }, // L&       GREEK LETTER KOPPA
    Interval{ 0x03E0, 0x03E0 }, // L&       GREEK LETTER SAMPI
    Interval{ 0x03E2, 0x03E2 }, // L&       COPTIC CAPITAL LETTER SHEI
    Interval{ 0x03E4, 0x03E4 }, // L&       COPTIC CAPITAL LETTER FEI
    Interval{ 0x03E6, 0x03E6 }, // L&       COPTIC CAPITAL LETTER KHEI
    Interval{ 0x03E8, 0x03E8 }, // L&       COPTIC CAPITAL LETTER HORI
    Interval{ 0x03EA, 0x03EA }, // L&       COPTIC CAPITAL LETTER GANGIA
    Interval{ 0x03EC, 0x03EC }, // L&       COPTIC CAPITAL LETTER SHIMA
    Interval{ 0x03EE, 0x03EE }, // L&       COPTIC CAPITAL LETTER DEI
    Interval{ 0x03F0, 0x03F1 }, // L&   [2] GREEK KAPPA SYMBOL..GREEK RHO SYMBOL
    Interval{ 0x03F4, 0x03F5 }, // L&   [2] GREEK CAPITAL THETA SYMBOL..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F7, 0x03F7 }, // L&       GREEK CAPITAL LETTER SHO
    Interval{ 0x03F9, 0x03FA }, // L&   [2] GREEK CAPITAL LUNATE SIGMA SYMBOL..GREEK CAPITAL LETTER SAN
    Interval{ 0x03FD, 0x042F }, // L&  [51] GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL..CYRILLIC CAPITAL LETTER YA
    Interval{ 0x0460, 0x0460 }, // L&       CYRILLIC CAPITAL LETTER OMEGA
    Interval{ 0x0462, 0x0462 }, // L&       CYRILLIC CAPITAL LETTER YAT
    Interval{ 0x0464, 0x0464 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED E
    Interval{ 0x0466, 0x0466 }, // L&       CYRILLIC CAPITAL LETTER LITTLE YUS
    Interval{ 0x0468, 0x0468 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
    Interval{ 0x046A, 0x046A }, // L&       CYRILLIC CAPITAL LETTER BIG YUS
    Interval{ 0x046C, 0x046C }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
    Interval{ 0x046E, 0x046E }, // L&       CYRILLIC CAPITAL LETTER KSI
    Interval{ 0x0470, 0x0470 }, // L&       CYRILLIC CAPITAL LETTER PSI
    Interval{ 0x0472, 0x0472 }, // L&       CYRILLIC CAPITAL LETTER FITA
    Interval{ 0x0474, 0x0474 }, // L&       CYRILLIC CAPITAL LETTER IZHITSA
    Interval{ 0x0476, 0x0476 }, // L&       CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
    Interval{ 0x0478, 0x0478 }, // L&       CYRILLIC CAPITAL LETTER UK
    Interval{ 0x047A, 0x047A }, // L&       CYRILLIC CAPITAL LETTER ROUND OMEGA
    Interval{ 0x047C, 0x047C }, // L&       CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
    Interval{ 0x047E, 0x047E }, // L&       CYRILLIC CAPITAL LETTER OT
    Interval{ 0x0480, 0x0480 }, // L&       CYRILLIC CAPITAL LETTER KOPPA
    Interval{ 0x048A, 0x048A }, // L&       CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
    Interval{ 0x048C, 0x048C }, // L&       CYRILLIC CAPITAL LETTER SEMISOFT SIGN
    Interval{ 0x048E, 0x048E }, // L&       CYRILLIC CAPITAL LETTER ER WITH TICK
    Interval{ 0x0490, 0x0490 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH UPTURN
    Interval{ 0x0492, 0x0492 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH STROKE
    Interval{ 0x0494, 0x0494 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
    Interval{ 0x0496, 0x0496 }, // L&       CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
    Interval{ 0x0498, 0x0498 }, // L&       CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
    Interval{ 0x049A, 0x049A }, // L&       CYRILLIC CAPITAL LETTER KA WITH DESCENDER
    Interval{ 0x049C, 0x049C }, // L&       CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
    Interval{ 0x049E, 0x049E }, // L&       CYRILLIC CAPITAL LETTER KA WITH STROKE
    Interval{ 0x04A0, 0x04A0 }, // L&       CYRILLIC CAPITAL LETTER BASHKIR KA
    Interval{ 0x04A2, 0x04A2 }, // L&       CYRILLIC CAPITAL LETTER EN WITH DESCENDER
    Interval{ 0x04A4, 0x04A4 }, // L&       CYRILLIC CAPITAL LIGATURE EN GHE
    Interval{ 0x04A6, 0x04A6 }, // L&       CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
    Interval{ 0x04A8, 0x04A8 }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN HA
    Interval{ 0x04AA, 0x04AA }, // L&       CYRILLIC CAPITAL LETTER ES WITH DESCENDER
    Interval{ 0x04AC, 0x04AC }, // L&       CYRILLIC CAPITAL LETTER TE WITH DESCENDER
    Interval{ 0x04AE, 0x04AE }, // L&       CYRILLIC CAPITAL LETTER STRAIGHT U
    Interval{ 0x04B0, 0x04B0 }, // L&       CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
    Interval{ 0x04B2, 0x04B2 }, // L&       CYRILLIC CAPITAL LETTER HA WITH DESCENDER
    Interval{ 0x04B4, 0x04B4 }, // L&       CYRILLIC CAPITAL LIGATURE TE TSE
    Interval{ 0x04B6, 0x04B6 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
    Interval{ 0x04B8, 0x04B8 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
    Interval{ 0x04BA, 0x04BA }, // L&       CYRILLIC CAPITAL LETTER SHHA
    Interval{ 0x04BC, 0x04BC }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN CHE
    Interval{ 0x04BE, 0x04BE }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
    Interval{ 0x04C0, 0x04C1 }, // L&   [2] CYRILLIC LETTER PALOCHKA..CYRILLIC CAPITAL LETTER ZHE WITH BREVE
    Interval{ 0x04C3, 0x04C3 }, // L&       CYRILLIC CAPITAL LETTER KA WITH HOOK
    Interval{ 0x04C5, 0x04C5 }, // L&       CYRILLIC CAPITAL LETTER EL WITH TAIL
    Interval{ 0x04C7, 0x04C7 }, // L&       CYRILLIC CAPITAL LETTER EN WITH HOOK
    Interval{ 0x04C9, 0x04C9 }, // L&       CYRILLIC CAPITAL LETTER EN WITH TAIL
    Interval{ 0x04CB, 0x04CB }, // L&       CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
    Interval{ 0x04CD, 0x04CD }, // L&       CYRILLIC CAPITAL LETTER EM WITH TAIL
    Interval{ 0x04D0, 0x04D0 }, // L&       CYRILLIC CAPITAL LETTER A WITH BREVE
    Interval{ 0x04D2, 0x04D2 }, // L&       CYRILLIC CAPITAL LETTER A WITH DIAERESIS
    Interval{ 0x04D4, 0x04D4 }, // L&       CYRILLIC CAPITAL LIGATURE A IE
    Interval{ 0x04D6, 0x04D6 }, // L&       CYRILLIC CAPITAL LETTER IE WITH BREVE
    Interval{ 0x04D8, 0x04D8 }, // L&       CYRILLIC CAPITAL LETTER SCHWA
    Interval{ 0x04DA, 0x04DA }, // L&       CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
    Interval{ 0x04DC, 0x04DC }, // L&       CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
    Interval{ 0x04DE, 0x04DE }, // L&       CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
    Interval{ 0x04E0, 0x04E0 }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN DZE
    Interval{ 0x04E2, 0x04E2 }, // L&       CYRILLIC CAPITAL LETTER I WITH MACRON
    Interval{ 0x04E4, 0x04E4 }, // L&       CYRILLIC CAPITAL LETTER I WITH DIAERESIS
    Interval{ 0x04E6, 0x04E6 }, // L&       CYRILLIC CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x04E8, 0x04E8 }, // L&       CYRILLIC CAPITAL LETTER BARRED O
    Interval{ 0x04EA, 0x04EA }, // L&       CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
    Interval{ 0x04EC, 0x04EC }, // L&       CYRILLIC CAPITAL LETTER E WITH DIAERESIS
    Interval{ 0x04EE, 0x04EE }, // L&       CYRILLIC CAPITAL LETTER U WITH MACRON
    Interval{ 0x04F0, 0x04F0 }, // L&       CYRILLIC CAPITAL LETTER U WITH DIAERESIS
    Interval{ 0x04F2, 0x04F2 }, // L&       CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x04F4, 0x04F4 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
    Interval{ 0x04F6, 0x04F6 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
    Interval{ 0x04F8, 0x04F8 }, // L&       CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
    Interval{ 0x04FA, 0x04FA }, // L&       CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK
    Interval{ 0x04FC, 0x04FC }, // L&       CYRILLIC CAPITAL LETTER HA WITH HOOK
    Interval{ 0x04FE, 0x04FE }, // L&       CYRILLIC CAPITAL LETTER HA WITH STROKE
    Interval{ 0x0500, 0x0500 }, // L&       CYRILLIC CAPITAL LETTER KOMI DE
    Interval{ 0x0502, 0x0502 }, // L&       CYRILLIC CAPITAL LETTER KOMI DJE
    Interval{ 0x0504, 0x0504 }, // L&       CYRILLIC CAPITAL LETTER KOMI ZJE
    Interval{ 0x0506, 0x0506 }, // L&       CYRILLIC CAPITAL LETTER KOMI DZJE
    Interval{ 0x0508, 0x0508 }, // L&       CYRILLIC CAPITAL LETTER KOMI LJE
    Interval{ 0x050A, 0x050A }, // L&       CYRILLIC CAPITAL LETTER KOMI NJE
    Interval{ 0x050C, 0x050C }, // L&       CYRILLIC CAPITAL LETTER KOMI SJE
    Interval{ 0x050E, 0x050E }, // L&       CYRILLIC CAPITAL LETTER KOMI TJE
    Interval{ 0x0510, 0x0510 }, // L&       CYRILLIC CAPITAL LETTER REVERSED ZE
    Interval{ 0x0512, 0x0512 }, // L&       CYRILLIC CAPITAL LETTER EL WITH HOOK
    Interval{ 0x0514, 0x0514 }, // L&       CYRILLIC CAPITAL LETTER LHA
    Interval{ 0x0516, 0x0516 }, // L&       CYRILLIC CAPITAL LETTER RHA
    Interval{ 0x0518, 0x0518 }, // L&       CYRILLIC CAPITAL LETTER YAE
    Interval{ 0x051A, 0x051A }, // L&       CYRILLIC CAPITAL LETTER QA
    Interval{ 0x051C, 0x051C }, // L&       CYRILLIC CAPITAL LETTER WE
    Interval{ 0x051E, 0x051E }, // L&       CYRILLIC CAPITAL LETTER ALEUT KA
    Interval{ 0x0520, 0x0520 }, // L&       CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
    Interval{ 0x0522, 0x0522 }, // L&       CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
    Interval{ 0x0524, 0x0524 }, // L&       CYRILLIC CAPITAL LETTER PE WITH DESCENDER
    Interval{ 0x0526, 0x0526 }, // L&       CYRILLIC CAPITAL LETTER SHHA WITH DESCENDER
    Interval{ 0x0528, 0x0528 }, // L&       CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK
    Interval{ 0x052A, 0x052A }, // L&       CYRILLIC CAPITAL LETTER DZZHE
    Interval{ 0x052C, 0x052C }, // L&       CYRILLIC CAPITAL LETTER DCHE
    Interval{ 0x052E, 0x052E }, // L&       CYRILLIC CAPITAL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x0587, 0x0587 }, // L&       ARMENIAN SMALL LIGATURE ECH YIWN
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1E00, 0x1E00 }, // L&       LATIN CAPITAL LETTER A WITH RING BELOW
    Interval{ 0x1E02, 0x1E02 }, // L&       LATIN CAPITAL LETTER B WITH DOT ABOVE
    Interval{ 0x1E04, 0x1E04 }, // L&       LATIN CAPITAL LETTER B WITH DOT BELOW
    Interval{ 0x1E06, 0x1E06 }, // L&       LATIN CAPITAL LETTER B WITH LINE BELOW
    Interval{ 0x1E08, 0x1E08 }, // L&       LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
    Interval{ 0x1E0A, 0x1E0A }, // L&       LATIN CAPITAL LETTER D WITH DOT ABOVE
    Interval{ 0x1E0C, 0x1E0C }, // L&       LATIN CAPITAL LETTER D WITH DOT BELOW
    Interval{ 0x1E0E, 0x1E0E }, // L&       LATIN CAPITAL LETTER D WITH LINE BELOW
    Interval{ 0x1E10, 0x1E10 }, // L&       LATIN CAPITAL LETTER D WITH CEDILLA
    Interval{ 0x1E12, 0x1E12 }, // L&       LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
    Interval{ 0x1E14, 0x1E14 }, // L&       LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
    Interval{ 0x1E16, 0x1E16 }, // L&       LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
    Interval{ 0x1E18, 0x1E18 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
    Interval{ 0x1E1A, 0x1E1A }, // L&       LATIN CAPITAL LETTER E WITH TILDE BELOW
    Interval{ 0x1E1C, 0x1E1C }, // L&       LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
    Interval{ 0x1E1E, 0x1E1E }, // L&       LATIN CAPITAL LETTER F WITH DOT ABOVE
    Interval{ 0x1E20, 0x1E20 }, // L&       LATIN CAPITAL LETTER G WITH MACRON
    Interval{ 0x1E22, 0x1E22 }, // L&       LATIN CAPITAL LETTER H WITH DOT ABOVE
    Interval{ 0x1E24, 0x1E24 }, // L&       LATIN CAPITAL LETTER H WITH DOT BELOW
    Interval{ 0x1E26, 0x1E26 }, // L&       LATIN CAPITAL LETTER H WITH DIAERESIS
    Interval{ 0x1E28, 0x1E28 }, // L&       LATIN CAPITAL LETTER H WITH CEDILLA
    Interval{ 0x1E2A, 0x1E2A }, // L&       LATIN CAPITAL LETTER H WITH BREVE BELOW
    Interval{ 0x1E2C, 0x1E2C }, // L&       LATIN CAPITAL LETTER I WITH TILDE BELOW
    Interval{ 0x1E2E, 0x1E2E }, // L&       LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
    Interval{ 0x1E30, 0x1E30 }, // L&       LATIN CAPITAL LETTER K WITH ACUTE
    Interval{ 0x1E32, 0x1E32 }, // L&       LATIN CAPITAL LETTER K WITH DOT BELOW
    Interval{ 0x1E34, 0x1E34 }, // L&       LATIN CAPITAL LETTER K WITH LINE BELOW
    Interval{ 0x1E36, 0x1E36 }, // L&       LATIN CAPITAL LETTER L WITH DOT BELOW
    Interval{ 0x1E38, 0x1E38 }, // L&       LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
    Interval{ 0x1E3A, 0x1E3A }, // L&       LATIN CAPITAL LETTER L WITH LINE BELOW
    Interval{ 0x1E3C, 0x1E3C }, // L&       LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
    Interval{ 0x1E3E, 0x1E3E }, // L&       LATIN CAPITAL LETTER M WITH ACUTE
    Interval{ 0x1E40, 0x1E40 }, // L&       LATIN CAPITAL LETTER M WITH DOT ABOVE
    Interval{ 0x1E42, 0x1E42 }, // L&       LATIN CAPITAL LETTER M WITH DOT BELOW
    Interval{ 0x1E44, 0x1E44 }, // L&       LATIN CAPITAL LETTER N WITH DOT ABOVE
    Interval{ 0x1E46, 0x1E46 }, // L&       LATIN CAPITAL LETTER N WITH DOT BELOW
    Interval{ 0x1E48, 0x1E48 }, // L&       LATIN CAPITAL LETTER N WITH LINE BELOW
    Interval{ 0x1E4A, 0x1E4A }, // L&       LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
    Interval{ 0x1E4C, 0x1E4C }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
    Interval{ 0x1E4E, 0x1E4E }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
    Interval{ 0x1E50, 0x1E50 }, // L&       LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
    Interval{ 0x1E52, 0x1E52 }, // L&       LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
    Interval{ 0x1E54, 0x1E54 }, // L&       LATIN CAPITAL LETTER P WITH ACUTE
    Interval{ 0x1E56, 0x1E56 }, // L&       LATIN CAPITAL LETTER P WITH DOT ABOVE
    Interval{ 0x1E58, 0x1E58 }, // L&       LATIN CAPITAL LETTER R WITH DOT ABOVE
    Interval{ 0x1E5A, 0x1E5A }, // L&       LATIN CAPITAL LETTER R WITH DOT BELOW
    Interval{ 0x1E5C, 0x1E5C }, // L&       LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
    Interval{ 0x1E5E, 0x1E5E }, // L&       LATIN CAPITAL LETTER R WITH LINE BELOW
    Interval{ 0x1E60, 0x1E60 }, // L&       LATIN CAPITAL LETTER S WITH DOT ABOVE
    Interval{ 0x1E62, 0x1E62 }, // L&       LATIN CAPITAL LETTER S WITH DOT BELOW
    Interval{ 0x1E64, 0x1E64 }, // L&       LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
    Interval{ 0x1E66, 0x1E66 }, // L&       LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
    Interval{ 0x1E68, 0x1E68 }, // L&       LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
    Interval{ 0x1E6A, 0x1E6A }, // L&       LATIN CAPITAL LETTER T WITH DOT ABOVE
    Interval{ 0x1E6C, 0x1E6C }, // L&       LATIN CAPITAL LETTER T WITH DOT BELOW
    Interval{ 0x1E6E, 0x1E6E }, // L&       LATIN CAPITAL LETTER T WITH LINE BELOW
    Interval{ 0x1E70, 0x1E70 }, // L&       LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
    Interval{ 0x1E72, 0x1E72 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
    Interval{ 0x1E74, 0x1E74 }, // L&       LATIN CAPITAL LETTER U WITH TILDE BELOW
    Interval{ 0x1E76, 0x1E76 }, // L&       LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
    Interval{ 0x1E78, 0x1E78 }, // L&       LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
    Interval{ 0x1E7A, 0x1E7A }, // L&       LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
    Interval{ 0x1E7C, 0x1E7C }, // L&       LATIN CAPITAL LETTER V WITH TILDE
    Interval{ 0x1E7E, 0x1E7E }, // L&       LATIN CAPITAL LETTER V WITH DOT BELOW
    Interval{ 0x1E80, 0x1E80 }, // L&       LATIN CAPITAL LETTER W WITH GRAVE
    Interval{ 0x1E82, 0x1E82 }, // L&       LATIN CAPITAL LETTER W WITH ACUTE
    Interval{ 0x1E84, 0x1E84 }, // L&       LATIN CAPITAL LETTER W WITH DIAERESIS
    Interval{ 0x1E86, 0x1E86 }, // L&       LATIN CAPITAL LETTER W WITH DOT ABOVE
    Interval{ 0x1E88, 0x1E88 }, // L&       LATIN CAPITAL LETTER W WITH DOT BELOW
    Interval{ 0x1E8A, 0x1E8A }, // L&       LATIN CAPITAL LETTER X WITH DOT ABOVE
    Interval{ 0x1E8C, 0x1E8C }, // L&       LATIN CAPITAL LETTER X WITH DIAERESIS
    Interval{ 0x1E8E, 0x1E8E }, // L&       LATIN CAPITAL LETTER Y WITH DOT ABOVE
    Interval{ 0x1E90, 0x1E90 }, // L&       LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
    Interval{ 0x1E92, 0x1E92 }, // L&       LATIN CAPITAL LETTER Z WITH DOT BELOW
    Interval{ 0x1E94, 0x1E94 }, // L&       LATIN CAPITAL LETTER Z WITH LINE BELOW
    Interval{ 0x1E9A, 0x1E9B }, // L&   [2] LATIN SMALL LETTER A WITH RIGHT HALF RING..LATIN SMALL LETTER LONG S WITH DOT ABOVE
    Interval{ 0x1E9E, 0x1E9E }, // L&       LATIN CAPITAL LETTER SHARP S
    Interval{ 0x1EA0, 0x1EA0 }, // L&       LATIN CAPITAL LETTER A WITH DOT BELOW
    Interval{ 0x1EA2, 0x1EA2 }, // L&       LATIN CAPITAL LETTER A WITH HOOK ABOVE
    Interval{ 0x1EA4, 0x1EA4 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EA6, 0x1EA6 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EA8, 0x1EA8 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EAA, 0x1EAA }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EAC, 0x1EAC }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EAE, 0x1EAE }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
    Interval{ 0x1EB0, 0x1EB0 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
    Interval{ 0x1EB2, 0x1EB2 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
    Interval{ 0x1EB4, 0x1EB4 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND TILDE
    Interval{ 0x1EB6, 0x1EB6 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
    Interval{ 0x1EB8, 0x1EB8 }, // L&       LATIN CAPITAL LETTER E WITH DOT BELOW
    Interval{ 0x1EBA, 0x1EBA }, // L&       LATIN CAPITAL LETTER E WITH HOOK ABOVE
    Interval{ 0x1EBC, 0x1EBC }, // L&       LATIN CAPITAL LETTER E WITH TILDE
    Interval{ 0x1EBE, 0x1EBE }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EC0, 0x1EC0 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EC2, 0x1EC2 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EC4, 0x1EC4 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EC6, 0x1EC6 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EC8, 0x1EC8 }, // L&       LATIN CAPITAL LETTER I WITH HOOK ABOVE
    Interval{ 0x1ECA, 0x1ECA }, // L&       LATIN CAPITAL LETTER I WITH DOT BELOW
    Interval{ 0x1ECC, 0x1ECC }, // L&       LATIN CAPITAL LETTER O WITH DOT BELOW
    Interval{ 0x1ECE, 0x1ECE }, // L&       LATIN CAPITAL LETTER O WITH HOOK ABOVE
    Interval{ 0x1ED0, 0x1ED0 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1ED2, 0x1ED2 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1ED4, 0x1ED4 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1ED6, 0x1ED6 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1ED8, 0x1ED8 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EDA, 0x1EDA }, // L&       LATIN CAPITAL LETTER O WITH HORN AND ACUTE
    Interval{ 0x1EDC, 0x1EDC }, // L&       LATIN CAPITAL LETTER O WITH HORN AND GRAVE
    Interval{ 0x1EDE, 0x1EDE }, // L&       LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
    Interval{ 0x1EE0, 0x1EE0 }, // L&       LATIN CAPITAL LETTER O WITH HORN AND TILDE
    Interval{ 0x1EE2, 0x1EE2 }, // L&       LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
    Interval{ 0x1EE4, 0x1EE4 }, // L&       LATIN CAPITAL LETTER U WITH DOT BELOW
    Interval{ 0x1EE6, 0x1EE6 }, // L&       LATIN CAPITAL LETTER U WITH HOOK ABOVE
    Interval{ 0x1EE8, 0x1EE8 }, // L&       LATIN CAPITAL LETTER U WITH HORN AND ACUTE
    Interval{ 0x1EEA, 0x1EEA }, // L&       LATIN CAPITAL LETTER U WITH HORN AND GRAVE
    Interval{ 0x1EEC, 0x1EEC }, // L&       LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
    Interval{ 0x1EEE, 0x1EEE }, // L&       LATIN CAPITAL LETTER U WITH HORN AND TILDE
    Interval{ 0x1EF0, 0x1EF0 }, // L&       LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
    Interval{ 0x1EF2, 0x1EF2 }, // L&       LATIN CAPITAL LETTER Y WITH GRAVE
    Interval{ 0x1EF4, 0x1EF4 }, // L&       LATIN CAPITAL LETTER Y WITH DOT BELOW
    Interval{ 0x1EF6, 0x1EF6 }, // L&       LATIN CAPITAL LETTER Y WITH HOOK ABOVE
    Interval{ 0x1EF8, 0x1EF8 }, // L&       LATIN CAPITAL LETTER Y WITH TILDE
    Interval{ 0x1EFA, 0x1EFA }, // L&       LATIN CAPITAL LETTER MIDDLE-WELSH LL
    Interval{ 0x1EFC, 0x1EFC }, // L&       LATIN CAPITAL LETTER MIDDLE-WELSH V
    Interval{ 0x1EFE, 0x1EFE }, // L&       LATIN CAPITAL LETTER Y WITH LOOP
    Interval{ 0x1F08, 0x1F0F }, // L&   [8] GREEK CAPITAL LETTER ALPHA WITH PSILI..GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F28, 0x1F2F }, // L&   [8] GREEK CAPITAL LETTER ETA WITH PSILI..GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F38, 0x1F3F }, // L&   [8] GREEK CAPITAL LETTER IOTA WITH PSILI..GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F5F }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F68, 0x1F6F }, // L&   [8] GREEK CAPITAL LETTER OMEGA WITH PSILI..GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F80, 0x1FAF }, // L&  [48] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
    Interval{ 0x1FB2, 0x1FB4 }, // L&   [3] GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB7, 0x1FBC }, // L&   [6] GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC7, 0x1FCC }, // L&   [6] GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD8, 0x1FDB }, // L&   [4] GREEK CAPITAL LETTER IOTA WITH VRACHY..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE8, 0x1FEC }, // L&   [5] GREEK CAPITAL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF7, 0x1FFC }, // L&   [6] GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x212A, 0x212B }, // L&   [2] KELVIN SIGN..ANGSTROM SIGN
    Interval{ 0x2132, 0x2132 }, // L&       TURNED CAPITAL F
    Interval{ 0x2160, 0x216F }, // Nl  [16] ROMAN NUMERAL ONE..ROMAN NUMERAL ONE THOUSAND
    Interval{ 0x2183, 0x2183 }, // L&       ROMAN NUMERAL REVERSED ONE HUNDRED
    Interval{ 0x24B6, 0x24CF }, // So  [26] CIRCLED LATIN CAPITAL LETTER A..CIRCLED LATIN CAPITAL LETTER Z
    Interval{ 0x2C00, 0x2C2F }, // L&  [48] GLAGOLITIC CAPITAL LETTER AZU..GLAGOLITIC CAPITAL LETTER CAUDATE CHRIVI
    Interval{ 0x2C60, 0x2C60 }, // L&       LATIN CAPITAL LETTER L WITH DOUBLE BAR
    Interval{ 0x2C62, 0x2C64 }, // L&   [3] LATIN CAPITAL LETTER L WITH MIDDLE TILDE..LATIN CAPITAL LETTER R WITH TAIL
    Interval{ 0x2C67, 0x2C67 }, // L&       LATIN CAPITAL LETTER H WITH DESCENDER
    Interval{ 0x2C69, 0x2C69 }, // L&       LATIN CAPITAL LETTER K WITH DESCENDER
    Interval{ 0x2C6B, 0x2C6B }, // L&       LATIN CAPITAL LETTER Z WITH DESCENDER
    Interval{ 0x2C6D, 0x2C70 }, // L&   [4] LATIN CAPITAL LETTER ALPHA..LATIN CAPITAL LETTER TURNED ALPHA
    Interval{ 0x2C72, 0x2C72 }, // L&       LATIN CAPITAL LETTER W WITH HOOK
    Interval{ 0x2C75, 0x2C75 }, // L&       LATIN CAPITAL LETTER HALF H
    Interval{ 0x2C7E, 0x2C80 }, // L&   [3] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC CAPITAL LETTER ALFA
    Interval{ 0x2C82, 0x2C82 }, // L&       COPTIC CAPITAL LETTER VIDA
    Interval{ 0x2C84, 0x2C84 }, // L&       COPTIC CAPITAL LETTER GAMMA
    Interval{ 0x2C86, 0x2C86 }, // L&       COPTIC CAPITAL LETTER DALDA
    Interval{ 0x2C88, 0x2C88 }, // L&       COPTIC CAPITAL LETTER EIE
    Interval{ 0x2C8A, 0x2C8A }, // L&       COPTIC CAPITAL LETTER SOU
    Interval{ 0x2C8C, 0x2C8C }, // L&       COPTIC CAPITAL LETTER ZATA
    Interval{ 0x2C8E, 0x2C8E }, // L&       COPTIC CAPITAL LETTER HATE
    Interval{ 0x2C90, 0x2C90 }, // L&       COPTIC CAPITAL LETTER THETHE
    Interval{ 0x2C92, 0x2C92 }, // L&       COPTIC CAPITAL LETTER IAUDA
    Interval{ 0x2C94, 0x2C94 }, // L&       COPTIC CAPITAL LETTER KAPA
    Interval{ 0x2C96, 0x2C96 }, // L&       COPTIC CAPITAL LETTER LAULA
    Interval{ 0x2C98, 0x2C98 }, // L&       COPTIC CAPITAL LETTER MI
    Interval{ 0x2C9A, 0x2C9A }, // L&       COPTIC CAPITAL LETTER NI
    Interval{ 0x2C9C, 0x2C9C }, // L&       COPTIC CAPITAL LETTER KSI
    Interval{ 0x2C9E, 0x2C9E }, // L&       COPTIC CAPITAL LETTER O
    Interval{ 0x2CA0, 0x2CA0 }, // L&       COPTIC CAPITAL LETTER PI
    Interval{ 0x2CA2, 0x2CA2 }, // L&       COPTIC CAPITAL LETTER RO
    Interval{ 0x2CA4, 0x2CA4 }, // L&       COPTIC CAPITAL LETTER SIMA
    Interval{ 0x2CA6, 0x2CA6 }, // L&       COPTIC CAPITAL LETTER TAU
    Interval{ 0x2CA8, 0x2CA8 }, // L&       COPTIC CAPITAL LETTER UA
    Interval{ 0x2CAA, 0x2CAA }, // L&       COPTIC CAPITAL LETTER FI
    Interval{ 0x2CAC, 0x2CAC }, // L&       COPTIC CAPITAL LETTER KHI
    Interval{ 0x2CAE, 0x2CAE }, // L&       COPTIC CAPITAL LETTER PSI
    Interval{ 0x2CB0, 0x2CB0 }, // L&       COPTIC CAPITAL LETTER OOU
    Interval{ 0x2CB2, 0x2CB2 }, // L&       COPTIC CAPITAL LETTER DIALECT-P ALEF
    Interval{ 0x2CB4, 0x2CB4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC AIN
    Interval{ 0x2CB6, 0x2CB6 }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
    Interval{ 0x2CB8, 0x2CB8 }, // L&       COPTIC CAPITAL LETTER DIALECT-P KAPA
    Interval{ 0x2CBA, 0x2CBA }, // L&       COPTIC CAPITAL LETTER DIALECT-P NI
    Interval{ 0x2CBC, 0x2CBC }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
    Interval{ 0x2CBE, 0x2CBE }, // L&       COPTIC CAPITAL LETTER OLD COPTIC OOU
    Interval{ 0x2CC0, 0x2CC0 }, // L&       COPTIC CAPITAL LETTER SAMPI
    Interval{ 0x2CC2, 0x2CC2 }, // L&       COPTIC CAPITAL LETTER CROSSED SHEI
    Interval{ 0x2CC4, 0x2CC4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC SHEI
    Interval{ 0x2CC6, 0x2CC6 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC ESH
    Interval{ 0x2CC8, 0x2CC8 }, // L&       COPTIC CAPITAL LETTER AKHMIMIC KHEI
    Interval{ 0x2CCA, 0x2CCA }, // L&       COPTIC CAPITAL LETTER DIALECT-P HORI
    Interval{ 0x2CCC, 0x2CCC }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HORI
    Interval{ 0x2CCE, 0x2CCE }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HA
    Interval{ 0x2CD0, 0x2CD0 }, // L&       COPTIC CAPITAL LETTER L-SHAPED HA
    Interval{ 0x2CD2, 0x2CD2 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HEI
    Interval{ 0x2CD4, 0x2CD4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HAT
    Interval{ 0x2CD6, 0x2CD6 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC GANGIA
    Interval{ 0x2CD8, 0x2CD8 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC DJA
    Interval{ 0x2CDA, 0x2CDA }, // L&       COPTIC CAPITAL LETTER OLD COPTIC SHIMA
    Interval{ 0x2CDC, 0x2CDC }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
    Interval{ 0x2CDE, 0x2CDE }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN NGI
    Interval{ 0x2CE0, 0x2CE0 }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN NYI
    Interval{ 0x2CE2, 0x2CE2 }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN WAU
    Interval{ 0x2CEB, 0x2CEB }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI
    Interval{ 0x2CED, 0x2CED }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF2, 0x2CF2 }, // L&       COPTIC CAPITAL LETTER BOHAIRIC KHEI
    Interval{ 0xA640, 0xA640 }, // L&       CYRILLIC CAPITAL LETTER ZEMLYA
    Interval{ 0xA642, 0xA642 }, // L&       CYRILLIC CAPITAL LETTER DZELO
    Interval{ 0xA644, 0xA644 }, // L&       CYRILLIC CAPITAL LETTER REVERSED DZE
    Interval{ 0xA646, 0xA646 }, // L&       CYRILLIC CAPITAL LETTER IOTA
    Interval{ 0xA648, 0xA648 }, // L&       CYRILLIC CAPITAL LETTER DJERV
    Interval{ 0xA64A, 0xA64A }, // L&       CYRILLIC CAPITAL LETTER MONOGRAPH UK
    Interval{ 0xA64C, 0xA64C }, // L&       CYRILLIC CAPITAL LETTER BROAD OMEGA
    Interval{ 0xA64E, 0xA64E }, // L&       CYRILLIC CAPITAL LETTER NEUTRAL YER
    Interval{ 0xA650, 0xA650 }, // L&       CYRILLIC CAPITAL LETTER YERU WITH BACK YER
    Interval{ 0xA652, 0xA652 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED YAT
    Interval{ 0xA654, 0xA654 }, // L&       CYRILLIC CAPITAL LETTER REVERSED YU
    Interval{ 0xA656, 0xA656 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED A
    Interval{ 0xA658, 0xA658 }, // L&       CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS
    Interval{ 0xA65A, 0xA65A }, // L&       CYRILLIC CAPITAL LETTER BLENDED YUS
    Interval{ 0xA65C, 0xA65C }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUS
    Interval{ 0xA65E, 0xA65E }, // L&       CYRILLIC CAPITAL LETTER YN
    Interval{ 0xA660, 0xA660 }, // L&       CYRILLIC CAPITAL LETTER REVERSED TSE
    Interval{ 0xA662, 0xA662 }, // L&       CYRILLIC CAPITAL LETTER SOFT DE
    Interval{ 0xA664, 0xA664 }, // L&       CYRILLIC CAPITAL LETTER SOFT EL
    Interval{ 0xA666, 0xA666 }, // L&       CYRILLIC CAPITAL LETTER SOFT EM
    Interval{ 0xA668, 0xA668 }, // L&       CYRILLIC CAPITAL LETTER MONOCULAR O
    Interval{ 0xA66A, 0xA66A }, // L&       CYRILLIC CAPITAL LETTER BINOCULAR O
    Interval{ 0xA66C, 0xA66C }, // L&       CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA680, 0xA680 }, // L&       CYRILLIC CAPITAL LETTER DWE
    Interval{ 0xA682, 0xA682 }, // L&       CYRILLIC CAPITAL LETTER DZWE
    Interval{ 0xA684, 0xA684 }, // L&       CYRILLIC CAPITAL LETTER ZHWE
    Interval{ 0xA686, 0xA686 }, // L&       CYRILLIC CAPITAL LETTER CCHE
    Interval{ 0xA688, 0xA688 }, // L&       CYRILLIC CAPITAL LETTER DZZE
    Interval{ 0xA68A, 0xA68A }, // L&       CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK
    Interval{ 0xA68C, 0xA68C }, // L&       CYRILLIC CAPITAL LETTER TWE
    Interval{ 0xA68E, 0xA68E }, // L&       CYRILLIC CAPITAL LETTER TSWE
    Interval{ 0xA690, 0xA690 }, // L&       CYRILLIC CAPITAL LETTER TSSE
    Interval{ 0xA692, 0xA692 }, // L&       CYRILLIC CAPITAL LETTER TCHE
    Interval{ 0xA694, 0xA694 }, // L&       CYRILLIC CAPITAL LETTER HWE
    Interval{ 0xA696, 0xA696 }, // L&       CYRILLIC CAPITAL LETTER SHWE
    Interval{ 0xA698, 0xA698 }, // L&       CYRILLIC CAPITAL LETTER DOUBLE O
    Interval{ 0xA69A, 0xA69A }, // L&       CYRILLIC CAPITAL LETTER CROSSED O
    Interval{ 0xA722, 0xA722 }, // L&       LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF
    Interval{ 0xA724, 0xA724 }, // L&       LATIN CAPITAL LETTER EGYPTOLOGICAL AIN
    Interval{ 0xA726, 0xA726 }, // L&       LATIN CAPITAL LETTER HENG
    Interval{ 0xA728, 0xA728 }, // L&       LATIN CAPITAL LETTER TZ
    Interval{ 0xA72A, 0xA72A }, // L&       LATIN CAPITAL LETTER TRESILLO
    Interval{ 0xA72C, 0xA72C }, // L&       LATIN CAPITAL LETTER CUATRILLO
    Interval{ 0xA72E, 0xA72E }, // L&       LATIN CAPITAL LETTER CUATRILLO WITH COMMA
    Interval{ 0xA732, 0xA732 }, // L&       LATIN CAPITAL LETTER AA
    Interval{ 0xA734, 0xA734 }, // L&       LATIN CAPITAL LETTER AO
    Interval{ 0xA736, 0xA736 }, // L&       LATIN CAPITAL LETTER AU
    Interval{ 0xA738, 0xA738 }, // L&       LATIN CAPITAL LETTER AV
    Interval{ 0xA73A, 0xA73A }, // L&       LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR
    Interval{ 0xA73C, 0xA73C }, // L&       LATIN CAPITAL LETTER AY
    Interval{ 0xA73E, 0xA73E }, // L&       LATIN CAPITAL LETTER REVERSED C WITH DOT
    Interval{ 0xA740, 0xA740 }, // L&       LATIN CAPITAL LETTER K WITH STROKE
    Interval{ 0xA742, 0xA742 }, // L&       LATIN CAPITAL LETTER K WITH DIAGONAL STROKE
    Interval{ 0xA744, 0xA744 }, // L&       LATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKE
    Interval{ 0xA746, 0xA746 }, // L&       LATIN CAPITAL LETTER BROKEN L
    Interval{ 0xA748, 0xA748 }, // L&       LATIN CAPITAL LETTER L WITH HIGH STROKE
    Interval{ 0xA74A, 0xA74A }, // L&       LATIN CAPITAL LETTER O WITH LONG STROKE OVERLAY
    Interval{ 0xA74C, 0xA74C }, // L&       LATIN CAPITAL LETTER O WITH LOOP
    Interval{ 0xA74E, 0xA74E }, // L&       LATIN CAPITAL LETTER OO
    Interval{ 0xA750, 0xA750 }, // L&       LATIN CAPITAL LETTER P WITH STROKE THROUGH DESCENDER
    Interval{ 0xA752, 0xA752 }, // L&       LATIN CAPITAL LETTER P WITH FLOURISH
    Interval{ 0xA754, 0xA754 }, // L&       LATIN CAPITAL LETTER P WITH SQUIRREL TAIL
    Interval{ 0xA756, 0xA756 }, // L&       LATIN CAPITAL LETTER Q WITH STROKE THROUGH DESCENDER
    Interval{ 0xA758, 0xA758 }, // L&       LATIN CAPITAL LETTER Q WITH DIAGONAL STROKE
    Interval{ 0xA75A, 0xA75A }, // L&       LATIN CAPITAL LETTER R ROTUNDA
    Interval{ 0xA75C, 0xA75C }, // L&       LATIN CAPITAL LETTER RUM ROTUNDA
    Interval{ 0xA75E, 0xA75E }, // L&       LATIN CAPITAL LETTER V WITH DIAGONAL STROKE
    Interval{ 0xA760, 0xA760 }, // L&       LATIN CAPITAL LETTER VY
    Interval{ 0xA762, 0xA762 }, // L&       LATIN CAPITAL LETTER VISIGOTHIC Z
    Interval{ 0xA764, 0xA764 }, // L&       LATIN CAPITAL LETTER THORN WITH STROKE
    Interval{ 0xA766, 0xA766 }, // L&       LATIN CAPITAL LETTER THORN WITH STROKE THROUGH DESCENDER
    Interval{ 0xA768, 0xA768 }, // L&       LATIN CAPITAL LETTER VEND
    Interval{ 0xA76A, 0xA76A }, // L&       LATIN CAPITAL LETTER ET
    Interval{ 0xA76C, 0xA76C }, // L&       LATIN CAPITAL LETTER IS
    Interval{ 0xA76E, 0xA76E }, // L&       LATIN CAPITAL LETTER CON
    Interval{ 0xA779, 0xA779 }, // L&       LATIN CAPITAL LETTER INSULAR D
    Interval{ 0xA77B, 0xA77B }, // L&       LATIN CAPITAL LETTER INSULAR F
    Interval{ 0xA77D, 0xA77E }, // L&   [2] LATIN CAPITAL LETTER INSULAR G..LATIN CAPITAL LETTER TURNED INSULAR G
    Interval{ 0xA780, 0xA780 }, // L&       LATIN CAPITAL LETTER TURNED L
    Interval{ 0xA782, 0xA782 }, // L&       LATIN CAPITAL LETTER INSULAR R
    Interval{ 0xA784, 0xA784 }, // L&       LATIN CAPITAL LETTER INSULAR S
    Interval{ 0xA786, 0xA786 }, // L&       LATIN CAPITAL LETTER INSULAR T
    Interval{ 0xA78B, 0xA78B }, // L&       LATIN CAPITAL LETTER SALTILLO
    Interval{ 0xA78D, 0xA78D }, // L&       LATIN CAPITAL LETTER TURNED H
    Interval{ 0xA790, 0xA790 }, // L&       LATIN CAPITAL LETTER N WITH DESCENDER
    Interval{ 0xA792, 0xA792 }, // L&       LATIN CAPITAL LETTER C WITH BAR
    Interval{ 0xA796, 0xA796 }, // L&       LATIN CAPITAL LETTER B WITH FLOURISH
    Interval{ 0xA798, 0xA798 }, // L&       LATIN CAPITAL LETTER F WITH STROKE
    Interval{ 0xA79A, 0xA79A }, // L&       LATIN CAPITAL LETTER VOLAPUK AE
    Interval{ 0xA79C, 0xA79C }, // L&       LATIN CAPITAL LETTER VOLAPUK OE
    Interval{ 0xA79E, 0xA79E }, // L&       LATIN CAPITAL LETTER VOLAPUK UE
    Interval{ 0xA7A0, 0xA7A0 }, // L&       LATIN CAPITAL LETTER G WITH OBLIQUE STROKE
    Interval{ 0xA7A2, 0xA7A2 }, // L&       LATIN CAPITAL LETTER K WITH OBLIQUE STROKE
    Interval{ 0xA7A4, 0xA7A4 }, // L&       LATIN CAPITAL LETTER N WITH OBLIQUE STROKE
    Interval{ 0xA7A6, 0xA7A6 }, // L&       LATIN CAPITAL LETTER R WITH OBLIQUE STROKE
    Interval{ 0xA7A8, 0xA7A8 }, // L&       LATIN CAPITAL LETTER S WITH OBLIQUE STROKE
    Interval{ 0xA7AA, 0xA7AE }, // L&   [5] LATIN CAPITAL LETTER H WITH HOOK..LATIN CAPITAL LETTER SMALL CAPITAL I
    Interval{ 0xA7B0, 0xA7B4 }, // L&   [5] LATIN CAPITAL LETTER TURNED K..LATIN CAPITAL LETTER BETA
    Interval{ 0xA7B6, 0xA7B6 }, // L&       LATIN CAPITAL LETTER OMEGA
    Interval{ 0xA7B8, 0xA7B8 }, // L&       LATIN CAPITAL LETTER U WITH STROKE
    Interval{ 0xA7BA, 0xA7BA }, // L&       LATIN CAPITAL LETTER GLOTTAL A
    Interval{ 0xA7BC, 0xA7BC }, // L&       LATIN CAPITAL LETTER GLOTTAL I
    Interval{ 0xA7BE, 0xA7BE }, // L&       LATIN CAPITAL LETTER GLOTTAL U
    Interval{ 0xA7C0, 0xA7C0 }, // L&       LATIN CAPITAL LETTER OLD POLISH O
    Interval{ 0xA7C2, 0xA7C2 }, // L&       LATIN CAPITAL LETTER ANGLICANA W
    Interval{ 0xA7C4, 0xA7C7 }, // L&   [4] LATIN CAPITAL LETTER C WITH PALATAL HOOK..LATIN CAPITAL LETTER D WITH SHORT STROKE OVERLAY
    Interval{ 0xA7C9, 0xA7C9 }, // L&       LATIN CAPITAL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D0 }, // L&       LATIN CAPITAL LETTER CLOSED INSULAR G
    Interval{ 0xA7D6, 0xA7D6 }, // L&       LATIN CAPITAL LETTER MIDDLE SCOTS S
    Interval{ 0xA7D8, 0xA7D8 }, // L&       LATIN CAPITAL LETTER SIGMOID S
    Interval{ 0xA7F5, 0xA7F5 }, // L&       LATIN CAPITAL LETTER REVERSED HALF H
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0x10400, 0x10427 }, // L&  [40] DESERET CAPITAL LETTER LONG I..DESERET CAPITAL LETTER EW
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x118A0, 0x118BF }, // L&  [32] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI CAPITAL LETTER VIYO
    Interval{ 0x16E40, 0x16E5F }, // L&  [32] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN CAPITAL LETTER Y
    Interval{ 0x1E900, 0x1E921 }, // L&  [34] ADLAM CAPITAL LETTER ALIF..ADLAM CAPITAL LETTER SHA
}; // }}}
auto static const Changes_When_Casemapped = std::array<Interval, 131>{ // {{{
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00F6 }, // L&  [31] LATIN CAPITAL LETTER O WITH STROKE..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x0137 }, // L&  [64] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER K WITH CEDILLA
    Interval{ 0x0139, 0x018C }, // L&  [84] LATIN CAPITAL LETTER L WITH ACUTE..LATIN SMALL LETTER D WITH TOPBAR
    Interval{ 0x018E, 0x019A }, // L&  [13] LATIN CAPITAL LETTER REVERSED E..LATIN SMALL LETTER L WITH BAR
    Interval{ 0x019C, 0x01A9 }, // L&  [14] LATIN CAPITAL LETTER TURNED M..LATIN CAPITAL LETTER ESH
    Interval{ 0x01AC, 0x01B9 }, // L&  [14] LATIN CAPITAL LETTER T WITH HOOK..LATIN SMALL LETTER EZH REVERSED
    Interval{ 0x01BC, 0x01BD }, // L&   [2] LATIN CAPITAL LETTER TONE FIVE..LATIN SMALL LETTER TONE FIVE
    Interval{ 0x01BF, 0x01BF }, // L&       LATIN LETTER WYNN
    Interval{ 0x01C4, 0x0220 }, // L&  [93] LATIN CAPITAL LETTER DZ WITH CARON..LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
    Interval{ 0x0222, 0x0233 }, // L&  [18] LATIN CAPITAL LETTER OU..LATIN SMALL LETTER Y WITH MACRON
    Interval{ 0x023A, 0x0254 }, // L&  [27] LATIN CAPITAL LETTER A WITH STROKE..LATIN SMALL LETTER OPEN O
    Interval{ 0x0256, 0x0257 }, // L&   [2] LATIN SMALL LETTER D WITH TAIL..LATIN SMALL LETTER D WITH HOOK
    Interval{ 0x0259, 0x0259 }, // L&       LATIN SMALL LETTER SCHWA
    Interval{ 0x025B, 0x025C }, // L&   [2] LATIN SMALL LETTER OPEN E..LATIN SMALL LETTER REVERSED OPEN E
    Interval{ 0x0260, 0x0261 }, // L&   [2] LATIN SMALL LETTER G WITH HOOK..LATIN SMALL LETTER SCRIPT G
    Interval{ 0x0263, 0x0263 }, // L&       LATIN SMALL LETTER GAMMA
    Interval{ 0x0265, 0x0266 }, // L&   [2] LATIN SMALL LETTER TURNED H..LATIN SMALL LETTER H WITH HOOK
    Interval{ 0x0268, 0x026C }, // L&   [5] LATIN SMALL LETTER I WITH STROKE..LATIN SMALL LETTER L WITH BELT
    Interval{ 0x026F, 0x026F }, // L&       LATIN SMALL LETTER TURNED M
    Interval{ 0x0271, 0x0272 }, // L&   [2] LATIN SMALL LETTER M WITH HOOK..LATIN SMALL LETTER N WITH LEFT HOOK
    Interval{ 0x0275, 0x0275 }, // L&       LATIN SMALL LETTER BARRED O
    Interval{ 0x027D, 0x027D }, // L&       LATIN SMALL LETTER R WITH TAIL
    Interval{ 0x0280, 0x0280 }, // L&       LATIN LETTER SMALL CAPITAL R
    Interval{ 0x0282, 0x0283 }, // L&   [2] LATIN SMALL LETTER S WITH HOOK..LATIN SMALL LETTER ESH
    Interval{ 0x0287, 0x028C }, // L&   [6] LATIN SMALL LETTER TURNED T..LATIN SMALL LETTER TURNED V
    Interval{ 0x0292, 0x0292 }, // L&       LATIN SMALL LETTER EZH
    Interval{ 0x029D, 0x029E }, // L&   [2] LATIN SMALL LETTER J WITH CROSSED-TAIL..LATIN SMALL LETTER TURNED K
    Interval{ 0x0345, 0x0345 }, // Mn       COMBINING GREEK YPOGEGRAMMENI
    Interval{ 0x0370, 0x0373 }, // L&   [4] GREEK CAPITAL LETTER HETA..GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0376, 0x0377 }, // L&   [2] GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA..GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x03A1 }, // L&  [20] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03D1 }, // L&  [47] GREEK CAPITAL LETTER SIGMA..GREEK THETA SYMBOL
    Interval{ 0x03D5, 0x03F5 }, // L&  [33] GREEK PHI SYMBOL..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F7, 0x03FB }, // L&   [5] GREEK CAPITAL LETTER SHO..GREEK SMALL LETTER SAN
    Interval{ 0x03FD, 0x0481 }, // L& [133] GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL..CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x048A, 0x052F }, // L& [166] CYRILLIC CAPITAL LETTER SHORT I WITH TAIL..CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x0561, 0x0587 }, // L&  [39] ARMENIAN SMALL LETTER AYB..ARMENIAN SMALL LIGATURE ECH YIWN
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1D79, 0x1D79 }, // L&       LATIN SMALL LETTER INSULAR G
    Interval{ 0x1D7D, 0x1D7D }, // L&       LATIN SMALL LETTER P WITH STROKE
    Interval{ 0x1D8E, 0x1D8E }, // L&       LATIN SMALL LETTER Z WITH PALATAL HOOK
    Interval{ 0x1E00, 0x1E9B }, // L& [156] LATIN CAPITAL LETTER A WITH RING BELOW..LATIN SMALL LETTER LONG S WITH DOT ABOVE
    Interval{ 0x1E9E, 0x1E9E }, // L&       LATIN CAPITAL LETTER SHARP S
    Interval{ 0x1EA0, 0x1F15 }, // L& [118] LATIN CAPITAL LETTER A WITH DOT BELOW..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F45 }, // L&  [38] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F7D }, // L&  [31] GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1FB4 }, // L&  [53] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FBC }, // L&   [7] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FCC }, // L&   [7] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FDB }, // L&   [6] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE0, 0x1FEC }, // L&  [13] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FFC }, // L&   [7] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x212A, 0x212B }, // L&   [2] KELVIN SIGN..ANGSTROM SIGN
    Interval{ 0x2132, 0x2132 }, // L&       TURNED CAPITAL F
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2160, 0x217F }, // Nl  [32] ROMAN NUMERAL ONE..SMALL ROMAN NUMERAL ONE THOUSAND
    Interval{ 0x2183, 0x2184 }, // L&   [2] ROMAN NUMERAL REVERSED ONE HUNDRED..LATIN SMALL LETTER REVERSED C
    Interval{ 0x24B6, 0x24E9 }, // So  [52] CIRCLED LATIN CAPITAL LETTER A..CIRCLED LATIN SMALL LETTER Z
    Interval{ 0x2C00, 0x2C70 }, // L& [113] GLAGOLITIC CAPITAL LETTER AZU..LATIN CAPITAL LETTER TURNED ALPHA
    Interval{ 0x2C72, 0x2C73 }, // L&   [2] LATIN CAPITAL LETTER W WITH HOOK..LATIN SMALL LETTER W WITH HOOK
    Interval{ 0x2C75, 0x2C76 }, // L&   [2] LATIN CAPITAL LETTER HALF H..LATIN SMALL LETTER HALF H
    Interval{ 0x2C7E, 0x2CE3 }, // L& [102] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC SMALL LETTER OLD NUBIAN WAU
    Interval{ 0x2CEB, 0x2CEE }, // L&   [4] COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI..COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF2, 0x2CF3 }, // L&   [2] COPTIC CAPITAL LETTER BOHAIRIC KHEI..COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0xA640, 0xA66D }, // L&  [46] CYRILLIC CAPITAL LETTER ZEMLYA..CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA680, 0xA69B }, // L&  [28] CYRILLIC CAPITAL LETTER DWE..CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA722, 0xA72F }, // L&  [14] LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF..LATIN SMALL LETTER CUATRILLO WITH COMMA
    Interval{ 0xA732, 0xA76F }, // L&  [62] LATIN CAPITAL LETTER AA..LATIN SMALL LETTER CON
    Interval{ 0xA779, 0xA787 }, // L&  [15] LATIN CAPITAL LETTER INSULAR D..LATIN SMALL LETTER INSULAR T
    Interval{ 0xA78B, 0xA78D }, // L&   [3] LATIN CAPITAL LETTER SALTILLO..LATIN CAPITAL LETTER TURNED H
    Interval{ 0xA790, 0xA794 }, // L&   [5] LATIN CAPITAL LETTER N WITH DESCENDER..LATIN SMALL LETTER C WITH PALATAL HOOK
    Interval{ 0xA796, 0xA7AE }, // L&  [25] LATIN CAPITAL LETTER B WITH FLOURISH..LATIN CAPITAL LETTER SMALL CAPITAL I
    Interval{ 0xA7B0, 0xA7CA }, // L&  [27] LATIN CAPITAL LETTER TURNED K..LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D1 }, // L&   [2] LATIN CAPITAL LETTER CLOSED INSULAR G..LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D6, 0xA7D9 }, // L&   [4] LATIN CAPITAL LETTER MIDDLE SCOTS S..LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F5, 0xA7F6 }, // L&   [2] LATIN CAPITAL LETTER REVERSED HALF H..LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xAB53, 0xAB53 }, // L&       LATIN SMALL LETTER CHI
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0x10400, 0x1044F }, // L&  [80] DESERET CAPITAL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x118A0, 0x118DF }, // L&  [64] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x16E40, 0x16E7F }, // L&  [64] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x1E900, 0x1E943 }, // L&  [68] ADLAM CAPITAL LETTER ALIF..ADLAM SMALL LETTER SHA
}; // }}}
auto static const Changes_When_Lowercased = std::array<Interval, 609>{ // {{{
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00DE }, // L&   [7] LATIN CAPITAL LETTER O WITH STROKE..LATIN CAPITAL LETTER THORN
    Interval{ 0x0100, 0x0100 }, // L&       LATIN CAPITAL LETTER A WITH MACRON
    Interval{ 0x0102, 0x0102 }, // L&       LATIN CAPITAL LETTER A WITH BREVE
    Interval{ 0x0104, 0x0104 }, // L&       LATIN CAPITAL LETTER A WITH OGONEK
    Interval{ 0x0106, 0x0106 }, // L&       LATIN CAPITAL LETTER C WITH ACUTE
    Interval{ 0x0108, 0x0108 }, // L&       LATIN CAPITAL LETTER C WITH CIRCUMFLEX
    Interval{ 0x010A, 0x010A }, // L&       LATIN CAPITAL LETTER C WITH DOT ABOVE
    Interval{ 0x010C, 0x010C }, // L&       LATIN CAPITAL LETTER C WITH CARON
    Interval{ 0x010E, 0x010E }, // L&       LATIN CAPITAL LETTER D WITH CARON
    Interval{ 0x0110, 0x0110 }, // L&       LATIN CAPITAL LETTER D WITH STROKE
    Interval{ 0x0112, 0x0112 }, // L&       LATIN CAPITAL LETTER E WITH MACRON
    Interval{ 0x0114, 0x0114 }, // L&       LATIN CAPITAL LETTER E WITH BREVE
    Interval{ 0x0116, 0x0116 }, // L&       LATIN CAPITAL LETTER E WITH DOT ABOVE
    Interval{ 0x0118, 0x0118 }, // L&       LATIN CAPITAL LETTER E WITH OGONEK
    Interval{ 0x011A, 0x011A }, // L&       LATIN CAPITAL LETTER E WITH CARON
    Interval{ 0x011C, 0x011C }, // L&       LATIN CAPITAL LETTER G WITH CIRCUMFLEX
    Interval{ 0x011E, 0x011E }, // L&       LATIN CAPITAL LETTER G WITH BREVE
    Interval{ 0x0120, 0x0120 }, // L&       LATIN CAPITAL LETTER G WITH DOT ABOVE
    Interval{ 0x0122, 0x0122 }, // L&       LATIN CAPITAL LETTER G WITH CEDILLA
    Interval{ 0x0124, 0x0124 }, // L&       LATIN CAPITAL LETTER H WITH CIRCUMFLEX
    Interval{ 0x0126, 0x0126 }, // L&       LATIN CAPITAL LETTER H WITH STROKE
    Interval{ 0x0128, 0x0128 }, // L&       LATIN CAPITAL LETTER I WITH TILDE
    Interval{ 0x012A, 0x012A }, // L&       LATIN CAPITAL LETTER I WITH MACRON
    Interval{ 0x012C, 0x012C }, // L&       LATIN CAPITAL LETTER I WITH BREVE
    Interval{ 0x012E, 0x012E }, // L&       LATIN CAPITAL LETTER I WITH OGONEK
    Interval{ 0x0130, 0x0130 }, // L&       LATIN CAPITAL LETTER I WITH DOT ABOVE
    Interval{ 0x0132, 0x0132 }, // L&       LATIN CAPITAL LIGATURE IJ
    Interval{ 0x0134, 0x0134 }, // L&       LATIN CAPITAL LETTER J WITH CIRCUMFLEX
    Interval{ 0x0136, 0x0136 }, // L&       LATIN CAPITAL LETTER K WITH CEDILLA
    Interval{ 0x0139, 0x0139 }, // L&       LATIN CAPITAL LETTER L WITH ACUTE
    Interval{ 0x013B, 0x013B }, // L&       LATIN CAPITAL LETTER L WITH CEDILLA
    Interval{ 0x013D, 0x013D }, // L&       LATIN CAPITAL LETTER L WITH CARON
    Interval{ 0x013F, 0x013F }, // L&       LATIN CAPITAL LETTER L WITH MIDDLE DOT
    Interval{ 0x0141, 0x0141 }, // L&       LATIN CAPITAL LETTER L WITH STROKE
    Interval{ 0x0143, 0x0143 }, // L&       LATIN CAPITAL LETTER N WITH ACUTE
    Interval{ 0x0145, 0x0145 }, // L&       LATIN CAPITAL LETTER N WITH CEDILLA
    Interval{ 0x0147, 0x0147 }, // L&       LATIN CAPITAL LETTER N WITH CARON
    Interval{ 0x014A, 0x014A }, // L&       LATIN CAPITAL LETTER ENG
    Interval{ 0x014C, 0x014C }, // L&       LATIN CAPITAL LETTER O WITH MACRON
    Interval{ 0x014E, 0x014E }, // L&       LATIN CAPITAL LETTER O WITH BREVE
    Interval{ 0x0150, 0x0150 }, // L&       LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
    Interval{ 0x0152, 0x0152 }, // L&       LATIN CAPITAL LIGATURE OE
    Interval{ 0x0154, 0x0154 }, // L&       LATIN CAPITAL LETTER R WITH ACUTE
    Interval{ 0x0156, 0x0156 }, // L&       LATIN CAPITAL LETTER R WITH CEDILLA
    Interval{ 0x0158, 0x0158 }, // L&       LATIN CAPITAL LETTER R WITH CARON
    Interval{ 0x015A, 0x015A }, // L&       LATIN CAPITAL LETTER S WITH ACUTE
    Interval{ 0x015C, 0x015C }, // L&       LATIN CAPITAL LETTER S WITH CIRCUMFLEX
    Interval{ 0x015E, 0x015E }, // L&       LATIN CAPITAL LETTER S WITH CEDILLA
    Interval{ 0x0160, 0x0160 }, // L&       LATIN CAPITAL LETTER S WITH CARON
    Interval{ 0x0162, 0x0162 }, // L&       LATIN CAPITAL LETTER T WITH CEDILLA
    Interval{ 0x0164, 0x0164 }, // L&       LATIN CAPITAL LETTER T WITH CARON
    Interval{ 0x0166, 0x0166 }, // L&       LATIN CAPITAL LETTER T WITH STROKE
    Interval{ 0x0168, 0x0168 }, // L&       LATIN CAPITAL LETTER U WITH TILDE
    Interval{ 0x016A, 0x016A }, // L&       LATIN CAPITAL LETTER U WITH MACRON
    Interval{ 0x016C, 0x016C }, // L&       LATIN CAPITAL LETTER U WITH BREVE
    Interval{ 0x016E, 0x016E }, // L&       LATIN CAPITAL LETTER U WITH RING ABOVE
    Interval{ 0x0170, 0x0170 }, // L&       LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x0172, 0x0172 }, // L&       LATIN CAPITAL LETTER U WITH OGONEK
    Interval{ 0x0174, 0x0174 }, // L&       LATIN CAPITAL LETTER W WITH CIRCUMFLEX
    Interval{ 0x0176, 0x0176 }, // L&       LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
    Interval{ 0x0178, 0x0179 }, // L&   [2] LATIN CAPITAL LETTER Y WITH DIAERESIS..LATIN CAPITAL LETTER Z WITH ACUTE
    Interval{ 0x017B, 0x017B }, // L&       LATIN CAPITAL LETTER Z WITH DOT ABOVE
    Interval{ 0x017D, 0x017D }, // L&       LATIN CAPITAL LETTER Z WITH CARON
    Interval{ 0x0181, 0x0182 }, // L&   [2] LATIN CAPITAL LETTER B WITH HOOK..LATIN CAPITAL LETTER B WITH TOPBAR
    Interval{ 0x0184, 0x0184 }, // L&       LATIN CAPITAL LETTER TONE SIX
    Interval{ 0x0186, 0x0187 }, // L&   [2] LATIN CAPITAL LETTER OPEN O..LATIN CAPITAL LETTER C WITH HOOK
    Interval{ 0x0189, 0x018B }, // L&   [3] LATIN CAPITAL LETTER AFRICAN D..LATIN CAPITAL LETTER D WITH TOPBAR
    Interval{ 0x018E, 0x0191 }, // L&   [4] LATIN CAPITAL LETTER REVERSED E..LATIN CAPITAL LETTER F WITH HOOK
    Interval{ 0x0193, 0x0194 }, // L&   [2] LATIN CAPITAL LETTER G WITH HOOK..LATIN CAPITAL LETTER GAMMA
    Interval{ 0x0196, 0x0198 }, // L&   [3] LATIN CAPITAL LETTER IOTA..LATIN CAPITAL LETTER K WITH HOOK
    Interval{ 0x019C, 0x019D }, // L&   [2] LATIN CAPITAL LETTER TURNED M..LATIN CAPITAL LETTER N WITH LEFT HOOK
    Interval{ 0x019F, 0x01A0 }, // L&   [2] LATIN CAPITAL LETTER O WITH MIDDLE TILDE..LATIN CAPITAL LETTER O WITH HORN
    Interval{ 0x01A2, 0x01A2 }, // L&       LATIN CAPITAL LETTER OI
    Interval{ 0x01A4, 0x01A4 }, // L&       LATIN CAPITAL LETTER P WITH HOOK
    Interval{ 0x01A6, 0x01A7 }, // L&   [2] LATIN LETTER YR..LATIN CAPITAL LETTER TONE TWO
    Interval{ 0x01A9, 0x01A9 }, // L&       LATIN CAPITAL LETTER ESH
    Interval{ 0x01AC, 0x01AC }, // L&       LATIN CAPITAL LETTER T WITH HOOK
    Interval{ 0x01AE, 0x01AF }, // L&   [2] LATIN CAPITAL LETTER T WITH RETROFLEX HOOK..LATIN CAPITAL LETTER U WITH HORN
    Interval{ 0x01B1, 0x01B3 }, // L&   [3] LATIN CAPITAL LETTER UPSILON..LATIN CAPITAL LETTER Y WITH HOOK
    Interval{ 0x01B5, 0x01B5 }, // L&       LATIN CAPITAL LETTER Z WITH STROKE
    Interval{ 0x01B7, 0x01B8 }, // L&   [2] LATIN CAPITAL LETTER EZH..LATIN CAPITAL LETTER EZH REVERSED
    Interval{ 0x01BC, 0x01BC }, // L&       LATIN CAPITAL LETTER TONE FIVE
    Interval{ 0x01C4, 0x01C5 }, // L&   [2] LATIN CAPITAL LETTER DZ WITH CARON..LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
    Interval{ 0x01C7, 0x01C8 }, // L&   [2] LATIN CAPITAL LETTER LJ..LATIN CAPITAL LETTER L WITH SMALL LETTER J
    Interval{ 0x01CA, 0x01CB }, // L&   [2] LATIN CAPITAL LETTER NJ..LATIN CAPITAL LETTER N WITH SMALL LETTER J
    Interval{ 0x01CD, 0x01CD }, // L&       LATIN CAPITAL LETTER A WITH CARON
    Interval{ 0x01CF, 0x01CF }, // L&       LATIN CAPITAL LETTER I WITH CARON
    Interval{ 0x01D1, 0x01D1 }, // L&       LATIN CAPITAL LETTER O WITH CARON
    Interval{ 0x01D3, 0x01D3 }, // L&       LATIN CAPITAL LETTER U WITH CARON
    Interval{ 0x01D5, 0x01D5 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
    Interval{ 0x01D7, 0x01D7 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
    Interval{ 0x01D9, 0x01D9 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
    Interval{ 0x01DB, 0x01DB }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
    Interval{ 0x01DE, 0x01DE }, // L&       LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
    Interval{ 0x01E0, 0x01E0 }, // L&       LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
    Interval{ 0x01E2, 0x01E2 }, // L&       LATIN CAPITAL LETTER AE WITH MACRON
    Interval{ 0x01E4, 0x01E4 }, // L&       LATIN CAPITAL LETTER G WITH STROKE
    Interval{ 0x01E6, 0x01E6 }, // L&       LATIN CAPITAL LETTER G WITH CARON
    Interval{ 0x01E8, 0x01E8 }, // L&       LATIN CAPITAL LETTER K WITH CARON
    Interval{ 0x01EA, 0x01EA }, // L&       LATIN CAPITAL LETTER O WITH OGONEK
    Interval{ 0x01EC, 0x01EC }, // L&       LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
    Interval{ 0x01EE, 0x01EE }, // L&       LATIN CAPITAL LETTER EZH WITH CARON
    Interval{ 0x01F1, 0x01F2 }, // L&   [2] LATIN CAPITAL LETTER DZ..LATIN CAPITAL LETTER D WITH SMALL LETTER Z
    Interval{ 0x01F4, 0x01F4 }, // L&       LATIN CAPITAL LETTER G WITH ACUTE
    Interval{ 0x01F6, 0x01F8 }, // L&   [3] LATIN CAPITAL LETTER HWAIR..LATIN CAPITAL LETTER N WITH GRAVE
    Interval{ 0x01FA, 0x01FA }, // L&       LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
    Interval{ 0x01FC, 0x01FC }, // L&       LATIN CAPITAL LETTER AE WITH ACUTE
    Interval{ 0x01FE, 0x01FE }, // L&       LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
    Interval{ 0x0200, 0x0200 }, // L&       LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
    Interval{ 0x0202, 0x0202 }, // L&       LATIN CAPITAL LETTER A WITH INVERTED BREVE
    Interval{ 0x0204, 0x0204 }, // L&       LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
    Interval{ 0x0206, 0x0206 }, // L&       LATIN CAPITAL LETTER E WITH INVERTED BREVE
    Interval{ 0x0208, 0x0208 }, // L&       LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
    Interval{ 0x020A, 0x020A }, // L&       LATIN CAPITAL LETTER I WITH INVERTED BREVE
    Interval{ 0x020C, 0x020C }, // L&       LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
    Interval{ 0x020E, 0x020E }, // L&       LATIN CAPITAL LETTER O WITH INVERTED BREVE
    Interval{ 0x0210, 0x0210 }, // L&       LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
    Interval{ 0x0212, 0x0212 }, // L&       LATIN CAPITAL LETTER R WITH INVERTED BREVE
    Interval{ 0x0214, 0x0214 }, // L&       LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
    Interval{ 0x0216, 0x0216 }, // L&       LATIN CAPITAL LETTER U WITH INVERTED BREVE
    Interval{ 0x0218, 0x0218 }, // L&       LATIN CAPITAL LETTER S WITH COMMA BELOW
    Interval{ 0x021A, 0x021A }, // L&       LATIN CAPITAL LETTER T WITH COMMA BELOW
    Interval{ 0x021C, 0x021C }, // L&       LATIN CAPITAL LETTER YOGH
    Interval{ 0x021E, 0x021E }, // L&       LATIN CAPITAL LETTER H WITH CARON
    Interval{ 0x0220, 0x0220 }, // L&       LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
    Interval{ 0x0222, 0x0222 }, // L&       LATIN CAPITAL LETTER OU
    Interval{ 0x0224, 0x0224 }, // L&       LATIN CAPITAL LETTER Z WITH HOOK
    Interval{ 0x0226, 0x0226 }, // L&       LATIN CAPITAL LETTER A WITH DOT ABOVE
    Interval{ 0x0228, 0x0228 }, // L&       LATIN CAPITAL LETTER E WITH CEDILLA
    Interval{ 0x022A, 0x022A }, // L&       LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
    Interval{ 0x022C, 0x022C }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND MACRON
    Interval{ 0x022E, 0x022E }, // L&       LATIN CAPITAL LETTER O WITH DOT ABOVE
    Interval{ 0x0230, 0x0230 }, // L&       LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
    Interval{ 0x0232, 0x0232 }, // L&       LATIN CAPITAL LETTER Y WITH MACRON
    Interval{ 0x023A, 0x023B }, // L&   [2] LATIN CAPITAL LETTER A WITH STROKE..LATIN CAPITAL LETTER C WITH STROKE
    Interval{ 0x023D, 0x023E }, // L&   [2] LATIN CAPITAL LETTER L WITH BAR..LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
    Interval{ 0x0241, 0x0241 }, // L&       LATIN CAPITAL LETTER GLOTTAL STOP
    Interval{ 0x0243, 0x0246 }, // L&   [4] LATIN CAPITAL LETTER B WITH STROKE..LATIN CAPITAL LETTER E WITH STROKE
    Interval{ 0x0248, 0x0248 }, // L&       LATIN CAPITAL LETTER J WITH STROKE
    Interval{ 0x024A, 0x024A }, // L&       LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
    Interval{ 0x024C, 0x024C }, // L&       LATIN CAPITAL LETTER R WITH STROKE
    Interval{ 0x024E, 0x024E }, // L&       LATIN CAPITAL LETTER Y WITH STROKE
    Interval{ 0x0370, 0x0370 }, // L&       GREEK CAPITAL LETTER HETA
    Interval{ 0x0372, 0x0372 }, // L&       GREEK CAPITAL LETTER ARCHAIC SAMPI
    Interval{ 0x0376, 0x0376 }, // L&       GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x038F }, // L&   [2] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER OMEGA WITH TONOS
    Interval{ 0x0391, 0x03A1 }, // L&  [17] GREEK CAPITAL LETTER ALPHA..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03AB }, // L&   [9] GREEK CAPITAL LETTER SIGMA..GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
    Interval{ 0x03CF, 0x03CF }, // L&       GREEK CAPITAL KAI SYMBOL
    Interval{ 0x03D8, 0x03D8 }, // L&       GREEK LETTER ARCHAIC KOPPA
    Interval{ 0x03DA, 0x03DA }, // L&       GREEK LETTER STIGMA
    Interval{ 0x03DC, 0x03DC }, // L&       GREEK LETTER DIGAMMA
    Interval{ 0x03DE, 0x03DE }, // L&       GREEK LETTER KOPPA
    Interval{ 0x03E0, 0x03E0 }, // L&       GREEK LETTER SAMPI
    Interval{ 0x03E2, 0x03E2 }, // L&       COPTIC CAPITAL LETTER SHEI
    Interval{ 0x03E4, 0x03E4 }, // L&       COPTIC CAPITAL LETTER FEI
    Interval{ 0x03E6, 0x03E6 }, // L&       COPTIC CAPITAL LETTER KHEI
    Interval{ 0x03E8, 0x03E8 }, // L&       COPTIC CAPITAL LETTER HORI
    Interval{ 0x03EA, 0x03EA }, // L&       COPTIC CAPITAL LETTER GANGIA
    Interval{ 0x03EC, 0x03EC }, // L&       COPTIC CAPITAL LETTER SHIMA
    Interval{ 0x03EE, 0x03EE }, // L&       COPTIC CAPITAL LETTER DEI
    Interval{ 0x03F4, 0x03F4 }, // L&       GREEK CAPITAL THETA SYMBOL
    Interval{ 0x03F7, 0x03F7 }, // L&       GREEK CAPITAL LETTER SHO
    Interval{ 0x03F9, 0x03FA }, // L&   [2] GREEK CAPITAL LUNATE SIGMA SYMBOL..GREEK CAPITAL LETTER SAN
    Interval{ 0x03FD, 0x042F }, // L&  [51] GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL..CYRILLIC CAPITAL LETTER YA
    Interval{ 0x0460, 0x0460 }, // L&       CYRILLIC CAPITAL LETTER OMEGA
    Interval{ 0x0462, 0x0462 }, // L&       CYRILLIC CAPITAL LETTER YAT
    Interval{ 0x0464, 0x0464 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED E
    Interval{ 0x0466, 0x0466 }, // L&       CYRILLIC CAPITAL LETTER LITTLE YUS
    Interval{ 0x0468, 0x0468 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
    Interval{ 0x046A, 0x046A }, // L&       CYRILLIC CAPITAL LETTER BIG YUS
    Interval{ 0x046C, 0x046C }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
    Interval{ 0x046E, 0x046E }, // L&       CYRILLIC CAPITAL LETTER KSI
    Interval{ 0x0470, 0x0470 }, // L&       CYRILLIC CAPITAL LETTER PSI
    Interval{ 0x0472, 0x0472 }, // L&       CYRILLIC CAPITAL LETTER FITA
    Interval{ 0x0474, 0x0474 }, // L&       CYRILLIC CAPITAL LETTER IZHITSA
    Interval{ 0x0476, 0x0476 }, // L&       CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
    Interval{ 0x0478, 0x0478 }, // L&       CYRILLIC CAPITAL LETTER UK
    Interval{ 0x047A, 0x047A }, // L&       CYRILLIC CAPITAL LETTER ROUND OMEGA
    Interval{ 0x047C, 0x047C }, // L&       CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
    Interval{ 0x047E, 0x047E }, // L&       CYRILLIC CAPITAL LETTER OT
    Interval{ 0x0480, 0x0480 }, // L&       CYRILLIC CAPITAL LETTER KOPPA
    Interval{ 0x048A, 0x048A }, // L&       CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
    Interval{ 0x048C, 0x048C }, // L&       CYRILLIC CAPITAL LETTER SEMISOFT SIGN
    Interval{ 0x048E, 0x048E }, // L&       CYRILLIC CAPITAL LETTER ER WITH TICK
    Interval{ 0x0490, 0x0490 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH UPTURN
    Interval{ 0x0492, 0x0492 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH STROKE
    Interval{ 0x0494, 0x0494 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
    Interval{ 0x0496, 0x0496 }, // L&       CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
    Interval{ 0x0498, 0x0498 }, // L&       CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
    Interval{ 0x049A, 0x049A }, // L&       CYRILLIC CAPITAL LETTER KA WITH DESCENDER
    Interval{ 0x049C, 0x049C }, // L&       CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
    Interval{ 0x049E, 0x049E }, // L&       CYRILLIC CAPITAL LETTER KA WITH STROKE
    Interval{ 0x04A0, 0x04A0 }, // L&       CYRILLIC CAPITAL LETTER BASHKIR KA
    Interval{ 0x04A2, 0x04A2 }, // L&       CYRILLIC CAPITAL LETTER EN WITH DESCENDER
    Interval{ 0x04A4, 0x04A4 }, // L&       CYRILLIC CAPITAL LIGATURE EN GHE
    Interval{ 0x04A6, 0x04A6 }, // L&       CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
    Interval{ 0x04A8, 0x04A8 }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN HA
    Interval{ 0x04AA, 0x04AA }, // L&       CYRILLIC CAPITAL LETTER ES WITH DESCENDER
    Interval{ 0x04AC, 0x04AC }, // L&       CYRILLIC CAPITAL LETTER TE WITH DESCENDER
    Interval{ 0x04AE, 0x04AE }, // L&       CYRILLIC CAPITAL LETTER STRAIGHT U
    Interval{ 0x04B0, 0x04B0 }, // L&       CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
    Interval{ 0x04B2, 0x04B2 }, // L&       CYRILLIC CAPITAL LETTER HA WITH DESCENDER
    Interval{ 0x04B4, 0x04B4 }, // L&       CYRILLIC CAPITAL LIGATURE TE TSE
    Interval{ 0x04B6, 0x04B6 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
    Interval{ 0x04B8, 0x04B8 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
    Interval{ 0x04BA, 0x04BA }, // L&       CYRILLIC CAPITAL LETTER SHHA
    Interval{ 0x04BC, 0x04BC }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN CHE
    Interval{ 0x04BE, 0x04BE }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
    Interval{ 0x04C0, 0x04C1 }, // L&   [2] CYRILLIC LETTER PALOCHKA..CYRILLIC CAPITAL LETTER ZHE WITH BREVE
    Interval{ 0x04C3, 0x04C3 }, // L&       CYRILLIC CAPITAL LETTER KA WITH HOOK
    Interval{ 0x04C5, 0x04C5 }, // L&       CYRILLIC CAPITAL LETTER EL WITH TAIL
    Interval{ 0x04C7, 0x04C7 }, // L&       CYRILLIC CAPITAL LETTER EN WITH HOOK
    Interval{ 0x04C9, 0x04C9 }, // L&       CYRILLIC CAPITAL LETTER EN WITH TAIL
    Interval{ 0x04CB, 0x04CB }, // L&       CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
    Interval{ 0x04CD, 0x04CD }, // L&       CYRILLIC CAPITAL LETTER EM WITH TAIL
    Interval{ 0x04D0, 0x04D0 }, // L&       CYRILLIC CAPITAL LETTER A WITH BREVE
    Interval{ 0x04D2, 0x04D2 }, // L&       CYRILLIC CAPITAL LETTER A WITH DIAERESIS
    Interval{ 0x04D4, 0x04D4 }, // L&       CYRILLIC CAPITAL LIGATURE A IE
    Interval{ 0x04D6, 0x04D6 }, // L&       CYRILLIC CAPITAL LETTER IE WITH BREVE
    Interval{ 0x04D8, 0x04D8 }, // L&       CYRILLIC CAPITAL LETTER SCHWA
    Interval{ 0x04DA, 0x04DA }, // L&       CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
    Interval{ 0x04DC, 0x04DC }, // L&       CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
    Interval{ 0x04DE, 0x04DE }, // L&       CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
    Interval{ 0x04E0, 0x04E0 }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN DZE
    Interval{ 0x04E2, 0x04E2 }, // L&       CYRILLIC CAPITAL LETTER I WITH MACRON
    Interval{ 0x04E4, 0x04E4 }, // L&       CYRILLIC CAPITAL LETTER I WITH DIAERESIS
    Interval{ 0x04E6, 0x04E6 }, // L&       CYRILLIC CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x04E8, 0x04E8 }, // L&       CYRILLIC CAPITAL LETTER BARRED O
    Interval{ 0x04EA, 0x04EA }, // L&       CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
    Interval{ 0x04EC, 0x04EC }, // L&       CYRILLIC CAPITAL LETTER E WITH DIAERESIS
    Interval{ 0x04EE, 0x04EE }, // L&       CYRILLIC CAPITAL LETTER U WITH MACRON
    Interval{ 0x04F0, 0x04F0 }, // L&       CYRILLIC CAPITAL LETTER U WITH DIAERESIS
    Interval{ 0x04F2, 0x04F2 }, // L&       CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x04F4, 0x04F4 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
    Interval{ 0x04F6, 0x04F6 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
    Interval{ 0x04F8, 0x04F8 }, // L&       CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
    Interval{ 0x04FA, 0x04FA }, // L&       CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK
    Interval{ 0x04FC, 0x04FC }, // L&       CYRILLIC CAPITAL LETTER HA WITH HOOK
    Interval{ 0x04FE, 0x04FE }, // L&       CYRILLIC CAPITAL LETTER HA WITH STROKE
    Interval{ 0x0500, 0x0500 }, // L&       CYRILLIC CAPITAL LETTER KOMI DE
    Interval{ 0x0502, 0x0502 }, // L&       CYRILLIC CAPITAL LETTER KOMI DJE
    Interval{ 0x0504, 0x0504 }, // L&       CYRILLIC CAPITAL LETTER KOMI ZJE
    Interval{ 0x0506, 0x0506 }, // L&       CYRILLIC CAPITAL LETTER KOMI DZJE
    Interval{ 0x0508, 0x0508 }, // L&       CYRILLIC CAPITAL LETTER KOMI LJE
    Interval{ 0x050A, 0x050A }, // L&       CYRILLIC CAPITAL LETTER KOMI NJE
    Interval{ 0x050C, 0x050C }, // L&       CYRILLIC CAPITAL LETTER KOMI SJE
    Interval{ 0x050E, 0x050E }, // L&       CYRILLIC CAPITAL LETTER KOMI TJE
    Interval{ 0x0510, 0x0510 }, // L&       CYRILLIC CAPITAL LETTER REVERSED ZE
    Interval{ 0x0512, 0x0512 }, // L&       CYRILLIC CAPITAL LETTER EL WITH HOOK
    Interval{ 0x0514, 0x0514 }, // L&       CYRILLIC CAPITAL LETTER LHA
    Interval{ 0x0516, 0x0516 }, // L&       CYRILLIC CAPITAL LETTER RHA
    Interval{ 0x0518, 0x0518 }, // L&       CYRILLIC CAPITAL LETTER YAE
    Interval{ 0x051A, 0x051A }, // L&       CYRILLIC CAPITAL LETTER QA
    Interval{ 0x051C, 0x051C }, // L&       CYRILLIC CAPITAL LETTER WE
    Interval{ 0x051E, 0x051E }, // L&       CYRILLIC CAPITAL LETTER ALEUT KA
    Interval{ 0x0520, 0x0520 }, // L&       CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
    Interval{ 0x0522, 0x0522 }, // L&       CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
    Interval{ 0x0524, 0x0524 }, // L&       CYRILLIC CAPITAL LETTER PE WITH DESCENDER
    Interval{ 0x0526, 0x0526 }, // L&       CYRILLIC CAPITAL LETTER SHHA WITH DESCENDER
    Interval{ 0x0528, 0x0528 }, // L&       CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK
    Interval{ 0x052A, 0x052A }, // L&       CYRILLIC CAPITAL LETTER DZZHE
    Interval{ 0x052C, 0x052C }, // L&       CYRILLIC CAPITAL LETTER DCHE
    Interval{ 0x052E, 0x052E }, // L&       CYRILLIC CAPITAL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1E00, 0x1E00 }, // L&       LATIN CAPITAL LETTER A WITH RING BELOW
    Interval{ 0x1E02, 0x1E02 }, // L&       LATIN CAPITAL LETTER B WITH DOT ABOVE
    Interval{ 0x1E04, 0x1E04 }, // L&       LATIN CAPITAL LETTER B WITH DOT BELOW
    Interval{ 0x1E06, 0x1E06 }, // L&       LATIN CAPITAL LETTER B WITH LINE BELOW
    Interval{ 0x1E08, 0x1E08 }, // L&       LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
    Interval{ 0x1E0A, 0x1E0A }, // L&       LATIN CAPITAL LETTER D WITH DOT ABOVE
    Interval{ 0x1E0C, 0x1E0C }, // L&       LATIN CAPITAL LETTER D WITH DOT BELOW
    Interval{ 0x1E0E, 0x1E0E }, // L&       LATIN CAPITAL LETTER D WITH LINE BELOW
    Interval{ 0x1E10, 0x1E10 }, // L&       LATIN CAPITAL LETTER D WITH CEDILLA
    Interval{ 0x1E12, 0x1E12 }, // L&       LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
    Interval{ 0x1E14, 0x1E14 }, // L&       LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
    Interval{ 0x1E16, 0x1E16 }, // L&       LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
    Interval{ 0x1E18, 0x1E18 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
    Interval{ 0x1E1A, 0x1E1A }, // L&       LATIN CAPITAL LETTER E WITH TILDE BELOW
    Interval{ 0x1E1C, 0x1E1C }, // L&       LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
    Interval{ 0x1E1E, 0x1E1E }, // L&       LATIN CAPITAL LETTER F WITH DOT ABOVE
    Interval{ 0x1E20, 0x1E20 }, // L&       LATIN CAPITAL LETTER G WITH MACRON
    Interval{ 0x1E22, 0x1E22 }, // L&       LATIN CAPITAL LETTER H WITH DOT ABOVE
    Interval{ 0x1E24, 0x1E24 }, // L&       LATIN CAPITAL LETTER H WITH DOT BELOW
    Interval{ 0x1E26, 0x1E26 }, // L&       LATIN CAPITAL LETTER H WITH DIAERESIS
    Interval{ 0x1E28, 0x1E28 }, // L&       LATIN CAPITAL LETTER H WITH CEDILLA
    Interval{ 0x1E2A, 0x1E2A }, // L&       LATIN CAPITAL LETTER H WITH BREVE BELOW
    Interval{ 0x1E2C, 0x1E2C }, // L&       LATIN CAPITAL LETTER I WITH TILDE BELOW
    Interval{ 0x1E2E, 0x1E2E }, // L&       LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
    Interval{ 0x1E30, 0x1E30 }, // L&       LATIN CAPITAL LETTER K WITH ACUTE
    Interval{ 0x1E32, 0x1E32 }, // L&       LATIN CAPITAL LETTER K WITH DOT BELOW
    Interval{ 0x1E34, 0x1E34 }, // L&       LATIN CAPITAL LETTER K WITH LINE BELOW
    Interval{ 0x1E36, 0x1E36 }, // L&       LATIN CAPITAL LETTER L WITH DOT BELOW
    Interval{ 0x1E38, 0x1E38 }, // L&       LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
    Interval{ 0x1E3A, 0x1E3A }, // L&       LATIN CAPITAL LETTER L WITH LINE BELOW
    Interval{ 0x1E3C, 0x1E3C }, // L&       LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
    Interval{ 0x1E3E, 0x1E3E }, // L&       LATIN CAPITAL LETTER M WITH ACUTE
    Interval{ 0x1E40, 0x1E40 }, // L&       LATIN CAPITAL LETTER M WITH DOT ABOVE
    Interval{ 0x1E42, 0x1E42 }, // L&       LATIN CAPITAL LETTER M WITH DOT BELOW
    Interval{ 0x1E44, 0x1E44 }, // L&       LATIN CAPITAL LETTER N WITH DOT ABOVE
    Interval{ 0x1E46, 0x1E46 }, // L&       LATIN CAPITAL LETTER N WITH DOT BELOW
    Interval{ 0x1E48, 0x1E48 }, // L&       LATIN CAPITAL LETTER N WITH LINE BELOW
    Interval{ 0x1E4A, 0x1E4A }, // L&       LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
    Interval{ 0x1E4C, 0x1E4C }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
    Interval{ 0x1E4E, 0x1E4E }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
    Interval{ 0x1E50, 0x1E50 }, // L&       LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
    Interval{ 0x1E52, 0x1E52 }, // L&       LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
    Interval{ 0x1E54, 0x1E54 }, // L&       LATIN CAPITAL LETTER P WITH ACUTE
    Interval{ 0x1E56, 0x1E56 }, // L&       LATIN CAPITAL LETTER P WITH DOT ABOVE
    Interval{ 0x1E58, 0x1E58 }, // L&       LATIN CAPITAL LETTER R WITH DOT ABOVE
    Interval{ 0x1E5A, 0x1E5A }, // L&       LATIN CAPITAL LETTER R WITH DOT BELOW
    Interval{ 0x1E5C, 0x1E5C }, // L&       LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
    Interval{ 0x1E5E, 0x1E5E }, // L&       LATIN CAPITAL LETTER R WITH LINE BELOW
    Interval{ 0x1E60, 0x1E60 }, // L&       LATIN CAPITAL LETTER S WITH DOT ABOVE
    Interval{ 0x1E62, 0x1E62 }, // L&       LATIN CAPITAL LETTER S WITH DOT BELOW
    Interval{ 0x1E64, 0x1E64 }, // L&       LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
    Interval{ 0x1E66, 0x1E66 }, // L&       LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
    Interval{ 0x1E68, 0x1E68 }, // L&       LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
    Interval{ 0x1E6A, 0x1E6A }, // L&       LATIN CAPITAL LETTER T WITH DOT ABOVE
    Interval{ 0x1E6C, 0x1E6C }, // L&       LATIN CAPITAL LETTER T WITH DOT BELOW
    Interval{ 0x1E6E, 0x1E6E }, // L&       LATIN CAPITAL LETTER T WITH LINE BELOW
    Interval{ 0x1E70, 0x1E70 }, // L&       LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
    Interval{ 0x1E72, 0x1E72 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
    Interval{ 0x1E74, 0x1E74 }, // L&       LATIN CAPITAL LETTER U WITH TILDE BELOW
    Interval{ 0x1E76, 0x1E76 }, // L&       LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
    Interval{ 0x1E78, 0x1E78 }, // L&       LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
    Interval{ 0x1E7A, 0x1E7A }, // L&       LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
    Interval{ 0x1E7C, 0x1E7C }, // L&       LATIN CAPITAL LETTER V WITH TILDE
    Interval{ 0x1E7E, 0x1E7E }, // L&       LATIN CAPITAL LETTER V WITH DOT BELOW
    Interval{ 0x1E80, 0x1E80 }, // L&       LATIN CAPITAL LETTER W WITH GRAVE
    Interval{ 0x1E82, 0x1E82 }, // L&       LATIN CAPITAL LETTER W WITH ACUTE
    Interval{ 0x1E84, 0x1E84 }, // L&       LATIN CAPITAL LETTER W WITH DIAERESIS
    Interval{ 0x1E86, 0x1E86 }, // L&       LATIN CAPITAL LETTER W WITH DOT ABOVE
    Interval{ 0x1E88, 0x1E88 }, // L&       LATIN CAPITAL LETTER W WITH DOT BELOW
    Interval{ 0x1E8A, 0x1E8A }, // L&       LATIN CAPITAL LETTER X WITH DOT ABOVE
    Interval{ 0x1E8C, 0x1E8C }, // L&       LATIN CAPITAL LETTER X WITH DIAERESIS
    Interval{ 0x1E8E, 0x1E8E }, // L&       LATIN CAPITAL LETTER Y WITH DOT ABOVE
    Interval{ 0x1E90, 0x1E90 }, // L&       LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
    Interval{ 0x1E92, 0x1E92 }, // L&       LATIN CAPITAL LETTER Z WITH DOT BELOW
    Interval{ 0x1E94, 0x1E94 }, // L&       LATIN CAPITAL LETTER Z WITH LINE BELOW
    Interval{ 0x1E9E, 0x1E9E }, // L&       LATIN CAPITAL LETTER SHARP S
    Interval{ 0x1EA0, 0x1EA0 }, // L&       LATIN CAPITAL LETTER A WITH DOT BELOW
    Interval{ 0x1EA2, 0x1EA2 }, // L&       LATIN CAPITAL LETTER A WITH HOOK ABOVE
    Interval{ 0x1EA4, 0x1EA4 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EA6, 0x1EA6 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EA8, 0x1EA8 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EAA, 0x1EAA }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EAC, 0x1EAC }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EAE, 0x1EAE }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
    Interval{ 0x1EB0, 0x1EB0 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
    Interval{ 0x1EB2, 0x1EB2 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
    Interval{ 0x1EB4, 0x1EB4 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND TILDE
    Interval{ 0x1EB6, 0x1EB6 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
    Interval{ 0x1EB8, 0x1EB8 }, // L&       LATIN CAPITAL LETTER E WITH DOT BELOW
    Interval{ 0x1EBA, 0x1EBA }, // L&       LATIN CAPITAL LETTER E WITH HOOK ABOVE
    Interval{ 0x1EBC, 0x1EBC }, // L&       LATIN CAPITAL LETTER E WITH TILDE
    Interval{ 0x1EBE, 0x1EBE }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EC0, 0x1EC0 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EC2, 0x1EC2 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EC4, 0x1EC4 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EC6, 0x1EC6 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EC8, 0x1EC8 }, // L&       LATIN CAPITAL LETTER I WITH HOOK ABOVE
    Interval{ 0x1ECA, 0x1ECA }, // L&       LATIN CAPITAL LETTER I WITH DOT BELOW
    Interval{ 0x1ECC, 0x1ECC }, // L&       LATIN CAPITAL LETTER O WITH DOT BELOW
    Interval{ 0x1ECE, 0x1ECE }, // L&       LATIN CAPITAL LETTER O WITH HOOK ABOVE
    Interval{ 0x1ED0, 0x1ED0 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1ED2, 0x1ED2 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1ED4, 0x1ED4 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1ED6, 0x1ED6 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1ED8, 0x1ED8 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EDA, 0x1EDA }, // L&       LATIN CAPITAL LETTER O WITH HORN AND ACUTE
    Interval{ 0x1EDC, 0x1EDC }, // L&       LATIN CAPITAL LETTER O WITH HORN AND GRAVE
    Interval{ 0x1EDE, 0x1EDE }, // L&       LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
    Interval{ 0x1EE0, 0x1EE0 }, // L&       LATIN CAPITAL LETTER O WITH HORN AND TILDE
    Interval{ 0x1EE2, 0x1EE2 }, // L&       LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
    Interval{ 0x1EE4, 0x1EE4 }, // L&       LATIN CAPITAL LETTER U WITH DOT BELOW
    Interval{ 0x1EE6, 0x1EE6 }, // L&       LATIN CAPITAL LETTER U WITH HOOK ABOVE
    Interval{ 0x1EE8, 0x1EE8 }, // L&       LATIN CAPITAL LETTER U WITH HORN AND ACUTE
    Interval{ 0x1EEA, 0x1EEA }, // L&       LATIN CAPITAL LETTER U WITH HORN AND GRAVE
    Interval{ 0x1EEC, 0x1EEC }, // L&       LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
    Interval{ 0x1EEE, 0x1EEE }, // L&       LATIN CAPITAL LETTER U WITH HORN AND TILDE
    Interval{ 0x1EF0, 0x1EF0 }, // L&       LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
    Interval{ 0x1EF2, 0x1EF2 }, // L&       LATIN CAPITAL LETTER Y WITH GRAVE
    Interval{ 0x1EF4, 0x1EF4 }, // L&       LATIN CAPITAL LETTER Y WITH DOT BELOW
    Interval{ 0x1EF6, 0x1EF6 }, // L&       LATIN CAPITAL LETTER Y WITH HOOK ABOVE
    Interval{ 0x1EF8, 0x1EF8 }, // L&       LATIN CAPITAL LETTER Y WITH TILDE
    Interval{ 0x1EFA, 0x1EFA }, // L&       LATIN CAPITAL LETTER MIDDLE-WELSH LL
    Interval{ 0x1EFC, 0x1EFC }, // L&       LATIN CAPITAL LETTER MIDDLE-WELSH V
    Interval{ 0x1EFE, 0x1EFE }, // L&       LATIN CAPITAL LETTER Y WITH LOOP
    Interval{ 0x1F08, 0x1F0F }, // L&   [8] GREEK CAPITAL LETTER ALPHA WITH PSILI..GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F28, 0x1F2F }, // L&   [8] GREEK CAPITAL LETTER ETA WITH PSILI..GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F38, 0x1F3F }, // L&   [8] GREEK CAPITAL LETTER IOTA WITH PSILI..GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F5F }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F68, 0x1F6F }, // L&   [8] GREEK CAPITAL LETTER OMEGA WITH PSILI..GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F88, 0x1F8F }, // L&   [8] GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI..GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
    Interval{ 0x1F98, 0x1F9F }, // L&   [8] GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI..GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
    Interval{ 0x1FA8, 0x1FAF }, // L&   [8] GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI..GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
    Interval{ 0x1FB8, 0x1FBC }, // L&   [5] GREEK CAPITAL LETTER ALPHA WITH VRACHY..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FC8, 0x1FCC }, // L&   [5] GREEK CAPITAL LETTER EPSILON WITH VARIA..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD8, 0x1FDB }, // L&   [4] GREEK CAPITAL LETTER IOTA WITH VRACHY..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE8, 0x1FEC }, // L&   [5] GREEK CAPITAL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF8, 0x1FFC }, // L&   [5] GREEK CAPITAL LETTER OMICRON WITH VARIA..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x212A, 0x212B }, // L&   [2] KELVIN SIGN..ANGSTROM SIGN
    Interval{ 0x2132, 0x2132 }, // L&       TURNED CAPITAL F
    Interval{ 0x2160, 0x216F }, // Nl  [16] ROMAN NUMERAL ONE..ROMAN NUMERAL ONE THOUSAND
    Interval{ 0x2183, 0x2183 }, // L&       ROMAN NUMERAL REVERSED ONE HUNDRED
    Interval{ 0x24B6, 0x24CF }, // So  [26] CIRCLED LATIN CAPITAL LETTER A..CIRCLED LATIN CAPITAL LETTER Z
    Interval{ 0x2C00, 0x2C2F }, // L&  [48] GLAGOLITIC CAPITAL LETTER AZU..GLAGOLITIC CAPITAL LETTER CAUDATE CHRIVI
    Interval{ 0x2C60, 0x2C60 }, // L&       LATIN CAPITAL LETTER L WITH DOUBLE BAR
    Interval{ 0x2C62, 0x2C64 }, // L&   [3] LATIN CAPITAL LETTER L WITH MIDDLE TILDE..LATIN CAPITAL LETTER R WITH TAIL
    Interval{ 0x2C67, 0x2C67 }, // L&       LATIN CAPITAL LETTER H WITH DESCENDER
    Interval{ 0x2C69, 0x2C69 }, // L&       LATIN CAPITAL LETTER K WITH DESCENDER
    Interval{ 0x2C6B, 0x2C6B }, // L&       LATIN CAPITAL LETTER Z WITH DESCENDER
    Interval{ 0x2C6D, 0x2C70 }, // L&   [4] LATIN CAPITAL LETTER ALPHA..LATIN CAPITAL LETTER TURNED ALPHA
    Interval{ 0x2C72, 0x2C72 }, // L&       LATIN CAPITAL LETTER W WITH HOOK
    Interval{ 0x2C75, 0x2C75 }, // L&       LATIN CAPITAL LETTER HALF H
    Interval{ 0x2C7E, 0x2C80 }, // L&   [3] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC CAPITAL LETTER ALFA
    Interval{ 0x2C82, 0x2C82 }, // L&       COPTIC CAPITAL LETTER VIDA
    Interval{ 0x2C84, 0x2C84 }, // L&       COPTIC CAPITAL LETTER GAMMA
    Interval{ 0x2C86, 0x2C86 }, // L&       COPTIC CAPITAL LETTER DALDA
    Interval{ 0x2C88, 0x2C88 }, // L&       COPTIC CAPITAL LETTER EIE
    Interval{ 0x2C8A, 0x2C8A }, // L&       COPTIC CAPITAL LETTER SOU
    Interval{ 0x2C8C, 0x2C8C }, // L&       COPTIC CAPITAL LETTER ZATA
    Interval{ 0x2C8E, 0x2C8E }, // L&       COPTIC CAPITAL LETTER HATE
    Interval{ 0x2C90, 0x2C90 }, // L&       COPTIC CAPITAL LETTER THETHE
    Interval{ 0x2C92, 0x2C92 }, // L&       COPTIC CAPITAL LETTER IAUDA
    Interval{ 0x2C94, 0x2C94 }, // L&       COPTIC CAPITAL LETTER KAPA
    Interval{ 0x2C96, 0x2C96 }, // L&       COPTIC CAPITAL LETTER LAULA
    Interval{ 0x2C98, 0x2C98 }, // L&       COPTIC CAPITAL LETTER MI
    Interval{ 0x2C9A, 0x2C9A }, // L&       COPTIC CAPITAL LETTER NI
    Interval{ 0x2C9C, 0x2C9C }, // L&       COPTIC CAPITAL LETTER KSI
    Interval{ 0x2C9E, 0x2C9E }, // L&       COPTIC CAPITAL LETTER O
    Interval{ 0x2CA0, 0x2CA0 }, // L&       COPTIC CAPITAL LETTER PI
    Interval{ 0x2CA2, 0x2CA2 }, // L&       COPTIC CAPITAL LETTER RO
    Interval{ 0x2CA4, 0x2CA4 }, // L&       COPTIC CAPITAL LETTER SIMA
    Interval{ 0x2CA6, 0x2CA6 }, // L&       COPTIC CAPITAL LETTER TAU
    Interval{ 0x2CA8, 0x2CA8 }, // L&       COPTIC CAPITAL LETTER UA
    Interval{ 0x2CAA, 0x2CAA }, // L&       COPTIC CAPITAL LETTER FI
    Interval{ 0x2CAC, 0x2CAC }, // L&       COPTIC CAPITAL LETTER KHI
    Interval{ 0x2CAE, 0x2CAE }, // L&       COPTIC CAPITAL LETTER PSI
    Interval{ 0x2CB0, 0x2CB0 }, // L&       COPTIC CAPITAL LETTER OOU
    Interval{ 0x2CB2, 0x2CB2 }, // L&       COPTIC CAPITAL LETTER DIALECT-P ALEF
    Interval{ 0x2CB4, 0x2CB4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC AIN
    Interval{ 0x2CB6, 0x2CB6 }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
    Interval{ 0x2CB8, 0x2CB8 }, // L&       COPTIC CAPITAL LETTER DIALECT-P KAPA
    Interval{ 0x2CBA, 0x2CBA }, // L&       COPTIC CAPITAL LETTER DIALECT-P NI
    Interval{ 0x2CBC, 0x2CBC }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
    Interval{ 0x2CBE, 0x2CBE }, // L&       COPTIC CAPITAL LETTER OLD COPTIC OOU
    Interval{ 0x2CC0, 0x2CC0 }, // L&       COPTIC CAPITAL LETTER SAMPI
    Interval{ 0x2CC2, 0x2CC2 }, // L&       COPTIC CAPITAL LETTER CROSSED SHEI
    Interval{ 0x2CC4, 0x2CC4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC SHEI
    Interval{ 0x2CC6, 0x2CC6 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC ESH
    Interval{ 0x2CC8, 0x2CC8 }, // L&       COPTIC CAPITAL LETTER AKHMIMIC KHEI
    Interval{ 0x2CCA, 0x2CCA }, // L&       COPTIC CAPITAL LETTER DIALECT-P HORI
    Interval{ 0x2CCC, 0x2CCC }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HORI
    Interval{ 0x2CCE, 0x2CCE }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HA
    Interval{ 0x2CD0, 0x2CD0 }, // L&       COPTIC CAPITAL LETTER L-SHAPED HA
    Interval{ 0x2CD2, 0x2CD2 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HEI
    Interval{ 0x2CD4, 0x2CD4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HAT
    Interval{ 0x2CD6, 0x2CD6 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC GANGIA
    Interval{ 0x2CD8, 0x2CD8 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC DJA
    Interval{ 0x2CDA, 0x2CDA }, // L&       COPTIC CAPITAL LETTER OLD COPTIC SHIMA
    Interval{ 0x2CDC, 0x2CDC }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
    Interval{ 0x2CDE, 0x2CDE }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN NGI
    Interval{ 0x2CE0, 0x2CE0 }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN NYI
    Interval{ 0x2CE2, 0x2CE2 }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN WAU
    Interval{ 0x2CEB, 0x2CEB }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI
    Interval{ 0x2CED, 0x2CED }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF2, 0x2CF2 }, // L&       COPTIC CAPITAL LETTER BOHAIRIC KHEI
    Interval{ 0xA640, 0xA640 }, // L&       CYRILLIC CAPITAL LETTER ZEMLYA
    Interval{ 0xA642, 0xA642 }, // L&       CYRILLIC CAPITAL LETTER DZELO
    Interval{ 0xA644, 0xA644 }, // L&       CYRILLIC CAPITAL LETTER REVERSED DZE
    Interval{ 0xA646, 0xA646 }, // L&       CYRILLIC CAPITAL LETTER IOTA
    Interval{ 0xA648, 0xA648 }, // L&       CYRILLIC CAPITAL LETTER DJERV
    Interval{ 0xA64A, 0xA64A }, // L&       CYRILLIC CAPITAL LETTER MONOGRAPH UK
    Interval{ 0xA64C, 0xA64C }, // L&       CYRILLIC CAPITAL LETTER BROAD OMEGA
    Interval{ 0xA64E, 0xA64E }, // L&       CYRILLIC CAPITAL LETTER NEUTRAL YER
    Interval{ 0xA650, 0xA650 }, // L&       CYRILLIC CAPITAL LETTER YERU WITH BACK YER
    Interval{ 0xA652, 0xA652 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED YAT
    Interval{ 0xA654, 0xA654 }, // L&       CYRILLIC CAPITAL LETTER REVERSED YU
    Interval{ 0xA656, 0xA656 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED A
    Interval{ 0xA658, 0xA658 }, // L&       CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS
    Interval{ 0xA65A, 0xA65A }, // L&       CYRILLIC CAPITAL LETTER BLENDED YUS
    Interval{ 0xA65C, 0xA65C }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUS
    Interval{ 0xA65E, 0xA65E }, // L&       CYRILLIC CAPITAL LETTER YN
    Interval{ 0xA660, 0xA660 }, // L&       CYRILLIC CAPITAL LETTER REVERSED TSE
    Interval{ 0xA662, 0xA662 }, // L&       CYRILLIC CAPITAL LETTER SOFT DE
    Interval{ 0xA664, 0xA664 }, // L&       CYRILLIC CAPITAL LETTER SOFT EL
    Interval{ 0xA666, 0xA666 }, // L&       CYRILLIC CAPITAL LETTER SOFT EM
    Interval{ 0xA668, 0xA668 }, // L&       CYRILLIC CAPITAL LETTER MONOCULAR O
    Interval{ 0xA66A, 0xA66A }, // L&       CYRILLIC CAPITAL LETTER BINOCULAR O
    Interval{ 0xA66C, 0xA66C }, // L&       CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA680, 0xA680 }, // L&       CYRILLIC CAPITAL LETTER DWE
    Interval{ 0xA682, 0xA682 }, // L&       CYRILLIC CAPITAL LETTER DZWE
    Interval{ 0xA684, 0xA684 }, // L&       CYRILLIC CAPITAL LETTER ZHWE
    Interval{ 0xA686, 0xA686 }, // L&       CYRILLIC CAPITAL LETTER CCHE
    Interval{ 0xA688, 0xA688 }, // L&       CYRILLIC CAPITAL LETTER DZZE
    Interval{ 0xA68A, 0xA68A }, // L&       CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK
    Interval{ 0xA68C, 0xA68C }, // L&       CYRILLIC CAPITAL LETTER TWE
    Interval{ 0xA68E, 0xA68E }, // L&       CYRILLIC CAPITAL LETTER TSWE
    Interval{ 0xA690, 0xA690 }, // L&       CYRILLIC CAPITAL LETTER TSSE
    Interval{ 0xA692, 0xA692 }, // L&       CYRILLIC CAPITAL LETTER TCHE
    Interval{ 0xA694, 0xA694 }, // L&       CYRILLIC CAPITAL LETTER HWE
    Interval{ 0xA696, 0xA696 }, // L&       CYRILLIC CAPITAL LETTER SHWE
    Interval{ 0xA698, 0xA698 }, // L&       CYRILLIC CAPITAL LETTER DOUBLE O
    Interval{ 0xA69A, 0xA69A }, // L&       CYRILLIC CAPITAL LETTER CROSSED O
    Interval{ 0xA722, 0xA722 }, // L&       LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF
    Interval{ 0xA724, 0xA724 }, // L&       LATIN CAPITAL LETTER EGYPTOLOGICAL AIN
    Interval{ 0xA726, 0xA726 }, // L&       LATIN CAPITAL LETTER HENG
    Interval{ 0xA728, 0xA728 }, // L&       LATIN CAPITAL LETTER TZ
    Interval{ 0xA72A, 0xA72A }, // L&       LATIN CAPITAL LETTER TRESILLO
    Interval{ 0xA72C, 0xA72C }, // L&       LATIN CAPITAL LETTER CUATRILLO
    Interval{ 0xA72E, 0xA72E }, // L&       LATIN CAPITAL LETTER CUATRILLO WITH COMMA
    Interval{ 0xA732, 0xA732 }, // L&       LATIN CAPITAL LETTER AA
    Interval{ 0xA734, 0xA734 }, // L&       LATIN CAPITAL LETTER AO
    Interval{ 0xA736, 0xA736 }, // L&       LATIN CAPITAL LETTER AU
    Interval{ 0xA738, 0xA738 }, // L&       LATIN CAPITAL LETTER AV
    Interval{ 0xA73A, 0xA73A }, // L&       LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR
    Interval{ 0xA73C, 0xA73C }, // L&       LATIN CAPITAL LETTER AY
    Interval{ 0xA73E, 0xA73E }, // L&       LATIN CAPITAL LETTER REVERSED C WITH DOT
    Interval{ 0xA740, 0xA740 }, // L&       LATIN CAPITAL LETTER K WITH STROKE
    Interval{ 0xA742, 0xA742 }, // L&       LATIN CAPITAL LETTER K WITH DIAGONAL STROKE
    Interval{ 0xA744, 0xA744 }, // L&       LATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKE
    Interval{ 0xA746, 0xA746 }, // L&       LATIN CAPITAL LETTER BROKEN L
    Interval{ 0xA748, 0xA748 }, // L&       LATIN CAPITAL LETTER L WITH HIGH STROKE
    Interval{ 0xA74A, 0xA74A }, // L&       LATIN CAPITAL LETTER O WITH LONG STROKE OVERLAY
    Interval{ 0xA74C, 0xA74C }, // L&       LATIN CAPITAL LETTER O WITH LOOP
    Interval{ 0xA74E, 0xA74E }, // L&       LATIN CAPITAL LETTER OO
    Interval{ 0xA750, 0xA750 }, // L&       LATIN CAPITAL LETTER P WITH STROKE THROUGH DESCENDER
    Interval{ 0xA752, 0xA752 }, // L&       LATIN CAPITAL LETTER P WITH FLOURISH
    Interval{ 0xA754, 0xA754 }, // L&       LATIN CAPITAL LETTER P WITH SQUIRREL TAIL
    Interval{ 0xA756, 0xA756 }, // L&       LATIN CAPITAL LETTER Q WITH STROKE THROUGH DESCENDER
    Interval{ 0xA758, 0xA758 }, // L&       LATIN CAPITAL LETTER Q WITH DIAGONAL STROKE
    Interval{ 0xA75A, 0xA75A }, // L&       LATIN CAPITAL LETTER R ROTUNDA
    Interval{ 0xA75C, 0xA75C }, // L&       LATIN CAPITAL LETTER RUM ROTUNDA
    Interval{ 0xA75E, 0xA75E }, // L&       LATIN CAPITAL LETTER V WITH DIAGONAL STROKE
    Interval{ 0xA760, 0xA760 }, // L&       LATIN CAPITAL LETTER VY
    Interval{ 0xA762, 0xA762 }, // L&       LATIN CAPITAL LETTER VISIGOTHIC Z
    Interval{ 0xA764, 0xA764 }, // L&       LATIN CAPITAL LETTER THORN WITH STROKE
    Interval{ 0xA766, 0xA766 }, // L&       LATIN CAPITAL LETTER THORN WITH STROKE THROUGH DESCENDER
    Interval{ 0xA768, 0xA768 }, // L&       LATIN CAPITAL LETTER VEND
    Interval{ 0xA76A, 0xA76A }, // L&       LATIN CAPITAL LETTER ET
    Interval{ 0xA76C, 0xA76C }, // L&       LATIN CAPITAL LETTER IS
    Interval{ 0xA76E, 0xA76E }, // L&       LATIN CAPITAL LETTER CON
    Interval{ 0xA779, 0xA779 }, // L&       LATIN CAPITAL LETTER INSULAR D
    Interval{ 0xA77B, 0xA77B }, // L&       LATIN CAPITAL LETTER INSULAR F
    Interval{ 0xA77D, 0xA77E }, // L&   [2] LATIN CAPITAL LETTER INSULAR G..LATIN CAPITAL LETTER TURNED INSULAR G
    Interval{ 0xA780, 0xA780 }, // L&       LATIN CAPITAL LETTER TURNED L
    Interval{ 0xA782, 0xA782 }, // L&       LATIN CAPITAL LETTER INSULAR R
    Interval{ 0xA784, 0xA784 }, // L&       LATIN CAPITAL LETTER INSULAR S
    Interval{ 0xA786, 0xA786 }, // L&       LATIN CAPITAL LETTER INSULAR T
    Interval{ 0xA78B, 0xA78B }, // L&       LATIN CAPITAL LETTER SALTILLO
    Interval{ 0xA78D, 0xA78D }, // L&       LATIN CAPITAL LETTER TURNED H
    Interval{ 0xA790, 0xA790 }, // L&       LATIN CAPITAL LETTER N WITH DESCENDER
    Interval{ 0xA792, 0xA792 }, // L&       LATIN CAPITAL LETTER C WITH BAR
    Interval{ 0xA796, 0xA796 }, // L&       LATIN CAPITAL LETTER B WITH FLOURISH
    Interval{ 0xA798, 0xA798 }, // L&       LATIN CAPITAL LETTER F WITH STROKE
    Interval{ 0xA79A, 0xA79A }, // L&       LATIN CAPITAL LETTER VOLAPUK AE
    Interval{ 0xA79C, 0xA79C }, // L&       LATIN CAPITAL LETTER VOLAPUK OE
    Interval{ 0xA79E, 0xA79E }, // L&       LATIN CAPITAL LETTER VOLAPUK UE
    Interval{ 0xA7A0, 0xA7A0 }, // L&       LATIN CAPITAL LETTER G WITH OBLIQUE STROKE
    Interval{ 0xA7A2, 0xA7A2 }, // L&       LATIN CAPITAL LETTER K WITH OBLIQUE STROKE
    Interval{ 0xA7A4, 0xA7A4 }, // L&       LATIN CAPITAL LETTER N WITH OBLIQUE STROKE
    Interval{ 0xA7A6, 0xA7A6 }, // L&       LATIN CAPITAL LETTER R WITH OBLIQUE STROKE
    Interval{ 0xA7A8, 0xA7A8 }, // L&       LATIN CAPITAL LETTER S WITH OBLIQUE STROKE
    Interval{ 0xA7AA, 0xA7AE }, // L&   [5] LATIN CAPITAL LETTER H WITH HOOK..LATIN CAPITAL LETTER SMALL CAPITAL I
    Interval{ 0xA7B0, 0xA7B4 }, // L&   [5] LATIN CAPITAL LETTER TURNED K..LATIN CAPITAL LETTER BETA
    Interval{ 0xA7B6, 0xA7B6 }, // L&       LATIN CAPITAL LETTER OMEGA
    Interval{ 0xA7B8, 0xA7B8 }, // L&       LATIN CAPITAL LETTER U WITH STROKE
    Interval{ 0xA7BA, 0xA7BA }, // L&       LATIN CAPITAL LETTER GLOTTAL A
    Interval{ 0xA7BC, 0xA7BC }, // L&       LATIN CAPITAL LETTER GLOTTAL I
    Interval{ 0xA7BE, 0xA7BE }, // L&       LATIN CAPITAL LETTER GLOTTAL U
    Interval{ 0xA7C0, 0xA7C0 }, // L&       LATIN CAPITAL LETTER OLD POLISH O
    Interval{ 0xA7C2, 0xA7C2 }, // L&       LATIN CAPITAL LETTER ANGLICANA W
    Interval{ 0xA7C4, 0xA7C7 }, // L&   [4] LATIN CAPITAL LETTER C WITH PALATAL HOOK..LATIN CAPITAL LETTER D WITH SHORT STROKE OVERLAY
    Interval{ 0xA7C9, 0xA7C9 }, // L&       LATIN CAPITAL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D0 }, // L&       LATIN CAPITAL LETTER CLOSED INSULAR G
    Interval{ 0xA7D6, 0xA7D6 }, // L&       LATIN CAPITAL LETTER MIDDLE SCOTS S
    Interval{ 0xA7D8, 0xA7D8 }, // L&       LATIN CAPITAL LETTER SIGMOID S
    Interval{ 0xA7F5, 0xA7F5 }, // L&       LATIN CAPITAL LETTER REVERSED HALF H
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0x10400, 0x10427 }, // L&  [40] DESERET CAPITAL LETTER LONG I..DESERET CAPITAL LETTER EW
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x118A0, 0x118BF }, // L&  [32] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI CAPITAL LETTER VIYO
    Interval{ 0x16E40, 0x16E5F }, // L&  [32] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN CAPITAL LETTER Y
    Interval{ 0x1E900, 0x1E921 }, // L&  [34] ADLAM CAPITAL LETTER ALIF..ADLAM CAPITAL LETTER SHA
}; // }}}
auto static const Changes_When_Titlecased = std::array<Interval, 626>{ // {{{
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00DF, 0x00F6 }, // L&  [24] LATIN SMALL LETTER SHARP S..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x00FF }, // L&   [8] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER Y WITH DIAERESIS
    Interval{ 0x0101, 0x0101 }, // L&       LATIN SMALL LETTER A WITH MACRON
    Interval{ 0x0103, 0x0103 }, // L&       LATIN SMALL LETTER A WITH BREVE
    Interval{ 0x0105, 0x0105 }, // L&       LATIN SMALL LETTER A WITH OGONEK
    Interval{ 0x0107, 0x0107 }, // L&       LATIN SMALL LETTER C WITH ACUTE
    Interval{ 0x0109, 0x0109 }, // L&       LATIN SMALL LETTER C WITH CIRCUMFLEX
    Interval{ 0x010B, 0x010B }, // L&       LATIN SMALL LETTER C WITH DOT ABOVE
    Interval{ 0x010D, 0x010D }, // L&       LATIN SMALL LETTER C WITH CARON
    Interval{ 0x010F, 0x010F }, // L&       LATIN SMALL LETTER D WITH CARON
    Interval{ 0x0111, 0x0111 }, // L&       LATIN SMALL LETTER D WITH STROKE
    Interval{ 0x0113, 0x0113 }, // L&       LATIN SMALL LETTER E WITH MACRON
    Interval{ 0x0115, 0x0115 }, // L&       LATIN SMALL LETTER E WITH BREVE
    Interval{ 0x0117, 0x0117 }, // L&       LATIN SMALL LETTER E WITH DOT ABOVE
    Interval{ 0x0119, 0x0119 }, // L&       LATIN SMALL LETTER E WITH OGONEK
    Interval{ 0x011B, 0x011B }, // L&       LATIN SMALL LETTER E WITH CARON
    Interval{ 0x011D, 0x011D }, // L&       LATIN SMALL LETTER G WITH CIRCUMFLEX
    Interval{ 0x011F, 0x011F }, // L&       LATIN SMALL LETTER G WITH BREVE
    Interval{ 0x0121, 0x0121 }, // L&       LATIN SMALL LETTER G WITH DOT ABOVE
    Interval{ 0x0123, 0x0123 }, // L&       LATIN SMALL LETTER G WITH CEDILLA
    Interval{ 0x0125, 0x0125 }, // L&       LATIN SMALL LETTER H WITH CIRCUMFLEX
    Interval{ 0x0127, 0x0127 }, // L&       LATIN SMALL LETTER H WITH STROKE
    Interval{ 0x0129, 0x0129 }, // L&       LATIN SMALL LETTER I WITH TILDE
    Interval{ 0x012B, 0x012B }, // L&       LATIN SMALL LETTER I WITH MACRON
    Interval{ 0x012D, 0x012D }, // L&       LATIN SMALL LETTER I WITH BREVE
    Interval{ 0x012F, 0x012F }, // L&       LATIN SMALL LETTER I WITH OGONEK
    Interval{ 0x0131, 0x0131 }, // L&       LATIN SMALL LETTER DOTLESS I
    Interval{ 0x0133, 0x0133 }, // L&       LATIN SMALL LIGATURE IJ
    Interval{ 0x0135, 0x0135 }, // L&       LATIN SMALL LETTER J WITH CIRCUMFLEX
    Interval{ 0x0137, 0x0137 }, // L&       LATIN SMALL LETTER K WITH CEDILLA
    Interval{ 0x013A, 0x013A }, // L&       LATIN SMALL LETTER L WITH ACUTE
    Interval{ 0x013C, 0x013C }, // L&       LATIN SMALL LETTER L WITH CEDILLA
    Interval{ 0x013E, 0x013E }, // L&       LATIN SMALL LETTER L WITH CARON
    Interval{ 0x0140, 0x0140 }, // L&       LATIN SMALL LETTER L WITH MIDDLE DOT
    Interval{ 0x0142, 0x0142 }, // L&       LATIN SMALL LETTER L WITH STROKE
    Interval{ 0x0144, 0x0144 }, // L&       LATIN SMALL LETTER N WITH ACUTE
    Interval{ 0x0146, 0x0146 }, // L&       LATIN SMALL LETTER N WITH CEDILLA
    Interval{ 0x0148, 0x0149 }, // L&   [2] LATIN SMALL LETTER N WITH CARON..LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
    Interval{ 0x014B, 0x014B }, // L&       LATIN SMALL LETTER ENG
    Interval{ 0x014D, 0x014D }, // L&       LATIN SMALL LETTER O WITH MACRON
    Interval{ 0x014F, 0x014F }, // L&       LATIN SMALL LETTER O WITH BREVE
    Interval{ 0x0151, 0x0151 }, // L&       LATIN SMALL LETTER O WITH DOUBLE ACUTE
    Interval{ 0x0153, 0x0153 }, // L&       LATIN SMALL LIGATURE OE
    Interval{ 0x0155, 0x0155 }, // L&       LATIN SMALL LETTER R WITH ACUTE
    Interval{ 0x0157, 0x0157 }, // L&       LATIN SMALL LETTER R WITH CEDILLA
    Interval{ 0x0159, 0x0159 }, // L&       LATIN SMALL LETTER R WITH CARON
    Interval{ 0x015B, 0x015B }, // L&       LATIN SMALL LETTER S WITH ACUTE
    Interval{ 0x015D, 0x015D }, // L&       LATIN SMALL LETTER S WITH CIRCUMFLEX
    Interval{ 0x015F, 0x015F }, // L&       LATIN SMALL LETTER S WITH CEDILLA
    Interval{ 0x0161, 0x0161 }, // L&       LATIN SMALL LETTER S WITH CARON
    Interval{ 0x0163, 0x0163 }, // L&       LATIN SMALL LETTER T WITH CEDILLA
    Interval{ 0x0165, 0x0165 }, // L&       LATIN SMALL LETTER T WITH CARON
    Interval{ 0x0167, 0x0167 }, // L&       LATIN SMALL LETTER T WITH STROKE
    Interval{ 0x0169, 0x0169 }, // L&       LATIN SMALL LETTER U WITH TILDE
    Interval{ 0x016B, 0x016B }, // L&       LATIN SMALL LETTER U WITH MACRON
    Interval{ 0x016D, 0x016D }, // L&       LATIN SMALL LETTER U WITH BREVE
    Interval{ 0x016F, 0x016F }, // L&       LATIN SMALL LETTER U WITH RING ABOVE
    Interval{ 0x0171, 0x0171 }, // L&       LATIN SMALL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x0173, 0x0173 }, // L&       LATIN SMALL LETTER U WITH OGONEK
    Interval{ 0x0175, 0x0175 }, // L&       LATIN SMALL LETTER W WITH CIRCUMFLEX
    Interval{ 0x0177, 0x0177 }, // L&       LATIN SMALL LETTER Y WITH CIRCUMFLEX
    Interval{ 0x017A, 0x017A }, // L&       LATIN SMALL LETTER Z WITH ACUTE
    Interval{ 0x017C, 0x017C }, // L&       LATIN SMALL LETTER Z WITH DOT ABOVE
    Interval{ 0x017E, 0x0180 }, // L&   [3] LATIN SMALL LETTER Z WITH CARON..LATIN SMALL LETTER B WITH STROKE
    Interval{ 0x0183, 0x0183 }, // L&       LATIN SMALL LETTER B WITH TOPBAR
    Interval{ 0x0185, 0x0185 }, // L&       LATIN SMALL LETTER TONE SIX
    Interval{ 0x0188, 0x0188 }, // L&       LATIN SMALL LETTER C WITH HOOK
    Interval{ 0x018C, 0x018C }, // L&       LATIN SMALL LETTER D WITH TOPBAR
    Interval{ 0x0192, 0x0192 }, // L&       LATIN SMALL LETTER F WITH HOOK
    Interval{ 0x0195, 0x0195 }, // L&       LATIN SMALL LETTER HV
    Interval{ 0x0199, 0x019A }, // L&   [2] LATIN SMALL LETTER K WITH HOOK..LATIN SMALL LETTER L WITH BAR
    Interval{ 0x019E, 0x019E }, // L&       LATIN SMALL LETTER N WITH LONG RIGHT LEG
    Interval{ 0x01A1, 0x01A1 }, // L&       LATIN SMALL LETTER O WITH HORN
    Interval{ 0x01A3, 0x01A3 }, // L&       LATIN SMALL LETTER OI
    Interval{ 0x01A5, 0x01A5 }, // L&       LATIN SMALL LETTER P WITH HOOK
    Interval{ 0x01A8, 0x01A8 }, // L&       LATIN SMALL LETTER TONE TWO
    Interval{ 0x01AD, 0x01AD }, // L&       LATIN SMALL LETTER T WITH HOOK
    Interval{ 0x01B0, 0x01B0 }, // L&       LATIN SMALL LETTER U WITH HORN
    Interval{ 0x01B4, 0x01B4 }, // L&       LATIN SMALL LETTER Y WITH HOOK
    Interval{ 0x01B6, 0x01B6 }, // L&       LATIN SMALL LETTER Z WITH STROKE
    Interval{ 0x01B9, 0x01B9 }, // L&       LATIN SMALL LETTER EZH REVERSED
    Interval{ 0x01BD, 0x01BD }, // L&       LATIN SMALL LETTER TONE FIVE
    Interval{ 0x01BF, 0x01BF }, // L&       LATIN LETTER WYNN
    Interval{ 0x01C4, 0x01C4 }, // L&       LATIN CAPITAL LETTER DZ WITH CARON
    Interval{ 0x01C6, 0x01C7 }, // L&   [2] LATIN SMALL LETTER DZ WITH CARON..LATIN CAPITAL LETTER LJ
    Interval{ 0x01C9, 0x01CA }, // L&   [2] LATIN SMALL LETTER LJ..LATIN CAPITAL LETTER NJ
    Interval{ 0x01CC, 0x01CC }, // L&       LATIN SMALL LETTER NJ
    Interval{ 0x01CE, 0x01CE }, // L&       LATIN SMALL LETTER A WITH CARON
    Interval{ 0x01D0, 0x01D0 }, // L&       LATIN SMALL LETTER I WITH CARON
    Interval{ 0x01D2, 0x01D2 }, // L&       LATIN SMALL LETTER O WITH CARON
    Interval{ 0x01D4, 0x01D4 }, // L&       LATIN SMALL LETTER U WITH CARON
    Interval{ 0x01D6, 0x01D6 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
    Interval{ 0x01D8, 0x01D8 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
    Interval{ 0x01DA, 0x01DA }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND CARON
    Interval{ 0x01DC, 0x01DD }, // L&   [2] LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE..LATIN SMALL LETTER TURNED E
    Interval{ 0x01DF, 0x01DF }, // L&       LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
    Interval{ 0x01E1, 0x01E1 }, // L&       LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
    Interval{ 0x01E3, 0x01E3 }, // L&       LATIN SMALL LETTER AE WITH MACRON
    Interval{ 0x01E5, 0x01E5 }, // L&       LATIN SMALL LETTER G WITH STROKE
    Interval{ 0x01E7, 0x01E7 }, // L&       LATIN SMALL LETTER G WITH CARON
    Interval{ 0x01E9, 0x01E9 }, // L&       LATIN SMALL LETTER K WITH CARON
    Interval{ 0x01EB, 0x01EB }, // L&       LATIN SMALL LETTER O WITH OGONEK
    Interval{ 0x01ED, 0x01ED }, // L&       LATIN SMALL LETTER O WITH OGONEK AND MACRON
    Interval{ 0x01EF, 0x01F1 }, // L&   [3] LATIN SMALL LETTER EZH WITH CARON..LATIN CAPITAL LETTER DZ
    Interval{ 0x01F3, 0x01F3 }, // L&       LATIN SMALL LETTER DZ
    Interval{ 0x01F5, 0x01F5 }, // L&       LATIN SMALL LETTER G WITH ACUTE
    Interval{ 0x01F9, 0x01F9 }, // L&       LATIN SMALL LETTER N WITH GRAVE
    Interval{ 0x01FB, 0x01FB }, // L&       LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
    Interval{ 0x01FD, 0x01FD }, // L&       LATIN SMALL LETTER AE WITH ACUTE
    Interval{ 0x01FF, 0x01FF }, // L&       LATIN SMALL LETTER O WITH STROKE AND ACUTE
    Interval{ 0x0201, 0x0201 }, // L&       LATIN SMALL LETTER A WITH DOUBLE GRAVE
    Interval{ 0x0203, 0x0203 }, // L&       LATIN SMALL LETTER A WITH INVERTED BREVE
    Interval{ 0x0205, 0x0205 }, // L&       LATIN SMALL LETTER E WITH DOUBLE GRAVE
    Interval{ 0x0207, 0x0207 }, // L&       LATIN SMALL LETTER E WITH INVERTED BREVE
    Interval{ 0x0209, 0x0209 }, // L&       LATIN SMALL LETTER I WITH DOUBLE GRAVE
    Interval{ 0x020B, 0x020B }, // L&       LATIN SMALL LETTER I WITH INVERTED BREVE
    Interval{ 0x020D, 0x020D }, // L&       LATIN SMALL LETTER O WITH DOUBLE GRAVE
    Interval{ 0x020F, 0x020F }, // L&       LATIN SMALL LETTER O WITH INVERTED BREVE
    Interval{ 0x0211, 0x0211 }, // L&       LATIN SMALL LETTER R WITH DOUBLE GRAVE
    Interval{ 0x0213, 0x0213 }, // L&       LATIN SMALL LETTER R WITH INVERTED BREVE
    Interval{ 0x0215, 0x0215 }, // L&       LATIN SMALL LETTER U WITH DOUBLE GRAVE
    Interval{ 0x0217, 0x0217 }, // L&       LATIN SMALL LETTER U WITH INVERTED BREVE
    Interval{ 0x0219, 0x0219 }, // L&       LATIN SMALL LETTER S WITH COMMA BELOW
    Interval{ 0x021B, 0x021B }, // L&       LATIN SMALL LETTER T WITH COMMA BELOW
    Interval{ 0x021D, 0x021D }, // L&       LATIN SMALL LETTER YOGH
    Interval{ 0x021F, 0x021F }, // L&       LATIN SMALL LETTER H WITH CARON
    Interval{ 0x0223, 0x0223 }, // L&       LATIN SMALL LETTER OU
    Interval{ 0x0225, 0x0225 }, // L&       LATIN SMALL LETTER Z WITH HOOK
    Interval{ 0x0227, 0x0227 }, // L&       LATIN SMALL LETTER A WITH DOT ABOVE
    Interval{ 0x0229, 0x0229 }, // L&       LATIN SMALL LETTER E WITH CEDILLA
    Interval{ 0x022B, 0x022B }, // L&       LATIN SMALL LETTER O WITH DIAERESIS AND MACRON
    Interval{ 0x022D, 0x022D }, // L&       LATIN SMALL LETTER O WITH TILDE AND MACRON
    Interval{ 0x022F, 0x022F }, // L&       LATIN SMALL LETTER O WITH DOT ABOVE
    Interval{ 0x0231, 0x0231 }, // L&       LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON
    Interval{ 0x0233, 0x0233 }, // L&       LATIN SMALL LETTER Y WITH MACRON
    Interval{ 0x023C, 0x023C }, // L&       LATIN SMALL LETTER C WITH STROKE
    Interval{ 0x023F, 0x0240 }, // L&   [2] LATIN SMALL LETTER S WITH SWASH TAIL..LATIN SMALL LETTER Z WITH SWASH TAIL
    Interval{ 0x0242, 0x0242 }, // L&       LATIN SMALL LETTER GLOTTAL STOP
    Interval{ 0x0247, 0x0247 }, // L&       LATIN SMALL LETTER E WITH STROKE
    Interval{ 0x0249, 0x0249 }, // L&       LATIN SMALL LETTER J WITH STROKE
    Interval{ 0x024B, 0x024B }, // L&       LATIN SMALL LETTER Q WITH HOOK TAIL
    Interval{ 0x024D, 0x024D }, // L&       LATIN SMALL LETTER R WITH STROKE
    Interval{ 0x024F, 0x0254 }, // L&   [6] LATIN SMALL LETTER Y WITH STROKE..LATIN SMALL LETTER OPEN O
    Interval{ 0x0256, 0x0257 }, // L&   [2] LATIN SMALL LETTER D WITH TAIL..LATIN SMALL LETTER D WITH HOOK
    Interval{ 0x0259, 0x0259 }, // L&       LATIN SMALL LETTER SCHWA
    Interval{ 0x025B, 0x025C }, // L&   [2] LATIN SMALL LETTER OPEN E..LATIN SMALL LETTER REVERSED OPEN E
    Interval{ 0x0260, 0x0261 }, // L&   [2] LATIN SMALL LETTER G WITH HOOK..LATIN SMALL LETTER SCRIPT G
    Interval{ 0x0263, 0x0263 }, // L&       LATIN SMALL LETTER GAMMA
    Interval{ 0x0265, 0x0266 }, // L&   [2] LATIN SMALL LETTER TURNED H..LATIN SMALL LETTER H WITH HOOK
    Interval{ 0x0268, 0x026C }, // L&   [5] LATIN SMALL LETTER I WITH STROKE..LATIN SMALL LETTER L WITH BELT
    Interval{ 0x026F, 0x026F }, // L&       LATIN SMALL LETTER TURNED M
    Interval{ 0x0271, 0x0272 }, // L&   [2] LATIN SMALL LETTER M WITH HOOK..LATIN SMALL LETTER N WITH LEFT HOOK
    Interval{ 0x0275, 0x0275 }, // L&       LATIN SMALL LETTER BARRED O
    Interval{ 0x027D, 0x027D }, // L&       LATIN SMALL LETTER R WITH TAIL
    Interval{ 0x0280, 0x0280 }, // L&       LATIN LETTER SMALL CAPITAL R
    Interval{ 0x0282, 0x0283 }, // L&   [2] LATIN SMALL LETTER S WITH HOOK..LATIN SMALL LETTER ESH
    Interval{ 0x0287, 0x028C }, // L&   [6] LATIN SMALL LETTER TURNED T..LATIN SMALL LETTER TURNED V
    Interval{ 0x0292, 0x0292 }, // L&       LATIN SMALL LETTER EZH
    Interval{ 0x029D, 0x029E }, // L&   [2] LATIN SMALL LETTER J WITH CROSSED-TAIL..LATIN SMALL LETTER TURNED K
    Interval{ 0x0345, 0x0345 }, // Mn       COMBINING GREEK YPOGEGRAMMENI
    Interval{ 0x0371, 0x0371 }, // L&       GREEK SMALL LETTER HETA
    Interval{ 0x0373, 0x0373 }, // L&       GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0377, 0x0377 }, // L&       GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x0390, 0x0390 }, // L&       GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
    Interval{ 0x03AC, 0x03CE }, // L&  [35] GREEK SMALL LETTER ALPHA WITH TONOS..GREEK SMALL LETTER OMEGA WITH TONOS
    Interval{ 0x03D0, 0x03D1 }, // L&   [2] GREEK BETA SYMBOL..GREEK THETA SYMBOL
    Interval{ 0x03D5, 0x03D7 }, // L&   [3] GREEK PHI SYMBOL..GREEK KAI SYMBOL
    Interval{ 0x03D9, 0x03D9 }, // L&       GREEK SMALL LETTER ARCHAIC KOPPA
    Interval{ 0x03DB, 0x03DB }, // L&       GREEK SMALL LETTER STIGMA
    Interval{ 0x03DD, 0x03DD }, // L&       GREEK SMALL LETTER DIGAMMA
    Interval{ 0x03DF, 0x03DF }, // L&       GREEK SMALL LETTER KOPPA
    Interval{ 0x03E1, 0x03E1 }, // L&       GREEK SMALL LETTER SAMPI
    Interval{ 0x03E3, 0x03E3 }, // L&       COPTIC SMALL LETTER SHEI
    Interval{ 0x03E5, 0x03E5 }, // L&       COPTIC SMALL LETTER FEI
    Interval{ 0x03E7, 0x03E7 }, // L&       COPTIC SMALL LETTER KHEI
    Interval{ 0x03E9, 0x03E9 }, // L&       COPTIC SMALL LETTER HORI
    Interval{ 0x03EB, 0x03EB }, // L&       COPTIC SMALL LETTER GANGIA
    Interval{ 0x03ED, 0x03ED }, // L&       COPTIC SMALL LETTER SHIMA
    Interval{ 0x03EF, 0x03F3 }, // L&   [5] COPTIC SMALL LETTER DEI..GREEK LETTER YOT
    Interval{ 0x03F5, 0x03F5 }, // L&       GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F8, 0x03F8 }, // L&       GREEK SMALL LETTER SHO
    Interval{ 0x03FB, 0x03FB }, // L&       GREEK SMALL LETTER SAN
    Interval{ 0x0430, 0x045F }, // L&  [48] CYRILLIC SMALL LETTER A..CYRILLIC SMALL LETTER DZHE
    Interval{ 0x0461, 0x0461 }, // L&       CYRILLIC SMALL LETTER OMEGA
    Interval{ 0x0463, 0x0463 }, // L&       CYRILLIC SMALL LETTER YAT
    Interval{ 0x0465, 0x0465 }, // L&       CYRILLIC SMALL LETTER IOTIFIED E
    Interval{ 0x0467, 0x0467 }, // L&       CYRILLIC SMALL LETTER LITTLE YUS
    Interval{ 0x0469, 0x0469 }, // L&       CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS
    Interval{ 0x046B, 0x046B }, // L&       CYRILLIC SMALL LETTER BIG YUS
    Interval{ 0x046D, 0x046D }, // L&       CYRILLIC SMALL LETTER IOTIFIED BIG YUS
    Interval{ 0x046F, 0x046F }, // L&       CYRILLIC SMALL LETTER KSI
    Interval{ 0x0471, 0x0471 }, // L&       CYRILLIC SMALL LETTER PSI
    Interval{ 0x0473, 0x0473 }, // L&       CYRILLIC SMALL LETTER FITA
    Interval{ 0x0475, 0x0475 }, // L&       CYRILLIC SMALL LETTER IZHITSA
    Interval{ 0x0477, 0x0477 }, // L&       CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
    Interval{ 0x0479, 0x0479 }, // L&       CYRILLIC SMALL LETTER UK
    Interval{ 0x047B, 0x047B }, // L&       CYRILLIC SMALL LETTER ROUND OMEGA
    Interval{ 0x047D, 0x047D }, // L&       CYRILLIC SMALL LETTER OMEGA WITH TITLO
    Interval{ 0x047F, 0x047F }, // L&       CYRILLIC SMALL LETTER OT
    Interval{ 0x0481, 0x0481 }, // L&       CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x048B, 0x048B }, // L&       CYRILLIC SMALL LETTER SHORT I WITH TAIL
    Interval{ 0x048D, 0x048D }, // L&       CYRILLIC SMALL LETTER SEMISOFT SIGN
    Interval{ 0x048F, 0x048F }, // L&       CYRILLIC SMALL LETTER ER WITH TICK
    Interval{ 0x0491, 0x0491 }, // L&       CYRILLIC SMALL LETTER GHE WITH UPTURN
    Interval{ 0x0493, 0x0493 }, // L&       CYRILLIC SMALL LETTER GHE WITH STROKE
    Interval{ 0x0495, 0x0495 }, // L&       CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
    Interval{ 0x0497, 0x0497 }, // L&       CYRILLIC SMALL LETTER ZHE WITH DESCENDER
    Interval{ 0x0499, 0x0499 }, // L&       CYRILLIC SMALL LETTER ZE WITH DESCENDER
    Interval{ 0x049B, 0x049B }, // L&       CYRILLIC SMALL LETTER KA WITH DESCENDER
    Interval{ 0x049D, 0x049D }, // L&       CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
    Interval{ 0x049F, 0x049F }, // L&       CYRILLIC SMALL LETTER KA WITH STROKE
    Interval{ 0x04A1, 0x04A1 }, // L&       CYRILLIC SMALL LETTER BASHKIR KA
    Interval{ 0x04A3, 0x04A3 }, // L&       CYRILLIC SMALL LETTER EN WITH DESCENDER
    Interval{ 0x04A5, 0x04A5 }, // L&       CYRILLIC SMALL LIGATURE EN GHE
    Interval{ 0x04A7, 0x04A7 }, // L&       CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
    Interval{ 0x04A9, 0x04A9 }, // L&       CYRILLIC SMALL LETTER ABKHASIAN HA
    Interval{ 0x04AB, 0x04AB }, // L&       CYRILLIC SMALL LETTER ES WITH DESCENDER
    Interval{ 0x04AD, 0x04AD }, // L&       CYRILLIC SMALL LETTER TE WITH DESCENDER
    Interval{ 0x04AF, 0x04AF }, // L&       CYRILLIC SMALL LETTER STRAIGHT U
    Interval{ 0x04B1, 0x04B1 }, // L&       CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
    Interval{ 0x04B3, 0x04B3 }, // L&       CYRILLIC SMALL LETTER HA WITH DESCENDER
    Interval{ 0x04B5, 0x04B5 }, // L&       CYRILLIC SMALL LIGATURE TE TSE
    Interval{ 0x04B7, 0x04B7 }, // L&       CYRILLIC SMALL LETTER CHE WITH DESCENDER
    Interval{ 0x04B9, 0x04B9 }, // L&       CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
    Interval{ 0x04BB, 0x04BB }, // L&       CYRILLIC SMALL LETTER SHHA
    Interval{ 0x04BD, 0x04BD }, // L&       CYRILLIC SMALL LETTER ABKHASIAN CHE
    Interval{ 0x04BF, 0x04BF }, // L&       CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
    Interval{ 0x04C2, 0x04C2 }, // L&       CYRILLIC SMALL LETTER ZHE WITH BREVE
    Interval{ 0x04C4, 0x04C4 }, // L&       CYRILLIC SMALL LETTER KA WITH HOOK
    Interval{ 0x04C6, 0x04C6 }, // L&       CYRILLIC SMALL LETTER EL WITH TAIL
    Interval{ 0x04C8, 0x04C8 }, // L&       CYRILLIC SMALL LETTER EN WITH HOOK
    Interval{ 0x04CA, 0x04CA }, // L&       CYRILLIC SMALL LETTER EN WITH TAIL
    Interval{ 0x04CC, 0x04CC }, // L&       CYRILLIC SMALL LETTER KHAKASSIAN CHE
    Interval{ 0x04CE, 0x04CF }, // L&   [2] CYRILLIC SMALL LETTER EM WITH TAIL..CYRILLIC SMALL LETTER PALOCHKA
    Interval{ 0x04D1, 0x04D1 }, // L&       CYRILLIC SMALL LETTER A WITH BREVE
    Interval{ 0x04D3, 0x04D3 }, // L&       CYRILLIC SMALL LETTER A WITH DIAERESIS
    Interval{ 0x04D5, 0x04D5 }, // L&       CYRILLIC SMALL LIGATURE A IE
    Interval{ 0x04D7, 0x04D7 }, // L&       CYRILLIC SMALL LETTER IE WITH BREVE
    Interval{ 0x04D9, 0x04D9 }, // L&       CYRILLIC SMALL LETTER SCHWA
    Interval{ 0x04DB, 0x04DB }, // L&       CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
    Interval{ 0x04DD, 0x04DD }, // L&       CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
    Interval{ 0x04DF, 0x04DF }, // L&       CYRILLIC SMALL LETTER ZE WITH DIAERESIS
    Interval{ 0x04E1, 0x04E1 }, // L&       CYRILLIC SMALL LETTER ABKHASIAN DZE
    Interval{ 0x04E3, 0x04E3 }, // L&       CYRILLIC SMALL LETTER I WITH MACRON
    Interval{ 0x04E5, 0x04E5 }, // L&       CYRILLIC SMALL LETTER I WITH DIAERESIS
    Interval{ 0x04E7, 0x04E7 }, // L&       CYRILLIC SMALL LETTER O WITH DIAERESIS
    Interval{ 0x04E9, 0x04E9 }, // L&       CYRILLIC SMALL LETTER BARRED O
    Interval{ 0x04EB, 0x04EB }, // L&       CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
    Interval{ 0x04ED, 0x04ED }, // L&       CYRILLIC SMALL LETTER E WITH DIAERESIS
    Interval{ 0x04EF, 0x04EF }, // L&       CYRILLIC SMALL LETTER U WITH MACRON
    Interval{ 0x04F1, 0x04F1 }, // L&       CYRILLIC SMALL LETTER U WITH DIAERESIS
    Interval{ 0x04F3, 0x04F3 }, // L&       CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x04F5, 0x04F5 }, // L&       CYRILLIC SMALL LETTER CHE WITH DIAERESIS
    Interval{ 0x04F7, 0x04F7 }, // L&       CYRILLIC SMALL LETTER GHE WITH DESCENDER
    Interval{ 0x04F9, 0x04F9 }, // L&       CYRILLIC SMALL LETTER YERU WITH DIAERESIS
    Interval{ 0x04FB, 0x04FB }, // L&       CYRILLIC SMALL LETTER GHE WITH STROKE AND HOOK
    Interval{ 0x04FD, 0x04FD }, // L&       CYRILLIC SMALL LETTER HA WITH HOOK
    Interval{ 0x04FF, 0x04FF }, // L&       CYRILLIC SMALL LETTER HA WITH STROKE
    Interval{ 0x0501, 0x0501 }, // L&       CYRILLIC SMALL LETTER KOMI DE
    Interval{ 0x0503, 0x0503 }, // L&       CYRILLIC SMALL LETTER KOMI DJE
    Interval{ 0x0505, 0x0505 }, // L&       CYRILLIC SMALL LETTER KOMI ZJE
    Interval{ 0x0507, 0x0507 }, // L&       CYRILLIC SMALL LETTER KOMI DZJE
    Interval{ 0x0509, 0x0509 }, // L&       CYRILLIC SMALL LETTER KOMI LJE
    Interval{ 0x050B, 0x050B }, // L&       CYRILLIC SMALL LETTER KOMI NJE
    Interval{ 0x050D, 0x050D }, // L&       CYRILLIC SMALL LETTER KOMI SJE
    Interval{ 0x050F, 0x050F }, // L&       CYRILLIC SMALL LETTER KOMI TJE
    Interval{ 0x0511, 0x0511 }, // L&       CYRILLIC SMALL LETTER REVERSED ZE
    Interval{ 0x0513, 0x0513 }, // L&       CYRILLIC SMALL LETTER EL WITH HOOK
    Interval{ 0x0515, 0x0515 }, // L&       CYRILLIC SMALL LETTER LHA
    Interval{ 0x0517, 0x0517 }, // L&       CYRILLIC SMALL LETTER RHA
    Interval{ 0x0519, 0x0519 }, // L&       CYRILLIC SMALL LETTER YAE
    Interval{ 0x051B, 0x051B }, // L&       CYRILLIC SMALL LETTER QA
    Interval{ 0x051D, 0x051D }, // L&       CYRILLIC SMALL LETTER WE
    Interval{ 0x051F, 0x051F }, // L&       CYRILLIC SMALL LETTER ALEUT KA
    Interval{ 0x0521, 0x0521 }, // L&       CYRILLIC SMALL LETTER EL WITH MIDDLE HOOK
    Interval{ 0x0523, 0x0523 }, // L&       CYRILLIC SMALL LETTER EN WITH MIDDLE HOOK
    Interval{ 0x0525, 0x0525 }, // L&       CYRILLIC SMALL LETTER PE WITH DESCENDER
    Interval{ 0x0527, 0x0527 }, // L&       CYRILLIC SMALL LETTER SHHA WITH DESCENDER
    Interval{ 0x0529, 0x0529 }, // L&       CYRILLIC SMALL LETTER EN WITH LEFT HOOK
    Interval{ 0x052B, 0x052B }, // L&       CYRILLIC SMALL LETTER DZZHE
    Interval{ 0x052D, 0x052D }, // L&       CYRILLIC SMALL LETTER DCHE
    Interval{ 0x052F, 0x052F }, // L&       CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0561, 0x0587 }, // L&  [39] ARMENIAN SMALL LETTER AYB..ARMENIAN SMALL LIGATURE ECH YIWN
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1D79, 0x1D79 }, // L&       LATIN SMALL LETTER INSULAR G
    Interval{ 0x1D7D, 0x1D7D }, // L&       LATIN SMALL LETTER P WITH STROKE
    Interval{ 0x1D8E, 0x1D8E }, // L&       LATIN SMALL LETTER Z WITH PALATAL HOOK
    Interval{ 0x1E01, 0x1E01 }, // L&       LATIN SMALL LETTER A WITH RING BELOW
    Interval{ 0x1E03, 0x1E03 }, // L&       LATIN SMALL LETTER B WITH DOT ABOVE
    Interval{ 0x1E05, 0x1E05 }, // L&       LATIN SMALL LETTER B WITH DOT BELOW
    Interval{ 0x1E07, 0x1E07 }, // L&       LATIN SMALL LETTER B WITH LINE BELOW
    Interval{ 0x1E09, 0x1E09 }, // L&       LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
    Interval{ 0x1E0B, 0x1E0B }, // L&       LATIN SMALL LETTER D WITH DOT ABOVE
    Interval{ 0x1E0D, 0x1E0D }, // L&       LATIN SMALL LETTER D WITH DOT BELOW
    Interval{ 0x1E0F, 0x1E0F }, // L&       LATIN SMALL LETTER D WITH LINE BELOW
    Interval{ 0x1E11, 0x1E11 }, // L&       LATIN SMALL LETTER D WITH CEDILLA
    Interval{ 0x1E13, 0x1E13 }, // L&       LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
    Interval{ 0x1E15, 0x1E15 }, // L&       LATIN SMALL LETTER E WITH MACRON AND GRAVE
    Interval{ 0x1E17, 0x1E17 }, // L&       LATIN SMALL LETTER E WITH MACRON AND ACUTE
    Interval{ 0x1E19, 0x1E19 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
    Interval{ 0x1E1B, 0x1E1B }, // L&       LATIN SMALL LETTER E WITH TILDE BELOW
    Interval{ 0x1E1D, 0x1E1D }, // L&       LATIN SMALL LETTER E WITH CEDILLA AND BREVE
    Interval{ 0x1E1F, 0x1E1F }, // L&       LATIN SMALL LETTER F WITH DOT ABOVE
    Interval{ 0x1E21, 0x1E21 }, // L&       LATIN SMALL LETTER G WITH MACRON
    Interval{ 0x1E23, 0x1E23 }, // L&       LATIN SMALL LETTER H WITH DOT ABOVE
    Interval{ 0x1E25, 0x1E25 }, // L&       LATIN SMALL LETTER H WITH DOT BELOW
    Interval{ 0x1E27, 0x1E27 }, // L&       LATIN SMALL LETTER H WITH DIAERESIS
    Interval{ 0x1E29, 0x1E29 }, // L&       LATIN SMALL LETTER H WITH CEDILLA
    Interval{ 0x1E2B, 0x1E2B }, // L&       LATIN SMALL LETTER H WITH BREVE BELOW
    Interval{ 0x1E2D, 0x1E2D }, // L&       LATIN SMALL LETTER I WITH TILDE BELOW
    Interval{ 0x1E2F, 0x1E2F }, // L&       LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
    Interval{ 0x1E31, 0x1E31 }, // L&       LATIN SMALL LETTER K WITH ACUTE
    Interval{ 0x1E33, 0x1E33 }, // L&       LATIN SMALL LETTER K WITH DOT BELOW
    Interval{ 0x1E35, 0x1E35 }, // L&       LATIN SMALL LETTER K WITH LINE BELOW
    Interval{ 0x1E37, 0x1E37 }, // L&       LATIN SMALL LETTER L WITH DOT BELOW
    Interval{ 0x1E39, 0x1E39 }, // L&       LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
    Interval{ 0x1E3B, 0x1E3B }, // L&       LATIN SMALL LETTER L WITH LINE BELOW
    Interval{ 0x1E3D, 0x1E3D }, // L&       LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
    Interval{ 0x1E3F, 0x1E3F }, // L&       LATIN SMALL LETTER M WITH ACUTE
    Interval{ 0x1E41, 0x1E41 }, // L&       LATIN SMALL LETTER M WITH DOT ABOVE
    Interval{ 0x1E43, 0x1E43 }, // L&       LATIN SMALL LETTER M WITH DOT BELOW
    Interval{ 0x1E45, 0x1E45 }, // L&       LATIN SMALL LETTER N WITH DOT ABOVE
    Interval{ 0x1E47, 0x1E47 }, // L&       LATIN SMALL LETTER N WITH DOT BELOW
    Interval{ 0x1E49, 0x1E49 }, // L&       LATIN SMALL LETTER N WITH LINE BELOW
    Interval{ 0x1E4B, 0x1E4B }, // L&       LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
    Interval{ 0x1E4D, 0x1E4D }, // L&       LATIN SMALL LETTER O WITH TILDE AND ACUTE
    Interval{ 0x1E4F, 0x1E4F }, // L&       LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
    Interval{ 0x1E51, 0x1E51 }, // L&       LATIN SMALL LETTER O WITH MACRON AND GRAVE
    Interval{ 0x1E53, 0x1E53 }, // L&       LATIN SMALL LETTER O WITH MACRON AND ACUTE
    Interval{ 0x1E55, 0x1E55 }, // L&       LATIN SMALL LETTER P WITH ACUTE
    Interval{ 0x1E57, 0x1E57 }, // L&       LATIN SMALL LETTER P WITH DOT ABOVE
    Interval{ 0x1E59, 0x1E59 }, // L&       LATIN SMALL LETTER R WITH DOT ABOVE
    Interval{ 0x1E5B, 0x1E5B }, // L&       LATIN SMALL LETTER R WITH DOT BELOW
    Interval{ 0x1E5D, 0x1E5D }, // L&       LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
    Interval{ 0x1E5F, 0x1E5F }, // L&       LATIN SMALL LETTER R WITH LINE BELOW
    Interval{ 0x1E61, 0x1E61 }, // L&       LATIN SMALL LETTER S WITH DOT ABOVE
    Interval{ 0x1E63, 0x1E63 }, // L&       LATIN SMALL LETTER S WITH DOT BELOW
    Interval{ 0x1E65, 0x1E65 }, // L&       LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
    Interval{ 0x1E67, 0x1E67 }, // L&       LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
    Interval{ 0x1E69, 0x1E69 }, // L&       LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
    Interval{ 0x1E6B, 0x1E6B }, // L&       LATIN SMALL LETTER T WITH DOT ABOVE
    Interval{ 0x1E6D, 0x1E6D }, // L&       LATIN SMALL LETTER T WITH DOT BELOW
    Interval{ 0x1E6F, 0x1E6F }, // L&       LATIN SMALL LETTER T WITH LINE BELOW
    Interval{ 0x1E71, 0x1E71 }, // L&       LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
    Interval{ 0x1E73, 0x1E73 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS BELOW
    Interval{ 0x1E75, 0x1E75 }, // L&       LATIN SMALL LETTER U WITH TILDE BELOW
    Interval{ 0x1E77, 0x1E77 }, // L&       LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
    Interval{ 0x1E79, 0x1E79 }, // L&       LATIN SMALL LETTER U WITH TILDE AND ACUTE
    Interval{ 0x1E7B, 0x1E7B }, // L&       LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
    Interval{ 0x1E7D, 0x1E7D }, // L&       LATIN SMALL LETTER V WITH TILDE
    Interval{ 0x1E7F, 0x1E7F }, // L&       LATIN SMALL LETTER V WITH DOT BELOW
    Interval{ 0x1E81, 0x1E81 }, // L&       LATIN SMALL LETTER W WITH GRAVE
    Interval{ 0x1E83, 0x1E83 }, // L&       LATIN SMALL LETTER W WITH ACUTE
    Interval{ 0x1E85, 0x1E85 }, // L&       LATIN SMALL LETTER W WITH DIAERESIS
    Interval{ 0x1E87, 0x1E87 }, // L&       LATIN SMALL LETTER W WITH DOT ABOVE
    Interval{ 0x1E89, 0x1E89 }, // L&       LATIN SMALL LETTER W WITH DOT BELOW
    Interval{ 0x1E8B, 0x1E8B }, // L&       LATIN SMALL LETTER X WITH DOT ABOVE
    Interval{ 0x1E8D, 0x1E8D }, // L&       LATIN SMALL LETTER X WITH DIAERESIS
    Interval{ 0x1E8F, 0x1E8F }, // L&       LATIN SMALL LETTER Y WITH DOT ABOVE
    Interval{ 0x1E91, 0x1E91 }, // L&       LATIN SMALL LETTER Z WITH CIRCUMFLEX
    Interval{ 0x1E93, 0x1E93 }, // L&       LATIN SMALL LETTER Z WITH DOT BELOW
    Interval{ 0x1E95, 0x1E9B }, // L&   [7] LATIN SMALL LETTER Z WITH LINE BELOW..LATIN SMALL LETTER LONG S WITH DOT ABOVE
    Interval{ 0x1EA1, 0x1EA1 }, // L&       LATIN SMALL LETTER A WITH DOT BELOW
    Interval{ 0x1EA3, 0x1EA3 }, // L&       LATIN SMALL LETTER A WITH HOOK ABOVE
    Interval{ 0x1EA5, 0x1EA5 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EA7, 0x1EA7 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EA9, 0x1EA9 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EAB, 0x1EAB }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EAD, 0x1EAD }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EAF, 0x1EAF }, // L&       LATIN SMALL LETTER A WITH BREVE AND ACUTE
    Interval{ 0x1EB1, 0x1EB1 }, // L&       LATIN SMALL LETTER A WITH BREVE AND GRAVE
    Interval{ 0x1EB3, 0x1EB3 }, // L&       LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
    Interval{ 0x1EB5, 0x1EB5 }, // L&       LATIN SMALL LETTER A WITH BREVE AND TILDE
    Interval{ 0x1EB7, 0x1EB7 }, // L&       LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
    Interval{ 0x1EB9, 0x1EB9 }, // L&       LATIN SMALL LETTER E WITH DOT BELOW
    Interval{ 0x1EBB, 0x1EBB }, // L&       LATIN SMALL LETTER E WITH HOOK ABOVE
    Interval{ 0x1EBD, 0x1EBD }, // L&       LATIN SMALL LETTER E WITH TILDE
    Interval{ 0x1EBF, 0x1EBF }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EC1, 0x1EC1 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EC3, 0x1EC3 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EC5, 0x1EC5 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EC7, 0x1EC7 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EC9, 0x1EC9 }, // L&       LATIN SMALL LETTER I WITH HOOK ABOVE
    Interval{ 0x1ECB, 0x1ECB }, // L&       LATIN SMALL LETTER I WITH DOT BELOW
    Interval{ 0x1ECD, 0x1ECD }, // L&       LATIN SMALL LETTER O WITH DOT BELOW
    Interval{ 0x1ECF, 0x1ECF }, // L&       LATIN SMALL LETTER O WITH HOOK ABOVE
    Interval{ 0x1ED1, 0x1ED1 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1ED3, 0x1ED3 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1ED5, 0x1ED5 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1ED7, 0x1ED7 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1ED9, 0x1ED9 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EDB, 0x1EDB }, // L&       LATIN SMALL LETTER O WITH HORN AND ACUTE
    Interval{ 0x1EDD, 0x1EDD }, // L&       LATIN SMALL LETTER O WITH HORN AND GRAVE
    Interval{ 0x1EDF, 0x1EDF }, // L&       LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
    Interval{ 0x1EE1, 0x1EE1 }, // L&       LATIN SMALL LETTER O WITH HORN AND TILDE
    Interval{ 0x1EE3, 0x1EE3 }, // L&       LATIN SMALL LETTER O WITH HORN AND DOT BELOW
    Interval{ 0x1EE5, 0x1EE5 }, // L&       LATIN SMALL LETTER U WITH DOT BELOW
    Interval{ 0x1EE7, 0x1EE7 }, // L&       LATIN SMALL LETTER U WITH HOOK ABOVE
    Interval{ 0x1EE9, 0x1EE9 }, // L&       LATIN SMALL LETTER U WITH HORN AND ACUTE
    Interval{ 0x1EEB, 0x1EEB }, // L&       LATIN SMALL LETTER U WITH HORN AND GRAVE
    Interval{ 0x1EED, 0x1EED }, // L&       LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
    Interval{ 0x1EEF, 0x1EEF }, // L&       LATIN SMALL LETTER U WITH HORN AND TILDE
    Interval{ 0x1EF1, 0x1EF1 }, // L&       LATIN SMALL LETTER U WITH HORN AND DOT BELOW
    Interval{ 0x1EF3, 0x1EF3 }, // L&       LATIN SMALL LETTER Y WITH GRAVE
    Interval{ 0x1EF5, 0x1EF5 }, // L&       LATIN SMALL LETTER Y WITH DOT BELOW
    Interval{ 0x1EF7, 0x1EF7 }, // L&       LATIN SMALL LETTER Y WITH HOOK ABOVE
    Interval{ 0x1EF9, 0x1EF9 }, // L&       LATIN SMALL LETTER Y WITH TILDE
    Interval{ 0x1EFB, 0x1EFB }, // L&       LATIN SMALL LETTER MIDDLE-WELSH LL
    Interval{ 0x1EFD, 0x1EFD }, // L&       LATIN SMALL LETTER MIDDLE-WELSH V
    Interval{ 0x1EFF, 0x1F07 }, // L&   [9] LATIN SMALL LETTER Y WITH LOOP..GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F10, 0x1F15 }, // L&   [6] GREEK SMALL LETTER EPSILON WITH PSILI..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F27 }, // L&   [8] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F30, 0x1F37 }, // L&   [8] GREEK SMALL LETTER IOTA WITH PSILI..GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F40, 0x1F45 }, // L&   [6] GREEK SMALL LETTER OMICRON WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F60, 0x1F67 }, // L&   [8] GREEK SMALL LETTER OMEGA WITH PSILI..GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F70, 0x1F7D }, // L&  [14] GREEK SMALL LETTER ALPHA WITH VARIA..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1F87 }, // L&   [8] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1F90, 0x1F97 }, // L&   [8] GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FA0, 0x1FA7 }, // L&   [8] GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FB0, 0x1FB4 }, // L&   [5] GREEK SMALL LETTER ALPHA WITH VRACHY..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FB7 }, // L&   [2] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FC7 }, // L&   [2] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FD7 }, // L&   [2] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
    Interval{ 0x1FE0, 0x1FE7 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FF7 }, // L&   [2] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2170, 0x217F }, // Nl  [16] SMALL ROMAN NUMERAL ONE..SMALL ROMAN NUMERAL ONE THOUSAND
    Interval{ 0x2184, 0x2184 }, // L&       LATIN SMALL LETTER REVERSED C
    Interval{ 0x24D0, 0x24E9 }, // So  [26] CIRCLED LATIN SMALL LETTER A..CIRCLED LATIN SMALL LETTER Z
    Interval{ 0x2C30, 0x2C5F }, // L&  [48] GLAGOLITIC SMALL LETTER AZU..GLAGOLITIC SMALL LETTER CAUDATE CHRIVI
    Interval{ 0x2C61, 0x2C61 }, // L&       LATIN SMALL LETTER L WITH DOUBLE BAR
    Interval{ 0x2C65, 0x2C66 }, // L&   [2] LATIN SMALL LETTER A WITH STROKE..LATIN SMALL LETTER T WITH DIAGONAL STROKE
    Interval{ 0x2C68, 0x2C68 }, // L&       LATIN SMALL LETTER H WITH DESCENDER
    Interval{ 0x2C6A, 0x2C6A }, // L&       LATIN SMALL LETTER K WITH DESCENDER
    Interval{ 0x2C6C, 0x2C6C }, // L&       LATIN SMALL LETTER Z WITH DESCENDER
    Interval{ 0x2C73, 0x2C73 }, // L&       LATIN SMALL LETTER W WITH HOOK
    Interval{ 0x2C76, 0x2C76 }, // L&       LATIN SMALL LETTER HALF H
    Interval{ 0x2C81, 0x2C81 }, // L&       COPTIC SMALL LETTER ALFA
    Interval{ 0x2C83, 0x2C83 }, // L&       COPTIC SMALL LETTER VIDA
    Interval{ 0x2C85, 0x2C85 }, // L&       COPTIC SMALL LETTER GAMMA
    Interval{ 0x2C87, 0x2C87 }, // L&       COPTIC SMALL LETTER DALDA
    Interval{ 0x2C89, 0x2C89 }, // L&       COPTIC SMALL LETTER EIE
    Interval{ 0x2C8B, 0x2C8B }, // L&       COPTIC SMALL LETTER SOU
    Interval{ 0x2C8D, 0x2C8D }, // L&       COPTIC SMALL LETTER ZATA
    Interval{ 0x2C8F, 0x2C8F }, // L&       COPTIC SMALL LETTER HATE
    Interval{ 0x2C91, 0x2C91 }, // L&       COPTIC SMALL LETTER THETHE
    Interval{ 0x2C93, 0x2C93 }, // L&       COPTIC SMALL LETTER IAUDA
    Interval{ 0x2C95, 0x2C95 }, // L&       COPTIC SMALL LETTER KAPA
    Interval{ 0x2C97, 0x2C97 }, // L&       COPTIC SMALL LETTER LAULA
    Interval{ 0x2C99, 0x2C99 }, // L&       COPTIC SMALL LETTER MI
    Interval{ 0x2C9B, 0x2C9B }, // L&       COPTIC SMALL LETTER NI
    Interval{ 0x2C9D, 0x2C9D }, // L&       COPTIC SMALL LETTER KSI
    Interval{ 0x2C9F, 0x2C9F }, // L&       COPTIC SMALL LETTER O
    Interval{ 0x2CA1, 0x2CA1 }, // L&       COPTIC SMALL LETTER PI
    Interval{ 0x2CA3, 0x2CA3 }, // L&       COPTIC SMALL LETTER RO
    Interval{ 0x2CA5, 0x2CA5 }, // L&       COPTIC SMALL LETTER SIMA
    Interval{ 0x2CA7, 0x2CA7 }, // L&       COPTIC SMALL LETTER TAU
    Interval{ 0x2CA9, 0x2CA9 }, // L&       COPTIC SMALL LETTER UA
    Interval{ 0x2CAB, 0x2CAB }, // L&       COPTIC SMALL LETTER FI
    Interval{ 0x2CAD, 0x2CAD }, // L&       COPTIC SMALL LETTER KHI
    Interval{ 0x2CAF, 0x2CAF }, // L&       COPTIC SMALL LETTER PSI
    Interval{ 0x2CB1, 0x2CB1 }, // L&       COPTIC SMALL LETTER OOU
    Interval{ 0x2CB3, 0x2CB3 }, // L&       COPTIC SMALL LETTER DIALECT-P ALEF
    Interval{ 0x2CB5, 0x2CB5 }, // L&       COPTIC SMALL LETTER OLD COPTIC AIN
    Interval{ 0x2CB7, 0x2CB7 }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC EIE
    Interval{ 0x2CB9, 0x2CB9 }, // L&       COPTIC SMALL LETTER DIALECT-P KAPA
    Interval{ 0x2CBB, 0x2CBB }, // L&       COPTIC SMALL LETTER DIALECT-P NI
    Interval{ 0x2CBD, 0x2CBD }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC NI
    Interval{ 0x2CBF, 0x2CBF }, // L&       COPTIC SMALL LETTER OLD COPTIC OOU
    Interval{ 0x2CC1, 0x2CC1 }, // L&       COPTIC SMALL LETTER SAMPI
    Interval{ 0x2CC3, 0x2CC3 }, // L&       COPTIC SMALL LETTER CROSSED SHEI
    Interval{ 0x2CC5, 0x2CC5 }, // L&       COPTIC SMALL LETTER OLD COPTIC SHEI
    Interval{ 0x2CC7, 0x2CC7 }, // L&       COPTIC SMALL LETTER OLD COPTIC ESH
    Interval{ 0x2CC9, 0x2CC9 }, // L&       COPTIC SMALL LETTER AKHMIMIC KHEI
    Interval{ 0x2CCB, 0x2CCB }, // L&       COPTIC SMALL LETTER DIALECT-P HORI
    Interval{ 0x2CCD, 0x2CCD }, // L&       COPTIC SMALL LETTER OLD COPTIC HORI
    Interval{ 0x2CCF, 0x2CCF }, // L&       COPTIC SMALL LETTER OLD COPTIC HA
    Interval{ 0x2CD1, 0x2CD1 }, // L&       COPTIC SMALL LETTER L-SHAPED HA
    Interval{ 0x2CD3, 0x2CD3 }, // L&       COPTIC SMALL LETTER OLD COPTIC HEI
    Interval{ 0x2CD5, 0x2CD5 }, // L&       COPTIC SMALL LETTER OLD COPTIC HAT
    Interval{ 0x2CD7, 0x2CD7 }, // L&       COPTIC SMALL LETTER OLD COPTIC GANGIA
    Interval{ 0x2CD9, 0x2CD9 }, // L&       COPTIC SMALL LETTER OLD COPTIC DJA
    Interval{ 0x2CDB, 0x2CDB }, // L&       COPTIC SMALL LETTER OLD COPTIC SHIMA
    Interval{ 0x2CDD, 0x2CDD }, // L&       COPTIC SMALL LETTER OLD NUBIAN SHIMA
    Interval{ 0x2CDF, 0x2CDF }, // L&       COPTIC SMALL LETTER OLD NUBIAN NGI
    Interval{ 0x2CE1, 0x2CE1 }, // L&       COPTIC SMALL LETTER OLD NUBIAN NYI
    Interval{ 0x2CE3, 0x2CE3 }, // L&       COPTIC SMALL LETTER OLD NUBIAN WAU
    Interval{ 0x2CEC, 0x2CEC }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC SHEI
    Interval{ 0x2CEE, 0x2CEE }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF3, 0x2CF3 }, // L&       COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0xA641, 0xA641 }, // L&       CYRILLIC SMALL LETTER ZEMLYA
    Interval{ 0xA643, 0xA643 }, // L&       CYRILLIC SMALL LETTER DZELO
    Interval{ 0xA645, 0xA645 }, // L&       CYRILLIC SMALL LETTER REVERSED DZE
    Interval{ 0xA647, 0xA647 }, // L&       CYRILLIC SMALL LETTER IOTA
    Interval{ 0xA649, 0xA649 }, // L&       CYRILLIC SMALL LETTER DJERV
    Interval{ 0xA64B, 0xA64B }, // L&       CYRILLIC SMALL LETTER MONOGRAPH UK
    Interval{ 0xA64D, 0xA64D }, // L&       CYRILLIC SMALL LETTER BROAD OMEGA
    Interval{ 0xA64F, 0xA64F }, // L&       CYRILLIC SMALL LETTER NEUTRAL YER
    Interval{ 0xA651, 0xA651 }, // L&       CYRILLIC SMALL LETTER YERU WITH BACK YER
    Interval{ 0xA653, 0xA653 }, // L&       CYRILLIC SMALL LETTER IOTIFIED YAT
    Interval{ 0xA655, 0xA655 }, // L&       CYRILLIC SMALL LETTER REVERSED YU
    Interval{ 0xA657, 0xA657 }, // L&       CYRILLIC SMALL LETTER IOTIFIED A
    Interval{ 0xA659, 0xA659 }, // L&       CYRILLIC SMALL LETTER CLOSED LITTLE YUS
    Interval{ 0xA65B, 0xA65B }, // L&       CYRILLIC SMALL LETTER BLENDED YUS
    Interval{ 0xA65D, 0xA65D }, // L&       CYRILLIC SMALL LETTER IOTIFIED CLOSED LITTLE YUS
    Interval{ 0xA65F, 0xA65F }, // L&       CYRILLIC SMALL LETTER YN
    Interval{ 0xA661, 0xA661 }, // L&       CYRILLIC SMALL LETTER REVERSED TSE
    Interval{ 0xA663, 0xA663 }, // L&       CYRILLIC SMALL LETTER SOFT DE
    Interval{ 0xA665, 0xA665 }, // L&       CYRILLIC SMALL LETTER SOFT EL
    Interval{ 0xA667, 0xA667 }, // L&       CYRILLIC SMALL LETTER SOFT EM
    Interval{ 0xA669, 0xA669 }, // L&       CYRILLIC SMALL LETTER MONOCULAR O
    Interval{ 0xA66B, 0xA66B }, // L&       CYRILLIC SMALL LETTER BINOCULAR O
    Interval{ 0xA66D, 0xA66D }, // L&       CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA681, 0xA681 }, // L&       CYRILLIC SMALL LETTER DWE
    Interval{ 0xA683, 0xA683 }, // L&       CYRILLIC SMALL LETTER DZWE
    Interval{ 0xA685, 0xA685 }, // L&       CYRILLIC SMALL LETTER ZHWE
    Interval{ 0xA687, 0xA687 }, // L&       CYRILLIC SMALL LETTER CCHE
    Interval{ 0xA689, 0xA689 }, // L&       CYRILLIC SMALL LETTER DZZE
    Interval{ 0xA68B, 0xA68B }, // L&       CYRILLIC SMALL LETTER TE WITH MIDDLE HOOK
    Interval{ 0xA68D, 0xA68D }, // L&       CYRILLIC SMALL LETTER TWE
    Interval{ 0xA68F, 0xA68F }, // L&       CYRILLIC SMALL LETTER TSWE
    Interval{ 0xA691, 0xA691 }, // L&       CYRILLIC SMALL LETTER TSSE
    Interval{ 0xA693, 0xA693 }, // L&       CYRILLIC SMALL LETTER TCHE
    Interval{ 0xA695, 0xA695 }, // L&       CYRILLIC SMALL LETTER HWE
    Interval{ 0xA697, 0xA697 }, // L&       CYRILLIC SMALL LETTER SHWE
    Interval{ 0xA699, 0xA699 }, // L&       CYRILLIC SMALL LETTER DOUBLE O
    Interval{ 0xA69B, 0xA69B }, // L&       CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA723, 0xA723 }, // L&       LATIN SMALL LETTER EGYPTOLOGICAL ALEF
    Interval{ 0xA725, 0xA725 }, // L&       LATIN SMALL LETTER EGYPTOLOGICAL AIN
    Interval{ 0xA727, 0xA727 }, // L&       LATIN SMALL LETTER HENG
    Interval{ 0xA729, 0xA729 }, // L&       LATIN SMALL LETTER TZ
    Interval{ 0xA72B, 0xA72B }, // L&       LATIN SMALL LETTER TRESILLO
    Interval{ 0xA72D, 0xA72D }, // L&       LATIN SMALL LETTER CUATRILLO
    Interval{ 0xA72F, 0xA72F }, // L&       LATIN SMALL LETTER CUATRILLO WITH COMMA
    Interval{ 0xA733, 0xA733 }, // L&       LATIN SMALL LETTER AA
    Interval{ 0xA735, 0xA735 }, // L&       LATIN SMALL LETTER AO
    Interval{ 0xA737, 0xA737 }, // L&       LATIN SMALL LETTER AU
    Interval{ 0xA739, 0xA739 }, // L&       LATIN SMALL LETTER AV
    Interval{ 0xA73B, 0xA73B }, // L&       LATIN SMALL LETTER AV WITH HORIZONTAL BAR
    Interval{ 0xA73D, 0xA73D }, // L&       LATIN SMALL LETTER AY
    Interval{ 0xA73F, 0xA73F }, // L&       LATIN SMALL LETTER REVERSED C WITH DOT
    Interval{ 0xA741, 0xA741 }, // L&       LATIN SMALL LETTER K WITH STROKE
    Interval{ 0xA743, 0xA743 }, // L&       LATIN SMALL LETTER K WITH DIAGONAL STROKE
    Interval{ 0xA745, 0xA745 }, // L&       LATIN SMALL LETTER K WITH STROKE AND DIAGONAL STROKE
    Interval{ 0xA747, 0xA747 }, // L&       LATIN SMALL LETTER BROKEN L
    Interval{ 0xA749, 0xA749 }, // L&       LATIN SMALL LETTER L WITH HIGH STROKE
    Interval{ 0xA74B, 0xA74B }, // L&       LATIN SMALL LETTER O WITH LONG STROKE OVERLAY
    Interval{ 0xA74D, 0xA74D }, // L&       LATIN SMALL LETTER O WITH LOOP
    Interval{ 0xA74F, 0xA74F }, // L&       LATIN SMALL LETTER OO
    Interval{ 0xA751, 0xA751 }, // L&       LATIN SMALL LETTER P WITH STROKE THROUGH DESCENDER
    Interval{ 0xA753, 0xA753 }, // L&       LATIN SMALL LETTER P WITH FLOURISH
    Interval{ 0xA755, 0xA755 }, // L&       LATIN SMALL LETTER P WITH SQUIRREL TAIL
    Interval{ 0xA757, 0xA757 }, // L&       LATIN SMALL LETTER Q WITH STROKE THROUGH DESCENDER
    Interval{ 0xA759, 0xA759 }, // L&       LATIN SMALL LETTER Q WITH DIAGONAL STROKE
    Interval{ 0xA75B, 0xA75B }, // L&       LATIN SMALL LETTER R ROTUNDA
    Interval{ 0xA75D, 0xA75D }, // L&       LATIN SMALL LETTER RUM ROTUNDA
    Interval{ 0xA75F, 0xA75F }, // L&       LATIN SMALL LETTER V WITH DIAGONAL STROKE
    Interval{ 0xA761, 0xA761 }, // L&       LATIN SMALL LETTER VY
    Interval{ 0xA763, 0xA763 }, // L&       LATIN SMALL LETTER VISIGOTHIC Z
    Interval{ 0xA765, 0xA765 }, // L&       LATIN SMALL LETTER THORN WITH STROKE
    Interval{ 0xA767, 0xA767 }, // L&       LATIN SMALL LETTER THORN WITH STROKE THROUGH DESCENDER
    Interval{ 0xA769, 0xA769 }, // L&       LATIN SMALL LETTER VEND
    Interval{ 0xA76B, 0xA76B }, // L&       LATIN SMALL LETTER ET
    Interval{ 0xA76D, 0xA76D }, // L&       LATIN SMALL LETTER IS
    Interval{ 0xA76F, 0xA76F }, // L&       LATIN SMALL LETTER CON
    Interval{ 0xA77A, 0xA77A }, // L&       LATIN SMALL LETTER INSULAR D
    Interval{ 0xA77C, 0xA77C }, // L&       LATIN SMALL LETTER INSULAR F
    Interval{ 0xA77F, 0xA77F }, // L&       LATIN SMALL LETTER TURNED INSULAR G
    Interval{ 0xA781, 0xA781 }, // L&       LATIN SMALL LETTER TURNED L
    Interval{ 0xA783, 0xA783 }, // L&       LATIN SMALL LETTER INSULAR R
    Interval{ 0xA785, 0xA785 }, // L&       LATIN SMALL LETTER INSULAR S
    Interval{ 0xA787, 0xA787 }, // L&       LATIN SMALL LETTER INSULAR T
    Interval{ 0xA78C, 0xA78C }, // L&       LATIN SMALL LETTER SALTILLO
    Interval{ 0xA791, 0xA791 }, // L&       LATIN SMALL LETTER N WITH DESCENDER
    Interval{ 0xA793, 0xA794 }, // L&   [2] LATIN SMALL LETTER C WITH BAR..LATIN SMALL LETTER C WITH PALATAL HOOK
    Interval{ 0xA797, 0xA797 }, // L&       LATIN SMALL LETTER B WITH FLOURISH
    Interval{ 0xA799, 0xA799 }, // L&       LATIN SMALL LETTER F WITH STROKE
    Interval{ 0xA79B, 0xA79B }, // L&       LATIN SMALL LETTER VOLAPUK AE
    Interval{ 0xA79D, 0xA79D }, // L&       LATIN SMALL LETTER VOLAPUK OE
    Interval{ 0xA79F, 0xA79F }, // L&       LATIN SMALL LETTER VOLAPUK UE
    Interval{ 0xA7A1, 0xA7A1 }, // L&       LATIN SMALL LETTER G WITH OBLIQUE STROKE
    Interval{ 0xA7A3, 0xA7A3 }, // L&       LATIN SMALL LETTER K WITH OBLIQUE STROKE
    Interval{ 0xA7A5, 0xA7A5 }, // L&       LATIN SMALL LETTER N WITH OBLIQUE STROKE
    Interval{ 0xA7A7, 0xA7A7 }, // L&       LATIN SMALL LETTER R WITH OBLIQUE STROKE
    Interval{ 0xA7A9, 0xA7A9 }, // L&       LATIN SMALL LETTER S WITH OBLIQUE STROKE
    Interval{ 0xA7B5, 0xA7B5 }, // L&       LATIN SMALL LETTER BETA
    Interval{ 0xA7B7, 0xA7B7 }, // L&       LATIN SMALL LETTER OMEGA
    Interval{ 0xA7B9, 0xA7B9 }, // L&       LATIN SMALL LETTER U WITH STROKE
    Interval{ 0xA7BB, 0xA7BB }, // L&       LATIN SMALL LETTER GLOTTAL A
    Interval{ 0xA7BD, 0xA7BD }, // L&       LATIN SMALL LETTER GLOTTAL I
    Interval{ 0xA7BF, 0xA7BF }, // L&       LATIN SMALL LETTER GLOTTAL U
    Interval{ 0xA7C1, 0xA7C1 }, // L&       LATIN SMALL LETTER OLD POLISH O
    Interval{ 0xA7C3, 0xA7C3 }, // L&       LATIN SMALL LETTER ANGLICANA W
    Interval{ 0xA7C8, 0xA7C8 }, // L&       LATIN SMALL LETTER D WITH SHORT STROKE OVERLAY
    Interval{ 0xA7CA, 0xA7CA }, // L&       LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D1, 0xA7D1 }, // L&       LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D7, 0xA7D7 }, // L&       LATIN SMALL LETTER MIDDLE SCOTS S
    Interval{ 0xA7D9, 0xA7D9 }, // L&       LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F6, 0xA7F6 }, // L&       LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xAB53, 0xAB53 }, // L&       LATIN SMALL LETTER CHI
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0x10428, 0x1044F }, // L&  [40] DESERET SMALL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x118C0, 0x118DF }, // L&  [32] WARANG CITI SMALL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x16E60, 0x16E7F }, // L&  [32] MEDEFAIDRIN SMALL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x1E922, 0x1E943 }, // L&  [34] ADLAM SMALL LETTER ALIF..ADLAM SMALL LETTER SHA
}; // }}}
auto static const Changes_When_Uppercased = std::array<Interval, 627>{ // {{{
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00DF, 0x00F6 }, // L&  [24] LATIN SMALL LETTER SHARP S..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x00FF }, // L&   [8] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER Y WITH DIAERESIS
    Interval{ 0x0101, 0x0101 }, // L&       LATIN SMALL LETTER A WITH MACRON
    Interval{ 0x0103, 0x0103 }, // L&       LATIN SMALL LETTER A WITH BREVE
    Interval{ 0x0105, 0x0105 }, // L&       LATIN SMALL LETTER A WITH OGONEK
    Interval{ 0x0107, 0x0107 }, // L&       LATIN SMALL LETTER C WITH ACUTE
    Interval{ 0x0109, 0x0109 }, // L&       LATIN SMALL LETTER C WITH CIRCUMFLEX
    Interval{ 0x010B, 0x010B }, // L&       LATIN SMALL LETTER C WITH DOT ABOVE
    Interval{ 0x010D, 0x010D }, // L&       LATIN SMALL LETTER C WITH CARON
    Interval{ 0x010F, 0x010F }, // L&       LATIN SMALL LETTER D WITH CARON
    Interval{ 0x0111, 0x0111 }, // L&       LATIN SMALL LETTER D WITH STROKE
    Interval{ 0x0113, 0x0113 }, // L&       LATIN SMALL LETTER E WITH MACRON
    Interval{ 0x0115, 0x0115 }, // L&       LATIN SMALL LETTER E WITH BREVE
    Interval{ 0x0117, 0x0117 }, // L&       LATIN SMALL LETTER E WITH DOT ABOVE
    Interval{ 0x0119, 0x0119 }, // L&       LATIN SMALL LETTER E WITH OGONEK
    Interval{ 0x011B, 0x011B }, // L&       LATIN SMALL LETTER E WITH CARON
    Interval{ 0x011D, 0x011D }, // L&       LATIN SMALL LETTER G WITH CIRCUMFLEX
    Interval{ 0x011F, 0x011F }, // L&       LATIN SMALL LETTER G WITH BREVE
    Interval{ 0x0121, 0x0121 }, // L&       LATIN SMALL LETTER G WITH DOT ABOVE
    Interval{ 0x0123, 0x0123 }, // L&       LATIN SMALL LETTER G WITH CEDILLA
    Interval{ 0x0125, 0x0125 }, // L&       LATIN SMALL LETTER H WITH CIRCUMFLEX
    Interval{ 0x0127, 0x0127 }, // L&       LATIN SMALL LETTER H WITH STROKE
    Interval{ 0x0129, 0x0129 }, // L&       LATIN SMALL LETTER I WITH TILDE
    Interval{ 0x012B, 0x012B }, // L&       LATIN SMALL LETTER I WITH MACRON
    Interval{ 0x012D, 0x012D }, // L&       LATIN SMALL LETTER I WITH BREVE
    Interval{ 0x012F, 0x012F }, // L&       LATIN SMALL LETTER I WITH OGONEK
    Interval{ 0x0131, 0x0131 }, // L&       LATIN SMALL LETTER DOTLESS I
    Interval{ 0x0133, 0x0133 }, // L&       LATIN SMALL LIGATURE IJ
    Interval{ 0x0135, 0x0135 }, // L&       LATIN SMALL LETTER J WITH CIRCUMFLEX
    Interval{ 0x0137, 0x0137 }, // L&       LATIN SMALL LETTER K WITH CEDILLA
    Interval{ 0x013A, 0x013A }, // L&       LATIN SMALL LETTER L WITH ACUTE
    Interval{ 0x013C, 0x013C }, // L&       LATIN SMALL LETTER L WITH CEDILLA
    Interval{ 0x013E, 0x013E }, // L&       LATIN SMALL LETTER L WITH CARON
    Interval{ 0x0140, 0x0140 }, // L&       LATIN SMALL LETTER L WITH MIDDLE DOT
    Interval{ 0x0142, 0x0142 }, // L&       LATIN SMALL LETTER L WITH STROKE
    Interval{ 0x0144, 0x0144 }, // L&       LATIN SMALL LETTER N WITH ACUTE
    Interval{ 0x0146, 0x0146 }, // L&       LATIN SMALL LETTER N WITH CEDILLA
    Interval{ 0x0148, 0x0149 }, // L&   [2] LATIN SMALL LETTER N WITH CARON..LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
    Interval{ 0x014B, 0x014B }, // L&       LATIN SMALL LETTER ENG
    Interval{ 0x014D, 0x014D }, // L&       LATIN SMALL LETTER O WITH MACRON
    Interval{ 0x014F, 0x014F }, // L&       LATIN SMALL LETTER O WITH BREVE
    Interval{ 0x0151, 0x0151 }, // L&       LATIN SMALL LETTER O WITH DOUBLE ACUTE
    Interval{ 0x0153, 0x0153 }, // L&       LATIN SMALL LIGATURE OE
    Interval{ 0x0155, 0x0155 }, // L&       LATIN SMALL LETTER R WITH ACUTE
    Interval{ 0x0157, 0x0157 }, // L&       LATIN SMALL LETTER R WITH CEDILLA
    Interval{ 0x0159, 0x0159 }, // L&       LATIN SMALL LETTER R WITH CARON
    Interval{ 0x015B, 0x015B }, // L&       LATIN SMALL LETTER S WITH ACUTE
    Interval{ 0x015D, 0x015D }, // L&       LATIN SMALL LETTER S WITH CIRCUMFLEX
    Interval{ 0x015F, 0x015F }, // L&       LATIN SMALL LETTER S WITH CEDILLA
    Interval{ 0x0161, 0x0161 }, // L&       LATIN SMALL LETTER S WITH CARON
    Interval{ 0x0163, 0x0163 }, // L&       LATIN SMALL LETTER T WITH CEDILLA
    Interval{ 0x0165, 0x0165 }, // L&       LATIN SMALL LETTER T WITH CARON
    Interval{ 0x0167, 0x0167 }, // L&       LATIN SMALL LETTER T WITH STROKE
    Interval{ 0x0169, 0x0169 }, // L&       LATIN SMALL LETTER U WITH TILDE
    Interval{ 0x016B, 0x016B }, // L&       LATIN SMALL LETTER U WITH MACRON
    Interval{ 0x016D, 0x016D }, // L&       LATIN SMALL LETTER U WITH BREVE
    Interval{ 0x016F, 0x016F }, // L&       LATIN SMALL LETTER U WITH RING ABOVE
    Interval{ 0x0171, 0x0171 }, // L&       LATIN SMALL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x0173, 0x0173 }, // L&       LATIN SMALL LETTER U WITH OGONEK
    Interval{ 0x0175, 0x0175 }, // L&       LATIN SMALL LETTER W WITH CIRCUMFLEX
    Interval{ 0x0177, 0x0177 }, // L&       LATIN SMALL LETTER Y WITH CIRCUMFLEX
    Interval{ 0x017A, 0x017A }, // L&       LATIN SMALL LETTER Z WITH ACUTE
    Interval{ 0x017C, 0x017C }, // L&       LATIN SMALL LETTER Z WITH DOT ABOVE
    Interval{ 0x017E, 0x0180 }, // L&   [3] LATIN SMALL LETTER Z WITH CARON..LATIN SMALL LETTER B WITH STROKE
    Interval{ 0x0183, 0x0183 }, // L&       LATIN SMALL LETTER B WITH TOPBAR
    Interval{ 0x0185, 0x0185 }, // L&       LATIN SMALL LETTER TONE SIX
    Interval{ 0x0188, 0x0188 }, // L&       LATIN SMALL LETTER C WITH HOOK
    Interval{ 0x018C, 0x018C }, // L&       LATIN SMALL LETTER D WITH TOPBAR
    Interval{ 0x0192, 0x0192 }, // L&       LATIN SMALL LETTER F WITH HOOK
    Interval{ 0x0195, 0x0195 }, // L&       LATIN SMALL LETTER HV
    Interval{ 0x0199, 0x019A }, // L&   [2] LATIN SMALL LETTER K WITH HOOK..LATIN SMALL LETTER L WITH BAR
    Interval{ 0x019E, 0x019E }, // L&       LATIN SMALL LETTER N WITH LONG RIGHT LEG
    Interval{ 0x01A1, 0x01A1 }, // L&       LATIN SMALL LETTER O WITH HORN
    Interval{ 0x01A3, 0x01A3 }, // L&       LATIN SMALL LETTER OI
    Interval{ 0x01A5, 0x01A5 }, // L&       LATIN SMALL LETTER P WITH HOOK
    Interval{ 0x01A8, 0x01A8 }, // L&       LATIN SMALL LETTER TONE TWO
    Interval{ 0x01AD, 0x01AD }, // L&       LATIN SMALL LETTER T WITH HOOK
    Interval{ 0x01B0, 0x01B0 }, // L&       LATIN SMALL LETTER U WITH HORN
    Interval{ 0x01B4, 0x01B4 }, // L&       LATIN SMALL LETTER Y WITH HOOK
    Interval{ 0x01B6, 0x01B6 }, // L&       LATIN SMALL LETTER Z WITH STROKE
    Interval{ 0x01B9, 0x01B9 }, // L&       LATIN SMALL LETTER EZH REVERSED
    Interval{ 0x01BD, 0x01BD }, // L&       LATIN SMALL LETTER TONE FIVE
    Interval{ 0x01BF, 0x01BF }, // L&       LATIN LETTER WYNN
    Interval{ 0x01C5, 0x01C6 }, // L&   [2] LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON..LATIN SMALL LETTER DZ WITH CARON
    Interval{ 0x01C8, 0x01C9 }, // L&   [2] LATIN CAPITAL LETTER L WITH SMALL LETTER J..LATIN SMALL LETTER LJ
    Interval{ 0x01CB, 0x01CC }, // L&   [2] LATIN CAPITAL LETTER N WITH SMALL LETTER J..LATIN SMALL LETTER NJ
    Interval{ 0x01CE, 0x01CE }, // L&       LATIN SMALL LETTER A WITH CARON
    Interval{ 0x01D0, 0x01D0 }, // L&       LATIN SMALL LETTER I WITH CARON
    Interval{ 0x01D2, 0x01D2 }, // L&       LATIN SMALL LETTER O WITH CARON
    Interval{ 0x01D4, 0x01D4 }, // L&       LATIN SMALL LETTER U WITH CARON
    Interval{ 0x01D6, 0x01D6 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
    Interval{ 0x01D8, 0x01D8 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
    Interval{ 0x01DA, 0x01DA }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND CARON
    Interval{ 0x01DC, 0x01DD }, // L&   [2] LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE..LATIN SMALL LETTER TURNED E
    Interval{ 0x01DF, 0x01DF }, // L&       LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
    Interval{ 0x01E1, 0x01E1 }, // L&       LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
    Interval{ 0x01E3, 0x01E3 }, // L&       LATIN SMALL LETTER AE WITH MACRON
    Interval{ 0x01E5, 0x01E5 }, // L&       LATIN SMALL LETTER G WITH STROKE
    Interval{ 0x01E7, 0x01E7 }, // L&       LATIN SMALL LETTER G WITH CARON
    Interval{ 0x01E9, 0x01E9 }, // L&       LATIN SMALL LETTER K WITH CARON
    Interval{ 0x01EB, 0x01EB }, // L&       LATIN SMALL LETTER O WITH OGONEK
    Interval{ 0x01ED, 0x01ED }, // L&       LATIN SMALL LETTER O WITH OGONEK AND MACRON
    Interval{ 0x01EF, 0x01F0 }, // L&   [2] LATIN SMALL LETTER EZH WITH CARON..LATIN SMALL LETTER J WITH CARON
    Interval{ 0x01F2, 0x01F3 }, // L&   [2] LATIN CAPITAL LETTER D WITH SMALL LETTER Z..LATIN SMALL LETTER DZ
    Interval{ 0x01F5, 0x01F5 }, // L&       LATIN SMALL LETTER G WITH ACUTE
    Interval{ 0x01F9, 0x01F9 }, // L&       LATIN SMALL LETTER N WITH GRAVE
    Interval{ 0x01FB, 0x01FB }, // L&       LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
    Interval{ 0x01FD, 0x01FD }, // L&       LATIN SMALL LETTER AE WITH ACUTE
    Interval{ 0x01FF, 0x01FF }, // L&       LATIN SMALL LETTER O WITH STROKE AND ACUTE
    Interval{ 0x0201, 0x0201 }, // L&       LATIN SMALL LETTER A WITH DOUBLE GRAVE
    Interval{ 0x0203, 0x0203 }, // L&       LATIN SMALL LETTER A WITH INVERTED BREVE
    Interval{ 0x0205, 0x0205 }, // L&       LATIN SMALL LETTER E WITH DOUBLE GRAVE
    Interval{ 0x0207, 0x0207 }, // L&       LATIN SMALL LETTER E WITH INVERTED BREVE
    Interval{ 0x0209, 0x0209 }, // L&       LATIN SMALL LETTER I WITH DOUBLE GRAVE
    Interval{ 0x020B, 0x020B }, // L&       LATIN SMALL LETTER I WITH INVERTED BREVE
    Interval{ 0x020D, 0x020D }, // L&       LATIN SMALL LETTER O WITH DOUBLE GRAVE
    Interval{ 0x020F, 0x020F }, // L&       LATIN SMALL LETTER O WITH INVERTED BREVE
    Interval{ 0x0211, 0x0211 }, // L&       LATIN SMALL LETTER R WITH DOUBLE GRAVE
    Interval{ 0x0213, 0x0213 }, // L&       LATIN SMALL LETTER R WITH INVERTED BREVE
    Interval{ 0x0215, 0x0215 }, // L&       LATIN SMALL LETTER U WITH DOUBLE GRAVE
    Interval{ 0x0217, 0x0217 }, // L&       LATIN SMALL LETTER U WITH INVERTED BREVE
    Interval{ 0x0219, 0x0219 }, // L&       LATIN SMALL LETTER S WITH COMMA BELOW
    Interval{ 0x021B, 0x021B }, // L&       LATIN SMALL LETTER T WITH COMMA BELOW
    Interval{ 0x021D, 0x021D }, // L&       LATIN SMALL LETTER YOGH
    Interval{ 0x021F, 0x021F }, // L&       LATIN SMALL LETTER H WITH CARON
    Interval{ 0x0223, 0x0223 }, // L&       LATIN SMALL LETTER OU
    Interval{ 0x0225, 0x0225 }, // L&       LATIN SMALL LETTER Z WITH HOOK
    Interval{ 0x0227, 0x0227 }, // L&       LATIN SMALL LETTER A WITH DOT ABOVE
    Interval{ 0x0229, 0x0229 }, // L&       LATIN SMALL LETTER E WITH CEDILLA
    Interval{ 0x022B, 0x022B }, // L&       LATIN SMALL LETTER O WITH DIAERESIS AND MACRON
    Interval{ 0x022D, 0x022D }, // L&       LATIN SMALL LETTER O WITH TILDE AND MACRON
    Interval{ 0x022F, 0x022F }, // L&       LATIN SMALL LETTER O WITH DOT ABOVE
    Interval{ 0x0231, 0x0231 }, // L&       LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON
    Interval{ 0x0233, 0x0233 }, // L&       LATIN SMALL LETTER Y WITH MACRON
    Interval{ 0x023C, 0x023C }, // L&       LATIN SMALL LETTER C WITH STROKE
    Interval{ 0x023F, 0x0240 }, // L&   [2] LATIN SMALL LETTER S WITH SWASH TAIL..LATIN SMALL LETTER Z WITH SWASH TAIL
    Interval{ 0x0242, 0x0242 }, // L&       LATIN SMALL LETTER GLOTTAL STOP
    Interval{ 0x0247, 0x0247 }, // L&       LATIN SMALL LETTER E WITH STROKE
    Interval{ 0x0249, 0x0249 }, // L&       LATIN SMALL LETTER J WITH STROKE
    Interval{ 0x024B, 0x024B }, // L&       LATIN SMALL LETTER Q WITH HOOK TAIL
    Interval{ 0x024D, 0x024D }, // L&       LATIN SMALL LETTER R WITH STROKE
    Interval{ 0x024F, 0x0254 }, // L&   [6] LATIN SMALL LETTER Y WITH STROKE..LATIN SMALL LETTER OPEN O
    Interval{ 0x0256, 0x0257 }, // L&   [2] LATIN SMALL LETTER D WITH TAIL..LATIN SMALL LETTER D WITH HOOK
    Interval{ 0x0259, 0x0259 }, // L&       LATIN SMALL LETTER SCHWA
    Interval{ 0x025B, 0x025C }, // L&   [2] LATIN SMALL LETTER OPEN E..LATIN SMALL LETTER REVERSED OPEN E
    Interval{ 0x0260, 0x0261 }, // L&   [2] LATIN SMALL LETTER G WITH HOOK..LATIN SMALL LETTER SCRIPT G
    Interval{ 0x0263, 0x0263 }, // L&       LATIN SMALL LETTER GAMMA
    Interval{ 0x0265, 0x0266 }, // L&   [2] LATIN SMALL LETTER TURNED H..LATIN SMALL LETTER H WITH HOOK
    Interval{ 0x0268, 0x026C }, // L&   [5] LATIN SMALL LETTER I WITH STROKE..LATIN SMALL LETTER L WITH BELT
    Interval{ 0x026F, 0x026F }, // L&       LATIN SMALL LETTER TURNED M
    Interval{ 0x0271, 0x0272 }, // L&   [2] LATIN SMALL LETTER M WITH HOOK..LATIN SMALL LETTER N WITH LEFT HOOK
    Interval{ 0x0275, 0x0275 }, // L&       LATIN SMALL LETTER BARRED O
    Interval{ 0x027D, 0x027D }, // L&       LATIN SMALL LETTER R WITH TAIL
    Interval{ 0x0280, 0x0280 }, // L&       LATIN LETTER SMALL CAPITAL R
    Interval{ 0x0282, 0x0283 }, // L&   [2] LATIN SMALL LETTER S WITH HOOK..LATIN SMALL LETTER ESH
    Interval{ 0x0287, 0x028C }, // L&   [6] LATIN SMALL LETTER TURNED T..LATIN SMALL LETTER TURNED V
    Interval{ 0x0292, 0x0292 }, // L&       LATIN SMALL LETTER EZH
    Interval{ 0x029D, 0x029E }, // L&   [2] LATIN SMALL LETTER J WITH CROSSED-TAIL..LATIN SMALL LETTER TURNED K
    Interval{ 0x0345, 0x0345 }, // Mn       COMBINING GREEK YPOGEGRAMMENI
    Interval{ 0x0371, 0x0371 }, // L&       GREEK SMALL LETTER HETA
    Interval{ 0x0373, 0x0373 }, // L&       GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0377, 0x0377 }, // L&       GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x0390, 0x0390 }, // L&       GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
    Interval{ 0x03AC, 0x03CE }, // L&  [35] GREEK SMALL LETTER ALPHA WITH TONOS..GREEK SMALL LETTER OMEGA WITH TONOS
    Interval{ 0x03D0, 0x03D1 }, // L&   [2] GREEK BETA SYMBOL..GREEK THETA SYMBOL
    Interval{ 0x03D5, 0x03D7 }, // L&   [3] GREEK PHI SYMBOL..GREEK KAI SYMBOL
    Interval{ 0x03D9, 0x03D9 }, // L&       GREEK SMALL LETTER ARCHAIC KOPPA
    Interval{ 0x03DB, 0x03DB }, // L&       GREEK SMALL LETTER STIGMA
    Interval{ 0x03DD, 0x03DD }, // L&       GREEK SMALL LETTER DIGAMMA
    Interval{ 0x03DF, 0x03DF }, // L&       GREEK SMALL LETTER KOPPA
    Interval{ 0x03E1, 0x03E1 }, // L&       GREEK SMALL LETTER SAMPI
    Interval{ 0x03E3, 0x03E3 }, // L&       COPTIC SMALL LETTER SHEI
    Interval{ 0x03E5, 0x03E5 }, // L&       COPTIC SMALL LETTER FEI
    Interval{ 0x03E7, 0x03E7 }, // L&       COPTIC SMALL LETTER KHEI
    Interval{ 0x03E9, 0x03E9 }, // L&       COPTIC SMALL LETTER HORI
    Interval{ 0x03EB, 0x03EB }, // L&       COPTIC SMALL LETTER GANGIA
    Interval{ 0x03ED, 0x03ED }, // L&       COPTIC SMALL LETTER SHIMA
    Interval{ 0x03EF, 0x03F3 }, // L&   [5] COPTIC SMALL LETTER DEI..GREEK LETTER YOT
    Interval{ 0x03F5, 0x03F5 }, // L&       GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F8, 0x03F8 }, // L&       GREEK SMALL LETTER SHO
    Interval{ 0x03FB, 0x03FB }, // L&       GREEK SMALL LETTER SAN
    Interval{ 0x0430, 0x045F }, // L&  [48] CYRILLIC SMALL LETTER A..CYRILLIC SMALL LETTER DZHE
    Interval{ 0x0461, 0x0461 }, // L&       CYRILLIC SMALL LETTER OMEGA
    Interval{ 0x0463, 0x0463 }, // L&       CYRILLIC SMALL LETTER YAT
    Interval{ 0x0465, 0x0465 }, // L&       CYRILLIC SMALL LETTER IOTIFIED E
    Interval{ 0x0467, 0x0467 }, // L&       CYRILLIC SMALL LETTER LITTLE YUS
    Interval{ 0x0469, 0x0469 }, // L&       CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS
    Interval{ 0x046B, 0x046B }, // L&       CYRILLIC SMALL LETTER BIG YUS
    Interval{ 0x046D, 0x046D }, // L&       CYRILLIC SMALL LETTER IOTIFIED BIG YUS
    Interval{ 0x046F, 0x046F }, // L&       CYRILLIC SMALL LETTER KSI
    Interval{ 0x0471, 0x0471 }, // L&       CYRILLIC SMALL LETTER PSI
    Interval{ 0x0473, 0x0473 }, // L&       CYRILLIC SMALL LETTER FITA
    Interval{ 0x0475, 0x0475 }, // L&       CYRILLIC SMALL LETTER IZHITSA
    Interval{ 0x0477, 0x0477 }, // L&       CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
    Interval{ 0x0479, 0x0479 }, // L&       CYRILLIC SMALL LETTER UK
    Interval{ 0x047B, 0x047B }, // L&       CYRILLIC SMALL LETTER ROUND OMEGA
    Interval{ 0x047D, 0x047D }, // L&       CYRILLIC SMALL LETTER OMEGA WITH TITLO
    Interval{ 0x047F, 0x047F }, // L&       CYRILLIC SMALL LETTER OT
    Interval{ 0x0481, 0x0481 }, // L&       CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x048B, 0x048B }, // L&       CYRILLIC SMALL LETTER SHORT I WITH TAIL
    Interval{ 0x048D, 0x048D }, // L&       CYRILLIC SMALL LETTER SEMISOFT SIGN
    Interval{ 0x048F, 0x048F }, // L&       CYRILLIC SMALL LETTER ER WITH TICK
    Interval{ 0x0491, 0x0491 }, // L&       CYRILLIC SMALL LETTER GHE WITH UPTURN
    Interval{ 0x0493, 0x0493 }, // L&       CYRILLIC SMALL LETTER GHE WITH STROKE
    Interval{ 0x0495, 0x0495 }, // L&       CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
    Interval{ 0x0497, 0x0497 }, // L&       CYRILLIC SMALL LETTER ZHE WITH DESCENDER
    Interval{ 0x0499, 0x0499 }, // L&       CYRILLIC SMALL LETTER ZE WITH DESCENDER
    Interval{ 0x049B, 0x049B }, // L&       CYRILLIC SMALL LETTER KA WITH DESCENDER
    Interval{ 0x049D, 0x049D }, // L&       CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
    Interval{ 0x049F, 0x049F }, // L&       CYRILLIC SMALL LETTER KA WITH STROKE
    Interval{ 0x04A1, 0x04A1 }, // L&       CYRILLIC SMALL LETTER BASHKIR KA
    Interval{ 0x04A3, 0x04A3 }, // L&       CYRILLIC SMALL LETTER EN WITH DESCENDER
    Interval{ 0x04A5, 0x04A5 }, // L&       CYRILLIC SMALL LIGATURE EN GHE
    Interval{ 0x04A7, 0x04A7 }, // L&       CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
    Interval{ 0x04A9, 0x04A9 }, // L&       CYRILLIC SMALL LETTER ABKHASIAN HA
    Interval{ 0x04AB, 0x04AB }, // L&       CYRILLIC SMALL LETTER ES WITH DESCENDER
    Interval{ 0x04AD, 0x04AD }, // L&       CYRILLIC SMALL LETTER TE WITH DESCENDER
    Interval{ 0x04AF, 0x04AF }, // L&       CYRILLIC SMALL LETTER STRAIGHT U
    Interval{ 0x04B1, 0x04B1 }, // L&       CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
    Interval{ 0x04B3, 0x04B3 }, // L&       CYRILLIC SMALL LETTER HA WITH DESCENDER
    Interval{ 0x04B5, 0x04B5 }, // L&       CYRILLIC SMALL LIGATURE TE TSE
    Interval{ 0x04B7, 0x04B7 }, // L&       CYRILLIC SMALL LETTER CHE WITH DESCENDER
    Interval{ 0x04B9, 0x04B9 }, // L&       CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
    Interval{ 0x04BB, 0x04BB }, // L&       CYRILLIC SMALL LETTER SHHA
    Interval{ 0x04BD, 0x04BD }, // L&       CYRILLIC SMALL LETTER ABKHASIAN CHE
    Interval{ 0x04BF, 0x04BF }, // L&       CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
    Interval{ 0x04C2, 0x04C2 }, // L&       CYRILLIC SMALL LETTER ZHE WITH BREVE
    Interval{ 0x04C4, 0x04C4 }, // L&       CYRILLIC SMALL LETTER KA WITH HOOK
    Interval{ 0x04C6, 0x04C6 }, // L&       CYRILLIC SMALL LETTER EL WITH TAIL
    Interval{ 0x04C8, 0x04C8 }, // L&       CYRILLIC SMALL LETTER EN WITH HOOK
    Interval{ 0x04CA, 0x04CA }, // L&       CYRILLIC SMALL LETTER EN WITH TAIL
    Interval{ 0x04CC, 0x04CC }, // L&       CYRILLIC SMALL LETTER KHAKASSIAN CHE
    Interval{ 0x04CE, 0x04CF }, // L&   [2] CYRILLIC SMALL LETTER EM WITH TAIL..CYRILLIC SMALL LETTER PALOCHKA
    Interval{ 0x04D1, 0x04D1 }, // L&       CYRILLIC SMALL LETTER A WITH BREVE
    Interval{ 0x04D3, 0x04D3 }, // L&       CYRILLIC SMALL LETTER A WITH DIAERESIS
    Interval{ 0x04D5, 0x04D5 }, // L&       CYRILLIC SMALL LIGATURE A IE
    Interval{ 0x04D7, 0x04D7 }, // L&       CYRILLIC SMALL LETTER IE WITH BREVE
    Interval{ 0x04D9, 0x04D9 }, // L&       CYRILLIC SMALL LETTER SCHWA
    Interval{ 0x04DB, 0x04DB }, // L&       CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
    Interval{ 0x04DD, 0x04DD }, // L&       CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
    Interval{ 0x04DF, 0x04DF }, // L&       CYRILLIC SMALL LETTER ZE WITH DIAERESIS
    Interval{ 0x04E1, 0x04E1 }, // L&       CYRILLIC SMALL LETTER ABKHASIAN DZE
    Interval{ 0x04E3, 0x04E3 }, // L&       CYRILLIC SMALL LETTER I WITH MACRON
    Interval{ 0x04E5, 0x04E5 }, // L&       CYRILLIC SMALL LETTER I WITH DIAERESIS
    Interval{ 0x04E7, 0x04E7 }, // L&       CYRILLIC SMALL LETTER O WITH DIAERESIS
    Interval{ 0x04E9, 0x04E9 }, // L&       CYRILLIC SMALL LETTER BARRED O
    Interval{ 0x04EB, 0x04EB }, // L&       CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
    Interval{ 0x04ED, 0x04ED }, // L&       CYRILLIC SMALL LETTER E WITH DIAERESIS
    Interval{ 0x04EF, 0x04EF }, // L&       CYRILLIC SMALL LETTER U WITH MACRON
    Interval{ 0x04F1, 0x04F1 }, // L&       CYRILLIC SMALL LETTER U WITH DIAERESIS
    Interval{ 0x04F3, 0x04F3 }, // L&       CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x04F5, 0x04F5 }, // L&       CYRILLIC SMALL LETTER CHE WITH DIAERESIS
    Interval{ 0x04F7, 0x04F7 }, // L&       CYRILLIC SMALL LETTER GHE WITH DESCENDER
    Interval{ 0x04F9, 0x04F9 }, // L&       CYRILLIC SMALL LETTER YERU WITH DIAERESIS
    Interval{ 0x04FB, 0x04FB }, // L&       CYRILLIC SMALL LETTER GHE WITH STROKE AND HOOK
    Interval{ 0x04FD, 0x04FD }, // L&       CYRILLIC SMALL LETTER HA WITH HOOK
    Interval{ 0x04FF, 0x04FF }, // L&       CYRILLIC SMALL LETTER HA WITH STROKE
    Interval{ 0x0501, 0x0501 }, // L&       CYRILLIC SMALL LETTER KOMI DE
    Interval{ 0x0503, 0x0503 }, // L&       CYRILLIC SMALL LETTER KOMI DJE
    Interval{ 0x0505, 0x0505 }, // L&       CYRILLIC SMALL LETTER KOMI ZJE
    Interval{ 0x0507, 0x0507 }, // L&       CYRILLIC SMALL LETTER KOMI DZJE
    Interval{ 0x0509, 0x0509 }, // L&       CYRILLIC SMALL LETTER KOMI LJE
    Interval{ 0x050B, 0x050B }, // L&       CYRILLIC SMALL LETTER KOMI NJE
    Interval{ 0x050D, 0x050D }, // L&       CYRILLIC SMALL LETTER KOMI SJE
    Interval{ 0x050F, 0x050F }, // L&       CYRILLIC SMALL LETTER KOMI TJE
    Interval{ 0x0511, 0x0511 }, // L&       CYRILLIC SMALL LETTER REVERSED ZE
    Interval{ 0x0513, 0x0513 }, // L&       CYRILLIC SMALL LETTER EL WITH HOOK
    Interval{ 0x0515, 0x0515 }, // L&       CYRILLIC SMALL LETTER LHA
    Interval{ 0x0517, 0x0517 }, // L&       CYRILLIC SMALL LETTER RHA
    Interval{ 0x0519, 0x0519 }, // L&       CYRILLIC SMALL LETTER YAE
    Interval{ 0x051B, 0x051B }, // L&       CYRILLIC SMALL LETTER QA
    Interval{ 0x051D, 0x051D }, // L&       CYRILLIC SMALL LETTER WE
    Interval{ 0x051F, 0x051F }, // L&       CYRILLIC SMALL LETTER ALEUT KA
    Interval{ 0x0521, 0x0521 }, // L&       CYRILLIC SMALL LETTER EL WITH MIDDLE HOOK
    Interval{ 0x0523, 0x0523 }, // L&       CYRILLIC SMALL LETTER EN WITH MIDDLE HOOK
    Interval{ 0x0525, 0x0525 }, // L&       CYRILLIC SMALL LETTER PE WITH DESCENDER
    Interval{ 0x0527, 0x0527 }, // L&       CYRILLIC SMALL LETTER SHHA WITH DESCENDER
    Interval{ 0x0529, 0x0529 }, // L&       CYRILLIC SMALL LETTER EN WITH LEFT HOOK
    Interval{ 0x052B, 0x052B }, // L&       CYRILLIC SMALL LETTER DZZHE
    Interval{ 0x052D, 0x052D }, // L&       CYRILLIC SMALL LETTER DCHE
    Interval{ 0x052F, 0x052F }, // L&       CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0561, 0x0587 }, // L&  [39] ARMENIAN SMALL LETTER AYB..ARMENIAN SMALL LIGATURE ECH YIWN
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1D79, 0x1D79 }, // L&       LATIN SMALL LETTER INSULAR G
    Interval{ 0x1D7D, 0x1D7D }, // L&       LATIN SMALL LETTER P WITH STROKE
    Interval{ 0x1D8E, 0x1D8E }, // L&       LATIN SMALL LETTER Z WITH PALATAL HOOK
    Interval{ 0x1E01, 0x1E01 }, // L&       LATIN SMALL LETTER A WITH RING BELOW
    Interval{ 0x1E03, 0x1E03 }, // L&       LATIN SMALL LETTER B WITH DOT ABOVE
    Interval{ 0x1E05, 0x1E05 }, // L&       LATIN SMALL LETTER B WITH DOT BELOW
    Interval{ 0x1E07, 0x1E07 }, // L&       LATIN SMALL LETTER B WITH LINE BELOW
    Interval{ 0x1E09, 0x1E09 }, // L&       LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
    Interval{ 0x1E0B, 0x1E0B }, // L&       LATIN SMALL LETTER D WITH DOT ABOVE
    Interval{ 0x1E0D, 0x1E0D }, // L&       LATIN SMALL LETTER D WITH DOT BELOW
    Interval{ 0x1E0F, 0x1E0F }, // L&       LATIN SMALL LETTER D WITH LINE BELOW
    Interval{ 0x1E11, 0x1E11 }, // L&       LATIN SMALL LETTER D WITH CEDILLA
    Interval{ 0x1E13, 0x1E13 }, // L&       LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
    Interval{ 0x1E15, 0x1E15 }, // L&       LATIN SMALL LETTER E WITH MACRON AND GRAVE
    Interval{ 0x1E17, 0x1E17 }, // L&       LATIN SMALL LETTER E WITH MACRON AND ACUTE
    Interval{ 0x1E19, 0x1E19 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
    Interval{ 0x1E1B, 0x1E1B }, // L&       LATIN SMALL LETTER E WITH TILDE BELOW
    Interval{ 0x1E1D, 0x1E1D }, // L&       LATIN SMALL LETTER E WITH CEDILLA AND BREVE
    Interval{ 0x1E1F, 0x1E1F }, // L&       LATIN SMALL LETTER F WITH DOT ABOVE
    Interval{ 0x1E21, 0x1E21 }, // L&       LATIN SMALL LETTER G WITH MACRON
    Interval{ 0x1E23, 0x1E23 }, // L&       LATIN SMALL LETTER H WITH DOT ABOVE
    Interval{ 0x1E25, 0x1E25 }, // L&       LATIN SMALL LETTER H WITH DOT BELOW
    Interval{ 0x1E27, 0x1E27 }, // L&       LATIN SMALL LETTER H WITH DIAERESIS
    Interval{ 0x1E29, 0x1E29 }, // L&       LATIN SMALL LETTER H WITH CEDILLA
    Interval{ 0x1E2B, 0x1E2B }, // L&       LATIN SMALL LETTER H WITH BREVE BELOW
    Interval{ 0x1E2D, 0x1E2D }, // L&       LATIN SMALL LETTER I WITH TILDE BELOW
    Interval{ 0x1E2F, 0x1E2F }, // L&       LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
    Interval{ 0x1E31, 0x1E31 }, // L&       LATIN SMALL LETTER K WITH ACUTE
    Interval{ 0x1E33, 0x1E33 }, // L&       LATIN SMALL LETTER K WITH DOT BELOW
    Interval{ 0x1E35, 0x1E35 }, // L&       LATIN SMALL LETTER K WITH LINE BELOW
    Interval{ 0x1E37, 0x1E37 }, // L&       LATIN SMALL LETTER L WITH DOT BELOW
    Interval{ 0x1E39, 0x1E39 }, // L&       LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
    Interval{ 0x1E3B, 0x1E3B }, // L&       LATIN SMALL LETTER L WITH LINE BELOW
    Interval{ 0x1E3D, 0x1E3D }, // L&       LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
    Interval{ 0x1E3F, 0x1E3F }, // L&       LATIN SMALL LETTER M WITH ACUTE
    Interval{ 0x1E41, 0x1E41 }, // L&       LATIN SMALL LETTER M WITH DOT ABOVE
    Interval{ 0x1E43, 0x1E43 }, // L&       LATIN SMALL LETTER M WITH DOT BELOW
    Interval{ 0x1E45, 0x1E45 }, // L&       LATIN SMALL LETTER N WITH DOT ABOVE
    Interval{ 0x1E47, 0x1E47 }, // L&       LATIN SMALL LETTER N WITH DOT BELOW
    Interval{ 0x1E49, 0x1E49 }, // L&       LATIN SMALL LETTER N WITH LINE BELOW
    Interval{ 0x1E4B, 0x1E4B }, // L&       LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
    Interval{ 0x1E4D, 0x1E4D }, // L&       LATIN SMALL LETTER O WITH TILDE AND ACUTE
    Interval{ 0x1E4F, 0x1E4F }, // L&       LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
    Interval{ 0x1E51, 0x1E51 }, // L&       LATIN SMALL LETTER O WITH MACRON AND GRAVE
    Interval{ 0x1E53, 0x1E53 }, // L&       LATIN SMALL LETTER O WITH MACRON AND ACUTE
    Interval{ 0x1E55, 0x1E55 }, // L&       LATIN SMALL LETTER P WITH ACUTE
    Interval{ 0x1E57, 0x1E57 }, // L&       LATIN SMALL LETTER P WITH DOT ABOVE
    Interval{ 0x1E59, 0x1E59 }, // L&       LATIN SMALL LETTER R WITH DOT ABOVE
    Interval{ 0x1E5B, 0x1E5B }, // L&       LATIN SMALL LETTER R WITH DOT BELOW
    Interval{ 0x1E5D, 0x1E5D }, // L&       LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
    Interval{ 0x1E5F, 0x1E5F }, // L&       LATIN SMALL LETTER R WITH LINE BELOW
    Interval{ 0x1E61, 0x1E61 }, // L&       LATIN SMALL LETTER S WITH DOT ABOVE
    Interval{ 0x1E63, 0x1E63 }, // L&       LATIN SMALL LETTER S WITH DOT BELOW
    Interval{ 0x1E65, 0x1E65 }, // L&       LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
    Interval{ 0x1E67, 0x1E67 }, // L&       LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
    Interval{ 0x1E69, 0x1E69 }, // L&       LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
    Interval{ 0x1E6B, 0x1E6B }, // L&       LATIN SMALL LETTER T WITH DOT ABOVE
    Interval{ 0x1E6D, 0x1E6D }, // L&       LATIN SMALL LETTER T WITH DOT BELOW
    Interval{ 0x1E6F, 0x1E6F }, // L&       LATIN SMALL LETTER T WITH LINE BELOW
    Interval{ 0x1E71, 0x1E71 }, // L&       LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
    Interval{ 0x1E73, 0x1E73 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS BELOW
    Interval{ 0x1E75, 0x1E75 }, // L&       LATIN SMALL LETTER U WITH TILDE BELOW
    Interval{ 0x1E77, 0x1E77 }, // L&       LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
    Interval{ 0x1E79, 0x1E79 }, // L&       LATIN SMALL LETTER U WITH TILDE AND ACUTE
    Interval{ 0x1E7B, 0x1E7B }, // L&       LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
    Interval{ 0x1E7D, 0x1E7D }, // L&       LATIN SMALL LETTER V WITH TILDE
    Interval{ 0x1E7F, 0x1E7F }, // L&       LATIN SMALL LETTER V WITH DOT BELOW
    Interval{ 0x1E81, 0x1E81 }, // L&       LATIN SMALL LETTER W WITH GRAVE
    Interval{ 0x1E83, 0x1E83 }, // L&       LATIN SMALL LETTER W WITH ACUTE
    Interval{ 0x1E85, 0x1E85 }, // L&       LATIN SMALL LETTER W WITH DIAERESIS
    Interval{ 0x1E87, 0x1E87 }, // L&       LATIN SMALL LETTER W WITH DOT ABOVE
    Interval{ 0x1E89, 0x1E89 }, // L&       LATIN SMALL LETTER W WITH DOT BELOW
    Interval{ 0x1E8B, 0x1E8B }, // L&       LATIN SMALL LETTER X WITH DOT ABOVE
    Interval{ 0x1E8D, 0x1E8D }, // L&       LATIN SMALL LETTER X WITH DIAERESIS
    Interval{ 0x1E8F, 0x1E8F }, // L&       LATIN SMALL LETTER Y WITH DOT ABOVE
    Interval{ 0x1E91, 0x1E91 }, // L&       LATIN SMALL LETTER Z WITH CIRCUMFLEX
    Interval{ 0x1E93, 0x1E93 }, // L&       LATIN SMALL LETTER Z WITH DOT BELOW
    Interval{ 0x1E95, 0x1E9B }, // L&   [7] LATIN SMALL LETTER Z WITH LINE BELOW..LATIN SMALL LETTER LONG S WITH DOT ABOVE
    Interval{ 0x1EA1, 0x1EA1 }, // L&       LATIN SMALL LETTER A WITH DOT BELOW
    Interval{ 0x1EA3, 0x1EA3 }, // L&       LATIN SMALL LETTER A WITH HOOK ABOVE
    Interval{ 0x1EA5, 0x1EA5 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EA7, 0x1EA7 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EA9, 0x1EA9 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EAB, 0x1EAB }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EAD, 0x1EAD }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EAF, 0x1EAF }, // L&       LATIN SMALL LETTER A WITH BREVE AND ACUTE
    Interval{ 0x1EB1, 0x1EB1 }, // L&       LATIN SMALL LETTER A WITH BREVE AND GRAVE
    Interval{ 0x1EB3, 0x1EB3 }, // L&       LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
    Interval{ 0x1EB5, 0x1EB5 }, // L&       LATIN SMALL LETTER A WITH BREVE AND TILDE
    Interval{ 0x1EB7, 0x1EB7 }, // L&       LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
    Interval{ 0x1EB9, 0x1EB9 }, // L&       LATIN SMALL LETTER E WITH DOT BELOW
    Interval{ 0x1EBB, 0x1EBB }, // L&       LATIN SMALL LETTER E WITH HOOK ABOVE
    Interval{ 0x1EBD, 0x1EBD }, // L&       LATIN SMALL LETTER E WITH TILDE
    Interval{ 0x1EBF, 0x1EBF }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EC1, 0x1EC1 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EC3, 0x1EC3 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EC5, 0x1EC5 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EC7, 0x1EC7 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EC9, 0x1EC9 }, // L&       LATIN SMALL LETTER I WITH HOOK ABOVE
    Interval{ 0x1ECB, 0x1ECB }, // L&       LATIN SMALL LETTER I WITH DOT BELOW
    Interval{ 0x1ECD, 0x1ECD }, // L&       LATIN SMALL LETTER O WITH DOT BELOW
    Interval{ 0x1ECF, 0x1ECF }, // L&       LATIN SMALL LETTER O WITH HOOK ABOVE
    Interval{ 0x1ED1, 0x1ED1 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1ED3, 0x1ED3 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1ED5, 0x1ED5 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1ED7, 0x1ED7 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1ED9, 0x1ED9 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EDB, 0x1EDB }, // L&       LATIN SMALL LETTER O WITH HORN AND ACUTE
    Interval{ 0x1EDD, 0x1EDD }, // L&       LATIN SMALL LETTER O WITH HORN AND GRAVE
    Interval{ 0x1EDF, 0x1EDF }, // L&       LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
    Interval{ 0x1EE1, 0x1EE1 }, // L&       LATIN SMALL LETTER O WITH HORN AND TILDE
    Interval{ 0x1EE3, 0x1EE3 }, // L&       LATIN SMALL LETTER O WITH HORN AND DOT BELOW
    Interval{ 0x1EE5, 0x1EE5 }, // L&       LATIN SMALL LETTER U WITH DOT BELOW
    Interval{ 0x1EE7, 0x1EE7 }, // L&       LATIN SMALL LETTER U WITH HOOK ABOVE
    Interval{ 0x1EE9, 0x1EE9 }, // L&       LATIN SMALL LETTER U WITH HORN AND ACUTE
    Interval{ 0x1EEB, 0x1EEB }, // L&       LATIN SMALL LETTER U WITH HORN AND GRAVE
    Interval{ 0x1EED, 0x1EED }, // L&       LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
    Interval{ 0x1EEF, 0x1EEF }, // L&       LATIN SMALL LETTER U WITH HORN AND TILDE
    Interval{ 0x1EF1, 0x1EF1 }, // L&       LATIN SMALL LETTER U WITH HORN AND DOT BELOW
    Interval{ 0x1EF3, 0x1EF3 }, // L&       LATIN SMALL LETTER Y WITH GRAVE
    Interval{ 0x1EF5, 0x1EF5 }, // L&       LATIN SMALL LETTER Y WITH DOT BELOW
    Interval{ 0x1EF7, 0x1EF7 }, // L&       LATIN SMALL LETTER Y WITH HOOK ABOVE
    Interval{ 0x1EF9, 0x1EF9 }, // L&       LATIN SMALL LETTER Y WITH TILDE
    Interval{ 0x1EFB, 0x1EFB }, // L&       LATIN SMALL LETTER MIDDLE-WELSH LL
    Interval{ 0x1EFD, 0x1EFD }, // L&       LATIN SMALL LETTER MIDDLE-WELSH V
    Interval{ 0x1EFF, 0x1F07 }, // L&   [9] LATIN SMALL LETTER Y WITH LOOP..GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F10, 0x1F15 }, // L&   [6] GREEK SMALL LETTER EPSILON WITH PSILI..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F27 }, // L&   [8] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F30, 0x1F37 }, // L&   [8] GREEK SMALL LETTER IOTA WITH PSILI..GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F40, 0x1F45 }, // L&   [6] GREEK SMALL LETTER OMICRON WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F60, 0x1F67 }, // L&   [8] GREEK SMALL LETTER OMEGA WITH PSILI..GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F70, 0x1F7D }, // L&  [14] GREEK SMALL LETTER ALPHA WITH VARIA..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1FB4 }, // L&  [53] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FB7 }, // L&   [2] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FBC, 0x1FBC }, // L&       GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FC7 }, // L&   [2] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FCC, 0x1FCC }, // L&       GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FD7 }, // L&   [2] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
    Interval{ 0x1FE0, 0x1FE7 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FF7 }, // L&   [2] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FFC, 0x1FFC }, // L&       GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2170, 0x217F }, // Nl  [16] SMALL ROMAN NUMERAL ONE..SMALL ROMAN NUMERAL ONE THOUSAND
    Interval{ 0x2184, 0x2184 }, // L&       LATIN SMALL LETTER REVERSED C
    Interval{ 0x24D0, 0x24E9 }, // So  [26] CIRCLED LATIN SMALL LETTER A..CIRCLED LATIN SMALL LETTER Z
    Interval{ 0x2C30, 0x2C5F }, // L&  [48] GLAGOLITIC SMALL LETTER AZU..GLAGOLITIC SMALL LETTER CAUDATE CHRIVI
    Interval{ 0x2C61, 0x2C61 }, // L&       LATIN SMALL LETTER L WITH DOUBLE BAR
    Interval{ 0x2C65, 0x2C66 }, // L&   [2] LATIN SMALL LETTER A WITH STROKE..LATIN SMALL LETTER T WITH DIAGONAL STROKE
    Interval{ 0x2C68, 0x2C68 }, // L&       LATIN SMALL LETTER H WITH DESCENDER
    Interval{ 0x2C6A, 0x2C6A }, // L&       LATIN SMALL LETTER K WITH DESCENDER
    Interval{ 0x2C6C, 0x2C6C }, // L&       LATIN SMALL LETTER Z WITH DESCENDER
    Interval{ 0x2C73, 0x2C73 }, // L&       LATIN SMALL LETTER W WITH HOOK
    Interval{ 0x2C76, 0x2C76 }, // L&       LATIN SMALL LETTER HALF H
    Interval{ 0x2C81, 0x2C81 }, // L&       COPTIC SMALL LETTER ALFA
    Interval{ 0x2C83, 0x2C83 }, // L&       COPTIC SMALL LETTER VIDA
    Interval{ 0x2C85, 0x2C85 }, // L&       COPTIC SMALL LETTER GAMMA
    Interval{ 0x2C87, 0x2C87 }, // L&       COPTIC SMALL LETTER DALDA
    Interval{ 0x2C89, 0x2C89 }, // L&       COPTIC SMALL LETTER EIE
    Interval{ 0x2C8B, 0x2C8B }, // L&       COPTIC SMALL LETTER SOU
    Interval{ 0x2C8D, 0x2C8D }, // L&       COPTIC SMALL LETTER ZATA
    Interval{ 0x2C8F, 0x2C8F }, // L&       COPTIC SMALL LETTER HATE
    Interval{ 0x2C91, 0x2C91 }, // L&       COPTIC SMALL LETTER THETHE
    Interval{ 0x2C93, 0x2C93 }, // L&       COPTIC SMALL LETTER IAUDA
    Interval{ 0x2C95, 0x2C95 }, // L&       COPTIC SMALL LETTER KAPA
    Interval{ 0x2C97, 0x2C97 }, // L&       COPTIC SMALL LETTER LAULA
    Interval{ 0x2C99, 0x2C99 }, // L&       COPTIC SMALL LETTER MI
    Interval{ 0x2C9B, 0x2C9B }, // L&       COPTIC SMALL LETTER NI
    Interval{ 0x2C9D, 0x2C9D }, // L&       COPTIC SMALL LETTER KSI
    Interval{ 0x2C9F, 0x2C9F }, // L&       COPTIC SMALL LETTER O
    Interval{ 0x2CA1, 0x2CA1 }, // L&       COPTIC SMALL LETTER PI
    Interval{ 0x2CA3, 0x2CA3 }, // L&       COPTIC SMALL LETTER RO
    Interval{ 0x2CA5, 0x2CA5 }, // L&       COPTIC SMALL LETTER SIMA
    Interval{ 0x2CA7, 0x2CA7 }, // L&       COPTIC SMALL LETTER TAU
    Interval{ 0x2CA9, 0x2CA9 }, // L&       COPTIC SMALL LETTER UA
    Interval{ 0x2CAB, 0x2CAB }, // L&       COPTIC SMALL LETTER FI
    Interval{ 0x2CAD, 0x2CAD }, // L&       COPTIC SMALL LETTER KHI
    Interval{ 0x2CAF, 0x2CAF }, // L&       COPTIC SMALL LETTER PSI
    Interval{ 0x2CB1, 0x2CB1 }, // L&       COPTIC SMALL LETTER OOU
    Interval{ 0x2CB3, 0x2CB3 }, // L&       COPTIC SMALL LETTER DIALECT-P ALEF
    Interval{ 0x2CB5, 0x2CB5 }, // L&       COPTIC SMALL LETTER OLD COPTIC AIN
    Interval{ 0x2CB7, 0x2CB7 }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC EIE
    Interval{ 0x2CB9, 0x2CB9 }, // L&       COPTIC SMALL LETTER DIALECT-P KAPA
    Interval{ 0x2CBB, 0x2CBB }, // L&       COPTIC SMALL LETTER DIALECT-P NI
    Interval{ 0x2CBD, 0x2CBD }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC NI
    Interval{ 0x2CBF, 0x2CBF }, // L&       COPTIC SMALL LETTER OLD COPTIC OOU
    Interval{ 0x2CC1, 0x2CC1 }, // L&       COPTIC SMALL LETTER SAMPI
    Interval{ 0x2CC3, 0x2CC3 }, // L&       COPTIC SMALL LETTER CROSSED SHEI
    Interval{ 0x2CC5, 0x2CC5 }, // L&       COPTIC SMALL LETTER OLD COPTIC SHEI
    Interval{ 0x2CC7, 0x2CC7 }, // L&       COPTIC SMALL LETTER OLD COPTIC ESH
    Interval{ 0x2CC9, 0x2CC9 }, // L&       COPTIC SMALL LETTER AKHMIMIC KHEI
    Interval{ 0x2CCB, 0x2CCB }, // L&       COPTIC SMALL LETTER DIALECT-P HORI
    Interval{ 0x2CCD, 0x2CCD }, // L&       COPTIC SMALL LETTER OLD COPTIC HORI
    Interval{ 0x2CCF, 0x2CCF }, // L&       COPTIC SMALL LETTER OLD COPTIC HA
    Interval{ 0x2CD1, 0x2CD1 }, // L&       COPTIC SMALL LETTER L-SHAPED HA
    Interval{ 0x2CD3, 0x2CD3 }, // L&       COPTIC SMALL LETTER OLD COPTIC HEI
    Interval{ 0x2CD5, 0x2CD5 }, // L&       COPTIC SMALL LETTER OLD COPTIC HAT
    Interval{ 0x2CD7, 0x2CD7 }, // L&       COPTIC SMALL LETTER OLD COPTIC GANGIA
    Interval{ 0x2CD9, 0x2CD9 }, // L&       COPTIC SMALL LETTER OLD COPTIC DJA
    Interval{ 0x2CDB, 0x2CDB }, // L&       COPTIC SMALL LETTER OLD COPTIC SHIMA
    Interval{ 0x2CDD, 0x2CDD }, // L&       COPTIC SMALL LETTER OLD NUBIAN SHIMA
    Interval{ 0x2CDF, 0x2CDF }, // L&       COPTIC SMALL LETTER OLD NUBIAN NGI
    Interval{ 0x2CE1, 0x2CE1 }, // L&       COPTIC SMALL LETTER OLD NUBIAN NYI
    Interval{ 0x2CE3, 0x2CE3 }, // L&       COPTIC SMALL LETTER OLD NUBIAN WAU
    Interval{ 0x2CEC, 0x2CEC }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC SHEI
    Interval{ 0x2CEE, 0x2CEE }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF3, 0x2CF3 }, // L&       COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0xA641, 0xA641 }, // L&       CYRILLIC SMALL LETTER ZEMLYA
    Interval{ 0xA643, 0xA643 }, // L&       CYRILLIC SMALL LETTER DZELO
    Interval{ 0xA645, 0xA645 }, // L&       CYRILLIC SMALL LETTER REVERSED DZE
    Interval{ 0xA647, 0xA647 }, // L&       CYRILLIC SMALL LETTER IOTA
    Interval{ 0xA649, 0xA649 }, // L&       CYRILLIC SMALL LETTER DJERV
    Interval{ 0xA64B, 0xA64B }, // L&       CYRILLIC SMALL LETTER MONOGRAPH UK
    Interval{ 0xA64D, 0xA64D }, // L&       CYRILLIC SMALL LETTER BROAD OMEGA
    Interval{ 0xA64F, 0xA64F }, // L&       CYRILLIC SMALL LETTER NEUTRAL YER
    Interval{ 0xA651, 0xA651 }, // L&       CYRILLIC SMALL LETTER YERU WITH BACK YER
    Interval{ 0xA653, 0xA653 }, // L&       CYRILLIC SMALL LETTER IOTIFIED YAT
    Interval{ 0xA655, 0xA655 }, // L&       CYRILLIC SMALL LETTER REVERSED YU
    Interval{ 0xA657, 0xA657 }, // L&       CYRILLIC SMALL LETTER IOTIFIED A
    Interval{ 0xA659, 0xA659 }, // L&       CYRILLIC SMALL LETTER CLOSED LITTLE YUS
    Interval{ 0xA65B, 0xA65B }, // L&       CYRILLIC SMALL LETTER BLENDED YUS
    Interval{ 0xA65D, 0xA65D }, // L&       CYRILLIC SMALL LETTER IOTIFIED CLOSED LITTLE YUS
    Interval{ 0xA65F, 0xA65F }, // L&       CYRILLIC SMALL LETTER YN
    Interval{ 0xA661, 0xA661 }, // L&       CYRILLIC SMALL LETTER REVERSED TSE
    Interval{ 0xA663, 0xA663 }, // L&       CYRILLIC SMALL LETTER SOFT DE
    Interval{ 0xA665, 0xA665 }, // L&       CYRILLIC SMALL LETTER SOFT EL
    Interval{ 0xA667, 0xA667 }, // L&       CYRILLIC SMALL LETTER SOFT EM
    Interval{ 0xA669, 0xA669 }, // L&       CYRILLIC SMALL LETTER MONOCULAR O
    Interval{ 0xA66B, 0xA66B }, // L&       CYRILLIC SMALL LETTER BINOCULAR O
    Interval{ 0xA66D, 0xA66D }, // L&       CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA681, 0xA681 }, // L&       CYRILLIC SMALL LETTER DWE
    Interval{ 0xA683, 0xA683 }, // L&       CYRILLIC SMALL LETTER DZWE
    Interval{ 0xA685, 0xA685 }, // L&       CYRILLIC SMALL LETTER ZHWE
    Interval{ 0xA687, 0xA687 }, // L&       CYRILLIC SMALL LETTER CCHE
    Interval{ 0xA689, 0xA689 }, // L&       CYRILLIC SMALL LETTER DZZE
    Interval{ 0xA68B, 0xA68B }, // L&       CYRILLIC SMALL LETTER TE WITH MIDDLE HOOK
    Interval{ 0xA68D, 0xA68D }, // L&       CYRILLIC SMALL LETTER TWE
    Interval{ 0xA68F, 0xA68F }, // L&       CYRILLIC SMALL LETTER TSWE
    Interval{ 0xA691, 0xA691 }, // L&       CYRILLIC SMALL LETTER TSSE
    Interval{ 0xA693, 0xA693 }, // L&       CYRILLIC SMALL LETTER TCHE
    Interval{ 0xA695, 0xA695 }, // L&       CYRILLIC SMALL LETTER HWE
    Interval{ 0xA697, 0xA697 }, // L&       CYRILLIC SMALL LETTER SHWE
    Interval{ 0xA699, 0xA699 }, // L&       CYRILLIC SMALL LETTER DOUBLE O
    Interval{ 0xA69B, 0xA69B }, // L&       CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA723, 0xA723 }, // L&       LATIN SMALL LETTER EGYPTOLOGICAL ALEF
    Interval{ 0xA725, 0xA725 }, // L&       LATIN SMALL LETTER EGYPTOLOGICAL AIN
    Interval{ 0xA727, 0xA727 }, // L&       LATIN SMALL LETTER HENG
    Interval{ 0xA729, 0xA729 }, // L&       LATIN SMALL LETTER TZ
    Interval{ 0xA72B, 0xA72B }, // L&       LATIN SMALL LETTER TRESILLO
    Interval{ 0xA72D, 0xA72D }, // L&       LATIN SMALL LETTER CUATRILLO
    Interval{ 0xA72F, 0xA72F }, // L&       LATIN SMALL LETTER CUATRILLO WITH COMMA
    Interval{ 0xA733, 0xA733 }, // L&       LATIN SMALL LETTER AA
    Interval{ 0xA735, 0xA735 }, // L&       LATIN SMALL LETTER AO
    Interval{ 0xA737, 0xA737 }, // L&       LATIN SMALL LETTER AU
    Interval{ 0xA739, 0xA739 }, // L&       LATIN SMALL LETTER AV
    Interval{ 0xA73B, 0xA73B }, // L&       LATIN SMALL LETTER AV WITH HORIZONTAL BAR
    Interval{ 0xA73D, 0xA73D }, // L&       LATIN SMALL LETTER AY
    Interval{ 0xA73F, 0xA73F }, // L&       LATIN SMALL LETTER REVERSED C WITH DOT
    Interval{ 0xA741, 0xA741 }, // L&       LATIN SMALL LETTER K WITH STROKE
    Interval{ 0xA743, 0xA743 }, // L&       LATIN SMALL LETTER K WITH DIAGONAL STROKE
    Interval{ 0xA745, 0xA745 }, // L&       LATIN SMALL LETTER K WITH STROKE AND DIAGONAL STROKE
    Interval{ 0xA747, 0xA747 }, // L&       LATIN SMALL LETTER BROKEN L
    Interval{ 0xA749, 0xA749 }, // L&       LATIN SMALL LETTER L WITH HIGH STROKE
    Interval{ 0xA74B, 0xA74B }, // L&       LATIN SMALL LETTER O WITH LONG STROKE OVERLAY
    Interval{ 0xA74D, 0xA74D }, // L&       LATIN SMALL LETTER O WITH LOOP
    Interval{ 0xA74F, 0xA74F }, // L&       LATIN SMALL LETTER OO
    Interval{ 0xA751, 0xA751 }, // L&       LATIN SMALL LETTER P WITH STROKE THROUGH DESCENDER
    Interval{ 0xA753, 0xA753 }, // L&       LATIN SMALL LETTER P WITH FLOURISH
    Interval{ 0xA755, 0xA755 }, // L&       LATIN SMALL LETTER P WITH SQUIRREL TAIL
    Interval{ 0xA757, 0xA757 }, // L&       LATIN SMALL LETTER Q WITH STROKE THROUGH DESCENDER
    Interval{ 0xA759, 0xA759 }, // L&       LATIN SMALL LETTER Q WITH DIAGONAL STROKE
    Interval{ 0xA75B, 0xA75B }, // L&       LATIN SMALL LETTER R ROTUNDA
    Interval{ 0xA75D, 0xA75D }, // L&       LATIN SMALL LETTER RUM ROTUNDA
    Interval{ 0xA75F, 0xA75F }, // L&       LATIN SMALL LETTER V WITH DIAGONAL STROKE
    Interval{ 0xA761, 0xA761 }, // L&       LATIN SMALL LETTER VY
    Interval{ 0xA763, 0xA763 }, // L&       LATIN SMALL LETTER VISIGOTHIC Z
    Interval{ 0xA765, 0xA765 }, // L&       LATIN SMALL LETTER THORN WITH STROKE
    Interval{ 0xA767, 0xA767 }, // L&       LATIN SMALL LETTER THORN WITH STROKE THROUGH DESCENDER
    Interval{ 0xA769, 0xA769 }, // L&       LATIN SMALL LETTER VEND
    Interval{ 0xA76B, 0xA76B }, // L&       LATIN SMALL LETTER ET
    Interval{ 0xA76D, 0xA76D }, // L&       LATIN SMALL LETTER IS
    Interval{ 0xA76F, 0xA76F }, // L&       LATIN SMALL LETTER CON
    Interval{ 0xA77A, 0xA77A }, // L&       LATIN SMALL LETTER INSULAR D
    Interval{ 0xA77C, 0xA77C }, // L&       LATIN SMALL LETTER INSULAR F
    Interval{ 0xA77F, 0xA77F }, // L&       LATIN SMALL LETTER TURNED INSULAR G
    Interval{ 0xA781, 0xA781 }, // L&       LATIN SMALL LETTER TURNED L
    Interval{ 0xA783, 0xA783 }, // L&       LATIN SMALL LETTER INSULAR R
    Interval{ 0xA785, 0xA785 }, // L&       LATIN SMALL LETTER INSULAR S
    Interval{ 0xA787, 0xA787 }, // L&       LATIN SMALL LETTER INSULAR T
    Interval{ 0xA78C, 0xA78C }, // L&       LATIN SMALL LETTER SALTILLO
    Interval{ 0xA791, 0xA791 }, // L&       LATIN SMALL LETTER N WITH DESCENDER
    Interval{ 0xA793, 0xA794 }, // L&   [2] LATIN SMALL LETTER C WITH BAR..LATIN SMALL LETTER C WITH PALATAL HOOK
    Interval{ 0xA797, 0xA797 }, // L&       LATIN SMALL LETTER B WITH FLOURISH
    Interval{ 0xA799, 0xA799 }, // L&       LATIN SMALL LETTER F WITH STROKE
    Interval{ 0xA79B, 0xA79B }, // L&       LATIN SMALL LETTER VOLAPUK AE
    Interval{ 0xA79D, 0xA79D }, // L&       LATIN SMALL LETTER VOLAPUK OE
    Interval{ 0xA79F, 0xA79F }, // L&       LATIN SMALL LETTER VOLAPUK UE
    Interval{ 0xA7A1, 0xA7A1 }, // L&       LATIN SMALL LETTER G WITH OBLIQUE STROKE
    Interval{ 0xA7A3, 0xA7A3 }, // L&       LATIN SMALL LETTER K WITH OBLIQUE STROKE
    Interval{ 0xA7A5, 0xA7A5 }, // L&       LATIN SMALL LETTER N WITH OBLIQUE STROKE
    Interval{ 0xA7A7, 0xA7A7 }, // L&       LATIN SMALL LETTER R WITH OBLIQUE STROKE
    Interval{ 0xA7A9, 0xA7A9 }, // L&       LATIN SMALL LETTER S WITH OBLIQUE STROKE
    Interval{ 0xA7B5, 0xA7B5 }, // L&       LATIN SMALL LETTER BETA
    Interval{ 0xA7B7, 0xA7B7 }, // L&       LATIN SMALL LETTER OMEGA
    Interval{ 0xA7B9, 0xA7B9 }, // L&       LATIN SMALL LETTER U WITH STROKE
    Interval{ 0xA7BB, 0xA7BB }, // L&       LATIN SMALL LETTER GLOTTAL A
    Interval{ 0xA7BD, 0xA7BD }, // L&       LATIN SMALL LETTER GLOTTAL I
    Interval{ 0xA7BF, 0xA7BF }, // L&       LATIN SMALL LETTER GLOTTAL U
    Interval{ 0xA7C1, 0xA7C1 }, // L&       LATIN SMALL LETTER OLD POLISH O
    Interval{ 0xA7C3, 0xA7C3 }, // L&       LATIN SMALL LETTER ANGLICANA W
    Interval{ 0xA7C8, 0xA7C8 }, // L&       LATIN SMALL LETTER D WITH SHORT STROKE OVERLAY
    Interval{ 0xA7CA, 0xA7CA }, // L&       LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D1, 0xA7D1 }, // L&       LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D7, 0xA7D7 }, // L&       LATIN SMALL LETTER MIDDLE SCOTS S
    Interval{ 0xA7D9, 0xA7D9 }, // L&       LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F6, 0xA7F6 }, // L&       LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xAB53, 0xAB53 }, // L&       LATIN SMALL LETTER CHI
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0x10428, 0x1044F }, // L&  [40] DESERET SMALL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x118C0, 0x118DF }, // L&  [32] WARANG CITI SMALL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x16E60, 0x16E7F }, // L&  [32] MEDEFAIDRIN SMALL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x1E922, 0x1E943 }, // L&  [34] ADLAM SMALL LETTER ALIF..ADLAM SMALL LETTER SHA
}; // }}}
auto static const Default_Ignorable_Code_Point = std::array<Interval, 27>{ // {{{
    Interval{ 0x00AD, 0x00AD }, // Cf       SOFT HYPHEN
    Interval{ 0x034F, 0x034F }, // Mn       COMBINING GRAPHEME JOINER
    Interval{ 0x061C, 0x061C }, // Cf       ARABIC LETTER MARK
    Interval{ 0x115F, 0x1160 }, // Lo   [2] HANGUL CHOSEONG FILLER..HANGUL JUNGSEONG FILLER
    Interval{ 0x17B4, 0x17B5 }, // Mn   [2] KHMER VOWEL INHERENT AQ..KHMER VOWEL INHERENT AA
    Interval{ 0x180B, 0x180D }, // Mn   [3] MONGOLIAN FREE VARIATION SELECTOR ONE..MONGOLIAN FREE VARIATION SELECTOR THREE
    Interval{ 0x180E, 0x180E }, // Cf       MONGOLIAN VOWEL SEPARATOR
    Interval{ 0x180F, 0x180F }, // Mn       MONGOLIAN FREE VARIATION SELECTOR FOUR
    Interval{ 0x200B, 0x200F }, // Cf   [5] ZERO WIDTH SPACE..RIGHT-TO-LEFT MARK
    Interval{ 0x202A, 0x202E }, // Cf   [5] LEFT-TO-RIGHT EMBEDDING..RIGHT-TO-LEFT OVERRIDE
    Interval{ 0x2060, 0x2064 }, // Cf   [5] WORD JOINER..INVISIBLE PLUS
    Interval{ 0x2065, 0x2065 }, // Cn       <reserved-2065>
    Interval{ 0x2066, 0x206F }, // Cf  [10] LEFT-TO-RIGHT ISOLATE..NOMINAL DIGIT SHAPES
    Interval{ 0x3164, 0x3164 }, // Lo       HANGUL FILLER
    Interval{ 0xFE00, 0xFE0F }, // Mn  [16] VARIATION SELECTOR-1..VARIATION SELECTOR-16
    Interval{ 0xFEFF, 0xFEFF }, // Cf       ZERO WIDTH NO-BREAK SPACE
    Interval{ 0xFFA0, 0xFFA0 }, // Lo       HALFWIDTH HANGUL FILLER
    Interval{ 0xFFF0, 0xFFF8 }, // Cn   [9] <reserved-FFF0>..<reserved-FFF8>
    Interval{ 0x1BCA0, 0x1BCA3 }, // Cf   [4] SHORTHAND FORMAT LETTER OVERLAP..SHORTHAND FORMAT UP STEP
    Interval{ 0x1D173, 0x1D17A }, // Cf   [8] MUSICAL SYMBOL BEGIN BEAM..MUSICAL SYMBOL END PHRASE
    Interval{ 0xE0000, 0xE0000 }, // Cn       <reserved-E0000>
    Interval{ 0xE0001, 0xE0001 }, // Cf       LANGUAGE TAG
    Interval{ 0xE0002, 0xE001F }, // Cn  [30] <reserved-E0002>..<reserved-E001F>
    Interval{ 0xE0020, 0xE007F }, // Cf  [96] TAG SPACE..CANCEL TAG
    Interval{ 0xE0080, 0xE00FF }, // Cn [128] <reserved-E0080>..<reserved-E00FF>
    Interval{ 0xE0100, 0xE01EF }, // Mn [240] VARIATION SELECTOR-17..VARIATION SELECTOR-256
    Interval{ 0xE01F0, 0xE0FFF }, // Cn [3600] <reserved-E01F0>..<reserved-E0FFF>
}; // }}}
auto static const Grapheme_Base = std::array<Interval, 1741>{ // {{{
    Interval{ 0x0020, 0x0020 }, // Zs       SPACE
    Interval{ 0x0021, 0x0023 }, // Po   [3] EXCLAMATION MARK..NUMBER SIGN
    Interval{ 0x0024, 0x0024 }, // Sc       DOLLAR SIGN
    Interval{ 0x0025, 0x0027 }, // Po   [3] PERCENT SIGN..APOSTROPHE
    Interval{ 0x0028, 0x0028 }, // Ps       LEFT PARENTHESIS
    Interval{ 0x0029, 0x0029 }, // Pe       RIGHT PARENTHESIS
    Interval{ 0x002A, 0x002A }, // Po       ASTERISK
    Interval{ 0x002B, 0x002B }, // Sm       PLUS SIGN
    Interval{ 0x002C, 0x002C }, // Po       COMMA
    Interval{ 0x002D, 0x002D }, // Pd       HYPHEN-MINUS
    Interval{ 0x002E, 0x002F }, // Po   [2] FULL STOP..SOLIDUS
    Interval{ 0x0030, 0x0039 }, // Nd  [10] DIGIT ZERO..DIGIT NINE
    Interval{ 0x003A, 0x003B }, // Po   [2] COLON..SEMICOLON
    Interval{ 0x003C, 0x003E }, // Sm   [3] LESS-THAN SIGN..GREATER-THAN SIGN
    Interval{ 0x003F, 0x0040 }, // Po   [2] QUESTION MARK..COMMERCIAL AT
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x005B, 0x005B }, // Ps       LEFT SQUARE BRACKET
    Interval{ 0x005C, 0x005C }, // Po       REVERSE SOLIDUS
    Interval{ 0x005D, 0x005D }, // Pe       RIGHT SQUARE BRACKET
    Interval{ 0x005E, 0x005E }, // Sk       CIRCUMFLEX ACCENT
    Interval{ 0x005F, 0x005F }, // Pc       LOW LINE
    Interval{ 0x0060, 0x0060 }, // Sk       GRAVE ACCENT
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x007B, 0x007B }, // Ps       LEFT CURLY BRACKET
    Interval{ 0x007C, 0x007C }, // Sm       VERTICAL LINE
    Interval{ 0x007D, 0x007D }, // Pe       RIGHT CURLY BRACKET
    Interval{ 0x007E, 0x007E }, // Sm       TILDE
    Interval{ 0x00A0, 0x00A0 }, // Zs       NO-BREAK SPACE
    Interval{ 0x00A1, 0x00A1 }, // Po       INVERTED EXCLAMATION MARK
    Interval{ 0x00A2, 0x00A5 }, // Sc   [4] CENT SIGN..YEN SIGN
    Interval{ 0x00A6, 0x00A6 }, // So       BROKEN BAR
    Interval{ 0x00A7, 0x00A7 }, // Po       SECTION SIGN
    Interval{ 0x00A8, 0x00A8 }, // Sk       DIAERESIS
    Interval{ 0x00A9, 0x00A9 }, // So       COPYRIGHT SIGN
    Interval{ 0x00AA, 0x00AA }, // Lo       FEMININE ORDINAL INDICATOR
    Interval{ 0x00AB, 0x00AB }, // Pi       LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
    Interval{ 0x00AC, 0x00AC }, // Sm       NOT SIGN
    Interval{ 0x00AE, 0x00AE }, // So       REGISTERED SIGN
    Interval{ 0x00AF, 0x00AF }, // Sk       MACRON
    Interval{ 0x00B0, 0x00B0 }, // So       DEGREE SIGN
    Interval{ 0x00B1, 0x00B1 }, // Sm       PLUS-MINUS SIGN
    Interval{ 0x00B2, 0x00B3 }, // No   [2] SUPERSCRIPT TWO..SUPERSCRIPT THREE
    Interval{ 0x00B4, 0x00B4 }, // Sk       ACUTE ACCENT
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00B6, 0x00B7 }, // Po   [2] PILCROW SIGN..MIDDLE DOT
    Interval{ 0x00B8, 0x00B8 }, // Sk       CEDILLA
    Interval{ 0x00B9, 0x00B9 }, // No       SUPERSCRIPT ONE
    Interval{ 0x00BA, 0x00BA }, // Lo       MASCULINE ORDINAL INDICATOR
    Interval{ 0x00BB, 0x00BB }, // Pf       RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
    Interval{ 0x00BC, 0x00BE }, // No   [3] VULGAR FRACTION ONE QUARTER..VULGAR FRACTION THREE QUARTERS
    Interval{ 0x00BF, 0x00BF }, // Po       INVERTED QUESTION MARK
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D7, 0x00D7 }, // Sm       MULTIPLICATION SIGN
    Interval{ 0x00D8, 0x00F6 }, // L&  [31] LATIN CAPITAL LETTER O WITH STROKE..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F7, 0x00F7 }, // Sm       DIVISION SIGN
    Interval{ 0x00F8, 0x01BA }, // L& [195] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER EZH WITH TAIL
    Interval{ 0x01BB, 0x01BB }, // Lo       LATIN LETTER TWO WITH STROKE
    Interval{ 0x01BC, 0x01BF }, // L&   [4] LATIN CAPITAL LETTER TONE FIVE..LATIN LETTER WYNN
    Interval{ 0x01C0, 0x01C3 }, // Lo   [4] LATIN LETTER DENTAL CLICK..LATIN LETTER RETROFLEX CLICK
    Interval{ 0x01C4, 0x0293 }, // L& [208] LATIN CAPITAL LETTER DZ WITH CARON..LATIN SMALL LETTER EZH WITH CURL
    Interval{ 0x0294, 0x0294 }, // Lo       LATIN LETTER GLOTTAL STOP
    Interval{ 0x0295, 0x02AF }, // L&  [27] LATIN LETTER PHARYNGEAL VOICED FRICATIVE..LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    Interval{ 0x02B0, 0x02C1 }, // Lm  [18] MODIFIER LETTER SMALL H..MODIFIER LETTER REVERSED GLOTTAL STOP
    Interval{ 0x02C2, 0x02C5 }, // Sk   [4] MODIFIER LETTER LEFT ARROWHEAD..MODIFIER LETTER DOWN ARROWHEAD
    Interval{ 0x02C6, 0x02D1 }, // Lm  [12] MODIFIER LETTER CIRCUMFLEX ACCENT..MODIFIER LETTER HALF TRIANGULAR COLON
    Interval{ 0x02D2, 0x02DF }, // Sk  [14] MODIFIER LETTER CENTRED RIGHT HALF RING..MODIFIER LETTER CROSS ACCENT
    Interval{ 0x02E0, 0x02E4 }, // Lm   [5] MODIFIER LETTER SMALL GAMMA..MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    Interval{ 0x02E5, 0x02EB }, // Sk   [7] MODIFIER LETTER EXTRA-HIGH TONE BAR..MODIFIER LETTER YANG DEPARTING TONE MARK
    Interval{ 0x02EC, 0x02EC }, // Lm       MODIFIER LETTER VOICING
    Interval{ 0x02ED, 0x02ED }, // Sk       MODIFIER LETTER UNASPIRATED
    Interval{ 0x02EE, 0x02EE }, // Lm       MODIFIER LETTER DOUBLE APOSTROPHE
    Interval{ 0x02EF, 0x02FF }, // Sk  [17] MODIFIER LETTER LOW DOWN ARROWHEAD..MODIFIER LETTER LOW LEFT ARROW
    Interval{ 0x0370, 0x0373 }, // L&   [4] GREEK CAPITAL LETTER HETA..GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0374, 0x0374 }, // Lm       GREEK NUMERAL SIGN
    Interval{ 0x0375, 0x0375 }, // Sk       GREEK LOWER NUMERAL SIGN
    Interval{ 0x0376, 0x0377 }, // L&   [2] GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA..GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037A, 0x037A }, // Lm       GREEK YPOGEGRAMMENI
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x037E, 0x037E }, // Po       GREEK QUESTION MARK
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0384, 0x0385 }, // Sk   [2] GREEK TONOS..GREEK DIALYTIKA TONOS
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0387, 0x0387 }, // Po       GREEK ANO TELEIA
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x03A1 }, // L&  [20] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03F5 }, // L&  [83] GREEK CAPITAL LETTER SIGMA..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F6, 0x03F6 }, // Sm       GREEK REVERSED LUNATE EPSILON SYMBOL
    Interval{ 0x03F7, 0x0481 }, // L& [139] GREEK CAPITAL LETTER SHO..CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x0482, 0x0482 }, // So       CYRILLIC THOUSANDS SIGN
    Interval{ 0x048A, 0x052F }, // L& [166] CYRILLIC CAPITAL LETTER SHORT I WITH TAIL..CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x0559, 0x0559 }, // Lm       ARMENIAN MODIFIER LETTER LEFT HALF RING
    Interval{ 0x055A, 0x055F }, // Po   [6] ARMENIAN APOSTROPHE..ARMENIAN ABBREVIATION MARK
    Interval{ 0x0560, 0x0588 }, // L&  [41] ARMENIAN SMALL LETTER TURNED AYB..ARMENIAN SMALL LETTER YI WITH STROKE
    Interval{ 0x0589, 0x0589 }, // Po       ARMENIAN FULL STOP
    Interval{ 0x058A, 0x058A }, // Pd       ARMENIAN HYPHEN
    Interval{ 0x058D, 0x058E }, // So   [2] RIGHT-FACING ARMENIAN ETERNITY SIGN..LEFT-FACING ARMENIAN ETERNITY SIGN
    Interval{ 0x058F, 0x058F }, // Sc       ARMENIAN DRAM SIGN
    Interval{ 0x05BE, 0x05BE }, // Pd       HEBREW PUNCTUATION MAQAF
    Interval{ 0x05C0, 0x05C0 }, // Po       HEBREW PUNCTUATION PASEQ
    Interval{ 0x05C3, 0x05C3 }, // Po       HEBREW PUNCTUATION SOF PASUQ
    Interval{ 0x05C6, 0x05C6 }, // Po       HEBREW PUNCTUATION NUN HAFUKHA
    Interval{ 0x05D0, 0x05EA }, // Lo  [27] HEBREW LETTER ALEF..HEBREW LETTER TAV
    Interval{ 0x05EF, 0x05F2 }, // Lo   [4] HEBREW YOD TRIANGLE..HEBREW LIGATURE YIDDISH DOUBLE YOD
    Interval{ 0x05F3, 0x05F4 }, // Po   [2] HEBREW PUNCTUATION GERESH..HEBREW PUNCTUATION GERSHAYIM
    Interval{ 0x0606, 0x0608 }, // Sm   [3] ARABIC-INDIC CUBE ROOT..ARABIC RAY
    Interval{ 0x0609, 0x060A }, // Po   [2] ARABIC-INDIC PER MILLE SIGN..ARABIC-INDIC PER TEN THOUSAND SIGN
    Interval{ 0x060B, 0x060B }, // Sc       AFGHANI SIGN
    Interval{ 0x060C, 0x060D }, // Po   [2] ARABIC COMMA..ARABIC DATE SEPARATOR
    Interval{ 0x060E, 0x060F }, // So   [2] ARABIC POETIC VERSE SIGN..ARABIC SIGN MISRA
    Interval{ 0x061B, 0x061B }, // Po       ARABIC SEMICOLON
    Interval{ 0x061D, 0x061F }, // Po   [3] ARABIC END OF TEXT MARK..ARABIC QUESTION MARK
    Interval{ 0x0620, 0x063F }, // Lo  [32] ARABIC LETTER KASHMIRI YEH..ARABIC LETTER FARSI YEH WITH THREE DOTS ABOVE
    Interval{ 0x0640, 0x0640 }, // Lm       ARABIC TATWEEL
    Interval{ 0x0641, 0x064A }, // Lo  [10] ARABIC LETTER FEH..ARABIC LETTER YEH
    Interval{ 0x0660, 0x0669 }, // Nd  [10] ARABIC-INDIC DIGIT ZERO..ARABIC-INDIC DIGIT NINE
    Interval{ 0x066A, 0x066D }, // Po   [4] ARABIC PERCENT SIGN..ARABIC FIVE POINTED STAR
    Interval{ 0x066E, 0x066F }, // Lo   [2] ARABIC LETTER DOTLESS BEH..ARABIC LETTER DOTLESS QAF
    Interval{ 0x0671, 0x06D3 }, // Lo  [99] ARABIC LETTER ALEF WASLA..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
    Interval{ 0x06D4, 0x06D4 }, // Po       ARABIC FULL STOP
    Interval{ 0x06D5, 0x06D5 }, // Lo       ARABIC LETTER AE
    Interval{ 0x06DE, 0x06DE }, // So       ARABIC START OF RUB EL HIZB
    Interval{ 0x06E5, 0x06E6 }, // Lm   [2] ARABIC SMALL WAW..ARABIC SMALL YEH
    Interval{ 0x06E9, 0x06E9 }, // So       ARABIC PLACE OF SAJDAH
    Interval{ 0x06EE, 0x06EF }, // Lo   [2] ARABIC LETTER DAL WITH INVERTED V..ARABIC LETTER REH WITH INVERTED V
    Interval{ 0x06F0, 0x06F9 }, // Nd  [10] EXTENDED ARABIC-INDIC DIGIT ZERO..EXTENDED ARABIC-INDIC DIGIT NINE
    Interval{ 0x06FA, 0x06FC }, // Lo   [3] ARABIC LETTER SHEEN WITH DOT BELOW..ARABIC LETTER GHAIN WITH DOT BELOW
    Interval{ 0x06FD, 0x06FE }, // So   [2] ARABIC SIGN SINDHI AMPERSAND..ARABIC SIGN SINDHI POSTPOSITION MEN
    Interval{ 0x06FF, 0x06FF }, // Lo       ARABIC LETTER HEH WITH INVERTED V
    Interval{ 0x0700, 0x070D }, // Po  [14] SYRIAC END OF PARAGRAPH..SYRIAC HARKLEAN ASTERISCUS
    Interval{ 0x0710, 0x0710 }, // Lo       SYRIAC LETTER ALAPH
    Interval{ 0x0712, 0x072F }, // Lo  [30] SYRIAC LETTER BETH..SYRIAC LETTER PERSIAN DHALATH
    Interval{ 0x074D, 0x07A5 }, // Lo  [89] SYRIAC LETTER SOGDIAN ZHAIN..THAANA LETTER WAAVU
    Interval{ 0x07B1, 0x07B1 }, // Lo       THAANA LETTER NAA
    Interval{ 0x07C0, 0x07C9 }, // Nd  [10] NKO DIGIT ZERO..NKO DIGIT NINE
    Interval{ 0x07CA, 0x07EA }, // Lo  [33] NKO LETTER A..NKO LETTER JONA RA
    Interval{ 0x07F4, 0x07F5 }, // Lm   [2] NKO HIGH TONE APOSTROPHE..NKO LOW TONE APOSTROPHE
    Interval{ 0x07F6, 0x07F6 }, // So       NKO SYMBOL OO DENNEN
    Interval{ 0x07F7, 0x07F9 }, // Po   [3] NKO SYMBOL GBAKURUNEN..NKO EXCLAMATION MARK
    Interval{ 0x07FA, 0x07FA }, // Lm       NKO LAJANYALAN
    Interval{ 0x07FE, 0x07FF }, // Sc   [2] NKO DOROME SIGN..NKO TAMAN SIGN
    Interval{ 0x0800, 0x0815 }, // Lo  [22] SAMARITAN LETTER ALAF..SAMARITAN LETTER TAAF
    Interval{ 0x081A, 0x081A }, // Lm       SAMARITAN MODIFIER LETTER EPENTHETIC YUT
    Interval{ 0x0824, 0x0824 }, // Lm       SAMARITAN MODIFIER LETTER SHORT A
    Interval{ 0x0828, 0x0828 }, // Lm       SAMARITAN MODIFIER LETTER I
    Interval{ 0x0830, 0x083E }, // Po  [15] SAMARITAN PUNCTUATION NEQUDAA..SAMARITAN PUNCTUATION ANNAAU
    Interval{ 0x0840, 0x0858 }, // Lo  [25] MANDAIC LETTER HALQA..MANDAIC LETTER AIN
    Interval{ 0x085E, 0x085E }, // Po       MANDAIC PUNCTUATION
    Interval{ 0x0860, 0x086A }, // Lo  [11] SYRIAC LETTER MALAYALAM NGA..SYRIAC LETTER MALAYALAM SSA
    Interval{ 0x0870, 0x0887 }, // Lo  [24] ARABIC LETTER ALEF WITH ATTACHED FATHA..ARABIC BASELINE ROUND DOT
    Interval{ 0x0888, 0x0888 }, // Sk       ARABIC RAISED ROUND DOT
    Interval{ 0x0889, 0x088E }, // Lo   [6] ARABIC LETTER NOON WITH INVERTED SMALL V..ARABIC VERTICAL TAIL
    Interval{ 0x08A0, 0x08C8 }, // Lo  [41] ARABIC LETTER BEH WITH SMALL V BELOW..ARABIC LETTER GRAF
    Interval{ 0x08C9, 0x08C9 }, // Lm       ARABIC SMALL FARSI YEH
    Interval{ 0x0903, 0x0903 }, // Mc       DEVANAGARI SIGN VISARGA
    Interval{ 0x0904, 0x0939 }, // Lo  [54] DEVANAGARI LETTER SHORT A..DEVANAGARI LETTER HA
    Interval{ 0x093B, 0x093B }, // Mc       DEVANAGARI VOWEL SIGN OOE
    Interval{ 0x093D, 0x093D }, // Lo       DEVANAGARI SIGN AVAGRAHA
    Interval{ 0x093E, 0x0940 }, // Mc   [3] DEVANAGARI VOWEL SIGN AA..DEVANAGARI VOWEL SIGN II
    Interval{ 0x0949, 0x094C }, // Mc   [4] DEVANAGARI VOWEL SIGN CANDRA O..DEVANAGARI VOWEL SIGN AU
    Interval{ 0x094E, 0x094F }, // Mc   [2] DEVANAGARI VOWEL SIGN PRISHTHAMATRA E..DEVANAGARI VOWEL SIGN AW
    Interval{ 0x0950, 0x0950 }, // Lo       DEVANAGARI OM
    Interval{ 0x0958, 0x0961 }, // Lo  [10] DEVANAGARI LETTER QA..DEVANAGARI LETTER VOCALIC LL
    Interval{ 0x0964, 0x0965 }, // Po   [2] DEVANAGARI DANDA..DEVANAGARI DOUBLE DANDA
    Interval{ 0x0966, 0x096F }, // Nd  [10] DEVANAGARI DIGIT ZERO..DEVANAGARI DIGIT NINE
    Interval{ 0x0970, 0x0970 }, // Po       DEVANAGARI ABBREVIATION SIGN
    Interval{ 0x0971, 0x0971 }, // Lm       DEVANAGARI SIGN HIGH SPACING DOT
    Interval{ 0x0972, 0x0980 }, // Lo  [15] DEVANAGARI LETTER CANDRA A..BENGALI ANJI
    Interval{ 0x0982, 0x0983 }, // Mc   [2] BENGALI SIGN ANUSVARA..BENGALI SIGN VISARGA
    Interval{ 0x0985, 0x098C }, // Lo   [8] BENGALI LETTER A..BENGALI LETTER VOCALIC L
    Interval{ 0x098F, 0x0990 }, // Lo   [2] BENGALI LETTER E..BENGALI LETTER AI
    Interval{ 0x0993, 0x09A8 }, // Lo  [22] BENGALI LETTER O..BENGALI LETTER NA
    Interval{ 0x09AA, 0x09B0 }, // Lo   [7] BENGALI LETTER PA..BENGALI LETTER RA
    Interval{ 0x09B2, 0x09B2 }, // Lo       BENGALI LETTER LA
    Interval{ 0x09B6, 0x09B9 }, // Lo   [4] BENGALI LETTER SHA..BENGALI LETTER HA
    Interval{ 0x09BD, 0x09BD }, // Lo       BENGALI SIGN AVAGRAHA
    Interval{ 0x09BF, 0x09C0 }, // Mc   [2] BENGALI VOWEL SIGN I..BENGALI VOWEL SIGN II
    Interval{ 0x09C7, 0x09C8 }, // Mc   [2] BENGALI VOWEL SIGN E..BENGALI VOWEL SIGN AI
    Interval{ 0x09CB, 0x09CC }, // Mc   [2] BENGALI VOWEL SIGN O..BENGALI VOWEL SIGN AU
    Interval{ 0x09CE, 0x09CE }, // Lo       BENGALI LETTER KHANDA TA
    Interval{ 0x09DC, 0x09DD }, // Lo   [2] BENGALI LETTER RRA..BENGALI LETTER RHA
    Interval{ 0x09DF, 0x09E1 }, // Lo   [3] BENGALI LETTER YYA..BENGALI LETTER VOCALIC LL
    Interval{ 0x09E6, 0x09EF }, // Nd  [10] BENGALI DIGIT ZERO..BENGALI DIGIT NINE
    Interval{ 0x09F0, 0x09F1 }, // Lo   [2] BENGALI LETTER RA WITH MIDDLE DIAGONAL..BENGALI LETTER RA WITH LOWER DIAGONAL
    Interval{ 0x09F2, 0x09F3 }, // Sc   [2] BENGALI RUPEE MARK..BENGALI RUPEE SIGN
    Interval{ 0x09F4, 0x09F9 }, // No   [6] BENGALI CURRENCY NUMERATOR ONE..BENGALI CURRENCY DENOMINATOR SIXTEEN
    Interval{ 0x09FA, 0x09FA }, // So       BENGALI ISSHAR
    Interval{ 0x09FB, 0x09FB }, // Sc       BENGALI GANDA MARK
    Interval{ 0x09FC, 0x09FC }, // Lo       BENGALI LETTER VEDIC ANUSVARA
    Interval{ 0x09FD, 0x09FD }, // Po       BENGALI ABBREVIATION SIGN
    Interval{ 0x0A03, 0x0A03 }, // Mc       GURMUKHI SIGN VISARGA
    Interval{ 0x0A05, 0x0A0A }, // Lo   [6] GURMUKHI LETTER A..GURMUKHI LETTER UU
    Interval{ 0x0A0F, 0x0A10 }, // Lo   [2] GURMUKHI LETTER EE..GURMUKHI LETTER AI
    Interval{ 0x0A13, 0x0A28 }, // Lo  [22] GURMUKHI LETTER OO..GURMUKHI LETTER NA
    Interval{ 0x0A2A, 0x0A30 }, // Lo   [7] GURMUKHI LETTER PA..GURMUKHI LETTER RA
    Interval{ 0x0A32, 0x0A33 }, // Lo   [2] GURMUKHI LETTER LA..GURMUKHI LETTER LLA
    Interval{ 0x0A35, 0x0A36 }, // Lo   [2] GURMUKHI LETTER VA..GURMUKHI LETTER SHA
    Interval{ 0x0A38, 0x0A39 }, // Lo   [2] GURMUKHI LETTER SA..GURMUKHI LETTER HA
    Interval{ 0x0A3E, 0x0A40 }, // Mc   [3] GURMUKHI VOWEL SIGN AA..GURMUKHI VOWEL SIGN II
    Interval{ 0x0A59, 0x0A5C }, // Lo   [4] GURMUKHI LETTER KHHA..GURMUKHI LETTER RRA
    Interval{ 0x0A5E, 0x0A5E }, // Lo       GURMUKHI LETTER FA
    Interval{ 0x0A66, 0x0A6F }, // Nd  [10] GURMUKHI DIGIT ZERO..GURMUKHI DIGIT NINE
    Interval{ 0x0A72, 0x0A74 }, // Lo   [3] GURMUKHI IRI..GURMUKHI EK ONKAR
    Interval{ 0x0A76, 0x0A76 }, // Po       GURMUKHI ABBREVIATION SIGN
    Interval{ 0x0A83, 0x0A83 }, // Mc       GUJARATI SIGN VISARGA
    Interval{ 0x0A85, 0x0A8D }, // Lo   [9] GUJARATI LETTER A..GUJARATI VOWEL CANDRA E
    Interval{ 0x0A8F, 0x0A91 }, // Lo   [3] GUJARATI LETTER E..GUJARATI VOWEL CANDRA O
    Interval{ 0x0A93, 0x0AA8 }, // Lo  [22] GUJARATI LETTER O..GUJARATI LETTER NA
    Interval{ 0x0AAA, 0x0AB0 }, // Lo   [7] GUJARATI LETTER PA..GUJARATI LETTER RA
    Interval{ 0x0AB2, 0x0AB3 }, // Lo   [2] GUJARATI LETTER LA..GUJARATI LETTER LLA
    Interval{ 0x0AB5, 0x0AB9 }, // Lo   [5] GUJARATI LETTER VA..GUJARATI LETTER HA
    Interval{ 0x0ABD, 0x0ABD }, // Lo       GUJARATI SIGN AVAGRAHA
    Interval{ 0x0ABE, 0x0AC0 }, // Mc   [3] GUJARATI VOWEL SIGN AA..GUJARATI VOWEL SIGN II
    Interval{ 0x0AC9, 0x0AC9 }, // Mc       GUJARATI VOWEL SIGN CANDRA O
    Interval{ 0x0ACB, 0x0ACC }, // Mc   [2] GUJARATI VOWEL SIGN O..GUJARATI VOWEL SIGN AU
    Interval{ 0x0AD0, 0x0AD0 }, // Lo       GUJARATI OM
    Interval{ 0x0AE0, 0x0AE1 }, // Lo   [2] GUJARATI LETTER VOCALIC RR..GUJARATI LETTER VOCALIC LL
    Interval{ 0x0AE6, 0x0AEF }, // Nd  [10] GUJARATI DIGIT ZERO..GUJARATI DIGIT NINE
    Interval{ 0x0AF0, 0x0AF0 }, // Po       GUJARATI ABBREVIATION SIGN
    Interval{ 0x0AF1, 0x0AF1 }, // Sc       GUJARATI RUPEE SIGN
    Interval{ 0x0AF9, 0x0AF9 }, // Lo       GUJARATI LETTER ZHA
    Interval{ 0x0B02, 0x0B03 }, // Mc   [2] ORIYA SIGN ANUSVARA..ORIYA SIGN VISARGA
    Interval{ 0x0B05, 0x0B0C }, // Lo   [8] ORIYA LETTER A..ORIYA LETTER VOCALIC L
    Interval{ 0x0B0F, 0x0B10 }, // Lo   [2] ORIYA LETTER E..ORIYA LETTER AI
    Interval{ 0x0B13, 0x0B28 }, // Lo  [22] ORIYA LETTER O..ORIYA LETTER NA
    Interval{ 0x0B2A, 0x0B30 }, // Lo   [7] ORIYA LETTER PA..ORIYA LETTER RA
    Interval{ 0x0B32, 0x0B33 }, // Lo   [2] ORIYA LETTER LA..ORIYA LETTER LLA
    Interval{ 0x0B35, 0x0B39 }, // Lo   [5] ORIYA LETTER VA..ORIYA LETTER HA
    Interval{ 0x0B3D, 0x0B3D }, // Lo       ORIYA SIGN AVAGRAHA
    Interval{ 0x0B40, 0x0B40 }, // Mc       ORIYA VOWEL SIGN II
    Interval{ 0x0B47, 0x0B48 }, // Mc   [2] ORIYA VOWEL SIGN E..ORIYA VOWEL SIGN AI
    Interval{ 0x0B4B, 0x0B4C }, // Mc   [2] ORIYA VOWEL SIGN O..ORIYA VOWEL SIGN AU
    Interval{ 0x0B5C, 0x0B5D }, // Lo   [2] ORIYA LETTER RRA..ORIYA LETTER RHA
    Interval{ 0x0B5F, 0x0B61 }, // Lo   [3] ORIYA LETTER YYA..ORIYA LETTER VOCALIC LL
    Interval{ 0x0B66, 0x0B6F }, // Nd  [10] ORIYA DIGIT ZERO..ORIYA DIGIT NINE
    Interval{ 0x0B70, 0x0B70 }, // So       ORIYA ISSHAR
    Interval{ 0x0B71, 0x0B71 }, // Lo       ORIYA LETTER WA
    Interval{ 0x0B72, 0x0B77 }, // No   [6] ORIYA FRACTION ONE QUARTER..ORIYA FRACTION THREE SIXTEENTHS
    Interval{ 0x0B83, 0x0B83 }, // Lo       TAMIL SIGN VISARGA
    Interval{ 0x0B85, 0x0B8A }, // Lo   [6] TAMIL LETTER A..TAMIL LETTER UU
    Interval{ 0x0B8E, 0x0B90 }, // Lo   [3] TAMIL LETTER E..TAMIL LETTER AI
    Interval{ 0x0B92, 0x0B95 }, // Lo   [4] TAMIL LETTER O..TAMIL LETTER KA
    Interval{ 0x0B99, 0x0B9A }, // Lo   [2] TAMIL LETTER NGA..TAMIL LETTER CA
    Interval{ 0x0B9C, 0x0B9C }, // Lo       TAMIL LETTER JA
    Interval{ 0x0B9E, 0x0B9F }, // Lo   [2] TAMIL LETTER NYA..TAMIL LETTER TTA
    Interval{ 0x0BA3, 0x0BA4 }, // Lo   [2] TAMIL LETTER NNA..TAMIL LETTER TA
    Interval{ 0x0BA8, 0x0BAA }, // Lo   [3] TAMIL LETTER NA..TAMIL LETTER PA
    Interval{ 0x0BAE, 0x0BB9 }, // Lo  [12] TAMIL LETTER MA..TAMIL LETTER HA
    Interval{ 0x0BBF, 0x0BBF }, // Mc       TAMIL VOWEL SIGN I
    Interval{ 0x0BC1, 0x0BC2 }, // Mc   [2] TAMIL VOWEL SIGN U..TAMIL VOWEL SIGN UU
    Interval{ 0x0BC6, 0x0BC8 }, // Mc   [3] TAMIL VOWEL SIGN E..TAMIL VOWEL SIGN AI
    Interval{ 0x0BCA, 0x0BCC }, // Mc   [3] TAMIL VOWEL SIGN O..TAMIL VOWEL SIGN AU
    Interval{ 0x0BD0, 0x0BD0 }, // Lo       TAMIL OM
    Interval{ 0x0BE6, 0x0BEF }, // Nd  [10] TAMIL DIGIT ZERO..TAMIL DIGIT NINE
    Interval{ 0x0BF0, 0x0BF2 }, // No   [3] TAMIL NUMBER TEN..TAMIL NUMBER ONE THOUSAND
    Interval{ 0x0BF3, 0x0BF8 }, // So   [6] TAMIL DAY SIGN..TAMIL AS ABOVE SIGN
    Interval{ 0x0BF9, 0x0BF9 }, // Sc       TAMIL RUPEE SIGN
    Interval{ 0x0BFA, 0x0BFA }, // So       TAMIL NUMBER SIGN
    Interval{ 0x0C01, 0x0C03 }, // Mc   [3] TELUGU SIGN CANDRABINDU..TELUGU SIGN VISARGA
    Interval{ 0x0C05, 0x0C0C }, // Lo   [8] TELUGU LETTER A..TELUGU LETTER VOCALIC L
    Interval{ 0x0C0E, 0x0C10 }, // Lo   [3] TELUGU LETTER E..TELUGU LETTER AI
    Interval{ 0x0C12, 0x0C28 }, // Lo  [23] TELUGU LETTER O..TELUGU LETTER NA
    Interval{ 0x0C2A, 0x0C39 }, // Lo  [16] TELUGU LETTER PA..TELUGU LETTER HA
    Interval{ 0x0C3D, 0x0C3D }, // Lo       TELUGU SIGN AVAGRAHA
    Interval{ 0x0C41, 0x0C44 }, // Mc   [4] TELUGU VOWEL SIGN U..TELUGU VOWEL SIGN VOCALIC RR
    Interval{ 0x0C58, 0x0C5A }, // Lo   [3] TELUGU LETTER TSA..TELUGU LETTER RRRA
    Interval{ 0x0C5D, 0x0C5D }, // Lo       TELUGU LETTER NAKAARA POLLU
    Interval{ 0x0C60, 0x0C61 }, // Lo   [2] TELUGU LETTER VOCALIC RR..TELUGU LETTER VOCALIC LL
    Interval{ 0x0C66, 0x0C6F }, // Nd  [10] TELUGU DIGIT ZERO..TELUGU DIGIT NINE
    Interval{ 0x0C77, 0x0C77 }, // Po       TELUGU SIGN SIDDHAM
    Interval{ 0x0C78, 0x0C7E }, // No   [7] TELUGU FRACTION DIGIT ZERO FOR ODD POWERS OF FOUR..TELUGU FRACTION DIGIT THREE FOR EVEN POWERS OF FOUR
    Interval{ 0x0C7F, 0x0C7F }, // So       TELUGU SIGN TUUMU
    Interval{ 0x0C80, 0x0C80 }, // Lo       KANNADA SIGN SPACING CANDRABINDU
    Interval{ 0x0C82, 0x0C83 }, // Mc   [2] KANNADA SIGN ANUSVARA..KANNADA SIGN VISARGA
    Interval{ 0x0C84, 0x0C84 }, // Po       KANNADA SIGN SIDDHAM
    Interval{ 0x0C85, 0x0C8C }, // Lo   [8] KANNADA LETTER A..KANNADA LETTER VOCALIC L
    Interval{ 0x0C8E, 0x0C90 }, // Lo   [3] KANNADA LETTER E..KANNADA LETTER AI
    Interval{ 0x0C92, 0x0CA8 }, // Lo  [23] KANNADA LETTER O..KANNADA LETTER NA
    Interval{ 0x0CAA, 0x0CB3 }, // Lo  [10] KANNADA LETTER PA..KANNADA LETTER LLA
    Interval{ 0x0CB5, 0x0CB9 }, // Lo   [5] KANNADA LETTER VA..KANNADA LETTER HA
    Interval{ 0x0CBD, 0x0CBD }, // Lo       KANNADA SIGN AVAGRAHA
    Interval{ 0x0CBE, 0x0CBE }, // Mc       KANNADA VOWEL SIGN AA
    Interval{ 0x0CC0, 0x0CC1 }, // Mc   [2] KANNADA VOWEL SIGN II..KANNADA VOWEL SIGN U
    Interval{ 0x0CC3, 0x0CC4 }, // Mc   [2] KANNADA VOWEL SIGN VOCALIC R..KANNADA VOWEL SIGN VOCALIC RR
    Interval{ 0x0CC7, 0x0CC8 }, // Mc   [2] KANNADA VOWEL SIGN EE..KANNADA VOWEL SIGN AI
    Interval{ 0x0CCA, 0x0CCB }, // Mc   [2] KANNADA VOWEL SIGN O..KANNADA VOWEL SIGN OO
    Interval{ 0x0CDD, 0x0CDE }, // Lo   [2] KANNADA LETTER NAKAARA POLLU..KANNADA LETTER FA
    Interval{ 0x0CE0, 0x0CE1 }, // Lo   [2] KANNADA LETTER VOCALIC RR..KANNADA LETTER VOCALIC LL
    Interval{ 0x0CE6, 0x0CEF }, // Nd  [10] KANNADA DIGIT ZERO..KANNADA DIGIT NINE
    Interval{ 0x0CF1, 0x0CF2 }, // Lo   [2] KANNADA SIGN JIHVAMULIYA..KANNADA SIGN UPADHMANIYA
    Interval{ 0x0CF3, 0x0CF3 }, // Mc       KANNADA SIGN COMBINING ANUSVARA ABOVE RIGHT
    Interval{ 0x0D02, 0x0D03 }, // Mc   [2] MALAYALAM SIGN ANUSVARA..MALAYALAM SIGN VISARGA
    Interval{ 0x0D04, 0x0D0C }, // Lo   [9] MALAYALAM LETTER VEDIC ANUSVARA..MALAYALAM LETTER VOCALIC L
    Interval{ 0x0D0E, 0x0D10 }, // Lo   [3] MALAYALAM LETTER E..MALAYALAM LETTER AI
    Interval{ 0x0D12, 0x0D3A }, // Lo  [41] MALAYALAM LETTER O..MALAYALAM LETTER TTTA
    Interval{ 0x0D3D, 0x0D3D }, // Lo       MALAYALAM SIGN AVAGRAHA
    Interval{ 0x0D3F, 0x0D40 }, // Mc   [2] MALAYALAM VOWEL SIGN I..MALAYALAM VOWEL SIGN II
    Interval{ 0x0D46, 0x0D48 }, // Mc   [3] MALAYALAM VOWEL SIGN E..MALAYALAM VOWEL SIGN AI
    Interval{ 0x0D4A, 0x0D4C }, // Mc   [3] MALAYALAM VOWEL SIGN O..MALAYALAM VOWEL SIGN AU
    Interval{ 0x0D4E, 0x0D4E }, // Lo       MALAYALAM LETTER DOT REPH
    Interval{ 0x0D4F, 0x0D4F }, // So       MALAYALAM SIGN PARA
    Interval{ 0x0D54, 0x0D56 }, // Lo   [3] MALAYALAM LETTER CHILLU M..MALAYALAM LETTER CHILLU LLL
    Interval{ 0x0D58, 0x0D5E }, // No   [7] MALAYALAM FRACTION ONE ONE-HUNDRED-AND-SIXTIETH..MALAYALAM FRACTION ONE FIFTH
    Interval{ 0x0D5F, 0x0D61 }, // Lo   [3] MALAYALAM LETTER ARCHAIC II..MALAYALAM LETTER VOCALIC LL
    Interval{ 0x0D66, 0x0D6F }, // Nd  [10] MALAYALAM DIGIT ZERO..MALAYALAM DIGIT NINE
    Interval{ 0x0D70, 0x0D78 }, // No   [9] MALAYALAM NUMBER TEN..MALAYALAM FRACTION THREE SIXTEENTHS
    Interval{ 0x0D79, 0x0D79 }, // So       MALAYALAM DATE MARK
    Interval{ 0x0D7A, 0x0D7F }, // Lo   [6] MALAYALAM LETTER CHILLU NN..MALAYALAM LETTER CHILLU K
    Interval{ 0x0D82, 0x0D83 }, // Mc   [2] SINHALA SIGN ANUSVARAYA..SINHALA SIGN VISARGAYA
    Interval{ 0x0D85, 0x0D96 }, // Lo  [18] SINHALA LETTER AYANNA..SINHALA LETTER AUYANNA
    Interval{ 0x0D9A, 0x0DB1 }, // Lo  [24] SINHALA LETTER ALPAPRAANA KAYANNA..SINHALA LETTER DANTAJA NAYANNA
    Interval{ 0x0DB3, 0x0DBB }, // Lo   [9] SINHALA LETTER SANYAKA DAYANNA..SINHALA LETTER RAYANNA
    Interval{ 0x0DBD, 0x0DBD }, // Lo       SINHALA LETTER DANTAJA LAYANNA
    Interval{ 0x0DC0, 0x0DC6 }, // Lo   [7] SINHALA LETTER VAYANNA..SINHALA LETTER FAYANNA
    Interval{ 0x0DD0, 0x0DD1 }, // Mc   [2] SINHALA VOWEL SIGN KETTI AEDA-PILLA..SINHALA VOWEL SIGN DIGA AEDA-PILLA
    Interval{ 0x0DD8, 0x0DDE }, // Mc   [7] SINHALA VOWEL SIGN GAETTA-PILLA..SINHALA VOWEL SIGN KOMBUVA HAA GAYANUKITTA
    Interval{ 0x0DE6, 0x0DEF }, // Nd  [10] SINHALA LITH DIGIT ZERO..SINHALA LITH DIGIT NINE
    Interval{ 0x0DF2, 0x0DF3 }, // Mc   [2] SINHALA VOWEL SIGN DIGA GAETTA-PILLA..SINHALA VOWEL SIGN DIGA GAYANUKITTA
    Interval{ 0x0DF4, 0x0DF4 }, // Po       SINHALA PUNCTUATION KUNDDALIYA
    Interval{ 0x0E01, 0x0E30 }, // Lo  [48] THAI CHARACTER KO KAI..THAI CHARACTER SARA A
    Interval{ 0x0E32, 0x0E33 }, // Lo   [2] THAI CHARACTER SARA AA..THAI CHARACTER SARA AM
    Interval{ 0x0E3F, 0x0E3F }, // Sc       THAI CURRENCY SYMBOL BAHT
    Interval{ 0x0E40, 0x0E45 }, // Lo   [6] THAI CHARACTER SARA E..THAI CHARACTER LAKKHANGYAO
    Interval{ 0x0E46, 0x0E46 }, // Lm       THAI CHARACTER MAIYAMOK
    Interval{ 0x0E4F, 0x0E4F }, // Po       THAI CHARACTER FONGMAN
    Interval{ 0x0E50, 0x0E59 }, // Nd  [10] THAI DIGIT ZERO..THAI DIGIT NINE
    Interval{ 0x0E5A, 0x0E5B }, // Po   [2] THAI CHARACTER ANGKHANKHU..THAI CHARACTER KHOMUT
    Interval{ 0x0E81, 0x0E82 }, // Lo   [2] LAO LETTER KO..LAO LETTER KHO SUNG
    Interval{ 0x0E84, 0x0E84 }, // Lo       LAO LETTER KHO TAM
    Interval{ 0x0E86, 0x0E8A }, // Lo   [5] LAO LETTER PALI GHA..LAO LETTER SO TAM
    Interval{ 0x0E8C, 0x0EA3 }, // Lo  [24] LAO LETTER PALI JHA..LAO LETTER LO LING
    Interval{ 0x0EA5, 0x0EA5 }, // Lo       LAO LETTER LO LOOT
    Interval{ 0x0EA7, 0x0EB0 }, // Lo  [10] LAO LETTER WO..LAO VOWEL SIGN A
    Interval{ 0x0EB2, 0x0EB3 }, // Lo   [2] LAO VOWEL SIGN AA..LAO VOWEL SIGN AM
    Interval{ 0x0EBD, 0x0EBD }, // Lo       LAO SEMIVOWEL SIGN NYO
    Interval{ 0x0EC0, 0x0EC4 }, // Lo   [5] LAO VOWEL SIGN E..LAO VOWEL SIGN AI
    Interval{ 0x0EC6, 0x0EC6 }, // Lm       LAO KO LA
    Interval{ 0x0ED0, 0x0ED9 }, // Nd  [10] LAO DIGIT ZERO..LAO DIGIT NINE
    Interval{ 0x0EDC, 0x0EDF }, // Lo   [4] LAO HO NO..LAO LETTER KHMU NYO
    Interval{ 0x0F00, 0x0F00 }, // Lo       TIBETAN SYLLABLE OM
    Interval{ 0x0F01, 0x0F03 }, // So   [3] TIBETAN MARK GTER YIG MGO TRUNCATED A..TIBETAN MARK GTER YIG MGO -UM GTER TSHEG MA
    Interval{ 0x0F04, 0x0F12 }, // Po  [15] TIBETAN MARK INITIAL YIG MGO MDUN MA..TIBETAN MARK RGYA GRAM SHAD
    Interval{ 0x0F13, 0x0F13 }, // So       TIBETAN MARK CARET -DZUD RTAGS ME LONG CAN
    Interval{ 0x0F14, 0x0F14 }, // Po       TIBETAN MARK GTER TSHEG
    Interval{ 0x0F15, 0x0F17 }, // So   [3] TIBETAN LOGOTYPE SIGN CHAD RTAGS..TIBETAN ASTROLOGICAL SIGN SGRA GCAN -CHAR RTAGS
    Interval{ 0x0F1A, 0x0F1F }, // So   [6] TIBETAN SIGN RDEL DKAR GCIG..TIBETAN SIGN RDEL DKAR RDEL NAG
    Interval{ 0x0F20, 0x0F29 }, // Nd  [10] TIBETAN DIGIT ZERO..TIBETAN DIGIT NINE
    Interval{ 0x0F2A, 0x0F33 }, // No  [10] TIBETAN DIGIT HALF ONE..TIBETAN DIGIT HALF ZERO
    Interval{ 0x0F34, 0x0F34 }, // So       TIBETAN MARK BSDUS RTAGS
    Interval{ 0x0F36, 0x0F36 }, // So       TIBETAN MARK CARET -DZUD RTAGS BZHI MIG CAN
    Interval{ 0x0F38, 0x0F38 }, // So       TIBETAN MARK CHE MGO
    Interval{ 0x0F3A, 0x0F3A }, // Ps       TIBETAN MARK GUG RTAGS GYON
    Interval{ 0x0F3B, 0x0F3B }, // Pe       TIBETAN MARK GUG RTAGS GYAS
    Interval{ 0x0F3C, 0x0F3C }, // Ps       TIBETAN MARK ANG KHANG GYON
    Interval{ 0x0F3D, 0x0F3D }, // Pe       TIBETAN MARK ANG KHANG GYAS
    Interval{ 0x0F3E, 0x0F3F }, // Mc   [2] TIBETAN SIGN YAR TSHES..TIBETAN SIGN MAR TSHES
    Interval{ 0x0F40, 0x0F47 }, // Lo   [8] TIBETAN LETTER KA..TIBETAN LETTER JA
    Interval{ 0x0F49, 0x0F6C }, // Lo  [36] TIBETAN LETTER NYA..TIBETAN LETTER RRA
    Interval{ 0x0F7F, 0x0F7F }, // Mc       TIBETAN SIGN RNAM BCAD
    Interval{ 0x0F85, 0x0F85 }, // Po       TIBETAN MARK PALUTA
    Interval{ 0x0F88, 0x0F8C }, // Lo   [5] TIBETAN SIGN LCE TSA CAN..TIBETAN SIGN INVERTED MCHU CAN
    Interval{ 0x0FBE, 0x0FC5 }, // So   [8] TIBETAN KU RU KHA..TIBETAN SYMBOL RDO RJE
    Interval{ 0x0FC7, 0x0FCC }, // So   [6] TIBETAN SYMBOL RDO RJE RGYA GRAM..TIBETAN SYMBOL NOR BU BZHI -KHYIL
    Interval{ 0x0FCE, 0x0FCF }, // So   [2] TIBETAN SIGN RDEL NAG RDEL DKAR..TIBETAN SIGN RDEL NAG GSUM
    Interval{ 0x0FD0, 0x0FD4 }, // Po   [5] TIBETAN MARK BSKA- SHOG GI MGO RGYAN..TIBETAN MARK CLOSING BRDA RNYING YIG MGO SGAB MA
    Interval{ 0x0FD5, 0x0FD8 }, // So   [4] RIGHT-FACING SVASTI SIGN..LEFT-FACING SVASTI SIGN WITH DOTS
    Interval{ 0x0FD9, 0x0FDA }, // Po   [2] TIBETAN MARK LEADING MCHAN RTAGS..TIBETAN MARK TRAILING MCHAN RTAGS
    Interval{ 0x1000, 0x102A }, // Lo  [43] MYANMAR LETTER KA..MYANMAR LETTER AU
    Interval{ 0x102B, 0x102C }, // Mc   [2] MYANMAR VOWEL SIGN TALL AA..MYANMAR VOWEL SIGN AA
    Interval{ 0x1031, 0x1031 }, // Mc       MYANMAR VOWEL SIGN E
    Interval{ 0x1038, 0x1038 }, // Mc       MYANMAR SIGN VISARGA
    Interval{ 0x103B, 0x103C }, // Mc   [2] MYANMAR CONSONANT SIGN MEDIAL YA..MYANMAR CONSONANT SIGN MEDIAL RA
    Interval{ 0x103F, 0x103F }, // Lo       MYANMAR LETTER GREAT SA
    Interval{ 0x1040, 0x1049 }, // Nd  [10] MYANMAR DIGIT ZERO..MYANMAR DIGIT NINE
    Interval{ 0x104A, 0x104F }, // Po   [6] MYANMAR SIGN LITTLE SECTION..MYANMAR SYMBOL GENITIVE
    Interval{ 0x1050, 0x1055 }, // Lo   [6] MYANMAR LETTER SHA..MYANMAR LETTER VOCALIC LL
    Interval{ 0x1056, 0x1057 }, // Mc   [2] MYANMAR VOWEL SIGN VOCALIC R..MYANMAR VOWEL SIGN VOCALIC RR
    Interval{ 0x105A, 0x105D }, // Lo   [4] MYANMAR LETTER MON NGA..MYANMAR LETTER MON BBE
    Interval{ 0x1061, 0x1061 }, // Lo       MYANMAR LETTER SGAW KAREN SHA
    Interval{ 0x1062, 0x1064 }, // Mc   [3] MYANMAR VOWEL SIGN SGAW KAREN EU..MYANMAR TONE MARK SGAW KAREN KE PHO
    Interval{ 0x1065, 0x1066 }, // Lo   [2] MYANMAR LETTER WESTERN PWO KAREN THA..MYANMAR LETTER WESTERN PWO KAREN PWA
    Interval{ 0x1067, 0x106D }, // Mc   [7] MYANMAR VOWEL SIGN WESTERN PWO KAREN EU..MYANMAR SIGN WESTERN PWO KAREN TONE-5
    Interval{ 0x106E, 0x1070 }, // Lo   [3] MYANMAR LETTER EASTERN PWO KAREN NNA..MYANMAR LETTER EASTERN PWO KAREN GHWA
    Interval{ 0x1075, 0x1081 }, // Lo  [13] MYANMAR LETTER SHAN KA..MYANMAR LETTER SHAN HA
    Interval{ 0x1083, 0x1084 }, // Mc   [2] MYANMAR VOWEL SIGN SHAN AA..MYANMAR VOWEL SIGN SHAN E
    Interval{ 0x1087, 0x108C }, // Mc   [6] MYANMAR SIGN SHAN TONE-2..MYANMAR SIGN SHAN COUNCIL TONE-3
    Interval{ 0x108E, 0x108E }, // Lo       MYANMAR LETTER RUMAI PALAUNG FA
    Interval{ 0x108F, 0x108F }, // Mc       MYANMAR SIGN RUMAI PALAUNG TONE-5
    Interval{ 0x1090, 0x1099 }, // Nd  [10] MYANMAR SHAN DIGIT ZERO..MYANMAR SHAN DIGIT NINE
    Interval{ 0x109A, 0x109C }, // Mc   [3] MYANMAR SIGN KHAMTI TONE-1..MYANMAR VOWEL SIGN AITON A
    Interval{ 0x109E, 0x109F }, // So   [2] MYANMAR SYMBOL SHAN ONE..MYANMAR SYMBOL SHAN EXCLAMATION
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FB, 0x10FB }, // Po       GEORGIAN PARAGRAPH SEPARATOR
    Interval{ 0x10FC, 0x10FC }, // Lm       MODIFIER LETTER GEORGIAN NAR
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x1100, 0x1248 }, // Lo [329] HANGUL CHOSEONG KIYEOK..ETHIOPIC SYLLABLE QWA
    Interval{ 0x124A, 0x124D }, // Lo   [4] ETHIOPIC SYLLABLE QWI..ETHIOPIC SYLLABLE QWE
    Interval{ 0x1250, 0x1256 }, // Lo   [7] ETHIOPIC SYLLABLE QHA..ETHIOPIC SYLLABLE QHO
    Interval{ 0x1258, 0x1258 }, // Lo       ETHIOPIC SYLLABLE QHWA
    Interval{ 0x125A, 0x125D }, // Lo   [4] ETHIOPIC SYLLABLE QHWI..ETHIOPIC SYLLABLE QHWE
    Interval{ 0x1260, 0x1288 }, // Lo  [41] ETHIOPIC SYLLABLE BA..ETHIOPIC SYLLABLE XWA
    Interval{ 0x128A, 0x128D }, // Lo   [4] ETHIOPIC SYLLABLE XWI..ETHIOPIC SYLLABLE XWE
    Interval{ 0x1290, 0x12B0 }, // Lo  [33] ETHIOPIC SYLLABLE NA..ETHIOPIC SYLLABLE KWA
    Interval{ 0x12B2, 0x12B5 }, // Lo   [4] ETHIOPIC SYLLABLE KWI..ETHIOPIC SYLLABLE KWE
    Interval{ 0x12B8, 0x12BE }, // Lo   [7] ETHIOPIC SYLLABLE KXA..ETHIOPIC SYLLABLE KXO
    Interval{ 0x12C0, 0x12C0 }, // Lo       ETHIOPIC SYLLABLE KXWA
    Interval{ 0x12C2, 0x12C5 }, // Lo   [4] ETHIOPIC SYLLABLE KXWI..ETHIOPIC SYLLABLE KXWE
    Interval{ 0x12C8, 0x12D6 }, // Lo  [15] ETHIOPIC SYLLABLE WA..ETHIOPIC SYLLABLE PHARYNGEAL O
    Interval{ 0x12D8, 0x1310 }, // Lo  [57] ETHIOPIC SYLLABLE ZA..ETHIOPIC SYLLABLE GWA
    Interval{ 0x1312, 0x1315 }, // Lo   [4] ETHIOPIC SYLLABLE GWI..ETHIOPIC SYLLABLE GWE
    Interval{ 0x1318, 0x135A }, // Lo  [67] ETHIOPIC SYLLABLE GGA..ETHIOPIC SYLLABLE FYA
    Interval{ 0x1360, 0x1368 }, // Po   [9] ETHIOPIC SECTION MARK..ETHIOPIC PARAGRAPH SEPARATOR
    Interval{ 0x1369, 0x137C }, // No  [20] ETHIOPIC DIGIT ONE..ETHIOPIC NUMBER TEN THOUSAND
    Interval{ 0x1380, 0x138F }, // Lo  [16] ETHIOPIC SYLLABLE SEBATBEIT MWA..ETHIOPIC SYLLABLE PWE
    Interval{ 0x1390, 0x1399 }, // So  [10] ETHIOPIC TONAL MARK YIZET..ETHIOPIC TONAL MARK KURT
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1400, 0x1400 }, // Pd       CANADIAN SYLLABICS HYPHEN
    Interval{ 0x1401, 0x166C }, // Lo [620] CANADIAN SYLLABICS E..CANADIAN SYLLABICS CARRIER TTSA
    Interval{ 0x166D, 0x166D }, // So       CANADIAN SYLLABICS CHI SIGN
    Interval{ 0x166E, 0x166E }, // Po       CANADIAN SYLLABICS FULL STOP
    Interval{ 0x166F, 0x167F }, // Lo  [17] CANADIAN SYLLABICS QAI..CANADIAN SYLLABICS BLACKFOOT W
    Interval{ 0x1680, 0x1680 }, // Zs       OGHAM SPACE MARK
    Interval{ 0x1681, 0x169A }, // Lo  [26] OGHAM LETTER BEITH..OGHAM LETTER PEITH
    Interval{ 0x169B, 0x169B }, // Ps       OGHAM FEATHER MARK
    Interval{ 0x169C, 0x169C }, // Pe       OGHAM REVERSED FEATHER MARK
    Interval{ 0x16A0, 0x16EA }, // Lo  [75] RUNIC LETTER FEHU FEOH FE F..RUNIC LETTER X
    Interval{ 0x16EB, 0x16ED }, // Po   [3] RUNIC SINGLE PUNCTUATION..RUNIC CROSS PUNCTUATION
    Interval{ 0x16EE, 0x16F0 }, // Nl   [3] RUNIC ARLAUG SYMBOL..RUNIC BELGTHOR SYMBOL
    Interval{ 0x16F1, 0x16F8 }, // Lo   [8] RUNIC LETTER K..RUNIC LETTER FRANKS CASKET AESC
    Interval{ 0x1700, 0x1711 }, // Lo  [18] TAGALOG LETTER A..TAGALOG LETTER HA
    Interval{ 0x1715, 0x1715 }, // Mc       TAGALOG SIGN PAMUDPOD
    Interval{ 0x171F, 0x1731 }, // Lo  [19] TAGALOG LETTER ARCHAIC RA..HANUNOO LETTER HA
    Interval{ 0x1734, 0x1734 }, // Mc       HANUNOO SIGN PAMUDPOD
    Interval{ 0x1735, 0x1736 }, // Po   [2] PHILIPPINE SINGLE PUNCTUATION..PHILIPPINE DOUBLE PUNCTUATION
    Interval{ 0x1740, 0x1751 }, // Lo  [18] BUHID LETTER A..BUHID LETTER HA
    Interval{ 0x1760, 0x176C }, // Lo  [13] TAGBANWA LETTER A..TAGBANWA LETTER YA
    Interval{ 0x176E, 0x1770 }, // Lo   [3] TAGBANWA LETTER LA..TAGBANWA LETTER SA
    Interval{ 0x1780, 0x17B3 }, // Lo  [52] KHMER LETTER KA..KHMER INDEPENDENT VOWEL QAU
    Interval{ 0x17B6, 0x17B6 }, // Mc       KHMER VOWEL SIGN AA
    Interval{ 0x17BE, 0x17C5 }, // Mc   [8] KHMER VOWEL SIGN OE..KHMER VOWEL SIGN AU
    Interval{ 0x17C7, 0x17C8 }, // Mc   [2] KHMER SIGN REAHMUK..KHMER SIGN YUUKALEAPINTU
    Interval{ 0x17D4, 0x17D6 }, // Po   [3] KHMER SIGN KHAN..KHMER SIGN CAMNUC PII KUUH
    Interval{ 0x17D7, 0x17D7 }, // Lm       KHMER SIGN LEK TOO
    Interval{ 0x17D8, 0x17DA }, // Po   [3] KHMER SIGN BEYYAL..KHMER SIGN KOOMUUT
    Interval{ 0x17DB, 0x17DB }, // Sc       KHMER CURRENCY SYMBOL RIEL
    Interval{ 0x17DC, 0x17DC }, // Lo       KHMER SIGN AVAKRAHASANYA
    Interval{ 0x17E0, 0x17E9 }, // Nd  [10] KHMER DIGIT ZERO..KHMER DIGIT NINE
    Interval{ 0x17F0, 0x17F9 }, // No  [10] KHMER SYMBOL LEK ATTAK SON..KHMER SYMBOL LEK ATTAK PRAM-BUON
    Interval{ 0x1800, 0x1805 }, // Po   [6] MONGOLIAN BIRGA..MONGOLIAN FOUR DOTS
    Interval{ 0x1806, 0x1806 }, // Pd       MONGOLIAN TODO SOFT HYPHEN
    Interval{ 0x1807, 0x180A }, // Po   [4] MONGOLIAN SIBE SYLLABLE BOUNDARY MARKER..MONGOLIAN NIRUGU
    Interval{ 0x1810, 0x1819 }, // Nd  [10] MONGOLIAN DIGIT ZERO..MONGOLIAN DIGIT NINE
    Interval{ 0x1820, 0x1842 }, // Lo  [35] MONGOLIAN LETTER A..MONGOLIAN LETTER CHI
    Interval{ 0x1843, 0x1843 }, // Lm       MONGOLIAN LETTER TODO LONG VOWEL SIGN
    Interval{ 0x1844, 0x1878 }, // Lo  [53] MONGOLIAN LETTER TODO E..MONGOLIAN LETTER CHA WITH TWO DOTS
    Interval{ 0x1880, 0x1884 }, // Lo   [5] MONGOLIAN LETTER ALI GALI ANUSVARA ONE..MONGOLIAN LETTER ALI GALI INVERTED UBADAMA
    Interval{ 0x1887, 0x18A8 }, // Lo  [34] MONGOLIAN LETTER ALI GALI A..MONGOLIAN LETTER MANCHU ALI GALI BHA
    Interval{ 0x18AA, 0x18AA }, // Lo       MONGOLIAN LETTER MANCHU ALI GALI LHA
    Interval{ 0x18B0, 0x18F5 }, // Lo  [70] CANADIAN SYLLABICS OY..CANADIAN SYLLABICS CARRIER DENTAL S
    Interval{ 0x1900, 0x191E }, // Lo  [31] LIMBU VOWEL-CARRIER LETTER..LIMBU LETTER TRA
    Interval{ 0x1923, 0x1926 }, // Mc   [4] LIMBU VOWEL SIGN EE..LIMBU VOWEL SIGN AU
    Interval{ 0x1929, 0x192B }, // Mc   [3] LIMBU SUBJOINED LETTER YA..LIMBU SUBJOINED LETTER WA
    Interval{ 0x1930, 0x1931 }, // Mc   [2] LIMBU SMALL LETTER KA..LIMBU SMALL LETTER NGA
    Interval{ 0x1933, 0x1938 }, // Mc   [6] LIMBU SMALL LETTER TA..LIMBU SMALL LETTER LA
    Interval{ 0x1940, 0x1940 }, // So       LIMBU SIGN LOO
    Interval{ 0x1944, 0x1945 }, // Po   [2] LIMBU EXCLAMATION MARK..LIMBU QUESTION MARK
    Interval{ 0x1946, 0x194F }, // Nd  [10] LIMBU DIGIT ZERO..LIMBU DIGIT NINE
    Interval{ 0x1950, 0x196D }, // Lo  [30] TAI LE LETTER KA..TAI LE LETTER AI
    Interval{ 0x1970, 0x1974 }, // Lo   [5] TAI LE LETTER TONE-2..TAI LE LETTER TONE-6
    Interval{ 0x1980, 0x19AB }, // Lo  [44] NEW TAI LUE LETTER HIGH QA..NEW TAI LUE LETTER LOW SUA
    Interval{ 0x19B0, 0x19C9 }, // Lo  [26] NEW TAI LUE VOWEL SIGN VOWEL SHORTENER..NEW TAI LUE TONE MARK-2
    Interval{ 0x19D0, 0x19D9 }, // Nd  [10] NEW TAI LUE DIGIT ZERO..NEW TAI LUE DIGIT NINE
    Interval{ 0x19DA, 0x19DA }, // No       NEW TAI LUE THAM DIGIT ONE
    Interval{ 0x19DE, 0x19FF }, // So  [34] NEW TAI LUE SIGN LAE..KHMER SYMBOL DAP-PRAM ROC
    Interval{ 0x1A00, 0x1A16 }, // Lo  [23] BUGINESE LETTER KA..BUGINESE LETTER HA
    Interval{ 0x1A19, 0x1A1A }, // Mc   [2] BUGINESE VOWEL SIGN E..BUGINESE VOWEL SIGN O
    Interval{ 0x1A1E, 0x1A1F }, // Po   [2] BUGINESE PALLAWA..BUGINESE END OF SECTION
    Interval{ 0x1A20, 0x1A54 }, // Lo  [53] TAI THAM LETTER HIGH KA..TAI THAM LETTER GREAT SA
    Interval{ 0x1A55, 0x1A55 }, // Mc       TAI THAM CONSONANT SIGN MEDIAL RA
    Interval{ 0x1A57, 0x1A57 }, // Mc       TAI THAM CONSONANT SIGN LA TANG LAI
    Interval{ 0x1A61, 0x1A61 }, // Mc       TAI THAM VOWEL SIGN A
    Interval{ 0x1A63, 0x1A64 }, // Mc   [2] TAI THAM VOWEL SIGN AA..TAI THAM VOWEL SIGN TALL AA
    Interval{ 0x1A6D, 0x1A72 }, // Mc   [6] TAI THAM VOWEL SIGN OY..TAI THAM VOWEL SIGN THAM AI
    Interval{ 0x1A80, 0x1A89 }, // Nd  [10] TAI THAM HORA DIGIT ZERO..TAI THAM HORA DIGIT NINE
    Interval{ 0x1A90, 0x1A99 }, // Nd  [10] TAI THAM THAM DIGIT ZERO..TAI THAM THAM DIGIT NINE
    Interval{ 0x1AA0, 0x1AA6 }, // Po   [7] TAI THAM SIGN WIANG..TAI THAM SIGN REVERSED ROTATED RANA
    Interval{ 0x1AA7, 0x1AA7 }, // Lm       TAI THAM SIGN MAI YAMOK
    Interval{ 0x1AA8, 0x1AAD }, // Po   [6] TAI THAM SIGN KAAN..TAI THAM SIGN CAANG
    Interval{ 0x1B04, 0x1B04 }, // Mc       BALINESE SIGN BISAH
    Interval{ 0x1B05, 0x1B33 }, // Lo  [47] BALINESE LETTER AKARA..BALINESE LETTER HA
    Interval{ 0x1B3B, 0x1B3B }, // Mc       BALINESE VOWEL SIGN RA REPA TEDUNG
    Interval{ 0x1B3D, 0x1B41 }, // Mc   [5] BALINESE VOWEL SIGN LA LENGA TEDUNG..BALINESE VOWEL SIGN TALING REPA TEDUNG
    Interval{ 0x1B43, 0x1B44 }, // Mc   [2] BALINESE VOWEL SIGN PEPET TEDUNG..BALINESE ADEG ADEG
    Interval{ 0x1B45, 0x1B4C }, // Lo   [8] BALINESE LETTER KAF SASAK..BALINESE LETTER ARCHAIC JNYA
    Interval{ 0x1B50, 0x1B59 }, // Nd  [10] BALINESE DIGIT ZERO..BALINESE DIGIT NINE
    Interval{ 0x1B5A, 0x1B60 }, // Po   [7] BALINESE PANTI..BALINESE PAMENENG
    Interval{ 0x1B61, 0x1B6A }, // So  [10] BALINESE MUSICAL SYMBOL DONG..BALINESE MUSICAL SYMBOL DANG GEDE
    Interval{ 0x1B74, 0x1B7C }, // So   [9] BALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DUG..BALINESE MUSICAL SYMBOL LEFT-HAND OPEN PING
    Interval{ 0x1B7D, 0x1B7E }, // Po   [2] BALINESE PANTI LANTANG..BALINESE PAMADA LANTANG
    Interval{ 0x1B82, 0x1B82 }, // Mc       SUNDANESE SIGN PANGWISAD
    Interval{ 0x1B83, 0x1BA0 }, // Lo  [30] SUNDANESE LETTER A..SUNDANESE LETTER HA
    Interval{ 0x1BA1, 0x1BA1 }, // Mc       SUNDANESE CONSONANT SIGN PAMINGKAL
    Interval{ 0x1BA6, 0x1BA7 }, // Mc   [2] SUNDANESE VOWEL SIGN PANAELAENG..SUNDANESE VOWEL SIGN PANOLONG
    Interval{ 0x1BAA, 0x1BAA }, // Mc       SUNDANESE SIGN PAMAAEH
    Interval{ 0x1BAE, 0x1BAF }, // Lo   [2] SUNDANESE LETTER KHA..SUNDANESE LETTER SYA
    Interval{ 0x1BB0, 0x1BB9 }, // Nd  [10] SUNDANESE DIGIT ZERO..SUNDANESE DIGIT NINE
    Interval{ 0x1BBA, 0x1BE5 }, // Lo  [44] SUNDANESE AVAGRAHA..BATAK LETTER U
    Interval{ 0x1BE7, 0x1BE7 }, // Mc       BATAK VOWEL SIGN E
    Interval{ 0x1BEA, 0x1BEC }, // Mc   [3] BATAK VOWEL SIGN I..BATAK VOWEL SIGN O
    Interval{ 0x1BEE, 0x1BEE }, // Mc       BATAK VOWEL SIGN U
    Interval{ 0x1BF2, 0x1BF3 }, // Mc   [2] BATAK PANGOLAT..BATAK PANONGONAN
    Interval{ 0x1BFC, 0x1BFF }, // Po   [4] BATAK SYMBOL BINDU NA METEK..BATAK SYMBOL BINDU PANGOLAT
    Interval{ 0x1C00, 0x1C23 }, // Lo  [36] LEPCHA LETTER KA..LEPCHA LETTER A
    Interval{ 0x1C24, 0x1C2B }, // Mc   [8] LEPCHA SUBJOINED LETTER YA..LEPCHA VOWEL SIGN UU
    Interval{ 0x1C34, 0x1C35 }, // Mc   [2] LEPCHA CONSONANT SIGN NYIN-DO..LEPCHA CONSONANT SIGN KANG
    Interval{ 0x1C3B, 0x1C3F }, // Po   [5] LEPCHA PUNCTUATION TA-ROL..LEPCHA PUNCTUATION TSHOOK
    Interval{ 0x1C40, 0x1C49 }, // Nd  [10] LEPCHA DIGIT ZERO..LEPCHA DIGIT NINE
    Interval{ 0x1C4D, 0x1C4F }, // Lo   [3] LEPCHA LETTER TTA..LEPCHA LETTER DDA
    Interval{ 0x1C50, 0x1C59 }, // Nd  [10] OL CHIKI DIGIT ZERO..OL CHIKI DIGIT NINE
    Interval{ 0x1C5A, 0x1C77 }, // Lo  [30] OL CHIKI LETTER LA..OL CHIKI LETTER OH
    Interval{ 0x1C78, 0x1C7D }, // Lm   [6] OL CHIKI MU TTUDDAG..OL CHIKI AHAD
    Interval{ 0x1C7E, 0x1C7F }, // Po   [2] OL CHIKI PUNCTUATION MUCAAD..OL CHIKI PUNCTUATION DOUBLE MUCAAD
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1CC0, 0x1CC7 }, // Po   [8] SUNDANESE PUNCTUATION BINDU SURYA..SUNDANESE PUNCTUATION BINDU BA SATANGA
    Interval{ 0x1CD3, 0x1CD3 }, // Po       VEDIC SIGN NIHSHVASA
    Interval{ 0x1CE1, 0x1CE1 }, // Mc       VEDIC TONE ATHARVAVEDIC INDEPENDENT SVARITA
    Interval{ 0x1CE9, 0x1CEC }, // Lo   [4] VEDIC SIGN ANUSVARA ANTARGOMUKHA..VEDIC SIGN ANUSVARA VAMAGOMUKHA WITH TAIL
    Interval{ 0x1CEE, 0x1CF3 }, // Lo   [6] VEDIC SIGN HEXIFORM LONG ANUSVARA..VEDIC SIGN ROTATED ARDHAVISARGA
    Interval{ 0x1CF5, 0x1CF6 }, // Lo   [2] VEDIC SIGN JIHVAMULIYA..VEDIC SIGN UPADHMANIYA
    Interval{ 0x1CF7, 0x1CF7 }, // Mc       VEDIC SIGN ATIKRAMA
    Interval{ 0x1CFA, 0x1CFA }, // Lo       VEDIC SIGN DOUBLE ANUSVARA ANTARGOMUKHA
    Interval{ 0x1D00, 0x1D2B }, // L&  [44] LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTER SMALL CAPITAL EL
    Interval{ 0x1D2C, 0x1D6A }, // Lm  [63] MODIFIER LETTER CAPITAL A..GREEK SUBSCRIPT SMALL LETTER CHI
    Interval{ 0x1D6B, 0x1D77 }, // L&  [13] LATIN SMALL LETTER UE..LATIN SMALL LETTER TURNED G
    Interval{ 0x1D78, 0x1D78 }, // Lm       MODIFIER LETTER CYRILLIC EN
    Interval{ 0x1D79, 0x1D9A }, // L&  [34] LATIN SMALL LETTER INSULAR G..LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
    Interval{ 0x1D9B, 0x1DBF }, // Lm  [37] MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER LETTER SMALL THETA
    Interval{ 0x1E00, 0x1F15 }, // L& [278] LATIN CAPITAL LETTER A WITH RING BELOW..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F45 }, // L&  [38] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F7D }, // L&  [31] GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1FB4 }, // L&  [53] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FBC }, // L&   [7] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FBD, 0x1FBD }, // Sk       GREEK KORONIS
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FBF, 0x1FC1 }, // Sk   [3] GREEK PSILI..GREEK DIALYTIKA AND PERISPOMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FCC }, // L&   [7] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FCD, 0x1FCF }, // Sk   [3] GREEK PSILI AND VARIA..GREEK PSILI AND PERISPOMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FDB }, // L&   [6] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FDD, 0x1FDF }, // Sk   [3] GREEK DASIA AND VARIA..GREEK DASIA AND PERISPOMENI
    Interval{ 0x1FE0, 0x1FEC }, // L&  [13] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FED, 0x1FEF }, // Sk   [3] GREEK DIALYTIKA AND VARIA..GREEK VARIA
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FFC }, // L&   [7] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x1FFD, 0x1FFE }, // Sk   [2] GREEK OXIA..GREEK DASIA
    Interval{ 0x2000, 0x200A }, // Zs  [11] EN QUAD..HAIR SPACE
    Interval{ 0x2010, 0x2015 }, // Pd   [6] HYPHEN..HORIZONTAL BAR
    Interval{ 0x2016, 0x2017 }, // Po   [2] DOUBLE VERTICAL LINE..DOUBLE LOW LINE
    Interval{ 0x2018, 0x2018 }, // Pi       LEFT SINGLE QUOTATION MARK
    Interval{ 0x2019, 0x2019 }, // Pf       RIGHT SINGLE QUOTATION MARK
    Interval{ 0x201A, 0x201A }, // Ps       SINGLE LOW-9 QUOTATION MARK
    Interval{ 0x201B, 0x201C }, // Pi   [2] SINGLE HIGH-REVERSED-9 QUOTATION MARK..LEFT DOUBLE QUOTATION MARK
    Interval{ 0x201D, 0x201D }, // Pf       RIGHT DOUBLE QUOTATION MARK
    Interval{ 0x201E, 0x201E }, // Ps       DOUBLE LOW-9 QUOTATION MARK
    Interval{ 0x201F, 0x201F }, // Pi       DOUBLE HIGH-REVERSED-9 QUOTATION MARK
    Interval{ 0x2020, 0x2027 }, // Po   [8] DAGGER..HYPHENATION POINT
    Interval{ 0x202F, 0x202F }, // Zs       NARROW NO-BREAK SPACE
    Interval{ 0x2030, 0x2038 }, // Po   [9] PER MILLE SIGN..CARET
    Interval{ 0x2039, 0x2039 }, // Pi       SINGLE LEFT-POINTING ANGLE QUOTATION MARK
    Interval{ 0x203A, 0x203A }, // Pf       SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
    Interval{ 0x203B, 0x203E }, // Po   [4] REFERENCE MARK..OVERLINE
    Interval{ 0x203F, 0x2040 }, // Pc   [2] UNDERTIE..CHARACTER TIE
    Interval{ 0x2041, 0x2043 }, // Po   [3] CARET INSERTION POINT..HYPHEN BULLET
    Interval{ 0x2044, 0x2044 }, // Sm       FRACTION SLASH
    Interval{ 0x2045, 0x2045 }, // Ps       LEFT SQUARE BRACKET WITH QUILL
    Interval{ 0x2046, 0x2046 }, // Pe       RIGHT SQUARE BRACKET WITH QUILL
    Interval{ 0x2047, 0x2051 }, // Po  [11] DOUBLE QUESTION MARK..TWO ASTERISKS ALIGNED VERTICALLY
    Interval{ 0x2052, 0x2052 }, // Sm       COMMERCIAL MINUS SIGN
    Interval{ 0x2053, 0x2053 }, // Po       SWUNG DASH
    Interval{ 0x2054, 0x2054 }, // Pc       INVERTED UNDERTIE
    Interval{ 0x2055, 0x205E }, // Po  [10] FLOWER PUNCTUATION MARK..VERTICAL FOUR DOTS
    Interval{ 0x205F, 0x205F }, // Zs       MEDIUM MATHEMATICAL SPACE
    Interval{ 0x2070, 0x2070 }, // No       SUPERSCRIPT ZERO
    Interval{ 0x2071, 0x2071 }, // Lm       SUPERSCRIPT LATIN SMALL LETTER I
    Interval{ 0x2074, 0x2079 }, // No   [6] SUPERSCRIPT FOUR..SUPERSCRIPT NINE
    Interval{ 0x207A, 0x207C }, // Sm   [3] SUPERSCRIPT PLUS SIGN..SUPERSCRIPT EQUALS SIGN
    Interval{ 0x207D, 0x207D }, // Ps       SUPERSCRIPT LEFT PARENTHESIS
    Interval{ 0x207E, 0x207E }, // Pe       SUPERSCRIPT RIGHT PARENTHESIS
    Interval{ 0x207F, 0x207F }, // Lm       SUPERSCRIPT LATIN SMALL LETTER N
    Interval{ 0x2080, 0x2089 }, // No  [10] SUBSCRIPT ZERO..SUBSCRIPT NINE
    Interval{ 0x208A, 0x208C }, // Sm   [3] SUBSCRIPT PLUS SIGN..SUBSCRIPT EQUALS SIGN
    Interval{ 0x208D, 0x208D }, // Ps       SUBSCRIPT LEFT PARENTHESIS
    Interval{ 0x208E, 0x208E }, // Pe       SUBSCRIPT RIGHT PARENTHESIS
    Interval{ 0x2090, 0x209C }, // Lm  [13] LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCRIPT SMALL LETTER T
    Interval{ 0x20A0, 0x20C0 }, // Sc  [33] EURO-CURRENCY SIGN..SOM SIGN
    Interval{ 0x2100, 0x2101 }, // So   [2] ACCOUNT OF..ADDRESSED TO THE SUBJECT
    Interval{ 0x2102, 0x2102 }, // L&       DOUBLE-STRUCK CAPITAL C
    Interval{ 0x2103, 0x2106 }, // So   [4] DEGREE CELSIUS..CADA UNA
    Interval{ 0x2107, 0x2107 }, // L&       EULER CONSTANT
    Interval{ 0x2108, 0x2109 }, // So   [2] SCRUPLE..DEGREE FAHRENHEIT
    Interval{ 0x210A, 0x2113 }, // L&  [10] SCRIPT SMALL G..SCRIPT SMALL L
    Interval{ 0x2114, 0x2114 }, // So       L B BAR SYMBOL
    Interval{ 0x2115, 0x2115 }, // L&       DOUBLE-STRUCK CAPITAL N
    Interval{ 0x2116, 0x2117 }, // So   [2] NUMERO SIGN..SOUND RECORDING COPYRIGHT
    Interval{ 0x2118, 0x2118 }, // Sm       SCRIPT CAPITAL P
    Interval{ 0x2119, 0x211D }, // L&   [5] DOUBLE-STRUCK CAPITAL P..DOUBLE-STRUCK CAPITAL R
    Interval{ 0x211E, 0x2123 }, // So   [6] PRESCRIPTION TAKE..VERSICLE
    Interval{ 0x2124, 0x2124 }, // L&       DOUBLE-STRUCK CAPITAL Z
    Interval{ 0x2125, 0x2125 }, // So       OUNCE SIGN
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x2127, 0x2127 }, // So       INVERTED OHM SIGN
    Interval{ 0x2128, 0x2128 }, // L&       BLACK-LETTER CAPITAL Z
    Interval{ 0x2129, 0x2129 }, // So       TURNED GREEK SMALL LETTER IOTA
    Interval{ 0x212A, 0x212D }, // L&   [4] KELVIN SIGN..BLACK-LETTER CAPITAL C
    Interval{ 0x212E, 0x212E }, // So       ESTIMATED SYMBOL
    Interval{ 0x212F, 0x2134 }, // L&   [6] SCRIPT SMALL E..SCRIPT SMALL O
    Interval{ 0x2135, 0x2138 }, // Lo   [4] ALEF SYMBOL..DALET SYMBOL
    Interval{ 0x2139, 0x2139 }, // L&       INFORMATION SOURCE
    Interval{ 0x213A, 0x213B }, // So   [2] ROTATED CAPITAL Q..FACSIMILE SIGN
    Interval{ 0x213C, 0x213F }, // L&   [4] DOUBLE-STRUCK SMALL PI..DOUBLE-STRUCK CAPITAL PI
    Interval{ 0x2140, 0x2144 }, // Sm   [5] DOUBLE-STRUCK N-ARY SUMMATION..TURNED SANS-SERIF CAPITAL Y
    Interval{ 0x2145, 0x2149 }, // L&   [5] DOUBLE-STRUCK ITALIC CAPITAL D..DOUBLE-STRUCK ITALIC SMALL J
    Interval{ 0x214A, 0x214A }, // So       PROPERTY LINE
    Interval{ 0x214B, 0x214B }, // Sm       TURNED AMPERSAND
    Interval{ 0x214C, 0x214D }, // So   [2] PER SIGN..AKTIESELSKAB
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x214F, 0x214F }, // So       SYMBOL FOR SAMARITAN SOURCE
    Interval{ 0x2150, 0x215F }, // No  [16] VULGAR FRACTION ONE SEVENTH..FRACTION NUMERATOR ONE
    Interval{ 0x2160, 0x2182 }, // Nl  [35] ROMAN NUMERAL ONE..ROMAN NUMERAL TEN THOUSAND
    Interval{ 0x2183, 0x2184 }, // L&   [2] ROMAN NUMERAL REVERSED ONE HUNDRED..LATIN SMALL LETTER REVERSED C
    Interval{ 0x2185, 0x2188 }, // Nl   [4] ROMAN NUMERAL SIX LATE FORM..ROMAN NUMERAL ONE HUNDRED THOUSAND
    Interval{ 0x2189, 0x2189 }, // No       VULGAR FRACTION ZERO THIRDS
    Interval{ 0x218A, 0x218B }, // So   [2] TURNED DIGIT TWO..TURNED DIGIT THREE
    Interval{ 0x2190, 0x2194 }, // Sm   [5] LEFTWARDS ARROW..LEFT RIGHT ARROW
    Interval{ 0x2195, 0x2199 }, // So   [5] UP DOWN ARROW..SOUTH WEST ARROW
    Interval{ 0x219A, 0x219B }, // Sm   [2] LEFTWARDS ARROW WITH STROKE..RIGHTWARDS ARROW WITH STROKE
    Interval{ 0x219C, 0x219F }, // So   [4] LEFTWARDS WAVE ARROW..UPWARDS TWO HEADED ARROW
    Interval{ 0x21A0, 0x21A0 }, // Sm       RIGHTWARDS TWO HEADED ARROW
    Interval{ 0x21A1, 0x21A2 }, // So   [2] DOWNWARDS TWO HEADED ARROW..LEFTWARDS ARROW WITH TAIL
    Interval{ 0x21A3, 0x21A3 }, // Sm       RIGHTWARDS ARROW WITH TAIL
    Interval{ 0x21A4, 0x21A5 }, // So   [2] LEFTWARDS ARROW FROM BAR..UPWARDS ARROW FROM BAR
    Interval{ 0x21A6, 0x21A6 }, // Sm       RIGHTWARDS ARROW FROM BAR
    Interval{ 0x21A7, 0x21AD }, // So   [7] DOWNWARDS ARROW FROM BAR..LEFT RIGHT WAVE ARROW
    Interval{ 0x21AE, 0x21AE }, // Sm       LEFT RIGHT ARROW WITH STROKE
    Interval{ 0x21AF, 0x21CD }, // So  [31] DOWNWARDS ZIGZAG ARROW..LEFTWARDS DOUBLE ARROW WITH STROKE
    Interval{ 0x21CE, 0x21CF }, // Sm   [2] LEFT RIGHT DOUBLE ARROW WITH STROKE..RIGHTWARDS DOUBLE ARROW WITH STROKE
    Interval{ 0x21D0, 0x21D1 }, // So   [2] LEFTWARDS DOUBLE ARROW..UPWARDS DOUBLE ARROW
    Interval{ 0x21D2, 0x21D2 }, // Sm       RIGHTWARDS DOUBLE ARROW
    Interval{ 0x21D3, 0x21D3 }, // So       DOWNWARDS DOUBLE ARROW
    Interval{ 0x21D4, 0x21D4 }, // Sm       LEFT RIGHT DOUBLE ARROW
    Interval{ 0x21D5, 0x21F3 }, // So  [31] UP DOWN DOUBLE ARROW..UP DOWN WHITE ARROW
    Interval{ 0x21F4, 0x22FF }, // Sm [268] RIGHT ARROW WITH SMALL CIRCLE..Z NOTATION BAG MEMBERSHIP
    Interval{ 0x2300, 0x2307 }, // So   [8] DIAMETER SIGN..WAVY LINE
    Interval{ 0x2308, 0x2308 }, // Ps       LEFT CEILING
    Interval{ 0x2309, 0x2309 }, // Pe       RIGHT CEILING
    Interval{ 0x230A, 0x230A }, // Ps       LEFT FLOOR
    Interval{ 0x230B, 0x230B }, // Pe       RIGHT FLOOR
    Interval{ 0x230C, 0x231F }, // So  [20] BOTTOM RIGHT CROP..BOTTOM RIGHT CORNER
    Interval{ 0x2320, 0x2321 }, // Sm   [2] TOP HALF INTEGRAL..BOTTOM HALF INTEGRAL
    Interval{ 0x2322, 0x2328 }, // So   [7] FROWN..KEYBOARD
    Interval{ 0x2329, 0x2329 }, // Ps       LEFT-POINTING ANGLE BRACKET
    Interval{ 0x232A, 0x232A }, // Pe       RIGHT-POINTING ANGLE BRACKET
    Interval{ 0x232B, 0x237B }, // So  [81] ERASE TO THE LEFT..NOT CHECK MARK
    Interval{ 0x237C, 0x237C }, // Sm       RIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROW
    Interval{ 0x237D, 0x239A }, // So  [30] SHOULDERED OPEN BOX..CLEAR SCREEN SYMBOL
    Interval{ 0x239B, 0x23B3 }, // Sm  [25] LEFT PARENTHESIS UPPER HOOK..SUMMATION BOTTOM
    Interval{ 0x23B4, 0x23DB }, // So  [40] TOP SQUARE BRACKET..FUSE
    Interval{ 0x23DC, 0x23E1 }, // Sm   [6] TOP PARENTHESIS..BOTTOM TORTOISE SHELL BRACKET
    Interval{ 0x23E2, 0x2426 }, // So  [69] WHITE TRAPEZIUM..SYMBOL FOR SUBSTITUTE FORM TWO
    Interval{ 0x2440, 0x244A }, // So  [11] OCR HOOK..OCR DOUBLE BACKSLASH
    Interval{ 0x2460, 0x249B }, // No  [60] CIRCLED DIGIT ONE..NUMBER TWENTY FULL STOP
    Interval{ 0x249C, 0x24E9 }, // So  [78] PARENTHESIZED LATIN SMALL LETTER A..CIRCLED LATIN SMALL LETTER Z
    Interval{ 0x24EA, 0x24FF }, // No  [22] CIRCLED DIGIT ZERO..NEGATIVE CIRCLED DIGIT ZERO
    Interval{ 0x2500, 0x25B6 }, // So [183] BOX DRAWINGS LIGHT HORIZONTAL..BLACK RIGHT-POINTING TRIANGLE
    Interval{ 0x25B7, 0x25B7 }, // Sm       WHITE RIGHT-POINTING TRIANGLE
    Interval{ 0x25B8, 0x25C0 }, // So   [9] BLACK RIGHT-POINTING SMALL TRIANGLE..BLACK LEFT-POINTING TRIANGLE
    Interval{ 0x25C1, 0x25C1 }, // Sm       WHITE LEFT-POINTING TRIANGLE
    Interval{ 0x25C2, 0x25F7 }, // So  [54] BLACK LEFT-POINTING SMALL TRIANGLE..WHITE CIRCLE WITH UPPER RIGHT QUADRANT
    Interval{ 0x25F8, 0x25FF }, // Sm   [8] UPPER LEFT TRIANGLE..LOWER RIGHT TRIANGLE
    Interval{ 0x2600, 0x266E }, // So [111] BLACK SUN WITH RAYS..MUSIC NATURAL SIGN
    Interval{ 0x266F, 0x266F }, // Sm       MUSIC SHARP SIGN
    Interval{ 0x2670, 0x2767 }, // So [248] WEST SYRIAC CROSS..ROTATED FLORAL HEART BULLET
    Interval{ 0x2768, 0x2768 }, // Ps       MEDIUM LEFT PARENTHESIS ORNAMENT
    Interval{ 0x2769, 0x2769 }, // Pe       MEDIUM RIGHT PARENTHESIS ORNAMENT
    Interval{ 0x276A, 0x276A }, // Ps       MEDIUM FLATTENED LEFT PARENTHESIS ORNAMENT
    Interval{ 0x276B, 0x276B }, // Pe       MEDIUM FLATTENED RIGHT PARENTHESIS ORNAMENT
    Interval{ 0x276C, 0x276C }, // Ps       MEDIUM LEFT-POINTING ANGLE BRACKET ORNAMENT
    Interval{ 0x276D, 0x276D }, // Pe       MEDIUM RIGHT-POINTING ANGLE BRACKET ORNAMENT
    Interval{ 0x276E, 0x276E }, // Ps       HEAVY LEFT-POINTING ANGLE QUOTATION MARK ORNAMENT
    Interval{ 0x276F, 0x276F }, // Pe       HEAVY RIGHT-POINTING ANGLE QUOTATION MARK ORNAMENT
    Interval{ 0x2770, 0x2770 }, // Ps       HEAVY LEFT-POINTING ANGLE BRACKET ORNAMENT
    Interval{ 0x2771, 0x2771 }, // Pe       HEAVY RIGHT-POINTING ANGLE BRACKET ORNAMENT
    Interval{ 0x2772, 0x2772 }, // Ps       LIGHT LEFT TORTOISE SHELL BRACKET ORNAMENT
    Interval{ 0x2773, 0x2773 }, // Pe       LIGHT RIGHT TORTOISE SHELL BRACKET ORNAMENT
    Interval{ 0x2774, 0x2774 }, // Ps       MEDIUM LEFT CURLY BRACKET ORNAMENT
    Interval{ 0x2775, 0x2775 }, // Pe       MEDIUM RIGHT CURLY BRACKET ORNAMENT
    Interval{ 0x2776, 0x2793 }, // No  [30] DINGBAT NEGATIVE CIRCLED DIGIT ONE..DINGBAT NEGATIVE CIRCLED SANS-SERIF NUMBER TEN
    Interval{ 0x2794, 0x27BF }, // So  [44] HEAVY WIDE-HEADED RIGHTWARDS ARROW..DOUBLE CURLY LOOP
    Interval{ 0x27C0, 0x27C4 }, // Sm   [5] THREE DIMENSIONAL ANGLE..OPEN SUPERSET
    Interval{ 0x27C5, 0x27C5 }, // Ps       LEFT S-SHAPED BAG DELIMITER
    Interval{ 0x27C6, 0x27C6 }, // Pe       RIGHT S-SHAPED BAG DELIMITER
    Interval{ 0x27C7, 0x27E5 }, // Sm  [31] OR WITH DOT INSIDE..WHITE SQUARE WITH RIGHTWARDS TICK
    Interval{ 0x27E6, 0x27E6 }, // Ps       MATHEMATICAL LEFT WHITE SQUARE BRACKET
    Interval{ 0x27E7, 0x27E7 }, // Pe       MATHEMATICAL RIGHT WHITE SQUARE BRACKET
    Interval{ 0x27E8, 0x27E8 }, // Ps       MATHEMATICAL LEFT ANGLE BRACKET
    Interval{ 0x27E9, 0x27E9 }, // Pe       MATHEMATICAL RIGHT ANGLE BRACKET
    Interval{ 0x27EA, 0x27EA }, // Ps       MATHEMATICAL LEFT DOUBLE ANGLE BRACKET
    Interval{ 0x27EB, 0x27EB }, // Pe       MATHEMATICAL RIGHT DOUBLE ANGLE BRACKET
    Interval{ 0x27EC, 0x27EC }, // Ps       MATHEMATICAL LEFT WHITE TORTOISE SHELL BRACKET
    Interval{ 0x27ED, 0x27ED }, // Pe       MATHEMATICAL RIGHT WHITE TORTOISE SHELL BRACKET
    Interval{ 0x27EE, 0x27EE }, // Ps       MATHEMATICAL LEFT FLATTENED PARENTHESIS
    Interval{ 0x27EF, 0x27EF }, // Pe       MATHEMATICAL RIGHT FLATTENED PARENTHESIS
    Interval{ 0x27F0, 0x27FF }, // Sm  [16] UPWARDS QUADRUPLE ARROW..LONG RIGHTWARDS SQUIGGLE ARROW
    Interval{ 0x2800, 0x28FF }, // So [256] BRAILLE PATTERN BLANK..BRAILLE PATTERN DOTS-12345678
    Interval{ 0x2900, 0x2982 }, // Sm [131] RIGHTWARDS TWO-HEADED ARROW WITH VERTICAL STROKE..Z NOTATION TYPE COLON
    Interval{ 0x2983, 0x2983 }, // Ps       LEFT WHITE CURLY BRACKET
    Interval{ 0x2984, 0x2984 }, // Pe       RIGHT WHITE CURLY BRACKET
    Interval{ 0x2985, 0x2985 }, // Ps       LEFT WHITE PARENTHESIS
    Interval{ 0x2986, 0x2986 }, // Pe       RIGHT WHITE PARENTHESIS
    Interval{ 0x2987, 0x2987 }, // Ps       Z NOTATION LEFT IMAGE BRACKET
    Interval{ 0x2988, 0x2988 }, // Pe       Z NOTATION RIGHT IMAGE BRACKET
    Interval{ 0x2989, 0x2989 }, // Ps       Z NOTATION LEFT BINDING BRACKET
    Interval{ 0x298A, 0x298A }, // Pe       Z NOTATION RIGHT BINDING BRACKET
    Interval{ 0x298B, 0x298B }, // Ps       LEFT SQUARE BRACKET WITH UNDERBAR
    Interval{ 0x298C, 0x298C }, // Pe       RIGHT SQUARE BRACKET WITH UNDERBAR
    Interval{ 0x298D, 0x298D }, // Ps       LEFT SQUARE BRACKET WITH TICK IN TOP CORNER
    Interval{ 0x298E, 0x298E }, // Pe       RIGHT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
    Interval{ 0x298F, 0x298F }, // Ps       LEFT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
    Interval{ 0x2990, 0x2990 }, // Pe       RIGHT SQUARE BRACKET WITH TICK IN TOP CORNER
    Interval{ 0x2991, 0x2991 }, // Ps       LEFT ANGLE BRACKET WITH DOT
    Interval{ 0x2992, 0x2992 }, // Pe       RIGHT ANGLE BRACKET WITH DOT
    Interval{ 0x2993, 0x2993 }, // Ps       LEFT ARC LESS-THAN BRACKET
    Interval{ 0x2994, 0x2994 }, // Pe       RIGHT ARC GREATER-THAN BRACKET
    Interval{ 0x2995, 0x2995 }, // Ps       DOUBLE LEFT ARC GREATER-THAN BRACKET
    Interval{ 0x2996, 0x2996 }, // Pe       DOUBLE RIGHT ARC LESS-THAN BRACKET
    Interval{ 0x2997, 0x2997 }, // Ps       LEFT BLACK TORTOISE SHELL BRACKET
    Interval{ 0x2998, 0x2998 }, // Pe       RIGHT BLACK TORTOISE SHELL BRACKET
    Interval{ 0x2999, 0x29D7 }, // Sm  [63] DOTTED FENCE..BLACK HOURGLASS
    Interval{ 0x29D8, 0x29D8 }, // Ps       LEFT WIGGLY FENCE
    Interval{ 0x29D9, 0x29D9 }, // Pe       RIGHT WIGGLY FENCE
    Interval{ 0x29DA, 0x29DA }, // Ps       LEFT DOUBLE WIGGLY FENCE
    Interval{ 0x29DB, 0x29DB }, // Pe       RIGHT DOUBLE WIGGLY FENCE
    Interval{ 0x29DC, 0x29FB }, // Sm  [32] INCOMPLETE INFINITY..TRIPLE PLUS
    Interval{ 0x29FC, 0x29FC }, // Ps       LEFT-POINTING CURVED ANGLE BRACKET
    Interval{ 0x29FD, 0x29FD }, // Pe       RIGHT-POINTING CURVED ANGLE BRACKET
    Interval{ 0x29FE, 0x2AFF }, // Sm [258] TINY..N-ARY WHITE VERTICAL BAR
    Interval{ 0x2B00, 0x2B2F }, // So  [48] NORTH EAST WHITE ARROW..WHITE VERTICAL ELLIPSE
    Interval{ 0x2B30, 0x2B44 }, // Sm  [21] LEFT ARROW WITH SMALL CIRCLE..RIGHTWARDS ARROW THROUGH SUPERSET
    Interval{ 0x2B45, 0x2B46 }, // So   [2] LEFTWARDS QUADRUPLE ARROW..RIGHTWARDS QUADRUPLE ARROW
    Interval{ 0x2B47, 0x2B4C }, // Sm   [6] REVERSE TILDE OPERATOR ABOVE RIGHTWARDS ARROW..RIGHTWARDS ARROW ABOVE REVERSE TILDE OPERATOR
    Interval{ 0x2B4D, 0x2B73 }, // So  [39] DOWNWARDS TRIANGLE-HEADED ZIGZAG ARROW..DOWNWARDS TRIANGLE-HEADED ARROW TO BAR
    Interval{ 0x2B76, 0x2B95 }, // So  [32] NORTH WEST TRIANGLE-HEADED ARROW TO BAR..RIGHTWARDS BLACK ARROW
    Interval{ 0x2B97, 0x2BFF }, // So [105] SYMBOL FOR TYPE A ELECTRONICS..HELLSCHREIBER PAUSE SYMBOL
    Interval{ 0x2C00, 0x2C7B }, // L& [124] GLAGOLITIC CAPITAL LETTER AZU..LATIN LETTER SMALL CAPITAL TURNED E
    Interval{ 0x2C7C, 0x2C7D }, // Lm   [2] LATIN SUBSCRIPT SMALL LETTER J..MODIFIER LETTER CAPITAL V
    Interval{ 0x2C7E, 0x2CE4 }, // L& [103] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC SYMBOL KAI
    Interval{ 0x2CE5, 0x2CEA }, // So   [6] COPTIC SYMBOL MI RO..COPTIC SYMBOL SHIMA SIMA
    Interval{ 0x2CEB, 0x2CEE }, // L&   [4] COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI..COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF2, 0x2CF3 }, // L&   [2] COPTIC CAPITAL LETTER BOHAIRIC KHEI..COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2CF9, 0x2CFC }, // Po   [4] COPTIC OLD NUBIAN FULL STOP..COPTIC OLD NUBIAN VERSE DIVIDER
    Interval{ 0x2CFD, 0x2CFD }, // No       COPTIC FRACTION ONE HALF
    Interval{ 0x2CFE, 0x2CFF }, // Po   [2] COPTIC FULL STOP..COPTIC MORPHOLOGICAL DIVIDER
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0x2D30, 0x2D67 }, // Lo  [56] TIFINAGH LETTER YA..TIFINAGH LETTER YO
    Interval{ 0x2D6F, 0x2D6F }, // Lm       TIFINAGH MODIFIER LETTER LABIALIZATION MARK
    Interval{ 0x2D70, 0x2D70 }, // Po       TIFINAGH SEPARATOR MARK
    Interval{ 0x2D80, 0x2D96 }, // Lo  [23] ETHIOPIC SYLLABLE LOA..ETHIOPIC SYLLABLE GGWE
    Interval{ 0x2DA0, 0x2DA6 }, // Lo   [7] ETHIOPIC SYLLABLE SSA..ETHIOPIC SYLLABLE SSO
    Interval{ 0x2DA8, 0x2DAE }, // Lo   [7] ETHIOPIC SYLLABLE CCA..ETHIOPIC SYLLABLE CCO
    Interval{ 0x2DB0, 0x2DB6 }, // Lo   [7] ETHIOPIC SYLLABLE ZZA..ETHIOPIC SYLLABLE ZZO
    Interval{ 0x2DB8, 0x2DBE }, // Lo   [7] ETHIOPIC SYLLABLE CCHA..ETHIOPIC SYLLABLE CCHO
    Interval{ 0x2DC0, 0x2DC6 }, // Lo   [7] ETHIOPIC SYLLABLE QYA..ETHIOPIC SYLLABLE QYO
    Interval{ 0x2DC8, 0x2DCE }, // Lo   [7] ETHIOPIC SYLLABLE KYA..ETHIOPIC SYLLABLE KYO
    Interval{ 0x2DD0, 0x2DD6 }, // Lo   [7] ETHIOPIC SYLLABLE XYA..ETHIOPIC SYLLABLE XYO
    Interval{ 0x2DD8, 0x2DDE }, // Lo   [7] ETHIOPIC SYLLABLE GYA..ETHIOPIC SYLLABLE GYO
    Interval{ 0x2E00, 0x2E01 }, // Po   [2] RIGHT ANGLE SUBSTITUTION MARKER..RIGHT ANGLE DOTTED SUBSTITUTION MARKER
    Interval{ 0x2E02, 0x2E02 }, // Pi       LEFT SUBSTITUTION BRACKET
    Interval{ 0x2E03, 0x2E03 }, // Pf       RIGHT SUBSTITUTION BRACKET
    Interval{ 0x2E04, 0x2E04 }, // Pi       LEFT DOTTED SUBSTITUTION BRACKET
    Interval{ 0x2E05, 0x2E05 }, // Pf       RIGHT DOTTED SUBSTITUTION BRACKET
    Interval{ 0x2E06, 0x2E08 }, // Po   [3] RAISED INTERPOLATION MARKER..DOTTED TRANSPOSITION MARKER
    Interval{ 0x2E09, 0x2E09 }, // Pi       LEFT TRANSPOSITION BRACKET
    Interval{ 0x2E0A, 0x2E0A }, // Pf       RIGHT TRANSPOSITION BRACKET
    Interval{ 0x2E0B, 0x2E0B }, // Po       RAISED SQUARE
    Interval{ 0x2E0C, 0x2E0C }, // Pi       LEFT RAISED OMISSION BRACKET
    Interval{ 0x2E0D, 0x2E0D }, // Pf       RIGHT RAISED OMISSION BRACKET
    Interval{ 0x2E0E, 0x2E16 }, // Po   [9] EDITORIAL CORONIS..DOTTED RIGHT-POINTING ANGLE
    Interval{ 0x2E17, 0x2E17 }, // Pd       DOUBLE OBLIQUE HYPHEN
    Interval{ 0x2E18, 0x2E19 }, // Po   [2] INVERTED INTERROBANG..PALM BRANCH
    Interval{ 0x2E1A, 0x2E1A }, // Pd       HYPHEN WITH DIAERESIS
    Interval{ 0x2E1B, 0x2E1B }, // Po       TILDE WITH RING ABOVE
    Interval{ 0x2E1C, 0x2E1C }, // Pi       LEFT LOW PARAPHRASE BRACKET
    Interval{ 0x2E1D, 0x2E1D }, // Pf       RIGHT LOW PARAPHRASE BRACKET
    Interval{ 0x2E1E, 0x2E1F }, // Po   [2] TILDE WITH DOT ABOVE..TILDE WITH DOT BELOW
    Interval{ 0x2E20, 0x2E20 }, // Pi       LEFT VERTICAL BAR WITH QUILL
    Interval{ 0x2E21, 0x2E21 }, // Pf       RIGHT VERTICAL BAR WITH QUILL
    Interval{ 0x2E22, 0x2E22 }, // Ps       TOP LEFT HALF BRACKET
    Interval{ 0x2E23, 0x2E23 }, // Pe       TOP RIGHT HALF BRACKET
    Interval{ 0x2E24, 0x2E24 }, // Ps       BOTTOM LEFT HALF BRACKET
    Interval{ 0x2E25, 0x2E25 }, // Pe       BOTTOM RIGHT HALF BRACKET
    Interval{ 0x2E26, 0x2E26 }, // Ps       LEFT SIDEWAYS U BRACKET
    Interval{ 0x2E27, 0x2E27 }, // Pe       RIGHT SIDEWAYS U BRACKET
    Interval{ 0x2E28, 0x2E28 }, // Ps       LEFT DOUBLE PARENTHESIS
    Interval{ 0x2E29, 0x2E29 }, // Pe       RIGHT DOUBLE PARENTHESIS
    Interval{ 0x2E2A, 0x2E2E }, // Po   [5] TWO DOTS OVER ONE DOT PUNCTUATION..REVERSED QUESTION MARK
    Interval{ 0x2E2F, 0x2E2F }, // Lm       VERTICAL TILDE
    Interval{ 0x2E30, 0x2E39 }, // Po  [10] RING POINT..TOP HALF SECTION SIGN
    Interval{ 0x2E3A, 0x2E3B }, // Pd   [2] TWO-EM DASH..THREE-EM DASH
    Interval{ 0x2E3C, 0x2E3F }, // Po   [4] STENOGRAPHIC FULL STOP..CAPITULUM
    Interval{ 0x2E40, 0x2E40 }, // Pd       DOUBLE HYPHEN
    Interval{ 0x2E41, 0x2E41 }, // Po       REVERSED COMMA
    Interval{ 0x2E42, 0x2E42 }, // Ps       DOUBLE LOW-REVERSED-9 QUOTATION MARK
    Interval{ 0x2E43, 0x2E4F }, // Po  [13] DASH WITH LEFT UPTURN..CORNISH VERSE DIVIDER
    Interval{ 0x2E50, 0x2E51 }, // So   [2] CROSS PATTY WITH RIGHT CROSSBAR..CROSS PATTY WITH LEFT CROSSBAR
    Interval{ 0x2E52, 0x2E54 }, // Po   [3] TIRONIAN SIGN CAPITAL ET..MEDIEVAL QUESTION MARK
    Interval{ 0x2E55, 0x2E55 }, // Ps       LEFT SQUARE BRACKET WITH STROKE
    Interval{ 0x2E56, 0x2E56 }, // Pe       RIGHT SQUARE BRACKET WITH STROKE
    Interval{ 0x2E57, 0x2E57 }, // Ps       LEFT SQUARE BRACKET WITH DOUBLE STROKE
    Interval{ 0x2E58, 0x2E58 }, // Pe       RIGHT SQUARE BRACKET WITH DOUBLE STROKE
    Interval{ 0x2E59, 0x2E59 }, // Ps       TOP HALF LEFT PARENTHESIS
    Interval{ 0x2E5A, 0x2E5A }, // Pe       TOP HALF RIGHT PARENTHESIS
    Interval{ 0x2E5B, 0x2E5B }, // Ps       BOTTOM HALF LEFT PARENTHESIS
    Interval{ 0x2E5C, 0x2E5C }, // Pe       BOTTOM HALF RIGHT PARENTHESIS
    Interval{ 0x2E5D, 0x2E5D }, // Pd       OBLIQUE HYPHEN
    Interval{ 0x2E80, 0x2E99 }, // So  [26] CJK RADICAL REPEAT..CJK RADICAL RAP
    Interval{ 0x2E9B, 0x2EF3 }, // So  [89] CJK RADICAL CHOKE..CJK RADICAL C-SIMPLIFIED TURTLE
    Interval{ 0x2F00, 0x2FD5 }, // So [214] KANGXI RADICAL ONE..KANGXI RADICAL FLUTE
    Interval{ 0x2FF0, 0x2FFB }, // So  [12] IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO RIGHT..IDEOGRAPHIC DESCRIPTION CHARACTER OVERLAID
    Interval{ 0x3000, 0x3000 }, // Zs       IDEOGRAPHIC SPACE
    Interval{ 0x3001, 0x3003 }, // Po   [3] IDEOGRAPHIC COMMA..DITTO MARK
    Interval{ 0x3004, 0x3004 }, // So       JAPANESE INDUSTRIAL STANDARD SYMBOL
    Interval{ 0x3005, 0x3005 }, // Lm       IDEOGRAPHIC ITERATION MARK
    Interval{ 0x3006, 0x3006 }, // Lo       IDEOGRAPHIC CLOSING MARK
    Interval{ 0x3007, 0x3007 }, // Nl       IDEOGRAPHIC NUMBER ZERO
    Interval{ 0x3008, 0x3008 }, // Ps       LEFT ANGLE BRACKET
    Interval{ 0x3009, 0x3009 }, // Pe       RIGHT ANGLE BRACKET
    Interval{ 0x300A, 0x300A }, // Ps       LEFT DOUBLE ANGLE BRACKET
    Interval{ 0x300B, 0x300B }, // Pe       RIGHT DOUBLE ANGLE BRACKET
    Interval{ 0x300C, 0x300C }, // Ps       LEFT CORNER BRACKET
    Interval{ 0x300D, 0x300D }, // Pe       RIGHT CORNER BRACKET
    Interval{ 0x300E, 0x300E }, // Ps       LEFT WHITE CORNER BRACKET
    Interval{ 0x300F, 0x300F }, // Pe       RIGHT WHITE CORNER BRACKET
    Interval{ 0x3010, 0x3010 }, // Ps       LEFT BLACK LENTICULAR BRACKET
    Interval{ 0x3011, 0x3011 }, // Pe       RIGHT BLACK LENTICULAR BRACKET
    Interval{ 0x3012, 0x3013 }, // So   [2] POSTAL MARK..GETA MARK
    Interval{ 0x3014, 0x3014 }, // Ps       LEFT TORTOISE SHELL BRACKET
    Interval{ 0x3015, 0x3015 }, // Pe       RIGHT TORTOISE SHELL BRACKET
    Interval{ 0x3016, 0x3016 }, // Ps       LEFT WHITE LENTICULAR BRACKET
    Interval{ 0x3017, 0x3017 }, // Pe       RIGHT WHITE LENTICULAR BRACKET
    Interval{ 0x3018, 0x3018 }, // Ps       LEFT WHITE TORTOISE SHELL BRACKET
    Interval{ 0x3019, 0x3019 }, // Pe       RIGHT WHITE TORTOISE SHELL BRACKET
    Interval{ 0x301A, 0x301A }, // Ps       LEFT WHITE SQUARE BRACKET
    Interval{ 0x301B, 0x301B }, // Pe       RIGHT WHITE SQUARE BRACKET
    Interval{ 0x301C, 0x301C }, // Pd       WAVE DASH
    Interval{ 0x301D, 0x301D }, // Ps       REVERSED DOUBLE PRIME QUOTATION MARK
    Interval{ 0x301E, 0x301F }, // Pe   [2] DOUBLE PRIME QUOTATION MARK..LOW DOUBLE PRIME QUOTATION MARK
    Interval{ 0x3020, 0x3020 }, // So       POSTAL MARK FACE
    Interval{ 0x3021, 0x3029 }, // Nl   [9] HANGZHOU NUMERAL ONE..HANGZHOU NUMERAL NINE
    Interval{ 0x3030, 0x3030 }, // Pd       WAVY DASH
    Interval{ 0x3031, 0x3035 }, // Lm   [5] VERTICAL KANA REPEAT MARK..VERTICAL KANA REPEAT MARK LOWER HALF
    Interval{ 0x3036, 0x3037 }, // So   [2] CIRCLED POSTAL MARK..IDEOGRAPHIC TELEGRAPH LINE FEED SEPARATOR SYMBOL
    Interval{ 0x3038, 0x303A }, // Nl   [3] HANGZHOU NUMERAL TEN..HANGZHOU NUMERAL THIRTY
    Interval{ 0x303B, 0x303B }, // Lm       VERTICAL IDEOGRAPHIC ITERATION MARK
    Interval{ 0x303C, 0x303C }, // Lo       MASU MARK
    Interval{ 0x303D, 0x303D }, // Po       PART ALTERNATION MARK
    Interval{ 0x303E, 0x303F }, // So   [2] IDEOGRAPHIC VARIATION INDICATOR..IDEOGRAPHIC HALF FILL SPACE
    Interval{ 0x3041, 0x3096 }, // Lo  [86] HIRAGANA LETTER SMALL A..HIRAGANA LETTER SMALL KE
    Interval{ 0x309B, 0x309C }, // Sk   [2] KATAKANA-HIRAGANA VOICED SOUND MARK..KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    Interval{ 0x309D, 0x309E }, // Lm   [2] HIRAGANA ITERATION MARK..HIRAGANA VOICED ITERATION MARK
    Interval{ 0x309F, 0x309F }, // Lo       HIRAGANA DIGRAPH YORI
    Interval{ 0x30A0, 0x30A0 }, // Pd       KATAKANA-HIRAGANA DOUBLE HYPHEN
    Interval{ 0x30A1, 0x30FA }, // Lo  [90] KATAKANA LETTER SMALL A..KATAKANA LETTER VO
    Interval{ 0x30FB, 0x30FB }, // Po       KATAKANA MIDDLE DOT
    Interval{ 0x30FC, 0x30FE }, // Lm   [3] KATAKANA-HIRAGANA PROLONGED SOUND MARK..KATAKANA VOICED ITERATION MARK
    Interval{ 0x30FF, 0x30FF }, // Lo       KATAKANA DIGRAPH KOTO
    Interval{ 0x3105, 0x312F }, // Lo  [43] BOPOMOFO LETTER B..BOPOMOFO LETTER NN
    Interval{ 0x3131, 0x318E }, // Lo  [94] HANGUL LETTER KIYEOK..HANGUL LETTER ARAEAE
    Interval{ 0x3190, 0x3191 }, // So   [2] IDEOGRAPHIC ANNOTATION LINKING MARK..IDEOGRAPHIC ANNOTATION REVERSE MARK
    Interval{ 0x3192, 0x3195 }, // No   [4] IDEOGRAPHIC ANNOTATION ONE MARK..IDEOGRAPHIC ANNOTATION FOUR MARK
    Interval{ 0x3196, 0x319F }, // So  [10] IDEOGRAPHIC ANNOTATION TOP MARK..IDEOGRAPHIC ANNOTATION MAN MARK
    Interval{ 0x31A0, 0x31BF }, // Lo  [32] BOPOMOFO LETTER BU..BOPOMOFO LETTER AH
    Interval{ 0x31C0, 0x31E3 }, // So  [36] CJK STROKE T..CJK STROKE Q
    Interval{ 0x31F0, 0x31FF }, // Lo  [16] KATAKANA LETTER SMALL KU..KATAKANA LETTER SMALL RO
    Interval{ 0x3200, 0x321E }, // So  [31] PARENTHESIZED HANGUL KIYEOK..PARENTHESIZED KOREAN CHARACTER O HU
    Interval{ 0x3220, 0x3229 }, // No  [10] PARENTHESIZED IDEOGRAPH ONE..PARENTHESIZED IDEOGRAPH TEN
    Interval{ 0x322A, 0x3247 }, // So  [30] PARENTHESIZED IDEOGRAPH MOON..CIRCLED IDEOGRAPH KOTO
    Interval{ 0x3248, 0x324F }, // No   [8] CIRCLED NUMBER TEN ON BLACK SQUARE..CIRCLED NUMBER EIGHTY ON BLACK SQUARE
    Interval{ 0x3250, 0x3250 }, // So       PARTNERSHIP SIGN
    Interval{ 0x3251, 0x325F }, // No  [15] CIRCLED NUMBER TWENTY ONE..CIRCLED NUMBER THIRTY FIVE
    Interval{ 0x3260, 0x327F }, // So  [32] CIRCLED HANGUL KIYEOK..KOREAN STANDARD SYMBOL
    Interval{ 0x3280, 0x3289 }, // No  [10] CIRCLED IDEOGRAPH ONE..CIRCLED IDEOGRAPH TEN
    Interval{ 0x328A, 0x32B0 }, // So  [39] CIRCLED IDEOGRAPH MOON..CIRCLED IDEOGRAPH NIGHT
    Interval{ 0x32B1, 0x32BF }, // No  [15] CIRCLED NUMBER THIRTY SIX..CIRCLED NUMBER FIFTY
    Interval{ 0x32C0, 0x33FF }, // So [320] IDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARY..SQUARE GAL
    Interval{ 0x3400, 0x4DBF }, // Lo [6592] CJK UNIFIED IDEOGRAPH-3400..CJK UNIFIED IDEOGRAPH-4DBF
    Interval{ 0x4DC0, 0x4DFF }, // So  [64] HEXAGRAM FOR THE CREATIVE HEAVEN..HEXAGRAM FOR BEFORE COMPLETION
    Interval{ 0x4E00, 0xA014 }, // Lo [21013] CJK UNIFIED IDEOGRAPH-4E00..YI SYLLABLE E
    Interval{ 0xA015, 0xA015 }, // Lm       YI SYLLABLE WU
    Interval{ 0xA016, 0xA48C }, // Lo [1143] YI SYLLABLE BIT..YI SYLLABLE YYR
    Interval{ 0xA490, 0xA4C6 }, // So  [55] YI RADICAL QOT..YI RADICAL KE
    Interval{ 0xA4D0, 0xA4F7 }, // Lo  [40] LISU LETTER BA..LISU LETTER OE
    Interval{ 0xA4F8, 0xA4FD }, // Lm   [6] LISU LETTER TONE MYA TI..LISU LETTER TONE MYA JEU
    Interval{ 0xA4FE, 0xA4FF }, // Po   [2] LISU PUNCTUATION COMMA..LISU PUNCTUATION FULL STOP
    Interval{ 0xA500, 0xA60B }, // Lo [268] VAI SYLLABLE EE..VAI SYLLABLE NG
    Interval{ 0xA60C, 0xA60C }, // Lm       VAI SYLLABLE LENGTHENER
    Interval{ 0xA60D, 0xA60F }, // Po   [3] VAI COMMA..VAI QUESTION MARK
    Interval{ 0xA610, 0xA61F }, // Lo  [16] VAI SYLLABLE NDOLE FA..VAI SYMBOL JONG
    Interval{ 0xA620, 0xA629 }, // Nd  [10] VAI DIGIT ZERO..VAI DIGIT NINE
    Interval{ 0xA62A, 0xA62B }, // Lo   [2] VAI SYLLABLE NDOLE MA..VAI SYLLABLE NDOLE DO
    Interval{ 0xA640, 0xA66D }, // L&  [46] CYRILLIC CAPITAL LETTER ZEMLYA..CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA66E, 0xA66E }, // Lo       CYRILLIC LETTER MULTIOCULAR O
    Interval{ 0xA673, 0xA673 }, // Po       SLAVONIC ASTERISK
    Interval{ 0xA67E, 0xA67E }, // Po       CYRILLIC KAVYKA
    Interval{ 0xA67F, 0xA67F }, // Lm       CYRILLIC PAYEROK
    Interval{ 0xA680, 0xA69B }, // L&  [28] CYRILLIC CAPITAL LETTER DWE..CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA69C, 0xA69D }, // Lm   [2] MODIFIER LETTER CYRILLIC HARD SIGN..MODIFIER LETTER CYRILLIC SOFT SIGN
    Interval{ 0xA6A0, 0xA6E5 }, // Lo  [70] BAMUM LETTER A..BAMUM LETTER KI
    Interval{ 0xA6E6, 0xA6EF }, // Nl  [10] BAMUM LETTER MO..BAMUM LETTER KOGHOM
    Interval{ 0xA6F2, 0xA6F7 }, // Po   [6] BAMUM NJAEMLI..BAMUM QUESTION MARK
    Interval{ 0xA700, 0xA716 }, // Sk  [23] MODIFIER LETTER CHINESE TONE YIN PING..MODIFIER LETTER EXTRA-LOW LEFT-STEM TONE BAR
    Interval{ 0xA717, 0xA71F }, // Lm   [9] MODIFIER LETTER DOT VERTICAL BAR..MODIFIER LETTER LOW INVERTED EXCLAMATION MARK
    Interval{ 0xA720, 0xA721 }, // Sk   [2] MODIFIER LETTER STRESS AND HIGH TONE..MODIFIER LETTER STRESS AND LOW TONE
    Interval{ 0xA722, 0xA76F }, // L&  [78] LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF..LATIN SMALL LETTER CON
    Interval{ 0xA770, 0xA770 }, // Lm       MODIFIER LETTER US
    Interval{ 0xA771, 0xA787 }, // L&  [23] LATIN SMALL LETTER DUM..LATIN SMALL LETTER INSULAR T
    Interval{ 0xA788, 0xA788 }, // Lm       MODIFIER LETTER LOW CIRCUMFLEX ACCENT
    Interval{ 0xA789, 0xA78A }, // Sk   [2] MODIFIER LETTER COLON..MODIFIER LETTER SHORT EQUALS SIGN
    Interval{ 0xA78B, 0xA78E }, // L&   [4] LATIN CAPITAL LETTER SALTILLO..LATIN SMALL LETTER L WITH RETROFLEX HOOK AND BELT
    Interval{ 0xA78F, 0xA78F }, // Lo       LATIN LETTER SINOLOGICAL DOT
    Interval{ 0xA790, 0xA7CA }, // L&  [59] LATIN CAPITAL LETTER N WITH DESCENDER..LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D1 }, // L&   [2] LATIN CAPITAL LETTER CLOSED INSULAR G..LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D3, 0xA7D3 }, // L&       LATIN SMALL LETTER DOUBLE THORN
    Interval{ 0xA7D5, 0xA7D9 }, // L&   [5] LATIN SMALL LETTER DOUBLE WYNN..LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F2, 0xA7F4 }, // Lm   [3] MODIFIER LETTER CAPITAL C..MODIFIER LETTER CAPITAL Q
    Interval{ 0xA7F5, 0xA7F6 }, // L&   [2] LATIN CAPITAL LETTER REVERSED HALF H..LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xA7F7, 0xA7F7 }, // Lo       LATIN EPIGRAPHIC LETTER SIDEWAYS I
    Interval{ 0xA7F8, 0xA7F9 }, // Lm   [2] MODIFIER LETTER CAPITAL H WITH STROKE..MODIFIER LETTER SMALL LIGATURE OE
    Interval{ 0xA7FA, 0xA7FA }, // L&       LATIN LETTER SMALL CAPITAL TURNED M
    Interval{ 0xA7FB, 0xA801 }, // Lo   [7] LATIN EPIGRAPHIC LETTER REVERSED F..SYLOTI NAGRI LETTER I
    Interval{ 0xA803, 0xA805 }, // Lo   [3] SYLOTI NAGRI LETTER U..SYLOTI NAGRI LETTER O
    Interval{ 0xA807, 0xA80A }, // Lo   [4] SYLOTI NAGRI LETTER KO..SYLOTI NAGRI LETTER GHO
    Interval{ 0xA80C, 0xA822 }, // Lo  [23] SYLOTI NAGRI LETTER CO..SYLOTI NAGRI LETTER HO
    Interval{ 0xA823, 0xA824 }, // Mc   [2] SYLOTI NAGRI VOWEL SIGN A..SYLOTI NAGRI VOWEL SIGN I
    Interval{ 0xA827, 0xA827 }, // Mc       SYLOTI NAGRI VOWEL SIGN OO
    Interval{ 0xA828, 0xA82B }, // So   [4] SYLOTI NAGRI POETRY MARK-1..SYLOTI NAGRI POETRY MARK-4
    Interval{ 0xA830, 0xA835 }, // No   [6] NORTH INDIC FRACTION ONE QUARTER..NORTH INDIC FRACTION THREE SIXTEENTHS
    Interval{ 0xA836, 0xA837 }, // So   [2] NORTH INDIC QUARTER MARK..NORTH INDIC PLACEHOLDER MARK
    Interval{ 0xA838, 0xA838 }, // Sc       NORTH INDIC RUPEE MARK
    Interval{ 0xA839, 0xA839 }, // So       NORTH INDIC QUANTITY MARK
    Interval{ 0xA840, 0xA873 }, // Lo  [52] PHAGS-PA LETTER KA..PHAGS-PA LETTER CANDRABINDU
    Interval{ 0xA874, 0xA877 }, // Po   [4] PHAGS-PA SINGLE HEAD MARK..PHAGS-PA MARK DOUBLE SHAD
    Interval{ 0xA880, 0xA881 }, // Mc   [2] SAURASHTRA SIGN ANUSVARA..SAURASHTRA SIGN VISARGA
    Interval{ 0xA882, 0xA8B3 }, // Lo  [50] SAURASHTRA LETTER A..SAURASHTRA LETTER LLA
    Interval{ 0xA8B4, 0xA8C3 }, // Mc  [16] SAURASHTRA CONSONANT SIGN HAARU..SAURASHTRA VOWEL SIGN AU
    Interval{ 0xA8CE, 0xA8CF }, // Po   [2] SAURASHTRA DANDA..SAURASHTRA DOUBLE DANDA
    Interval{ 0xA8D0, 0xA8D9 }, // Nd  [10] SAURASHTRA DIGIT ZERO..SAURASHTRA DIGIT NINE
    Interval{ 0xA8F2, 0xA8F7 }, // Lo   [6] DEVANAGARI SIGN SPACING CANDRABINDU..DEVANAGARI SIGN CANDRABINDU AVAGRAHA
    Interval{ 0xA8F8, 0xA8FA }, // Po   [3] DEVANAGARI SIGN PUSHPIKA..DEVANAGARI CARET
    Interval{ 0xA8FB, 0xA8FB }, // Lo       DEVANAGARI HEADSTROKE
    Interval{ 0xA8FC, 0xA8FC }, // Po       DEVANAGARI SIGN SIDDHAM
    Interval{ 0xA8FD, 0xA8FE }, // Lo   [2] DEVANAGARI JAIN OM..DEVANAGARI LETTER AY
    Interval{ 0xA900, 0xA909 }, // Nd  [10] KAYAH LI DIGIT ZERO..KAYAH LI DIGIT NINE
    Interval{ 0xA90A, 0xA925 }, // Lo  [28] KAYAH LI LETTER KA..KAYAH LI LETTER OO
    Interval{ 0xA92E, 0xA92F }, // Po   [2] KAYAH LI SIGN CWI..KAYAH LI SIGN SHYA
    Interval{ 0xA930, 0xA946 }, // Lo  [23] REJANG LETTER KA..REJANG LETTER A
    Interval{ 0xA952, 0xA953 }, // Mc   [2] REJANG CONSONANT SIGN H..REJANG VIRAMA
    Interval{ 0xA95F, 0xA95F }, // Po       REJANG SECTION MARK
    Interval{ 0xA960, 0xA97C }, // Lo  [29] HANGUL CHOSEONG TIKEUT-MIEUM..HANGUL CHOSEONG SSANGYEORINHIEUH
    Interval{ 0xA983, 0xA983 }, // Mc       JAVANESE SIGN WIGNYAN
    Interval{ 0xA984, 0xA9B2 }, // Lo  [47] JAVANESE LETTER A..JAVANESE LETTER HA
    Interval{ 0xA9B4, 0xA9B5 }, // Mc   [2] JAVANESE VOWEL SIGN TARUNG..JAVANESE VOWEL SIGN TOLONG
    Interval{ 0xA9BA, 0xA9BB }, // Mc   [2] JAVANESE VOWEL SIGN TALING..JAVANESE VOWEL SIGN DIRGA MURE
    Interval{ 0xA9BE, 0xA9C0 }, // Mc   [3] JAVANESE CONSONANT SIGN PENGKAL..JAVANESE PANGKON
    Interval{ 0xA9C1, 0xA9CD }, // Po  [13] JAVANESE LEFT RERENGGAN..JAVANESE TURNED PADA PISELEH
    Interval{ 0xA9CF, 0xA9CF }, // Lm       JAVANESE PANGRANGKEP
    Interval{ 0xA9D0, 0xA9D9 }, // Nd  [10] JAVANESE DIGIT ZERO..JAVANESE DIGIT NINE
    Interval{ 0xA9DE, 0xA9DF }, // Po   [2] JAVANESE PADA TIRTA TUMETES..JAVANESE PADA ISEN-ISEN
    Interval{ 0xA9E0, 0xA9E4 }, // Lo   [5] MYANMAR LETTER SHAN GHA..MYANMAR LETTER SHAN BHA
    Interval{ 0xA9E6, 0xA9E6 }, // Lm       MYANMAR MODIFIER LETTER SHAN REDUPLICATION
    Interval{ 0xA9E7, 0xA9EF }, // Lo   [9] MYANMAR LETTER TAI LAING NYA..MYANMAR LETTER TAI LAING NNA
    Interval{ 0xA9F0, 0xA9F9 }, // Nd  [10] MYANMAR TAI LAING DIGIT ZERO..MYANMAR TAI LAING DIGIT NINE
    Interval{ 0xA9FA, 0xA9FE }, // Lo   [5] MYANMAR LETTER TAI LAING LLA..MYANMAR LETTER TAI LAING BHA
    Interval{ 0xAA00, 0xAA28 }, // Lo  [41] CHAM LETTER A..CHAM LETTER HA
    Interval{ 0xAA2F, 0xAA30 }, // Mc   [2] CHAM VOWEL SIGN O..CHAM VOWEL SIGN AI
    Interval{ 0xAA33, 0xAA34 }, // Mc   [2] CHAM CONSONANT SIGN YA..CHAM CONSONANT SIGN RA
    Interval{ 0xAA40, 0xAA42 }, // Lo   [3] CHAM LETTER FINAL K..CHAM LETTER FINAL NG
    Interval{ 0xAA44, 0xAA4B }, // Lo   [8] CHAM LETTER FINAL CH..CHAM LETTER FINAL SS
    Interval{ 0xAA4D, 0xAA4D }, // Mc       CHAM CONSONANT SIGN FINAL H
    Interval{ 0xAA50, 0xAA59 }, // Nd  [10] CHAM DIGIT ZERO..CHAM DIGIT NINE
    Interval{ 0xAA5C, 0xAA5F }, // Po   [4] CHAM PUNCTUATION SPIRAL..CHAM PUNCTUATION TRIPLE DANDA
    Interval{ 0xAA60, 0xAA6F }, // Lo  [16] MYANMAR LETTER KHAMTI GA..MYANMAR LETTER KHAMTI FA
    Interval{ 0xAA70, 0xAA70 }, // Lm       MYANMAR MODIFIER LETTER KHAMTI REDUPLICATION
    Interval{ 0xAA71, 0xAA76 }, // Lo   [6] MYANMAR LETTER KHAMTI XA..MYANMAR LOGOGRAM KHAMTI HM
    Interval{ 0xAA77, 0xAA79 }, // So   [3] MYANMAR SYMBOL AITON EXCLAMATION..MYANMAR SYMBOL AITON TWO
    Interval{ 0xAA7A, 0xAA7A }, // Lo       MYANMAR LETTER AITON RA
    Interval{ 0xAA7B, 0xAA7B }, // Mc       MYANMAR SIGN PAO KAREN TONE
    Interval{ 0xAA7D, 0xAA7D }, // Mc       MYANMAR SIGN TAI LAING TONE-5
    Interval{ 0xAA7E, 0xAAAF }, // Lo  [50] MYANMAR LETTER SHWE PALAUNG CHA..TAI VIET LETTER HIGH O
    Interval{ 0xAAB1, 0xAAB1 }, // Lo       TAI VIET VOWEL AA
    Interval{ 0xAAB5, 0xAAB6 }, // Lo   [2] TAI VIET VOWEL E..TAI VIET VOWEL O
    Interval{ 0xAAB9, 0xAABD }, // Lo   [5] TAI VIET VOWEL UEA..TAI VIET VOWEL AN
    Interval{ 0xAAC0, 0xAAC0 }, // Lo       TAI VIET TONE MAI NUENG
    Interval{ 0xAAC2, 0xAAC2 }, // Lo       TAI VIET TONE MAI SONG
    Interval{ 0xAADB, 0xAADC }, // Lo   [2] TAI VIET SYMBOL KON..TAI VIET SYMBOL NUENG
    Interval{ 0xAADD, 0xAADD }, // Lm       TAI VIET SYMBOL SAM
    Interval{ 0xAADE, 0xAADF }, // Po   [2] TAI VIET SYMBOL HO HOI..TAI VIET SYMBOL KOI KOI
    Interval{ 0xAAE0, 0xAAEA }, // Lo  [11] MEETEI MAYEK LETTER E..MEETEI MAYEK LETTER SSA
    Interval{ 0xAAEB, 0xAAEB }, // Mc       MEETEI MAYEK VOWEL SIGN II
    Interval{ 0xAAEE, 0xAAEF }, // Mc   [2] MEETEI MAYEK VOWEL SIGN AU..MEETEI MAYEK VOWEL SIGN AAU
    Interval{ 0xAAF0, 0xAAF1 }, // Po   [2] MEETEI MAYEK CHEIKHAN..MEETEI MAYEK AHANG KHUDAM
    Interval{ 0xAAF2, 0xAAF2 }, // Lo       MEETEI MAYEK ANJI
    Interval{ 0xAAF3, 0xAAF4 }, // Lm   [2] MEETEI MAYEK SYLLABLE REPETITION MARK..MEETEI MAYEK WORD REPETITION MARK
    Interval{ 0xAAF5, 0xAAF5 }, // Mc       MEETEI MAYEK VOWEL SIGN VISARGA
    Interval{ 0xAB01, 0xAB06 }, // Lo   [6] ETHIOPIC SYLLABLE TTHU..ETHIOPIC SYLLABLE TTHO
    Interval{ 0xAB09, 0xAB0E }, // Lo   [6] ETHIOPIC SYLLABLE DDHU..ETHIOPIC SYLLABLE DDHO
    Interval{ 0xAB11, 0xAB16 }, // Lo   [6] ETHIOPIC SYLLABLE DZU..ETHIOPIC SYLLABLE DZO
    Interval{ 0xAB20, 0xAB26 }, // Lo   [7] ETHIOPIC SYLLABLE CCHHA..ETHIOPIC SYLLABLE CCHHO
    Interval{ 0xAB28, 0xAB2E }, // Lo   [7] ETHIOPIC SYLLABLE BBA..ETHIOPIC SYLLABLE BBO
    Interval{ 0xAB30, 0xAB5A }, // L&  [43] LATIN SMALL LETTER BARRED ALPHA..LATIN SMALL LETTER Y WITH SHORT RIGHT LEG
    Interval{ 0xAB5B, 0xAB5B }, // Sk       MODIFIER BREVE WITH INVERTED BREVE
    Interval{ 0xAB5C, 0xAB5F }, // Lm   [4] MODIFIER LETTER SMALL HENG..MODIFIER LETTER SMALL U WITH LEFT HOOK
    Interval{ 0xAB60, 0xAB68 }, // L&   [9] LATIN SMALL LETTER SAKHA YAT..LATIN SMALL LETTER TURNED R WITH MIDDLE TILDE
    Interval{ 0xAB69, 0xAB69 }, // Lm       MODIFIER LETTER SMALL TURNED W
    Interval{ 0xAB6A, 0xAB6B }, // Sk   [2] MODIFIER LETTER LEFT TACK..MODIFIER LETTER RIGHT TACK
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xABC0, 0xABE2 }, // Lo  [35] MEETEI MAYEK LETTER KOK..MEETEI MAYEK LETTER I LONSUM
    Interval{ 0xABE3, 0xABE4 }, // Mc   [2] MEETEI MAYEK VOWEL SIGN ONAP..MEETEI MAYEK VOWEL SIGN INAP
    Interval{ 0xABE6, 0xABE7 }, // Mc   [2] MEETEI MAYEK VOWEL SIGN YENAP..MEETEI MAYEK VOWEL SIGN SOUNAP
    Interval{ 0xABE9, 0xABEA }, // Mc   [2] MEETEI MAYEK VOWEL SIGN CHEINAP..MEETEI MAYEK VOWEL SIGN NUNG
    Interval{ 0xABEB, 0xABEB }, // Po       MEETEI MAYEK CHEIKHEI
    Interval{ 0xABEC, 0xABEC }, // Mc       MEETEI MAYEK LUM IYEK
    Interval{ 0xABF0, 0xABF9 }, // Nd  [10] MEETEI MAYEK DIGIT ZERO..MEETEI MAYEK DIGIT NINE
    Interval{ 0xAC00, 0xD7A3 }, // Lo [11172] HANGUL SYLLABLE GA..HANGUL SYLLABLE HIH
    Interval{ 0xD7B0, 0xD7C6 }, // Lo  [23] HANGUL JUNGSEONG O-YEO..HANGUL JUNGSEONG ARAEA-E
    Interval{ 0xD7CB, 0xD7FB }, // Lo  [49] HANGUL JONGSEONG NIEUN-RIEUL..HANGUL JONGSEONG PHIEUPH-THIEUTH
    Interval{ 0xF900, 0xFA6D }, // Lo [366] CJK COMPATIBILITY IDEOGRAPH-F900..CJK COMPATIBILITY IDEOGRAPH-FA6D
    Interval{ 0xFA70, 0xFAD9 }, // Lo [106] CJK COMPATIBILITY IDEOGRAPH-FA70..CJK COMPATIBILITY IDEOGRAPH-FAD9
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFB1D, 0xFB1D }, // Lo       HEBREW LETTER YOD WITH HIRIQ
    Interval{ 0xFB1F, 0xFB28 }, // Lo  [10] HEBREW LIGATURE YIDDISH YOD YOD PATAH..HEBREW LETTER WIDE TAV
    Interval{ 0xFB29, 0xFB29 }, // Sm       HEBREW LETTER ALTERNATIVE PLUS SIGN
    Interval{ 0xFB2A, 0xFB36 }, // Lo  [13] HEBREW LETTER SHIN WITH SHIN DOT..HEBREW LETTER ZAYIN WITH DAGESH
    Interval{ 0xFB38, 0xFB3C }, // Lo   [5] HEBREW LETTER TET WITH DAGESH..HEBREW LETTER LAMED WITH DAGESH
    Interval{ 0xFB3E, 0xFB3E }, // Lo       HEBREW LETTER MEM WITH DAGESH
    Interval{ 0xFB40, 0xFB41 }, // Lo   [2] HEBREW LETTER NUN WITH DAGESH..HEBREW LETTER SAMEKH WITH DAGESH
    Interval{ 0xFB43, 0xFB44 }, // Lo   [2] HEBREW LETTER FINAL PE WITH DAGESH..HEBREW LETTER PE WITH DAGESH
    Interval{ 0xFB46, 0xFBB1 }, // Lo [108] HEBREW LETTER TSADI WITH DAGESH..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
    Interval{ 0xFBB2, 0xFBC2 }, // Sk  [17] ARABIC SYMBOL DOT ABOVE..ARABIC SYMBOL WASLA ABOVE
    Interval{ 0xFBD3, 0xFD3D }, // Lo [363] ARABIC LETTER NG ISOLATED FORM..ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
    Interval{ 0xFD3E, 0xFD3E }, // Pe       ORNATE LEFT PARENTHESIS
    Interval{ 0xFD3F, 0xFD3F }, // Ps       ORNATE RIGHT PARENTHESIS
    Interval{ 0xFD40, 0xFD4F }, // So  [16] ARABIC LIGATURE RAHIMAHU ALLAAH..ARABIC LIGATURE RAHIMAHUM ALLAAH
    Interval{ 0xFD50, 0xFD8F }, // Lo  [64] ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM..ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
    Interval{ 0xFD92, 0xFDC7 }, // Lo  [54] ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM..ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
    Interval{ 0xFDCF, 0xFDCF }, // So       ARABIC LIGATURE SALAAMUHU ALAYNAA
    Interval{ 0xFDF0, 0xFDFB }, // Lo  [12] ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM..ARABIC LIGATURE JALLAJALALOUHOU
    Interval{ 0xFDFC, 0xFDFC }, // Sc       RIAL SIGN
    Interval{ 0xFDFD, 0xFDFF }, // So   [3] ARABIC LIGATURE BISMILLAH AR-RAHMAN AR-RAHEEM..ARABIC LIGATURE AZZA WA JALL
    Interval{ 0xFE10, 0xFE16 }, // Po   [7] PRESENTATION FORM FOR VERTICAL COMMA..PRESENTATION FORM FOR VERTICAL QUESTION MARK
    Interval{ 0xFE17, 0xFE17 }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT WHITE LENTICULAR BRACKET
    Interval{ 0xFE18, 0xFE18 }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT WHITE LENTICULAR BRAKCET
    Interval{ 0xFE19, 0xFE19 }, // Po       PRESENTATION FORM FOR VERTICAL HORIZONTAL ELLIPSIS
    Interval{ 0xFE30, 0xFE30 }, // Po       PRESENTATION FORM FOR VERTICAL TWO DOT LEADER
    Interval{ 0xFE31, 0xFE32 }, // Pd   [2] PRESENTATION FORM FOR VERTICAL EM DASH..PRESENTATION FORM FOR VERTICAL EN DASH
    Interval{ 0xFE33, 0xFE34 }, // Pc   [2] PRESENTATION FORM FOR VERTICAL LOW LINE..PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
    Interval{ 0xFE35, 0xFE35 }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT PARENTHESIS
    Interval{ 0xFE36, 0xFE36 }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT PARENTHESIS
    Interval{ 0xFE37, 0xFE37 }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKET
    Interval{ 0xFE38, 0xFE38 }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKET
    Interval{ 0xFE39, 0xFE39 }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKET
    Interval{ 0xFE3A, 0xFE3A }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKET
    Interval{ 0xFE3B, 0xFE3B }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKET
    Interval{ 0xFE3C, 0xFE3C }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKET
    Interval{ 0xFE3D, 0xFE3D }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKET
    Interval{ 0xFE3E, 0xFE3E }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT DOUBLE ANGLE BRACKET
    Interval{ 0xFE3F, 0xFE3F }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKET
    Interval{ 0xFE40, 0xFE40 }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKET
    Interval{ 0xFE41, 0xFE41 }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKET
    Interval{ 0xFE42, 0xFE42 }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKET
    Interval{ 0xFE43, 0xFE43 }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT WHITE CORNER BRACKET
    Interval{ 0xFE44, 0xFE44 }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKET
    Interval{ 0xFE45, 0xFE46 }, // Po   [2] SESAME DOT..WHITE SESAME DOT
    Interval{ 0xFE47, 0xFE47 }, // Ps       PRESENTATION FORM FOR VERTICAL LEFT SQUARE BRACKET
    Interval{ 0xFE48, 0xFE48 }, // Pe       PRESENTATION FORM FOR VERTICAL RIGHT SQUARE BRACKET
    Interval{ 0xFE49, 0xFE4C }, // Po   [4] DASHED OVERLINE..DOUBLE WAVY OVERLINE
    Interval{ 0xFE4D, 0xFE4F }, // Pc   [3] DASHED LOW LINE..WAVY LOW LINE
    Interval{ 0xFE50, 0xFE52 }, // Po   [3] SMALL COMMA..SMALL FULL STOP
    Interval{ 0xFE54, 0xFE57 }, // Po   [4] SMALL SEMICOLON..SMALL EXCLAMATION MARK
    Interval{ 0xFE58, 0xFE58 }, // Pd       SMALL EM DASH
    Interval{ 0xFE59, 0xFE59 }, // Ps       SMALL LEFT PARENTHESIS
    Interval{ 0xFE5A, 0xFE5A }, // Pe       SMALL RIGHT PARENTHESIS
    Interval{ 0xFE5B, 0xFE5B }, // Ps       SMALL LEFT CURLY BRACKET
    Interval{ 0xFE5C, 0xFE5C }, // Pe       SMALL RIGHT CURLY BRACKET
    Interval{ 0xFE5D, 0xFE5D }, // Ps       SMALL LEFT TORTOISE SHELL BRACKET
    Interval{ 0xFE5E, 0xFE5E }, // Pe       SMALL RIGHT TORTOISE SHELL BRACKET
    Interval{ 0xFE5F, 0xFE61 }, // Po   [3] SMALL NUMBER SIGN..SMALL ASTERISK
    Interval{ 0xFE62, 0xFE62 }, // Sm       SMALL PLUS SIGN
    Interval{ 0xFE63, 0xFE63 }, // Pd       SMALL HYPHEN-MINUS
    Interval{ 0xFE64, 0xFE66 }, // Sm   [3] SMALL LESS-THAN SIGN..SMALL EQUALS SIGN
    Interval{ 0xFE68, 0xFE68 }, // Po       SMALL REVERSE SOLIDUS
    Interval{ 0xFE69, 0xFE69 }, // Sc       SMALL DOLLAR SIGN
    Interval{ 0xFE6A, 0xFE6B }, // Po   [2] SMALL PERCENT SIGN..SMALL COMMERCIAL AT
    Interval{ 0xFE70, 0xFE74 }, // Lo   [5] ARABIC FATHATAN ISOLATED FORM..ARABIC KASRATAN ISOLATED FORM
    Interval{ 0xFE76, 0xFEFC }, // Lo [135] ARABIC FATHA ISOLATED FORM..ARABIC LIGATURE LAM WITH ALEF FINAL FORM
    Interval{ 0xFF01, 0xFF03 }, // Po   [3] FULLWIDTH EXCLAMATION MARK..FULLWIDTH NUMBER SIGN
    Interval{ 0xFF04, 0xFF04 }, // Sc       FULLWIDTH DOLLAR SIGN
    Interval{ 0xFF05, 0xFF07 }, // Po   [3] FULLWIDTH PERCENT SIGN..FULLWIDTH APOSTROPHE
    Interval{ 0xFF08, 0xFF08 }, // Ps       FULLWIDTH LEFT PARENTHESIS
    Interval{ 0xFF09, 0xFF09 }, // Pe       FULLWIDTH RIGHT PARENTHESIS
    Interval{ 0xFF0A, 0xFF0A }, // Po       FULLWIDTH ASTERISK
    Interval{ 0xFF0B, 0xFF0B }, // Sm       FULLWIDTH PLUS SIGN
    Interval{ 0xFF0C, 0xFF0C }, // Po       FULLWIDTH COMMA
    Interval{ 0xFF0D, 0xFF0D }, // Pd       FULLWIDTH HYPHEN-MINUS
    Interval{ 0xFF0E, 0xFF0F }, // Po   [2] FULLWIDTH FULL STOP..FULLWIDTH SOLIDUS
    Interval{ 0xFF10, 0xFF19 }, // Nd  [10] FULLWIDTH DIGIT ZERO..FULLWIDTH DIGIT NINE
    Interval{ 0xFF1A, 0xFF1B }, // Po   [2] FULLWIDTH COLON..FULLWIDTH SEMICOLON
    Interval{ 0xFF1C, 0xFF1E }, // Sm   [3] FULLWIDTH LESS-THAN SIGN..FULLWIDTH GREATER-THAN SIGN
    Interval{ 0xFF1F, 0xFF20 }, // Po   [2] FULLWIDTH QUESTION MARK..FULLWIDTH COMMERCIAL AT
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0xFF3B, 0xFF3B }, // Ps       FULLWIDTH LEFT SQUARE BRACKET
    Interval{ 0xFF3C, 0xFF3C }, // Po       FULLWIDTH REVERSE SOLIDUS
    Interval{ 0xFF3D, 0xFF3D }, // Pe       FULLWIDTH RIGHT SQUARE BRACKET
    Interval{ 0xFF3E, 0xFF3E }, // Sk       FULLWIDTH CIRCUMFLEX ACCENT
    Interval{ 0xFF3F, 0xFF3F }, // Pc       FULLWIDTH LOW LINE
    Interval{ 0xFF40, 0xFF40 }, // Sk       FULLWIDTH GRAVE ACCENT
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0xFF5B, 0xFF5B }, // Ps       FULLWIDTH LEFT CURLY BRACKET
    Interval{ 0xFF5C, 0xFF5C }, // Sm       FULLWIDTH VERTICAL LINE
    Interval{ 0xFF5D, 0xFF5D }, // Pe       FULLWIDTH RIGHT CURLY BRACKET
    Interval{ 0xFF5E, 0xFF5E }, // Sm       FULLWIDTH TILDE
    Interval{ 0xFF5F, 0xFF5F }, // Ps       FULLWIDTH LEFT WHITE PARENTHESIS
    Interval{ 0xFF60, 0xFF60 }, // Pe       FULLWIDTH RIGHT WHITE PARENTHESIS
    Interval{ 0xFF61, 0xFF61 }, // Po       HALFWIDTH IDEOGRAPHIC FULL STOP
    Interval{ 0xFF62, 0xFF62 }, // Ps       HALFWIDTH LEFT CORNER BRACKET
    Interval{ 0xFF63, 0xFF63 }, // Pe       HALFWIDTH RIGHT CORNER BRACKET
    Interval{ 0xFF64, 0xFF65 }, // Po   [2] HALFWIDTH IDEOGRAPHIC COMMA..HALFWIDTH KATAKANA MIDDLE DOT
    Interval{ 0xFF66, 0xFF6F }, // Lo  [10] HALFWIDTH KATAKANA LETTER WO..HALFWIDTH KATAKANA LETTER SMALL TU
    Interval{ 0xFF70, 0xFF70 }, // Lm       HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
    Interval{ 0xFF71, 0xFF9D }, // Lo  [45] HALFWIDTH KATAKANA LETTER A..HALFWIDTH KATAKANA LETTER N
    Interval{ 0xFFA0, 0xFFBE }, // Lo  [31] HALFWIDTH HANGUL FILLER..HALFWIDTH HANGUL LETTER HIEUH
    Interval{ 0xFFC2, 0xFFC7 }, // Lo   [6] HALFWIDTH HANGUL LETTER A..HALFWIDTH HANGUL LETTER E
    Interval{ 0xFFCA, 0xFFCF }, // Lo   [6] HALFWIDTH HANGUL LETTER YEO..HALFWIDTH HANGUL LETTER OE
    Interval{ 0xFFD2, 0xFFD7 }, // Lo   [6] HALFWIDTH HANGUL LETTER YO..HALFWIDTH HANGUL LETTER YU
    Interval{ 0xFFDA, 0xFFDC }, // Lo   [3] HALFWIDTH HANGUL LETTER EU..HALFWIDTH HANGUL LETTER I
    Interval{ 0xFFE0, 0xFFE1 }, // Sc   [2] FULLWIDTH CENT SIGN..FULLWIDTH POUND SIGN
    Interval{ 0xFFE2, 0xFFE2 }, // Sm       FULLWIDTH NOT SIGN
    Interval{ 0xFFE3, 0xFFE3 }, // Sk       FULLWIDTH MACRON
    Interval{ 0xFFE4, 0xFFE4 }, // So       FULLWIDTH BROKEN BAR
    Interval{ 0xFFE5, 0xFFE6 }, // Sc   [2] FULLWIDTH YEN SIGN..FULLWIDTH WON SIGN
    Interval{ 0xFFE8, 0xFFE8 }, // So       HALFWIDTH FORMS LIGHT VERTICAL
    Interval{ 0xFFE9, 0xFFEC }, // Sm   [4] HALFWIDTH LEFTWARDS ARROW..HALFWIDTH DOWNWARDS ARROW
    Interval{ 0xFFED, 0xFFEE }, // So   [2] HALFWIDTH BLACK SQUARE..HALFWIDTH WHITE CIRCLE
    Interval{ 0xFFFC, 0xFFFD }, // So   [2] OBJECT REPLACEMENT CHARACTER..REPLACEMENT CHARACTER
    Interval{ 0x10000, 0x1000B }, // Lo  [12] LINEAR B SYLLABLE B008 A..LINEAR B SYLLABLE B046 JE
    Interval{ 0x1000D, 0x10026 }, // Lo  [26] LINEAR B SYLLABLE B036 JO..LINEAR B SYLLABLE B032 QO
    Interval{ 0x10028, 0x1003A }, // Lo  [19] LINEAR B SYLLABLE B060 RA..LINEAR B SYLLABLE B042 WO
    Interval{ 0x1003C, 0x1003D }, // Lo   [2] LINEAR B SYLLABLE B017 ZA..LINEAR B SYLLABLE B074 ZE
    Interval{ 0x1003F, 0x1004D }, // Lo  [15] LINEAR B SYLLABLE B020 ZO..LINEAR B SYLLABLE B091 TWO
    Interval{ 0x10050, 0x1005D }, // Lo  [14] LINEAR B SYMBOL B018..LINEAR B SYMBOL B089
    Interval{ 0x10080, 0x100FA }, // Lo [123] LINEAR B IDEOGRAM B100 MAN..LINEAR B IDEOGRAM VESSEL B305
    Interval{ 0x10100, 0x10102 }, // Po   [3] AEGEAN WORD SEPARATOR LINE..AEGEAN CHECK MARK
    Interval{ 0x10107, 0x10133 }, // No  [45] AEGEAN NUMBER ONE..AEGEAN NUMBER NINETY THOUSAND
    Interval{ 0x10137, 0x1013F }, // So   [9] AEGEAN WEIGHT BASE UNIT..AEGEAN MEASURE THIRD SUBUNIT
    Interval{ 0x10140, 0x10174 }, // Nl  [53] GREEK ACROPHONIC ATTIC ONE QUARTER..GREEK ACROPHONIC STRATIAN FIFTY MNAS
    Interval{ 0x10175, 0x10178 }, // No   [4] GREEK ONE HALF SIGN..GREEK THREE QUARTERS SIGN
    Interval{ 0x10179, 0x10189 }, // So  [17] GREEK YEAR SIGN..GREEK TRYBLION BASE SIGN
    Interval{ 0x1018A, 0x1018B }, // No   [2] GREEK ZERO SIGN..GREEK ONE QUARTER SIGN
    Interval{ 0x1018C, 0x1018E }, // So   [3] GREEK SINUSOID SIGN..NOMISMA SIGN
    Interval{ 0x10190, 0x1019C }, // So  [13] ROMAN SEXTANS SIGN..ASCIA SYMBOL
    Interval{ 0x101A0, 0x101A0 }, // So       GREEK SYMBOL TAU RHO
    Interval{ 0x101D0, 0x101FC }, // So  [45] PHAISTOS DISC SIGN PEDESTRIAN..PHAISTOS DISC SIGN WAVY BAND
    Interval{ 0x10280, 0x1029C }, // Lo  [29] LYCIAN LETTER A..LYCIAN LETTER X
    Interval{ 0x102A0, 0x102D0 }, // Lo  [49] CARIAN LETTER A..CARIAN LETTER UUU3
    Interval{ 0x102E1, 0x102FB }, // No  [27] COPTIC EPACT DIGIT ONE..COPTIC EPACT NUMBER NINE HUNDRED
    Interval{ 0x10300, 0x1031F }, // Lo  [32] OLD ITALIC LETTER A..OLD ITALIC LETTER ESS
    Interval{ 0x10320, 0x10323 }, // No   [4] OLD ITALIC NUMERAL ONE..OLD ITALIC NUMERAL FIFTY
    Interval{ 0x1032D, 0x10340 }, // Lo  [20] OLD ITALIC LETTER YE..GOTHIC LETTER PAIRTHRA
    Interval{ 0x10341, 0x10341 }, // Nl       GOTHIC LETTER NINETY
    Interval{ 0x10342, 0x10349 }, // Lo   [8] GOTHIC LETTER RAIDA..GOTHIC LETTER OTHAL
    Interval{ 0x1034A, 0x1034A }, // Nl       GOTHIC LETTER NINE HUNDRED
    Interval{ 0x10350, 0x10375 }, // Lo  [38] OLD PERMIC LETTER AN..OLD PERMIC LETTER IA
    Interval{ 0x10380, 0x1039D }, // Lo  [30] UGARITIC LETTER ALPA..UGARITIC LETTER SSU
    Interval{ 0x1039F, 0x1039F }, // Po       UGARITIC WORD DIVIDER
    Interval{ 0x103A0, 0x103C3 }, // Lo  [36] OLD PERSIAN SIGN A..OLD PERSIAN SIGN HA
    Interval{ 0x103C8, 0x103CF }, // Lo   [8] OLD PERSIAN SIGN AURAMAZDAA..OLD PERSIAN SIGN BUUMISH
    Interval{ 0x103D0, 0x103D0 }, // Po       OLD PERSIAN WORD DIVIDER
    Interval{ 0x103D1, 0x103D5 }, // Nl   [5] OLD PERSIAN NUMBER ONE..OLD PERSIAN NUMBER HUNDRED
    Interval{ 0x10400, 0x1044F }, // L&  [80] DESERET CAPITAL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x10450, 0x1049D }, // Lo  [78] SHAVIAN LETTER PEEP..OSMANYA LETTER OO
    Interval{ 0x104A0, 0x104A9 }, // Nd  [10] OSMANYA DIGIT ZERO..OSMANYA DIGIT NINE
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10500, 0x10527 }, // Lo  [40] ELBASAN LETTER A..ELBASAN LETTER KHE
    Interval{ 0x10530, 0x10563 }, // Lo  [52] CAUCASIAN ALBANIAN LETTER ALT..CAUCASIAN ALBANIAN LETTER KIW
    Interval{ 0x1056F, 0x1056F }, // Po       CAUCASIAN ALBANIAN CITATION MARK
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10600, 0x10736 }, // Lo [311] LINEAR A SIGN AB001..LINEAR A SIGN A664
    Interval{ 0x10740, 0x10755 }, // Lo  [22] LINEAR A SIGN A701 A..LINEAR A SIGN A732 JE
    Interval{ 0x10760, 0x10767 }, // Lo   [8] LINEAR A SIGN A800..LINEAR A SIGN A807
    Interval{ 0x10780, 0x10785 }, // Lm   [6] MODIFIER LETTER SMALL CAPITAL AA..MODIFIER LETTER SMALL B WITH HOOK
    Interval{ 0x10787, 0x107B0 }, // Lm  [42] MODIFIER LETTER SMALL DZ DIGRAPH..MODIFIER LETTER SMALL V WITH RIGHT HOOK
    Interval{ 0x107B2, 0x107BA }, // Lm   [9] MODIFIER LETTER SMALL CAPITAL Y..MODIFIER LETTER SMALL S WITH CURL
    Interval{ 0x10800, 0x10805 }, // Lo   [6] CYPRIOT SYLLABLE A..CYPRIOT SYLLABLE JA
    Interval{ 0x10808, 0x10808 }, // Lo       CYPRIOT SYLLABLE JO
    Interval{ 0x1080A, 0x10835 }, // Lo  [44] CYPRIOT SYLLABLE KA..CYPRIOT SYLLABLE WO
    Interval{ 0x10837, 0x10838 }, // Lo   [2] CYPRIOT SYLLABLE XA..CYPRIOT SYLLABLE XE
    Interval{ 0x1083C, 0x1083C }, // Lo       CYPRIOT SYLLABLE ZA
    Interval{ 0x1083F, 0x10855 }, // Lo  [23] CYPRIOT SYLLABLE ZO..IMPERIAL ARAMAIC LETTER TAW
    Interval{ 0x10857, 0x10857 }, // Po       IMPERIAL ARAMAIC SECTION SIGN
    Interval{ 0x10858, 0x1085F }, // No   [8] IMPERIAL ARAMAIC NUMBER ONE..IMPERIAL ARAMAIC NUMBER TEN THOUSAND
    Interval{ 0x10860, 0x10876 }, // Lo  [23] PALMYRENE LETTER ALEPH..PALMYRENE LETTER TAW
    Interval{ 0x10877, 0x10878 }, // So   [2] PALMYRENE LEFT-POINTING FLEURON..PALMYRENE RIGHT-POINTING FLEURON
    Interval{ 0x10879, 0x1087F }, // No   [7] PALMYRENE NUMBER ONE..PALMYRENE NUMBER TWENTY
    Interval{ 0x10880, 0x1089E }, // Lo  [31] NABATAEAN LETTER FINAL ALEPH..NABATAEAN LETTER TAW
    Interval{ 0x108A7, 0x108AF }, // No   [9] NABATAEAN NUMBER ONE..NABATAEAN NUMBER ONE HUNDRED
    Interval{ 0x108E0, 0x108F2 }, // Lo  [19] HATRAN LETTER ALEPH..HATRAN LETTER QOPH
    Interval{ 0x108F4, 0x108F5 }, // Lo   [2] HATRAN LETTER SHIN..HATRAN LETTER TAW
    Interval{ 0x108FB, 0x108FF }, // No   [5] HATRAN NUMBER ONE..HATRAN NUMBER ONE HUNDRED
    Interval{ 0x10900, 0x10915 }, // Lo  [22] PHOENICIAN LETTER ALF..PHOENICIAN LETTER TAU
    Interval{ 0x10916, 0x1091B }, // No   [6] PHOENICIAN NUMBER ONE..PHOENICIAN NUMBER THREE
    Interval{ 0x1091F, 0x1091F }, // Po       PHOENICIAN WORD SEPARATOR
    Interval{ 0x10920, 0x10939 }, // Lo  [26] LYDIAN LETTER A..LYDIAN LETTER C
    Interval{ 0x1093F, 0x1093F }, // Po       LYDIAN TRIANGULAR MARK
    Interval{ 0x10980, 0x109B7 }, // Lo  [56] MEROITIC HIEROGLYPHIC LETTER A..MEROITIC CURSIVE LETTER DA
    Interval{ 0x109BC, 0x109BD }, // No   [2] MEROITIC CURSIVE FRACTION ELEVEN TWELFTHS..MEROITIC CURSIVE FRACTION ONE HALF
    Interval{ 0x109BE, 0x109BF }, // Lo   [2] MEROITIC CURSIVE LOGOGRAM RMT..MEROITIC CURSIVE LOGOGRAM IMN
    Interval{ 0x109C0, 0x109CF }, // No  [16] MEROITIC CURSIVE NUMBER ONE..MEROITIC CURSIVE NUMBER SEVENTY
    Interval{ 0x109D2, 0x109FF }, // No  [46] MEROITIC CURSIVE NUMBER ONE HUNDRED..MEROITIC CURSIVE FRACTION TEN TWELFTHS
    Interval{ 0x10A00, 0x10A00 }, // Lo       KHAROSHTHI LETTER A
    Interval{ 0x10A10, 0x10A13 }, // Lo   [4] KHAROSHTHI LETTER KA..KHAROSHTHI LETTER GHA
    Interval{ 0x10A15, 0x10A17 }, // Lo   [3] KHAROSHTHI LETTER CA..KHAROSHTHI LETTER JA
    Interval{ 0x10A19, 0x10A35 }, // Lo  [29] KHAROSHTHI LETTER NYA..KHAROSHTHI LETTER VHA
    Interval{ 0x10A40, 0x10A48 }, // No   [9] KHAROSHTHI DIGIT ONE..KHAROSHTHI FRACTION ONE HALF
    Interval{ 0x10A50, 0x10A58 }, // Po   [9] KHAROSHTHI PUNCTUATION DOT..KHAROSHTHI PUNCTUATION LINES
    Interval{ 0x10A60, 0x10A7C }, // Lo  [29] OLD SOUTH ARABIAN LETTER HE..OLD SOUTH ARABIAN LETTER THETH
    Interval{ 0x10A7D, 0x10A7E }, // No   [2] OLD SOUTH ARABIAN NUMBER ONE..OLD SOUTH ARABIAN NUMBER FIFTY
    Interval{ 0x10A7F, 0x10A7F }, // Po       OLD SOUTH ARABIAN NUMERIC INDICATOR
    Interval{ 0x10A80, 0x10A9C }, // Lo  [29] OLD NORTH ARABIAN LETTER HEH..OLD NORTH ARABIAN LETTER ZAH
    Interval{ 0x10A9D, 0x10A9F }, // No   [3] OLD NORTH ARABIAN NUMBER ONE..OLD NORTH ARABIAN NUMBER TWENTY
    Interval{ 0x10AC0, 0x10AC7 }, // Lo   [8] MANICHAEAN LETTER ALEPH..MANICHAEAN LETTER WAW
    Interval{ 0x10AC8, 0x10AC8 }, // So       MANICHAEAN SIGN UD
    Interval{ 0x10AC9, 0x10AE4 }, // Lo  [28] MANICHAEAN LETTER ZAYIN..MANICHAEAN LETTER TAW
    Interval{ 0x10AEB, 0x10AEF }, // No   [5] MANICHAEAN NUMBER ONE..MANICHAEAN NUMBER ONE HUNDRED
    Interval{ 0x10AF0, 0x10AF6 }, // Po   [7] MANICHAEAN PUNCTUATION STAR..MANICHAEAN PUNCTUATION LINE FILLER
    Interval{ 0x10B00, 0x10B35 }, // Lo  [54] AVESTAN LETTER A..AVESTAN LETTER HE
    Interval{ 0x10B39, 0x10B3F }, // Po   [7] AVESTAN ABBREVIATION MARK..LARGE ONE RING OVER TWO RINGS PUNCTUATION
    Interval{ 0x10B40, 0x10B55 }, // Lo  [22] INSCRIPTIONAL PARTHIAN LETTER ALEPH..INSCRIPTIONAL PARTHIAN LETTER TAW
    Interval{ 0x10B58, 0x10B5F }, // No   [8] INSCRIPTIONAL PARTHIAN NUMBER ONE..INSCRIPTIONAL PARTHIAN NUMBER ONE THOUSAND
    Interval{ 0x10B60, 0x10B72 }, // Lo  [19] INSCRIPTIONAL PAHLAVI LETTER ALEPH..INSCRIPTIONAL PAHLAVI LETTER TAW
    Interval{ 0x10B78, 0x10B7F }, // No   [8] INSCRIPTIONAL PAHLAVI NUMBER ONE..INSCRIPTIONAL PAHLAVI NUMBER ONE THOUSAND
    Interval{ 0x10B80, 0x10B91 }, // Lo  [18] PSALTER PAHLAVI LETTER ALEPH..PSALTER PAHLAVI LETTER TAW
    Interval{ 0x10B99, 0x10B9C }, // Po   [4] PSALTER PAHLAVI SECTION MARK..PSALTER PAHLAVI FOUR DOTS WITH DOT
    Interval{ 0x10BA9, 0x10BAF }, // No   [7] PSALTER PAHLAVI NUMBER ONE..PSALTER PAHLAVI NUMBER ONE HUNDRED
    Interval{ 0x10C00, 0x10C48 }, // Lo  [73] OLD TURKIC LETTER ORKHON A..OLD TURKIC LETTER ORKHON BASH
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x10CFA, 0x10CFF }, // No   [6] OLD HUNGARIAN NUMBER ONE..OLD HUNGARIAN NUMBER ONE THOUSAND
    Interval{ 0x10D00, 0x10D23 }, // Lo  [36] HANIFI ROHINGYA LETTER A..HANIFI ROHINGYA MARK NA KHONNA
    Interval{ 0x10D30, 0x10D39 }, // Nd  [10] HANIFI ROHINGYA DIGIT ZERO..HANIFI ROHINGYA DIGIT NINE
    Interval{ 0x10E60, 0x10E7E }, // No  [31] RUMI DIGIT ONE..RUMI FRACTION TWO THIRDS
    Interval{ 0x10E80, 0x10EA9 }, // Lo  [42] YEZIDI LETTER ELIF..YEZIDI LETTER ET
    Interval{ 0x10EAD, 0x10EAD }, // Pd       YEZIDI HYPHENATION MARK
    Interval{ 0x10EB0, 0x10EB1 }, // Lo   [2] YEZIDI LETTER LAM WITH DOT ABOVE..YEZIDI LETTER YOT WITH CIRCUMFLEX ABOVE
    Interval{ 0x10F00, 0x10F1C }, // Lo  [29] OLD SOGDIAN LETTER ALEPH..OLD SOGDIAN LETTER FINAL TAW WITH VERTICAL TAIL
    Interval{ 0x10F1D, 0x10F26 }, // No  [10] OLD SOGDIAN NUMBER ONE..OLD SOGDIAN FRACTION ONE HALF
    Interval{ 0x10F27, 0x10F27 }, // Lo       OLD SOGDIAN LIGATURE AYIN-DALETH
    Interval{ 0x10F30, 0x10F45 }, // Lo  [22] SOGDIAN LETTER ALEPH..SOGDIAN INDEPENDENT SHIN
    Interval{ 0x10F51, 0x10F54 }, // No   [4] SOGDIAN NUMBER ONE..SOGDIAN NUMBER ONE HUNDRED
    Interval{ 0x10F55, 0x10F59 }, // Po   [5] SOGDIAN PUNCTUATION TWO VERTICAL BARS..SOGDIAN PUNCTUATION HALF CIRCLE WITH DOT
    Interval{ 0x10F70, 0x10F81 }, // Lo  [18] OLD UYGHUR LETTER ALEPH..OLD UYGHUR LETTER LESH
    Interval{ 0x10F86, 0x10F89 }, // Po   [4] OLD UYGHUR PUNCTUATION BAR..OLD UYGHUR PUNCTUATION FOUR DOTS
    Interval{ 0x10FB0, 0x10FC4 }, // Lo  [21] CHORASMIAN LETTER ALEPH..CHORASMIAN LETTER TAW
    Interval{ 0x10FC5, 0x10FCB }, // No   [7] CHORASMIAN NUMBER ONE..CHORASMIAN NUMBER ONE HUNDRED
    Interval{ 0x10FE0, 0x10FF6 }, // Lo  [23] ELYMAIC LETTER ALEPH..ELYMAIC LIGATURE ZAYIN-YODH
    Interval{ 0x11000, 0x11000 }, // Mc       BRAHMI SIGN CANDRABINDU
    Interval{ 0x11002, 0x11002 }, // Mc       BRAHMI SIGN VISARGA
    Interval{ 0x11003, 0x11037 }, // Lo  [53] BRAHMI SIGN JIHVAMULIYA..BRAHMI LETTER OLD TAMIL NNNA
    Interval{ 0x11047, 0x1104D }, // Po   [7] BRAHMI DANDA..BRAHMI PUNCTUATION LOTUS
    Interval{ 0x11052, 0x11065 }, // No  [20] BRAHMI NUMBER ONE..BRAHMI NUMBER ONE THOUSAND
    Interval{ 0x11066, 0x1106F }, // Nd  [10] BRAHMI DIGIT ZERO..BRAHMI DIGIT NINE
    Interval{ 0x11071, 0x11072 }, // Lo   [2] BRAHMI LETTER OLD TAMIL SHORT E..BRAHMI LETTER OLD TAMIL SHORT O
    Interval{ 0x11075, 0x11075 }, // Lo       BRAHMI LETTER OLD TAMIL LLA
    Interval{ 0x11082, 0x11082 }, // Mc       KAITHI SIGN VISARGA
    Interval{ 0x11083, 0x110AF }, // Lo  [45] KAITHI LETTER A..KAITHI LETTER HA
    Interval{ 0x110B0, 0x110B2 }, // Mc   [3] KAITHI VOWEL SIGN AA..KAITHI VOWEL SIGN II
    Interval{ 0x110B7, 0x110B8 }, // Mc   [2] KAITHI VOWEL SIGN O..KAITHI VOWEL SIGN AU
    Interval{ 0x110BB, 0x110BC }, // Po   [2] KAITHI ABBREVIATION SIGN..KAITHI ENUMERATION SIGN
    Interval{ 0x110BE, 0x110C1 }, // Po   [4] KAITHI SECTION MARK..KAITHI DOUBLE DANDA
    Interval{ 0x110D0, 0x110E8 }, // Lo  [25] SORA SOMPENG LETTER SAH..SORA SOMPENG LETTER MAE
    Interval{ 0x110F0, 0x110F9 }, // Nd  [10] SORA SOMPENG DIGIT ZERO..SORA SOMPENG DIGIT NINE
    Interval{ 0x11103, 0x11126 }, // Lo  [36] CHAKMA LETTER AA..CHAKMA LETTER HAA
    Interval{ 0x1112C, 0x1112C }, // Mc       CHAKMA VOWEL SIGN E
    Interval{ 0x11136, 0x1113F }, // Nd  [10] CHAKMA DIGIT ZERO..CHAKMA DIGIT NINE
    Interval{ 0x11140, 0x11143 }, // Po   [4] CHAKMA SECTION MARK..CHAKMA QUESTION MARK
    Interval{ 0x11144, 0x11144 }, // Lo       CHAKMA LETTER LHAA
    Interval{ 0x11145, 0x11146 }, // Mc   [2] CHAKMA VOWEL SIGN AA..CHAKMA VOWEL SIGN EI
    Interval{ 0x11147, 0x11147 }, // Lo       CHAKMA LETTER VAA
    Interval{ 0x11150, 0x11172 }, // Lo  [35] MAHAJANI LETTER A..MAHAJANI LETTER RRA
    Interval{ 0x11174, 0x11175 }, // Po   [2] MAHAJANI ABBREVIATION SIGN..MAHAJANI SECTION MARK
    Interval{ 0x11176, 0x11176 }, // Lo       MAHAJANI LIGATURE SHRI
    Interval{ 0x11182, 0x11182 }, // Mc       SHARADA SIGN VISARGA
    Interval{ 0x11183, 0x111B2 }, // Lo  [48] SHARADA LETTER A..SHARADA LETTER HA
    Interval{ 0x111B3, 0x111B5 }, // Mc   [3] SHARADA VOWEL SIGN AA..SHARADA VOWEL SIGN II
    Interval{ 0x111BF, 0x111C0 }, // Mc   [2] SHARADA VOWEL SIGN AU..SHARADA SIGN VIRAMA
    Interval{ 0x111C1, 0x111C4 }, // Lo   [4] SHARADA SIGN AVAGRAHA..SHARADA OM
    Interval{ 0x111C5, 0x111C8 }, // Po   [4] SHARADA DANDA..SHARADA SEPARATOR
    Interval{ 0x111CD, 0x111CD }, // Po       SHARADA SUTRA MARK
    Interval{ 0x111CE, 0x111CE }, // Mc       SHARADA VOWEL SIGN PRISHTHAMATRA E
    Interval{ 0x111D0, 0x111D9 }, // Nd  [10] SHARADA DIGIT ZERO..SHARADA DIGIT NINE
    Interval{ 0x111DA, 0x111DA }, // Lo       SHARADA EKAM
    Interval{ 0x111DB, 0x111DB }, // Po       SHARADA SIGN SIDDHAM
    Interval{ 0x111DC, 0x111DC }, // Lo       SHARADA HEADSTROKE
    Interval{ 0x111DD, 0x111DF }, // Po   [3] SHARADA CONTINUATION SIGN..SHARADA SECTION MARK-2
    Interval{ 0x111E1, 0x111F4 }, // No  [20] SINHALA ARCHAIC DIGIT ONE..SINHALA ARCHAIC NUMBER ONE THOUSAND
    Interval{ 0x11200, 0x11211 }, // Lo  [18] KHOJKI LETTER A..KHOJKI LETTER JJA
    Interval{ 0x11213, 0x1122B }, // Lo  [25] KHOJKI LETTER NYA..KHOJKI LETTER LLA
    Interval{ 0x1122C, 0x1122E }, // Mc   [3] KHOJKI VOWEL SIGN AA..KHOJKI VOWEL SIGN II
    Interval{ 0x11232, 0x11233 }, // Mc   [2] KHOJKI VOWEL SIGN O..KHOJKI VOWEL SIGN AU
    Interval{ 0x11235, 0x11235 }, // Mc       KHOJKI SIGN VIRAMA
    Interval{ 0x11238, 0x1123D }, // Po   [6] KHOJKI DANDA..KHOJKI ABBREVIATION SIGN
    Interval{ 0x1123F, 0x11240 }, // Lo   [2] KHOJKI LETTER QA..KHOJKI LETTER SHORT I
    Interval{ 0x11280, 0x11286 }, // Lo   [7] MULTANI LETTER A..MULTANI LETTER GA
    Interval{ 0x11288, 0x11288 }, // Lo       MULTANI LETTER GHA
    Interval{ 0x1128A, 0x1128D }, // Lo   [4] MULTANI LETTER CA..MULTANI LETTER JJA
    Interval{ 0x1128F, 0x1129D }, // Lo  [15] MULTANI LETTER NYA..MULTANI LETTER BA
    Interval{ 0x1129F, 0x112A8 }, // Lo  [10] MULTANI LETTER BHA..MULTANI LETTER RHA
    Interval{ 0x112A9, 0x112A9 }, // Po       MULTANI SECTION MARK
    Interval{ 0x112B0, 0x112DE }, // Lo  [47] KHUDAWADI LETTER A..KHUDAWADI LETTER HA
    Interval{ 0x112E0, 0x112E2 }, // Mc   [3] KHUDAWADI VOWEL SIGN AA..KHUDAWADI VOWEL SIGN II
    Interval{ 0x112F0, 0x112F9 }, // Nd  [10] KHUDAWADI DIGIT ZERO..KHUDAWADI DIGIT NINE
    Interval{ 0x11302, 0x11303 }, // Mc   [2] GRANTHA SIGN ANUSVARA..GRANTHA SIGN VISARGA
    Interval{ 0x11305, 0x1130C }, // Lo   [8] GRANTHA LETTER A..GRANTHA LETTER VOCALIC L
    Interval{ 0x1130F, 0x11310 }, // Lo   [2] GRANTHA LETTER EE..GRANTHA LETTER AI
    Interval{ 0x11313, 0x11328 }, // Lo  [22] GRANTHA LETTER OO..GRANTHA LETTER NA
    Interval{ 0x1132A, 0x11330 }, // Lo   [7] GRANTHA LETTER PA..GRANTHA LETTER RA
    Interval{ 0x11332, 0x11333 }, // Lo   [2] GRANTHA LETTER LA..GRANTHA LETTER LLA
    Interval{ 0x11335, 0x11339 }, // Lo   [5] GRANTHA LETTER VA..GRANTHA LETTER HA
    Interval{ 0x1133D, 0x1133D }, // Lo       GRANTHA SIGN AVAGRAHA
    Interval{ 0x1133F, 0x1133F }, // Mc       GRANTHA VOWEL SIGN I
    Interval{ 0x11341, 0x11344 }, // Mc   [4] GRANTHA VOWEL SIGN U..GRANTHA VOWEL SIGN VOCALIC RR
    Interval{ 0x11347, 0x11348 }, // Mc   [2] GRANTHA VOWEL SIGN EE..GRANTHA VOWEL SIGN AI
    Interval{ 0x1134B, 0x1134D }, // Mc   [3] GRANTHA VOWEL SIGN OO..GRANTHA SIGN VIRAMA
    Interval{ 0x11350, 0x11350 }, // Lo       GRANTHA OM
    Interval{ 0x1135D, 0x11361 }, // Lo   [5] GRANTHA SIGN PLUTA..GRANTHA LETTER VOCALIC LL
    Interval{ 0x11362, 0x11363 }, // Mc   [2] GRANTHA VOWEL SIGN VOCALIC L..GRANTHA VOWEL SIGN VOCALIC LL
    Interval{ 0x11400, 0x11434 }, // Lo  [53] NEWA LETTER A..NEWA LETTER HA
    Interval{ 0x11435, 0x11437 }, // Mc   [3] NEWA VOWEL SIGN AA..NEWA VOWEL SIGN II
    Interval{ 0x11440, 0x11441 }, // Mc   [2] NEWA VOWEL SIGN O..NEWA VOWEL SIGN AU
    Interval{ 0x11445, 0x11445 }, // Mc       NEWA SIGN VISARGA
    Interval{ 0x11447, 0x1144A }, // Lo   [4] NEWA SIGN AVAGRAHA..NEWA SIDDHI
    Interval{ 0x1144B, 0x1144F }, // Po   [5] NEWA DANDA..NEWA ABBREVIATION SIGN
    Interval{ 0x11450, 0x11459 }, // Nd  [10] NEWA DIGIT ZERO..NEWA DIGIT NINE
    Interval{ 0x1145A, 0x1145B }, // Po   [2] NEWA DOUBLE COMMA..NEWA PLACEHOLDER MARK
    Interval{ 0x1145D, 0x1145D }, // Po       NEWA INSERTION SIGN
    Interval{ 0x1145F, 0x11461 }, // Lo   [3] NEWA LETTER VEDIC ANUSVARA..NEWA SIGN UPADHMANIYA
    Interval{ 0x11480, 0x114AF }, // Lo  [48] TIRHUTA ANJI..TIRHUTA LETTER HA
    Interval{ 0x114B1, 0x114B2 }, // Mc   [2] TIRHUTA VOWEL SIGN I..TIRHUTA VOWEL SIGN II
    Interval{ 0x114B9, 0x114B9 }, // Mc       TIRHUTA VOWEL SIGN E
    Interval{ 0x114BB, 0x114BC }, // Mc   [2] TIRHUTA VOWEL SIGN AI..TIRHUTA VOWEL SIGN O
    Interval{ 0x114BE, 0x114BE }, // Mc       TIRHUTA VOWEL SIGN AU
    Interval{ 0x114C1, 0x114C1 }, // Mc       TIRHUTA SIGN VISARGA
    Interval{ 0x114C4, 0x114C5 }, // Lo   [2] TIRHUTA SIGN AVAGRAHA..TIRHUTA GVANG
    Interval{ 0x114C6, 0x114C6 }, // Po       TIRHUTA ABBREVIATION SIGN
    Interval{ 0x114C7, 0x114C7 }, // Lo       TIRHUTA OM
    Interval{ 0x114D0, 0x114D9 }, // Nd  [10] TIRHUTA DIGIT ZERO..TIRHUTA DIGIT NINE
    Interval{ 0x11580, 0x115AE }, // Lo  [47] SIDDHAM LETTER A..SIDDHAM LETTER HA
    Interval{ 0x115B0, 0x115B1 }, // Mc   [2] SIDDHAM VOWEL SIGN I..SIDDHAM VOWEL SIGN II
    Interval{ 0x115B8, 0x115BB }, // Mc   [4] SIDDHAM VOWEL SIGN E..SIDDHAM VOWEL SIGN AU
    Interval{ 0x115BE, 0x115BE }, // Mc       SIDDHAM SIGN VISARGA
    Interval{ 0x115C1, 0x115D7 }, // Po  [23] SIDDHAM SIGN SIDDHAM..SIDDHAM SECTION MARK WITH CIRCLES AND FOUR ENCLOSURES
    Interval{ 0x115D8, 0x115DB }, // Lo   [4] SIDDHAM LETTER THREE-CIRCLE ALTERNATE I..SIDDHAM LETTER ALTERNATE U
    Interval{ 0x11600, 0x1162F }, // Lo  [48] MODI LETTER A..MODI LETTER LLA
    Interval{ 0x11630, 0x11632 }, // Mc   [3] MODI VOWEL SIGN AA..MODI VOWEL SIGN II
    Interval{ 0x1163B, 0x1163C }, // Mc   [2] MODI VOWEL SIGN O..MODI VOWEL SIGN AU
    Interval{ 0x1163E, 0x1163E }, // Mc       MODI SIGN VISARGA
    Interval{ 0x11641, 0x11643 }, // Po   [3] MODI DANDA..MODI ABBREVIATION SIGN
    Interval{ 0x11644, 0x11644 }, // Lo       MODI SIGN HUVA
    Interval{ 0x11650, 0x11659 }, // Nd  [10] MODI DIGIT ZERO..MODI DIGIT NINE
    Interval{ 0x11660, 0x1166C }, // Po  [13] MONGOLIAN BIRGA WITH ORNAMENT..MONGOLIAN TURNED SWIRL BIRGA WITH DOUBLE ORNAMENT
    Interval{ 0x11680, 0x116AA }, // Lo  [43] TAKRI LETTER A..TAKRI LETTER RRA
    Interval{ 0x116AC, 0x116AC }, // Mc       TAKRI SIGN VISARGA
    Interval{ 0x116AE, 0x116AF }, // Mc   [2] TAKRI VOWEL SIGN I..TAKRI VOWEL SIGN II
    Interval{ 0x116B6, 0x116B6 }, // Mc       TAKRI SIGN VIRAMA
    Interval{ 0x116B8, 0x116B8 }, // Lo       TAKRI LETTER ARCHAIC KHA
    Interval{ 0x116B9, 0x116B9 }, // Po       TAKRI ABBREVIATION SIGN
    Interval{ 0x116C0, 0x116C9 }, // Nd  [10] TAKRI DIGIT ZERO..TAKRI DIGIT NINE
    Interval{ 0x11700, 0x1171A }, // Lo  [27] AHOM LETTER KA..AHOM LETTER ALTERNATE BA
    Interval{ 0x11720, 0x11721 }, // Mc   [2] AHOM VOWEL SIGN A..AHOM VOWEL SIGN AA
    Interval{ 0x11726, 0x11726 }, // Mc       AHOM VOWEL SIGN E
    Interval{ 0x11730, 0x11739 }, // Nd  [10] AHOM DIGIT ZERO..AHOM DIGIT NINE
    Interval{ 0x1173A, 0x1173B }, // No   [2] AHOM NUMBER TEN..AHOM NUMBER TWENTY
    Interval{ 0x1173C, 0x1173E }, // Po   [3] AHOM SIGN SMALL SECTION..AHOM SIGN RULAI
    Interval{ 0x1173F, 0x1173F }, // So       AHOM SYMBOL VI
    Interval{ 0x11740, 0x11746 }, // Lo   [7] AHOM LETTER CA..AHOM LETTER LLA
    Interval{ 0x11800, 0x1182B }, // Lo  [44] DOGRA LETTER A..DOGRA LETTER RRA
    Interval{ 0x1182C, 0x1182E }, // Mc   [3] DOGRA VOWEL SIGN AA..DOGRA VOWEL SIGN II
    Interval{ 0x11838, 0x11838 }, // Mc       DOGRA SIGN VISARGA
    Interval{ 0x1183B, 0x1183B }, // Po       DOGRA ABBREVIATION SIGN
    Interval{ 0x118A0, 0x118DF }, // L&  [64] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x118E0, 0x118E9 }, // Nd  [10] WARANG CITI DIGIT ZERO..WARANG CITI DIGIT NINE
    Interval{ 0x118EA, 0x118F2 }, // No   [9] WARANG CITI NUMBER TEN..WARANG CITI NUMBER NINETY
    Interval{ 0x118FF, 0x11906 }, // Lo   [8] WARANG CITI OM..DIVES AKURU LETTER E
    Interval{ 0x11909, 0x11909 }, // Lo       DIVES AKURU LETTER O
    Interval{ 0x1190C, 0x11913 }, // Lo   [8] DIVES AKURU LETTER KA..DIVES AKURU LETTER JA
    Interval{ 0x11915, 0x11916 }, // Lo   [2] DIVES AKURU LETTER NYA..DIVES AKURU LETTER TTA
    Interval{ 0x11918, 0x1192F }, // Lo  [24] DIVES AKURU LETTER DDA..DIVES AKURU LETTER ZA
    Interval{ 0x11931, 0x11935 }, // Mc   [5] DIVES AKURU VOWEL SIGN I..DIVES AKURU VOWEL SIGN E
    Interval{ 0x11937, 0x11938 }, // Mc   [2] DIVES AKURU VOWEL SIGN AI..DIVES AKURU VOWEL SIGN O
    Interval{ 0x1193D, 0x1193D }, // Mc       DIVES AKURU SIGN HALANTA
    Interval{ 0x1193F, 0x1193F }, // Lo       DIVES AKURU PREFIXED NASAL SIGN
    Interval{ 0x11940, 0x11940 }, // Mc       DIVES AKURU MEDIAL YA
    Interval{ 0x11941, 0x11941 }, // Lo       DIVES AKURU INITIAL RA
    Interval{ 0x11942, 0x11942 }, // Mc       DIVES AKURU MEDIAL RA
    Interval{ 0x11944, 0x11946 }, // Po   [3] DIVES AKURU DOUBLE DANDA..DIVES AKURU END OF TEXT MARK
    Interval{ 0x11950, 0x11959 }, // Nd  [10] DIVES AKURU DIGIT ZERO..DIVES AKURU DIGIT NINE
    Interval{ 0x119A0, 0x119A7 }, // Lo   [8] NANDINAGARI LETTER A..NANDINAGARI LETTER VOCALIC RR
    Interval{ 0x119AA, 0x119D0 }, // Lo  [39] NANDINAGARI LETTER E..NANDINAGARI LETTER RRA
    Interval{ 0x119D1, 0x119D3 }, // Mc   [3] NANDINAGARI VOWEL SIGN AA..NANDINAGARI VOWEL SIGN II
    Interval{ 0x119DC, 0x119DF }, // Mc   [4] NANDINAGARI VOWEL SIGN O..NANDINAGARI SIGN VISARGA
    Interval{ 0x119E1, 0x119E1 }, // Lo       NANDINAGARI SIGN AVAGRAHA
    Interval{ 0x119E2, 0x119E2 }, // Po       NANDINAGARI SIGN SIDDHAM
    Interval{ 0x119E3, 0x119E3 }, // Lo       NANDINAGARI HEADSTROKE
    Interval{ 0x119E4, 0x119E4 }, // Mc       NANDINAGARI VOWEL SIGN PRISHTHAMATRA E
    Interval{ 0x11A00, 0x11A00 }, // Lo       ZANABAZAR SQUARE LETTER A
    Interval{ 0x11A0B, 0x11A32 }, // Lo  [40] ZANABAZAR SQUARE LETTER KA..ZANABAZAR SQUARE LETTER KSSA
    Interval{ 0x11A39, 0x11A39 }, // Mc       ZANABAZAR SQUARE SIGN VISARGA
    Interval{ 0x11A3A, 0x11A3A }, // Lo       ZANABAZAR SQUARE CLUSTER-INITIAL LETTER RA
    Interval{ 0x11A3F, 0x11A46 }, // Po   [8] ZANABAZAR SQUARE INITIAL HEAD MARK..ZANABAZAR SQUARE CLOSING DOUBLE-LINED HEAD MARK
    Interval{ 0x11A50, 0x11A50 }, // Lo       SOYOMBO LETTER A
    Interval{ 0x11A57, 0x11A58 }, // Mc   [2] SOYOMBO VOWEL SIGN AI..SOYOMBO VOWEL SIGN AU
    Interval{ 0x11A5C, 0x11A89 }, // Lo  [46] SOYOMBO LETTER KA..SOYOMBO CLUSTER-INITIAL LETTER SA
    Interval{ 0x11A97, 0x11A97 }, // Mc       SOYOMBO SIGN VISARGA
    Interval{ 0x11A9A, 0x11A9C }, // Po   [3] SOYOMBO MARK TSHEG..SOYOMBO MARK DOUBLE SHAD
    Interval{ 0x11A9D, 0x11A9D }, // Lo       SOYOMBO MARK PLUTA
    Interval{ 0x11A9E, 0x11AA2 }, // Po   [5] SOYOMBO HEAD MARK WITH MOON AND SUN AND TRIPLE FLAME..SOYOMBO TERMINAL MARK-2
    Interval{ 0x11AB0, 0x11AF8 }, // Lo  [73] CANADIAN SYLLABICS NATTILIK HI..PAU CIN HAU GLOTTAL STOP FINAL
    Interval{ 0x11B00, 0x11B09 }, // Po  [10] DEVANAGARI HEAD MARK..DEVANAGARI SIGN MINDU
    Interval{ 0x11C00, 0x11C08 }, // Lo   [9] BHAIKSUKI LETTER A..BHAIKSUKI LETTER VOCALIC L
    Interval{ 0x11C0A, 0x11C2E }, // Lo  [37] BHAIKSUKI LETTER E..BHAIKSUKI LETTER HA
    Interval{ 0x11C2F, 0x11C2F }, // Mc       BHAIKSUKI VOWEL SIGN AA
    Interval{ 0x11C3E, 0x11C3E }, // Mc       BHAIKSUKI SIGN VISARGA
    Interval{ 0x11C40, 0x11C40 }, // Lo       BHAIKSUKI SIGN AVAGRAHA
    Interval{ 0x11C41, 0x11C45 }, // Po   [5] BHAIKSUKI DANDA..BHAIKSUKI GAP FILLER-2
    Interval{ 0x11C50, 0x11C59 }, // Nd  [10] BHAIKSUKI DIGIT ZERO..BHAIKSUKI DIGIT NINE
    Interval{ 0x11C5A, 0x11C6C }, // No  [19] BHAIKSUKI NUMBER ONE..BHAIKSUKI HUNDREDS UNIT MARK
    Interval{ 0x11C70, 0x11C71 }, // Po   [2] MARCHEN HEAD MARK..MARCHEN MARK SHAD
    Interval{ 0x11C72, 0x11C8F }, // Lo  [30] MARCHEN LETTER KA..MARCHEN LETTER A
    Interval{ 0x11CA9, 0x11CA9 }, // Mc       MARCHEN SUBJOINED LETTER YA
    Interval{ 0x11CB1, 0x11CB1 }, // Mc       MARCHEN VOWEL SIGN I
    Interval{ 0x11CB4, 0x11CB4 }, // Mc       MARCHEN VOWEL SIGN O
    Interval{ 0x11D00, 0x11D06 }, // Lo   [7] MASARAM GONDI LETTER A..MASARAM GONDI LETTER E
    Interval{ 0x11D08, 0x11D09 }, // Lo   [2] MASARAM GONDI LETTER AI..MASARAM GONDI LETTER O
    Interval{ 0x11D0B, 0x11D30 }, // Lo  [38] MASARAM GONDI LETTER AU..MASARAM GONDI LETTER TRA
    Interval{ 0x11D46, 0x11D46 }, // Lo       MASARAM GONDI REPHA
    Interval{ 0x11D50, 0x11D59 }, // Nd  [10] MASARAM GONDI DIGIT ZERO..MASARAM GONDI DIGIT NINE
    Interval{ 0x11D60, 0x11D65 }, // Lo   [6] GUNJALA GONDI LETTER A..GUNJALA GONDI LETTER UU
    Interval{ 0x11D67, 0x11D68 }, // Lo   [2] GUNJALA GONDI LETTER EE..GUNJALA GONDI LETTER AI
    Interval{ 0x11D6A, 0x11D89 }, // Lo  [32] GUNJALA GONDI LETTER OO..GUNJALA GONDI LETTER SA
    Interval{ 0x11D8A, 0x11D8E }, // Mc   [5] GUNJALA GONDI VOWEL SIGN AA..GUNJALA GONDI VOWEL SIGN UU
    Interval{ 0x11D93, 0x11D94 }, // Mc   [2] GUNJALA GONDI VOWEL SIGN OO..GUNJALA GONDI VOWEL SIGN AU
    Interval{ 0x11D96, 0x11D96 }, // Mc       GUNJALA GONDI SIGN VISARGA
    Interval{ 0x11D98, 0x11D98 }, // Lo       GUNJALA GONDI OM
    Interval{ 0x11DA0, 0x11DA9 }, // Nd  [10] GUNJALA GONDI DIGIT ZERO..GUNJALA GONDI DIGIT NINE
    Interval{ 0x11EE0, 0x11EF2 }, // Lo  [19] MAKASAR LETTER KA..MAKASAR ANGKA
    Interval{ 0x11EF5, 0x11EF6 }, // Mc   [2] MAKASAR VOWEL SIGN E..MAKASAR VOWEL SIGN O
    Interval{ 0x11EF7, 0x11EF8 }, // Po   [2] MAKASAR PASSIMBANG..MAKASAR END OF SECTION
    Interval{ 0x11F02, 0x11F02 }, // Lo       KAWI SIGN REPHA
    Interval{ 0x11F03, 0x11F03 }, // Mc       KAWI SIGN VISARGA
    Interval{ 0x11F04, 0x11F10 }, // Lo  [13] KAWI LETTER A..KAWI LETTER O
    Interval{ 0x11F12, 0x11F33 }, // Lo  [34] KAWI LETTER KA..KAWI LETTER JNYA
    Interval{ 0x11F34, 0x11F35 }, // Mc   [2] KAWI VOWEL SIGN AA..KAWI VOWEL SIGN ALTERNATE AA
    Interval{ 0x11F3E, 0x11F3F }, // Mc   [2] KAWI VOWEL SIGN E..KAWI VOWEL SIGN AI
    Interval{ 0x11F41, 0x11F41 }, // Mc       KAWI SIGN KILLER
    Interval{ 0x11F43, 0x11F4F }, // Po  [13] KAWI DANDA..KAWI PUNCTUATION CLOSING SPIRAL
    Interval{ 0x11F50, 0x11F59 }, // Nd  [10] KAWI DIGIT ZERO..KAWI DIGIT NINE
    Interval{ 0x11FB0, 0x11FB0 }, // Lo       LISU LETTER YHA
    Interval{ 0x11FC0, 0x11FD4 }, // No  [21] TAMIL FRACTION ONE THREE-HUNDRED-AND-TWENTIETH..TAMIL FRACTION DOWNSCALING FACTOR KIIZH
    Interval{ 0x11FD5, 0x11FDC }, // So   [8] TAMIL SIGN NEL..TAMIL SIGN MUKKURUNI
    Interval{ 0x11FDD, 0x11FE0 }, // Sc   [4] TAMIL SIGN KAACU..TAMIL SIGN VARAAKAN
    Interval{ 0x11FE1, 0x11FF1 }, // So  [17] TAMIL SIGN PAARAM..TAMIL SIGN VAKAIYARAA
    Interval{ 0x11FFF, 0x11FFF }, // Po       TAMIL PUNCTUATION END OF TEXT
    Interval{ 0x12000, 0x12399 }, // Lo [922] CUNEIFORM SIGN A..CUNEIFORM SIGN U U
    Interval{ 0x12400, 0x1246E }, // Nl [111] CUNEIFORM NUMERIC SIGN TWO ASH..CUNEIFORM NUMERIC SIGN NINE U VARIANT FORM
    Interval{ 0x12470, 0x12474 }, // Po   [5] CUNEIFORM PUNCTUATION SIGN OLD ASSYRIAN WORD DIVIDER..CUNEIFORM PUNCTUATION SIGN DIAGONAL QUADCOLON
    Interval{ 0x12480, 0x12543 }, // Lo [196] CUNEIFORM SIGN AB TIMES NUN TENU..CUNEIFORM SIGN ZU5 TIMES THREE DISH TENU
    Interval{ 0x12F90, 0x12FF0 }, // Lo  [97] CYPRO-MINOAN SIGN CM001..CYPRO-MINOAN SIGN CM114
    Interval{ 0x12FF1, 0x12FF2 }, // Po   [2] CYPRO-MINOAN SIGN CM301..CYPRO-MINOAN SIGN CM302
    Interval{ 0x13000, 0x1342F }, // Lo [1072] EGYPTIAN HIEROGLYPH A001..EGYPTIAN HIEROGLYPH V011D
    Interval{ 0x13441, 0x13446 }, // Lo   [6] EGYPTIAN HIEROGLYPH FULL BLANK..EGYPTIAN HIEROGLYPH WIDE LOST SIGN
    Interval{ 0x14400, 0x14646 }, // Lo [583] ANATOLIAN HIEROGLYPH A001..ANATOLIAN HIEROGLYPH A530
    Interval{ 0x16800, 0x16A38 }, // Lo [569] BAMUM LETTER PHASE-A NGKUE MFON..BAMUM LETTER PHASE-F VUEQ
    Interval{ 0x16A40, 0x16A5E }, // Lo  [31] MRO LETTER TA..MRO LETTER TEK
    Interval{ 0x16A60, 0x16A69 }, // Nd  [10] MRO DIGIT ZERO..MRO DIGIT NINE
    Interval{ 0x16A6E, 0x16A6F }, // Po   [2] MRO DANDA..MRO DOUBLE DANDA
    Interval{ 0x16A70, 0x16ABE }, // Lo  [79] TANGSA LETTER OZ..TANGSA LETTER ZA
    Interval{ 0x16AC0, 0x16AC9 }, // Nd  [10] TANGSA DIGIT ZERO..TANGSA DIGIT NINE
    Interval{ 0x16AD0, 0x16AED }, // Lo  [30] BASSA VAH LETTER ENNI..BASSA VAH LETTER I
    Interval{ 0x16AF5, 0x16AF5 }, // Po       BASSA VAH FULL STOP
    Interval{ 0x16B00, 0x16B2F }, // Lo  [48] PAHAWH HMONG VOWEL KEEB..PAHAWH HMONG CONSONANT CAU
    Interval{ 0x16B37, 0x16B3B }, // Po   [5] PAHAWH HMONG SIGN VOS THOM..PAHAWH HMONG SIGN VOS FEEM
    Interval{ 0x16B3C, 0x16B3F }, // So   [4] PAHAWH HMONG SIGN XYEEM NTXIV..PAHAWH HMONG SIGN XYEEM FAIB
    Interval{ 0x16B40, 0x16B43 }, // Lm   [4] PAHAWH HMONG SIGN VOS SEEV..PAHAWH HMONG SIGN IB YAM
    Interval{ 0x16B44, 0x16B44 }, // Po       PAHAWH HMONG SIGN XAUS
    Interval{ 0x16B45, 0x16B45 }, // So       PAHAWH HMONG SIGN CIM TSOV ROG
    Interval{ 0x16B50, 0x16B59 }, // Nd  [10] PAHAWH HMONG DIGIT ZERO..PAHAWH HMONG DIGIT NINE
    Interval{ 0x16B5B, 0x16B61 }, // No   [7] PAHAWH HMONG NUMBER TENS..PAHAWH HMONG NUMBER TRILLIONS
    Interval{ 0x16B63, 0x16B77 }, // Lo  [21] PAHAWH HMONG SIGN VOS LUB..PAHAWH HMONG SIGN CIM NRES TOS
    Interval{ 0x16B7D, 0x16B8F }, // Lo  [19] PAHAWH HMONG CLAN SIGN TSHEEJ..PAHAWH HMONG CLAN SIGN VWJ
    Interval{ 0x16E40, 0x16E7F }, // L&  [64] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x16E80, 0x16E96 }, // No  [23] MEDEFAIDRIN DIGIT ZERO..MEDEFAIDRIN DIGIT THREE ALTERNATE FORM
    Interval{ 0x16E97, 0x16E9A }, // Po   [4] MEDEFAIDRIN COMMA..MEDEFAIDRIN EXCLAMATION OH
    Interval{ 0x16F00, 0x16F4A }, // Lo  [75] MIAO LETTER PA..MIAO LETTER RTE
    Interval{ 0x16F50, 0x16F50 }, // Lo       MIAO LETTER NASALIZATION
    Interval{ 0x16F51, 0x16F87 }, // Mc  [55] MIAO SIGN ASPIRATION..MIAO VOWEL SIGN UI
    Interval{ 0x16F93, 0x16F9F }, // Lm  [13] MIAO LETTER TONE-2..MIAO LETTER REFORMED TONE-8
    Interval{ 0x16FE0, 0x16FE1 }, // Lm   [2] TANGUT ITERATION MARK..NUSHU ITERATION MARK
    Interval{ 0x16FE2, 0x16FE2 }, // Po       OLD CHINESE HOOK MARK
    Interval{ 0x16FE3, 0x16FE3 }, // Lm       OLD CHINESE ITERATION MARK
    Interval{ 0x16FF0, 0x16FF1 }, // Mc   [2] VIETNAMESE ALTERNATE READING MARK CA..VIETNAMESE ALTERNATE READING MARK NHAY
    Interval{ 0x17000, 0x187F7 }, // Lo [6136] TANGUT IDEOGRAPH-17000..TANGUT IDEOGRAPH-187F7
    Interval{ 0x18800, 0x18CD5 }, // Lo [1238] TANGUT COMPONENT-001..KHITAN SMALL SCRIPT CHARACTER-18CD5
    Interval{ 0x18D00, 0x18D08 }, // Lo   [9] TANGUT IDEOGRAPH-18D00..TANGUT IDEOGRAPH-18D08
    Interval{ 0x1AFF0, 0x1AFF3 }, // Lm   [4] KATAKANA LETTER MINNAN TONE-2..KATAKANA LETTER MINNAN TONE-5
    Interval{ 0x1AFF5, 0x1AFFB }, // Lm   [7] KATAKANA LETTER MINNAN TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-5
    Interval{ 0x1AFFD, 0x1AFFE }, // Lm   [2] KATAKANA LETTER MINNAN NASALIZED TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-8
    Interval{ 0x1B000, 0x1B122 }, // Lo [291] KATAKANA LETTER ARCHAIC E..KATAKANA LETTER ARCHAIC WU
    Interval{ 0x1B132, 0x1B132 }, // Lo       HIRAGANA LETTER SMALL KO
    Interval{ 0x1B150, 0x1B152 }, // Lo   [3] HIRAGANA LETTER SMALL WI..HIRAGANA LETTER SMALL WO
    Interval{ 0x1B155, 0x1B155 }, // Lo       KATAKANA LETTER SMALL KO
    Interval{ 0x1B164, 0x1B167 }, // Lo   [4] KATAKANA LETTER SMALL WI..KATAKANA LETTER SMALL N
    Interval{ 0x1B170, 0x1B2FB }, // Lo [396] NUSHU CHARACTER-1B170..NUSHU CHARACTER-1B2FB
    Interval{ 0x1BC00, 0x1BC6A }, // Lo [107] DUPLOYAN LETTER H..DUPLOYAN LETTER VOCALIC M
    Interval{ 0x1BC70, 0x1BC7C }, // Lo  [13] DUPLOYAN AFFIX LEFT HORIZONTAL SECANT..DUPLOYAN AFFIX ATTACHED TANGENT HOOK
    Interval{ 0x1BC80, 0x1BC88 }, // Lo   [9] DUPLOYAN AFFIX HIGH ACUTE..DUPLOYAN AFFIX HIGH VERTICAL
    Interval{ 0x1BC90, 0x1BC99 }, // Lo  [10] DUPLOYAN AFFIX LOW ACUTE..DUPLOYAN AFFIX LOW ARROW
    Interval{ 0x1BC9C, 0x1BC9C }, // So       DUPLOYAN SIGN O WITH CROSS
    Interval{ 0x1BC9F, 0x1BC9F }, // Po       DUPLOYAN PUNCTUATION CHINOOK FULL STOP
    Interval{ 0x1CF50, 0x1CFC3 }, // So [116] ZNAMENNY NEUME KRYUK..ZNAMENNY NEUME PAUK
    Interval{ 0x1D000, 0x1D0F5 }, // So [246] BYZANTINE MUSICAL SYMBOL PSILI..BYZANTINE MUSICAL SYMBOL GORGON NEO KATO
    Interval{ 0x1D100, 0x1D126 }, // So  [39] MUSICAL SYMBOL SINGLE BARLINE..MUSICAL SYMBOL DRUM CLEF-2
    Interval{ 0x1D129, 0x1D164 }, // So  [60] MUSICAL SYMBOL MULTIPLE MEASURE REST..MUSICAL SYMBOL ONE HUNDRED TWENTY-EIGHTH NOTE
    Interval{ 0x1D166, 0x1D166 }, // Mc       MUSICAL SYMBOL COMBINING SPRECHGESANG STEM
    Interval{ 0x1D16A, 0x1D16C }, // So   [3] MUSICAL SYMBOL FINGERED TREMOLO-1..MUSICAL SYMBOL FINGERED TREMOLO-3
    Interval{ 0x1D16D, 0x1D16D }, // Mc       MUSICAL SYMBOL COMBINING AUGMENTATION DOT
    Interval{ 0x1D183, 0x1D184 }, // So   [2] MUSICAL SYMBOL ARPEGGIATO UP..MUSICAL SYMBOL ARPEGGIATO DOWN
    Interval{ 0x1D18C, 0x1D1A9 }, // So  [30] MUSICAL SYMBOL RINFORZANDO..MUSICAL SYMBOL DEGREE SLASH
    Interval{ 0x1D1AE, 0x1D1EA }, // So  [61] MUSICAL SYMBOL PEDAL MARK..MUSICAL SYMBOL KORON
    Interval{ 0x1D200, 0x1D241 }, // So  [66] GREEK VOCAL NOTATION SYMBOL-1..GREEK INSTRUMENTAL NOTATION SYMBOL-54
    Interval{ 0x1D245, 0x1D245 }, // So       GREEK MUSICAL LEIMMA
    Interval{ 0x1D2C0, 0x1D2D3 }, // No  [20] KAKTOVIK NUMERAL ZERO..KAKTOVIK NUMERAL NINETEEN
    Interval{ 0x1D2E0, 0x1D2F3 }, // No  [20] MAYAN NUMERAL ZERO..MAYAN NUMERAL NINETEEN
    Interval{ 0x1D300, 0x1D356 }, // So  [87] MONOGRAM FOR EARTH..TETRAGRAM FOR FOSTERING
    Interval{ 0x1D360, 0x1D378 }, // No  [25] COUNTING ROD UNIT DIGIT ONE..TALLY MARK FIVE
    Interval{ 0x1D400, 0x1D454 }, // L&  [85] MATHEMATICAL BOLD CAPITAL A..MATHEMATICAL ITALIC SMALL G
    Interval{ 0x1D456, 0x1D49C }, // L&  [71] MATHEMATICAL ITALIC SMALL I..MATHEMATICAL SCRIPT CAPITAL A
    Interval{ 0x1D49E, 0x1D49F }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL C..MATHEMATICAL SCRIPT CAPITAL D
    Interval{ 0x1D4A2, 0x1D4A2 }, // L&       MATHEMATICAL SCRIPT CAPITAL G
    Interval{ 0x1D4A5, 0x1D4A6 }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL J..MATHEMATICAL SCRIPT CAPITAL K
    Interval{ 0x1D4A9, 0x1D4AC }, // L&   [4] MATHEMATICAL SCRIPT CAPITAL N..MATHEMATICAL SCRIPT CAPITAL Q
    Interval{ 0x1D4AE, 0x1D4B9 }, // L&  [12] MATHEMATICAL SCRIPT CAPITAL S..MATHEMATICAL SCRIPT SMALL D
    Interval{ 0x1D4BB, 0x1D4BB }, // L&       MATHEMATICAL SCRIPT SMALL F
    Interval{ 0x1D4BD, 0x1D4C3 }, // L&   [7] MATHEMATICAL SCRIPT SMALL H..MATHEMATICAL SCRIPT SMALL N
    Interval{ 0x1D4C5, 0x1D505 }, // L&  [65] MATHEMATICAL SCRIPT SMALL P..MATHEMATICAL FRAKTUR CAPITAL B
    Interval{ 0x1D507, 0x1D50A }, // L&   [4] MATHEMATICAL FRAKTUR CAPITAL D..MATHEMATICAL FRAKTUR CAPITAL G
    Interval{ 0x1D50D, 0x1D514 }, // L&   [8] MATHEMATICAL FRAKTUR CAPITAL J..MATHEMATICAL FRAKTUR CAPITAL Q
    Interval{ 0x1D516, 0x1D51C }, // L&   [7] MATHEMATICAL FRAKTUR CAPITAL S..MATHEMATICAL FRAKTUR CAPITAL Y
    Interval{ 0x1D51E, 0x1D539 }, // L&  [28] MATHEMATICAL FRAKTUR SMALL A..MATHEMATICAL DOUBLE-STRUCK CAPITAL B
    Interval{ 0x1D53B, 0x1D53E }, // L&   [4] MATHEMATICAL DOUBLE-STRUCK CAPITAL D..MATHEMATICAL DOUBLE-STRUCK CAPITAL G
    Interval{ 0x1D540, 0x1D544 }, // L&   [5] MATHEMATICAL DOUBLE-STRUCK CAPITAL I..MATHEMATICAL DOUBLE-STRUCK CAPITAL M
    Interval{ 0x1D546, 0x1D546 }, // L&       MATHEMATICAL DOUBLE-STRUCK CAPITAL O
    Interval{ 0x1D54A, 0x1D550 }, // L&   [7] MATHEMATICAL DOUBLE-STRUCK CAPITAL S..MATHEMATICAL DOUBLE-STRUCK CAPITAL Y
    Interval{ 0x1D552, 0x1D6A5 }, // L& [340] MATHEMATICAL DOUBLE-STRUCK SMALL A..MATHEMATICAL ITALIC SMALL DOTLESS J
    Interval{ 0x1D6A8, 0x1D6C0 }, // L&  [25] MATHEMATICAL BOLD CAPITAL ALPHA..MATHEMATICAL BOLD CAPITAL OMEGA
    Interval{ 0x1D6C1, 0x1D6C1 }, // Sm       MATHEMATICAL BOLD NABLA
    Interval{ 0x1D6C2, 0x1D6DA }, // L&  [25] MATHEMATICAL BOLD SMALL ALPHA..MATHEMATICAL BOLD SMALL OMEGA
    Interval{ 0x1D6DB, 0x1D6DB }, // Sm       MATHEMATICAL BOLD PARTIAL DIFFERENTIAL
    Interval{ 0x1D6DC, 0x1D6FA }, // L&  [31] MATHEMATICAL BOLD EPSILON SYMBOL..MATHEMATICAL ITALIC CAPITAL OMEGA
    Interval{ 0x1D6FB, 0x1D6FB }, // Sm       MATHEMATICAL ITALIC NABLA
    Interval{ 0x1D6FC, 0x1D714 }, // L&  [25] MATHEMATICAL ITALIC SMALL ALPHA..MATHEMATICAL ITALIC SMALL OMEGA
    Interval{ 0x1D715, 0x1D715 }, // Sm       MATHEMATICAL ITALIC PARTIAL DIFFERENTIAL
    Interval{ 0x1D716, 0x1D734 }, // L&  [31] MATHEMATICAL ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D735, 0x1D735 }, // Sm       MATHEMATICAL BOLD ITALIC NABLA
    Interval{ 0x1D736, 0x1D74E }, // L&  [25] MATHEMATICAL BOLD ITALIC SMALL ALPHA..MATHEMATICAL BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D74F, 0x1D74F }, // Sm       MATHEMATICAL BOLD ITALIC PARTIAL DIFFERENTIAL
    Interval{ 0x1D750, 0x1D76E }, // L&  [31] MATHEMATICAL BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGA
    Interval{ 0x1D76F, 0x1D76F }, // Sm       MATHEMATICAL SANS-SERIF BOLD NABLA
    Interval{ 0x1D770, 0x1D788 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD SMALL OMEGA
    Interval{ 0x1D789, 0x1D789 }, // Sm       MATHEMATICAL SANS-SERIF BOLD PARTIAL DIFFERENTIAL
    Interval{ 0x1D78A, 0x1D7A8 }, // L&  [31] MATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D7A9, 0x1D7A9 }, // Sm       MATHEMATICAL SANS-SERIF BOLD ITALIC NABLA
    Interval{ 0x1D7AA, 0x1D7C2 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D7C3, 0x1D7C3 }, // Sm       MATHEMATICAL SANS-SERIF BOLD ITALIC PARTIAL DIFFERENTIAL
    Interval{ 0x1D7C4, 0x1D7CB }, // L&   [8] MATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD SMALL DIGAMMA
    Interval{ 0x1D7CE, 0x1D7FF }, // Nd  [50] MATHEMATICAL BOLD DIGIT ZERO..MATHEMATICAL MONOSPACE DIGIT NINE
    Interval{ 0x1D800, 0x1D9FF }, // So [512] SIGNWRITING HAND-FIST INDEX..SIGNWRITING HEAD
    Interval{ 0x1DA37, 0x1DA3A }, // So   [4] SIGNWRITING AIR BLOW SMALL ROTATIONS..SIGNWRITING BREATH EXHALE
    Interval{ 0x1DA6D, 0x1DA74 }, // So   [8] SIGNWRITING SHOULDER HIP SPINE..SIGNWRITING TORSO-FLOORPLANE TWISTING
    Interval{ 0x1DA76, 0x1DA83 }, // So  [14] SIGNWRITING LIMB COMBINATION..SIGNWRITING LOCATION DEPTH
    Interval{ 0x1DA85, 0x1DA86 }, // So   [2] SIGNWRITING LOCATION TORSO..SIGNWRITING LOCATION LIMBS DIGITS
    Interval{ 0x1DA87, 0x1DA8B }, // Po   [5] SIGNWRITING COMMA..SIGNWRITING PARENTHESIS
    Interval{ 0x1DF00, 0x1DF09 }, // L&  [10] LATIN SMALL LETTER FENG DIGRAPH WITH TRILL..LATIN SMALL LETTER T WITH HOOK AND RETROFLEX HOOK
    Interval{ 0x1DF0A, 0x1DF0A }, // Lo       LATIN LETTER RETROFLEX CLICK WITH RETROFLEX HOOK
    Interval{ 0x1DF0B, 0x1DF1E }, // L&  [20] LATIN SMALL LETTER ESH WITH DOUBLE BAR..LATIN SMALL LETTER S WITH CURL
    Interval{ 0x1DF25, 0x1DF2A }, // L&   [6] LATIN SMALL LETTER D WITH MID-HEIGHT LEFT HOOK..LATIN SMALL LETTER T WITH MID-HEIGHT LEFT HOOK
    Interval{ 0x1E030, 0x1E06D }, // Lm  [62] MODIFIER LETTER CYRILLIC SMALL A..MODIFIER LETTER CYRILLIC SMALL STRAIGHT U WITH STROKE
    Interval{ 0x1E100, 0x1E12C }, // Lo  [45] NYIAKENG PUACHUE HMONG LETTER MA..NYIAKENG PUACHUE HMONG LETTER W
    Interval{ 0x1E137, 0x1E13D }, // Lm   [7] NYIAKENG PUACHUE HMONG SIGN FOR PERSON..NYIAKENG PUACHUE HMONG SYLLABLE LENGTHENER
    Interval{ 0x1E140, 0x1E149 }, // Nd  [10] NYIAKENG PUACHUE HMONG DIGIT ZERO..NYIAKENG PUACHUE HMONG DIGIT NINE
    Interval{ 0x1E14E, 0x1E14E }, // Lo       NYIAKENG PUACHUE HMONG LOGOGRAM NYAJ
    Interval{ 0x1E14F, 0x1E14F }, // So       NYIAKENG PUACHUE HMONG CIRCLED CA
    Interval{ 0x1E290, 0x1E2AD }, // Lo  [30] TOTO LETTER PA..TOTO LETTER A
    Interval{ 0x1E2C0, 0x1E2EB }, // Lo  [44] WANCHO LETTER AA..WANCHO LETTER YIH
    Interval{ 0x1E2F0, 0x1E2F9 }, // Nd  [10] WANCHO DIGIT ZERO..WANCHO DIGIT NINE
    Interval{ 0x1E2FF, 0x1E2FF }, // Sc       WANCHO NGUN SIGN
    Interval{ 0x1E4D0, 0x1E4EA }, // Lo  [27] NAG MUNDARI LETTER O..NAG MUNDARI LETTER ELL
    Interval{ 0x1E4EB, 0x1E4EB }, // Lm       NAG MUNDARI SIGN OJOD
    Interval{ 0x1E4F0, 0x1E4F9 }, // Nd  [10] NAG MUNDARI DIGIT ZERO..NAG MUNDARI DIGIT NINE
    Interval{ 0x1E7E0, 0x1E7E6 }, // Lo   [7] ETHIOPIC SYLLABLE HHYA..ETHIOPIC SYLLABLE HHYO
    Interval{ 0x1E7E8, 0x1E7EB }, // Lo   [4] ETHIOPIC SYLLABLE GURAGE HHWA..ETHIOPIC SYLLABLE HHWE
    Interval{ 0x1E7ED, 0x1E7EE }, // Lo   [2] ETHIOPIC SYLLABLE GURAGE MWI..ETHIOPIC SYLLABLE GURAGE MWEE
    Interval{ 0x1E7F0, 0x1E7FE }, // Lo  [15] ETHIOPIC SYLLABLE GURAGE QWI..ETHIOPIC SYLLABLE GURAGE PWEE
    Interval{ 0x1E800, 0x1E8C4 }, // Lo [197] MENDE KIKAKUI SYLLABLE M001 KI..MENDE KIKAKUI SYLLABLE M060 NYON
    Interval{ 0x1E8C7, 0x1E8CF }, // No   [9] MENDE KIKAKUI DIGIT ONE..MENDE KIKAKUI DIGIT NINE
    Interval{ 0x1E900, 0x1E943 }, // L&  [68] ADLAM CAPITAL LETTER ALIF..ADLAM SMALL LETTER SHA
    Interval{ 0x1E94B, 0x1E94B }, // Lm       ADLAM NASALIZATION MARK
    Interval{ 0x1E950, 0x1E959 }, // Nd  [10] ADLAM DIGIT ZERO..ADLAM DIGIT NINE
    Interval{ 0x1E95E, 0x1E95F }, // Po   [2] ADLAM INITIAL EXCLAMATION MARK..ADLAM INITIAL QUESTION MARK
    Interval{ 0x1EC71, 0x1ECAB }, // No  [59] INDIC SIYAQ NUMBER ONE..INDIC SIYAQ NUMBER PREFIXED NINE
    Interval{ 0x1ECAC, 0x1ECAC }, // So       INDIC SIYAQ PLACEHOLDER
    Interval{ 0x1ECAD, 0x1ECAF }, // No   [3] INDIC SIYAQ FRACTION ONE QUARTER..INDIC SIYAQ FRACTION THREE QUARTERS
    Interval{ 0x1ECB0, 0x1ECB0 }, // Sc       INDIC SIYAQ RUPEE MARK
    Interval{ 0x1ECB1, 0x1ECB4 }, // No   [4] INDIC SIYAQ NUMBER ALTERNATE ONE..INDIC SIYAQ ALTERNATE LAKH MARK
    Interval{ 0x1ED01, 0x1ED2D }, // No  [45] OTTOMAN SIYAQ NUMBER ONE..OTTOMAN SIYAQ NUMBER NINETY THOUSAND
    Interval{ 0x1ED2E, 0x1ED2E }, // So       OTTOMAN SIYAQ MARRATAN
    Interval{ 0x1ED2F, 0x1ED3D }, // No  [15] OTTOMAN SIYAQ ALTERNATE NUMBER TWO..OTTOMAN SIYAQ FRACTION ONE SIXTH
    Interval{ 0x1EE00, 0x1EE03 }, // Lo   [4] ARABIC MATHEMATICAL ALEF..ARABIC MATHEMATICAL DAL
    Interval{ 0x1EE05, 0x1EE1F }, // Lo  [27] ARABIC MATHEMATICAL WAW..ARABIC MATHEMATICAL DOTLESS QAF
    Interval{ 0x1EE21, 0x1EE22 }, // Lo   [2] ARABIC MATHEMATICAL INITIAL BEH..ARABIC MATHEMATICAL INITIAL JEEM
    Interval{ 0x1EE24, 0x1EE24 }, // Lo       ARABIC MATHEMATICAL INITIAL HEH
    Interval{ 0x1EE27, 0x1EE27 }, // Lo       ARABIC MATHEMATICAL INITIAL HAH
    Interval{ 0x1EE29, 0x1EE32 }, // Lo  [10] ARABIC MATHEMATICAL INITIAL YEH..ARABIC MATHEMATICAL INITIAL QAF
    Interval{ 0x1EE34, 0x1EE37 }, // Lo   [4] ARABIC MATHEMATICAL INITIAL SHEEN..ARABIC MATHEMATICAL INITIAL KHAH
    Interval{ 0x1EE39, 0x1EE39 }, // Lo       ARABIC MATHEMATICAL INITIAL DAD
    Interval{ 0x1EE3B, 0x1EE3B }, // Lo       ARABIC MATHEMATICAL INITIAL GHAIN
    Interval{ 0x1EE42, 0x1EE42 }, // Lo       ARABIC MATHEMATICAL TAILED JEEM
    Interval{ 0x1EE47, 0x1EE47 }, // Lo       ARABIC MATHEMATICAL TAILED HAH
    Interval{ 0x1EE49, 0x1EE49 }, // Lo       ARABIC MATHEMATICAL TAILED YEH
    Interval{ 0x1EE4B, 0x1EE4B }, // Lo       ARABIC MATHEMATICAL TAILED LAM
    Interval{ 0x1EE4D, 0x1EE4F }, // Lo   [3] ARABIC MATHEMATICAL TAILED NOON..ARABIC MATHEMATICAL TAILED AIN
    Interval{ 0x1EE51, 0x1EE52 }, // Lo   [2] ARABIC MATHEMATICAL TAILED SAD..ARABIC MATHEMATICAL TAILED QAF
    Interval{ 0x1EE54, 0x1EE54 }, // Lo       ARABIC MATHEMATICAL TAILED SHEEN
    Interval{ 0x1EE57, 0x1EE57 }, // Lo       ARABIC MATHEMATICAL TAILED KHAH
    Interval{ 0x1EE59, 0x1EE59 }, // Lo       ARABIC MATHEMATICAL TAILED DAD
    Interval{ 0x1EE5B, 0x1EE5B }, // Lo       ARABIC MATHEMATICAL TAILED GHAIN
    Interval{ 0x1EE5D, 0x1EE5D }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS NOON
    Interval{ 0x1EE5F, 0x1EE5F }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS QAF
    Interval{ 0x1EE61, 0x1EE62 }, // Lo   [2] ARABIC MATHEMATICAL STRETCHED BEH..ARABIC MATHEMATICAL STRETCHED JEEM
    Interval{ 0x1EE64, 0x1EE64 }, // Lo       ARABIC MATHEMATICAL STRETCHED HEH
    Interval{ 0x1EE67, 0x1EE6A }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED HAH..ARABIC MATHEMATICAL STRETCHED KAF
    Interval{ 0x1EE6C, 0x1EE72 }, // Lo   [7] ARABIC MATHEMATICAL STRETCHED MEEM..ARABIC MATHEMATICAL STRETCHED QAF
    Interval{ 0x1EE74, 0x1EE77 }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED SHEEN..ARABIC MATHEMATICAL STRETCHED KHAH
    Interval{ 0x1EE79, 0x1EE7C }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED DAD..ARABIC MATHEMATICAL STRETCHED DOTLESS BEH
    Interval{ 0x1EE7E, 0x1EE7E }, // Lo       ARABIC MATHEMATICAL STRETCHED DOTLESS FEH
    Interval{ 0x1EE80, 0x1EE89 }, // Lo  [10] ARABIC MATHEMATICAL LOOPED ALEF..ARABIC MATHEMATICAL LOOPED YEH
    Interval{ 0x1EE8B, 0x1EE9B }, // Lo  [17] ARABIC MATHEMATICAL LOOPED LAM..ARABIC MATHEMATICAL LOOPED GHAIN
    Interval{ 0x1EEA1, 0x1EEA3 }, // Lo   [3] ARABIC MATHEMATICAL DOUBLE-STRUCK BEH..ARABIC MATHEMATICAL DOUBLE-STRUCK DAL
    Interval{ 0x1EEA5, 0x1EEA9 }, // Lo   [5] ARABIC MATHEMATICAL DOUBLE-STRUCK WAW..ARABIC MATHEMATICAL DOUBLE-STRUCK YEH
    Interval{ 0x1EEAB, 0x1EEBB }, // Lo  [17] ARABIC MATHEMATICAL DOUBLE-STRUCK LAM..ARABIC MATHEMATICAL DOUBLE-STRUCK GHAIN
    Interval{ 0x1EEF0, 0x1EEF1 }, // Sm   [2] ARABIC MATHEMATICAL OPERATOR MEEM WITH HAH WITH TATWEEL..ARABIC MATHEMATICAL OPERATOR HAH WITH DAL
    Interval{ 0x1F000, 0x1F02B }, // So  [44] MAHJONG TILE EAST WIND..MAHJONG TILE BACK
    Interval{ 0x1F030, 0x1F093 }, // So [100] DOMINO TILE HORIZONTAL BACK..DOMINO TILE VERTICAL-06-06
    Interval{ 0x1F0A0, 0x1F0AE }, // So  [15] PLAYING CARD BACK..PLAYING CARD KING OF SPADES
    Interval{ 0x1F0B1, 0x1F0BF }, // So  [15] PLAYING CARD ACE OF HEARTS..PLAYING CARD RED JOKER
    Interval{ 0x1F0C1, 0x1F0CF }, // So  [15] PLAYING CARD ACE OF DIAMONDS..PLAYING CARD BLACK JOKER
    Interval{ 0x1F0D1, 0x1F0F5 }, // So  [37] PLAYING CARD ACE OF CLUBS..PLAYING CARD TRUMP-21
    Interval{ 0x1F100, 0x1F10C }, // No  [13] DIGIT ZERO FULL STOP..DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ZERO
    Interval{ 0x1F10D, 0x1F1AD }, // So [161] CIRCLED ZERO WITH SLASH..MASK WORK SYMBOL
    Interval{ 0x1F1E6, 0x1F202 }, // So  [29] REGIONAL INDICATOR SYMBOL LETTER A..SQUARED KATAKANA SA
    Interval{ 0x1F210, 0x1F23B }, // So  [44] SQUARED CJK UNIFIED IDEOGRAPH-624B..SQUARED CJK UNIFIED IDEOGRAPH-914D
    Interval{ 0x1F240, 0x1F248 }, // So   [9] TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-672C..TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-6557
    Interval{ 0x1F250, 0x1F251 }, // So   [2] CIRCLED IDEOGRAPH ADVANTAGE..CIRCLED IDEOGRAPH ACCEPT
    Interval{ 0x1F260, 0x1F265 }, // So   [6] ROUNDED SYMBOL FOR FU..ROUNDED SYMBOL FOR CAI
    Interval{ 0x1F300, 0x1F3FA }, // So [251] CYCLONE..AMPHORA
    Interval{ 0x1F3FB, 0x1F3FF }, // Sk   [5] EMOJI MODIFIER FITZPATRICK TYPE-1-2..EMOJI MODIFIER FITZPATRICK TYPE-6
    Interval{ 0x1F400, 0x1F6D7 }, // So [728] RAT..ELEVATOR
    Interval{ 0x1F6DC, 0x1F6EC }, // So  [17] WIRELESS..AIRPLANE ARRIVING
    Interval{ 0x1F6F0, 0x1F6FC }, // So  [13] SATELLITE..ROLLER SKATE
    Interval{ 0x1F700, 0x1F776 }, // So [119] ALCHEMICAL SYMBOL FOR QUINTESSENCE..LUNAR ECLIPSE
    Interval{ 0x1F77B, 0x1F7D9 }, // So  [95] HAUMEA..NINE POINTED WHITE STAR
    Interval{ 0x1F7E0, 0x1F7EB }, // So  [12] LARGE ORANGE CIRCLE..LARGE BROWN SQUARE
    Interval{ 0x1F7F0, 0x1F7F0 }, // So       HEAVY EQUALS SIGN
    Interval{ 0x1F800, 0x1F80B }, // So  [12] LEFTWARDS ARROW WITH SMALL TRIANGLE ARROWHEAD..DOWNWARDS ARROW WITH LARGE TRIANGLE ARROWHEAD
    Interval{ 0x1F810, 0x1F847 }, // So  [56] LEFTWARDS ARROW WITH SMALL EQUILATERAL ARROWHEAD..DOWNWARDS HEAVY ARROW
    Interval{ 0x1F850, 0x1F859 }, // So  [10] LEFTWARDS SANS-SERIF ARROW..UP DOWN SANS-SERIF ARROW
    Interval{ 0x1F860, 0x1F887 }, // So  [40] WIDE-HEADED LEFTWARDS LIGHT BARB ARROW..WIDE-HEADED SOUTH WEST VERY HEAVY BARB ARROW
    Interval{ 0x1F890, 0x1F8AD }, // So  [30] LEFTWARDS TRIANGLE ARROWHEAD..WHITE ARROW SHAFT WIDTH TWO THIRDS
    Interval{ 0x1F8B0, 0x1F8B1 }, // So   [2] ARROW POINTING UPWARDS THEN NORTH WEST..ARROW POINTING RIGHTWARDS THEN CURVING SOUTH WEST
    Interval{ 0x1F900, 0x1FA53 }, // So [340] CIRCLED CROSS FORMEE WITH FOUR DOTS..BLACK CHESS KNIGHT-BISHOP
    Interval{ 0x1FA60, 0x1FA6D }, // So  [14] XIANGQI RED GENERAL..XIANGQI BLACK SOLDIER
    Interval{ 0x1FA70, 0x1FA7C }, // So  [13] BALLET SHOES..CRUTCH
    Interval{ 0x1FA80, 0x1FA88 }, // So   [9] YO-YO..FLUTE
    Interval{ 0x1FA90, 0x1FABD }, // So  [46] RINGED PLANET..WING
    Interval{ 0x1FABF, 0x1FAC5 }, // So   [7] GOOSE..PERSON WITH CROWN
    Interval{ 0x1FACE, 0x1FADB }, // So  [14] MOOSE..PEA POD
    Interval{ 0x1FAE0, 0x1FAE8 }, // So   [9] MELTING FACE..SHAKING FACE
    Interval{ 0x1FAF0, 0x1FAF8 }, // So   [9] HAND WITH INDEX FINGER AND THUMB CROSSED..RIGHTWARDS PUSHING HAND
    Interval{ 0x1FB00, 0x1FB92 }, // So [147] BLOCK SEXTANT-1..UPPER HALF INVERSE MEDIUM SHADE AND LOWER HALF BLOCK
    Interval{ 0x1FB94, 0x1FBCA }, // So  [55] LEFT HALF INVERSE MEDIUM SHADE AND RIGHT HALF BLOCK..WHITE UP-POINTING CHEVRON
    Interval{ 0x1FBF0, 0x1FBF9 }, // Nd  [10] SEGMENTED DIGIT ZERO..SEGMENTED DIGIT NINE
    Interval{ 0x20000, 0x2A6DF }, // Lo [42720] CJK UNIFIED IDEOGRAPH-20000..CJK UNIFIED IDEOGRAPH-2A6DF
    Interval{ 0x2A700, 0x2B739 }, // Lo [4154] CJK UNIFIED IDEOGRAPH-2A700..CJK UNIFIED IDEOGRAPH-2B739
    Interval{ 0x2B740, 0x2B81D }, // Lo [222] CJK UNIFIED IDEOGRAPH-2B740..CJK UNIFIED IDEOGRAPH-2B81D
    Interval{ 0x2B820, 0x2CEA1 }, // Lo [5762] CJK UNIFIED IDEOGRAPH-2B820..CJK UNIFIED IDEOGRAPH-2CEA1
    Interval{ 0x2CEB0, 0x2EBE0 }, // Lo [7473] CJK UNIFIED IDEOGRAPH-2CEB0..CJK UNIFIED IDEOGRAPH-2EBE0
    Interval{ 0x2F800, 0x2FA1D }, // Lo [542] CJK COMPATIBILITY IDEOGRAPH-2F800..CJK COMPATIBILITY IDEOGRAPH-2FA1D
    Interval{ 0x30000, 0x3134A }, // Lo [4939] CJK UNIFIED IDEOGRAPH-30000..CJK UNIFIED IDEOGRAPH-3134A
    Interval{ 0x31350, 0x323AF }, // Lo [4192] CJK UNIFIED IDEOGRAPH-31350..CJK UNIFIED IDEOGRAPH-323AF
}; // }}}
auto static const Grapheme_Extend = std::array<Interval, 376>{ // {{{
    Interval{ 0x0300, 0x036F }, // Mn [112] COMBINING GRAVE ACCENT..COMBINING LATIN SMALL LETTER X
    Interval{ 0x0483, 0x0487 }, // Mn   [5] COMBINING CYRILLIC TITLO..COMBINING CYRILLIC POKRYTIE
    Interval{ 0x0488, 0x0489 }, // Me   [2] COMBINING CYRILLIC HUNDRED THOUSANDS SIGN..COMBINING CYRILLIC MILLIONS SIGN
    Interval{ 0x0591, 0x05BD }, // Mn  [45] HEBREW ACCENT ETNAHTA..HEBREW POINT METEG
    Interval{ 0x05BF, 0x05BF }, // Mn       HEBREW POINT RAFE
    Interval{ 0x05C1, 0x05C2 }, // Mn   [2] HEBREW POINT SHIN DOT..HEBREW POINT SIN DOT
    Interval{ 0x05C4, 0x05C5 }, // Mn   [2] HEBREW MARK UPPER DOT..HEBREW MARK LOWER DOT
    Interval{ 0x05C7, 0x05C7 }, // Mn       HEBREW POINT QAMATS QATAN
    Interval{ 0x0610, 0x061A }, // Mn  [11] ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM..ARABIC SMALL KASRA
    Interval{ 0x064B, 0x065F }, // Mn  [21] ARABIC FATHATAN..ARABIC WAVY HAMZA BELOW
    Interval{ 0x0670, 0x0670 }, // Mn       ARABIC LETTER SUPERSCRIPT ALEF
    Interval{ 0x06D6, 0x06DC }, // Mn   [7] ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA..ARABIC SMALL HIGH SEEN
    Interval{ 0x06DF, 0x06E4 }, // Mn   [6] ARABIC SMALL HIGH ROUNDED ZERO..ARABIC SMALL HIGH MADDA
    Interval{ 0x06E7, 0x06E8 }, // Mn   [2] ARABIC SMALL HIGH YEH..ARABIC SMALL HIGH NOON
    Interval{ 0x06EA, 0x06ED }, // Mn   [4] ARABIC EMPTY CENTRE LOW STOP..ARABIC SMALL LOW MEEM
    Interval{ 0x0711, 0x0711 }, // Mn       SYRIAC LETTER SUPERSCRIPT ALAPH
    Interval{ 0x0730, 0x074A }, // Mn  [27] SYRIAC PTHAHA ABOVE..SYRIAC BARREKH
    Interval{ 0x07A6, 0x07B0 }, // Mn  [11] THAANA ABAFILI..THAANA SUKUN
    Interval{ 0x07EB, 0x07F3 }, // Mn   [9] NKO COMBINING SHORT HIGH TONE..NKO COMBINING DOUBLE DOT ABOVE
    Interval{ 0x07FD, 0x07FD }, // Mn       NKO DANTAYALAN
    Interval{ 0x0816, 0x0819 }, // Mn   [4] SAMARITAN MARK IN..SAMARITAN MARK DAGESH
    Interval{ 0x081B, 0x0823 }, // Mn   [9] SAMARITAN MARK EPENTHETIC YUT..SAMARITAN VOWEL SIGN A
    Interval{ 0x0825, 0x0827 }, // Mn   [3] SAMARITAN VOWEL SIGN SHORT A..SAMARITAN VOWEL SIGN U
    Interval{ 0x0829, 0x082D }, // Mn   [5] SAMARITAN VOWEL SIGN LONG I..SAMARITAN MARK NEQUDAA
    Interval{ 0x0859, 0x085B }, // Mn   [3] MANDAIC AFFRICATION MARK..MANDAIC GEMINATION MARK
    Interval{ 0x0898, 0x089F }, // Mn   [8] ARABIC SMALL HIGH WORD AL-JUZ..ARABIC HALF MADDA OVER MADDA
    Interval{ 0x08CA, 0x08E1 }, // Mn  [24] ARABIC SMALL HIGH FARSI YEH..ARABIC SMALL HIGH SIGN SAFHA
    Interval{ 0x08E3, 0x0902 }, // Mn  [32] ARABIC TURNED DAMMA BELOW..DEVANAGARI SIGN ANUSVARA
    Interval{ 0x093A, 0x093A }, // Mn       DEVANAGARI VOWEL SIGN OE
    Interval{ 0x093C, 0x093C }, // Mn       DEVANAGARI SIGN NUKTA
    Interval{ 0x0941, 0x0948 }, // Mn   [8] DEVANAGARI VOWEL SIGN U..DEVANAGARI VOWEL SIGN AI
    Interval{ 0x094D, 0x094D }, // Mn       DEVANAGARI SIGN VIRAMA
    Interval{ 0x0951, 0x0957 }, // Mn   [7] DEVANAGARI STRESS SIGN UDATTA..DEVANAGARI VOWEL SIGN UUE
    Interval{ 0x0962, 0x0963 }, // Mn   [2] DEVANAGARI VOWEL SIGN VOCALIC L..DEVANAGARI VOWEL SIGN VOCALIC LL
    Interval{ 0x0981, 0x0981 }, // Mn       BENGALI SIGN CANDRABINDU
    Interval{ 0x09BC, 0x09BC }, // Mn       BENGALI SIGN NUKTA
    Interval{ 0x09BE, 0x09BE }, // Mc       BENGALI VOWEL SIGN AA
    Interval{ 0x09C1, 0x09C4 }, // Mn   [4] BENGALI VOWEL SIGN U..BENGALI VOWEL SIGN VOCALIC RR
    Interval{ 0x09CD, 0x09CD }, // Mn       BENGALI SIGN VIRAMA
    Interval{ 0x09D7, 0x09D7 }, // Mc       BENGALI AU LENGTH MARK
    Interval{ 0x09E2, 0x09E3 }, // Mn   [2] BENGALI VOWEL SIGN VOCALIC L..BENGALI VOWEL SIGN VOCALIC LL
    Interval{ 0x09FE, 0x09FE }, // Mn       BENGALI SANDHI MARK
    Interval{ 0x0A01, 0x0A02 }, // Mn   [2] GURMUKHI SIGN ADAK BINDI..GURMUKHI SIGN BINDI
    Interval{ 0x0A3C, 0x0A3C }, // Mn       GURMUKHI SIGN NUKTA
    Interval{ 0x0A41, 0x0A42 }, // Mn   [2] GURMUKHI VOWEL SIGN U..GURMUKHI VOWEL SIGN UU
    Interval{ 0x0A47, 0x0A48 }, // Mn   [2] GURMUKHI VOWEL SIGN EE..GURMUKHI VOWEL SIGN AI
    Interval{ 0x0A4B, 0x0A4D }, // Mn   [3] GURMUKHI VOWEL SIGN OO..GURMUKHI SIGN VIRAMA
    Interval{ 0x0A51, 0x0A51 }, // Mn       GURMUKHI SIGN UDAAT
    Interval{ 0x0A70, 0x0A71 }, // Mn   [2] GURMUKHI TIPPI..GURMUKHI ADDAK
    Interval{ 0x0A75, 0x0A75 }, // Mn       GURMUKHI SIGN YAKASH
    Interval{ 0x0A81, 0x0A82 }, // Mn   [2] GUJARATI SIGN CANDRABINDU..GUJARATI SIGN ANUSVARA
    Interval{ 0x0ABC, 0x0ABC }, // Mn       GUJARATI SIGN NUKTA
    Interval{ 0x0AC1, 0x0AC5 }, // Mn   [5] GUJARATI VOWEL SIGN U..GUJARATI VOWEL SIGN CANDRA E
    Interval{ 0x0AC7, 0x0AC8 }, // Mn   [2] GUJARATI VOWEL SIGN E..GUJARATI VOWEL SIGN AI
    Interval{ 0x0ACD, 0x0ACD }, // Mn       GUJARATI SIGN VIRAMA
    Interval{ 0x0AE2, 0x0AE3 }, // Mn   [2] GUJARATI VOWEL SIGN VOCALIC L..GUJARATI VOWEL SIGN VOCALIC LL
    Interval{ 0x0AFA, 0x0AFF }, // Mn   [6] GUJARATI SIGN SUKUN..GUJARATI SIGN TWO-CIRCLE NUKTA ABOVE
    Interval{ 0x0B01, 0x0B01 }, // Mn       ORIYA SIGN CANDRABINDU
    Interval{ 0x0B3C, 0x0B3C }, // Mn       ORIYA SIGN NUKTA
    Interval{ 0x0B3E, 0x0B3E }, // Mc       ORIYA VOWEL SIGN AA
    Interval{ 0x0B3F, 0x0B3F }, // Mn       ORIYA VOWEL SIGN I
    Interval{ 0x0B41, 0x0B44 }, // Mn   [4] ORIYA VOWEL SIGN U..ORIYA VOWEL SIGN VOCALIC RR
    Interval{ 0x0B4D, 0x0B4D }, // Mn       ORIYA SIGN VIRAMA
    Interval{ 0x0B55, 0x0B56 }, // Mn   [2] ORIYA SIGN OVERLINE..ORIYA AI LENGTH MARK
    Interval{ 0x0B57, 0x0B57 }, // Mc       ORIYA AU LENGTH MARK
    Interval{ 0x0B62, 0x0B63 }, // Mn   [2] ORIYA VOWEL SIGN VOCALIC L..ORIYA VOWEL SIGN VOCALIC LL
    Interval{ 0x0B82, 0x0B82 }, // Mn       TAMIL SIGN ANUSVARA
    Interval{ 0x0BBE, 0x0BBE }, // Mc       TAMIL VOWEL SIGN AA
    Interval{ 0x0BC0, 0x0BC0 }, // Mn       TAMIL VOWEL SIGN II
    Interval{ 0x0BCD, 0x0BCD }, // Mn       TAMIL SIGN VIRAMA
    Interval{ 0x0BD7, 0x0BD7 }, // Mc       TAMIL AU LENGTH MARK
    Interval{ 0x0C00, 0x0C00 }, // Mn       TELUGU SIGN COMBINING CANDRABINDU ABOVE
    Interval{ 0x0C04, 0x0C04 }, // Mn       TELUGU SIGN COMBINING ANUSVARA ABOVE
    Interval{ 0x0C3C, 0x0C3C }, // Mn       TELUGU SIGN NUKTA
    Interval{ 0x0C3E, 0x0C40 }, // Mn   [3] TELUGU VOWEL SIGN AA..TELUGU VOWEL SIGN II
    Interval{ 0x0C46, 0x0C48 }, // Mn   [3] TELUGU VOWEL SIGN E..TELUGU VOWEL SIGN AI
    Interval{ 0x0C4A, 0x0C4D }, // Mn   [4] TELUGU VOWEL SIGN O..TELUGU SIGN VIRAMA
    Interval{ 0x0C55, 0x0C56 }, // Mn   [2] TELUGU LENGTH MARK..TELUGU AI LENGTH MARK
    Interval{ 0x0C62, 0x0C63 }, // Mn   [2] TELUGU VOWEL SIGN VOCALIC L..TELUGU VOWEL SIGN VOCALIC LL
    Interval{ 0x0C81, 0x0C81 }, // Mn       KANNADA SIGN CANDRABINDU
    Interval{ 0x0CBC, 0x0CBC }, // Mn       KANNADA SIGN NUKTA
    Interval{ 0x0CBF, 0x0CBF }, // Mn       KANNADA VOWEL SIGN I
    Interval{ 0x0CC2, 0x0CC2 }, // Mc       KANNADA VOWEL SIGN UU
    Interval{ 0x0CC6, 0x0CC6 }, // Mn       KANNADA VOWEL SIGN E
    Interval{ 0x0CCC, 0x0CCD }, // Mn   [2] KANNADA VOWEL SIGN AU..KANNADA SIGN VIRAMA
    Interval{ 0x0CD5, 0x0CD6 }, // Mc   [2] KANNADA LENGTH MARK..KANNADA AI LENGTH MARK
    Interval{ 0x0CE2, 0x0CE3 }, // Mn   [2] KANNADA VOWEL SIGN VOCALIC L..KANNADA VOWEL SIGN VOCALIC LL
    Interval{ 0x0D00, 0x0D01 }, // Mn   [2] MALAYALAM SIGN COMBINING ANUSVARA ABOVE..MALAYALAM SIGN CANDRABINDU
    Interval{ 0x0D3B, 0x0D3C }, // Mn   [2] MALAYALAM SIGN VERTICAL BAR VIRAMA..MALAYALAM SIGN CIRCULAR VIRAMA
    Interval{ 0x0D3E, 0x0D3E }, // Mc       MALAYALAM VOWEL SIGN AA
    Interval{ 0x0D41, 0x0D44 }, // Mn   [4] MALAYALAM VOWEL SIGN U..MALAYALAM VOWEL SIGN VOCALIC RR
    Interval{ 0x0D4D, 0x0D4D }, // Mn       MALAYALAM SIGN VIRAMA
    Interval{ 0x0D57, 0x0D57 }, // Mc       MALAYALAM AU LENGTH MARK
    Interval{ 0x0D62, 0x0D63 }, // Mn   [2] MALAYALAM VOWEL SIGN VOCALIC L..MALAYALAM VOWEL SIGN VOCALIC LL
    Interval{ 0x0D81, 0x0D81 }, // Mn       SINHALA SIGN CANDRABINDU
    Interval{ 0x0DCA, 0x0DCA }, // Mn       SINHALA SIGN AL-LAKUNA
    Interval{ 0x0DCF, 0x0DCF }, // Mc       SINHALA VOWEL SIGN AELA-PILLA
    Interval{ 0x0DD2, 0x0DD4 }, // Mn   [3] SINHALA VOWEL SIGN KETTI IS-PILLA..SINHALA VOWEL SIGN KETTI PAA-PILLA
    Interval{ 0x0DD6, 0x0DD6 }, // Mn       SINHALA VOWEL SIGN DIGA PAA-PILLA
    Interval{ 0x0DDF, 0x0DDF }, // Mc       SINHALA VOWEL SIGN GAYANUKITTA
    Interval{ 0x0E31, 0x0E31 }, // Mn       THAI CHARACTER MAI HAN-AKAT
    Interval{ 0x0E34, 0x0E3A }, // Mn   [7] THAI CHARACTER SARA I..THAI CHARACTER PHINTHU
    Interval{ 0x0E47, 0x0E4E }, // Mn   [8] THAI CHARACTER MAITAIKHU..THAI CHARACTER YAMAKKAN
    Interval{ 0x0EB1, 0x0EB1 }, // Mn       LAO VOWEL SIGN MAI KAN
    Interval{ 0x0EB4, 0x0EBC }, // Mn   [9] LAO VOWEL SIGN I..LAO SEMIVOWEL SIGN LO
    Interval{ 0x0EC8, 0x0ECE }, // Mn   [7] LAO TONE MAI EK..LAO YAMAKKAN
    Interval{ 0x0F18, 0x0F19 }, // Mn   [2] TIBETAN ASTROLOGICAL SIGN -KHYUD PA..TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS
    Interval{ 0x0F35, 0x0F35 }, // Mn       TIBETAN MARK NGAS BZUNG NYI ZLA
    Interval{ 0x0F37, 0x0F37 }, // Mn       TIBETAN MARK NGAS BZUNG SGOR RTAGS
    Interval{ 0x0F39, 0x0F39 }, // Mn       TIBETAN MARK TSA -PHRU
    Interval{ 0x0F71, 0x0F7E }, // Mn  [14] TIBETAN VOWEL SIGN AA..TIBETAN SIGN RJES SU NGA RO
    Interval{ 0x0F80, 0x0F84 }, // Mn   [5] TIBETAN VOWEL SIGN REVERSED I..TIBETAN MARK HALANTA
    Interval{ 0x0F86, 0x0F87 }, // Mn   [2] TIBETAN SIGN LCI RTAGS..TIBETAN SIGN YANG RTAGS
    Interval{ 0x0F8D, 0x0F97 }, // Mn  [11] TIBETAN SUBJOINED SIGN LCE TSA CAN..TIBETAN SUBJOINED LETTER JA
    Interval{ 0x0F99, 0x0FBC }, // Mn  [36] TIBETAN SUBJOINED LETTER NYA..TIBETAN SUBJOINED LETTER FIXED-FORM RA
    Interval{ 0x0FC6, 0x0FC6 }, // Mn       TIBETAN SYMBOL PADMA GDAN
    Interval{ 0x102D, 0x1030 }, // Mn   [4] MYANMAR VOWEL SIGN I..MYANMAR VOWEL SIGN UU
    Interval{ 0x1032, 0x1037 }, // Mn   [6] MYANMAR VOWEL SIGN AI..MYANMAR SIGN DOT BELOW
    Interval{ 0x1039, 0x103A }, // Mn   [2] MYANMAR SIGN VIRAMA..MYANMAR SIGN ASAT
    Interval{ 0x103D, 0x103E }, // Mn   [2] MYANMAR CONSONANT SIGN MEDIAL WA..MYANMAR CONSONANT SIGN MEDIAL HA
    Interval{ 0x1058, 0x1059 }, // Mn   [2] MYANMAR VOWEL SIGN VOCALIC L..MYANMAR VOWEL SIGN VOCALIC LL
    Interval{ 0x105E, 0x1060 }, // Mn   [3] MYANMAR CONSONANT SIGN MON MEDIAL NA..MYANMAR CONSONANT SIGN MON MEDIAL LA
    Interval{ 0x1071, 0x1074 }, // Mn   [4] MYANMAR VOWEL SIGN GEBA KAREN I..MYANMAR VOWEL SIGN KAYAH EE
    Interval{ 0x1082, 0x1082 }, // Mn       MYANMAR CONSONANT SIGN SHAN MEDIAL WA
    Interval{ 0x1085, 0x1086 }, // Mn   [2] MYANMAR VOWEL SIGN SHAN E ABOVE..MYANMAR VOWEL SIGN SHAN FINAL Y
    Interval{ 0x108D, 0x108D }, // Mn       MYANMAR SIGN SHAN COUNCIL EMPHATIC TONE
    Interval{ 0x109D, 0x109D }, // Mn       MYANMAR VOWEL SIGN AITON AI
    Interval{ 0x135D, 0x135F }, // Mn   [3] ETHIOPIC COMBINING GEMINATION AND VOWEL LENGTH MARK..ETHIOPIC COMBINING GEMINATION MARK
    Interval{ 0x1712, 0x1714 }, // Mn   [3] TAGALOG VOWEL SIGN I..TAGALOG SIGN VIRAMA
    Interval{ 0x1732, 0x1733 }, // Mn   [2] HANUNOO VOWEL SIGN I..HANUNOO VOWEL SIGN U
    Interval{ 0x1752, 0x1753 }, // Mn   [2] BUHID VOWEL SIGN I..BUHID VOWEL SIGN U
    Interval{ 0x1772, 0x1773 }, // Mn   [2] TAGBANWA VOWEL SIGN I..TAGBANWA VOWEL SIGN U
    Interval{ 0x17B4, 0x17B5 }, // Mn   [2] KHMER VOWEL INHERENT AQ..KHMER VOWEL INHERENT AA
    Interval{ 0x17B7, 0x17BD }, // Mn   [7] KHMER VOWEL SIGN I..KHMER VOWEL SIGN UA
    Interval{ 0x17C6, 0x17C6 }, // Mn       KHMER SIGN NIKAHIT
    Interval{ 0x17C9, 0x17D3 }, // Mn  [11] KHMER SIGN MUUSIKATOAN..KHMER SIGN BATHAMASAT
    Interval{ 0x17DD, 0x17DD }, // Mn       KHMER SIGN ATTHACAN
    Interval{ 0x180B, 0x180D }, // Mn   [3] MONGOLIAN FREE VARIATION SELECTOR ONE..MONGOLIAN FREE VARIATION SELECTOR THREE
    Interval{ 0x180F, 0x180F }, // Mn       MONGOLIAN FREE VARIATION SELECTOR FOUR
    Interval{ 0x1885, 0x1886 }, // Mn   [2] MONGOLIAN LETTER ALI GALI BALUDA..MONGOLIAN LETTER ALI GALI THREE BALUDA
    Interval{ 0x18A9, 0x18A9 }, // Mn       MONGOLIAN LETTER ALI GALI DAGALGA
    Interval{ 0x1920, 0x1922 }, // Mn   [3] LIMBU VOWEL SIGN A..LIMBU VOWEL SIGN U
    Interval{ 0x1927, 0x1928 }, // Mn   [2] LIMBU VOWEL SIGN E..LIMBU VOWEL SIGN O
    Interval{ 0x1932, 0x1932 }, // Mn       LIMBU SMALL LETTER ANUSVARA
    Interval{ 0x1939, 0x193B }, // Mn   [3] LIMBU SIGN MUKPHRENG..LIMBU SIGN SA-I
    Interval{ 0x1A17, 0x1A18 }, // Mn   [2] BUGINESE VOWEL SIGN I..BUGINESE VOWEL SIGN U
    Interval{ 0x1A1B, 0x1A1B }, // Mn       BUGINESE VOWEL SIGN AE
    Interval{ 0x1A56, 0x1A56 }, // Mn       TAI THAM CONSONANT SIGN MEDIAL LA
    Interval{ 0x1A58, 0x1A5E }, // Mn   [7] TAI THAM SIGN MAI KANG LAI..TAI THAM CONSONANT SIGN SA
    Interval{ 0x1A60, 0x1A60 }, // Mn       TAI THAM SIGN SAKOT
    Interval{ 0x1A62, 0x1A62 }, // Mn       TAI THAM VOWEL SIGN MAI SAT
    Interval{ 0x1A65, 0x1A6C }, // Mn   [8] TAI THAM VOWEL SIGN I..TAI THAM VOWEL SIGN OA BELOW
    Interval{ 0x1A73, 0x1A7C }, // Mn  [10] TAI THAM VOWEL SIGN OA ABOVE..TAI THAM SIGN KHUEN-LUE KARAN
    Interval{ 0x1A7F, 0x1A7F }, // Mn       TAI THAM COMBINING CRYPTOGRAMMIC DOT
    Interval{ 0x1AB0, 0x1ABD }, // Mn  [14] COMBINING DOUBLED CIRCUMFLEX ACCENT..COMBINING PARENTHESES BELOW
    Interval{ 0x1ABE, 0x1ABE }, // Me       COMBINING PARENTHESES OVERLAY
    Interval{ 0x1ABF, 0x1ACE }, // Mn  [16] COMBINING LATIN SMALL LETTER W BELOW..COMBINING LATIN SMALL LETTER INSULAR T
    Interval{ 0x1B00, 0x1B03 }, // Mn   [4] BALINESE SIGN ULU RICEM..BALINESE SIGN SURANG
    Interval{ 0x1B34, 0x1B34 }, // Mn       BALINESE SIGN REREKAN
    Interval{ 0x1B35, 0x1B35 }, // Mc       BALINESE VOWEL SIGN TEDUNG
    Interval{ 0x1B36, 0x1B3A }, // Mn   [5] BALINESE VOWEL SIGN ULU..BALINESE VOWEL SIGN RA REPA
    Interval{ 0x1B3C, 0x1B3C }, // Mn       BALINESE VOWEL SIGN LA LENGA
    Interval{ 0x1B42, 0x1B42 }, // Mn       BALINESE VOWEL SIGN PEPET
    Interval{ 0x1B6B, 0x1B73 }, // Mn   [9] BALINESE MUSICAL SYMBOL COMBINING TEGEH..BALINESE MUSICAL SYMBOL COMBINING GONG
    Interval{ 0x1B80, 0x1B81 }, // Mn   [2] SUNDANESE SIGN PANYECEK..SUNDANESE SIGN PANGLAYAR
    Interval{ 0x1BA2, 0x1BA5 }, // Mn   [4] SUNDANESE CONSONANT SIGN PANYAKRA..SUNDANESE VOWEL SIGN PANYUKU
    Interval{ 0x1BA8, 0x1BA9 }, // Mn   [2] SUNDANESE VOWEL SIGN PAMEPET..SUNDANESE VOWEL SIGN PANEULEUNG
    Interval{ 0x1BAB, 0x1BAD }, // Mn   [3] SUNDANESE SIGN VIRAMA..SUNDANESE CONSONANT SIGN PASANGAN WA
    Interval{ 0x1BE6, 0x1BE6 }, // Mn       BATAK SIGN TOMPI
    Interval{ 0x1BE8, 0x1BE9 }, // Mn   [2] BATAK VOWEL SIGN PAKPAK E..BATAK VOWEL SIGN EE
    Interval{ 0x1BED, 0x1BED }, // Mn       BATAK VOWEL SIGN KARO O
    Interval{ 0x1BEF, 0x1BF1 }, // Mn   [3] BATAK VOWEL SIGN U FOR SIMALUNGUN SA..BATAK CONSONANT SIGN H
    Interval{ 0x1C2C, 0x1C33 }, // Mn   [8] LEPCHA VOWEL SIGN E..LEPCHA CONSONANT SIGN T
    Interval{ 0x1C36, 0x1C37 }, // Mn   [2] LEPCHA SIGN RAN..LEPCHA SIGN NUKTA
    Interval{ 0x1CD0, 0x1CD2 }, // Mn   [3] VEDIC TONE KARSHANA..VEDIC TONE PRENKHA
    Interval{ 0x1CD4, 0x1CE0 }, // Mn  [13] VEDIC SIGN YAJURVEDIC MIDLINE SVARITA..VEDIC TONE RIGVEDIC KASHMIRI INDEPENDENT SVARITA
    Interval{ 0x1CE2, 0x1CE8 }, // Mn   [7] VEDIC SIGN VISARGA SVARITA..VEDIC SIGN VISARGA ANUDATTA WITH TAIL
    Interval{ 0x1CED, 0x1CED }, // Mn       VEDIC SIGN TIRYAK
    Interval{ 0x1CF4, 0x1CF4 }, // Mn       VEDIC TONE CANDRA ABOVE
    Interval{ 0x1CF8, 0x1CF9 }, // Mn   [2] VEDIC TONE RING ABOVE..VEDIC TONE DOUBLE RING ABOVE
    Interval{ 0x1DC0, 0x1DFF }, // Mn  [64] COMBINING DOTTED GRAVE ACCENT..COMBINING RIGHT ARROWHEAD AND DOWN ARROWHEAD BELOW
    Interval{ 0x200C, 0x200C }, // Cf       ZERO WIDTH NON-JOINER
    Interval{ 0x20D0, 0x20DC }, // Mn  [13] COMBINING LEFT HARPOON ABOVE..COMBINING FOUR DOTS ABOVE
    Interval{ 0x20DD, 0x20E0 }, // Me   [4] COMBINING ENCLOSING CIRCLE..COMBINING ENCLOSING CIRCLE BACKSLASH
    Interval{ 0x20E1, 0x20E1 }, // Mn       COMBINING LEFT RIGHT ARROW ABOVE
    Interval{ 0x20E2, 0x20E4 }, // Me   [3] COMBINING ENCLOSING SCREEN..COMBINING ENCLOSING UPWARD POINTING TRIANGLE
    Interval{ 0x20E5, 0x20F0 }, // Mn  [12] COMBINING REVERSE SOLIDUS OVERLAY..COMBINING ASTERISK ABOVE
    Interval{ 0x2CEF, 0x2CF1 }, // Mn   [3] COPTIC COMBINING NI ABOVE..COPTIC COMBINING SPIRITUS LENIS
    Interval{ 0x2D7F, 0x2D7F }, // Mn       TIFINAGH CONSONANT JOINER
    Interval{ 0x2DE0, 0x2DFF }, // Mn  [32] COMBINING CYRILLIC LETTER BE..COMBINING CYRILLIC LETTER IOTIFIED BIG YUS
    Interval{ 0x302A, 0x302D }, // Mn   [4] IDEOGRAPHIC LEVEL TONE MARK..IDEOGRAPHIC ENTERING TONE MARK
    Interval{ 0x302E, 0x302F }, // Mc   [2] HANGUL SINGLE DOT TONE MARK..HANGUL DOUBLE DOT TONE MARK
    Interval{ 0x3099, 0x309A }, // Mn   [2] COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK..COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    Interval{ 0xA66F, 0xA66F }, // Mn       COMBINING CYRILLIC VZMET
    Interval{ 0xA670, 0xA672 }, // Me   [3] COMBINING CYRILLIC TEN MILLIONS SIGN..COMBINING CYRILLIC THOUSAND MILLIONS SIGN
    Interval{ 0xA674, 0xA67D }, // Mn  [10] COMBINING CYRILLIC LETTER UKRAINIAN IE..COMBINING CYRILLIC PAYEROK
    Interval{ 0xA69E, 0xA69F }, // Mn   [2] COMBINING CYRILLIC LETTER EF..COMBINING CYRILLIC LETTER IOTIFIED E
    Interval{ 0xA6F0, 0xA6F1 }, // Mn   [2] BAMUM COMBINING MARK KOQNDON..BAMUM COMBINING MARK TUKWENTIS
    Interval{ 0xA802, 0xA802 }, // Mn       SYLOTI NAGRI SIGN DVISVARA
    Interval{ 0xA806, 0xA806 }, // Mn       SYLOTI NAGRI SIGN HASANTA
    Interval{ 0xA80B, 0xA80B }, // Mn       SYLOTI NAGRI SIGN ANUSVARA
    Interval{ 0xA825, 0xA826 }, // Mn   [2] SYLOTI NAGRI VOWEL SIGN U..SYLOTI NAGRI VOWEL SIGN E
    Interval{ 0xA82C, 0xA82C }, // Mn       SYLOTI NAGRI SIGN ALTERNATE HASANTA
    Interval{ 0xA8C4, 0xA8C5 }, // Mn   [2] SAURASHTRA SIGN VIRAMA..SAURASHTRA SIGN CANDRABINDU
    Interval{ 0xA8E0, 0xA8F1 }, // Mn  [18] COMBINING DEVANAGARI DIGIT ZERO..COMBINING DEVANAGARI SIGN AVAGRAHA
    Interval{ 0xA8FF, 0xA8FF }, // Mn       DEVANAGARI VOWEL SIGN AY
    Interval{ 0xA926, 0xA92D }, // Mn   [8] KAYAH LI VOWEL UE..KAYAH LI TONE CALYA PLOPHU
    Interval{ 0xA947, 0xA951 }, // Mn  [11] REJANG VOWEL SIGN I..REJANG CONSONANT SIGN R
    Interval{ 0xA980, 0xA982 }, // Mn   [3] JAVANESE SIGN PANYANGGA..JAVANESE SIGN LAYAR
    Interval{ 0xA9B3, 0xA9B3 }, // Mn       JAVANESE SIGN CECAK TELU
    Interval{ 0xA9B6, 0xA9B9 }, // Mn   [4] JAVANESE VOWEL SIGN WULU..JAVANESE VOWEL SIGN SUKU MENDUT
    Interval{ 0xA9BC, 0xA9BD }, // Mn   [2] JAVANESE VOWEL SIGN PEPET..JAVANESE CONSONANT SIGN KERET
    Interval{ 0xA9E5, 0xA9E5 }, // Mn       MYANMAR SIGN SHAN SAW
    Interval{ 0xAA29, 0xAA2E }, // Mn   [6] CHAM VOWEL SIGN AA..CHAM VOWEL SIGN OE
    Interval{ 0xAA31, 0xAA32 }, // Mn   [2] CHAM VOWEL SIGN AU..CHAM VOWEL SIGN UE
    Interval{ 0xAA35, 0xAA36 }, // Mn   [2] CHAM CONSONANT SIGN LA..CHAM CONSONANT SIGN WA
    Interval{ 0xAA43, 0xAA43 }, // Mn       CHAM CONSONANT SIGN FINAL NG
    Interval{ 0xAA4C, 0xAA4C }, // Mn       CHAM CONSONANT SIGN FINAL M
    Interval{ 0xAA7C, 0xAA7C }, // Mn       MYANMAR SIGN TAI LAING TONE-2
    Interval{ 0xAAB0, 0xAAB0 }, // Mn       TAI VIET MAI KANG
    Interval{ 0xAAB2, 0xAAB4 }, // Mn   [3] TAI VIET VOWEL I..TAI VIET VOWEL U
    Interval{ 0xAAB7, 0xAAB8 }, // Mn   [2] TAI VIET MAI KHIT..TAI VIET VOWEL IA
    Interval{ 0xAABE, 0xAABF }, // Mn   [2] TAI VIET VOWEL AM..TAI VIET TONE MAI EK
    Interval{ 0xAAC1, 0xAAC1 }, // Mn       TAI VIET TONE MAI THO
    Interval{ 0xAAEC, 0xAAED }, // Mn   [2] MEETEI MAYEK VOWEL SIGN UU..MEETEI MAYEK VOWEL SIGN AAI
    Interval{ 0xAAF6, 0xAAF6 }, // Mn       MEETEI MAYEK VIRAMA
    Interval{ 0xABE5, 0xABE5 }, // Mn       MEETEI MAYEK VOWEL SIGN ANAP
    Interval{ 0xABE8, 0xABE8 }, // Mn       MEETEI MAYEK VOWEL SIGN UNAP
    Interval{ 0xABED, 0xABED }, // Mn       MEETEI MAYEK APUN IYEK
    Interval{ 0xFB1E, 0xFB1E }, // Mn       HEBREW POINT JUDEO-SPANISH VARIKA
    Interval{ 0xFE00, 0xFE0F }, // Mn  [16] VARIATION SELECTOR-1..VARIATION SELECTOR-16
    Interval{ 0xFE20, 0xFE2F }, // Mn  [16] COMBINING LIGATURE LEFT HALF..COMBINING CYRILLIC TITLO RIGHT HALF
    Interval{ 0xFF9E, 0xFF9F }, // Lm   [2] HALFWIDTH KATAKANA VOICED SOUND MARK..HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
    Interval{ 0x101FD, 0x101FD }, // Mn       PHAISTOS DISC SIGN COMBINING OBLIQUE STROKE
    Interval{ 0x102E0, 0x102E0 }, // Mn       COPTIC EPACT THOUSANDS MARK
    Interval{ 0x10376, 0x1037A }, // Mn   [5] COMBINING OLD PERMIC LETTER AN..COMBINING OLD PERMIC LETTER SII
    Interval{ 0x10A01, 0x10A03 }, // Mn   [3] KHAROSHTHI VOWEL SIGN I..KHAROSHTHI VOWEL SIGN VOCALIC R
    Interval{ 0x10A05, 0x10A06 }, // Mn   [2] KHAROSHTHI VOWEL SIGN E..KHAROSHTHI VOWEL SIGN O
    Interval{ 0x10A0C, 0x10A0F }, // Mn   [4] KHAROSHTHI VOWEL LENGTH MARK..KHAROSHTHI SIGN VISARGA
    Interval{ 0x10A38, 0x10A3A }, // Mn   [3] KHAROSHTHI SIGN BAR ABOVE..KHAROSHTHI SIGN DOT BELOW
    Interval{ 0x10A3F, 0x10A3F }, // Mn       KHAROSHTHI VIRAMA
    Interval{ 0x10AE5, 0x10AE6 }, // Mn   [2] MANICHAEAN ABBREVIATION MARK ABOVE..MANICHAEAN ABBREVIATION MARK BELOW
    Interval{ 0x10D24, 0x10D27 }, // Mn   [4] HANIFI ROHINGYA SIGN HARBAHAY..HANIFI ROHINGYA SIGN TASSI
    Interval{ 0x10EAB, 0x10EAC }, // Mn   [2] YEZIDI COMBINING HAMZA MARK..YEZIDI COMBINING MADDA MARK
    Interval{ 0x10EFD, 0x10EFF }, // Mn   [3] ARABIC SMALL LOW WORD SAKTA..ARABIC SMALL LOW WORD MADDA
    Interval{ 0x10F46, 0x10F50 }, // Mn  [11] SOGDIAN COMBINING DOT BELOW..SOGDIAN COMBINING STROKE BELOW
    Interval{ 0x10F82, 0x10F85 }, // Mn   [4] OLD UYGHUR COMBINING DOT ABOVE..OLD UYGHUR COMBINING TWO DOTS BELOW
    Interval{ 0x11001, 0x11001 }, // Mn       BRAHMI SIGN ANUSVARA
    Interval{ 0x11038, 0x11046 }, // Mn  [15] BRAHMI VOWEL SIGN AA..BRAHMI VIRAMA
    Interval{ 0x11070, 0x11070 }, // Mn       BRAHMI SIGN OLD TAMIL VIRAMA
    Interval{ 0x11073, 0x11074 }, // Mn   [2] BRAHMI VOWEL SIGN OLD TAMIL SHORT E..BRAHMI VOWEL SIGN OLD TAMIL SHORT O
    Interval{ 0x1107F, 0x11081 }, // Mn   [3] BRAHMI NUMBER JOINER..KAITHI SIGN ANUSVARA
    Interval{ 0x110B3, 0x110B6 }, // Mn   [4] KAITHI VOWEL SIGN U..KAITHI VOWEL SIGN AI
    Interval{ 0x110B9, 0x110BA }, // Mn   [2] KAITHI SIGN VIRAMA..KAITHI SIGN NUKTA
    Interval{ 0x110C2, 0x110C2 }, // Mn       KAITHI VOWEL SIGN VOCALIC R
    Interval{ 0x11100, 0x11102 }, // Mn   [3] CHAKMA SIGN CANDRABINDU..CHAKMA SIGN VISARGA
    Interval{ 0x11127, 0x1112B }, // Mn   [5] CHAKMA VOWEL SIGN A..CHAKMA VOWEL SIGN UU
    Interval{ 0x1112D, 0x11134 }, // Mn   [8] CHAKMA VOWEL SIGN AI..CHAKMA MAAYYAA
    Interval{ 0x11173, 0x11173 }, // Mn       MAHAJANI SIGN NUKTA
    Interval{ 0x11180, 0x11181 }, // Mn   [2] SHARADA SIGN CANDRABINDU..SHARADA SIGN ANUSVARA
    Interval{ 0x111B6, 0x111BE }, // Mn   [9] SHARADA VOWEL SIGN U..SHARADA VOWEL SIGN O
    Interval{ 0x111C9, 0x111CC }, // Mn   [4] SHARADA SANDHI MARK..SHARADA EXTRA SHORT VOWEL MARK
    Interval{ 0x111CF, 0x111CF }, // Mn       SHARADA SIGN INVERTED CANDRABINDU
    Interval{ 0x1122F, 0x11231 }, // Mn   [3] KHOJKI VOWEL SIGN U..KHOJKI VOWEL SIGN AI
    Interval{ 0x11234, 0x11234 }, // Mn       KHOJKI SIGN ANUSVARA
    Interval{ 0x11236, 0x11237 }, // Mn   [2] KHOJKI SIGN NUKTA..KHOJKI SIGN SHADDA
    Interval{ 0x1123E, 0x1123E }, // Mn       KHOJKI SIGN SUKUN
    Interval{ 0x11241, 0x11241 }, // Mn       KHOJKI VOWEL SIGN VOCALIC R
    Interval{ 0x112DF, 0x112DF }, // Mn       KHUDAWADI SIGN ANUSVARA
    Interval{ 0x112E3, 0x112EA }, // Mn   [8] KHUDAWADI VOWEL SIGN U..KHUDAWADI SIGN VIRAMA
    Interval{ 0x11300, 0x11301 }, // Mn   [2] GRANTHA SIGN COMBINING ANUSVARA ABOVE..GRANTHA SIGN CANDRABINDU
    Interval{ 0x1133B, 0x1133C }, // Mn   [2] COMBINING BINDU BELOW..GRANTHA SIGN NUKTA
    Interval{ 0x1133E, 0x1133E }, // Mc       GRANTHA VOWEL SIGN AA
    Interval{ 0x11340, 0x11340 }, // Mn       GRANTHA VOWEL SIGN II
    Interval{ 0x11357, 0x11357 }, // Mc       GRANTHA AU LENGTH MARK
    Interval{ 0x11366, 0x1136C }, // Mn   [7] COMBINING GRANTHA DIGIT ZERO..COMBINING GRANTHA DIGIT SIX
    Interval{ 0x11370, 0x11374 }, // Mn   [5] COMBINING GRANTHA LETTER A..COMBINING GRANTHA LETTER PA
    Interval{ 0x11438, 0x1143F }, // Mn   [8] NEWA VOWEL SIGN U..NEWA VOWEL SIGN AI
    Interval{ 0x11442, 0x11444 }, // Mn   [3] NEWA SIGN VIRAMA..NEWA SIGN ANUSVARA
    Interval{ 0x11446, 0x11446 }, // Mn       NEWA SIGN NUKTA
    Interval{ 0x1145E, 0x1145E }, // Mn       NEWA SANDHI MARK
    Interval{ 0x114B0, 0x114B0 }, // Mc       TIRHUTA VOWEL SIGN AA
    Interval{ 0x114B3, 0x114B8 }, // Mn   [6] TIRHUTA VOWEL SIGN U..TIRHUTA VOWEL SIGN VOCALIC LL
    Interval{ 0x114BA, 0x114BA }, // Mn       TIRHUTA VOWEL SIGN SHORT E
    Interval{ 0x114BD, 0x114BD }, // Mc       TIRHUTA VOWEL SIGN SHORT O
    Interval{ 0x114BF, 0x114C0 }, // Mn   [2] TIRHUTA SIGN CANDRABINDU..TIRHUTA SIGN ANUSVARA
    Interval{ 0x114C2, 0x114C3 }, // Mn   [2] TIRHUTA SIGN VIRAMA..TIRHUTA SIGN NUKTA
    Interval{ 0x115AF, 0x115AF }, // Mc       SIDDHAM VOWEL SIGN AA
    Interval{ 0x115B2, 0x115B5 }, // Mn   [4] SIDDHAM VOWEL SIGN U..SIDDHAM VOWEL SIGN VOCALIC RR
    Interval{ 0x115BC, 0x115BD }, // Mn   [2] SIDDHAM SIGN CANDRABINDU..SIDDHAM SIGN ANUSVARA
    Interval{ 0x115BF, 0x115C0 }, // Mn   [2] SIDDHAM SIGN VIRAMA..SIDDHAM SIGN NUKTA
    Interval{ 0x115DC, 0x115DD }, // Mn   [2] SIDDHAM VOWEL SIGN ALTERNATE U..SIDDHAM VOWEL SIGN ALTERNATE UU
    Interval{ 0x11633, 0x1163A }, // Mn   [8] MODI VOWEL SIGN U..MODI VOWEL SIGN AI
    Interval{ 0x1163D, 0x1163D }, // Mn       MODI SIGN ANUSVARA
    Interval{ 0x1163F, 0x11640 }, // Mn   [2] MODI SIGN VIRAMA..MODI SIGN ARDHACANDRA
    Interval{ 0x116AB, 0x116AB }, // Mn       TAKRI SIGN ANUSVARA
    Interval{ 0x116AD, 0x116AD }, // Mn       TAKRI VOWEL SIGN AA
    Interval{ 0x116B0, 0x116B5 }, // Mn   [6] TAKRI VOWEL SIGN U..TAKRI VOWEL SIGN AU
    Interval{ 0x116B7, 0x116B7 }, // Mn       TAKRI SIGN NUKTA
    Interval{ 0x1171D, 0x1171F }, // Mn   [3] AHOM CONSONANT SIGN MEDIAL LA..AHOM CONSONANT SIGN MEDIAL LIGATING RA
    Interval{ 0x11722, 0x11725 }, // Mn   [4] AHOM VOWEL SIGN I..AHOM VOWEL SIGN UU
    Interval{ 0x11727, 0x1172B }, // Mn   [5] AHOM VOWEL SIGN AW..AHOM SIGN KILLER
    Interval{ 0x1182F, 0x11837 }, // Mn   [9] DOGRA VOWEL SIGN U..DOGRA SIGN ANUSVARA
    Interval{ 0x11839, 0x1183A }, // Mn   [2] DOGRA SIGN VIRAMA..DOGRA SIGN NUKTA
    Interval{ 0x11930, 0x11930 }, // Mc       DIVES AKURU VOWEL SIGN AA
    Interval{ 0x1193B, 0x1193C }, // Mn   [2] DIVES AKURU SIGN ANUSVARA..DIVES AKURU SIGN CANDRABINDU
    Interval{ 0x1193E, 0x1193E }, // Mn       DIVES AKURU VIRAMA
    Interval{ 0x11943, 0x11943 }, // Mn       DIVES AKURU SIGN NUKTA
    Interval{ 0x119D4, 0x119D7 }, // Mn   [4] NANDINAGARI VOWEL SIGN U..NANDINAGARI VOWEL SIGN VOCALIC RR
    Interval{ 0x119DA, 0x119DB }, // Mn   [2] NANDINAGARI VOWEL SIGN E..NANDINAGARI VOWEL SIGN AI
    Interval{ 0x119E0, 0x119E0 }, // Mn       NANDINAGARI SIGN VIRAMA
    Interval{ 0x11A01, 0x11A0A }, // Mn  [10] ZANABAZAR SQUARE VOWEL SIGN I..ZANABAZAR SQUARE VOWEL LENGTH MARK
    Interval{ 0x11A33, 0x11A38 }, // Mn   [6] ZANABAZAR SQUARE FINAL CONSONANT MARK..ZANABAZAR SQUARE SIGN ANUSVARA
    Interval{ 0x11A3B, 0x11A3E }, // Mn   [4] ZANABAZAR SQUARE CLUSTER-FINAL LETTER YA..ZANABAZAR SQUARE CLUSTER-FINAL LETTER VA
    Interval{ 0x11A47, 0x11A47 }, // Mn       ZANABAZAR SQUARE SUBJOINER
    Interval{ 0x11A51, 0x11A56 }, // Mn   [6] SOYOMBO VOWEL SIGN I..SOYOMBO VOWEL SIGN OE
    Interval{ 0x11A59, 0x11A5B }, // Mn   [3] SOYOMBO VOWEL SIGN VOCALIC R..SOYOMBO VOWEL LENGTH MARK
    Interval{ 0x11A8A, 0x11A96 }, // Mn  [13] SOYOMBO FINAL CONSONANT SIGN G..SOYOMBO SIGN ANUSVARA
    Interval{ 0x11A98, 0x11A99 }, // Mn   [2] SOYOMBO GEMINATION MARK..SOYOMBO SUBJOINER
    Interval{ 0x11C30, 0x11C36 }, // Mn   [7] BHAIKSUKI VOWEL SIGN I..BHAIKSUKI VOWEL SIGN VOCALIC L
    Interval{ 0x11C38, 0x11C3D }, // Mn   [6] BHAIKSUKI VOWEL SIGN E..BHAIKSUKI SIGN ANUSVARA
    Interval{ 0x11C3F, 0x11C3F }, // Mn       BHAIKSUKI SIGN VIRAMA
    Interval{ 0x11C92, 0x11CA7 }, // Mn  [22] MARCHEN SUBJOINED LETTER KA..MARCHEN SUBJOINED LETTER ZA
    Interval{ 0x11CAA, 0x11CB0 }, // Mn   [7] MARCHEN SUBJOINED LETTER RA..MARCHEN VOWEL SIGN AA
    Interval{ 0x11CB2, 0x11CB3 }, // Mn   [2] MARCHEN VOWEL SIGN U..MARCHEN VOWEL SIGN E
    Interval{ 0x11CB5, 0x11CB6 }, // Mn   [2] MARCHEN SIGN ANUSVARA..MARCHEN SIGN CANDRABINDU
    Interval{ 0x11D31, 0x11D36 }, // Mn   [6] MASARAM GONDI VOWEL SIGN AA..MASARAM GONDI VOWEL SIGN VOCALIC R
    Interval{ 0x11D3A, 0x11D3A }, // Mn       MASARAM GONDI VOWEL SIGN E
    Interval{ 0x11D3C, 0x11D3D }, // Mn   [2] MASARAM GONDI VOWEL SIGN AI..MASARAM GONDI VOWEL SIGN O
    Interval{ 0x11D3F, 0x11D45 }, // Mn   [7] MASARAM GONDI VOWEL SIGN AU..MASARAM GONDI VIRAMA
    Interval{ 0x11D47, 0x11D47 }, // Mn       MASARAM GONDI RA-KARA
    Interval{ 0x11D90, 0x11D91 }, // Mn   [2] GUNJALA GONDI VOWEL SIGN EE..GUNJALA GONDI VOWEL SIGN AI
    Interval{ 0x11D95, 0x11D95 }, // Mn       GUNJALA GONDI SIGN ANUSVARA
    Interval{ 0x11D97, 0x11D97 }, // Mn       GUNJALA GONDI VIRAMA
    Interval{ 0x11EF3, 0x11EF4 }, // Mn   [2] MAKASAR VOWEL SIGN I..MAKASAR VOWEL SIGN U
    Interval{ 0x11F00, 0x11F01 }, // Mn   [2] KAWI SIGN CANDRABINDU..KAWI SIGN ANUSVARA
    Interval{ 0x11F36, 0x11F3A }, // Mn   [5] KAWI VOWEL SIGN I..KAWI VOWEL SIGN VOCALIC R
    Interval{ 0x11F40, 0x11F40 }, // Mn       KAWI VOWEL SIGN EU
    Interval{ 0x11F42, 0x11F42 }, // Mn       KAWI CONJOINER
    Interval{ 0x13440, 0x13440 }, // Mn       EGYPTIAN HIEROGLYPH MIRROR HORIZONTALLY
    Interval{ 0x13447, 0x13455 }, // Mn  [15] EGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT TOP START..EGYPTIAN HIEROGLYPH MODIFIER DAMAGED
    Interval{ 0x16AF0, 0x16AF4 }, // Mn   [5] BASSA VAH COMBINING HIGH TONE..BASSA VAH COMBINING HIGH-LOW TONE
    Interval{ 0x16B30, 0x16B36 }, // Mn   [7] PAHAWH HMONG MARK CIM TUB..PAHAWH HMONG MARK CIM TAUM
    Interval{ 0x16F4F, 0x16F4F }, // Mn       MIAO SIGN CONSONANT MODIFIER BAR
    Interval{ 0x16F8F, 0x16F92 }, // Mn   [4] MIAO TONE RIGHT..MIAO TONE BELOW
    Interval{ 0x16FE4, 0x16FE4 }, // Mn       KHITAN SMALL SCRIPT FILLER
    Interval{ 0x1BC9D, 0x1BC9E }, // Mn   [2] DUPLOYAN THICK LETTER SELECTOR..DUPLOYAN DOUBLE MARK
    Interval{ 0x1CF00, 0x1CF2D }, // Mn  [46] ZNAMENNY COMBINING MARK GORAZDO NIZKO S KRYZHEM ON LEFT..ZNAMENNY COMBINING MARK KRYZH ON LEFT
    Interval{ 0x1CF30, 0x1CF46 }, // Mn  [23] ZNAMENNY COMBINING TONAL RANGE MARK MRACHNO..ZNAMENNY PRIZNAK MODIFIER ROG
    Interval{ 0x1D165, 0x1D165 }, // Mc       MUSICAL SYMBOL COMBINING STEM
    Interval{ 0x1D167, 0x1D169 }, // Mn   [3] MUSICAL SYMBOL COMBINING TREMOLO-1..MUSICAL SYMBOL COMBINING TREMOLO-3
    Interval{ 0x1D16E, 0x1D172 }, // Mc   [5] MUSICAL SYMBOL COMBINING FLAG-1..MUSICAL SYMBOL COMBINING FLAG-5
    Interval{ 0x1D17B, 0x1D182 }, // Mn   [8] MUSICAL SYMBOL COMBINING ACCENT..MUSICAL SYMBOL COMBINING LOURE
    Interval{ 0x1D185, 0x1D18B }, // Mn   [7] MUSICAL SYMBOL COMBINING DOIT..MUSICAL SYMBOL COMBINING TRIPLE TONGUE
    Interval{ 0x1D1AA, 0x1D1AD }, // Mn   [4] MUSICAL SYMBOL COMBINING DOWN BOW..MUSICAL SYMBOL COMBINING SNAP PIZZICATO
    Interval{ 0x1D242, 0x1D244 }, // Mn   [3] COMBINING GREEK MUSICAL TRISEME..COMBINING GREEK MUSICAL PENTASEME
    Interval{ 0x1DA00, 0x1DA36 }, // Mn  [55] SIGNWRITING HEAD RIM..SIGNWRITING AIR SUCKING IN
    Interval{ 0x1DA3B, 0x1DA6C }, // Mn  [50] SIGNWRITING MOUTH CLOSED NEUTRAL..SIGNWRITING EXCITEMENT
    Interval{ 0x1DA75, 0x1DA75 }, // Mn       SIGNWRITING UPPER BODY TILTING FROM HIP JOINTS
    Interval{ 0x1DA84, 0x1DA84 }, // Mn       SIGNWRITING LOCATION HEAD NECK
    Interval{ 0x1DA9B, 0x1DA9F }, // Mn   [5] SIGNWRITING FILL MODIFIER-2..SIGNWRITING FILL MODIFIER-6
    Interval{ 0x1DAA1, 0x1DAAF }, // Mn  [15] SIGNWRITING ROTATION MODIFIER-2..SIGNWRITING ROTATION MODIFIER-16
    Interval{ 0x1E000, 0x1E006 }, // Mn   [7] COMBINING GLAGOLITIC LETTER AZU..COMBINING GLAGOLITIC LETTER ZHIVETE
    Interval{ 0x1E008, 0x1E018 }, // Mn  [17] COMBINING GLAGOLITIC LETTER ZEMLJA..COMBINING GLAGOLITIC LETTER HERU
    Interval{ 0x1E01B, 0x1E021 }, // Mn   [7] COMBINING GLAGOLITIC LETTER SHTA..COMBINING GLAGOLITIC LETTER YATI
    Interval{ 0x1E023, 0x1E024 }, // Mn   [2] COMBINING GLAGOLITIC LETTER YU..COMBINING GLAGOLITIC LETTER SMALL YUS
    Interval{ 0x1E026, 0x1E02A }, // Mn   [5] COMBINING GLAGOLITIC LETTER YO..COMBINING GLAGOLITIC LETTER FITA
    Interval{ 0x1E08F, 0x1E08F }, // Mn       COMBINING CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
    Interval{ 0x1E130, 0x1E136 }, // Mn   [7] NYIAKENG PUACHUE HMONG TONE-B..NYIAKENG PUACHUE HMONG TONE-D
    Interval{ 0x1E2AE, 0x1E2AE }, // Mn       TOTO SIGN RISING TONE
    Interval{ 0x1E2EC, 0x1E2EF }, // Mn   [4] WANCHO TONE TUP..WANCHO TONE KOINI
    Interval{ 0x1E4EC, 0x1E4EF }, // Mn   [4] NAG MUNDARI SIGN MUHOR..NAG MUNDARI SIGN SUTUH
    Interval{ 0x1E8D0, 0x1E8D6 }, // Mn   [7] MENDE KIKAKUI COMBINING NUMBER TEENS..MENDE KIKAKUI COMBINING NUMBER MILLIONS
    Interval{ 0x1E944, 0x1E94A }, // Mn   [7] ADLAM ALIF LENGTHENER..ADLAM NUKTA
    Interval{ 0xE0020, 0xE007F }, // Cf  [96] TAG SPACE..CANCEL TAG
    Interval{ 0xE0100, 0xE01EF }, // Mn [240] VARIATION SELECTOR-17..VARIATION SELECTOR-256
}; // }}}
auto static const Grapheme_Link = std::array<Interval, 60>{ // {{{
    Interval{ 0x094D, 0x094D }, // Mn       DEVANAGARI SIGN VIRAMA
    Interval{ 0x09CD, 0x09CD }, // Mn       BENGALI SIGN VIRAMA
    Interval{ 0x0A4D, 0x0A4D }, // Mn       GURMUKHI SIGN VIRAMA
    Interval{ 0x0ACD, 0x0ACD }, // Mn       GUJARATI SIGN VIRAMA
    Interval{ 0x0B4D, 0x0B4D }, // Mn       ORIYA SIGN VIRAMA
    Interval{ 0x0BCD, 0x0BCD }, // Mn       TAMIL SIGN VIRAMA
    Interval{ 0x0C4D, 0x0C4D }, // Mn       TELUGU SIGN VIRAMA
    Interval{ 0x0CCD, 0x0CCD }, // Mn       KANNADA SIGN VIRAMA
    Interval{ 0x0D3B, 0x0D3C }, // Mn   [2] MALAYALAM SIGN VERTICAL BAR VIRAMA..MALAYALAM SIGN CIRCULAR VIRAMA
    Interval{ 0x0D4D, 0x0D4D }, // Mn       MALAYALAM SIGN VIRAMA
    Interval{ 0x0DCA, 0x0DCA }, // Mn       SINHALA SIGN AL-LAKUNA
    Interval{ 0x0E3A, 0x0E3A }, // Mn       THAI CHARACTER PHINTHU
    Interval{ 0x0EBA, 0x0EBA }, // Mn       LAO SIGN PALI VIRAMA
    Interval{ 0x0F84, 0x0F84 }, // Mn       TIBETAN MARK HALANTA
    Interval{ 0x1039, 0x103A }, // Mn   [2] MYANMAR SIGN VIRAMA..MYANMAR SIGN ASAT
    Interval{ 0x1714, 0x1714 }, // Mn       TAGALOG SIGN VIRAMA
    Interval{ 0x1715, 0x1715 }, // Mc       TAGALOG SIGN PAMUDPOD
    Interval{ 0x1734, 0x1734 }, // Mc       HANUNOO SIGN PAMUDPOD
    Interval{ 0x17D2, 0x17D2 }, // Mn       KHMER SIGN COENG
    Interval{ 0x1A60, 0x1A60 }, // Mn       TAI THAM SIGN SAKOT
    Interval{ 0x1B44, 0x1B44 }, // Mc       BALINESE ADEG ADEG
    Interval{ 0x1BAA, 0x1BAA }, // Mc       SUNDANESE SIGN PAMAAEH
    Interval{ 0x1BAB, 0x1BAB }, // Mn       SUNDANESE SIGN VIRAMA
    Interval{ 0x1BF2, 0x1BF3 }, // Mc   [2] BATAK PANGOLAT..BATAK PANONGONAN
    Interval{ 0x2D7F, 0x2D7F }, // Mn       TIFINAGH CONSONANT JOINER
    Interval{ 0xA806, 0xA806 }, // Mn       SYLOTI NAGRI SIGN HASANTA
    Interval{ 0xA82C, 0xA82C }, // Mn       SYLOTI NAGRI SIGN ALTERNATE HASANTA
    Interval{ 0xA8C4, 0xA8C4 }, // Mn       SAURASHTRA SIGN VIRAMA
    Interval{ 0xA953, 0xA953 }, // Mc       REJANG VIRAMA
    Interval{ 0xA9C0, 0xA9C0 }, // Mc       JAVANESE PANGKON
    Interval{ 0xAAF6, 0xAAF6 }, // Mn       MEETEI MAYEK VIRAMA
    Interval{ 0xABED, 0xABED }, // Mn       MEETEI MAYEK APUN IYEK
    Interval{ 0x10A3F, 0x10A3F }, // Mn       KHAROSHTHI VIRAMA
    Interval{ 0x11046, 0x11046 }, // Mn       BRAHMI VIRAMA
    Interval{ 0x11070, 0x11070 }, // Mn       BRAHMI SIGN OLD TAMIL VIRAMA
    Interval{ 0x1107F, 0x1107F }, // Mn       BRAHMI NUMBER JOINER
    Interval{ 0x110B9, 0x110B9 }, // Mn       KAITHI SIGN VIRAMA
    Interval{ 0x11133, 0x11134 }, // Mn   [2] CHAKMA VIRAMA..CHAKMA MAAYYAA
    Interval{ 0x111C0, 0x111C0 }, // Mc       SHARADA SIGN VIRAMA
    Interval{ 0x11235, 0x11235 }, // Mc       KHOJKI SIGN VIRAMA
    Interval{ 0x112EA, 0x112EA }, // Mn       KHUDAWADI SIGN VIRAMA
    Interval{ 0x1134D, 0x1134D }, // Mc       GRANTHA SIGN VIRAMA
    Interval{ 0x11442, 0x11442 }, // Mn       NEWA SIGN VIRAMA
    Interval{ 0x114C2, 0x114C2 }, // Mn       TIRHUTA SIGN VIRAMA
    Interval{ 0x115BF, 0x115BF }, // Mn       SIDDHAM SIGN VIRAMA
    Interval{ 0x1163F, 0x1163F }, // Mn       MODI SIGN VIRAMA
    Interval{ 0x116B6, 0x116B6 }, // Mc       TAKRI SIGN VIRAMA
    Interval{ 0x1172B, 0x1172B }, // Mn       AHOM SIGN KILLER
    Interval{ 0x11839, 0x11839 }, // Mn       DOGRA SIGN VIRAMA
    Interval{ 0x1193D, 0x1193D }, // Mc       DIVES AKURU SIGN HALANTA
    Interval{ 0x1193E, 0x1193E }, // Mn       DIVES AKURU VIRAMA
    Interval{ 0x119E0, 0x119E0 }, // Mn       NANDINAGARI SIGN VIRAMA
    Interval{ 0x11A34, 0x11A34 }, // Mn       ZANABAZAR SQUARE SIGN VIRAMA
    Interval{ 0x11A47, 0x11A47 }, // Mn       ZANABAZAR SQUARE SUBJOINER
    Interval{ 0x11A99, 0x11A99 }, // Mn       SOYOMBO SUBJOINER
    Interval{ 0x11C3F, 0x11C3F }, // Mn       BHAIKSUKI SIGN VIRAMA
    Interval{ 0x11D44, 0x11D45 }, // Mn   [2] MASARAM GONDI SIGN HALANTA..MASARAM GONDI VIRAMA
    Interval{ 0x11D97, 0x11D97 }, // Mn       GUNJALA GONDI VIRAMA
    Interval{ 0x11F41, 0x11F41 }, // Mc       KAWI SIGN KILLER
    Interval{ 0x11F42, 0x11F42 }, // Mn       KAWI CONJOINER
}; // }}}
auto static const ID_Continue = std::array<Interval, 1340>{ // {{{
    Interval{ 0x0030, 0x0039 }, // Nd  [10] DIGIT ZERO..DIGIT NINE
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x005F, 0x005F }, // Pc       LOW LINE
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00AA, 0x00AA }, // Lo       FEMININE ORDINAL INDICATOR
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00B7, 0x00B7 }, // Po       MIDDLE DOT
    Interval{ 0x00BA, 0x00BA }, // Lo       MASCULINE ORDINAL INDICATOR
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00F6 }, // L&  [31] LATIN CAPITAL LETTER O WITH STROKE..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x01BA }, // L& [195] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER EZH WITH TAIL
    Interval{ 0x01BB, 0x01BB }, // Lo       LATIN LETTER TWO WITH STROKE
    Interval{ 0x01BC, 0x01BF }, // L&   [4] LATIN CAPITAL LETTER TONE FIVE..LATIN LETTER WYNN
    Interval{ 0x01C0, 0x01C3 }, // Lo   [4] LATIN LETTER DENTAL CLICK..LATIN LETTER RETROFLEX CLICK
    Interval{ 0x01C4, 0x0293 }, // L& [208] LATIN CAPITAL LETTER DZ WITH CARON..LATIN SMALL LETTER EZH WITH CURL
    Interval{ 0x0294, 0x0294 }, // Lo       LATIN LETTER GLOTTAL STOP
    Interval{ 0x0295, 0x02AF }, // L&  [27] LATIN LETTER PHARYNGEAL VOICED FRICATIVE..LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    Interval{ 0x02B0, 0x02C1 }, // Lm  [18] MODIFIER LETTER SMALL H..MODIFIER LETTER REVERSED GLOTTAL STOP
    Interval{ 0x02C6, 0x02D1 }, // Lm  [12] MODIFIER LETTER CIRCUMFLEX ACCENT..MODIFIER LETTER HALF TRIANGULAR COLON
    Interval{ 0x02E0, 0x02E4 }, // Lm   [5] MODIFIER LETTER SMALL GAMMA..MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    Interval{ 0x02EC, 0x02EC }, // Lm       MODIFIER LETTER VOICING
    Interval{ 0x02EE, 0x02EE }, // Lm       MODIFIER LETTER DOUBLE APOSTROPHE
    Interval{ 0x0300, 0x036F }, // Mn [112] COMBINING GRAVE ACCENT..COMBINING LATIN SMALL LETTER X
    Interval{ 0x0370, 0x0373 }, // L&   [4] GREEK CAPITAL LETTER HETA..GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0374, 0x0374 }, // Lm       GREEK NUMERAL SIGN
    Interval{ 0x0376, 0x0377 }, // L&   [2] GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA..GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037A, 0x037A }, // Lm       GREEK YPOGEGRAMMENI
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0387, 0x0387 }, // Po       GREEK ANO TELEIA
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x03A1 }, // L&  [20] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03F5 }, // L&  [83] GREEK CAPITAL LETTER SIGMA..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F7, 0x0481 }, // L& [139] GREEK CAPITAL LETTER SHO..CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x0483, 0x0487 }, // Mn   [5] COMBINING CYRILLIC TITLO..COMBINING CYRILLIC POKRYTIE
    Interval{ 0x048A, 0x052F }, // L& [166] CYRILLIC CAPITAL LETTER SHORT I WITH TAIL..CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x0559, 0x0559 }, // Lm       ARMENIAN MODIFIER LETTER LEFT HALF RING
    Interval{ 0x0560, 0x0588 }, // L&  [41] ARMENIAN SMALL LETTER TURNED AYB..ARMENIAN SMALL LETTER YI WITH STROKE
    Interval{ 0x0591, 0x05BD }, // Mn  [45] HEBREW ACCENT ETNAHTA..HEBREW POINT METEG
    Interval{ 0x05BF, 0x05BF }, // Mn       HEBREW POINT RAFE
    Interval{ 0x05C1, 0x05C2 }, // Mn   [2] HEBREW POINT SHIN DOT..HEBREW POINT SIN DOT
    Interval{ 0x05C4, 0x05C5 }, // Mn   [2] HEBREW MARK UPPER DOT..HEBREW MARK LOWER DOT
    Interval{ 0x05C7, 0x05C7 }, // Mn       HEBREW POINT QAMATS QATAN
    Interval{ 0x05D0, 0x05EA }, // Lo  [27] HEBREW LETTER ALEF..HEBREW LETTER TAV
    Interval{ 0x05EF, 0x05F2 }, // Lo   [4] HEBREW YOD TRIANGLE..HEBREW LIGATURE YIDDISH DOUBLE YOD
    Interval{ 0x0610, 0x061A }, // Mn  [11] ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM..ARABIC SMALL KASRA
    Interval{ 0x0620, 0x063F }, // Lo  [32] ARABIC LETTER KASHMIRI YEH..ARABIC LETTER FARSI YEH WITH THREE DOTS ABOVE
    Interval{ 0x0640, 0x0640 }, // Lm       ARABIC TATWEEL
    Interval{ 0x0641, 0x064A }, // Lo  [10] ARABIC LETTER FEH..ARABIC LETTER YEH
    Interval{ 0x064B, 0x065F }, // Mn  [21] ARABIC FATHATAN..ARABIC WAVY HAMZA BELOW
    Interval{ 0x0660, 0x0669 }, // Nd  [10] ARABIC-INDIC DIGIT ZERO..ARABIC-INDIC DIGIT NINE
    Interval{ 0x066E, 0x066F }, // Lo   [2] ARABIC LETTER DOTLESS BEH..ARABIC LETTER DOTLESS QAF
    Interval{ 0x0670, 0x0670 }, // Mn       ARABIC LETTER SUPERSCRIPT ALEF
    Interval{ 0x0671, 0x06D3 }, // Lo  [99] ARABIC LETTER ALEF WASLA..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
    Interval{ 0x06D5, 0x06D5 }, // Lo       ARABIC LETTER AE
    Interval{ 0x06D6, 0x06DC }, // Mn   [7] ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA..ARABIC SMALL HIGH SEEN
    Interval{ 0x06DF, 0x06E4 }, // Mn   [6] ARABIC SMALL HIGH ROUNDED ZERO..ARABIC SMALL HIGH MADDA
    Interval{ 0x06E5, 0x06E6 }, // Lm   [2] ARABIC SMALL WAW..ARABIC SMALL YEH
    Interval{ 0x06E7, 0x06E8 }, // Mn   [2] ARABIC SMALL HIGH YEH..ARABIC SMALL HIGH NOON
    Interval{ 0x06EA, 0x06ED }, // Mn   [4] ARABIC EMPTY CENTRE LOW STOP..ARABIC SMALL LOW MEEM
    Interval{ 0x06EE, 0x06EF }, // Lo   [2] ARABIC LETTER DAL WITH INVERTED V..ARABIC LETTER REH WITH INVERTED V
    Interval{ 0x06F0, 0x06F9 }, // Nd  [10] EXTENDED ARABIC-INDIC DIGIT ZERO..EXTENDED ARABIC-INDIC DIGIT NINE
    Interval{ 0x06FA, 0x06FC }, // Lo   [3] ARABIC LETTER SHEEN WITH DOT BELOW..ARABIC LETTER GHAIN WITH DOT BELOW
    Interval{ 0x06FF, 0x06FF }, // Lo       ARABIC LETTER HEH WITH INVERTED V
    Interval{ 0x0710, 0x0710 }, // Lo       SYRIAC LETTER ALAPH
    Interval{ 0x0711, 0x0711 }, // Mn       SYRIAC LETTER SUPERSCRIPT ALAPH
    Interval{ 0x0712, 0x072F }, // Lo  [30] SYRIAC LETTER BETH..SYRIAC LETTER PERSIAN DHALATH
    Interval{ 0x0730, 0x074A }, // Mn  [27] SYRIAC PTHAHA ABOVE..SYRIAC BARREKH
    Interval{ 0x074D, 0x07A5 }, // Lo  [89] SYRIAC LETTER SOGDIAN ZHAIN..THAANA LETTER WAAVU
    Interval{ 0x07A6, 0x07B0 }, // Mn  [11] THAANA ABAFILI..THAANA SUKUN
    Interval{ 0x07B1, 0x07B1 }, // Lo       THAANA LETTER NAA
    Interval{ 0x07C0, 0x07C9 }, // Nd  [10] NKO DIGIT ZERO..NKO DIGIT NINE
    Interval{ 0x07CA, 0x07EA }, // Lo  [33] NKO LETTER A..NKO LETTER JONA RA
    Interval{ 0x07EB, 0x07F3 }, // Mn   [9] NKO COMBINING SHORT HIGH TONE..NKO COMBINING DOUBLE DOT ABOVE
    Interval{ 0x07F4, 0x07F5 }, // Lm   [2] NKO HIGH TONE APOSTROPHE..NKO LOW TONE APOSTROPHE
    Interval{ 0x07FA, 0x07FA }, // Lm       NKO LAJANYALAN
    Interval{ 0x07FD, 0x07FD }, // Mn       NKO DANTAYALAN
    Interval{ 0x0800, 0x0815 }, // Lo  [22] SAMARITAN LETTER ALAF..SAMARITAN LETTER TAAF
    Interval{ 0x0816, 0x0819 }, // Mn   [4] SAMARITAN MARK IN..SAMARITAN MARK DAGESH
    Interval{ 0x081A, 0x081A }, // Lm       SAMARITAN MODIFIER LETTER EPENTHETIC YUT
    Interval{ 0x081B, 0x0823 }, // Mn   [9] SAMARITAN MARK EPENTHETIC YUT..SAMARITAN VOWEL SIGN A
    Interval{ 0x0824, 0x0824 }, // Lm       SAMARITAN MODIFIER LETTER SHORT A
    Interval{ 0x0825, 0x0827 }, // Mn   [3] SAMARITAN VOWEL SIGN SHORT A..SAMARITAN VOWEL SIGN U
    Interval{ 0x0828, 0x0828 }, // Lm       SAMARITAN MODIFIER LETTER I
    Interval{ 0x0829, 0x082D }, // Mn   [5] SAMARITAN VOWEL SIGN LONG I..SAMARITAN MARK NEQUDAA
    Interval{ 0x0840, 0x0858 }, // Lo  [25] MANDAIC LETTER HALQA..MANDAIC LETTER AIN
    Interval{ 0x0859, 0x085B }, // Mn   [3] MANDAIC AFFRICATION MARK..MANDAIC GEMINATION MARK
    Interval{ 0x0860, 0x086A }, // Lo  [11] SYRIAC LETTER MALAYALAM NGA..SYRIAC LETTER MALAYALAM SSA
    Interval{ 0x0870, 0x0887 }, // Lo  [24] ARABIC LETTER ALEF WITH ATTACHED FATHA..ARABIC BASELINE ROUND DOT
    Interval{ 0x0889, 0x088E }, // Lo   [6] ARABIC LETTER NOON WITH INVERTED SMALL V..ARABIC VERTICAL TAIL
    Interval{ 0x0898, 0x089F }, // Mn   [8] ARABIC SMALL HIGH WORD AL-JUZ..ARABIC HALF MADDA OVER MADDA
    Interval{ 0x08A0, 0x08C8 }, // Lo  [41] ARABIC LETTER BEH WITH SMALL V BELOW..ARABIC LETTER GRAF
    Interval{ 0x08C9, 0x08C9 }, // Lm       ARABIC SMALL FARSI YEH
    Interval{ 0x08CA, 0x08E1 }, // Mn  [24] ARABIC SMALL HIGH FARSI YEH..ARABIC SMALL HIGH SIGN SAFHA
    Interval{ 0x08E3, 0x0902 }, // Mn  [32] ARABIC TURNED DAMMA BELOW..DEVANAGARI SIGN ANUSVARA
    Interval{ 0x0903, 0x0903 }, // Mc       DEVANAGARI SIGN VISARGA
    Interval{ 0x0904, 0x0939 }, // Lo  [54] DEVANAGARI LETTER SHORT A..DEVANAGARI LETTER HA
    Interval{ 0x093A, 0x093A }, // Mn       DEVANAGARI VOWEL SIGN OE
    Interval{ 0x093B, 0x093B }, // Mc       DEVANAGARI VOWEL SIGN OOE
    Interval{ 0x093C, 0x093C }, // Mn       DEVANAGARI SIGN NUKTA
    Interval{ 0x093D, 0x093D }, // Lo       DEVANAGARI SIGN AVAGRAHA
    Interval{ 0x093E, 0x0940 }, // Mc   [3] DEVANAGARI VOWEL SIGN AA..DEVANAGARI VOWEL SIGN II
    Interval{ 0x0941, 0x0948 }, // Mn   [8] DEVANAGARI VOWEL SIGN U..DEVANAGARI VOWEL SIGN AI
    Interval{ 0x0949, 0x094C }, // Mc   [4] DEVANAGARI VOWEL SIGN CANDRA O..DEVANAGARI VOWEL SIGN AU
    Interval{ 0x094D, 0x094D }, // Mn       DEVANAGARI SIGN VIRAMA
    Interval{ 0x094E, 0x094F }, // Mc   [2] DEVANAGARI VOWEL SIGN PRISHTHAMATRA E..DEVANAGARI VOWEL SIGN AW
    Interval{ 0x0950, 0x0950 }, // Lo       DEVANAGARI OM
    Interval{ 0x0951, 0x0957 }, // Mn   [7] DEVANAGARI STRESS SIGN UDATTA..DEVANAGARI VOWEL SIGN UUE
    Interval{ 0x0958, 0x0961 }, // Lo  [10] DEVANAGARI LETTER QA..DEVANAGARI LETTER VOCALIC LL
    Interval{ 0x0962, 0x0963 }, // Mn   [2] DEVANAGARI VOWEL SIGN VOCALIC L..DEVANAGARI VOWEL SIGN VOCALIC LL
    Interval{ 0x0966, 0x096F }, // Nd  [10] DEVANAGARI DIGIT ZERO..DEVANAGARI DIGIT NINE
    Interval{ 0x0971, 0x0971 }, // Lm       DEVANAGARI SIGN HIGH SPACING DOT
    Interval{ 0x0972, 0x0980 }, // Lo  [15] DEVANAGARI LETTER CANDRA A..BENGALI ANJI
    Interval{ 0x0981, 0x0981 }, // Mn       BENGALI SIGN CANDRABINDU
    Interval{ 0x0982, 0x0983 }, // Mc   [2] BENGALI SIGN ANUSVARA..BENGALI SIGN VISARGA
    Interval{ 0x0985, 0x098C }, // Lo   [8] BENGALI LETTER A..BENGALI LETTER VOCALIC L
    Interval{ 0x098F, 0x0990 }, // Lo   [2] BENGALI LETTER E..BENGALI LETTER AI
    Interval{ 0x0993, 0x09A8 }, // Lo  [22] BENGALI LETTER O..BENGALI LETTER NA
    Interval{ 0x09AA, 0x09B0 }, // Lo   [7] BENGALI LETTER PA..BENGALI LETTER RA
    Interval{ 0x09B2, 0x09B2 }, // Lo       BENGALI LETTER LA
    Interval{ 0x09B6, 0x09B9 }, // Lo   [4] BENGALI LETTER SHA..BENGALI LETTER HA
    Interval{ 0x09BC, 0x09BC }, // Mn       BENGALI SIGN NUKTA
    Interval{ 0x09BD, 0x09BD }, // Lo       BENGALI SIGN AVAGRAHA
    Interval{ 0x09BE, 0x09C0 }, // Mc   [3] BENGALI VOWEL SIGN AA..BENGALI VOWEL SIGN II
    Interval{ 0x09C1, 0x09C4 }, // Mn   [4] BENGALI VOWEL SIGN U..BENGALI VOWEL SIGN VOCALIC RR
    Interval{ 0x09C7, 0x09C8 }, // Mc   [2] BENGALI VOWEL SIGN E..BENGALI VOWEL SIGN AI
    Interval{ 0x09CB, 0x09CC }, // Mc   [2] BENGALI VOWEL SIGN O..BENGALI VOWEL SIGN AU
    Interval{ 0x09CD, 0x09CD }, // Mn       BENGALI SIGN VIRAMA
    Interval{ 0x09CE, 0x09CE }, // Lo       BENGALI LETTER KHANDA TA
    Interval{ 0x09D7, 0x09D7 }, // Mc       BENGALI AU LENGTH MARK
    Interval{ 0x09DC, 0x09DD }, // Lo   [2] BENGALI LETTER RRA..BENGALI LETTER RHA
    Interval{ 0x09DF, 0x09E1 }, // Lo   [3] BENGALI LETTER YYA..BENGALI LETTER VOCALIC LL
    Interval{ 0x09E2, 0x09E3 }, // Mn   [2] BENGALI VOWEL SIGN VOCALIC L..BENGALI VOWEL SIGN VOCALIC LL
    Interval{ 0x09E6, 0x09EF }, // Nd  [10] BENGALI DIGIT ZERO..BENGALI DIGIT NINE
    Interval{ 0x09F0, 0x09F1 }, // Lo   [2] BENGALI LETTER RA WITH MIDDLE DIAGONAL..BENGALI LETTER RA WITH LOWER DIAGONAL
    Interval{ 0x09FC, 0x09FC }, // Lo       BENGALI LETTER VEDIC ANUSVARA
    Interval{ 0x09FE, 0x09FE }, // Mn       BENGALI SANDHI MARK
    Interval{ 0x0A01, 0x0A02 }, // Mn   [2] GURMUKHI SIGN ADAK BINDI..GURMUKHI SIGN BINDI
    Interval{ 0x0A03, 0x0A03 }, // Mc       GURMUKHI SIGN VISARGA
    Interval{ 0x0A05, 0x0A0A }, // Lo   [6] GURMUKHI LETTER A..GURMUKHI LETTER UU
    Interval{ 0x0A0F, 0x0A10 }, // Lo   [2] GURMUKHI LETTER EE..GURMUKHI LETTER AI
    Interval{ 0x0A13, 0x0A28 }, // Lo  [22] GURMUKHI LETTER OO..GURMUKHI LETTER NA
    Interval{ 0x0A2A, 0x0A30 }, // Lo   [7] GURMUKHI LETTER PA..GURMUKHI LETTER RA
    Interval{ 0x0A32, 0x0A33 }, // Lo   [2] GURMUKHI LETTER LA..GURMUKHI LETTER LLA
    Interval{ 0x0A35, 0x0A36 }, // Lo   [2] GURMUKHI LETTER VA..GURMUKHI LETTER SHA
    Interval{ 0x0A38, 0x0A39 }, // Lo   [2] GURMUKHI LETTER SA..GURMUKHI LETTER HA
    Interval{ 0x0A3C, 0x0A3C }, // Mn       GURMUKHI SIGN NUKTA
    Interval{ 0x0A3E, 0x0A40 }, // Mc   [3] GURMUKHI VOWEL SIGN AA..GURMUKHI VOWEL SIGN II
    Interval{ 0x0A41, 0x0A42 }, // Mn   [2] GURMUKHI VOWEL SIGN U..GURMUKHI VOWEL SIGN UU
    Interval{ 0x0A47, 0x0A48 }, // Mn   [2] GURMUKHI VOWEL SIGN EE..GURMUKHI VOWEL SIGN AI
    Interval{ 0x0A4B, 0x0A4D }, // Mn   [3] GURMUKHI VOWEL SIGN OO..GURMUKHI SIGN VIRAMA
    Interval{ 0x0A51, 0x0A51 }, // Mn       GURMUKHI SIGN UDAAT
    Interval{ 0x0A59, 0x0A5C }, // Lo   [4] GURMUKHI LETTER KHHA..GURMUKHI LETTER RRA
    Interval{ 0x0A5E, 0x0A5E }, // Lo       GURMUKHI LETTER FA
    Interval{ 0x0A66, 0x0A6F }, // Nd  [10] GURMUKHI DIGIT ZERO..GURMUKHI DIGIT NINE
    Interval{ 0x0A70, 0x0A71 }, // Mn   [2] GURMUKHI TIPPI..GURMUKHI ADDAK
    Interval{ 0x0A72, 0x0A74 }, // Lo   [3] GURMUKHI IRI..GURMUKHI EK ONKAR
    Interval{ 0x0A75, 0x0A75 }, // Mn       GURMUKHI SIGN YAKASH
    Interval{ 0x0A81, 0x0A82 }, // Mn   [2] GUJARATI SIGN CANDRABINDU..GUJARATI SIGN ANUSVARA
    Interval{ 0x0A83, 0x0A83 }, // Mc       GUJARATI SIGN VISARGA
    Interval{ 0x0A85, 0x0A8D }, // Lo   [9] GUJARATI LETTER A..GUJARATI VOWEL CANDRA E
    Interval{ 0x0A8F, 0x0A91 }, // Lo   [3] GUJARATI LETTER E..GUJARATI VOWEL CANDRA O
    Interval{ 0x0A93, 0x0AA8 }, // Lo  [22] GUJARATI LETTER O..GUJARATI LETTER NA
    Interval{ 0x0AAA, 0x0AB0 }, // Lo   [7] GUJARATI LETTER PA..GUJARATI LETTER RA
    Interval{ 0x0AB2, 0x0AB3 }, // Lo   [2] GUJARATI LETTER LA..GUJARATI LETTER LLA
    Interval{ 0x0AB5, 0x0AB9 }, // Lo   [5] GUJARATI LETTER VA..GUJARATI LETTER HA
    Interval{ 0x0ABC, 0x0ABC }, // Mn       GUJARATI SIGN NUKTA
    Interval{ 0x0ABD, 0x0ABD }, // Lo       GUJARATI SIGN AVAGRAHA
    Interval{ 0x0ABE, 0x0AC0 }, // Mc   [3] GUJARATI VOWEL SIGN AA..GUJARATI VOWEL SIGN II
    Interval{ 0x0AC1, 0x0AC5 }, // Mn   [5] GUJARATI VOWEL SIGN U..GUJARATI VOWEL SIGN CANDRA E
    Interval{ 0x0AC7, 0x0AC8 }, // Mn   [2] GUJARATI VOWEL SIGN E..GUJARATI VOWEL SIGN AI
    Interval{ 0x0AC9, 0x0AC9 }, // Mc       GUJARATI VOWEL SIGN CANDRA O
    Interval{ 0x0ACB, 0x0ACC }, // Mc   [2] GUJARATI VOWEL SIGN O..GUJARATI VOWEL SIGN AU
    Interval{ 0x0ACD, 0x0ACD }, // Mn       GUJARATI SIGN VIRAMA
    Interval{ 0x0AD0, 0x0AD0 }, // Lo       GUJARATI OM
    Interval{ 0x0AE0, 0x0AE1 }, // Lo   [2] GUJARATI LETTER VOCALIC RR..GUJARATI LETTER VOCALIC LL
    Interval{ 0x0AE2, 0x0AE3 }, // Mn   [2] GUJARATI VOWEL SIGN VOCALIC L..GUJARATI VOWEL SIGN VOCALIC LL
    Interval{ 0x0AE6, 0x0AEF }, // Nd  [10] GUJARATI DIGIT ZERO..GUJARATI DIGIT NINE
    Interval{ 0x0AF9, 0x0AF9 }, // Lo       GUJARATI LETTER ZHA
    Interval{ 0x0AFA, 0x0AFF }, // Mn   [6] GUJARATI SIGN SUKUN..GUJARATI SIGN TWO-CIRCLE NUKTA ABOVE
    Interval{ 0x0B01, 0x0B01 }, // Mn       ORIYA SIGN CANDRABINDU
    Interval{ 0x0B02, 0x0B03 }, // Mc   [2] ORIYA SIGN ANUSVARA..ORIYA SIGN VISARGA
    Interval{ 0x0B05, 0x0B0C }, // Lo   [8] ORIYA LETTER A..ORIYA LETTER VOCALIC L
    Interval{ 0x0B0F, 0x0B10 }, // Lo   [2] ORIYA LETTER E..ORIYA LETTER AI
    Interval{ 0x0B13, 0x0B28 }, // Lo  [22] ORIYA LETTER O..ORIYA LETTER NA
    Interval{ 0x0B2A, 0x0B30 }, // Lo   [7] ORIYA LETTER PA..ORIYA LETTER RA
    Interval{ 0x0B32, 0x0B33 }, // Lo   [2] ORIYA LETTER LA..ORIYA LETTER LLA
    Interval{ 0x0B35, 0x0B39 }, // Lo   [5] ORIYA LETTER VA..ORIYA LETTER HA
    Interval{ 0x0B3C, 0x0B3C }, // Mn       ORIYA SIGN NUKTA
    Interval{ 0x0B3D, 0x0B3D }, // Lo       ORIYA SIGN AVAGRAHA
    Interval{ 0x0B3E, 0x0B3E }, // Mc       ORIYA VOWEL SIGN AA
    Interval{ 0x0B3F, 0x0B3F }, // Mn       ORIYA VOWEL SIGN I
    Interval{ 0x0B40, 0x0B40 }, // Mc       ORIYA VOWEL SIGN II
    Interval{ 0x0B41, 0x0B44 }, // Mn   [4] ORIYA VOWEL SIGN U..ORIYA VOWEL SIGN VOCALIC RR
    Interval{ 0x0B47, 0x0B48 }, // Mc   [2] ORIYA VOWEL SIGN E..ORIYA VOWEL SIGN AI
    Interval{ 0x0B4B, 0x0B4C }, // Mc   [2] ORIYA VOWEL SIGN O..ORIYA VOWEL SIGN AU
    Interval{ 0x0B4D, 0x0B4D }, // Mn       ORIYA SIGN VIRAMA
    Interval{ 0x0B55, 0x0B56 }, // Mn   [2] ORIYA SIGN OVERLINE..ORIYA AI LENGTH MARK
    Interval{ 0x0B57, 0x0B57 }, // Mc       ORIYA AU LENGTH MARK
    Interval{ 0x0B5C, 0x0B5D }, // Lo   [2] ORIYA LETTER RRA..ORIYA LETTER RHA
    Interval{ 0x0B5F, 0x0B61 }, // Lo   [3] ORIYA LETTER YYA..ORIYA LETTER VOCALIC LL
    Interval{ 0x0B62, 0x0B63 }, // Mn   [2] ORIYA VOWEL SIGN VOCALIC L..ORIYA VOWEL SIGN VOCALIC LL
    Interval{ 0x0B66, 0x0B6F }, // Nd  [10] ORIYA DIGIT ZERO..ORIYA DIGIT NINE
    Interval{ 0x0B71, 0x0B71 }, // Lo       ORIYA LETTER WA
    Interval{ 0x0B82, 0x0B82 }, // Mn       TAMIL SIGN ANUSVARA
    Interval{ 0x0B83, 0x0B83 }, // Lo       TAMIL SIGN VISARGA
    Interval{ 0x0B85, 0x0B8A }, // Lo   [6] TAMIL LETTER A..TAMIL LETTER UU
    Interval{ 0x0B8E, 0x0B90 }, // Lo   [3] TAMIL LETTER E..TAMIL LETTER AI
    Interval{ 0x0B92, 0x0B95 }, // Lo   [4] TAMIL LETTER O..TAMIL LETTER KA
    Interval{ 0x0B99, 0x0B9A }, // Lo   [2] TAMIL LETTER NGA..TAMIL LETTER CA
    Interval{ 0x0B9C, 0x0B9C }, // Lo       TAMIL LETTER JA
    Interval{ 0x0B9E, 0x0B9F }, // Lo   [2] TAMIL LETTER NYA..TAMIL LETTER TTA
    Interval{ 0x0BA3, 0x0BA4 }, // Lo   [2] TAMIL LETTER NNA..TAMIL LETTER TA
    Interval{ 0x0BA8, 0x0BAA }, // Lo   [3] TAMIL LETTER NA..TAMIL LETTER PA
    Interval{ 0x0BAE, 0x0BB9 }, // Lo  [12] TAMIL LETTER MA..TAMIL LETTER HA
    Interval{ 0x0BBE, 0x0BBF }, // Mc   [2] TAMIL VOWEL SIGN AA..TAMIL VOWEL SIGN I
    Interval{ 0x0BC0, 0x0BC0 }, // Mn       TAMIL VOWEL SIGN II
    Interval{ 0x0BC1, 0x0BC2 }, // Mc   [2] TAMIL VOWEL SIGN U..TAMIL VOWEL SIGN UU
    Interval{ 0x0BC6, 0x0BC8 }, // Mc   [3] TAMIL VOWEL SIGN E..TAMIL VOWEL SIGN AI
    Interval{ 0x0BCA, 0x0BCC }, // Mc   [3] TAMIL VOWEL SIGN O..TAMIL VOWEL SIGN AU
    Interval{ 0x0BCD, 0x0BCD }, // Mn       TAMIL SIGN VIRAMA
    Interval{ 0x0BD0, 0x0BD0 }, // Lo       TAMIL OM
    Interval{ 0x0BD7, 0x0BD7 }, // Mc       TAMIL AU LENGTH MARK
    Interval{ 0x0BE6, 0x0BEF }, // Nd  [10] TAMIL DIGIT ZERO..TAMIL DIGIT NINE
    Interval{ 0x0C00, 0x0C00 }, // Mn       TELUGU SIGN COMBINING CANDRABINDU ABOVE
    Interval{ 0x0C01, 0x0C03 }, // Mc   [3] TELUGU SIGN CANDRABINDU..TELUGU SIGN VISARGA
    Interval{ 0x0C04, 0x0C04 }, // Mn       TELUGU SIGN COMBINING ANUSVARA ABOVE
    Interval{ 0x0C05, 0x0C0C }, // Lo   [8] TELUGU LETTER A..TELUGU LETTER VOCALIC L
    Interval{ 0x0C0E, 0x0C10 }, // Lo   [3] TELUGU LETTER E..TELUGU LETTER AI
    Interval{ 0x0C12, 0x0C28 }, // Lo  [23] TELUGU LETTER O..TELUGU LETTER NA
    Interval{ 0x0C2A, 0x0C39 }, // Lo  [16] TELUGU LETTER PA..TELUGU LETTER HA
    Interval{ 0x0C3C, 0x0C3C }, // Mn       TELUGU SIGN NUKTA
    Interval{ 0x0C3D, 0x0C3D }, // Lo       TELUGU SIGN AVAGRAHA
    Interval{ 0x0C3E, 0x0C40 }, // Mn   [3] TELUGU VOWEL SIGN AA..TELUGU VOWEL SIGN II
    Interval{ 0x0C41, 0x0C44 }, // Mc   [4] TELUGU VOWEL SIGN U..TELUGU VOWEL SIGN VOCALIC RR
    Interval{ 0x0C46, 0x0C48 }, // Mn   [3] TELUGU VOWEL SIGN E..TELUGU VOWEL SIGN AI
    Interval{ 0x0C4A, 0x0C4D }, // Mn   [4] TELUGU VOWEL SIGN O..TELUGU SIGN VIRAMA
    Interval{ 0x0C55, 0x0C56 }, // Mn   [2] TELUGU LENGTH MARK..TELUGU AI LENGTH MARK
    Interval{ 0x0C58, 0x0C5A }, // Lo   [3] TELUGU LETTER TSA..TELUGU LETTER RRRA
    Interval{ 0x0C5D, 0x0C5D }, // Lo       TELUGU LETTER NAKAARA POLLU
    Interval{ 0x0C60, 0x0C61 }, // Lo   [2] TELUGU LETTER VOCALIC RR..TELUGU LETTER VOCALIC LL
    Interval{ 0x0C62, 0x0C63 }, // Mn   [2] TELUGU VOWEL SIGN VOCALIC L..TELUGU VOWEL SIGN VOCALIC LL
    Interval{ 0x0C66, 0x0C6F }, // Nd  [10] TELUGU DIGIT ZERO..TELUGU DIGIT NINE
    Interval{ 0x0C80, 0x0C80 }, // Lo       KANNADA SIGN SPACING CANDRABINDU
    Interval{ 0x0C81, 0x0C81 }, // Mn       KANNADA SIGN CANDRABINDU
    Interval{ 0x0C82, 0x0C83 }, // Mc   [2] KANNADA SIGN ANUSVARA..KANNADA SIGN VISARGA
    Interval{ 0x0C85, 0x0C8C }, // Lo   [8] KANNADA LETTER A..KANNADA LETTER VOCALIC L
    Interval{ 0x0C8E, 0x0C90 }, // Lo   [3] KANNADA LETTER E..KANNADA LETTER AI
    Interval{ 0x0C92, 0x0CA8 }, // Lo  [23] KANNADA LETTER O..KANNADA LETTER NA
    Interval{ 0x0CAA, 0x0CB3 }, // Lo  [10] KANNADA LETTER PA..KANNADA LETTER LLA
    Interval{ 0x0CB5, 0x0CB9 }, // Lo   [5] KANNADA LETTER VA..KANNADA LETTER HA
    Interval{ 0x0CBC, 0x0CBC }, // Mn       KANNADA SIGN NUKTA
    Interval{ 0x0CBD, 0x0CBD }, // Lo       KANNADA SIGN AVAGRAHA
    Interval{ 0x0CBE, 0x0CBE }, // Mc       KANNADA VOWEL SIGN AA
    Interval{ 0x0CBF, 0x0CBF }, // Mn       KANNADA VOWEL SIGN I
    Interval{ 0x0CC0, 0x0CC4 }, // Mc   [5] KANNADA VOWEL SIGN II..KANNADA VOWEL SIGN VOCALIC RR
    Interval{ 0x0CC6, 0x0CC6 }, // Mn       KANNADA VOWEL SIGN E
    Interval{ 0x0CC7, 0x0CC8 }, // Mc   [2] KANNADA VOWEL SIGN EE..KANNADA VOWEL SIGN AI
    Interval{ 0x0CCA, 0x0CCB }, // Mc   [2] KANNADA VOWEL SIGN O..KANNADA VOWEL SIGN OO
    Interval{ 0x0CCC, 0x0CCD }, // Mn   [2] KANNADA VOWEL SIGN AU..KANNADA SIGN VIRAMA
    Interval{ 0x0CD5, 0x0CD6 }, // Mc   [2] KANNADA LENGTH MARK..KANNADA AI LENGTH MARK
    Interval{ 0x0CDD, 0x0CDE }, // Lo   [2] KANNADA LETTER NAKAARA POLLU..KANNADA LETTER FA
    Interval{ 0x0CE0, 0x0CE1 }, // Lo   [2] KANNADA LETTER VOCALIC RR..KANNADA LETTER VOCALIC LL
    Interval{ 0x0CE2, 0x0CE3 }, // Mn   [2] KANNADA VOWEL SIGN VOCALIC L..KANNADA VOWEL SIGN VOCALIC LL
    Interval{ 0x0CE6, 0x0CEF }, // Nd  [10] KANNADA DIGIT ZERO..KANNADA DIGIT NINE
    Interval{ 0x0CF1, 0x0CF2 }, // Lo   [2] KANNADA SIGN JIHVAMULIYA..KANNADA SIGN UPADHMANIYA
    Interval{ 0x0CF3, 0x0CF3 }, // Mc       KANNADA SIGN COMBINING ANUSVARA ABOVE RIGHT
    Interval{ 0x0D00, 0x0D01 }, // Mn   [2] MALAYALAM SIGN COMBINING ANUSVARA ABOVE..MALAYALAM SIGN CANDRABINDU
    Interval{ 0x0D02, 0x0D03 }, // Mc   [2] MALAYALAM SIGN ANUSVARA..MALAYALAM SIGN VISARGA
    Interval{ 0x0D04, 0x0D0C }, // Lo   [9] MALAYALAM LETTER VEDIC ANUSVARA..MALAYALAM LETTER VOCALIC L
    Interval{ 0x0D0E, 0x0D10 }, // Lo   [3] MALAYALAM LETTER E..MALAYALAM LETTER AI
    Interval{ 0x0D12, 0x0D3A }, // Lo  [41] MALAYALAM LETTER O..MALAYALAM LETTER TTTA
    Interval{ 0x0D3B, 0x0D3C }, // Mn   [2] MALAYALAM SIGN VERTICAL BAR VIRAMA..MALAYALAM SIGN CIRCULAR VIRAMA
    Interval{ 0x0D3D, 0x0D3D }, // Lo       MALAYALAM SIGN AVAGRAHA
    Interval{ 0x0D3E, 0x0D40 }, // Mc   [3] MALAYALAM VOWEL SIGN AA..MALAYALAM VOWEL SIGN II
    Interval{ 0x0D41, 0x0D44 }, // Mn   [4] MALAYALAM VOWEL SIGN U..MALAYALAM VOWEL SIGN VOCALIC RR
    Interval{ 0x0D46, 0x0D48 }, // Mc   [3] MALAYALAM VOWEL SIGN E..MALAYALAM VOWEL SIGN AI
    Interval{ 0x0D4A, 0x0D4C }, // Mc   [3] MALAYALAM VOWEL SIGN O..MALAYALAM VOWEL SIGN AU
    Interval{ 0x0D4D, 0x0D4D }, // Mn       MALAYALAM SIGN VIRAMA
    Interval{ 0x0D4E, 0x0D4E }, // Lo       MALAYALAM LETTER DOT REPH
    Interval{ 0x0D54, 0x0D56 }, // Lo   [3] MALAYALAM LETTER CHILLU M..MALAYALAM LETTER CHILLU LLL
    Interval{ 0x0D57, 0x0D57 }, // Mc       MALAYALAM AU LENGTH MARK
    Interval{ 0x0D5F, 0x0D61 }, // Lo   [3] MALAYALAM LETTER ARCHAIC II..MALAYALAM LETTER VOCALIC LL
    Interval{ 0x0D62, 0x0D63 }, // Mn   [2] MALAYALAM VOWEL SIGN VOCALIC L..MALAYALAM VOWEL SIGN VOCALIC LL
    Interval{ 0x0D66, 0x0D6F }, // Nd  [10] MALAYALAM DIGIT ZERO..MALAYALAM DIGIT NINE
    Interval{ 0x0D7A, 0x0D7F }, // Lo   [6] MALAYALAM LETTER CHILLU NN..MALAYALAM LETTER CHILLU K
    Interval{ 0x0D81, 0x0D81 }, // Mn       SINHALA SIGN CANDRABINDU
    Interval{ 0x0D82, 0x0D83 }, // Mc   [2] SINHALA SIGN ANUSVARAYA..SINHALA SIGN VISARGAYA
    Interval{ 0x0D85, 0x0D96 }, // Lo  [18] SINHALA LETTER AYANNA..SINHALA LETTER AUYANNA
    Interval{ 0x0D9A, 0x0DB1 }, // Lo  [24] SINHALA LETTER ALPAPRAANA KAYANNA..SINHALA LETTER DANTAJA NAYANNA
    Interval{ 0x0DB3, 0x0DBB }, // Lo   [9] SINHALA LETTER SANYAKA DAYANNA..SINHALA LETTER RAYANNA
    Interval{ 0x0DBD, 0x0DBD }, // Lo       SINHALA LETTER DANTAJA LAYANNA
    Interval{ 0x0DC0, 0x0DC6 }, // Lo   [7] SINHALA LETTER VAYANNA..SINHALA LETTER FAYANNA
    Interval{ 0x0DCA, 0x0DCA }, // Mn       SINHALA SIGN AL-LAKUNA
    Interval{ 0x0DCF, 0x0DD1 }, // Mc   [3] SINHALA VOWEL SIGN AELA-PILLA..SINHALA VOWEL SIGN DIGA AEDA-PILLA
    Interval{ 0x0DD2, 0x0DD4 }, // Mn   [3] SINHALA VOWEL SIGN KETTI IS-PILLA..SINHALA VOWEL SIGN KETTI PAA-PILLA
    Interval{ 0x0DD6, 0x0DD6 }, // Mn       SINHALA VOWEL SIGN DIGA PAA-PILLA
    Interval{ 0x0DD8, 0x0DDF }, // Mc   [8] SINHALA VOWEL SIGN GAETTA-PILLA..SINHALA VOWEL SIGN GAYANUKITTA
    Interval{ 0x0DE6, 0x0DEF }, // Nd  [10] SINHALA LITH DIGIT ZERO..SINHALA LITH DIGIT NINE
    Interval{ 0x0DF2, 0x0DF3 }, // Mc   [2] SINHALA VOWEL SIGN DIGA GAETTA-PILLA..SINHALA VOWEL SIGN DIGA GAYANUKITTA
    Interval{ 0x0E01, 0x0E30 }, // Lo  [48] THAI CHARACTER KO KAI..THAI CHARACTER SARA A
    Interval{ 0x0E31, 0x0E31 }, // Mn       THAI CHARACTER MAI HAN-AKAT
    Interval{ 0x0E32, 0x0E33 }, // Lo   [2] THAI CHARACTER SARA AA..THAI CHARACTER SARA AM
    Interval{ 0x0E34, 0x0E3A }, // Mn   [7] THAI CHARACTER SARA I..THAI CHARACTER PHINTHU
    Interval{ 0x0E40, 0x0E45 }, // Lo   [6] THAI CHARACTER SARA E..THAI CHARACTER LAKKHANGYAO
    Interval{ 0x0E46, 0x0E46 }, // Lm       THAI CHARACTER MAIYAMOK
    Interval{ 0x0E47, 0x0E4E }, // Mn   [8] THAI CHARACTER MAITAIKHU..THAI CHARACTER YAMAKKAN
    Interval{ 0x0E50, 0x0E59 }, // Nd  [10] THAI DIGIT ZERO..THAI DIGIT NINE
    Interval{ 0x0E81, 0x0E82 }, // Lo   [2] LAO LETTER KO..LAO LETTER KHO SUNG
    Interval{ 0x0E84, 0x0E84 }, // Lo       LAO LETTER KHO TAM
    Interval{ 0x0E86, 0x0E8A }, // Lo   [5] LAO LETTER PALI GHA..LAO LETTER SO TAM
    Interval{ 0x0E8C, 0x0EA3 }, // Lo  [24] LAO LETTER PALI JHA..LAO LETTER LO LING
    Interval{ 0x0EA5, 0x0EA5 }, // Lo       LAO LETTER LO LOOT
    Interval{ 0x0EA7, 0x0EB0 }, // Lo  [10] LAO LETTER WO..LAO VOWEL SIGN A
    Interval{ 0x0EB1, 0x0EB1 }, // Mn       LAO VOWEL SIGN MAI KAN
    Interval{ 0x0EB2, 0x0EB3 }, // Lo   [2] LAO VOWEL SIGN AA..LAO VOWEL SIGN AM
    Interval{ 0x0EB4, 0x0EBC }, // Mn   [9] LAO VOWEL SIGN I..LAO SEMIVOWEL SIGN LO
    Interval{ 0x0EBD, 0x0EBD }, // Lo       LAO SEMIVOWEL SIGN NYO
    Interval{ 0x0EC0, 0x0EC4 }, // Lo   [5] LAO VOWEL SIGN E..LAO VOWEL SIGN AI
    Interval{ 0x0EC6, 0x0EC6 }, // Lm       LAO KO LA
    Interval{ 0x0EC8, 0x0ECE }, // Mn   [7] LAO TONE MAI EK..LAO YAMAKKAN
    Interval{ 0x0ED0, 0x0ED9 }, // Nd  [10] LAO DIGIT ZERO..LAO DIGIT NINE
    Interval{ 0x0EDC, 0x0EDF }, // Lo   [4] LAO HO NO..LAO LETTER KHMU NYO
    Interval{ 0x0F00, 0x0F00 }, // Lo       TIBETAN SYLLABLE OM
    Interval{ 0x0F18, 0x0F19 }, // Mn   [2] TIBETAN ASTROLOGICAL SIGN -KHYUD PA..TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS
    Interval{ 0x0F20, 0x0F29 }, // Nd  [10] TIBETAN DIGIT ZERO..TIBETAN DIGIT NINE
    Interval{ 0x0F35, 0x0F35 }, // Mn       TIBETAN MARK NGAS BZUNG NYI ZLA
    Interval{ 0x0F37, 0x0F37 }, // Mn       TIBETAN MARK NGAS BZUNG SGOR RTAGS
    Interval{ 0x0F39, 0x0F39 }, // Mn       TIBETAN MARK TSA -PHRU
    Interval{ 0x0F3E, 0x0F3F }, // Mc   [2] TIBETAN SIGN YAR TSHES..TIBETAN SIGN MAR TSHES
    Interval{ 0x0F40, 0x0F47 }, // Lo   [8] TIBETAN LETTER KA..TIBETAN LETTER JA
    Interval{ 0x0F49, 0x0F6C }, // Lo  [36] TIBETAN LETTER NYA..TIBETAN LETTER RRA
    Interval{ 0x0F71, 0x0F7E }, // Mn  [14] TIBETAN VOWEL SIGN AA..TIBETAN SIGN RJES SU NGA RO
    Interval{ 0x0F7F, 0x0F7F }, // Mc       TIBETAN SIGN RNAM BCAD
    Interval{ 0x0F80, 0x0F84 }, // Mn   [5] TIBETAN VOWEL SIGN REVERSED I..TIBETAN MARK HALANTA
    Interval{ 0x0F86, 0x0F87 }, // Mn   [2] TIBETAN SIGN LCI RTAGS..TIBETAN SIGN YANG RTAGS
    Interval{ 0x0F88, 0x0F8C }, // Lo   [5] TIBETAN SIGN LCE TSA CAN..TIBETAN SIGN INVERTED MCHU CAN
    Interval{ 0x0F8D, 0x0F97 }, // Mn  [11] TIBETAN SUBJOINED SIGN LCE TSA CAN..TIBETAN SUBJOINED LETTER JA
    Interval{ 0x0F99, 0x0FBC }, // Mn  [36] TIBETAN SUBJOINED LETTER NYA..TIBETAN SUBJOINED LETTER FIXED-FORM RA
    Interval{ 0x0FC6, 0x0FC6 }, // Mn       TIBETAN SYMBOL PADMA GDAN
    Interval{ 0x1000, 0x102A }, // Lo  [43] MYANMAR LETTER KA..MYANMAR LETTER AU
    Interval{ 0x102B, 0x102C }, // Mc   [2] MYANMAR VOWEL SIGN TALL AA..MYANMAR VOWEL SIGN AA
    Interval{ 0x102D, 0x1030 }, // Mn   [4] MYANMAR VOWEL SIGN I..MYANMAR VOWEL SIGN UU
    Interval{ 0x1031, 0x1031 }, // Mc       MYANMAR VOWEL SIGN E
    Interval{ 0x1032, 0x1037 }, // Mn   [6] MYANMAR VOWEL SIGN AI..MYANMAR SIGN DOT BELOW
    Interval{ 0x1038, 0x1038 }, // Mc       MYANMAR SIGN VISARGA
    Interval{ 0x1039, 0x103A }, // Mn   [2] MYANMAR SIGN VIRAMA..MYANMAR SIGN ASAT
    Interval{ 0x103B, 0x103C }, // Mc   [2] MYANMAR CONSONANT SIGN MEDIAL YA..MYANMAR CONSONANT SIGN MEDIAL RA
    Interval{ 0x103D, 0x103E }, // Mn   [2] MYANMAR CONSONANT SIGN MEDIAL WA..MYANMAR CONSONANT SIGN MEDIAL HA
    Interval{ 0x103F, 0x103F }, // Lo       MYANMAR LETTER GREAT SA
    Interval{ 0x1040, 0x1049 }, // Nd  [10] MYANMAR DIGIT ZERO..MYANMAR DIGIT NINE
    Interval{ 0x1050, 0x1055 }, // Lo   [6] MYANMAR LETTER SHA..MYANMAR LETTER VOCALIC LL
    Interval{ 0x1056, 0x1057 }, // Mc   [2] MYANMAR VOWEL SIGN VOCALIC R..MYANMAR VOWEL SIGN VOCALIC RR
    Interval{ 0x1058, 0x1059 }, // Mn   [2] MYANMAR VOWEL SIGN VOCALIC L..MYANMAR VOWEL SIGN VOCALIC LL
    Interval{ 0x105A, 0x105D }, // Lo   [4] MYANMAR LETTER MON NGA..MYANMAR LETTER MON BBE
    Interval{ 0x105E, 0x1060 }, // Mn   [3] MYANMAR CONSONANT SIGN MON MEDIAL NA..MYANMAR CONSONANT SIGN MON MEDIAL LA
    Interval{ 0x1061, 0x1061 }, // Lo       MYANMAR LETTER SGAW KAREN SHA
    Interval{ 0x1062, 0x1064 }, // Mc   [3] MYANMAR VOWEL SIGN SGAW KAREN EU..MYANMAR TONE MARK SGAW KAREN KE PHO
    Interval{ 0x1065, 0x1066 }, // Lo   [2] MYANMAR LETTER WESTERN PWO KAREN THA..MYANMAR LETTER WESTERN PWO KAREN PWA
    Interval{ 0x1067, 0x106D }, // Mc   [7] MYANMAR VOWEL SIGN WESTERN PWO KAREN EU..MYANMAR SIGN WESTERN PWO KAREN TONE-5
    Interval{ 0x106E, 0x1070 }, // Lo   [3] MYANMAR LETTER EASTERN PWO KAREN NNA..MYANMAR LETTER EASTERN PWO KAREN GHWA
    Interval{ 0x1071, 0x1074 }, // Mn   [4] MYANMAR VOWEL SIGN GEBA KAREN I..MYANMAR VOWEL SIGN KAYAH EE
    Interval{ 0x1075, 0x1081 }, // Lo  [13] MYANMAR LETTER SHAN KA..MYANMAR LETTER SHAN HA
    Interval{ 0x1082, 0x1082 }, // Mn       MYANMAR CONSONANT SIGN SHAN MEDIAL WA
    Interval{ 0x1083, 0x1084 }, // Mc   [2] MYANMAR VOWEL SIGN SHAN AA..MYANMAR VOWEL SIGN SHAN E
    Interval{ 0x1085, 0x1086 }, // Mn   [2] MYANMAR VOWEL SIGN SHAN E ABOVE..MYANMAR VOWEL SIGN SHAN FINAL Y
    Interval{ 0x1087, 0x108C }, // Mc   [6] MYANMAR SIGN SHAN TONE-2..MYANMAR SIGN SHAN COUNCIL TONE-3
    Interval{ 0x108D, 0x108D }, // Mn       MYANMAR SIGN SHAN COUNCIL EMPHATIC TONE
    Interval{ 0x108E, 0x108E }, // Lo       MYANMAR LETTER RUMAI PALAUNG FA
    Interval{ 0x108F, 0x108F }, // Mc       MYANMAR SIGN RUMAI PALAUNG TONE-5
    Interval{ 0x1090, 0x1099 }, // Nd  [10] MYANMAR SHAN DIGIT ZERO..MYANMAR SHAN DIGIT NINE
    Interval{ 0x109A, 0x109C }, // Mc   [3] MYANMAR SIGN KHAMTI TONE-1..MYANMAR VOWEL SIGN AITON A
    Interval{ 0x109D, 0x109D }, // Mn       MYANMAR VOWEL SIGN AITON AI
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FC, 0x10FC }, // Lm       MODIFIER LETTER GEORGIAN NAR
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x1100, 0x1248 }, // Lo [329] HANGUL CHOSEONG KIYEOK..ETHIOPIC SYLLABLE QWA
    Interval{ 0x124A, 0x124D }, // Lo   [4] ETHIOPIC SYLLABLE QWI..ETHIOPIC SYLLABLE QWE
    Interval{ 0x1250, 0x1256 }, // Lo   [7] ETHIOPIC SYLLABLE QHA..ETHIOPIC SYLLABLE QHO
    Interval{ 0x1258, 0x1258 }, // Lo       ETHIOPIC SYLLABLE QHWA
    Interval{ 0x125A, 0x125D }, // Lo   [4] ETHIOPIC SYLLABLE QHWI..ETHIOPIC SYLLABLE QHWE
    Interval{ 0x1260, 0x1288 }, // Lo  [41] ETHIOPIC SYLLABLE BA..ETHIOPIC SYLLABLE XWA
    Interval{ 0x128A, 0x128D }, // Lo   [4] ETHIOPIC SYLLABLE XWI..ETHIOPIC SYLLABLE XWE
    Interval{ 0x1290, 0x12B0 }, // Lo  [33] ETHIOPIC SYLLABLE NA..ETHIOPIC SYLLABLE KWA
    Interval{ 0x12B2, 0x12B5 }, // Lo   [4] ETHIOPIC SYLLABLE KWI..ETHIOPIC SYLLABLE KWE
    Interval{ 0x12B8, 0x12BE }, // Lo   [7] ETHIOPIC SYLLABLE KXA..ETHIOPIC SYLLABLE KXO
    Interval{ 0x12C0, 0x12C0 }, // Lo       ETHIOPIC SYLLABLE KXWA
    Interval{ 0x12C2, 0x12C5 }, // Lo   [4] ETHIOPIC SYLLABLE KXWI..ETHIOPIC SYLLABLE KXWE
    Interval{ 0x12C8, 0x12D6 }, // Lo  [15] ETHIOPIC SYLLABLE WA..ETHIOPIC SYLLABLE PHARYNGEAL O
    Interval{ 0x12D8, 0x1310 }, // Lo  [57] ETHIOPIC SYLLABLE ZA..ETHIOPIC SYLLABLE GWA
    Interval{ 0x1312, 0x1315 }, // Lo   [4] ETHIOPIC SYLLABLE GWI..ETHIOPIC SYLLABLE GWE
    Interval{ 0x1318, 0x135A }, // Lo  [67] ETHIOPIC SYLLABLE GGA..ETHIOPIC SYLLABLE FYA
    Interval{ 0x135D, 0x135F }, // Mn   [3] ETHIOPIC COMBINING GEMINATION AND VOWEL LENGTH MARK..ETHIOPIC COMBINING GEMINATION MARK
    Interval{ 0x1369, 0x1371 }, // No   [9] ETHIOPIC DIGIT ONE..ETHIOPIC DIGIT NINE
    Interval{ 0x1380, 0x138F }, // Lo  [16] ETHIOPIC SYLLABLE SEBATBEIT MWA..ETHIOPIC SYLLABLE PWE
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1401, 0x166C }, // Lo [620] CANADIAN SYLLABICS E..CANADIAN SYLLABICS CARRIER TTSA
    Interval{ 0x166F, 0x167F }, // Lo  [17] CANADIAN SYLLABICS QAI..CANADIAN SYLLABICS BLACKFOOT W
    Interval{ 0x1681, 0x169A }, // Lo  [26] OGHAM LETTER BEITH..OGHAM LETTER PEITH
    Interval{ 0x16A0, 0x16EA }, // Lo  [75] RUNIC LETTER FEHU FEOH FE F..RUNIC LETTER X
    Interval{ 0x16EE, 0x16F0 }, // Nl   [3] RUNIC ARLAUG SYMBOL..RUNIC BELGTHOR SYMBOL
    Interval{ 0x16F1, 0x16F8 }, // Lo   [8] RUNIC LETTER K..RUNIC LETTER FRANKS CASKET AESC
    Interval{ 0x1700, 0x1711 }, // Lo  [18] TAGALOG LETTER A..TAGALOG LETTER HA
    Interval{ 0x1712, 0x1714 }, // Mn   [3] TAGALOG VOWEL SIGN I..TAGALOG SIGN VIRAMA
    Interval{ 0x1715, 0x1715 }, // Mc       TAGALOG SIGN PAMUDPOD
    Interval{ 0x171F, 0x1731 }, // Lo  [19] TAGALOG LETTER ARCHAIC RA..HANUNOO LETTER HA
    Interval{ 0x1732, 0x1733 }, // Mn   [2] HANUNOO VOWEL SIGN I..HANUNOO VOWEL SIGN U
    Interval{ 0x1734, 0x1734 }, // Mc       HANUNOO SIGN PAMUDPOD
    Interval{ 0x1740, 0x1751 }, // Lo  [18] BUHID LETTER A..BUHID LETTER HA
    Interval{ 0x1752, 0x1753 }, // Mn   [2] BUHID VOWEL SIGN I..BUHID VOWEL SIGN U
    Interval{ 0x1760, 0x176C }, // Lo  [13] TAGBANWA LETTER A..TAGBANWA LETTER YA
    Interval{ 0x176E, 0x1770 }, // Lo   [3] TAGBANWA LETTER LA..TAGBANWA LETTER SA
    Interval{ 0x1772, 0x1773 }, // Mn   [2] TAGBANWA VOWEL SIGN I..TAGBANWA VOWEL SIGN U
    Interval{ 0x1780, 0x17B3 }, // Lo  [52] KHMER LETTER KA..KHMER INDEPENDENT VOWEL QAU
    Interval{ 0x17B4, 0x17B5 }, // Mn   [2] KHMER VOWEL INHERENT AQ..KHMER VOWEL INHERENT AA
    Interval{ 0x17B6, 0x17B6 }, // Mc       KHMER VOWEL SIGN AA
    Interval{ 0x17B7, 0x17BD }, // Mn   [7] KHMER VOWEL SIGN I..KHMER VOWEL SIGN UA
    Interval{ 0x17BE, 0x17C5 }, // Mc   [8] KHMER VOWEL SIGN OE..KHMER VOWEL SIGN AU
    Interval{ 0x17C6, 0x17C6 }, // Mn       KHMER SIGN NIKAHIT
    Interval{ 0x17C7, 0x17C8 }, // Mc   [2] KHMER SIGN REAHMUK..KHMER SIGN YUUKALEAPINTU
    Interval{ 0x17C9, 0x17D3 }, // Mn  [11] KHMER SIGN MUUSIKATOAN..KHMER SIGN BATHAMASAT
    Interval{ 0x17D7, 0x17D7 }, // Lm       KHMER SIGN LEK TOO
    Interval{ 0x17DC, 0x17DC }, // Lo       KHMER SIGN AVAKRAHASANYA
    Interval{ 0x17DD, 0x17DD }, // Mn       KHMER SIGN ATTHACAN
    Interval{ 0x17E0, 0x17E9 }, // Nd  [10] KHMER DIGIT ZERO..KHMER DIGIT NINE
    Interval{ 0x180B, 0x180D }, // Mn   [3] MONGOLIAN FREE VARIATION SELECTOR ONE..MONGOLIAN FREE VARIATION SELECTOR THREE
    Interval{ 0x180F, 0x180F }, // Mn       MONGOLIAN FREE VARIATION SELECTOR FOUR
    Interval{ 0x1810, 0x1819 }, // Nd  [10] MONGOLIAN DIGIT ZERO..MONGOLIAN DIGIT NINE
    Interval{ 0x1820, 0x1842 }, // Lo  [35] MONGOLIAN LETTER A..MONGOLIAN LETTER CHI
    Interval{ 0x1843, 0x1843 }, // Lm       MONGOLIAN LETTER TODO LONG VOWEL SIGN
    Interval{ 0x1844, 0x1878 }, // Lo  [53] MONGOLIAN LETTER TODO E..MONGOLIAN LETTER CHA WITH TWO DOTS
    Interval{ 0x1880, 0x1884 }, // Lo   [5] MONGOLIAN LETTER ALI GALI ANUSVARA ONE..MONGOLIAN LETTER ALI GALI INVERTED UBADAMA
    Interval{ 0x1885, 0x1886 }, // Mn   [2] MONGOLIAN LETTER ALI GALI BALUDA..MONGOLIAN LETTER ALI GALI THREE BALUDA
    Interval{ 0x1887, 0x18A8 }, // Lo  [34] MONGOLIAN LETTER ALI GALI A..MONGOLIAN LETTER MANCHU ALI GALI BHA
    Interval{ 0x18A9, 0x18A9 }, // Mn       MONGOLIAN LETTER ALI GALI DAGALGA
    Interval{ 0x18AA, 0x18AA }, // Lo       MONGOLIAN LETTER MANCHU ALI GALI LHA
    Interval{ 0x18B0, 0x18F5 }, // Lo  [70] CANADIAN SYLLABICS OY..CANADIAN SYLLABICS CARRIER DENTAL S
    Interval{ 0x1900, 0x191E }, // Lo  [31] LIMBU VOWEL-CARRIER LETTER..LIMBU LETTER TRA
    Interval{ 0x1920, 0x1922 }, // Mn   [3] LIMBU VOWEL SIGN A..LIMBU VOWEL SIGN U
    Interval{ 0x1923, 0x1926 }, // Mc   [4] LIMBU VOWEL SIGN EE..LIMBU VOWEL SIGN AU
    Interval{ 0x1927, 0x1928 }, // Mn   [2] LIMBU VOWEL SIGN E..LIMBU VOWEL SIGN O
    Interval{ 0x1929, 0x192B }, // Mc   [3] LIMBU SUBJOINED LETTER YA..LIMBU SUBJOINED LETTER WA
    Interval{ 0x1930, 0x1931 }, // Mc   [2] LIMBU SMALL LETTER KA..LIMBU SMALL LETTER NGA
    Interval{ 0x1932, 0x1932 }, // Mn       LIMBU SMALL LETTER ANUSVARA
    Interval{ 0x1933, 0x1938 }, // Mc   [6] LIMBU SMALL LETTER TA..LIMBU SMALL LETTER LA
    Interval{ 0x1939, 0x193B }, // Mn   [3] LIMBU SIGN MUKPHRENG..LIMBU SIGN SA-I
    Interval{ 0x1946, 0x194F }, // Nd  [10] LIMBU DIGIT ZERO..LIMBU DIGIT NINE
    Interval{ 0x1950, 0x196D }, // Lo  [30] TAI LE LETTER KA..TAI LE LETTER AI
    Interval{ 0x1970, 0x1974 }, // Lo   [5] TAI LE LETTER TONE-2..TAI LE LETTER TONE-6
    Interval{ 0x1980, 0x19AB }, // Lo  [44] NEW TAI LUE LETTER HIGH QA..NEW TAI LUE LETTER LOW SUA
    Interval{ 0x19B0, 0x19C9 }, // Lo  [26] NEW TAI LUE VOWEL SIGN VOWEL SHORTENER..NEW TAI LUE TONE MARK-2
    Interval{ 0x19D0, 0x19D9 }, // Nd  [10] NEW TAI LUE DIGIT ZERO..NEW TAI LUE DIGIT NINE
    Interval{ 0x19DA, 0x19DA }, // No       NEW TAI LUE THAM DIGIT ONE
    Interval{ 0x1A00, 0x1A16 }, // Lo  [23] BUGINESE LETTER KA..BUGINESE LETTER HA
    Interval{ 0x1A17, 0x1A18 }, // Mn   [2] BUGINESE VOWEL SIGN I..BUGINESE VOWEL SIGN U
    Interval{ 0x1A19, 0x1A1A }, // Mc   [2] BUGINESE VOWEL SIGN E..BUGINESE VOWEL SIGN O
    Interval{ 0x1A1B, 0x1A1B }, // Mn       BUGINESE VOWEL SIGN AE
    Interval{ 0x1A20, 0x1A54 }, // Lo  [53] TAI THAM LETTER HIGH KA..TAI THAM LETTER GREAT SA
    Interval{ 0x1A55, 0x1A55 }, // Mc       TAI THAM CONSONANT SIGN MEDIAL RA
    Interval{ 0x1A56, 0x1A56 }, // Mn       TAI THAM CONSONANT SIGN MEDIAL LA
    Interval{ 0x1A57, 0x1A57 }, // Mc       TAI THAM CONSONANT SIGN LA TANG LAI
    Interval{ 0x1A58, 0x1A5E }, // Mn   [7] TAI THAM SIGN MAI KANG LAI..TAI THAM CONSONANT SIGN SA
    Interval{ 0x1A60, 0x1A60 }, // Mn       TAI THAM SIGN SAKOT
    Interval{ 0x1A61, 0x1A61 }, // Mc       TAI THAM VOWEL SIGN A
    Interval{ 0x1A62, 0x1A62 }, // Mn       TAI THAM VOWEL SIGN MAI SAT
    Interval{ 0x1A63, 0x1A64 }, // Mc   [2] TAI THAM VOWEL SIGN AA..TAI THAM VOWEL SIGN TALL AA
    Interval{ 0x1A65, 0x1A6C }, // Mn   [8] TAI THAM VOWEL SIGN I..TAI THAM VOWEL SIGN OA BELOW
    Interval{ 0x1A6D, 0x1A72 }, // Mc   [6] TAI THAM VOWEL SIGN OY..TAI THAM VOWEL SIGN THAM AI
    Interval{ 0x1A73, 0x1A7C }, // Mn  [10] TAI THAM VOWEL SIGN OA ABOVE..TAI THAM SIGN KHUEN-LUE KARAN
    Interval{ 0x1A7F, 0x1A7F }, // Mn       TAI THAM COMBINING CRYPTOGRAMMIC DOT
    Interval{ 0x1A80, 0x1A89 }, // Nd  [10] TAI THAM HORA DIGIT ZERO..TAI THAM HORA DIGIT NINE
    Interval{ 0x1A90, 0x1A99 }, // Nd  [10] TAI THAM THAM DIGIT ZERO..TAI THAM THAM DIGIT NINE
    Interval{ 0x1AA7, 0x1AA7 }, // Lm       TAI THAM SIGN MAI YAMOK
    Interval{ 0x1AB0, 0x1ABD }, // Mn  [14] COMBINING DOUBLED CIRCUMFLEX ACCENT..COMBINING PARENTHESES BELOW
    Interval{ 0x1ABF, 0x1ACE }, // Mn  [16] COMBINING LATIN SMALL LETTER W BELOW..COMBINING LATIN SMALL LETTER INSULAR T
    Interval{ 0x1B00, 0x1B03 }, // Mn   [4] BALINESE SIGN ULU RICEM..BALINESE SIGN SURANG
    Interval{ 0x1B04, 0x1B04 }, // Mc       BALINESE SIGN BISAH
    Interval{ 0x1B05, 0x1B33 }, // Lo  [47] BALINESE LETTER AKARA..BALINESE LETTER HA
    Interval{ 0x1B34, 0x1B34 }, // Mn       BALINESE SIGN REREKAN
    Interval{ 0x1B35, 0x1B35 }, // Mc       BALINESE VOWEL SIGN TEDUNG
    Interval{ 0x1B36, 0x1B3A }, // Mn   [5] BALINESE VOWEL SIGN ULU..BALINESE VOWEL SIGN RA REPA
    Interval{ 0x1B3B, 0x1B3B }, // Mc       BALINESE VOWEL SIGN RA REPA TEDUNG
    Interval{ 0x1B3C, 0x1B3C }, // Mn       BALINESE VOWEL SIGN LA LENGA
    Interval{ 0x1B3D, 0x1B41 }, // Mc   [5] BALINESE VOWEL SIGN LA LENGA TEDUNG..BALINESE VOWEL SIGN TALING REPA TEDUNG
    Interval{ 0x1B42, 0x1B42 }, // Mn       BALINESE VOWEL SIGN PEPET
    Interval{ 0x1B43, 0x1B44 }, // Mc   [2] BALINESE VOWEL SIGN PEPET TEDUNG..BALINESE ADEG ADEG
    Interval{ 0x1B45, 0x1B4C }, // Lo   [8] BALINESE LETTER KAF SASAK..BALINESE LETTER ARCHAIC JNYA
    Interval{ 0x1B50, 0x1B59 }, // Nd  [10] BALINESE DIGIT ZERO..BALINESE DIGIT NINE
    Interval{ 0x1B6B, 0x1B73 }, // Mn   [9] BALINESE MUSICAL SYMBOL COMBINING TEGEH..BALINESE MUSICAL SYMBOL COMBINING GONG
    Interval{ 0x1B80, 0x1B81 }, // Mn   [2] SUNDANESE SIGN PANYECEK..SUNDANESE SIGN PANGLAYAR
    Interval{ 0x1B82, 0x1B82 }, // Mc       SUNDANESE SIGN PANGWISAD
    Interval{ 0x1B83, 0x1BA0 }, // Lo  [30] SUNDANESE LETTER A..SUNDANESE LETTER HA
    Interval{ 0x1BA1, 0x1BA1 }, // Mc       SUNDANESE CONSONANT SIGN PAMINGKAL
    Interval{ 0x1BA2, 0x1BA5 }, // Mn   [4] SUNDANESE CONSONANT SIGN PANYAKRA..SUNDANESE VOWEL SIGN PANYUKU
    Interval{ 0x1BA6, 0x1BA7 }, // Mc   [2] SUNDANESE VOWEL SIGN PANAELAENG..SUNDANESE VOWEL SIGN PANOLONG
    Interval{ 0x1BA8, 0x1BA9 }, // Mn   [2] SUNDANESE VOWEL SIGN PAMEPET..SUNDANESE VOWEL SIGN PANEULEUNG
    Interval{ 0x1BAA, 0x1BAA }, // Mc       SUNDANESE SIGN PAMAAEH
    Interval{ 0x1BAB, 0x1BAD }, // Mn   [3] SUNDANESE SIGN VIRAMA..SUNDANESE CONSONANT SIGN PASANGAN WA
    Interval{ 0x1BAE, 0x1BAF }, // Lo   [2] SUNDANESE LETTER KHA..SUNDANESE LETTER SYA
    Interval{ 0x1BB0, 0x1BB9 }, // Nd  [10] SUNDANESE DIGIT ZERO..SUNDANESE DIGIT NINE
    Interval{ 0x1BBA, 0x1BE5 }, // Lo  [44] SUNDANESE AVAGRAHA..BATAK LETTER U
    Interval{ 0x1BE6, 0x1BE6 }, // Mn       BATAK SIGN TOMPI
    Interval{ 0x1BE7, 0x1BE7 }, // Mc       BATAK VOWEL SIGN E
    Interval{ 0x1BE8, 0x1BE9 }, // Mn   [2] BATAK VOWEL SIGN PAKPAK E..BATAK VOWEL SIGN EE
    Interval{ 0x1BEA, 0x1BEC }, // Mc   [3] BATAK VOWEL SIGN I..BATAK VOWEL SIGN O
    Interval{ 0x1BED, 0x1BED }, // Mn       BATAK VOWEL SIGN KARO O
    Interval{ 0x1BEE, 0x1BEE }, // Mc       BATAK VOWEL SIGN U
    Interval{ 0x1BEF, 0x1BF1 }, // Mn   [3] BATAK VOWEL SIGN U FOR SIMALUNGUN SA..BATAK CONSONANT SIGN H
    Interval{ 0x1BF2, 0x1BF3 }, // Mc   [2] BATAK PANGOLAT..BATAK PANONGONAN
    Interval{ 0x1C00, 0x1C23 }, // Lo  [36] LEPCHA LETTER KA..LEPCHA LETTER A
    Interval{ 0x1C24, 0x1C2B }, // Mc   [8] LEPCHA SUBJOINED LETTER YA..LEPCHA VOWEL SIGN UU
    Interval{ 0x1C2C, 0x1C33 }, // Mn   [8] LEPCHA VOWEL SIGN E..LEPCHA CONSONANT SIGN T
    Interval{ 0x1C34, 0x1C35 }, // Mc   [2] LEPCHA CONSONANT SIGN NYIN-DO..LEPCHA CONSONANT SIGN KANG
    Interval{ 0x1C36, 0x1C37 }, // Mn   [2] LEPCHA SIGN RAN..LEPCHA SIGN NUKTA
    Interval{ 0x1C40, 0x1C49 }, // Nd  [10] LEPCHA DIGIT ZERO..LEPCHA DIGIT NINE
    Interval{ 0x1C4D, 0x1C4F }, // Lo   [3] LEPCHA LETTER TTA..LEPCHA LETTER DDA
    Interval{ 0x1C50, 0x1C59 }, // Nd  [10] OL CHIKI DIGIT ZERO..OL CHIKI DIGIT NINE
    Interval{ 0x1C5A, 0x1C77 }, // Lo  [30] OL CHIKI LETTER LA..OL CHIKI LETTER OH
    Interval{ 0x1C78, 0x1C7D }, // Lm   [6] OL CHIKI MU TTUDDAG..OL CHIKI AHAD
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1CD0, 0x1CD2 }, // Mn   [3] VEDIC TONE KARSHANA..VEDIC TONE PRENKHA
    Interval{ 0x1CD4, 0x1CE0 }, // Mn  [13] VEDIC SIGN YAJURVEDIC MIDLINE SVARITA..VEDIC TONE RIGVEDIC KASHMIRI INDEPENDENT SVARITA
    Interval{ 0x1CE1, 0x1CE1 }, // Mc       VEDIC TONE ATHARVAVEDIC INDEPENDENT SVARITA
    Interval{ 0x1CE2, 0x1CE8 }, // Mn   [7] VEDIC SIGN VISARGA SVARITA..VEDIC SIGN VISARGA ANUDATTA WITH TAIL
    Interval{ 0x1CE9, 0x1CEC }, // Lo   [4] VEDIC SIGN ANUSVARA ANTARGOMUKHA..VEDIC SIGN ANUSVARA VAMAGOMUKHA WITH TAIL
    Interval{ 0x1CED, 0x1CED }, // Mn       VEDIC SIGN TIRYAK
    Interval{ 0x1CEE, 0x1CF3 }, // Lo   [6] VEDIC SIGN HEXIFORM LONG ANUSVARA..VEDIC SIGN ROTATED ARDHAVISARGA
    Interval{ 0x1CF4, 0x1CF4 }, // Mn       VEDIC TONE CANDRA ABOVE
    Interval{ 0x1CF5, 0x1CF6 }, // Lo   [2] VEDIC SIGN JIHVAMULIYA..VEDIC SIGN UPADHMANIYA
    Interval{ 0x1CF7, 0x1CF7 }, // Mc       VEDIC SIGN ATIKRAMA
    Interval{ 0x1CF8, 0x1CF9 }, // Mn   [2] VEDIC TONE RING ABOVE..VEDIC TONE DOUBLE RING ABOVE
    Interval{ 0x1CFA, 0x1CFA }, // Lo       VEDIC SIGN DOUBLE ANUSVARA ANTARGOMUKHA
    Interval{ 0x1D00, 0x1D2B }, // L&  [44] LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTER SMALL CAPITAL EL
    Interval{ 0x1D2C, 0x1D6A }, // Lm  [63] MODIFIER LETTER CAPITAL A..GREEK SUBSCRIPT SMALL LETTER CHI
    Interval{ 0x1D6B, 0x1D77 }, // L&  [13] LATIN SMALL LETTER UE..LATIN SMALL LETTER TURNED G
    Interval{ 0x1D78, 0x1D78 }, // Lm       MODIFIER LETTER CYRILLIC EN
    Interval{ 0x1D79, 0x1D9A }, // L&  [34] LATIN SMALL LETTER INSULAR G..LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
    Interval{ 0x1D9B, 0x1DBF }, // Lm  [37] MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER LETTER SMALL THETA
    Interval{ 0x1DC0, 0x1DFF }, // Mn  [64] COMBINING DOTTED GRAVE ACCENT..COMBINING RIGHT ARROWHEAD AND DOWN ARROWHEAD BELOW
    Interval{ 0x1E00, 0x1F15 }, // L& [278] LATIN CAPITAL LETTER A WITH RING BELOW..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F45 }, // L&  [38] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F7D }, // L&  [31] GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1FB4 }, // L&  [53] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FBC }, // L&   [7] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FCC }, // L&   [7] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FDB }, // L&   [6] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE0, 0x1FEC }, // L&  [13] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FFC }, // L&   [7] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x203F, 0x2040 }, // Pc   [2] UNDERTIE..CHARACTER TIE
    Interval{ 0x2054, 0x2054 }, // Pc       INVERTED UNDERTIE
    Interval{ 0x2071, 0x2071 }, // Lm       SUPERSCRIPT LATIN SMALL LETTER I
    Interval{ 0x207F, 0x207F }, // Lm       SUPERSCRIPT LATIN SMALL LETTER N
    Interval{ 0x2090, 0x209C }, // Lm  [13] LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCRIPT SMALL LETTER T
    Interval{ 0x20D0, 0x20DC }, // Mn  [13] COMBINING LEFT HARPOON ABOVE..COMBINING FOUR DOTS ABOVE
    Interval{ 0x20E1, 0x20E1 }, // Mn       COMBINING LEFT RIGHT ARROW ABOVE
    Interval{ 0x20E5, 0x20F0 }, // Mn  [12] COMBINING REVERSE SOLIDUS OVERLAY..COMBINING ASTERISK ABOVE
    Interval{ 0x2102, 0x2102 }, // L&       DOUBLE-STRUCK CAPITAL C
    Interval{ 0x2107, 0x2107 }, // L&       EULER CONSTANT
    Interval{ 0x210A, 0x2113 }, // L&  [10] SCRIPT SMALL G..SCRIPT SMALL L
    Interval{ 0x2115, 0x2115 }, // L&       DOUBLE-STRUCK CAPITAL N
    Interval{ 0x2118, 0x2118 }, // Sm       SCRIPT CAPITAL P
    Interval{ 0x2119, 0x211D }, // L&   [5] DOUBLE-STRUCK CAPITAL P..DOUBLE-STRUCK CAPITAL R
    Interval{ 0x2124, 0x2124 }, // L&       DOUBLE-STRUCK CAPITAL Z
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x2128, 0x2128 }, // L&       BLACK-LETTER CAPITAL Z
    Interval{ 0x212A, 0x212D }, // L&   [4] KELVIN SIGN..BLACK-LETTER CAPITAL C
    Interval{ 0x212E, 0x212E }, // So       ESTIMATED SYMBOL
    Interval{ 0x212F, 0x2134 }, // L&   [6] SCRIPT SMALL E..SCRIPT SMALL O
    Interval{ 0x2135, 0x2138 }, // Lo   [4] ALEF SYMBOL..DALET SYMBOL
    Interval{ 0x2139, 0x2139 }, // L&       INFORMATION SOURCE
    Interval{ 0x213C, 0x213F }, // L&   [4] DOUBLE-STRUCK SMALL PI..DOUBLE-STRUCK CAPITAL PI
    Interval{ 0x2145, 0x2149 }, // L&   [5] DOUBLE-STRUCK ITALIC CAPITAL D..DOUBLE-STRUCK ITALIC SMALL J
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2160, 0x2182 }, // Nl  [35] ROMAN NUMERAL ONE..ROMAN NUMERAL TEN THOUSAND
    Interval{ 0x2183, 0x2184 }, // L&   [2] ROMAN NUMERAL REVERSED ONE HUNDRED..LATIN SMALL LETTER REVERSED C
    Interval{ 0x2185, 0x2188 }, // Nl   [4] ROMAN NUMERAL SIX LATE FORM..ROMAN NUMERAL ONE HUNDRED THOUSAND
    Interval{ 0x2C00, 0x2C7B }, // L& [124] GLAGOLITIC CAPITAL LETTER AZU..LATIN LETTER SMALL CAPITAL TURNED E
    Interval{ 0x2C7C, 0x2C7D }, // Lm   [2] LATIN SUBSCRIPT SMALL LETTER J..MODIFIER LETTER CAPITAL V
    Interval{ 0x2C7E, 0x2CE4 }, // L& [103] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC SYMBOL KAI
    Interval{ 0x2CEB, 0x2CEE }, // L&   [4] COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI..COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CEF, 0x2CF1 }, // Mn   [3] COPTIC COMBINING NI ABOVE..COPTIC COMBINING SPIRITUS LENIS
    Interval{ 0x2CF2, 0x2CF3 }, // L&   [2] COPTIC CAPITAL LETTER BOHAIRIC KHEI..COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0x2D30, 0x2D67 }, // Lo  [56] TIFINAGH LETTER YA..TIFINAGH LETTER YO
    Interval{ 0x2D6F, 0x2D6F }, // Lm       TIFINAGH MODIFIER LETTER LABIALIZATION MARK
    Interval{ 0x2D7F, 0x2D7F }, // Mn       TIFINAGH CONSONANT JOINER
    Interval{ 0x2D80, 0x2D96 }, // Lo  [23] ETHIOPIC SYLLABLE LOA..ETHIOPIC SYLLABLE GGWE
    Interval{ 0x2DA0, 0x2DA6 }, // Lo   [7] ETHIOPIC SYLLABLE SSA..ETHIOPIC SYLLABLE SSO
    Interval{ 0x2DA8, 0x2DAE }, // Lo   [7] ETHIOPIC SYLLABLE CCA..ETHIOPIC SYLLABLE CCO
    Interval{ 0x2DB0, 0x2DB6 }, // Lo   [7] ETHIOPIC SYLLABLE ZZA..ETHIOPIC SYLLABLE ZZO
    Interval{ 0x2DB8, 0x2DBE }, // Lo   [7] ETHIOPIC SYLLABLE CCHA..ETHIOPIC SYLLABLE CCHO
    Interval{ 0x2DC0, 0x2DC6 }, // Lo   [7] ETHIOPIC SYLLABLE QYA..ETHIOPIC SYLLABLE QYO
    Interval{ 0x2DC8, 0x2DCE }, // Lo   [7] ETHIOPIC SYLLABLE KYA..ETHIOPIC SYLLABLE KYO
    Interval{ 0x2DD0, 0x2DD6 }, // Lo   [7] ETHIOPIC SYLLABLE XYA..ETHIOPIC SYLLABLE XYO
    Interval{ 0x2DD8, 0x2DDE }, // Lo   [7] ETHIOPIC SYLLABLE GYA..ETHIOPIC SYLLABLE GYO
    Interval{ 0x2DE0, 0x2DFF }, // Mn  [32] COMBINING CYRILLIC LETTER BE..COMBINING CYRILLIC LETTER IOTIFIED BIG YUS
    Interval{ 0x3005, 0x3005 }, // Lm       IDEOGRAPHIC ITERATION MARK
    Interval{ 0x3006, 0x3006 }, // Lo       IDEOGRAPHIC CLOSING MARK
    Interval{ 0x3007, 0x3007 }, // Nl       IDEOGRAPHIC NUMBER ZERO
    Interval{ 0x3021, 0x3029 }, // Nl   [9] HANGZHOU NUMERAL ONE..HANGZHOU NUMERAL NINE
    Interval{ 0x302A, 0x302D }, // Mn   [4] IDEOGRAPHIC LEVEL TONE MARK..IDEOGRAPHIC ENTERING TONE MARK
    Interval{ 0x302E, 0x302F }, // Mc   [2] HANGUL SINGLE DOT TONE MARK..HANGUL DOUBLE DOT TONE MARK
    Interval{ 0x3031, 0x3035 }, // Lm   [5] VERTICAL KANA REPEAT MARK..VERTICAL KANA REPEAT MARK LOWER HALF
    Interval{ 0x3038, 0x303A }, // Nl   [3] HANGZHOU NUMERAL TEN..HANGZHOU NUMERAL THIRTY
    Interval{ 0x303B, 0x303B }, // Lm       VERTICAL IDEOGRAPHIC ITERATION MARK
    Interval{ 0x303C, 0x303C }, // Lo       MASU MARK
    Interval{ 0x3041, 0x3096 }, // Lo  [86] HIRAGANA LETTER SMALL A..HIRAGANA LETTER SMALL KE
    Interval{ 0x3099, 0x309A }, // Mn   [2] COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK..COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    Interval{ 0x309B, 0x309C }, // Sk   [2] KATAKANA-HIRAGANA VOICED SOUND MARK..KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    Interval{ 0x309D, 0x309E }, // Lm   [2] HIRAGANA ITERATION MARK..HIRAGANA VOICED ITERATION MARK
    Interval{ 0x309F, 0x309F }, // Lo       HIRAGANA DIGRAPH YORI
    Interval{ 0x30A1, 0x30FA }, // Lo  [90] KATAKANA LETTER SMALL A..KATAKANA LETTER VO
    Interval{ 0x30FC, 0x30FE }, // Lm   [3] KATAKANA-HIRAGANA PROLONGED SOUND MARK..KATAKANA VOICED ITERATION MARK
    Interval{ 0x30FF, 0x30FF }, // Lo       KATAKANA DIGRAPH KOTO
    Interval{ 0x3105, 0x312F }, // Lo  [43] BOPOMOFO LETTER B..BOPOMOFO LETTER NN
    Interval{ 0x3131, 0x318E }, // Lo  [94] HANGUL LETTER KIYEOK..HANGUL LETTER ARAEAE
    Interval{ 0x31A0, 0x31BF }, // Lo  [32] BOPOMOFO LETTER BU..BOPOMOFO LETTER AH
    Interval{ 0x31F0, 0x31FF }, // Lo  [16] KATAKANA LETTER SMALL KU..KATAKANA LETTER SMALL RO
    Interval{ 0x3400, 0x4DBF }, // Lo [6592] CJK UNIFIED IDEOGRAPH-3400..CJK UNIFIED IDEOGRAPH-4DBF
    Interval{ 0x4E00, 0xA014 }, // Lo [21013] CJK UNIFIED IDEOGRAPH-4E00..YI SYLLABLE E
    Interval{ 0xA015, 0xA015 }, // Lm       YI SYLLABLE WU
    Interval{ 0xA016, 0xA48C }, // Lo [1143] YI SYLLABLE BIT..YI SYLLABLE YYR
    Interval{ 0xA4D0, 0xA4F7 }, // Lo  [40] LISU LETTER BA..LISU LETTER OE
    Interval{ 0xA4F8, 0xA4FD }, // Lm   [6] LISU LETTER TONE MYA TI..LISU LETTER TONE MYA JEU
    Interval{ 0xA500, 0xA60B }, // Lo [268] VAI SYLLABLE EE..VAI SYLLABLE NG
    Interval{ 0xA60C, 0xA60C }, // Lm       VAI SYLLABLE LENGTHENER
    Interval{ 0xA610, 0xA61F }, // Lo  [16] VAI SYLLABLE NDOLE FA..VAI SYMBOL JONG
    Interval{ 0xA620, 0xA629 }, // Nd  [10] VAI DIGIT ZERO..VAI DIGIT NINE
    Interval{ 0xA62A, 0xA62B }, // Lo   [2] VAI SYLLABLE NDOLE MA..VAI SYLLABLE NDOLE DO
    Interval{ 0xA640, 0xA66D }, // L&  [46] CYRILLIC CAPITAL LETTER ZEMLYA..CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA66E, 0xA66E }, // Lo       CYRILLIC LETTER MULTIOCULAR O
    Interval{ 0xA66F, 0xA66F }, // Mn       COMBINING CYRILLIC VZMET
    Interval{ 0xA674, 0xA67D }, // Mn  [10] COMBINING CYRILLIC LETTER UKRAINIAN IE..COMBINING CYRILLIC PAYEROK
    Interval{ 0xA67F, 0xA67F }, // Lm       CYRILLIC PAYEROK
    Interval{ 0xA680, 0xA69B }, // L&  [28] CYRILLIC CAPITAL LETTER DWE..CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA69C, 0xA69D }, // Lm   [2] MODIFIER LETTER CYRILLIC HARD SIGN..MODIFIER LETTER CYRILLIC SOFT SIGN
    Interval{ 0xA69E, 0xA69F }, // Mn   [2] COMBINING CYRILLIC LETTER EF..COMBINING CYRILLIC LETTER IOTIFIED E
    Interval{ 0xA6A0, 0xA6E5 }, // Lo  [70] BAMUM LETTER A..BAMUM LETTER KI
    Interval{ 0xA6E6, 0xA6EF }, // Nl  [10] BAMUM LETTER MO..BAMUM LETTER KOGHOM
    Interval{ 0xA6F0, 0xA6F1 }, // Mn   [2] BAMUM COMBINING MARK KOQNDON..BAMUM COMBINING MARK TUKWENTIS
    Interval{ 0xA717, 0xA71F }, // Lm   [9] MODIFIER LETTER DOT VERTICAL BAR..MODIFIER LETTER LOW INVERTED EXCLAMATION MARK
    Interval{ 0xA722, 0xA76F }, // L&  [78] LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF..LATIN SMALL LETTER CON
    Interval{ 0xA770, 0xA770 }, // Lm       MODIFIER LETTER US
    Interval{ 0xA771, 0xA787 }, // L&  [23] LATIN SMALL LETTER DUM..LATIN SMALL LETTER INSULAR T
    Interval{ 0xA788, 0xA788 }, // Lm       MODIFIER LETTER LOW CIRCUMFLEX ACCENT
    Interval{ 0xA78B, 0xA78E }, // L&   [4] LATIN CAPITAL LETTER SALTILLO..LATIN SMALL LETTER L WITH RETROFLEX HOOK AND BELT
    Interval{ 0xA78F, 0xA78F }, // Lo       LATIN LETTER SINOLOGICAL DOT
    Interval{ 0xA790, 0xA7CA }, // L&  [59] LATIN CAPITAL LETTER N WITH DESCENDER..LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D1 }, // L&   [2] LATIN CAPITAL LETTER CLOSED INSULAR G..LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D3, 0xA7D3 }, // L&       LATIN SMALL LETTER DOUBLE THORN
    Interval{ 0xA7D5, 0xA7D9 }, // L&   [5] LATIN SMALL LETTER DOUBLE WYNN..LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F2, 0xA7F4 }, // Lm   [3] MODIFIER LETTER CAPITAL C..MODIFIER LETTER CAPITAL Q
    Interval{ 0xA7F5, 0xA7F6 }, // L&   [2] LATIN CAPITAL LETTER REVERSED HALF H..LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xA7F7, 0xA7F7 }, // Lo       LATIN EPIGRAPHIC LETTER SIDEWAYS I
    Interval{ 0xA7F8, 0xA7F9 }, // Lm   [2] MODIFIER LETTER CAPITAL H WITH STROKE..MODIFIER LETTER SMALL LIGATURE OE
    Interval{ 0xA7FA, 0xA7FA }, // L&       LATIN LETTER SMALL CAPITAL TURNED M
    Interval{ 0xA7FB, 0xA801 }, // Lo   [7] LATIN EPIGRAPHIC LETTER REVERSED F..SYLOTI NAGRI LETTER I
    Interval{ 0xA802, 0xA802 }, // Mn       SYLOTI NAGRI SIGN DVISVARA
    Interval{ 0xA803, 0xA805 }, // Lo   [3] SYLOTI NAGRI LETTER U..SYLOTI NAGRI LETTER O
    Interval{ 0xA806, 0xA806 }, // Mn       SYLOTI NAGRI SIGN HASANTA
    Interval{ 0xA807, 0xA80A }, // Lo   [4] SYLOTI NAGRI LETTER KO..SYLOTI NAGRI LETTER GHO
    Interval{ 0xA80B, 0xA80B }, // Mn       SYLOTI NAGRI SIGN ANUSVARA
    Interval{ 0xA80C, 0xA822 }, // Lo  [23] SYLOTI NAGRI LETTER CO..SYLOTI NAGRI LETTER HO
    Interval{ 0xA823, 0xA824 }, // Mc   [2] SYLOTI NAGRI VOWEL SIGN A..SYLOTI NAGRI VOWEL SIGN I
    Interval{ 0xA825, 0xA826 }, // Mn   [2] SYLOTI NAGRI VOWEL SIGN U..SYLOTI NAGRI VOWEL SIGN E
    Interval{ 0xA827, 0xA827 }, // Mc       SYLOTI NAGRI VOWEL SIGN OO
    Interval{ 0xA82C, 0xA82C }, // Mn       SYLOTI NAGRI SIGN ALTERNATE HASANTA
    Interval{ 0xA840, 0xA873 }, // Lo  [52] PHAGS-PA LETTER KA..PHAGS-PA LETTER CANDRABINDU
    Interval{ 0xA880, 0xA881 }, // Mc   [2] SAURASHTRA SIGN ANUSVARA..SAURASHTRA SIGN VISARGA
    Interval{ 0xA882, 0xA8B3 }, // Lo  [50] SAURASHTRA LETTER A..SAURASHTRA LETTER LLA
    Interval{ 0xA8B4, 0xA8C3 }, // Mc  [16] SAURASHTRA CONSONANT SIGN HAARU..SAURASHTRA VOWEL SIGN AU
    Interval{ 0xA8C4, 0xA8C5 }, // Mn   [2] SAURASHTRA SIGN VIRAMA..SAURASHTRA SIGN CANDRABINDU
    Interval{ 0xA8D0, 0xA8D9 }, // Nd  [10] SAURASHTRA DIGIT ZERO..SAURASHTRA DIGIT NINE
    Interval{ 0xA8E0, 0xA8F1 }, // Mn  [18] COMBINING DEVANAGARI DIGIT ZERO..COMBINING DEVANAGARI SIGN AVAGRAHA
    Interval{ 0xA8F2, 0xA8F7 }, // Lo   [6] DEVANAGARI SIGN SPACING CANDRABINDU..DEVANAGARI SIGN CANDRABINDU AVAGRAHA
    Interval{ 0xA8FB, 0xA8FB }, // Lo       DEVANAGARI HEADSTROKE
    Interval{ 0xA8FD, 0xA8FE }, // Lo   [2] DEVANAGARI JAIN OM..DEVANAGARI LETTER AY
    Interval{ 0xA8FF, 0xA8FF }, // Mn       DEVANAGARI VOWEL SIGN AY
    Interval{ 0xA900, 0xA909 }, // Nd  [10] KAYAH LI DIGIT ZERO..KAYAH LI DIGIT NINE
    Interval{ 0xA90A, 0xA925 }, // Lo  [28] KAYAH LI LETTER KA..KAYAH LI LETTER OO
    Interval{ 0xA926, 0xA92D }, // Mn   [8] KAYAH LI VOWEL UE..KAYAH LI TONE CALYA PLOPHU
    Interval{ 0xA930, 0xA946 }, // Lo  [23] REJANG LETTER KA..REJANG LETTER A
    Interval{ 0xA947, 0xA951 }, // Mn  [11] REJANG VOWEL SIGN I..REJANG CONSONANT SIGN R
    Interval{ 0xA952, 0xA953 }, // Mc   [2] REJANG CONSONANT SIGN H..REJANG VIRAMA
    Interval{ 0xA960, 0xA97C }, // Lo  [29] HANGUL CHOSEONG TIKEUT-MIEUM..HANGUL CHOSEONG SSANGYEORINHIEUH
    Interval{ 0xA980, 0xA982 }, // Mn   [3] JAVANESE SIGN PANYANGGA..JAVANESE SIGN LAYAR
    Interval{ 0xA983, 0xA983 }, // Mc       JAVANESE SIGN WIGNYAN
    Interval{ 0xA984, 0xA9B2 }, // Lo  [47] JAVANESE LETTER A..JAVANESE LETTER HA
    Interval{ 0xA9B3, 0xA9B3 }, // Mn       JAVANESE SIGN CECAK TELU
    Interval{ 0xA9B4, 0xA9B5 }, // Mc   [2] JAVANESE VOWEL SIGN TARUNG..JAVANESE VOWEL SIGN TOLONG
    Interval{ 0xA9B6, 0xA9B9 }, // Mn   [4] JAVANESE VOWEL SIGN WULU..JAVANESE VOWEL SIGN SUKU MENDUT
    Interval{ 0xA9BA, 0xA9BB }, // Mc   [2] JAVANESE VOWEL SIGN TALING..JAVANESE VOWEL SIGN DIRGA MURE
    Interval{ 0xA9BC, 0xA9BD }, // Mn   [2] JAVANESE VOWEL SIGN PEPET..JAVANESE CONSONANT SIGN KERET
    Interval{ 0xA9BE, 0xA9C0 }, // Mc   [3] JAVANESE CONSONANT SIGN PENGKAL..JAVANESE PANGKON
    Interval{ 0xA9CF, 0xA9CF }, // Lm       JAVANESE PANGRANGKEP
    Interval{ 0xA9D0, 0xA9D9 }, // Nd  [10] JAVANESE DIGIT ZERO..JAVANESE DIGIT NINE
    Interval{ 0xA9E0, 0xA9E4 }, // Lo   [5] MYANMAR LETTER SHAN GHA..MYANMAR LETTER SHAN BHA
    Interval{ 0xA9E5, 0xA9E5 }, // Mn       MYANMAR SIGN SHAN SAW
    Interval{ 0xA9E6, 0xA9E6 }, // Lm       MYANMAR MODIFIER LETTER SHAN REDUPLICATION
    Interval{ 0xA9E7, 0xA9EF }, // Lo   [9] MYANMAR LETTER TAI LAING NYA..MYANMAR LETTER TAI LAING NNA
    Interval{ 0xA9F0, 0xA9F9 }, // Nd  [10] MYANMAR TAI LAING DIGIT ZERO..MYANMAR TAI LAING DIGIT NINE
    Interval{ 0xA9FA, 0xA9FE }, // Lo   [5] MYANMAR LETTER TAI LAING LLA..MYANMAR LETTER TAI LAING BHA
    Interval{ 0xAA00, 0xAA28 }, // Lo  [41] CHAM LETTER A..CHAM LETTER HA
    Interval{ 0xAA29, 0xAA2E }, // Mn   [6] CHAM VOWEL SIGN AA..CHAM VOWEL SIGN OE
    Interval{ 0xAA2F, 0xAA30 }, // Mc   [2] CHAM VOWEL SIGN O..CHAM VOWEL SIGN AI
    Interval{ 0xAA31, 0xAA32 }, // Mn   [2] CHAM VOWEL SIGN AU..CHAM VOWEL SIGN UE
    Interval{ 0xAA33, 0xAA34 }, // Mc   [2] CHAM CONSONANT SIGN YA..CHAM CONSONANT SIGN RA
    Interval{ 0xAA35, 0xAA36 }, // Mn   [2] CHAM CONSONANT SIGN LA..CHAM CONSONANT SIGN WA
    Interval{ 0xAA40, 0xAA42 }, // Lo   [3] CHAM LETTER FINAL K..CHAM LETTER FINAL NG
    Interval{ 0xAA43, 0xAA43 }, // Mn       CHAM CONSONANT SIGN FINAL NG
    Interval{ 0xAA44, 0xAA4B }, // Lo   [8] CHAM LETTER FINAL CH..CHAM LETTER FINAL SS
    Interval{ 0xAA4C, 0xAA4C }, // Mn       CHAM CONSONANT SIGN FINAL M
    Interval{ 0xAA4D, 0xAA4D }, // Mc       CHAM CONSONANT SIGN FINAL H
    Interval{ 0xAA50, 0xAA59 }, // Nd  [10] CHAM DIGIT ZERO..CHAM DIGIT NINE
    Interval{ 0xAA60, 0xAA6F }, // Lo  [16] MYANMAR LETTER KHAMTI GA..MYANMAR LETTER KHAMTI FA
    Interval{ 0xAA70, 0xAA70 }, // Lm       MYANMAR MODIFIER LETTER KHAMTI REDUPLICATION
    Interval{ 0xAA71, 0xAA76 }, // Lo   [6] MYANMAR LETTER KHAMTI XA..MYANMAR LOGOGRAM KHAMTI HM
    Interval{ 0xAA7A, 0xAA7A }, // Lo       MYANMAR LETTER AITON RA
    Interval{ 0xAA7B, 0xAA7B }, // Mc       MYANMAR SIGN PAO KAREN TONE
    Interval{ 0xAA7C, 0xAA7C }, // Mn       MYANMAR SIGN TAI LAING TONE-2
    Interval{ 0xAA7D, 0xAA7D }, // Mc       MYANMAR SIGN TAI LAING TONE-5
    Interval{ 0xAA7E, 0xAAAF }, // Lo  [50] MYANMAR LETTER SHWE PALAUNG CHA..TAI VIET LETTER HIGH O
    Interval{ 0xAAB0, 0xAAB0 }, // Mn       TAI VIET MAI KANG
    Interval{ 0xAAB1, 0xAAB1 }, // Lo       TAI VIET VOWEL AA
    Interval{ 0xAAB2, 0xAAB4 }, // Mn   [3] TAI VIET VOWEL I..TAI VIET VOWEL U
    Interval{ 0xAAB5, 0xAAB6 }, // Lo   [2] TAI VIET VOWEL E..TAI VIET VOWEL O
    Interval{ 0xAAB7, 0xAAB8 }, // Mn   [2] TAI VIET MAI KHIT..TAI VIET VOWEL IA
    Interval{ 0xAAB9, 0xAABD }, // Lo   [5] TAI VIET VOWEL UEA..TAI VIET VOWEL AN
    Interval{ 0xAABE, 0xAABF }, // Mn   [2] TAI VIET VOWEL AM..TAI VIET TONE MAI EK
    Interval{ 0xAAC0, 0xAAC0 }, // Lo       TAI VIET TONE MAI NUENG
    Interval{ 0xAAC1, 0xAAC1 }, // Mn       TAI VIET TONE MAI THO
    Interval{ 0xAAC2, 0xAAC2 }, // Lo       TAI VIET TONE MAI SONG
    Interval{ 0xAADB, 0xAADC }, // Lo   [2] TAI VIET SYMBOL KON..TAI VIET SYMBOL NUENG
    Interval{ 0xAADD, 0xAADD }, // Lm       TAI VIET SYMBOL SAM
    Interval{ 0xAAE0, 0xAAEA }, // Lo  [11] MEETEI MAYEK LETTER E..MEETEI MAYEK LETTER SSA
    Interval{ 0xAAEB, 0xAAEB }, // Mc       MEETEI MAYEK VOWEL SIGN II
    Interval{ 0xAAEC, 0xAAED }, // Mn   [2] MEETEI MAYEK VOWEL SIGN UU..MEETEI MAYEK VOWEL SIGN AAI
    Interval{ 0xAAEE, 0xAAEF }, // Mc   [2] MEETEI MAYEK VOWEL SIGN AU..MEETEI MAYEK VOWEL SIGN AAU
    Interval{ 0xAAF2, 0xAAF2 }, // Lo       MEETEI MAYEK ANJI
    Interval{ 0xAAF3, 0xAAF4 }, // Lm   [2] MEETEI MAYEK SYLLABLE REPETITION MARK..MEETEI MAYEK WORD REPETITION MARK
    Interval{ 0xAAF5, 0xAAF5 }, // Mc       MEETEI MAYEK VOWEL SIGN VISARGA
    Interval{ 0xAAF6, 0xAAF6 }, // Mn       MEETEI MAYEK VIRAMA
    Interval{ 0xAB01, 0xAB06 }, // Lo   [6] ETHIOPIC SYLLABLE TTHU..ETHIOPIC SYLLABLE TTHO
    Interval{ 0xAB09, 0xAB0E }, // Lo   [6] ETHIOPIC SYLLABLE DDHU..ETHIOPIC SYLLABLE DDHO
    Interval{ 0xAB11, 0xAB16 }, // Lo   [6] ETHIOPIC SYLLABLE DZU..ETHIOPIC SYLLABLE DZO
    Interval{ 0xAB20, 0xAB26 }, // Lo   [7] ETHIOPIC SYLLABLE CCHHA..ETHIOPIC SYLLABLE CCHHO
    Interval{ 0xAB28, 0xAB2E }, // Lo   [7] ETHIOPIC SYLLABLE BBA..ETHIOPIC SYLLABLE BBO
    Interval{ 0xAB30, 0xAB5A }, // L&  [43] LATIN SMALL LETTER BARRED ALPHA..LATIN SMALL LETTER Y WITH SHORT RIGHT LEG
    Interval{ 0xAB5C, 0xAB5F }, // Lm   [4] MODIFIER LETTER SMALL HENG..MODIFIER LETTER SMALL U WITH LEFT HOOK
    Interval{ 0xAB60, 0xAB68 }, // L&   [9] LATIN SMALL LETTER SAKHA YAT..LATIN SMALL LETTER TURNED R WITH MIDDLE TILDE
    Interval{ 0xAB69, 0xAB69 }, // Lm       MODIFIER LETTER SMALL TURNED W
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xABC0, 0xABE2 }, // Lo  [35] MEETEI MAYEK LETTER KOK..MEETEI MAYEK LETTER I LONSUM
    Interval{ 0xABE3, 0xABE4 }, // Mc   [2] MEETEI MAYEK VOWEL SIGN ONAP..MEETEI MAYEK VOWEL SIGN INAP
    Interval{ 0xABE5, 0xABE5 }, // Mn       MEETEI MAYEK VOWEL SIGN ANAP
    Interval{ 0xABE6, 0xABE7 }, // Mc   [2] MEETEI MAYEK VOWEL SIGN YENAP..MEETEI MAYEK VOWEL SIGN SOUNAP
    Interval{ 0xABE8, 0xABE8 }, // Mn       MEETEI MAYEK VOWEL SIGN UNAP
    Interval{ 0xABE9, 0xABEA }, // Mc   [2] MEETEI MAYEK VOWEL SIGN CHEINAP..MEETEI MAYEK VOWEL SIGN NUNG
    Interval{ 0xABEC, 0xABEC }, // Mc       MEETEI MAYEK LUM IYEK
    Interval{ 0xABED, 0xABED }, // Mn       MEETEI MAYEK APUN IYEK
    Interval{ 0xABF0, 0xABF9 }, // Nd  [10] MEETEI MAYEK DIGIT ZERO..MEETEI MAYEK DIGIT NINE
    Interval{ 0xAC00, 0xD7A3 }, // Lo [11172] HANGUL SYLLABLE GA..HANGUL SYLLABLE HIH
    Interval{ 0xD7B0, 0xD7C6 }, // Lo  [23] HANGUL JUNGSEONG O-YEO..HANGUL JUNGSEONG ARAEA-E
    Interval{ 0xD7CB, 0xD7FB }, // Lo  [49] HANGUL JONGSEONG NIEUN-RIEUL..HANGUL JONGSEONG PHIEUPH-THIEUTH
    Interval{ 0xF900, 0xFA6D }, // Lo [366] CJK COMPATIBILITY IDEOGRAPH-F900..CJK COMPATIBILITY IDEOGRAPH-FA6D
    Interval{ 0xFA70, 0xFAD9 }, // Lo [106] CJK COMPATIBILITY IDEOGRAPH-FA70..CJK COMPATIBILITY IDEOGRAPH-FAD9
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFB1D, 0xFB1D }, // Lo       HEBREW LETTER YOD WITH HIRIQ
    Interval{ 0xFB1E, 0xFB1E }, // Mn       HEBREW POINT JUDEO-SPANISH VARIKA
    Interval{ 0xFB1F, 0xFB28 }, // Lo  [10] HEBREW LIGATURE YIDDISH YOD YOD PATAH..HEBREW LETTER WIDE TAV
    Interval{ 0xFB2A, 0xFB36 }, // Lo  [13] HEBREW LETTER SHIN WITH SHIN DOT..HEBREW LETTER ZAYIN WITH DAGESH
    Interval{ 0xFB38, 0xFB3C }, // Lo   [5] HEBREW LETTER TET WITH DAGESH..HEBREW LETTER LAMED WITH DAGESH
    Interval{ 0xFB3E, 0xFB3E }, // Lo       HEBREW LETTER MEM WITH DAGESH
    Interval{ 0xFB40, 0xFB41 }, // Lo   [2] HEBREW LETTER NUN WITH DAGESH..HEBREW LETTER SAMEKH WITH DAGESH
    Interval{ 0xFB43, 0xFB44 }, // Lo   [2] HEBREW LETTER FINAL PE WITH DAGESH..HEBREW LETTER PE WITH DAGESH
    Interval{ 0xFB46, 0xFBB1 }, // Lo [108] HEBREW LETTER TSADI WITH DAGESH..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
    Interval{ 0xFBD3, 0xFD3D }, // Lo [363] ARABIC LETTER NG ISOLATED FORM..ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
    Interval{ 0xFD50, 0xFD8F }, // Lo  [64] ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM..ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
    Interval{ 0xFD92, 0xFDC7 }, // Lo  [54] ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM..ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
    Interval{ 0xFDF0, 0xFDFB }, // Lo  [12] ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM..ARABIC LIGATURE JALLAJALALOUHOU
    Interval{ 0xFE00, 0xFE0F }, // Mn  [16] VARIATION SELECTOR-1..VARIATION SELECTOR-16
    Interval{ 0xFE20, 0xFE2F }, // Mn  [16] COMBINING LIGATURE LEFT HALF..COMBINING CYRILLIC TITLO RIGHT HALF
    Interval{ 0xFE33, 0xFE34 }, // Pc   [2] PRESENTATION FORM FOR VERTICAL LOW LINE..PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
    Interval{ 0xFE4D, 0xFE4F }, // Pc   [3] DASHED LOW LINE..WAVY LOW LINE
    Interval{ 0xFE70, 0xFE74 }, // Lo   [5] ARABIC FATHATAN ISOLATED FORM..ARABIC KASRATAN ISOLATED FORM
    Interval{ 0xFE76, 0xFEFC }, // Lo [135] ARABIC FATHA ISOLATED FORM..ARABIC LIGATURE LAM WITH ALEF FINAL FORM
    Interval{ 0xFF10, 0xFF19 }, // Nd  [10] FULLWIDTH DIGIT ZERO..FULLWIDTH DIGIT NINE
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0xFF3F, 0xFF3F }, // Pc       FULLWIDTH LOW LINE
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0xFF66, 0xFF6F }, // Lo  [10] HALFWIDTH KATAKANA LETTER WO..HALFWIDTH KATAKANA LETTER SMALL TU
    Interval{ 0xFF70, 0xFF70 }, // Lm       HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
    Interval{ 0xFF71, 0xFF9D }, // Lo  [45] HALFWIDTH KATAKANA LETTER A..HALFWIDTH KATAKANA LETTER N
    Interval{ 0xFF9E, 0xFF9F }, // Lm   [2] HALFWIDTH KATAKANA VOICED SOUND MARK..HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
    Interval{ 0xFFA0, 0xFFBE }, // Lo  [31] HALFWIDTH HANGUL FILLER..HALFWIDTH HANGUL LETTER HIEUH
    Interval{ 0xFFC2, 0xFFC7 }, // Lo   [6] HALFWIDTH HANGUL LETTER A..HALFWIDTH HANGUL LETTER E
    Interval{ 0xFFCA, 0xFFCF }, // Lo   [6] HALFWIDTH HANGUL LETTER YEO..HALFWIDTH HANGUL LETTER OE
    Interval{ 0xFFD2, 0xFFD7 }, // Lo   [6] HALFWIDTH HANGUL LETTER YO..HALFWIDTH HANGUL LETTER YU
    Interval{ 0xFFDA, 0xFFDC }, // Lo   [3] HALFWIDTH HANGUL LETTER EU..HALFWIDTH HANGUL LETTER I
    Interval{ 0x10000, 0x1000B }, // Lo  [12] LINEAR B SYLLABLE B008 A..LINEAR B SYLLABLE B046 JE
    Interval{ 0x1000D, 0x10026 }, // Lo  [26] LINEAR B SYLLABLE B036 JO..LINEAR B SYLLABLE B032 QO
    Interval{ 0x10028, 0x1003A }, // Lo  [19] LINEAR B SYLLABLE B060 RA..LINEAR B SYLLABLE B042 WO
    Interval{ 0x1003C, 0x1003D }, // Lo   [2] LINEAR B SYLLABLE B017 ZA..LINEAR B SYLLABLE B074 ZE
    Interval{ 0x1003F, 0x1004D }, // Lo  [15] LINEAR B SYLLABLE B020 ZO..LINEAR B SYLLABLE B091 TWO
    Interval{ 0x10050, 0x1005D }, // Lo  [14] LINEAR B SYMBOL B018..LINEAR B SYMBOL B089
    Interval{ 0x10080, 0x100FA }, // Lo [123] LINEAR B IDEOGRAM B100 MAN..LINEAR B IDEOGRAM VESSEL B305
    Interval{ 0x10140, 0x10174 }, // Nl  [53] GREEK ACROPHONIC ATTIC ONE QUARTER..GREEK ACROPHONIC STRATIAN FIFTY MNAS
    Interval{ 0x101FD, 0x101FD }, // Mn       PHAISTOS DISC SIGN COMBINING OBLIQUE STROKE
    Interval{ 0x10280, 0x1029C }, // Lo  [29] LYCIAN LETTER A..LYCIAN LETTER X
    Interval{ 0x102A0, 0x102D0 }, // Lo  [49] CARIAN LETTER A..CARIAN LETTER UUU3
    Interval{ 0x102E0, 0x102E0 }, // Mn       COPTIC EPACT THOUSANDS MARK
    Interval{ 0x10300, 0x1031F }, // Lo  [32] OLD ITALIC LETTER A..OLD ITALIC LETTER ESS
    Interval{ 0x1032D, 0x10340 }, // Lo  [20] OLD ITALIC LETTER YE..GOTHIC LETTER PAIRTHRA
    Interval{ 0x10341, 0x10341 }, // Nl       GOTHIC LETTER NINETY
    Interval{ 0x10342, 0x10349 }, // Lo   [8] GOTHIC LETTER RAIDA..GOTHIC LETTER OTHAL
    Interval{ 0x1034A, 0x1034A }, // Nl       GOTHIC LETTER NINE HUNDRED
    Interval{ 0x10350, 0x10375 }, // Lo  [38] OLD PERMIC LETTER AN..OLD PERMIC LETTER IA
    Interval{ 0x10376, 0x1037A }, // Mn   [5] COMBINING OLD PERMIC LETTER AN..COMBINING OLD PERMIC LETTER SII
    Interval{ 0x10380, 0x1039D }, // Lo  [30] UGARITIC LETTER ALPA..UGARITIC LETTER SSU
    Interval{ 0x103A0, 0x103C3 }, // Lo  [36] OLD PERSIAN SIGN A..OLD PERSIAN SIGN HA
    Interval{ 0x103C8, 0x103CF }, // Lo   [8] OLD PERSIAN SIGN AURAMAZDAA..OLD PERSIAN SIGN BUUMISH
    Interval{ 0x103D1, 0x103D5 }, // Nl   [5] OLD PERSIAN NUMBER ONE..OLD PERSIAN NUMBER HUNDRED
    Interval{ 0x10400, 0x1044F }, // L&  [80] DESERET CAPITAL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x10450, 0x1049D }, // Lo  [78] SHAVIAN LETTER PEEP..OSMANYA LETTER OO
    Interval{ 0x104A0, 0x104A9 }, // Nd  [10] OSMANYA DIGIT ZERO..OSMANYA DIGIT NINE
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10500, 0x10527 }, // Lo  [40] ELBASAN LETTER A..ELBASAN LETTER KHE
    Interval{ 0x10530, 0x10563 }, // Lo  [52] CAUCASIAN ALBANIAN LETTER ALT..CAUCASIAN ALBANIAN LETTER KIW
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10600, 0x10736 }, // Lo [311] LINEAR A SIGN AB001..LINEAR A SIGN A664
    Interval{ 0x10740, 0x10755 }, // Lo  [22] LINEAR A SIGN A701 A..LINEAR A SIGN A732 JE
    Interval{ 0x10760, 0x10767 }, // Lo   [8] LINEAR A SIGN A800..LINEAR A SIGN A807
    Interval{ 0x10780, 0x10785 }, // Lm   [6] MODIFIER LETTER SMALL CAPITAL AA..MODIFIER LETTER SMALL B WITH HOOK
    Interval{ 0x10787, 0x107B0 }, // Lm  [42] MODIFIER LETTER SMALL DZ DIGRAPH..MODIFIER LETTER SMALL V WITH RIGHT HOOK
    Interval{ 0x107B2, 0x107BA }, // Lm   [9] MODIFIER LETTER SMALL CAPITAL Y..MODIFIER LETTER SMALL S WITH CURL
    Interval{ 0x10800, 0x10805 }, // Lo   [6] CYPRIOT SYLLABLE A..CYPRIOT SYLLABLE JA
    Interval{ 0x10808, 0x10808 }, // Lo       CYPRIOT SYLLABLE JO
    Interval{ 0x1080A, 0x10835 }, // Lo  [44] CYPRIOT SYLLABLE KA..CYPRIOT SYLLABLE WO
    Interval{ 0x10837, 0x10838 }, // Lo   [2] CYPRIOT SYLLABLE XA..CYPRIOT SYLLABLE XE
    Interval{ 0x1083C, 0x1083C }, // Lo       CYPRIOT SYLLABLE ZA
    Interval{ 0x1083F, 0x10855 }, // Lo  [23] CYPRIOT SYLLABLE ZO..IMPERIAL ARAMAIC LETTER TAW
    Interval{ 0x10860, 0x10876 }, // Lo  [23] PALMYRENE LETTER ALEPH..PALMYRENE LETTER TAW
    Interval{ 0x10880, 0x1089E }, // Lo  [31] NABATAEAN LETTER FINAL ALEPH..NABATAEAN LETTER TAW
    Interval{ 0x108E0, 0x108F2 }, // Lo  [19] HATRAN LETTER ALEPH..HATRAN LETTER QOPH
    Interval{ 0x108F4, 0x108F5 }, // Lo   [2] HATRAN LETTER SHIN..HATRAN LETTER TAW
    Interval{ 0x10900, 0x10915 }, // Lo  [22] PHOENICIAN LETTER ALF..PHOENICIAN LETTER TAU
    Interval{ 0x10920, 0x10939 }, // Lo  [26] LYDIAN LETTER A..LYDIAN LETTER C
    Interval{ 0x10980, 0x109B7 }, // Lo  [56] MEROITIC HIEROGLYPHIC LETTER A..MEROITIC CURSIVE LETTER DA
    Interval{ 0x109BE, 0x109BF }, // Lo   [2] MEROITIC CURSIVE LOGOGRAM RMT..MEROITIC CURSIVE LOGOGRAM IMN
    Interval{ 0x10A00, 0x10A00 }, // Lo       KHAROSHTHI LETTER A
    Interval{ 0x10A01, 0x10A03 }, // Mn   [3] KHAROSHTHI VOWEL SIGN I..KHAROSHTHI VOWEL SIGN VOCALIC R
    Interval{ 0x10A05, 0x10A06 }, // Mn   [2] KHAROSHTHI VOWEL SIGN E..KHAROSHTHI VOWEL SIGN O
    Interval{ 0x10A0C, 0x10A0F }, // Mn   [4] KHAROSHTHI VOWEL LENGTH MARK..KHAROSHTHI SIGN VISARGA
    Interval{ 0x10A10, 0x10A13 }, // Lo   [4] KHAROSHTHI LETTER KA..KHAROSHTHI LETTER GHA
    Interval{ 0x10A15, 0x10A17 }, // Lo   [3] KHAROSHTHI LETTER CA..KHAROSHTHI LETTER JA
    Interval{ 0x10A19, 0x10A35 }, // Lo  [29] KHAROSHTHI LETTER NYA..KHAROSHTHI LETTER VHA
    Interval{ 0x10A38, 0x10A3A }, // Mn   [3] KHAROSHTHI SIGN BAR ABOVE..KHAROSHTHI SIGN DOT BELOW
    Interval{ 0x10A3F, 0x10A3F }, // Mn       KHAROSHTHI VIRAMA
    Interval{ 0x10A60, 0x10A7C }, // Lo  [29] OLD SOUTH ARABIAN LETTER HE..OLD SOUTH ARABIAN LETTER THETH
    Interval{ 0x10A80, 0x10A9C }, // Lo  [29] OLD NORTH ARABIAN LETTER HEH..OLD NORTH ARABIAN LETTER ZAH
    Interval{ 0x10AC0, 0x10AC7 }, // Lo   [8] MANICHAEAN LETTER ALEPH..MANICHAEAN LETTER WAW
    Interval{ 0x10AC9, 0x10AE4 }, // Lo  [28] MANICHAEAN LETTER ZAYIN..MANICHAEAN LETTER TAW
    Interval{ 0x10AE5, 0x10AE6 }, // Mn   [2] MANICHAEAN ABBREVIATION MARK ABOVE..MANICHAEAN ABBREVIATION MARK BELOW
    Interval{ 0x10B00, 0x10B35 }, // Lo  [54] AVESTAN LETTER A..AVESTAN LETTER HE
    Interval{ 0x10B40, 0x10B55 }, // Lo  [22] INSCRIPTIONAL PARTHIAN LETTER ALEPH..INSCRIPTIONAL PARTHIAN LETTER TAW
    Interval{ 0x10B60, 0x10B72 }, // Lo  [19] INSCRIPTIONAL PAHLAVI LETTER ALEPH..INSCRIPTIONAL PAHLAVI LETTER TAW
    Interval{ 0x10B80, 0x10B91 }, // Lo  [18] PSALTER PAHLAVI LETTER ALEPH..PSALTER PAHLAVI LETTER TAW
    Interval{ 0x10C00, 0x10C48 }, // Lo  [73] OLD TURKIC LETTER ORKHON A..OLD TURKIC LETTER ORKHON BASH
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x10D00, 0x10D23 }, // Lo  [36] HANIFI ROHINGYA LETTER A..HANIFI ROHINGYA MARK NA KHONNA
    Interval{ 0x10D24, 0x10D27 }, // Mn   [4] HANIFI ROHINGYA SIGN HARBAHAY..HANIFI ROHINGYA SIGN TASSI
    Interval{ 0x10D30, 0x10D39 }, // Nd  [10] HANIFI ROHINGYA DIGIT ZERO..HANIFI ROHINGYA DIGIT NINE
    Interval{ 0x10E80, 0x10EA9 }, // Lo  [42] YEZIDI LETTER ELIF..YEZIDI LETTER ET
    Interval{ 0x10EAB, 0x10EAC }, // Mn   [2] YEZIDI COMBINING HAMZA MARK..YEZIDI COMBINING MADDA MARK
    Interval{ 0x10EB0, 0x10EB1 }, // Lo   [2] YEZIDI LETTER LAM WITH DOT ABOVE..YEZIDI LETTER YOT WITH CIRCUMFLEX ABOVE
    Interval{ 0x10EFD, 0x10EFF }, // Mn   [3] ARABIC SMALL LOW WORD SAKTA..ARABIC SMALL LOW WORD MADDA
    Interval{ 0x10F00, 0x10F1C }, // Lo  [29] OLD SOGDIAN LETTER ALEPH..OLD SOGDIAN LETTER FINAL TAW WITH VERTICAL TAIL
    Interval{ 0x10F27, 0x10F27 }, // Lo       OLD SOGDIAN LIGATURE AYIN-DALETH
    Interval{ 0x10F30, 0x10F45 }, // Lo  [22] SOGDIAN LETTER ALEPH..SOGDIAN INDEPENDENT SHIN
    Interval{ 0x10F46, 0x10F50 }, // Mn  [11] SOGDIAN COMBINING DOT BELOW..SOGDIAN COMBINING STROKE BELOW
    Interval{ 0x10F70, 0x10F81 }, // Lo  [18] OLD UYGHUR LETTER ALEPH..OLD UYGHUR LETTER LESH
    Interval{ 0x10F82, 0x10F85 }, // Mn   [4] OLD UYGHUR COMBINING DOT ABOVE..OLD UYGHUR COMBINING TWO DOTS BELOW
    Interval{ 0x10FB0, 0x10FC4 }, // Lo  [21] CHORASMIAN LETTER ALEPH..CHORASMIAN LETTER TAW
    Interval{ 0x10FE0, 0x10FF6 }, // Lo  [23] ELYMAIC LETTER ALEPH..ELYMAIC LIGATURE ZAYIN-YODH
    Interval{ 0x11000, 0x11000 }, // Mc       BRAHMI SIGN CANDRABINDU
    Interval{ 0x11001, 0x11001 }, // Mn       BRAHMI SIGN ANUSVARA
    Interval{ 0x11002, 0x11002 }, // Mc       BRAHMI SIGN VISARGA
    Interval{ 0x11003, 0x11037 }, // Lo  [53] BRAHMI SIGN JIHVAMULIYA..BRAHMI LETTER OLD TAMIL NNNA
    Interval{ 0x11038, 0x11046 }, // Mn  [15] BRAHMI VOWEL SIGN AA..BRAHMI VIRAMA
    Interval{ 0x11066, 0x1106F }, // Nd  [10] BRAHMI DIGIT ZERO..BRAHMI DIGIT NINE
    Interval{ 0x11070, 0x11070 }, // Mn       BRAHMI SIGN OLD TAMIL VIRAMA
    Interval{ 0x11071, 0x11072 }, // Lo   [2] BRAHMI LETTER OLD TAMIL SHORT E..BRAHMI LETTER OLD TAMIL SHORT O
    Interval{ 0x11073, 0x11074 }, // Mn   [2] BRAHMI VOWEL SIGN OLD TAMIL SHORT E..BRAHMI VOWEL SIGN OLD TAMIL SHORT O
    Interval{ 0x11075, 0x11075 }, // Lo       BRAHMI LETTER OLD TAMIL LLA
    Interval{ 0x1107F, 0x11081 }, // Mn   [3] BRAHMI NUMBER JOINER..KAITHI SIGN ANUSVARA
    Interval{ 0x11082, 0x11082 }, // Mc       KAITHI SIGN VISARGA
    Interval{ 0x11083, 0x110AF }, // Lo  [45] KAITHI LETTER A..KAITHI LETTER HA
    Interval{ 0x110B0, 0x110B2 }, // Mc   [3] KAITHI VOWEL SIGN AA..KAITHI VOWEL SIGN II
    Interval{ 0x110B3, 0x110B6 }, // Mn   [4] KAITHI VOWEL SIGN U..KAITHI VOWEL SIGN AI
    Interval{ 0x110B7, 0x110B8 }, // Mc   [2] KAITHI VOWEL SIGN O..KAITHI VOWEL SIGN AU
    Interval{ 0x110B9, 0x110BA }, // Mn   [2] KAITHI SIGN VIRAMA..KAITHI SIGN NUKTA
    Interval{ 0x110C2, 0x110C2 }, // Mn       KAITHI VOWEL SIGN VOCALIC R
    Interval{ 0x110D0, 0x110E8 }, // Lo  [25] SORA SOMPENG LETTER SAH..SORA SOMPENG LETTER MAE
    Interval{ 0x110F0, 0x110F9 }, // Nd  [10] SORA SOMPENG DIGIT ZERO..SORA SOMPENG DIGIT NINE
    Interval{ 0x11100, 0x11102 }, // Mn   [3] CHAKMA SIGN CANDRABINDU..CHAKMA SIGN VISARGA
    Interval{ 0x11103, 0x11126 }, // Lo  [36] CHAKMA LETTER AA..CHAKMA LETTER HAA
    Interval{ 0x11127, 0x1112B }, // Mn   [5] CHAKMA VOWEL SIGN A..CHAKMA VOWEL SIGN UU
    Interval{ 0x1112C, 0x1112C }, // Mc       CHAKMA VOWEL SIGN E
    Interval{ 0x1112D, 0x11134 }, // Mn   [8] CHAKMA VOWEL SIGN AI..CHAKMA MAAYYAA
    Interval{ 0x11136, 0x1113F }, // Nd  [10] CHAKMA DIGIT ZERO..CHAKMA DIGIT NINE
    Interval{ 0x11144, 0x11144 }, // Lo       CHAKMA LETTER LHAA
    Interval{ 0x11145, 0x11146 }, // Mc   [2] CHAKMA VOWEL SIGN AA..CHAKMA VOWEL SIGN EI
    Interval{ 0x11147, 0x11147 }, // Lo       CHAKMA LETTER VAA
    Interval{ 0x11150, 0x11172 }, // Lo  [35] MAHAJANI LETTER A..MAHAJANI LETTER RRA
    Interval{ 0x11173, 0x11173 }, // Mn       MAHAJANI SIGN NUKTA
    Interval{ 0x11176, 0x11176 }, // Lo       MAHAJANI LIGATURE SHRI
    Interval{ 0x11180, 0x11181 }, // Mn   [2] SHARADA SIGN CANDRABINDU..SHARADA SIGN ANUSVARA
    Interval{ 0x11182, 0x11182 }, // Mc       SHARADA SIGN VISARGA
    Interval{ 0x11183, 0x111B2 }, // Lo  [48] SHARADA LETTER A..SHARADA LETTER HA
    Interval{ 0x111B3, 0x111B5 }, // Mc   [3] SHARADA VOWEL SIGN AA..SHARADA VOWEL SIGN II
    Interval{ 0x111B6, 0x111BE }, // Mn   [9] SHARADA VOWEL SIGN U..SHARADA VOWEL SIGN O
    Interval{ 0x111BF, 0x111C0 }, // Mc   [2] SHARADA VOWEL SIGN AU..SHARADA SIGN VIRAMA
    Interval{ 0x111C1, 0x111C4 }, // Lo   [4] SHARADA SIGN AVAGRAHA..SHARADA OM
    Interval{ 0x111C9, 0x111CC }, // Mn   [4] SHARADA SANDHI MARK..SHARADA EXTRA SHORT VOWEL MARK
    Interval{ 0x111CE, 0x111CE }, // Mc       SHARADA VOWEL SIGN PRISHTHAMATRA E
    Interval{ 0x111CF, 0x111CF }, // Mn       SHARADA SIGN INVERTED CANDRABINDU
    Interval{ 0x111D0, 0x111D9 }, // Nd  [10] SHARADA DIGIT ZERO..SHARADA DIGIT NINE
    Interval{ 0x111DA, 0x111DA }, // Lo       SHARADA EKAM
    Interval{ 0x111DC, 0x111DC }, // Lo       SHARADA HEADSTROKE
    Interval{ 0x11200, 0x11211 }, // Lo  [18] KHOJKI LETTER A..KHOJKI LETTER JJA
    Interval{ 0x11213, 0x1122B }, // Lo  [25] KHOJKI LETTER NYA..KHOJKI LETTER LLA
    Interval{ 0x1122C, 0x1122E }, // Mc   [3] KHOJKI VOWEL SIGN AA..KHOJKI VOWEL SIGN II
    Interval{ 0x1122F, 0x11231 }, // Mn   [3] KHOJKI VOWEL SIGN U..KHOJKI VOWEL SIGN AI
    Interval{ 0x11232, 0x11233 }, // Mc   [2] KHOJKI VOWEL SIGN O..KHOJKI VOWEL SIGN AU
    Interval{ 0x11234, 0x11234 }, // Mn       KHOJKI SIGN ANUSVARA
    Interval{ 0x11235, 0x11235 }, // Mc       KHOJKI SIGN VIRAMA
    Interval{ 0x11236, 0x11237 }, // Mn   [2] KHOJKI SIGN NUKTA..KHOJKI SIGN SHADDA
    Interval{ 0x1123E, 0x1123E }, // Mn       KHOJKI SIGN SUKUN
    Interval{ 0x1123F, 0x11240 }, // Lo   [2] KHOJKI LETTER QA..KHOJKI LETTER SHORT I
    Interval{ 0x11241, 0x11241 }, // Mn       KHOJKI VOWEL SIGN VOCALIC R
    Interval{ 0x11280, 0x11286 }, // Lo   [7] MULTANI LETTER A..MULTANI LETTER GA
    Interval{ 0x11288, 0x11288 }, // Lo       MULTANI LETTER GHA
    Interval{ 0x1128A, 0x1128D }, // Lo   [4] MULTANI LETTER CA..MULTANI LETTER JJA
    Interval{ 0x1128F, 0x1129D }, // Lo  [15] MULTANI LETTER NYA..MULTANI LETTER BA
    Interval{ 0x1129F, 0x112A8 }, // Lo  [10] MULTANI LETTER BHA..MULTANI LETTER RHA
    Interval{ 0x112B0, 0x112DE }, // Lo  [47] KHUDAWADI LETTER A..KHUDAWADI LETTER HA
    Interval{ 0x112DF, 0x112DF }, // Mn       KHUDAWADI SIGN ANUSVARA
    Interval{ 0x112E0, 0x112E2 }, // Mc   [3] KHUDAWADI VOWEL SIGN AA..KHUDAWADI VOWEL SIGN II
    Interval{ 0x112E3, 0x112EA }, // Mn   [8] KHUDAWADI VOWEL SIGN U..KHUDAWADI SIGN VIRAMA
    Interval{ 0x112F0, 0x112F9 }, // Nd  [10] KHUDAWADI DIGIT ZERO..KHUDAWADI DIGIT NINE
    Interval{ 0x11300, 0x11301 }, // Mn   [2] GRANTHA SIGN COMBINING ANUSVARA ABOVE..GRANTHA SIGN CANDRABINDU
    Interval{ 0x11302, 0x11303 }, // Mc   [2] GRANTHA SIGN ANUSVARA..GRANTHA SIGN VISARGA
    Interval{ 0x11305, 0x1130C }, // Lo   [8] GRANTHA LETTER A..GRANTHA LETTER VOCALIC L
    Interval{ 0x1130F, 0x11310 }, // Lo   [2] GRANTHA LETTER EE..GRANTHA LETTER AI
    Interval{ 0x11313, 0x11328 }, // Lo  [22] GRANTHA LETTER OO..GRANTHA LETTER NA
    Interval{ 0x1132A, 0x11330 }, // Lo   [7] GRANTHA LETTER PA..GRANTHA LETTER RA
    Interval{ 0x11332, 0x11333 }, // Lo   [2] GRANTHA LETTER LA..GRANTHA LETTER LLA
    Interval{ 0x11335, 0x11339 }, // Lo   [5] GRANTHA LETTER VA..GRANTHA LETTER HA
    Interval{ 0x1133B, 0x1133C }, // Mn   [2] COMBINING BINDU BELOW..GRANTHA SIGN NUKTA
    Interval{ 0x1133D, 0x1133D }, // Lo       GRANTHA SIGN AVAGRAHA
    Interval{ 0x1133E, 0x1133F }, // Mc   [2] GRANTHA VOWEL SIGN AA..GRANTHA VOWEL SIGN I
    Interval{ 0x11340, 0x11340 }, // Mn       GRANTHA VOWEL SIGN II
    Interval{ 0x11341, 0x11344 }, // Mc   [4] GRANTHA VOWEL SIGN U..GRANTHA VOWEL SIGN VOCALIC RR
    Interval{ 0x11347, 0x11348 }, // Mc   [2] GRANTHA VOWEL SIGN EE..GRANTHA VOWEL SIGN AI
    Interval{ 0x1134B, 0x1134D }, // Mc   [3] GRANTHA VOWEL SIGN OO..GRANTHA SIGN VIRAMA
    Interval{ 0x11350, 0x11350 }, // Lo       GRANTHA OM
    Interval{ 0x11357, 0x11357 }, // Mc       GRANTHA AU LENGTH MARK
    Interval{ 0x1135D, 0x11361 }, // Lo   [5] GRANTHA SIGN PLUTA..GRANTHA LETTER VOCALIC LL
    Interval{ 0x11362, 0x11363 }, // Mc   [2] GRANTHA VOWEL SIGN VOCALIC L..GRANTHA VOWEL SIGN VOCALIC LL
    Interval{ 0x11366, 0x1136C }, // Mn   [7] COMBINING GRANTHA DIGIT ZERO..COMBINING GRANTHA DIGIT SIX
    Interval{ 0x11370, 0x11374 }, // Mn   [5] COMBINING GRANTHA LETTER A..COMBINING GRANTHA LETTER PA
    Interval{ 0x11400, 0x11434 }, // Lo  [53] NEWA LETTER A..NEWA LETTER HA
    Interval{ 0x11435, 0x11437 }, // Mc   [3] NEWA VOWEL SIGN AA..NEWA VOWEL SIGN II
    Interval{ 0x11438, 0x1143F }, // Mn   [8] NEWA VOWEL SIGN U..NEWA VOWEL SIGN AI
    Interval{ 0x11440, 0x11441 }, // Mc   [2] NEWA VOWEL SIGN O..NEWA VOWEL SIGN AU
    Interval{ 0x11442, 0x11444 }, // Mn   [3] NEWA SIGN VIRAMA..NEWA SIGN ANUSVARA
    Interval{ 0x11445, 0x11445 }, // Mc       NEWA SIGN VISARGA
    Interval{ 0x11446, 0x11446 }, // Mn       NEWA SIGN NUKTA
    Interval{ 0x11447, 0x1144A }, // Lo   [4] NEWA SIGN AVAGRAHA..NEWA SIDDHI
    Interval{ 0x11450, 0x11459 }, // Nd  [10] NEWA DIGIT ZERO..NEWA DIGIT NINE
    Interval{ 0x1145E, 0x1145E }, // Mn       NEWA SANDHI MARK
    Interval{ 0x1145F, 0x11461 }, // Lo   [3] NEWA LETTER VEDIC ANUSVARA..NEWA SIGN UPADHMANIYA
    Interval{ 0x11480, 0x114AF }, // Lo  [48] TIRHUTA ANJI..TIRHUTA LETTER HA
    Interval{ 0x114B0, 0x114B2 }, // Mc   [3] TIRHUTA VOWEL SIGN AA..TIRHUTA VOWEL SIGN II
    Interval{ 0x114B3, 0x114B8 }, // Mn   [6] TIRHUTA VOWEL SIGN U..TIRHUTA VOWEL SIGN VOCALIC LL
    Interval{ 0x114B9, 0x114B9 }, // Mc       TIRHUTA VOWEL SIGN E
    Interval{ 0x114BA, 0x114BA }, // Mn       TIRHUTA VOWEL SIGN SHORT E
    Interval{ 0x114BB, 0x114BE }, // Mc   [4] TIRHUTA VOWEL SIGN AI..TIRHUTA VOWEL SIGN AU
    Interval{ 0x114BF, 0x114C0 }, // Mn   [2] TIRHUTA SIGN CANDRABINDU..TIRHUTA SIGN ANUSVARA
    Interval{ 0x114C1, 0x114C1 }, // Mc       TIRHUTA SIGN VISARGA
    Interval{ 0x114C2, 0x114C3 }, // Mn   [2] TIRHUTA SIGN VIRAMA..TIRHUTA SIGN NUKTA
    Interval{ 0x114C4, 0x114C5 }, // Lo   [2] TIRHUTA SIGN AVAGRAHA..TIRHUTA GVANG
    Interval{ 0x114C7, 0x114C7 }, // Lo       TIRHUTA OM
    Interval{ 0x114D0, 0x114D9 }, // Nd  [10] TIRHUTA DIGIT ZERO..TIRHUTA DIGIT NINE
    Interval{ 0x11580, 0x115AE }, // Lo  [47] SIDDHAM LETTER A..SIDDHAM LETTER HA
    Interval{ 0x115AF, 0x115B1 }, // Mc   [3] SIDDHAM VOWEL SIGN AA..SIDDHAM VOWEL SIGN II
    Interval{ 0x115B2, 0x115B5 }, // Mn   [4] SIDDHAM VOWEL SIGN U..SIDDHAM VOWEL SIGN VOCALIC RR
    Interval{ 0x115B8, 0x115BB }, // Mc   [4] SIDDHAM VOWEL SIGN E..SIDDHAM VOWEL SIGN AU
    Interval{ 0x115BC, 0x115BD }, // Mn   [2] SIDDHAM SIGN CANDRABINDU..SIDDHAM SIGN ANUSVARA
    Interval{ 0x115BE, 0x115BE }, // Mc       SIDDHAM SIGN VISARGA
    Interval{ 0x115BF, 0x115C0 }, // Mn   [2] SIDDHAM SIGN VIRAMA..SIDDHAM SIGN NUKTA
    Interval{ 0x115D8, 0x115DB }, // Lo   [4] SIDDHAM LETTER THREE-CIRCLE ALTERNATE I..SIDDHAM LETTER ALTERNATE U
    Interval{ 0x115DC, 0x115DD }, // Mn   [2] SIDDHAM VOWEL SIGN ALTERNATE U..SIDDHAM VOWEL SIGN ALTERNATE UU
    Interval{ 0x11600, 0x1162F }, // Lo  [48] MODI LETTER A..MODI LETTER LLA
    Interval{ 0x11630, 0x11632 }, // Mc   [3] MODI VOWEL SIGN AA..MODI VOWEL SIGN II
    Interval{ 0x11633, 0x1163A }, // Mn   [8] MODI VOWEL SIGN U..MODI VOWEL SIGN AI
    Interval{ 0x1163B, 0x1163C }, // Mc   [2] MODI VOWEL SIGN O..MODI VOWEL SIGN AU
    Interval{ 0x1163D, 0x1163D }, // Mn       MODI SIGN ANUSVARA
    Interval{ 0x1163E, 0x1163E }, // Mc       MODI SIGN VISARGA
    Interval{ 0x1163F, 0x11640 }, // Mn   [2] MODI SIGN VIRAMA..MODI SIGN ARDHACANDRA
    Interval{ 0x11644, 0x11644 }, // Lo       MODI SIGN HUVA
    Interval{ 0x11650, 0x11659 }, // Nd  [10] MODI DIGIT ZERO..MODI DIGIT NINE
    Interval{ 0x11680, 0x116AA }, // Lo  [43] TAKRI LETTER A..TAKRI LETTER RRA
    Interval{ 0x116AB, 0x116AB }, // Mn       TAKRI SIGN ANUSVARA
    Interval{ 0x116AC, 0x116AC }, // Mc       TAKRI SIGN VISARGA
    Interval{ 0x116AD, 0x116AD }, // Mn       TAKRI VOWEL SIGN AA
    Interval{ 0x116AE, 0x116AF }, // Mc   [2] TAKRI VOWEL SIGN I..TAKRI VOWEL SIGN II
    Interval{ 0x116B0, 0x116B5 }, // Mn   [6] TAKRI VOWEL SIGN U..TAKRI VOWEL SIGN AU
    Interval{ 0x116B6, 0x116B6 }, // Mc       TAKRI SIGN VIRAMA
    Interval{ 0x116B7, 0x116B7 }, // Mn       TAKRI SIGN NUKTA
    Interval{ 0x116B8, 0x116B8 }, // Lo       TAKRI LETTER ARCHAIC KHA
    Interval{ 0x116C0, 0x116C9 }, // Nd  [10] TAKRI DIGIT ZERO..TAKRI DIGIT NINE
    Interval{ 0x11700, 0x1171A }, // Lo  [27] AHOM LETTER KA..AHOM LETTER ALTERNATE BA
    Interval{ 0x1171D, 0x1171F }, // Mn   [3] AHOM CONSONANT SIGN MEDIAL LA..AHOM CONSONANT SIGN MEDIAL LIGATING RA
    Interval{ 0x11720, 0x11721 }, // Mc   [2] AHOM VOWEL SIGN A..AHOM VOWEL SIGN AA
    Interval{ 0x11722, 0x11725 }, // Mn   [4] AHOM VOWEL SIGN I..AHOM VOWEL SIGN UU
    Interval{ 0x11726, 0x11726 }, // Mc       AHOM VOWEL SIGN E
    Interval{ 0x11727, 0x1172B }, // Mn   [5] AHOM VOWEL SIGN AW..AHOM SIGN KILLER
    Interval{ 0x11730, 0x11739 }, // Nd  [10] AHOM DIGIT ZERO..AHOM DIGIT NINE
    Interval{ 0x11740, 0x11746 }, // Lo   [7] AHOM LETTER CA..AHOM LETTER LLA
    Interval{ 0x11800, 0x1182B }, // Lo  [44] DOGRA LETTER A..DOGRA LETTER RRA
    Interval{ 0x1182C, 0x1182E }, // Mc   [3] DOGRA VOWEL SIGN AA..DOGRA VOWEL SIGN II
    Interval{ 0x1182F, 0x11837 }, // Mn   [9] DOGRA VOWEL SIGN U..DOGRA SIGN ANUSVARA
    Interval{ 0x11838, 0x11838 }, // Mc       DOGRA SIGN VISARGA
    Interval{ 0x11839, 0x1183A }, // Mn   [2] DOGRA SIGN VIRAMA..DOGRA SIGN NUKTA
    Interval{ 0x118A0, 0x118DF }, // L&  [64] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x118E0, 0x118E9 }, // Nd  [10] WARANG CITI DIGIT ZERO..WARANG CITI DIGIT NINE
    Interval{ 0x118FF, 0x11906 }, // Lo   [8] WARANG CITI OM..DIVES AKURU LETTER E
    Interval{ 0x11909, 0x11909 }, // Lo       DIVES AKURU LETTER O
    Interval{ 0x1190C, 0x11913 }, // Lo   [8] DIVES AKURU LETTER KA..DIVES AKURU LETTER JA
    Interval{ 0x11915, 0x11916 }, // Lo   [2] DIVES AKURU LETTER NYA..DIVES AKURU LETTER TTA
    Interval{ 0x11918, 0x1192F }, // Lo  [24] DIVES AKURU LETTER DDA..DIVES AKURU LETTER ZA
    Interval{ 0x11930, 0x11935 }, // Mc   [6] DIVES AKURU VOWEL SIGN AA..DIVES AKURU VOWEL SIGN E
    Interval{ 0x11937, 0x11938 }, // Mc   [2] DIVES AKURU VOWEL SIGN AI..DIVES AKURU VOWEL SIGN O
    Interval{ 0x1193B, 0x1193C }, // Mn   [2] DIVES AKURU SIGN ANUSVARA..DIVES AKURU SIGN CANDRABINDU
    Interval{ 0x1193D, 0x1193D }, // Mc       DIVES AKURU SIGN HALANTA
    Interval{ 0x1193E, 0x1193E }, // Mn       DIVES AKURU VIRAMA
    Interval{ 0x1193F, 0x1193F }, // Lo       DIVES AKURU PREFIXED NASAL SIGN
    Interval{ 0x11940, 0x11940 }, // Mc       DIVES AKURU MEDIAL YA
    Interval{ 0x11941, 0x11941 }, // Lo       DIVES AKURU INITIAL RA
    Interval{ 0x11942, 0x11942 }, // Mc       DIVES AKURU MEDIAL RA
    Interval{ 0x11943, 0x11943 }, // Mn       DIVES AKURU SIGN NUKTA
    Interval{ 0x11950, 0x11959 }, // Nd  [10] DIVES AKURU DIGIT ZERO..DIVES AKURU DIGIT NINE
    Interval{ 0x119A0, 0x119A7 }, // Lo   [8] NANDINAGARI LETTER A..NANDINAGARI LETTER VOCALIC RR
    Interval{ 0x119AA, 0x119D0 }, // Lo  [39] NANDINAGARI LETTER E..NANDINAGARI LETTER RRA
    Interval{ 0x119D1, 0x119D3 }, // Mc   [3] NANDINAGARI VOWEL SIGN AA..NANDINAGARI VOWEL SIGN II
    Interval{ 0x119D4, 0x119D7 }, // Mn   [4] NANDINAGARI VOWEL SIGN U..NANDINAGARI VOWEL SIGN VOCALIC RR
    Interval{ 0x119DA, 0x119DB }, // Mn   [2] NANDINAGARI VOWEL SIGN E..NANDINAGARI VOWEL SIGN AI
    Interval{ 0x119DC, 0x119DF }, // Mc   [4] NANDINAGARI VOWEL SIGN O..NANDINAGARI SIGN VISARGA
    Interval{ 0x119E0, 0x119E0 }, // Mn       NANDINAGARI SIGN VIRAMA
    Interval{ 0x119E1, 0x119E1 }, // Lo       NANDINAGARI SIGN AVAGRAHA
    Interval{ 0x119E3, 0x119E3 }, // Lo       NANDINAGARI HEADSTROKE
    Interval{ 0x119E4, 0x119E4 }, // Mc       NANDINAGARI VOWEL SIGN PRISHTHAMATRA E
    Interval{ 0x11A00, 0x11A00 }, // Lo       ZANABAZAR SQUARE LETTER A
    Interval{ 0x11A01, 0x11A0A }, // Mn  [10] ZANABAZAR SQUARE VOWEL SIGN I..ZANABAZAR SQUARE VOWEL LENGTH MARK
    Interval{ 0x11A0B, 0x11A32 }, // Lo  [40] ZANABAZAR SQUARE LETTER KA..ZANABAZAR SQUARE LETTER KSSA
    Interval{ 0x11A33, 0x11A38 }, // Mn   [6] ZANABAZAR SQUARE FINAL CONSONANT MARK..ZANABAZAR SQUARE SIGN ANUSVARA
    Interval{ 0x11A39, 0x11A39 }, // Mc       ZANABAZAR SQUARE SIGN VISARGA
    Interval{ 0x11A3A, 0x11A3A }, // Lo       ZANABAZAR SQUARE CLUSTER-INITIAL LETTER RA
    Interval{ 0x11A3B, 0x11A3E }, // Mn   [4] ZANABAZAR SQUARE CLUSTER-FINAL LETTER YA..ZANABAZAR SQUARE CLUSTER-FINAL LETTER VA
    Interval{ 0x11A47, 0x11A47 }, // Mn       ZANABAZAR SQUARE SUBJOINER
    Interval{ 0x11A50, 0x11A50 }, // Lo       SOYOMBO LETTER A
    Interval{ 0x11A51, 0x11A56 }, // Mn   [6] SOYOMBO VOWEL SIGN I..SOYOMBO VOWEL SIGN OE
    Interval{ 0x11A57, 0x11A58 }, // Mc   [2] SOYOMBO VOWEL SIGN AI..SOYOMBO VOWEL SIGN AU
    Interval{ 0x11A59, 0x11A5B }, // Mn   [3] SOYOMBO VOWEL SIGN VOCALIC R..SOYOMBO VOWEL LENGTH MARK
    Interval{ 0x11A5C, 0x11A89 }, // Lo  [46] SOYOMBO LETTER KA..SOYOMBO CLUSTER-INITIAL LETTER SA
    Interval{ 0x11A8A, 0x11A96 }, // Mn  [13] SOYOMBO FINAL CONSONANT SIGN G..SOYOMBO SIGN ANUSVARA
    Interval{ 0x11A97, 0x11A97 }, // Mc       SOYOMBO SIGN VISARGA
    Interval{ 0x11A98, 0x11A99 }, // Mn   [2] SOYOMBO GEMINATION MARK..SOYOMBO SUBJOINER
    Interval{ 0x11A9D, 0x11A9D }, // Lo       SOYOMBO MARK PLUTA
    Interval{ 0x11AB0, 0x11AF8 }, // Lo  [73] CANADIAN SYLLABICS NATTILIK HI..PAU CIN HAU GLOTTAL STOP FINAL
    Interval{ 0x11C00, 0x11C08 }, // Lo   [9] BHAIKSUKI LETTER A..BHAIKSUKI LETTER VOCALIC L
    Interval{ 0x11C0A, 0x11C2E }, // Lo  [37] BHAIKSUKI LETTER E..BHAIKSUKI LETTER HA
    Interval{ 0x11C2F, 0x11C2F }, // Mc       BHAIKSUKI VOWEL SIGN AA
    Interval{ 0x11C30, 0x11C36 }, // Mn   [7] BHAIKSUKI VOWEL SIGN I..BHAIKSUKI VOWEL SIGN VOCALIC L
    Interval{ 0x11C38, 0x11C3D }, // Mn   [6] BHAIKSUKI VOWEL SIGN E..BHAIKSUKI SIGN ANUSVARA
    Interval{ 0x11C3E, 0x11C3E }, // Mc       BHAIKSUKI SIGN VISARGA
    Interval{ 0x11C3F, 0x11C3F }, // Mn       BHAIKSUKI SIGN VIRAMA
    Interval{ 0x11C40, 0x11C40 }, // Lo       BHAIKSUKI SIGN AVAGRAHA
    Interval{ 0x11C50, 0x11C59 }, // Nd  [10] BHAIKSUKI DIGIT ZERO..BHAIKSUKI DIGIT NINE
    Interval{ 0x11C72, 0x11C8F }, // Lo  [30] MARCHEN LETTER KA..MARCHEN LETTER A
    Interval{ 0x11C92, 0x11CA7 }, // Mn  [22] MARCHEN SUBJOINED LETTER KA..MARCHEN SUBJOINED LETTER ZA
    Interval{ 0x11CA9, 0x11CA9 }, // Mc       MARCHEN SUBJOINED LETTER YA
    Interval{ 0x11CAA, 0x11CB0 }, // Mn   [7] MARCHEN SUBJOINED LETTER RA..MARCHEN VOWEL SIGN AA
    Interval{ 0x11CB1, 0x11CB1 }, // Mc       MARCHEN VOWEL SIGN I
    Interval{ 0x11CB2, 0x11CB3 }, // Mn   [2] MARCHEN VOWEL SIGN U..MARCHEN VOWEL SIGN E
    Interval{ 0x11CB4, 0x11CB4 }, // Mc       MARCHEN VOWEL SIGN O
    Interval{ 0x11CB5, 0x11CB6 }, // Mn   [2] MARCHEN SIGN ANUSVARA..MARCHEN SIGN CANDRABINDU
    Interval{ 0x11D00, 0x11D06 }, // Lo   [7] MASARAM GONDI LETTER A..MASARAM GONDI LETTER E
    Interval{ 0x11D08, 0x11D09 }, // Lo   [2] MASARAM GONDI LETTER AI..MASARAM GONDI LETTER O
    Interval{ 0x11D0B, 0x11D30 }, // Lo  [38] MASARAM GONDI LETTER AU..MASARAM GONDI LETTER TRA
    Interval{ 0x11D31, 0x11D36 }, // Mn   [6] MASARAM GONDI VOWEL SIGN AA..MASARAM GONDI VOWEL SIGN VOCALIC R
    Interval{ 0x11D3A, 0x11D3A }, // Mn       MASARAM GONDI VOWEL SIGN E
    Interval{ 0x11D3C, 0x11D3D }, // Mn   [2] MASARAM GONDI VOWEL SIGN AI..MASARAM GONDI VOWEL SIGN O
    Interval{ 0x11D3F, 0x11D45 }, // Mn   [7] MASARAM GONDI VOWEL SIGN AU..MASARAM GONDI VIRAMA
    Interval{ 0x11D46, 0x11D46 }, // Lo       MASARAM GONDI REPHA
    Interval{ 0x11D47, 0x11D47 }, // Mn       MASARAM GONDI RA-KARA
    Interval{ 0x11D50, 0x11D59 }, // Nd  [10] MASARAM GONDI DIGIT ZERO..MASARAM GONDI DIGIT NINE
    Interval{ 0x11D60, 0x11D65 }, // Lo   [6] GUNJALA GONDI LETTER A..GUNJALA GONDI LETTER UU
    Interval{ 0x11D67, 0x11D68 }, // Lo   [2] GUNJALA GONDI LETTER EE..GUNJALA GONDI LETTER AI
    Interval{ 0x11D6A, 0x11D89 }, // Lo  [32] GUNJALA GONDI LETTER OO..GUNJALA GONDI LETTER SA
    Interval{ 0x11D8A, 0x11D8E }, // Mc   [5] GUNJALA GONDI VOWEL SIGN AA..GUNJALA GONDI VOWEL SIGN UU
    Interval{ 0x11D90, 0x11D91 }, // Mn   [2] GUNJALA GONDI VOWEL SIGN EE..GUNJALA GONDI VOWEL SIGN AI
    Interval{ 0x11D93, 0x11D94 }, // Mc   [2] GUNJALA GONDI VOWEL SIGN OO..GUNJALA GONDI VOWEL SIGN AU
    Interval{ 0x11D95, 0x11D95 }, // Mn       GUNJALA GONDI SIGN ANUSVARA
    Interval{ 0x11D96, 0x11D96 }, // Mc       GUNJALA GONDI SIGN VISARGA
    Interval{ 0x11D97, 0x11D97 }, // Mn       GUNJALA GONDI VIRAMA
    Interval{ 0x11D98, 0x11D98 }, // Lo       GUNJALA GONDI OM
    Interval{ 0x11DA0, 0x11DA9 }, // Nd  [10] GUNJALA GONDI DIGIT ZERO..GUNJALA GONDI DIGIT NINE
    Interval{ 0x11EE0, 0x11EF2 }, // Lo  [19] MAKASAR LETTER KA..MAKASAR ANGKA
    Interval{ 0x11EF3, 0x11EF4 }, // Mn   [2] MAKASAR VOWEL SIGN I..MAKASAR VOWEL SIGN U
    Interval{ 0x11EF5, 0x11EF6 }, // Mc   [2] MAKASAR VOWEL SIGN E..MAKASAR VOWEL SIGN O
    Interval{ 0x11F00, 0x11F01 }, // Mn   [2] KAWI SIGN CANDRABINDU..KAWI SIGN ANUSVARA
    Interval{ 0x11F02, 0x11F02 }, // Lo       KAWI SIGN REPHA
    Interval{ 0x11F03, 0x11F03 }, // Mc       KAWI SIGN VISARGA
    Interval{ 0x11F04, 0x11F10 }, // Lo  [13] KAWI LETTER A..KAWI LETTER O
    Interval{ 0x11F12, 0x11F33 }, // Lo  [34] KAWI LETTER KA..KAWI LETTER JNYA
    Interval{ 0x11F34, 0x11F35 }, // Mc   [2] KAWI VOWEL SIGN AA..KAWI VOWEL SIGN ALTERNATE AA
    Interval{ 0x11F36, 0x11F3A }, // Mn   [5] KAWI VOWEL SIGN I..KAWI VOWEL SIGN VOCALIC R
    Interval{ 0x11F3E, 0x11F3F }, // Mc   [2] KAWI VOWEL SIGN E..KAWI VOWEL SIGN AI
    Interval{ 0x11F40, 0x11F40 }, // Mn       KAWI VOWEL SIGN EU
    Interval{ 0x11F41, 0x11F41 }, // Mc       KAWI SIGN KILLER
    Interval{ 0x11F42, 0x11F42 }, // Mn       KAWI CONJOINER
    Interval{ 0x11F50, 0x11F59 }, // Nd  [10] KAWI DIGIT ZERO..KAWI DIGIT NINE
    Interval{ 0x11FB0, 0x11FB0 }, // Lo       LISU LETTER YHA
    Interval{ 0x12000, 0x12399 }, // Lo [922] CUNEIFORM SIGN A..CUNEIFORM SIGN U U
    Interval{ 0x12400, 0x1246E }, // Nl [111] CUNEIFORM NUMERIC SIGN TWO ASH..CUNEIFORM NUMERIC SIGN NINE U VARIANT FORM
    Interval{ 0x12480, 0x12543 }, // Lo [196] CUNEIFORM SIGN AB TIMES NUN TENU..CUNEIFORM SIGN ZU5 TIMES THREE DISH TENU
    Interval{ 0x12F90, 0x12FF0 }, // Lo  [97] CYPRO-MINOAN SIGN CM001..CYPRO-MINOAN SIGN CM114
    Interval{ 0x13000, 0x1342F }, // Lo [1072] EGYPTIAN HIEROGLYPH A001..EGYPTIAN HIEROGLYPH V011D
    Interval{ 0x13440, 0x13440 }, // Mn       EGYPTIAN HIEROGLYPH MIRROR HORIZONTALLY
    Interval{ 0x13441, 0x13446 }, // Lo   [6] EGYPTIAN HIEROGLYPH FULL BLANK..EGYPTIAN HIEROGLYPH WIDE LOST SIGN
    Interval{ 0x13447, 0x13455 }, // Mn  [15] EGYPTIAN HIEROGLYPH MODIFIER DAMAGED AT TOP START..EGYPTIAN HIEROGLYPH MODIFIER DAMAGED
    Interval{ 0x14400, 0x14646 }, // Lo [583] ANATOLIAN HIEROGLYPH A001..ANATOLIAN HIEROGLYPH A530
    Interval{ 0x16800, 0x16A38 }, // Lo [569] BAMUM LETTER PHASE-A NGKUE MFON..BAMUM LETTER PHASE-F VUEQ
    Interval{ 0x16A40, 0x16A5E }, // Lo  [31] MRO LETTER TA..MRO LETTER TEK
    Interval{ 0x16A60, 0x16A69 }, // Nd  [10] MRO DIGIT ZERO..MRO DIGIT NINE
    Interval{ 0x16A70, 0x16ABE }, // Lo  [79] TANGSA LETTER OZ..TANGSA LETTER ZA
    Interval{ 0x16AC0, 0x16AC9 }, // Nd  [10] TANGSA DIGIT ZERO..TANGSA DIGIT NINE
    Interval{ 0x16AD0, 0x16AED }, // Lo  [30] BASSA VAH LETTER ENNI..BASSA VAH LETTER I
    Interval{ 0x16AF0, 0x16AF4 }, // Mn   [5] BASSA VAH COMBINING HIGH TONE..BASSA VAH COMBINING HIGH-LOW TONE
    Interval{ 0x16B00, 0x16B2F }, // Lo  [48] PAHAWH HMONG VOWEL KEEB..PAHAWH HMONG CONSONANT CAU
    Interval{ 0x16B30, 0x16B36 }, // Mn   [7] PAHAWH HMONG MARK CIM TUB..PAHAWH HMONG MARK CIM TAUM
    Interval{ 0x16B40, 0x16B43 }, // Lm   [4] PAHAWH HMONG SIGN VOS SEEV..PAHAWH HMONG SIGN IB YAM
    Interval{ 0x16B50, 0x16B59 }, // Nd  [10] PAHAWH HMONG DIGIT ZERO..PAHAWH HMONG DIGIT NINE
    Interval{ 0x16B63, 0x16B77 }, // Lo  [21] PAHAWH HMONG SIGN VOS LUB..PAHAWH HMONG SIGN CIM NRES TOS
    Interval{ 0x16B7D, 0x16B8F }, // Lo  [19] PAHAWH HMONG CLAN SIGN TSHEEJ..PAHAWH HMONG CLAN SIGN VWJ
    Interval{ 0x16E40, 0x16E7F }, // L&  [64] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x16F00, 0x16F4A }, // Lo  [75] MIAO LETTER PA..MIAO LETTER RTE
    Interval{ 0x16F4F, 0x16F4F }, // Mn       MIAO SIGN CONSONANT MODIFIER BAR
    Interval{ 0x16F50, 0x16F50 }, // Lo       MIAO LETTER NASALIZATION
    Interval{ 0x16F51, 0x16F87 }, // Mc  [55] MIAO SIGN ASPIRATION..MIAO VOWEL SIGN UI
    Interval{ 0x16F8F, 0x16F92 }, // Mn   [4] MIAO TONE RIGHT..MIAO TONE BELOW
    Interval{ 0x16F93, 0x16F9F }, // Lm  [13] MIAO LETTER TONE-2..MIAO LETTER REFORMED TONE-8
    Interval{ 0x16FE0, 0x16FE1 }, // Lm   [2] TANGUT ITERATION MARK..NUSHU ITERATION MARK
    Interval{ 0x16FE3, 0x16FE3 }, // Lm       OLD CHINESE ITERATION MARK
    Interval{ 0x16FE4, 0x16FE4 }, // Mn       KHITAN SMALL SCRIPT FILLER
    Interval{ 0x16FF0, 0x16FF1 }, // Mc   [2] VIETNAMESE ALTERNATE READING MARK CA..VIETNAMESE ALTERNATE READING MARK NHAY
    Interval{ 0x17000, 0x187F7 }, // Lo [6136] TANGUT IDEOGRAPH-17000..TANGUT IDEOGRAPH-187F7
    Interval{ 0x18800, 0x18CD5 }, // Lo [1238] TANGUT COMPONENT-001..KHITAN SMALL SCRIPT CHARACTER-18CD5
    Interval{ 0x18D00, 0x18D08 }, // Lo   [9] TANGUT IDEOGRAPH-18D00..TANGUT IDEOGRAPH-18D08
    Interval{ 0x1AFF0, 0x1AFF3 }, // Lm   [4] KATAKANA LETTER MINNAN TONE-2..KATAKANA LETTER MINNAN TONE-5
    Interval{ 0x1AFF5, 0x1AFFB }, // Lm   [7] KATAKANA LETTER MINNAN TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-5
    Interval{ 0x1AFFD, 0x1AFFE }, // Lm   [2] KATAKANA LETTER MINNAN NASALIZED TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-8
    Interval{ 0x1B000, 0x1B122 }, // Lo [291] KATAKANA LETTER ARCHAIC E..KATAKANA LETTER ARCHAIC WU
    Interval{ 0x1B132, 0x1B132 }, // Lo       HIRAGANA LETTER SMALL KO
    Interval{ 0x1B150, 0x1B152 }, // Lo   [3] HIRAGANA LETTER SMALL WI..HIRAGANA LETTER SMALL WO
    Interval{ 0x1B155, 0x1B155 }, // Lo       KATAKANA LETTER SMALL KO
    Interval{ 0x1B164, 0x1B167 }, // Lo   [4] KATAKANA LETTER SMALL WI..KATAKANA LETTER SMALL N
    Interval{ 0x1B170, 0x1B2FB }, // Lo [396] NUSHU CHARACTER-1B170..NUSHU CHARACTER-1B2FB
    Interval{ 0x1BC00, 0x1BC6A }, // Lo [107] DUPLOYAN LETTER H..DUPLOYAN LETTER VOCALIC M
    Interval{ 0x1BC70, 0x1BC7C }, // Lo  [13] DUPLOYAN AFFIX LEFT HORIZONTAL SECANT..DUPLOYAN AFFIX ATTACHED TANGENT HOOK
    Interval{ 0x1BC80, 0x1BC88 }, // Lo   [9] DUPLOYAN AFFIX HIGH ACUTE..DUPLOYAN AFFIX HIGH VERTICAL
    Interval{ 0x1BC90, 0x1BC99 }, // Lo  [10] DUPLOYAN AFFIX LOW ACUTE..DUPLOYAN AFFIX LOW ARROW
    Interval{ 0x1BC9D, 0x1BC9E }, // Mn   [2] DUPLOYAN THICK LETTER SELECTOR..DUPLOYAN DOUBLE MARK
    Interval{ 0x1CF00, 0x1CF2D }, // Mn  [46] ZNAMENNY COMBINING MARK GORAZDO NIZKO S KRYZHEM ON LEFT..ZNAMENNY COMBINING MARK KRYZH ON LEFT
    Interval{ 0x1CF30, 0x1CF46 }, // Mn  [23] ZNAMENNY COMBINING TONAL RANGE MARK MRACHNO..ZNAMENNY PRIZNAK MODIFIER ROG
    Interval{ 0x1D165, 0x1D166 }, // Mc   [2] MUSICAL SYMBOL COMBINING STEM..MUSICAL SYMBOL COMBINING SPRECHGESANG STEM
    Interval{ 0x1D167, 0x1D169 }, // Mn   [3] MUSICAL SYMBOL COMBINING TREMOLO-1..MUSICAL SYMBOL COMBINING TREMOLO-3
    Interval{ 0x1D16D, 0x1D172 }, // Mc   [6] MUSICAL SYMBOL COMBINING AUGMENTATION DOT..MUSICAL SYMBOL COMBINING FLAG-5
    Interval{ 0x1D17B, 0x1D182 }, // Mn   [8] MUSICAL SYMBOL COMBINING ACCENT..MUSICAL SYMBOL COMBINING LOURE
    Interval{ 0x1D185, 0x1D18B }, // Mn   [7] MUSICAL SYMBOL COMBINING DOIT..MUSICAL SYMBOL COMBINING TRIPLE TONGUE
    Interval{ 0x1D1AA, 0x1D1AD }, // Mn   [4] MUSICAL SYMBOL COMBINING DOWN BOW..MUSICAL SYMBOL COMBINING SNAP PIZZICATO
    Interval{ 0x1D242, 0x1D244 }, // Mn   [3] COMBINING GREEK MUSICAL TRISEME..COMBINING GREEK MUSICAL PENTASEME
    Interval{ 0x1D400, 0x1D454 }, // L&  [85] MATHEMATICAL BOLD CAPITAL A..MATHEMATICAL ITALIC SMALL G
    Interval{ 0x1D456, 0x1D49C }, // L&  [71] MATHEMATICAL ITALIC SMALL I..MATHEMATICAL SCRIPT CAPITAL A
    Interval{ 0x1D49E, 0x1D49F }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL C..MATHEMATICAL SCRIPT CAPITAL D
    Interval{ 0x1D4A2, 0x1D4A2 }, // L&       MATHEMATICAL SCRIPT CAPITAL G
    Interval{ 0x1D4A5, 0x1D4A6 }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL J..MATHEMATICAL SCRIPT CAPITAL K
    Interval{ 0x1D4A9, 0x1D4AC }, // L&   [4] MATHEMATICAL SCRIPT CAPITAL N..MATHEMATICAL SCRIPT CAPITAL Q
    Interval{ 0x1D4AE, 0x1D4B9 }, // L&  [12] MATHEMATICAL SCRIPT CAPITAL S..MATHEMATICAL SCRIPT SMALL D
    Interval{ 0x1D4BB, 0x1D4BB }, // L&       MATHEMATICAL SCRIPT SMALL F
    Interval{ 0x1D4BD, 0x1D4C3 }, // L&   [7] MATHEMATICAL SCRIPT SMALL H..MATHEMATICAL SCRIPT SMALL N
    Interval{ 0x1D4C5, 0x1D505 }, // L&  [65] MATHEMATICAL SCRIPT SMALL P..MATHEMATICAL FRAKTUR CAPITAL B
    Interval{ 0x1D507, 0x1D50A }, // L&   [4] MATHEMATICAL FRAKTUR CAPITAL D..MATHEMATICAL FRAKTUR CAPITAL G
    Interval{ 0x1D50D, 0x1D514 }, // L&   [8] MATHEMATICAL FRAKTUR CAPITAL J..MATHEMATICAL FRAKTUR CAPITAL Q
    Interval{ 0x1D516, 0x1D51C }, // L&   [7] MATHEMATICAL FRAKTUR CAPITAL S..MATHEMATICAL FRAKTUR CAPITAL Y
    Interval{ 0x1D51E, 0x1D539 }, // L&  [28] MATHEMATICAL FRAKTUR SMALL A..MATHEMATICAL DOUBLE-STRUCK CAPITAL B
    Interval{ 0x1D53B, 0x1D53E }, // L&   [4] MATHEMATICAL DOUBLE-STRUCK CAPITAL D..MATHEMATICAL DOUBLE-STRUCK CAPITAL G
    Interval{ 0x1D540, 0x1D544 }, // L&   [5] MATHEMATICAL DOUBLE-STRUCK CAPITAL I..MATHEMATICAL DOUBLE-STRUCK CAPITAL M
    Interval{ 0x1D546, 0x1D546 }, // L&       MATHEMATICAL DOUBLE-STRUCK CAPITAL O
    Interval{ 0x1D54A, 0x1D550 }, // L&   [7] MATHEMATICAL DOUBLE-STRUCK CAPITAL S..MATHEMATICAL DOUBLE-STRUCK CAPITAL Y
    Interval{ 0x1D552, 0x1D6A5 }, // L& [340] MATHEMATICAL DOUBLE-STRUCK SMALL A..MATHEMATICAL ITALIC SMALL DOTLESS J
    Interval{ 0x1D6A8, 0x1D6C0 }, // L&  [25] MATHEMATICAL BOLD CAPITAL ALPHA..MATHEMATICAL BOLD CAPITAL OMEGA
    Interval{ 0x1D6C2, 0x1D6DA }, // L&  [25] MATHEMATICAL BOLD SMALL ALPHA..MATHEMATICAL BOLD SMALL OMEGA
    Interval{ 0x1D6DC, 0x1D6FA }, // L&  [31] MATHEMATICAL BOLD EPSILON SYMBOL..MATHEMATICAL ITALIC CAPITAL OMEGA
    Interval{ 0x1D6FC, 0x1D714 }, // L&  [25] MATHEMATICAL ITALIC SMALL ALPHA..MATHEMATICAL ITALIC SMALL OMEGA
    Interval{ 0x1D716, 0x1D734 }, // L&  [31] MATHEMATICAL ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D736, 0x1D74E }, // L&  [25] MATHEMATICAL BOLD ITALIC SMALL ALPHA..MATHEMATICAL BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D750, 0x1D76E }, // L&  [31] MATHEMATICAL BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGA
    Interval{ 0x1D770, 0x1D788 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD SMALL OMEGA
    Interval{ 0x1D78A, 0x1D7A8 }, // L&  [31] MATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D7AA, 0x1D7C2 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D7C4, 0x1D7CB }, // L&   [8] MATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD SMALL DIGAMMA
    Interval{ 0x1D7CE, 0x1D7FF }, // Nd  [50] MATHEMATICAL BOLD DIGIT ZERO..MATHEMATICAL MONOSPACE DIGIT NINE
    Interval{ 0x1DA00, 0x1DA36 }, // Mn  [55] SIGNWRITING HEAD RIM..SIGNWRITING AIR SUCKING IN
    Interval{ 0x1DA3B, 0x1DA6C }, // Mn  [50] SIGNWRITING MOUTH CLOSED NEUTRAL..SIGNWRITING EXCITEMENT
    Interval{ 0x1DA75, 0x1DA75 }, // Mn       SIGNWRITING UPPER BODY TILTING FROM HIP JOINTS
    Interval{ 0x1DA84, 0x1DA84 }, // Mn       SIGNWRITING LOCATION HEAD NECK
    Interval{ 0x1DA9B, 0x1DA9F }, // Mn   [5] SIGNWRITING FILL MODIFIER-2..SIGNWRITING FILL MODIFIER-6
    Interval{ 0x1DAA1, 0x1DAAF }, // Mn  [15] SIGNWRITING ROTATION MODIFIER-2..SIGNWRITING ROTATION MODIFIER-16
    Interval{ 0x1DF00, 0x1DF09 }, // L&  [10] LATIN SMALL LETTER FENG DIGRAPH WITH TRILL..LATIN SMALL LETTER T WITH HOOK AND RETROFLEX HOOK
    Interval{ 0x1DF0A, 0x1DF0A }, // Lo       LATIN LETTER RETROFLEX CLICK WITH RETROFLEX HOOK
    Interval{ 0x1DF0B, 0x1DF1E }, // L&  [20] LATIN SMALL LETTER ESH WITH DOUBLE BAR..LATIN SMALL LETTER S WITH CURL
    Interval{ 0x1DF25, 0x1DF2A }, // L&   [6] LATIN SMALL LETTER D WITH MID-HEIGHT LEFT HOOK..LATIN SMALL LETTER T WITH MID-HEIGHT LEFT HOOK
    Interval{ 0x1E000, 0x1E006 }, // Mn   [7] COMBINING GLAGOLITIC LETTER AZU..COMBINING GLAGOLITIC LETTER ZHIVETE
    Interval{ 0x1E008, 0x1E018 }, // Mn  [17] COMBINING GLAGOLITIC LETTER ZEMLJA..COMBINING GLAGOLITIC LETTER HERU
    Interval{ 0x1E01B, 0x1E021 }, // Mn   [7] COMBINING GLAGOLITIC LETTER SHTA..COMBINING GLAGOLITIC LETTER YATI
    Interval{ 0x1E023, 0x1E024 }, // Mn   [2] COMBINING GLAGOLITIC LETTER YU..COMBINING GLAGOLITIC LETTER SMALL YUS
    Interval{ 0x1E026, 0x1E02A }, // Mn   [5] COMBINING GLAGOLITIC LETTER YO..COMBINING GLAGOLITIC LETTER FITA
    Interval{ 0x1E030, 0x1E06D }, // Lm  [62] MODIFIER LETTER CYRILLIC SMALL A..MODIFIER LETTER CYRILLIC SMALL STRAIGHT U WITH STROKE
    Interval{ 0x1E08F, 0x1E08F }, // Mn       COMBINING CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
    Interval{ 0x1E100, 0x1E12C }, // Lo  [45] NYIAKENG PUACHUE HMONG LETTER MA..NYIAKENG PUACHUE HMONG LETTER W
    Interval{ 0x1E130, 0x1E136 }, // Mn   [7] NYIAKENG PUACHUE HMONG TONE-B..NYIAKENG PUACHUE HMONG TONE-D
    Interval{ 0x1E137, 0x1E13D }, // Lm   [7] NYIAKENG PUACHUE HMONG SIGN FOR PERSON..NYIAKENG PUACHUE HMONG SYLLABLE LENGTHENER
    Interval{ 0x1E140, 0x1E149 }, // Nd  [10] NYIAKENG PUACHUE HMONG DIGIT ZERO..NYIAKENG PUACHUE HMONG DIGIT NINE
    Interval{ 0x1E14E, 0x1E14E }, // Lo       NYIAKENG PUACHUE HMONG LOGOGRAM NYAJ
    Interval{ 0x1E290, 0x1E2AD }, // Lo  [30] TOTO LETTER PA..TOTO LETTER A
    Interval{ 0x1E2AE, 0x1E2AE }, // Mn       TOTO SIGN RISING TONE
    Interval{ 0x1E2C0, 0x1E2EB }, // Lo  [44] WANCHO LETTER AA..WANCHO LETTER YIH
    Interval{ 0x1E2EC, 0x1E2EF }, // Mn   [4] WANCHO TONE TUP..WANCHO TONE KOINI
    Interval{ 0x1E2F0, 0x1E2F9 }, // Nd  [10] WANCHO DIGIT ZERO..WANCHO DIGIT NINE
    Interval{ 0x1E4D0, 0x1E4EA }, // Lo  [27] NAG MUNDARI LETTER O..NAG MUNDARI LETTER ELL
    Interval{ 0x1E4EB, 0x1E4EB }, // Lm       NAG MUNDARI SIGN OJOD
    Interval{ 0x1E4EC, 0x1E4EF }, // Mn   [4] NAG MUNDARI SIGN MUHOR..NAG MUNDARI SIGN SUTUH
    Interval{ 0x1E4F0, 0x1E4F9 }, // Nd  [10] NAG MUNDARI DIGIT ZERO..NAG MUNDARI DIGIT NINE
    Interval{ 0x1E7E0, 0x1E7E6 }, // Lo   [7] ETHIOPIC SYLLABLE HHYA..ETHIOPIC SYLLABLE HHYO
    Interval{ 0x1E7E8, 0x1E7EB }, // Lo   [4] ETHIOPIC SYLLABLE GURAGE HHWA..ETHIOPIC SYLLABLE HHWE
    Interval{ 0x1E7ED, 0x1E7EE }, // Lo   [2] ETHIOPIC SYLLABLE GURAGE MWI..ETHIOPIC SYLLABLE GURAGE MWEE
    Interval{ 0x1E7F0, 0x1E7FE }, // Lo  [15] ETHIOPIC SYLLABLE GURAGE QWI..ETHIOPIC SYLLABLE GURAGE PWEE
    Interval{ 0x1E800, 0x1E8C4 }, // Lo [197] MENDE KIKAKUI SYLLABLE M001 KI..MENDE KIKAKUI SYLLABLE M060 NYON
    Interval{ 0x1E8D0, 0x1E8D6 }, // Mn   [7] MENDE KIKAKUI COMBINING NUMBER TEENS..MENDE KIKAKUI COMBINING NUMBER MILLIONS
    Interval{ 0x1E900, 0x1E943 }, // L&  [68] ADLAM CAPITAL LETTER ALIF..ADLAM SMALL LETTER SHA
    Interval{ 0x1E944, 0x1E94A }, // Mn   [7] ADLAM ALIF LENGTHENER..ADLAM NUKTA
    Interval{ 0x1E94B, 0x1E94B }, // Lm       ADLAM NASALIZATION MARK
    Interval{ 0x1E950, 0x1E959 }, // Nd  [10] ADLAM DIGIT ZERO..ADLAM DIGIT NINE
    Interval{ 0x1EE00, 0x1EE03 }, // Lo   [4] ARABIC MATHEMATICAL ALEF..ARABIC MATHEMATICAL DAL
    Interval{ 0x1EE05, 0x1EE1F }, // Lo  [27] ARABIC MATHEMATICAL WAW..ARABIC MATHEMATICAL DOTLESS QAF
    Interval{ 0x1EE21, 0x1EE22 }, // Lo   [2] ARABIC MATHEMATICAL INITIAL BEH..ARABIC MATHEMATICAL INITIAL JEEM
    Interval{ 0x1EE24, 0x1EE24 }, // Lo       ARABIC MATHEMATICAL INITIAL HEH
    Interval{ 0x1EE27, 0x1EE27 }, // Lo       ARABIC MATHEMATICAL INITIAL HAH
    Interval{ 0x1EE29, 0x1EE32 }, // Lo  [10] ARABIC MATHEMATICAL INITIAL YEH..ARABIC MATHEMATICAL INITIAL QAF
    Interval{ 0x1EE34, 0x1EE37 }, // Lo   [4] ARABIC MATHEMATICAL INITIAL SHEEN..ARABIC MATHEMATICAL INITIAL KHAH
    Interval{ 0x1EE39, 0x1EE39 }, // Lo       ARABIC MATHEMATICAL INITIAL DAD
    Interval{ 0x1EE3B, 0x1EE3B }, // Lo       ARABIC MATHEMATICAL INITIAL GHAIN
    Interval{ 0x1EE42, 0x1EE42 }, // Lo       ARABIC MATHEMATICAL TAILED JEEM
    Interval{ 0x1EE47, 0x1EE47 }, // Lo       ARABIC MATHEMATICAL TAILED HAH
    Interval{ 0x1EE49, 0x1EE49 }, // Lo       ARABIC MATHEMATICAL TAILED YEH
    Interval{ 0x1EE4B, 0x1EE4B }, // Lo       ARABIC MATHEMATICAL TAILED LAM
    Interval{ 0x1EE4D, 0x1EE4F }, // Lo   [3] ARABIC MATHEMATICAL TAILED NOON..ARABIC MATHEMATICAL TAILED AIN
    Interval{ 0x1EE51, 0x1EE52 }, // Lo   [2] ARABIC MATHEMATICAL TAILED SAD..ARABIC MATHEMATICAL TAILED QAF
    Interval{ 0x1EE54, 0x1EE54 }, // Lo       ARABIC MATHEMATICAL TAILED SHEEN
    Interval{ 0x1EE57, 0x1EE57 }, // Lo       ARABIC MATHEMATICAL TAILED KHAH
    Interval{ 0x1EE59, 0x1EE59 }, // Lo       ARABIC MATHEMATICAL TAILED DAD
    Interval{ 0x1EE5B, 0x1EE5B }, // Lo       ARABIC MATHEMATICAL TAILED GHAIN
    Interval{ 0x1EE5D, 0x1EE5D }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS NOON
    Interval{ 0x1EE5F, 0x1EE5F }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS QAF
    Interval{ 0x1EE61, 0x1EE62 }, // Lo   [2] ARABIC MATHEMATICAL STRETCHED BEH..ARABIC MATHEMATICAL STRETCHED JEEM
    Interval{ 0x1EE64, 0x1EE64 }, // Lo       ARABIC MATHEMATICAL STRETCHED HEH
    Interval{ 0x1EE67, 0x1EE6A }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED HAH..ARABIC MATHEMATICAL STRETCHED KAF
    Interval{ 0x1EE6C, 0x1EE72 }, // Lo   [7] ARABIC MATHEMATICAL STRETCHED MEEM..ARABIC MATHEMATICAL STRETCHED QAF
    Interval{ 0x1EE74, 0x1EE77 }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED SHEEN..ARABIC MATHEMATICAL STRETCHED KHAH
    Interval{ 0x1EE79, 0x1EE7C }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED DAD..ARABIC MATHEMATICAL STRETCHED DOTLESS BEH
    Interval{ 0x1EE7E, 0x1EE7E }, // Lo       ARABIC MATHEMATICAL STRETCHED DOTLESS FEH
    Interval{ 0x1EE80, 0x1EE89 }, // Lo  [10] ARABIC MATHEMATICAL LOOPED ALEF..ARABIC MATHEMATICAL LOOPED YEH
    Interval{ 0x1EE8B, 0x1EE9B }, // Lo  [17] ARABIC MATHEMATICAL LOOPED LAM..ARABIC MATHEMATICAL LOOPED GHAIN
    Interval{ 0x1EEA1, 0x1EEA3 }, // Lo   [3] ARABIC MATHEMATICAL DOUBLE-STRUCK BEH..ARABIC MATHEMATICAL DOUBLE-STRUCK DAL
    Interval{ 0x1EEA5, 0x1EEA9 }, // Lo   [5] ARABIC MATHEMATICAL DOUBLE-STRUCK WAW..ARABIC MATHEMATICAL DOUBLE-STRUCK YEH
    Interval{ 0x1EEAB, 0x1EEBB }, // Lo  [17] ARABIC MATHEMATICAL DOUBLE-STRUCK LAM..ARABIC MATHEMATICAL DOUBLE-STRUCK GHAIN
    Interval{ 0x1FBF0, 0x1FBF9 }, // Nd  [10] SEGMENTED DIGIT ZERO..SEGMENTED DIGIT NINE
    Interval{ 0x20000, 0x2A6DF }, // Lo [42720] CJK UNIFIED IDEOGRAPH-20000..CJK UNIFIED IDEOGRAPH-2A6DF
    Interval{ 0x2A700, 0x2B739 }, // Lo [4154] CJK UNIFIED IDEOGRAPH-2A700..CJK UNIFIED IDEOGRAPH-2B739
    Interval{ 0x2B740, 0x2B81D }, // Lo [222] CJK UNIFIED IDEOGRAPH-2B740..CJK UNIFIED IDEOGRAPH-2B81D
    Interval{ 0x2B820, 0x2CEA1 }, // Lo [5762] CJK UNIFIED IDEOGRAPH-2B820..CJK UNIFIED IDEOGRAPH-2CEA1
    Interval{ 0x2CEB0, 0x2EBE0 }, // Lo [7473] CJK UNIFIED IDEOGRAPH-2CEB0..CJK UNIFIED IDEOGRAPH-2EBE0
    Interval{ 0x2F800, 0x2FA1D }, // Lo [542] CJK COMPATIBILITY IDEOGRAPH-2F800..CJK COMPATIBILITY IDEOGRAPH-2FA1D
    Interval{ 0x30000, 0x3134A }, // Lo [4939] CJK UNIFIED IDEOGRAPH-30000..CJK UNIFIED IDEOGRAPH-3134A
    Interval{ 0x31350, 0x323AF }, // Lo [4192] CJK UNIFIED IDEOGRAPH-31350..CJK UNIFIED IDEOGRAPH-323AF
    Interval{ 0xE0100, 0xE01EF }, // Mn [240] VARIATION SELECTOR-17..VARIATION SELECTOR-256
}; // }}}
auto static const ID_Start = std::array<Interval, 739>{ // {{{
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00AA, 0x00AA }, // Lo       FEMININE ORDINAL INDICATOR
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00BA, 0x00BA }, // Lo       MASCULINE ORDINAL INDICATOR
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00F6 }, // L&  [31] LATIN CAPITAL LETTER O WITH STROKE..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x01BA }, // L& [195] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER EZH WITH TAIL
    Interval{ 0x01BB, 0x01BB }, // Lo       LATIN LETTER TWO WITH STROKE
    Interval{ 0x01BC, 0x01BF }, // L&   [4] LATIN CAPITAL LETTER TONE FIVE..LATIN LETTER WYNN
    Interval{ 0x01C0, 0x01C3 }, // Lo   [4] LATIN LETTER DENTAL CLICK..LATIN LETTER RETROFLEX CLICK
    Interval{ 0x01C4, 0x0293 }, // L& [208] LATIN CAPITAL LETTER DZ WITH CARON..LATIN SMALL LETTER EZH WITH CURL
    Interval{ 0x0294, 0x0294 }, // Lo       LATIN LETTER GLOTTAL STOP
    Interval{ 0x0295, 0x02AF }, // L&  [27] LATIN LETTER PHARYNGEAL VOICED FRICATIVE..LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    Interval{ 0x02B0, 0x02C1 }, // Lm  [18] MODIFIER LETTER SMALL H..MODIFIER LETTER REVERSED GLOTTAL STOP
    Interval{ 0x02C6, 0x02D1 }, // Lm  [12] MODIFIER LETTER CIRCUMFLEX ACCENT..MODIFIER LETTER HALF TRIANGULAR COLON
    Interval{ 0x02E0, 0x02E4 }, // Lm   [5] MODIFIER LETTER SMALL GAMMA..MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    Interval{ 0x02EC, 0x02EC }, // Lm       MODIFIER LETTER VOICING
    Interval{ 0x02EE, 0x02EE }, // Lm       MODIFIER LETTER DOUBLE APOSTROPHE
    Interval{ 0x0370, 0x0373 }, // L&   [4] GREEK CAPITAL LETTER HETA..GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0374, 0x0374 }, // Lm       GREEK NUMERAL SIGN
    Interval{ 0x0376, 0x0377 }, // L&   [2] GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA..GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037A, 0x037A }, // Lm       GREEK YPOGEGRAMMENI
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x03A1 }, // L&  [20] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03F5 }, // L&  [83] GREEK CAPITAL LETTER SIGMA..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F7, 0x0481 }, // L& [139] GREEK CAPITAL LETTER SHO..CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x048A, 0x052F }, // L& [166] CYRILLIC CAPITAL LETTER SHORT I WITH TAIL..CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x0559, 0x0559 }, // Lm       ARMENIAN MODIFIER LETTER LEFT HALF RING
    Interval{ 0x0560, 0x0588 }, // L&  [41] ARMENIAN SMALL LETTER TURNED AYB..ARMENIAN SMALL LETTER YI WITH STROKE
    Interval{ 0x05D0, 0x05EA }, // Lo  [27] HEBREW LETTER ALEF..HEBREW LETTER TAV
    Interval{ 0x05EF, 0x05F2 }, // Lo   [4] HEBREW YOD TRIANGLE..HEBREW LIGATURE YIDDISH DOUBLE YOD
    Interval{ 0x0620, 0x063F }, // Lo  [32] ARABIC LETTER KASHMIRI YEH..ARABIC LETTER FARSI YEH WITH THREE DOTS ABOVE
    Interval{ 0x0640, 0x0640 }, // Lm       ARABIC TATWEEL
    Interval{ 0x0641, 0x064A }, // Lo  [10] ARABIC LETTER FEH..ARABIC LETTER YEH
    Interval{ 0x066E, 0x066F }, // Lo   [2] ARABIC LETTER DOTLESS BEH..ARABIC LETTER DOTLESS QAF
    Interval{ 0x0671, 0x06D3 }, // Lo  [99] ARABIC LETTER ALEF WASLA..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
    Interval{ 0x06D5, 0x06D5 }, // Lo       ARABIC LETTER AE
    Interval{ 0x06E5, 0x06E6 }, // Lm   [2] ARABIC SMALL WAW..ARABIC SMALL YEH
    Interval{ 0x06EE, 0x06EF }, // Lo   [2] ARABIC LETTER DAL WITH INVERTED V..ARABIC LETTER REH WITH INVERTED V
    Interval{ 0x06FA, 0x06FC }, // Lo   [3] ARABIC LETTER SHEEN WITH DOT BELOW..ARABIC LETTER GHAIN WITH DOT BELOW
    Interval{ 0x06FF, 0x06FF }, // Lo       ARABIC LETTER HEH WITH INVERTED V
    Interval{ 0x0710, 0x0710 }, // Lo       SYRIAC LETTER ALAPH
    Interval{ 0x0712, 0x072F }, // Lo  [30] SYRIAC LETTER BETH..SYRIAC LETTER PERSIAN DHALATH
    Interval{ 0x074D, 0x07A5 }, // Lo  [89] SYRIAC LETTER SOGDIAN ZHAIN..THAANA LETTER WAAVU
    Interval{ 0x07B1, 0x07B1 }, // Lo       THAANA LETTER NAA
    Interval{ 0x07CA, 0x07EA }, // Lo  [33] NKO LETTER A..NKO LETTER JONA RA
    Interval{ 0x07F4, 0x07F5 }, // Lm   [2] NKO HIGH TONE APOSTROPHE..NKO LOW TONE APOSTROPHE
    Interval{ 0x07FA, 0x07FA }, // Lm       NKO LAJANYALAN
    Interval{ 0x0800, 0x0815 }, // Lo  [22] SAMARITAN LETTER ALAF..SAMARITAN LETTER TAAF
    Interval{ 0x081A, 0x081A }, // Lm       SAMARITAN MODIFIER LETTER EPENTHETIC YUT
    Interval{ 0x0824, 0x0824 }, // Lm       SAMARITAN MODIFIER LETTER SHORT A
    Interval{ 0x0828, 0x0828 }, // Lm       SAMARITAN MODIFIER LETTER I
    Interval{ 0x0840, 0x0858 }, // Lo  [25] MANDAIC LETTER HALQA..MANDAIC LETTER AIN
    Interval{ 0x0860, 0x086A }, // Lo  [11] SYRIAC LETTER MALAYALAM NGA..SYRIAC LETTER MALAYALAM SSA
    Interval{ 0x0870, 0x0887 }, // Lo  [24] ARABIC LETTER ALEF WITH ATTACHED FATHA..ARABIC BASELINE ROUND DOT
    Interval{ 0x0889, 0x088E }, // Lo   [6] ARABIC LETTER NOON WITH INVERTED SMALL V..ARABIC VERTICAL TAIL
    Interval{ 0x08A0, 0x08C8 }, // Lo  [41] ARABIC LETTER BEH WITH SMALL V BELOW..ARABIC LETTER GRAF
    Interval{ 0x08C9, 0x08C9 }, // Lm       ARABIC SMALL FARSI YEH
    Interval{ 0x0904, 0x0939 }, // Lo  [54] DEVANAGARI LETTER SHORT A..DEVANAGARI LETTER HA
    Interval{ 0x093D, 0x093D }, // Lo       DEVANAGARI SIGN AVAGRAHA
    Interval{ 0x0950, 0x0950 }, // Lo       DEVANAGARI OM
    Interval{ 0x0958, 0x0961 }, // Lo  [10] DEVANAGARI LETTER QA..DEVANAGARI LETTER VOCALIC LL
    Interval{ 0x0971, 0x0971 }, // Lm       DEVANAGARI SIGN HIGH SPACING DOT
    Interval{ 0x0972, 0x0980 }, // Lo  [15] DEVANAGARI LETTER CANDRA A..BENGALI ANJI
    Interval{ 0x0985, 0x098C }, // Lo   [8] BENGALI LETTER A..BENGALI LETTER VOCALIC L
    Interval{ 0x098F, 0x0990 }, // Lo   [2] BENGALI LETTER E..BENGALI LETTER AI
    Interval{ 0x0993, 0x09A8 }, // Lo  [22] BENGALI LETTER O..BENGALI LETTER NA
    Interval{ 0x09AA, 0x09B0 }, // Lo   [7] BENGALI LETTER PA..BENGALI LETTER RA
    Interval{ 0x09B2, 0x09B2 }, // Lo       BENGALI LETTER LA
    Interval{ 0x09B6, 0x09B9 }, // Lo   [4] BENGALI LETTER SHA..BENGALI LETTER HA
    Interval{ 0x09BD, 0x09BD }, // Lo       BENGALI SIGN AVAGRAHA
    Interval{ 0x09CE, 0x09CE }, // Lo       BENGALI LETTER KHANDA TA
    Interval{ 0x09DC, 0x09DD }, // Lo   [2] BENGALI LETTER RRA..BENGALI LETTER RHA
    Interval{ 0x09DF, 0x09E1 }, // Lo   [3] BENGALI LETTER YYA..BENGALI LETTER VOCALIC LL
    Interval{ 0x09F0, 0x09F1 }, // Lo   [2] BENGALI LETTER RA WITH MIDDLE DIAGONAL..BENGALI LETTER RA WITH LOWER DIAGONAL
    Interval{ 0x09FC, 0x09FC }, // Lo       BENGALI LETTER VEDIC ANUSVARA
    Interval{ 0x0A05, 0x0A0A }, // Lo   [6] GURMUKHI LETTER A..GURMUKHI LETTER UU
    Interval{ 0x0A0F, 0x0A10 }, // Lo   [2] GURMUKHI LETTER EE..GURMUKHI LETTER AI
    Interval{ 0x0A13, 0x0A28 }, // Lo  [22] GURMUKHI LETTER OO..GURMUKHI LETTER NA
    Interval{ 0x0A2A, 0x0A30 }, // Lo   [7] GURMUKHI LETTER PA..GURMUKHI LETTER RA
    Interval{ 0x0A32, 0x0A33 }, // Lo   [2] GURMUKHI LETTER LA..GURMUKHI LETTER LLA
    Interval{ 0x0A35, 0x0A36 }, // Lo   [2] GURMUKHI LETTER VA..GURMUKHI LETTER SHA
    Interval{ 0x0A38, 0x0A39 }, // Lo   [2] GURMUKHI LETTER SA..GURMUKHI LETTER HA
    Interval{ 0x0A59, 0x0A5C }, // Lo   [4] GURMUKHI LETTER KHHA..GURMUKHI LETTER RRA
    Interval{ 0x0A5E, 0x0A5E }, // Lo       GURMUKHI LETTER FA
    Interval{ 0x0A72, 0x0A74 }, // Lo   [3] GURMUKHI IRI..GURMUKHI EK ONKAR
    Interval{ 0x0A85, 0x0A8D }, // Lo   [9] GUJARATI LETTER A..GUJARATI VOWEL CANDRA E
    Interval{ 0x0A8F, 0x0A91 }, // Lo   [3] GUJARATI LETTER E..GUJARATI VOWEL CANDRA O
    Interval{ 0x0A93, 0x0AA8 }, // Lo  [22] GUJARATI LETTER O..GUJARATI LETTER NA
    Interval{ 0x0AAA, 0x0AB0 }, // Lo   [7] GUJARATI LETTER PA..GUJARATI LETTER RA
    Interval{ 0x0AB2, 0x0AB3 }, // Lo   [2] GUJARATI LETTER LA..GUJARATI LETTER LLA
    Interval{ 0x0AB5, 0x0AB9 }, // Lo   [5] GUJARATI LETTER VA..GUJARATI LETTER HA
    Interval{ 0x0ABD, 0x0ABD }, // Lo       GUJARATI SIGN AVAGRAHA
    Interval{ 0x0AD0, 0x0AD0 }, // Lo       GUJARATI OM
    Interval{ 0x0AE0, 0x0AE1 }, // Lo   [2] GUJARATI LETTER VOCALIC RR..GUJARATI LETTER VOCALIC LL
    Interval{ 0x0AF9, 0x0AF9 }, // Lo       GUJARATI LETTER ZHA
    Interval{ 0x0B05, 0x0B0C }, // Lo   [8] ORIYA LETTER A..ORIYA LETTER VOCALIC L
    Interval{ 0x0B0F, 0x0B10 }, // Lo   [2] ORIYA LETTER E..ORIYA LETTER AI
    Interval{ 0x0B13, 0x0B28 }, // Lo  [22] ORIYA LETTER O..ORIYA LETTER NA
    Interval{ 0x0B2A, 0x0B30 }, // Lo   [7] ORIYA LETTER PA..ORIYA LETTER RA
    Interval{ 0x0B32, 0x0B33 }, // Lo   [2] ORIYA LETTER LA..ORIYA LETTER LLA
    Interval{ 0x0B35, 0x0B39 }, // Lo   [5] ORIYA LETTER VA..ORIYA LETTER HA
    Interval{ 0x0B3D, 0x0B3D }, // Lo       ORIYA SIGN AVAGRAHA
    Interval{ 0x0B5C, 0x0B5D }, // Lo   [2] ORIYA LETTER RRA..ORIYA LETTER RHA
    Interval{ 0x0B5F, 0x0B61 }, // Lo   [3] ORIYA LETTER YYA..ORIYA LETTER VOCALIC LL
    Interval{ 0x0B71, 0x0B71 }, // Lo       ORIYA LETTER WA
    Interval{ 0x0B83, 0x0B83 }, // Lo       TAMIL SIGN VISARGA
    Interval{ 0x0B85, 0x0B8A }, // Lo   [6] TAMIL LETTER A..TAMIL LETTER UU
    Interval{ 0x0B8E, 0x0B90 }, // Lo   [3] TAMIL LETTER E..TAMIL LETTER AI
    Interval{ 0x0B92, 0x0B95 }, // Lo   [4] TAMIL LETTER O..TAMIL LETTER KA
    Interval{ 0x0B99, 0x0B9A }, // Lo   [2] TAMIL LETTER NGA..TAMIL LETTER CA
    Interval{ 0x0B9C, 0x0B9C }, // Lo       TAMIL LETTER JA
    Interval{ 0x0B9E, 0x0B9F }, // Lo   [2] TAMIL LETTER NYA..TAMIL LETTER TTA
    Interval{ 0x0BA3, 0x0BA4 }, // Lo   [2] TAMIL LETTER NNA..TAMIL LETTER TA
    Interval{ 0x0BA8, 0x0BAA }, // Lo   [3] TAMIL LETTER NA..TAMIL LETTER PA
    Interval{ 0x0BAE, 0x0BB9 }, // Lo  [12] TAMIL LETTER MA..TAMIL LETTER HA
    Interval{ 0x0BD0, 0x0BD0 }, // Lo       TAMIL OM
    Interval{ 0x0C05, 0x0C0C }, // Lo   [8] TELUGU LETTER A..TELUGU LETTER VOCALIC L
    Interval{ 0x0C0E, 0x0C10 }, // Lo   [3] TELUGU LETTER E..TELUGU LETTER AI
    Interval{ 0x0C12, 0x0C28 }, // Lo  [23] TELUGU LETTER O..TELUGU LETTER NA
    Interval{ 0x0C2A, 0x0C39 }, // Lo  [16] TELUGU LETTER PA..TELUGU LETTER HA
    Interval{ 0x0C3D, 0x0C3D }, // Lo       TELUGU SIGN AVAGRAHA
    Interval{ 0x0C58, 0x0C5A }, // Lo   [3] TELUGU LETTER TSA..TELUGU LETTER RRRA
    Interval{ 0x0C5D, 0x0C5D }, // Lo       TELUGU LETTER NAKAARA POLLU
    Interval{ 0x0C60, 0x0C61 }, // Lo   [2] TELUGU LETTER VOCALIC RR..TELUGU LETTER VOCALIC LL
    Interval{ 0x0C80, 0x0C80 }, // Lo       KANNADA SIGN SPACING CANDRABINDU
    Interval{ 0x0C85, 0x0C8C }, // Lo   [8] KANNADA LETTER A..KANNADA LETTER VOCALIC L
    Interval{ 0x0C8E, 0x0C90 }, // Lo   [3] KANNADA LETTER E..KANNADA LETTER AI
    Interval{ 0x0C92, 0x0CA8 }, // Lo  [23] KANNADA LETTER O..KANNADA LETTER NA
    Interval{ 0x0CAA, 0x0CB3 }, // Lo  [10] KANNADA LETTER PA..KANNADA LETTER LLA
    Interval{ 0x0CB5, 0x0CB9 }, // Lo   [5] KANNADA LETTER VA..KANNADA LETTER HA
    Interval{ 0x0CBD, 0x0CBD }, // Lo       KANNADA SIGN AVAGRAHA
    Interval{ 0x0CDD, 0x0CDE }, // Lo   [2] KANNADA LETTER NAKAARA POLLU..KANNADA LETTER FA
    Interval{ 0x0CE0, 0x0CE1 }, // Lo   [2] KANNADA LETTER VOCALIC RR..KANNADA LETTER VOCALIC LL
    Interval{ 0x0CF1, 0x0CF2 }, // Lo   [2] KANNADA SIGN JIHVAMULIYA..KANNADA SIGN UPADHMANIYA
    Interval{ 0x0D04, 0x0D0C }, // Lo   [9] MALAYALAM LETTER VEDIC ANUSVARA..MALAYALAM LETTER VOCALIC L
    Interval{ 0x0D0E, 0x0D10 }, // Lo   [3] MALAYALAM LETTER E..MALAYALAM LETTER AI
    Interval{ 0x0D12, 0x0D3A }, // Lo  [41] MALAYALAM LETTER O..MALAYALAM LETTER TTTA
    Interval{ 0x0D3D, 0x0D3D }, // Lo       MALAYALAM SIGN AVAGRAHA
    Interval{ 0x0D4E, 0x0D4E }, // Lo       MALAYALAM LETTER DOT REPH
    Interval{ 0x0D54, 0x0D56 }, // Lo   [3] MALAYALAM LETTER CHILLU M..MALAYALAM LETTER CHILLU LLL
    Interval{ 0x0D5F, 0x0D61 }, // Lo   [3] MALAYALAM LETTER ARCHAIC II..MALAYALAM LETTER VOCALIC LL
    Interval{ 0x0D7A, 0x0D7F }, // Lo   [6] MALAYALAM LETTER CHILLU NN..MALAYALAM LETTER CHILLU K
    Interval{ 0x0D85, 0x0D96 }, // Lo  [18] SINHALA LETTER AYANNA..SINHALA LETTER AUYANNA
    Interval{ 0x0D9A, 0x0DB1 }, // Lo  [24] SINHALA LETTER ALPAPRAANA KAYANNA..SINHALA LETTER DANTAJA NAYANNA
    Interval{ 0x0DB3, 0x0DBB }, // Lo   [9] SINHALA LETTER SANYAKA DAYANNA..SINHALA LETTER RAYANNA
    Interval{ 0x0DBD, 0x0DBD }, // Lo       SINHALA LETTER DANTAJA LAYANNA
    Interval{ 0x0DC0, 0x0DC6 }, // Lo   [7] SINHALA LETTER VAYANNA..SINHALA LETTER FAYANNA
    Interval{ 0x0E01, 0x0E30 }, // Lo  [48] THAI CHARACTER KO KAI..THAI CHARACTER SARA A
    Interval{ 0x0E32, 0x0E33 }, // Lo   [2] THAI CHARACTER SARA AA..THAI CHARACTER SARA AM
    Interval{ 0x0E40, 0x0E45 }, // Lo   [6] THAI CHARACTER SARA E..THAI CHARACTER LAKKHANGYAO
    Interval{ 0x0E46, 0x0E46 }, // Lm       THAI CHARACTER MAIYAMOK
    Interval{ 0x0E81, 0x0E82 }, // Lo   [2] LAO LETTER KO..LAO LETTER KHO SUNG
    Interval{ 0x0E84, 0x0E84 }, // Lo       LAO LETTER KHO TAM
    Interval{ 0x0E86, 0x0E8A }, // Lo   [5] LAO LETTER PALI GHA..LAO LETTER SO TAM
    Interval{ 0x0E8C, 0x0EA3 }, // Lo  [24] LAO LETTER PALI JHA..LAO LETTER LO LING
    Interval{ 0x0EA5, 0x0EA5 }, // Lo       LAO LETTER LO LOOT
    Interval{ 0x0EA7, 0x0EB0 }, // Lo  [10] LAO LETTER WO..LAO VOWEL SIGN A
    Interval{ 0x0EB2, 0x0EB3 }, // Lo   [2] LAO VOWEL SIGN AA..LAO VOWEL SIGN AM
    Interval{ 0x0EBD, 0x0EBD }, // Lo       LAO SEMIVOWEL SIGN NYO
    Interval{ 0x0EC0, 0x0EC4 }, // Lo   [5] LAO VOWEL SIGN E..LAO VOWEL SIGN AI
    Interval{ 0x0EC6, 0x0EC6 }, // Lm       LAO KO LA
    Interval{ 0x0EDC, 0x0EDF }, // Lo   [4] LAO HO NO..LAO LETTER KHMU NYO
    Interval{ 0x0F00, 0x0F00 }, // Lo       TIBETAN SYLLABLE OM
    Interval{ 0x0F40, 0x0F47 }, // Lo   [8] TIBETAN LETTER KA..TIBETAN LETTER JA
    Interval{ 0x0F49, 0x0F6C }, // Lo  [36] TIBETAN LETTER NYA..TIBETAN LETTER RRA
    Interval{ 0x0F88, 0x0F8C }, // Lo   [5] TIBETAN SIGN LCE TSA CAN..TIBETAN SIGN INVERTED MCHU CAN
    Interval{ 0x1000, 0x102A }, // Lo  [43] MYANMAR LETTER KA..MYANMAR LETTER AU
    Interval{ 0x103F, 0x103F }, // Lo       MYANMAR LETTER GREAT SA
    Interval{ 0x1050, 0x1055 }, // Lo   [6] MYANMAR LETTER SHA..MYANMAR LETTER VOCALIC LL
    Interval{ 0x105A, 0x105D }, // Lo   [4] MYANMAR LETTER MON NGA..MYANMAR LETTER MON BBE
    Interval{ 0x1061, 0x1061 }, // Lo       MYANMAR LETTER SGAW KAREN SHA
    Interval{ 0x1065, 0x1066 }, // Lo   [2] MYANMAR LETTER WESTERN PWO KAREN THA..MYANMAR LETTER WESTERN PWO KAREN PWA
    Interval{ 0x106E, 0x1070 }, // Lo   [3] MYANMAR LETTER EASTERN PWO KAREN NNA..MYANMAR LETTER EASTERN PWO KAREN GHWA
    Interval{ 0x1075, 0x1081 }, // Lo  [13] MYANMAR LETTER SHAN KA..MYANMAR LETTER SHAN HA
    Interval{ 0x108E, 0x108E }, // Lo       MYANMAR LETTER RUMAI PALAUNG FA
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FC, 0x10FC }, // Lm       MODIFIER LETTER GEORGIAN NAR
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x1100, 0x1248 }, // Lo [329] HANGUL CHOSEONG KIYEOK..ETHIOPIC SYLLABLE QWA
    Interval{ 0x124A, 0x124D }, // Lo   [4] ETHIOPIC SYLLABLE QWI..ETHIOPIC SYLLABLE QWE
    Interval{ 0x1250, 0x1256 }, // Lo   [7] ETHIOPIC SYLLABLE QHA..ETHIOPIC SYLLABLE QHO
    Interval{ 0x1258, 0x1258 }, // Lo       ETHIOPIC SYLLABLE QHWA
    Interval{ 0x125A, 0x125D }, // Lo   [4] ETHIOPIC SYLLABLE QHWI..ETHIOPIC SYLLABLE QHWE
    Interval{ 0x1260, 0x1288 }, // Lo  [41] ETHIOPIC SYLLABLE BA..ETHIOPIC SYLLABLE XWA
    Interval{ 0x128A, 0x128D }, // Lo   [4] ETHIOPIC SYLLABLE XWI..ETHIOPIC SYLLABLE XWE
    Interval{ 0x1290, 0x12B0 }, // Lo  [33] ETHIOPIC SYLLABLE NA..ETHIOPIC SYLLABLE KWA
    Interval{ 0x12B2, 0x12B5 }, // Lo   [4] ETHIOPIC SYLLABLE KWI..ETHIOPIC SYLLABLE KWE
    Interval{ 0x12B8, 0x12BE }, // Lo   [7] ETHIOPIC SYLLABLE KXA..ETHIOPIC SYLLABLE KXO
    Interval{ 0x12C0, 0x12C0 }, // Lo       ETHIOPIC SYLLABLE KXWA
    Interval{ 0x12C2, 0x12C5 }, // Lo   [4] ETHIOPIC SYLLABLE KXWI..ETHIOPIC SYLLABLE KXWE
    Interval{ 0x12C8, 0x12D6 }, // Lo  [15] ETHIOPIC SYLLABLE WA..ETHIOPIC SYLLABLE PHARYNGEAL O
    Interval{ 0x12D8, 0x1310 }, // Lo  [57] ETHIOPIC SYLLABLE ZA..ETHIOPIC SYLLABLE GWA
    Interval{ 0x1312, 0x1315 }, // Lo   [4] ETHIOPIC SYLLABLE GWI..ETHIOPIC SYLLABLE GWE
    Interval{ 0x1318, 0x135A }, // Lo  [67] ETHIOPIC SYLLABLE GGA..ETHIOPIC SYLLABLE FYA
    Interval{ 0x1380, 0x138F }, // Lo  [16] ETHIOPIC SYLLABLE SEBATBEIT MWA..ETHIOPIC SYLLABLE PWE
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1401, 0x166C }, // Lo [620] CANADIAN SYLLABICS E..CANADIAN SYLLABICS CARRIER TTSA
    Interval{ 0x166F, 0x167F }, // Lo  [17] CANADIAN SYLLABICS QAI..CANADIAN SYLLABICS BLACKFOOT W
    Interval{ 0x1681, 0x169A }, // Lo  [26] OGHAM LETTER BEITH..OGHAM LETTER PEITH
    Interval{ 0x16A0, 0x16EA }, // Lo  [75] RUNIC LETTER FEHU FEOH FE F..RUNIC LETTER X
    Interval{ 0x16EE, 0x16F0 }, // Nl   [3] RUNIC ARLAUG SYMBOL..RUNIC BELGTHOR SYMBOL
    Interval{ 0x16F1, 0x16F8 }, // Lo   [8] RUNIC LETTER K..RUNIC LETTER FRANKS CASKET AESC
    Interval{ 0x1700, 0x1711 }, // Lo  [18] TAGALOG LETTER A..TAGALOG LETTER HA
    Interval{ 0x171F, 0x1731 }, // Lo  [19] TAGALOG LETTER ARCHAIC RA..HANUNOO LETTER HA
    Interval{ 0x1740, 0x1751 }, // Lo  [18] BUHID LETTER A..BUHID LETTER HA
    Interval{ 0x1760, 0x176C }, // Lo  [13] TAGBANWA LETTER A..TAGBANWA LETTER YA
    Interval{ 0x176E, 0x1770 }, // Lo   [3] TAGBANWA LETTER LA..TAGBANWA LETTER SA
    Interval{ 0x1780, 0x17B3 }, // Lo  [52] KHMER LETTER KA..KHMER INDEPENDENT VOWEL QAU
    Interval{ 0x17D7, 0x17D7 }, // Lm       KHMER SIGN LEK TOO
    Interval{ 0x17DC, 0x17DC }, // Lo       KHMER SIGN AVAKRAHASANYA
    Interval{ 0x1820, 0x1842 }, // Lo  [35] MONGOLIAN LETTER A..MONGOLIAN LETTER CHI
    Interval{ 0x1843, 0x1843 }, // Lm       MONGOLIAN LETTER TODO LONG VOWEL SIGN
    Interval{ 0x1844, 0x1878 }, // Lo  [53] MONGOLIAN LETTER TODO E..MONGOLIAN LETTER CHA WITH TWO DOTS
    Interval{ 0x1880, 0x1884 }, // Lo   [5] MONGOLIAN LETTER ALI GALI ANUSVARA ONE..MONGOLIAN LETTER ALI GALI INVERTED UBADAMA
    Interval{ 0x1885, 0x1886 }, // Mn   [2] MONGOLIAN LETTER ALI GALI BALUDA..MONGOLIAN LETTER ALI GALI THREE BALUDA
    Interval{ 0x1887, 0x18A8 }, // Lo  [34] MONGOLIAN LETTER ALI GALI A..MONGOLIAN LETTER MANCHU ALI GALI BHA
    Interval{ 0x18AA, 0x18AA }, // Lo       MONGOLIAN LETTER MANCHU ALI GALI LHA
    Interval{ 0x18B0, 0x18F5 }, // Lo  [70] CANADIAN SYLLABICS OY..CANADIAN SYLLABICS CARRIER DENTAL S
    Interval{ 0x1900, 0x191E }, // Lo  [31] LIMBU VOWEL-CARRIER LETTER..LIMBU LETTER TRA
    Interval{ 0x1950, 0x196D }, // Lo  [30] TAI LE LETTER KA..TAI LE LETTER AI
    Interval{ 0x1970, 0x1974 }, // Lo   [5] TAI LE LETTER TONE-2..TAI LE LETTER TONE-6
    Interval{ 0x1980, 0x19AB }, // Lo  [44] NEW TAI LUE LETTER HIGH QA..NEW TAI LUE LETTER LOW SUA
    Interval{ 0x19B0, 0x19C9 }, // Lo  [26] NEW TAI LUE VOWEL SIGN VOWEL SHORTENER..NEW TAI LUE TONE MARK-2
    Interval{ 0x1A00, 0x1A16 }, // Lo  [23] BUGINESE LETTER KA..BUGINESE LETTER HA
    Interval{ 0x1A20, 0x1A54 }, // Lo  [53] TAI THAM LETTER HIGH KA..TAI THAM LETTER GREAT SA
    Interval{ 0x1AA7, 0x1AA7 }, // Lm       TAI THAM SIGN MAI YAMOK
    Interval{ 0x1B05, 0x1B33 }, // Lo  [47] BALINESE LETTER AKARA..BALINESE LETTER HA
    Interval{ 0x1B45, 0x1B4C }, // Lo   [8] BALINESE LETTER KAF SASAK..BALINESE LETTER ARCHAIC JNYA
    Interval{ 0x1B83, 0x1BA0 }, // Lo  [30] SUNDANESE LETTER A..SUNDANESE LETTER HA
    Interval{ 0x1BAE, 0x1BAF }, // Lo   [2] SUNDANESE LETTER KHA..SUNDANESE LETTER SYA
    Interval{ 0x1BBA, 0x1BE5 }, // Lo  [44] SUNDANESE AVAGRAHA..BATAK LETTER U
    Interval{ 0x1C00, 0x1C23 }, // Lo  [36] LEPCHA LETTER KA..LEPCHA LETTER A
    Interval{ 0x1C4D, 0x1C4F }, // Lo   [3] LEPCHA LETTER TTA..LEPCHA LETTER DDA
    Interval{ 0x1C5A, 0x1C77 }, // Lo  [30] OL CHIKI LETTER LA..OL CHIKI LETTER OH
    Interval{ 0x1C78, 0x1C7D }, // Lm   [6] OL CHIKI MU TTUDDAG..OL CHIKI AHAD
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1CE9, 0x1CEC }, // Lo   [4] VEDIC SIGN ANUSVARA ANTARGOMUKHA..VEDIC SIGN ANUSVARA VAMAGOMUKHA WITH TAIL
    Interval{ 0x1CEE, 0x1CF3 }, // Lo   [6] VEDIC SIGN HEXIFORM LONG ANUSVARA..VEDIC SIGN ROTATED ARDHAVISARGA
    Interval{ 0x1CF5, 0x1CF6 }, // Lo   [2] VEDIC SIGN JIHVAMULIYA..VEDIC SIGN UPADHMANIYA
    Interval{ 0x1CFA, 0x1CFA }, // Lo       VEDIC SIGN DOUBLE ANUSVARA ANTARGOMUKHA
    Interval{ 0x1D00, 0x1D2B }, // L&  [44] LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTER SMALL CAPITAL EL
    Interval{ 0x1D2C, 0x1D6A }, // Lm  [63] MODIFIER LETTER CAPITAL A..GREEK SUBSCRIPT SMALL LETTER CHI
    Interval{ 0x1D6B, 0x1D77 }, // L&  [13] LATIN SMALL LETTER UE..LATIN SMALL LETTER TURNED G
    Interval{ 0x1D78, 0x1D78 }, // Lm       MODIFIER LETTER CYRILLIC EN
    Interval{ 0x1D79, 0x1D9A }, // L&  [34] LATIN SMALL LETTER INSULAR G..LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
    Interval{ 0x1D9B, 0x1DBF }, // Lm  [37] MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER LETTER SMALL THETA
    Interval{ 0x1E00, 0x1F15 }, // L& [278] LATIN CAPITAL LETTER A WITH RING BELOW..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F45 }, // L&  [38] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F7D }, // L&  [31] GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1FB4 }, // L&  [53] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FBC }, // L&   [7] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FCC }, // L&   [7] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FDB }, // L&   [6] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE0, 0x1FEC }, // L&  [13] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FFC }, // L&   [7] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x2071, 0x2071 }, // Lm       SUPERSCRIPT LATIN SMALL LETTER I
    Interval{ 0x207F, 0x207F }, // Lm       SUPERSCRIPT LATIN SMALL LETTER N
    Interval{ 0x2090, 0x209C }, // Lm  [13] LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCRIPT SMALL LETTER T
    Interval{ 0x2102, 0x2102 }, // L&       DOUBLE-STRUCK CAPITAL C
    Interval{ 0x2107, 0x2107 }, // L&       EULER CONSTANT
    Interval{ 0x210A, 0x2113 }, // L&  [10] SCRIPT SMALL G..SCRIPT SMALL L
    Interval{ 0x2115, 0x2115 }, // L&       DOUBLE-STRUCK CAPITAL N
    Interval{ 0x2118, 0x2118 }, // Sm       SCRIPT CAPITAL P
    Interval{ 0x2119, 0x211D }, // L&   [5] DOUBLE-STRUCK CAPITAL P..DOUBLE-STRUCK CAPITAL R
    Interval{ 0x2124, 0x2124 }, // L&       DOUBLE-STRUCK CAPITAL Z
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x2128, 0x2128 }, // L&       BLACK-LETTER CAPITAL Z
    Interval{ 0x212A, 0x212D }, // L&   [4] KELVIN SIGN..BLACK-LETTER CAPITAL C
    Interval{ 0x212E, 0x212E }, // So       ESTIMATED SYMBOL
    Interval{ 0x212F, 0x2134 }, // L&   [6] SCRIPT SMALL E..SCRIPT SMALL O
    Interval{ 0x2135, 0x2138 }, // Lo   [4] ALEF SYMBOL..DALET SYMBOL
    Interval{ 0x2139, 0x2139 }, // L&       INFORMATION SOURCE
    Interval{ 0x213C, 0x213F }, // L&   [4] DOUBLE-STRUCK SMALL PI..DOUBLE-STRUCK CAPITAL PI
    Interval{ 0x2145, 0x2149 }, // L&   [5] DOUBLE-STRUCK ITALIC CAPITAL D..DOUBLE-STRUCK ITALIC SMALL J
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2160, 0x2182 }, // Nl  [35] ROMAN NUMERAL ONE..ROMAN NUMERAL TEN THOUSAND
    Interval{ 0x2183, 0x2184 }, // L&   [2] ROMAN NUMERAL REVERSED ONE HUNDRED..LATIN SMALL LETTER REVERSED C
    Interval{ 0x2185, 0x2188 }, // Nl   [4] ROMAN NUMERAL SIX LATE FORM..ROMAN NUMERAL ONE HUNDRED THOUSAND
    Interval{ 0x2C00, 0x2C7B }, // L& [124] GLAGOLITIC CAPITAL LETTER AZU..LATIN LETTER SMALL CAPITAL TURNED E
    Interval{ 0x2C7C, 0x2C7D }, // Lm   [2] LATIN SUBSCRIPT SMALL LETTER J..MODIFIER LETTER CAPITAL V
    Interval{ 0x2C7E, 0x2CE4 }, // L& [103] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC SYMBOL KAI
    Interval{ 0x2CEB, 0x2CEE }, // L&   [4] COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI..COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF2, 0x2CF3 }, // L&   [2] COPTIC CAPITAL LETTER BOHAIRIC KHEI..COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0x2D30, 0x2D67 }, // Lo  [56] TIFINAGH LETTER YA..TIFINAGH LETTER YO
    Interval{ 0x2D6F, 0x2D6F }, // Lm       TIFINAGH MODIFIER LETTER LABIALIZATION MARK
    Interval{ 0x2D80, 0x2D96 }, // Lo  [23] ETHIOPIC SYLLABLE LOA..ETHIOPIC SYLLABLE GGWE
    Interval{ 0x2DA0, 0x2DA6 }, // Lo   [7] ETHIOPIC SYLLABLE SSA..ETHIOPIC SYLLABLE SSO
    Interval{ 0x2DA8, 0x2DAE }, // Lo   [7] ETHIOPIC SYLLABLE CCA..ETHIOPIC SYLLABLE CCO
    Interval{ 0x2DB0, 0x2DB6 }, // Lo   [7] ETHIOPIC SYLLABLE ZZA..ETHIOPIC SYLLABLE ZZO
    Interval{ 0x2DB8, 0x2DBE }, // Lo   [7] ETHIOPIC SYLLABLE CCHA..ETHIOPIC SYLLABLE CCHO
    Interval{ 0x2DC0, 0x2DC6 }, // Lo   [7] ETHIOPIC SYLLABLE QYA..ETHIOPIC SYLLABLE QYO
    Interval{ 0x2DC8, 0x2DCE }, // Lo   [7] ETHIOPIC SYLLABLE KYA..ETHIOPIC SYLLABLE KYO
    Interval{ 0x2DD0, 0x2DD6 }, // Lo   [7] ETHIOPIC SYLLABLE XYA..ETHIOPIC SYLLABLE XYO
    Interval{ 0x2DD8, 0x2DDE }, // Lo   [7] ETHIOPIC SYLLABLE GYA..ETHIOPIC SYLLABLE GYO
    Interval{ 0x3005, 0x3005 }, // Lm       IDEOGRAPHIC ITERATION MARK
    Interval{ 0x3006, 0x3006 }, // Lo       IDEOGRAPHIC CLOSING MARK
    Interval{ 0x3007, 0x3007 }, // Nl       IDEOGRAPHIC NUMBER ZERO
    Interval{ 0x3021, 0x3029 }, // Nl   [9] HANGZHOU NUMERAL ONE..HANGZHOU NUMERAL NINE
    Interval{ 0x3031, 0x3035 }, // Lm   [5] VERTICAL KANA REPEAT MARK..VERTICAL KANA REPEAT MARK LOWER HALF
    Interval{ 0x3038, 0x303A }, // Nl   [3] HANGZHOU NUMERAL TEN..HANGZHOU NUMERAL THIRTY
    Interval{ 0x303B, 0x303B }, // Lm       VERTICAL IDEOGRAPHIC ITERATION MARK
    Interval{ 0x303C, 0x303C }, // Lo       MASU MARK
    Interval{ 0x3041, 0x3096 }, // Lo  [86] HIRAGANA LETTER SMALL A..HIRAGANA LETTER SMALL KE
    Interval{ 0x309B, 0x309C }, // Sk   [2] KATAKANA-HIRAGANA VOICED SOUND MARK..KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    Interval{ 0x309D, 0x309E }, // Lm   [2] HIRAGANA ITERATION MARK..HIRAGANA VOICED ITERATION MARK
    Interval{ 0x309F, 0x309F }, // Lo       HIRAGANA DIGRAPH YORI
    Interval{ 0x30A1, 0x30FA }, // Lo  [90] KATAKANA LETTER SMALL A..KATAKANA LETTER VO
    Interval{ 0x30FC, 0x30FE }, // Lm   [3] KATAKANA-HIRAGANA PROLONGED SOUND MARK..KATAKANA VOICED ITERATION MARK
    Interval{ 0x30FF, 0x30FF }, // Lo       KATAKANA DIGRAPH KOTO
    Interval{ 0x3105, 0x312F }, // Lo  [43] BOPOMOFO LETTER B..BOPOMOFO LETTER NN
    Interval{ 0x3131, 0x318E }, // Lo  [94] HANGUL LETTER KIYEOK..HANGUL LETTER ARAEAE
    Interval{ 0x31A0, 0x31BF }, // Lo  [32] BOPOMOFO LETTER BU..BOPOMOFO LETTER AH
    Interval{ 0x31F0, 0x31FF }, // Lo  [16] KATAKANA LETTER SMALL KU..KATAKANA LETTER SMALL RO
    Interval{ 0x3400, 0x4DBF }, // Lo [6592] CJK UNIFIED IDEOGRAPH-3400..CJK UNIFIED IDEOGRAPH-4DBF
    Interval{ 0x4E00, 0xA014 }, // Lo [21013] CJK UNIFIED IDEOGRAPH-4E00..YI SYLLABLE E
    Interval{ 0xA015, 0xA015 }, // Lm       YI SYLLABLE WU
    Interval{ 0xA016, 0xA48C }, // Lo [1143] YI SYLLABLE BIT..YI SYLLABLE YYR
    Interval{ 0xA4D0, 0xA4F7 }, // Lo  [40] LISU LETTER BA..LISU LETTER OE
    Interval{ 0xA4F8, 0xA4FD }, // Lm   [6] LISU LETTER TONE MYA TI..LISU LETTER TONE MYA JEU
    Interval{ 0xA500, 0xA60B }, // Lo [268] VAI SYLLABLE EE..VAI SYLLABLE NG
    Interval{ 0xA60C, 0xA60C }, // Lm       VAI SYLLABLE LENGTHENER
    Interval{ 0xA610, 0xA61F }, // Lo  [16] VAI SYLLABLE NDOLE FA..VAI SYMBOL JONG
    Interval{ 0xA62A, 0xA62B }, // Lo   [2] VAI SYLLABLE NDOLE MA..VAI SYLLABLE NDOLE DO
    Interval{ 0xA640, 0xA66D }, // L&  [46] CYRILLIC CAPITAL LETTER ZEMLYA..CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA66E, 0xA66E }, // Lo       CYRILLIC LETTER MULTIOCULAR O
    Interval{ 0xA67F, 0xA67F }, // Lm       CYRILLIC PAYEROK
    Interval{ 0xA680, 0xA69B }, // L&  [28] CYRILLIC CAPITAL LETTER DWE..CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA69C, 0xA69D }, // Lm   [2] MODIFIER LETTER CYRILLIC HARD SIGN..MODIFIER LETTER CYRILLIC SOFT SIGN
    Interval{ 0xA6A0, 0xA6E5 }, // Lo  [70] BAMUM LETTER A..BAMUM LETTER KI
    Interval{ 0xA6E6, 0xA6EF }, // Nl  [10] BAMUM LETTER MO..BAMUM LETTER KOGHOM
    Interval{ 0xA717, 0xA71F }, // Lm   [9] MODIFIER LETTER DOT VERTICAL BAR..MODIFIER LETTER LOW INVERTED EXCLAMATION MARK
    Interval{ 0xA722, 0xA76F }, // L&  [78] LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF..LATIN SMALL LETTER CON
    Interval{ 0xA770, 0xA770 }, // Lm       MODIFIER LETTER US
    Interval{ 0xA771, 0xA787 }, // L&  [23] LATIN SMALL LETTER DUM..LATIN SMALL LETTER INSULAR T
    Interval{ 0xA788, 0xA788 }, // Lm       MODIFIER LETTER LOW CIRCUMFLEX ACCENT
    Interval{ 0xA78B, 0xA78E }, // L&   [4] LATIN CAPITAL LETTER SALTILLO..LATIN SMALL LETTER L WITH RETROFLEX HOOK AND BELT
    Interval{ 0xA78F, 0xA78F }, // Lo       LATIN LETTER SINOLOGICAL DOT
    Interval{ 0xA790, 0xA7CA }, // L&  [59] LATIN CAPITAL LETTER N WITH DESCENDER..LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D1 }, // L&   [2] LATIN CAPITAL LETTER CLOSED INSULAR G..LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D3, 0xA7D3 }, // L&       LATIN SMALL LETTER DOUBLE THORN
    Interval{ 0xA7D5, 0xA7D9 }, // L&   [5] LATIN SMALL LETTER DOUBLE WYNN..LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F2, 0xA7F4 }, // Lm   [3] MODIFIER LETTER CAPITAL C..MODIFIER LETTER CAPITAL Q
    Interval{ 0xA7F5, 0xA7F6 }, // L&   [2] LATIN CAPITAL LETTER REVERSED HALF H..LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xA7F7, 0xA7F7 }, // Lo       LATIN EPIGRAPHIC LETTER SIDEWAYS I
    Interval{ 0xA7F8, 0xA7F9 }, // Lm   [2] MODIFIER LETTER CAPITAL H WITH STROKE..MODIFIER LETTER SMALL LIGATURE OE
    Interval{ 0xA7FA, 0xA7FA }, // L&       LATIN LETTER SMALL CAPITAL TURNED M
    Interval{ 0xA7FB, 0xA801 }, // Lo   [7] LATIN EPIGRAPHIC LETTER REVERSED F..SYLOTI NAGRI LETTER I
    Interval{ 0xA803, 0xA805 }, // Lo   [3] SYLOTI NAGRI LETTER U..SYLOTI NAGRI LETTER O
    Interval{ 0xA807, 0xA80A }, // Lo   [4] SYLOTI NAGRI LETTER KO..SYLOTI NAGRI LETTER GHO
    Interval{ 0xA80C, 0xA822 }, // Lo  [23] SYLOTI NAGRI LETTER CO..SYLOTI NAGRI LETTER HO
    Interval{ 0xA840, 0xA873 }, // Lo  [52] PHAGS-PA LETTER KA..PHAGS-PA LETTER CANDRABINDU
    Interval{ 0xA882, 0xA8B3 }, // Lo  [50] SAURASHTRA LETTER A..SAURASHTRA LETTER LLA
    Interval{ 0xA8F2, 0xA8F7 }, // Lo   [6] DEVANAGARI SIGN SPACING CANDRABINDU..DEVANAGARI SIGN CANDRABINDU AVAGRAHA
    Interval{ 0xA8FB, 0xA8FB }, // Lo       DEVANAGARI HEADSTROKE
    Interval{ 0xA8FD, 0xA8FE }, // Lo   [2] DEVANAGARI JAIN OM..DEVANAGARI LETTER AY
    Interval{ 0xA90A, 0xA925 }, // Lo  [28] KAYAH LI LETTER KA..KAYAH LI LETTER OO
    Interval{ 0xA930, 0xA946 }, // Lo  [23] REJANG LETTER KA..REJANG LETTER A
    Interval{ 0xA960, 0xA97C }, // Lo  [29] HANGUL CHOSEONG TIKEUT-MIEUM..HANGUL CHOSEONG SSANGYEORINHIEUH
    Interval{ 0xA984, 0xA9B2 }, // Lo  [47] JAVANESE LETTER A..JAVANESE LETTER HA
    Interval{ 0xA9CF, 0xA9CF }, // Lm       JAVANESE PANGRANGKEP
    Interval{ 0xA9E0, 0xA9E4 }, // Lo   [5] MYANMAR LETTER SHAN GHA..MYANMAR LETTER SHAN BHA
    Interval{ 0xA9E6, 0xA9E6 }, // Lm       MYANMAR MODIFIER LETTER SHAN REDUPLICATION
    Interval{ 0xA9E7, 0xA9EF }, // Lo   [9] MYANMAR LETTER TAI LAING NYA..MYANMAR LETTER TAI LAING NNA
    Interval{ 0xA9FA, 0xA9FE }, // Lo   [5] MYANMAR LETTER TAI LAING LLA..MYANMAR LETTER TAI LAING BHA
    Interval{ 0xAA00, 0xAA28 }, // Lo  [41] CHAM LETTER A..CHAM LETTER HA
    Interval{ 0xAA40, 0xAA42 }, // Lo   [3] CHAM LETTER FINAL K..CHAM LETTER FINAL NG
    Interval{ 0xAA44, 0xAA4B }, // Lo   [8] CHAM LETTER FINAL CH..CHAM LETTER FINAL SS
    Interval{ 0xAA60, 0xAA6F }, // Lo  [16] MYANMAR LETTER KHAMTI GA..MYANMAR LETTER KHAMTI FA
    Interval{ 0xAA70, 0xAA70 }, // Lm       MYANMAR MODIFIER LETTER KHAMTI REDUPLICATION
    Interval{ 0xAA71, 0xAA76 }, // Lo   [6] MYANMAR LETTER KHAMTI XA..MYANMAR LOGOGRAM KHAMTI HM
    Interval{ 0xAA7A, 0xAA7A }, // Lo       MYANMAR LETTER AITON RA
    Interval{ 0xAA7E, 0xAAAF }, // Lo  [50] MYANMAR LETTER SHWE PALAUNG CHA..TAI VIET LETTER HIGH O
    Interval{ 0xAAB1, 0xAAB1 }, // Lo       TAI VIET VOWEL AA
    Interval{ 0xAAB5, 0xAAB6 }, // Lo   [2] TAI VIET VOWEL E..TAI VIET VOWEL O
    Interval{ 0xAAB9, 0xAABD }, // Lo   [5] TAI VIET VOWEL UEA..TAI VIET VOWEL AN
    Interval{ 0xAAC0, 0xAAC0 }, // Lo       TAI VIET TONE MAI NUENG
    Interval{ 0xAAC2, 0xAAC2 }, // Lo       TAI VIET TONE MAI SONG
    Interval{ 0xAADB, 0xAADC }, // Lo   [2] TAI VIET SYMBOL KON..TAI VIET SYMBOL NUENG
    Interval{ 0xAADD, 0xAADD }, // Lm       TAI VIET SYMBOL SAM
    Interval{ 0xAAE0, 0xAAEA }, // Lo  [11] MEETEI MAYEK LETTER E..MEETEI MAYEK LETTER SSA
    Interval{ 0xAAF2, 0xAAF2 }, // Lo       MEETEI MAYEK ANJI
    Interval{ 0xAAF3, 0xAAF4 }, // Lm   [2] MEETEI MAYEK SYLLABLE REPETITION MARK..MEETEI MAYEK WORD REPETITION MARK
    Interval{ 0xAB01, 0xAB06 }, // Lo   [6] ETHIOPIC SYLLABLE TTHU..ETHIOPIC SYLLABLE TTHO
    Interval{ 0xAB09, 0xAB0E }, // Lo   [6] ETHIOPIC SYLLABLE DDHU..ETHIOPIC SYLLABLE DDHO
    Interval{ 0xAB11, 0xAB16 }, // Lo   [6] ETHIOPIC SYLLABLE DZU..ETHIOPIC SYLLABLE DZO
    Interval{ 0xAB20, 0xAB26 }, // Lo   [7] ETHIOPIC SYLLABLE CCHHA..ETHIOPIC SYLLABLE CCHHO
    Interval{ 0xAB28, 0xAB2E }, // Lo   [7] ETHIOPIC SYLLABLE BBA..ETHIOPIC SYLLABLE BBO
    Interval{ 0xAB30, 0xAB5A }, // L&  [43] LATIN SMALL LETTER BARRED ALPHA..LATIN SMALL LETTER Y WITH SHORT RIGHT LEG
    Interval{ 0xAB5C, 0xAB5F }, // Lm   [4] MODIFIER LETTER SMALL HENG..MODIFIER LETTER SMALL U WITH LEFT HOOK
    Interval{ 0xAB60, 0xAB68 }, // L&   [9] LATIN SMALL LETTER SAKHA YAT..LATIN SMALL LETTER TURNED R WITH MIDDLE TILDE
    Interval{ 0xAB69, 0xAB69 }, // Lm       MODIFIER LETTER SMALL TURNED W
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xABC0, 0xABE2 }, // Lo  [35] MEETEI MAYEK LETTER KOK..MEETEI MAYEK LETTER I LONSUM
    Interval{ 0xAC00, 0xD7A3 }, // Lo [11172] HANGUL SYLLABLE GA..HANGUL SYLLABLE HIH
    Interval{ 0xD7B0, 0xD7C6 }, // Lo  [23] HANGUL JUNGSEONG O-YEO..HANGUL JUNGSEONG ARAEA-E
    Interval{ 0xD7CB, 0xD7FB }, // Lo  [49] HANGUL JONGSEONG NIEUN-RIEUL..HANGUL JONGSEONG PHIEUPH-THIEUTH
    Interval{ 0xF900, 0xFA6D }, // Lo [366] CJK COMPATIBILITY IDEOGRAPH-F900..CJK COMPATIBILITY IDEOGRAPH-FA6D
    Interval{ 0xFA70, 0xFAD9 }, // Lo [106] CJK COMPATIBILITY IDEOGRAPH-FA70..CJK COMPATIBILITY IDEOGRAPH-FAD9
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFB1D, 0xFB1D }, // Lo       HEBREW LETTER YOD WITH HIRIQ
    Interval{ 0xFB1F, 0xFB28 }, // Lo  [10] HEBREW LIGATURE YIDDISH YOD YOD PATAH..HEBREW LETTER WIDE TAV
    Interval{ 0xFB2A, 0xFB36 }, // Lo  [13] HEBREW LETTER SHIN WITH SHIN DOT..HEBREW LETTER ZAYIN WITH DAGESH
    Interval{ 0xFB38, 0xFB3C }, // Lo   [5] HEBREW LETTER TET WITH DAGESH..HEBREW LETTER LAMED WITH DAGESH
    Interval{ 0xFB3E, 0xFB3E }, // Lo       HEBREW LETTER MEM WITH DAGESH
    Interval{ 0xFB40, 0xFB41 }, // Lo   [2] HEBREW LETTER NUN WITH DAGESH..HEBREW LETTER SAMEKH WITH DAGESH
    Interval{ 0xFB43, 0xFB44 }, // Lo   [2] HEBREW LETTER FINAL PE WITH DAGESH..HEBREW LETTER PE WITH DAGESH
    Interval{ 0xFB46, 0xFBB1 }, // Lo [108] HEBREW LETTER TSADI WITH DAGESH..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
    Interval{ 0xFBD3, 0xFD3D }, // Lo [363] ARABIC LETTER NG ISOLATED FORM..ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
    Interval{ 0xFD50, 0xFD8F }, // Lo  [64] ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM..ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
    Interval{ 0xFD92, 0xFDC7 }, // Lo  [54] ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM..ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
    Interval{ 0xFDF0, 0xFDFB }, // Lo  [12] ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM..ARABIC LIGATURE JALLAJALALOUHOU
    Interval{ 0xFE70, 0xFE74 }, // Lo   [5] ARABIC FATHATAN ISOLATED FORM..ARABIC KASRATAN ISOLATED FORM
    Interval{ 0xFE76, 0xFEFC }, // Lo [135] ARABIC FATHA ISOLATED FORM..ARABIC LIGATURE LAM WITH ALEF FINAL FORM
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0xFF66, 0xFF6F }, // Lo  [10] HALFWIDTH KATAKANA LETTER WO..HALFWIDTH KATAKANA LETTER SMALL TU
    Interval{ 0xFF70, 0xFF70 }, // Lm       HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
    Interval{ 0xFF71, 0xFF9D }, // Lo  [45] HALFWIDTH KATAKANA LETTER A..HALFWIDTH KATAKANA LETTER N
    Interval{ 0xFF9E, 0xFF9F }, // Lm   [2] HALFWIDTH KATAKANA VOICED SOUND MARK..HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
    Interval{ 0xFFA0, 0xFFBE }, // Lo  [31] HALFWIDTH HANGUL FILLER..HALFWIDTH HANGUL LETTER HIEUH
    Interval{ 0xFFC2, 0xFFC7 }, // Lo   [6] HALFWIDTH HANGUL LETTER A..HALFWIDTH HANGUL LETTER E
    Interval{ 0xFFCA, 0xFFCF }, // Lo   [6] HALFWIDTH HANGUL LETTER YEO..HALFWIDTH HANGUL LETTER OE
    Interval{ 0xFFD2, 0xFFD7 }, // Lo   [6] HALFWIDTH HANGUL LETTER YO..HALFWIDTH HANGUL LETTER YU
    Interval{ 0xFFDA, 0xFFDC }, // Lo   [3] HALFWIDTH HANGUL LETTER EU..HALFWIDTH HANGUL LETTER I
    Interval{ 0x10000, 0x1000B }, // Lo  [12] LINEAR B SYLLABLE B008 A..LINEAR B SYLLABLE B046 JE
    Interval{ 0x1000D, 0x10026 }, // Lo  [26] LINEAR B SYLLABLE B036 JO..LINEAR B SYLLABLE B032 QO
    Interval{ 0x10028, 0x1003A }, // Lo  [19] LINEAR B SYLLABLE B060 RA..LINEAR B SYLLABLE B042 WO
    Interval{ 0x1003C, 0x1003D }, // Lo   [2] LINEAR B SYLLABLE B017 ZA..LINEAR B SYLLABLE B074 ZE
    Interval{ 0x1003F, 0x1004D }, // Lo  [15] LINEAR B SYLLABLE B020 ZO..LINEAR B SYLLABLE B091 TWO
    Interval{ 0x10050, 0x1005D }, // Lo  [14] LINEAR B SYMBOL B018..LINEAR B SYMBOL B089
    Interval{ 0x10080, 0x100FA }, // Lo [123] LINEAR B IDEOGRAM B100 MAN..LINEAR B IDEOGRAM VESSEL B305
    Interval{ 0x10140, 0x10174 }, // Nl  [53] GREEK ACROPHONIC ATTIC ONE QUARTER..GREEK ACROPHONIC STRATIAN FIFTY MNAS
    Interval{ 0x10280, 0x1029C }, // Lo  [29] LYCIAN LETTER A..LYCIAN LETTER X
    Interval{ 0x102A0, 0x102D0 }, // Lo  [49] CARIAN LETTER A..CARIAN LETTER UUU3
    Interval{ 0x10300, 0x1031F }, // Lo  [32] OLD ITALIC LETTER A..OLD ITALIC LETTER ESS
    Interval{ 0x1032D, 0x10340 }, // Lo  [20] OLD ITALIC LETTER YE..GOTHIC LETTER PAIRTHRA
    Interval{ 0x10341, 0x10341 }, // Nl       GOTHIC LETTER NINETY
    Interval{ 0x10342, 0x10349 }, // Lo   [8] GOTHIC LETTER RAIDA..GOTHIC LETTER OTHAL
    Interval{ 0x1034A, 0x1034A }, // Nl       GOTHIC LETTER NINE HUNDRED
    Interval{ 0x10350, 0x10375 }, // Lo  [38] OLD PERMIC LETTER AN..OLD PERMIC LETTER IA
    Interval{ 0x10380, 0x1039D }, // Lo  [30] UGARITIC LETTER ALPA..UGARITIC LETTER SSU
    Interval{ 0x103A0, 0x103C3 }, // Lo  [36] OLD PERSIAN SIGN A..OLD PERSIAN SIGN HA
    Interval{ 0x103C8, 0x103CF }, // Lo   [8] OLD PERSIAN SIGN AURAMAZDAA..OLD PERSIAN SIGN BUUMISH
    Interval{ 0x103D1, 0x103D5 }, // Nl   [5] OLD PERSIAN NUMBER ONE..OLD PERSIAN NUMBER HUNDRED
    Interval{ 0x10400, 0x1044F }, // L&  [80] DESERET CAPITAL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x10450, 0x1049D }, // Lo  [78] SHAVIAN LETTER PEEP..OSMANYA LETTER OO
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10500, 0x10527 }, // Lo  [40] ELBASAN LETTER A..ELBASAN LETTER KHE
    Interval{ 0x10530, 0x10563 }, // Lo  [52] CAUCASIAN ALBANIAN LETTER ALT..CAUCASIAN ALBANIAN LETTER KIW
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10600, 0x10736 }, // Lo [311] LINEAR A SIGN AB001..LINEAR A SIGN A664
    Interval{ 0x10740, 0x10755 }, // Lo  [22] LINEAR A SIGN A701 A..LINEAR A SIGN A732 JE
    Interval{ 0x10760, 0x10767 }, // Lo   [8] LINEAR A SIGN A800..LINEAR A SIGN A807
    Interval{ 0x10780, 0x10785 }, // Lm   [6] MODIFIER LETTER SMALL CAPITAL AA..MODIFIER LETTER SMALL B WITH HOOK
    Interval{ 0x10787, 0x107B0 }, // Lm  [42] MODIFIER LETTER SMALL DZ DIGRAPH..MODIFIER LETTER SMALL V WITH RIGHT HOOK
    Interval{ 0x107B2, 0x107BA }, // Lm   [9] MODIFIER LETTER SMALL CAPITAL Y..MODIFIER LETTER SMALL S WITH CURL
    Interval{ 0x10800, 0x10805 }, // Lo   [6] CYPRIOT SYLLABLE A..CYPRIOT SYLLABLE JA
    Interval{ 0x10808, 0x10808 }, // Lo       CYPRIOT SYLLABLE JO
    Interval{ 0x1080A, 0x10835 }, // Lo  [44] CYPRIOT SYLLABLE KA..CYPRIOT SYLLABLE WO
    Interval{ 0x10837, 0x10838 }, // Lo   [2] CYPRIOT SYLLABLE XA..CYPRIOT SYLLABLE XE
    Interval{ 0x1083C, 0x1083C }, // Lo       CYPRIOT SYLLABLE ZA
    Interval{ 0x1083F, 0x10855 }, // Lo  [23] CYPRIOT SYLLABLE ZO..IMPERIAL ARAMAIC LETTER TAW
    Interval{ 0x10860, 0x10876 }, // Lo  [23] PALMYRENE LETTER ALEPH..PALMYRENE LETTER TAW
    Interval{ 0x10880, 0x1089E }, // Lo  [31] NABATAEAN LETTER FINAL ALEPH..NABATAEAN LETTER TAW
    Interval{ 0x108E0, 0x108F2 }, // Lo  [19] HATRAN LETTER ALEPH..HATRAN LETTER QOPH
    Interval{ 0x108F4, 0x108F5 }, // Lo   [2] HATRAN LETTER SHIN..HATRAN LETTER TAW
    Interval{ 0x10900, 0x10915 }, // Lo  [22] PHOENICIAN LETTER ALF..PHOENICIAN LETTER TAU
    Interval{ 0x10920, 0x10939 }, // Lo  [26] LYDIAN LETTER A..LYDIAN LETTER C
    Interval{ 0x10980, 0x109B7 }, // Lo  [56] MEROITIC HIEROGLYPHIC LETTER A..MEROITIC CURSIVE LETTER DA
    Interval{ 0x109BE, 0x109BF }, // Lo   [2] MEROITIC CURSIVE LOGOGRAM RMT..MEROITIC CURSIVE LOGOGRAM IMN
    Interval{ 0x10A00, 0x10A00 }, // Lo       KHAROSHTHI LETTER A
    Interval{ 0x10A10, 0x10A13 }, // Lo   [4] KHAROSHTHI LETTER KA..KHAROSHTHI LETTER GHA
    Interval{ 0x10A15, 0x10A17 }, // Lo   [3] KHAROSHTHI LETTER CA..KHAROSHTHI LETTER JA
    Interval{ 0x10A19, 0x10A35 }, // Lo  [29] KHAROSHTHI LETTER NYA..KHAROSHTHI LETTER VHA
    Interval{ 0x10A60, 0x10A7C }, // Lo  [29] OLD SOUTH ARABIAN LETTER HE..OLD SOUTH ARABIAN LETTER THETH
    Interval{ 0x10A80, 0x10A9C }, // Lo  [29] OLD NORTH ARABIAN LETTER HEH..OLD NORTH ARABIAN LETTER ZAH
    Interval{ 0x10AC0, 0x10AC7 }, // Lo   [8] MANICHAEAN LETTER ALEPH..MANICHAEAN LETTER WAW
    Interval{ 0x10AC9, 0x10AE4 }, // Lo  [28] MANICHAEAN LETTER ZAYIN..MANICHAEAN LETTER TAW
    Interval{ 0x10B00, 0x10B35 }, // Lo  [54] AVESTAN LETTER A..AVESTAN LETTER HE
    Interval{ 0x10B40, 0x10B55 }, // Lo  [22] INSCRIPTIONAL PARTHIAN LETTER ALEPH..INSCRIPTIONAL PARTHIAN LETTER TAW
    Interval{ 0x10B60, 0x10B72 }, // Lo  [19] INSCRIPTIONAL PAHLAVI LETTER ALEPH..INSCRIPTIONAL PAHLAVI LETTER TAW
    Interval{ 0x10B80, 0x10B91 }, // Lo  [18] PSALTER PAHLAVI LETTER ALEPH..PSALTER PAHLAVI LETTER TAW
    Interval{ 0x10C00, 0x10C48 }, // Lo  [73] OLD TURKIC LETTER ORKHON A..OLD TURKIC LETTER ORKHON BASH
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x10D00, 0x10D23 }, // Lo  [36] HANIFI ROHINGYA LETTER A..HANIFI ROHINGYA MARK NA KHONNA
    Interval{ 0x10E80, 0x10EA9 }, // Lo  [42] YEZIDI LETTER ELIF..YEZIDI LETTER ET
    Interval{ 0x10EB0, 0x10EB1 }, // Lo   [2] YEZIDI LETTER LAM WITH DOT ABOVE..YEZIDI LETTER YOT WITH CIRCUMFLEX ABOVE
    Interval{ 0x10F00, 0x10F1C }, // Lo  [29] OLD SOGDIAN LETTER ALEPH..OLD SOGDIAN LETTER FINAL TAW WITH VERTICAL TAIL
    Interval{ 0x10F27, 0x10F27 }, // Lo       OLD SOGDIAN LIGATURE AYIN-DALETH
    Interval{ 0x10F30, 0x10F45 }, // Lo  [22] SOGDIAN LETTER ALEPH..SOGDIAN INDEPENDENT SHIN
    Interval{ 0x10F70, 0x10F81 }, // Lo  [18] OLD UYGHUR LETTER ALEPH..OLD UYGHUR LETTER LESH
    Interval{ 0x10FB0, 0x10FC4 }, // Lo  [21] CHORASMIAN LETTER ALEPH..CHORASMIAN LETTER TAW
    Interval{ 0x10FE0, 0x10FF6 }, // Lo  [23] ELYMAIC LETTER ALEPH..ELYMAIC LIGATURE ZAYIN-YODH
    Interval{ 0x11003, 0x11037 }, // Lo  [53] BRAHMI SIGN JIHVAMULIYA..BRAHMI LETTER OLD TAMIL NNNA
    Interval{ 0x11071, 0x11072 }, // Lo   [2] BRAHMI LETTER OLD TAMIL SHORT E..BRAHMI LETTER OLD TAMIL SHORT O
    Interval{ 0x11075, 0x11075 }, // Lo       BRAHMI LETTER OLD TAMIL LLA
    Interval{ 0x11083, 0x110AF }, // Lo  [45] KAITHI LETTER A..KAITHI LETTER HA
    Interval{ 0x110D0, 0x110E8 }, // Lo  [25] SORA SOMPENG LETTER SAH..SORA SOMPENG LETTER MAE
    Interval{ 0x11103, 0x11126 }, // Lo  [36] CHAKMA LETTER AA..CHAKMA LETTER HAA
    Interval{ 0x11144, 0x11144 }, // Lo       CHAKMA LETTER LHAA
    Interval{ 0x11147, 0x11147 }, // Lo       CHAKMA LETTER VAA
    Interval{ 0x11150, 0x11172 }, // Lo  [35] MAHAJANI LETTER A..MAHAJANI LETTER RRA
    Interval{ 0x11176, 0x11176 }, // Lo       MAHAJANI LIGATURE SHRI
    Interval{ 0x11183, 0x111B2 }, // Lo  [48] SHARADA LETTER A..SHARADA LETTER HA
    Interval{ 0x111C1, 0x111C4 }, // Lo   [4] SHARADA SIGN AVAGRAHA..SHARADA OM
    Interval{ 0x111DA, 0x111DA }, // Lo       SHARADA EKAM
    Interval{ 0x111DC, 0x111DC }, // Lo       SHARADA HEADSTROKE
    Interval{ 0x11200, 0x11211 }, // Lo  [18] KHOJKI LETTER A..KHOJKI LETTER JJA
    Interval{ 0x11213, 0x1122B }, // Lo  [25] KHOJKI LETTER NYA..KHOJKI LETTER LLA
    Interval{ 0x1123F, 0x11240 }, // Lo   [2] KHOJKI LETTER QA..KHOJKI LETTER SHORT I
    Interval{ 0x11280, 0x11286 }, // Lo   [7] MULTANI LETTER A..MULTANI LETTER GA
    Interval{ 0x11288, 0x11288 }, // Lo       MULTANI LETTER GHA
    Interval{ 0x1128A, 0x1128D }, // Lo   [4] MULTANI LETTER CA..MULTANI LETTER JJA
    Interval{ 0x1128F, 0x1129D }, // Lo  [15] MULTANI LETTER NYA..MULTANI LETTER BA
    Interval{ 0x1129F, 0x112A8 }, // Lo  [10] MULTANI LETTER BHA..MULTANI LETTER RHA
    Interval{ 0x112B0, 0x112DE }, // Lo  [47] KHUDAWADI LETTER A..KHUDAWADI LETTER HA
    Interval{ 0x11305, 0x1130C }, // Lo   [8] GRANTHA LETTER A..GRANTHA LETTER VOCALIC L
    Interval{ 0x1130F, 0x11310 }, // Lo   [2] GRANTHA LETTER EE..GRANTHA LETTER AI
    Interval{ 0x11313, 0x11328 }, // Lo  [22] GRANTHA LETTER OO..GRANTHA LETTER NA
    Interval{ 0x1132A, 0x11330 }, // Lo   [7] GRANTHA LETTER PA..GRANTHA LETTER RA
    Interval{ 0x11332, 0x11333 }, // Lo   [2] GRANTHA LETTER LA..GRANTHA LETTER LLA
    Interval{ 0x11335, 0x11339 }, // Lo   [5] GRANTHA LETTER VA..GRANTHA LETTER HA
    Interval{ 0x1133D, 0x1133D }, // Lo       GRANTHA SIGN AVAGRAHA
    Interval{ 0x11350, 0x11350 }, // Lo       GRANTHA OM
    Interval{ 0x1135D, 0x11361 }, // Lo   [5] GRANTHA SIGN PLUTA..GRANTHA LETTER VOCALIC LL
    Interval{ 0x11400, 0x11434 }, // Lo  [53] NEWA LETTER A..NEWA LETTER HA
    Interval{ 0x11447, 0x1144A }, // Lo   [4] NEWA SIGN AVAGRAHA..NEWA SIDDHI
    Interval{ 0x1145F, 0x11461 }, // Lo   [3] NEWA LETTER VEDIC ANUSVARA..NEWA SIGN UPADHMANIYA
    Interval{ 0x11480, 0x114AF }, // Lo  [48] TIRHUTA ANJI..TIRHUTA LETTER HA
    Interval{ 0x114C4, 0x114C5 }, // Lo   [2] TIRHUTA SIGN AVAGRAHA..TIRHUTA GVANG
    Interval{ 0x114C7, 0x114C7 }, // Lo       TIRHUTA OM
    Interval{ 0x11580, 0x115AE }, // Lo  [47] SIDDHAM LETTER A..SIDDHAM LETTER HA
    Interval{ 0x115D8, 0x115DB }, // Lo   [4] SIDDHAM LETTER THREE-CIRCLE ALTERNATE I..SIDDHAM LETTER ALTERNATE U
    Interval{ 0x11600, 0x1162F }, // Lo  [48] MODI LETTER A..MODI LETTER LLA
    Interval{ 0x11644, 0x11644 }, // Lo       MODI SIGN HUVA
    Interval{ 0x11680, 0x116AA }, // Lo  [43] TAKRI LETTER A..TAKRI LETTER RRA
    Interval{ 0x116B8, 0x116B8 }, // Lo       TAKRI LETTER ARCHAIC KHA
    Interval{ 0x11700, 0x1171A }, // Lo  [27] AHOM LETTER KA..AHOM LETTER ALTERNATE BA
    Interval{ 0x11740, 0x11746 }, // Lo   [7] AHOM LETTER CA..AHOM LETTER LLA
    Interval{ 0x11800, 0x1182B }, // Lo  [44] DOGRA LETTER A..DOGRA LETTER RRA
    Interval{ 0x118A0, 0x118DF }, // L&  [64] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x118FF, 0x11906 }, // Lo   [8] WARANG CITI OM..DIVES AKURU LETTER E
    Interval{ 0x11909, 0x11909 }, // Lo       DIVES AKURU LETTER O
    Interval{ 0x1190C, 0x11913 }, // Lo   [8] DIVES AKURU LETTER KA..DIVES AKURU LETTER JA
    Interval{ 0x11915, 0x11916 }, // Lo   [2] DIVES AKURU LETTER NYA..DIVES AKURU LETTER TTA
    Interval{ 0x11918, 0x1192F }, // Lo  [24] DIVES AKURU LETTER DDA..DIVES AKURU LETTER ZA
    Interval{ 0x1193F, 0x1193F }, // Lo       DIVES AKURU PREFIXED NASAL SIGN
    Interval{ 0x11941, 0x11941 }, // Lo       DIVES AKURU INITIAL RA
    Interval{ 0x119A0, 0x119A7 }, // Lo   [8] NANDINAGARI LETTER A..NANDINAGARI LETTER VOCALIC RR
    Interval{ 0x119AA, 0x119D0 }, // Lo  [39] NANDINAGARI LETTER E..NANDINAGARI LETTER RRA
    Interval{ 0x119E1, 0x119E1 }, // Lo       NANDINAGARI SIGN AVAGRAHA
    Interval{ 0x119E3, 0x119E3 }, // Lo       NANDINAGARI HEADSTROKE
    Interval{ 0x11A00, 0x11A00 }, // Lo       ZANABAZAR SQUARE LETTER A
    Interval{ 0x11A0B, 0x11A32 }, // Lo  [40] ZANABAZAR SQUARE LETTER KA..ZANABAZAR SQUARE LETTER KSSA
    Interval{ 0x11A3A, 0x11A3A }, // Lo       ZANABAZAR SQUARE CLUSTER-INITIAL LETTER RA
    Interval{ 0x11A50, 0x11A50 }, // Lo       SOYOMBO LETTER A
    Interval{ 0x11A5C, 0x11A89 }, // Lo  [46] SOYOMBO LETTER KA..SOYOMBO CLUSTER-INITIAL LETTER SA
    Interval{ 0x11A9D, 0x11A9D }, // Lo       SOYOMBO MARK PLUTA
    Interval{ 0x11AB0, 0x11AF8 }, // Lo  [73] CANADIAN SYLLABICS NATTILIK HI..PAU CIN HAU GLOTTAL STOP FINAL
    Interval{ 0x11C00, 0x11C08 }, // Lo   [9] BHAIKSUKI LETTER A..BHAIKSUKI LETTER VOCALIC L
    Interval{ 0x11C0A, 0x11C2E }, // Lo  [37] BHAIKSUKI LETTER E..BHAIKSUKI LETTER HA
    Interval{ 0x11C40, 0x11C40 }, // Lo       BHAIKSUKI SIGN AVAGRAHA
    Interval{ 0x11C72, 0x11C8F }, // Lo  [30] MARCHEN LETTER KA..MARCHEN LETTER A
    Interval{ 0x11D00, 0x11D06 }, // Lo   [7] MASARAM GONDI LETTER A..MASARAM GONDI LETTER E
    Interval{ 0x11D08, 0x11D09 }, // Lo   [2] MASARAM GONDI LETTER AI..MASARAM GONDI LETTER O
    Interval{ 0x11D0B, 0x11D30 }, // Lo  [38] MASARAM GONDI LETTER AU..MASARAM GONDI LETTER TRA
    Interval{ 0x11D46, 0x11D46 }, // Lo       MASARAM GONDI REPHA
    Interval{ 0x11D60, 0x11D65 }, // Lo   [6] GUNJALA GONDI LETTER A..GUNJALA GONDI LETTER UU
    Interval{ 0x11D67, 0x11D68 }, // Lo   [2] GUNJALA GONDI LETTER EE..GUNJALA GONDI LETTER AI
    Interval{ 0x11D6A, 0x11D89 }, // Lo  [32] GUNJALA GONDI LETTER OO..GUNJALA GONDI LETTER SA
    Interval{ 0x11D98, 0x11D98 }, // Lo       GUNJALA GONDI OM
    Interval{ 0x11EE0, 0x11EF2 }, // Lo  [19] MAKASAR LETTER KA..MAKASAR ANGKA
    Interval{ 0x11F02, 0x11F02 }, // Lo       KAWI SIGN REPHA
    Interval{ 0x11F04, 0x11F10 }, // Lo  [13] KAWI LETTER A..KAWI LETTER O
    Interval{ 0x11F12, 0x11F33 }, // Lo  [34] KAWI LETTER KA..KAWI LETTER JNYA
    Interval{ 0x11FB0, 0x11FB0 }, // Lo       LISU LETTER YHA
    Interval{ 0x12000, 0x12399 }, // Lo [922] CUNEIFORM SIGN A..CUNEIFORM SIGN U U
    Interval{ 0x12400, 0x1246E }, // Nl [111] CUNEIFORM NUMERIC SIGN TWO ASH..CUNEIFORM NUMERIC SIGN NINE U VARIANT FORM
    Interval{ 0x12480, 0x12543 }, // Lo [196] CUNEIFORM SIGN AB TIMES NUN TENU..CUNEIFORM SIGN ZU5 TIMES THREE DISH TENU
    Interval{ 0x12F90, 0x12FF0 }, // Lo  [97] CYPRO-MINOAN SIGN CM001..CYPRO-MINOAN SIGN CM114
    Interval{ 0x13000, 0x1342F }, // Lo [1072] EGYPTIAN HIEROGLYPH A001..EGYPTIAN HIEROGLYPH V011D
    Interval{ 0x13441, 0x13446 }, // Lo   [6] EGYPTIAN HIEROGLYPH FULL BLANK..EGYPTIAN HIEROGLYPH WIDE LOST SIGN
    Interval{ 0x14400, 0x14646 }, // Lo [583] ANATOLIAN HIEROGLYPH A001..ANATOLIAN HIEROGLYPH A530
    Interval{ 0x16800, 0x16A38 }, // Lo [569] BAMUM LETTER PHASE-A NGKUE MFON..BAMUM LETTER PHASE-F VUEQ
    Interval{ 0x16A40, 0x16A5E }, // Lo  [31] MRO LETTER TA..MRO LETTER TEK
    Interval{ 0x16A70, 0x16ABE }, // Lo  [79] TANGSA LETTER OZ..TANGSA LETTER ZA
    Interval{ 0x16AD0, 0x16AED }, // Lo  [30] BASSA VAH LETTER ENNI..BASSA VAH LETTER I
    Interval{ 0x16B00, 0x16B2F }, // Lo  [48] PAHAWH HMONG VOWEL KEEB..PAHAWH HMONG CONSONANT CAU
    Interval{ 0x16B40, 0x16B43 }, // Lm   [4] PAHAWH HMONG SIGN VOS SEEV..PAHAWH HMONG SIGN IB YAM
    Interval{ 0x16B63, 0x16B77 }, // Lo  [21] PAHAWH HMONG SIGN VOS LUB..PAHAWH HMONG SIGN CIM NRES TOS
    Interval{ 0x16B7D, 0x16B8F }, // Lo  [19] PAHAWH HMONG CLAN SIGN TSHEEJ..PAHAWH HMONG CLAN SIGN VWJ
    Interval{ 0x16E40, 0x16E7F }, // L&  [64] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x16F00, 0x16F4A }, // Lo  [75] MIAO LETTER PA..MIAO LETTER RTE
    Interval{ 0x16F50, 0x16F50 }, // Lo       MIAO LETTER NASALIZATION
    Interval{ 0x16F93, 0x16F9F }, // Lm  [13] MIAO LETTER TONE-2..MIAO LETTER REFORMED TONE-8
    Interval{ 0x16FE0, 0x16FE1 }, // Lm   [2] TANGUT ITERATION MARK..NUSHU ITERATION MARK
    Interval{ 0x16FE3, 0x16FE3 }, // Lm       OLD CHINESE ITERATION MARK
    Interval{ 0x17000, 0x187F7 }, // Lo [6136] TANGUT IDEOGRAPH-17000..TANGUT IDEOGRAPH-187F7
    Interval{ 0x18800, 0x18CD5 }, // Lo [1238] TANGUT COMPONENT-001..KHITAN SMALL SCRIPT CHARACTER-18CD5
    Interval{ 0x18D00, 0x18D08 }, // Lo   [9] TANGUT IDEOGRAPH-18D00..TANGUT IDEOGRAPH-18D08
    Interval{ 0x1AFF0, 0x1AFF3 }, // Lm   [4] KATAKANA LETTER MINNAN TONE-2..KATAKANA LETTER MINNAN TONE-5
    Interval{ 0x1AFF5, 0x1AFFB }, // Lm   [7] KATAKANA LETTER MINNAN TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-5
    Interval{ 0x1AFFD, 0x1AFFE }, // Lm   [2] KATAKANA LETTER MINNAN NASALIZED TONE-7..KATAKANA LETTER MINNAN NASALIZED TONE-8
    Interval{ 0x1B000, 0x1B122 }, // Lo [291] KATAKANA LETTER ARCHAIC E..KATAKANA LETTER ARCHAIC WU
    Interval{ 0x1B132, 0x1B132 }, // Lo       HIRAGANA LETTER SMALL KO
    Interval{ 0x1B150, 0x1B152 }, // Lo   [3] HIRAGANA LETTER SMALL WI..HIRAGANA LETTER SMALL WO
    Interval{ 0x1B155, 0x1B155 }, // Lo       KATAKANA LETTER SMALL KO
    Interval{ 0x1B164, 0x1B167 }, // Lo   [4] KATAKANA LETTER SMALL WI..KATAKANA LETTER SMALL N
    Interval{ 0x1B170, 0x1B2FB }, // Lo [396] NUSHU CHARACTER-1B170..NUSHU CHARACTER-1B2FB
    Interval{ 0x1BC00, 0x1BC6A }, // Lo [107] DUPLOYAN LETTER H..DUPLOYAN LETTER VOCALIC M
    Interval{ 0x1BC70, 0x1BC7C }, // Lo  [13] DUPLOYAN AFFIX LEFT HORIZONTAL SECANT..DUPLOYAN AFFIX ATTACHED TANGENT HOOK
    Interval{ 0x1BC80, 0x1BC88 }, // Lo   [9] DUPLOYAN AFFIX HIGH ACUTE..DUPLOYAN AFFIX HIGH VERTICAL
    Interval{ 0x1BC90, 0x1BC99 }, // Lo  [10] DUPLOYAN AFFIX LOW ACUTE..DUPLOYAN AFFIX LOW ARROW
    Interval{ 0x1D400, 0x1D454 }, // L&  [85] MATHEMATICAL BOLD CAPITAL A..MATHEMATICAL ITALIC SMALL G
    Interval{ 0x1D456, 0x1D49C }, // L&  [71] MATHEMATICAL ITALIC SMALL I..MATHEMATICAL SCRIPT CAPITAL A
    Interval{ 0x1D49E, 0x1D49F }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL C..MATHEMATICAL SCRIPT CAPITAL D
    Interval{ 0x1D4A2, 0x1D4A2 }, // L&       MATHEMATICAL SCRIPT CAPITAL G
    Interval{ 0x1D4A5, 0x1D4A6 }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL J..MATHEMATICAL SCRIPT CAPITAL K
    Interval{ 0x1D4A9, 0x1D4AC }, // L&   [4] MATHEMATICAL SCRIPT CAPITAL N..MATHEMATICAL SCRIPT CAPITAL Q
    Interval{ 0x1D4AE, 0x1D4B9 }, // L&  [12] MATHEMATICAL SCRIPT CAPITAL S..MATHEMATICAL SCRIPT SMALL D
    Interval{ 0x1D4BB, 0x1D4BB }, // L&       MATHEMATICAL SCRIPT SMALL F
    Interval{ 0x1D4BD, 0x1D4C3 }, // L&   [7] MATHEMATICAL SCRIPT SMALL H..MATHEMATICAL SCRIPT SMALL N
    Interval{ 0x1D4C5, 0x1D505 }, // L&  [65] MATHEMATICAL SCRIPT SMALL P..MATHEMATICAL FRAKTUR CAPITAL B
    Interval{ 0x1D507, 0x1D50A }, // L&   [4] MATHEMATICAL FRAKTUR CAPITAL D..MATHEMATICAL FRAKTUR CAPITAL G
    Interval{ 0x1D50D, 0x1D514 }, // L&   [8] MATHEMATICAL FRAKTUR CAPITAL J..MATHEMATICAL FRAKTUR CAPITAL Q
    Interval{ 0x1D516, 0x1D51C }, // L&   [7] MATHEMATICAL FRAKTUR CAPITAL S..MATHEMATICAL FRAKTUR CAPITAL Y
    Interval{ 0x1D51E, 0x1D539 }, // L&  [28] MATHEMATICAL FRAKTUR SMALL A..MATHEMATICAL DOUBLE-STRUCK CAPITAL B
    Interval{ 0x1D53B, 0x1D53E }, // L&   [4] MATHEMATICAL DOUBLE-STRUCK CAPITAL D..MATHEMATICAL DOUBLE-STRUCK CAPITAL G
    Interval{ 0x1D540, 0x1D544 }, // L&   [5] MATHEMATICAL DOUBLE-STRUCK CAPITAL I..MATHEMATICAL DOUBLE-STRUCK CAPITAL M
    Interval{ 0x1D546, 0x1D546 }, // L&       MATHEMATICAL DOUBLE-STRUCK CAPITAL O
    Interval{ 0x1D54A, 0x1D550 }, // L&   [7] MATHEMATICAL DOUBLE-STRUCK CAPITAL S..MATHEMATICAL DOUBLE-STRUCK CAPITAL Y
    Interval{ 0x1D552, 0x1D6A5 }, // L& [340] MATHEMATICAL DOUBLE-STRUCK SMALL A..MATHEMATICAL ITALIC SMALL DOTLESS J
    Interval{ 0x1D6A8, 0x1D6C0 }, // L&  [25] MATHEMATICAL BOLD CAPITAL ALPHA..MATHEMATICAL BOLD CAPITAL OMEGA
    Interval{ 0x1D6C2, 0x1D6DA }, // L&  [25] MATHEMATICAL BOLD SMALL ALPHA..MATHEMATICAL BOLD SMALL OMEGA
    Interval{ 0x1D6DC, 0x1D6FA }, // L&  [31] MATHEMATICAL BOLD EPSILON SYMBOL..MATHEMATICAL ITALIC CAPITAL OMEGA
    Interval{ 0x1D6FC, 0x1D714 }, // L&  [25] MATHEMATICAL ITALIC SMALL ALPHA..MATHEMATICAL ITALIC SMALL OMEGA
    Interval{ 0x1D716, 0x1D734 }, // L&  [31] MATHEMATICAL ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D736, 0x1D74E }, // L&  [25] MATHEMATICAL BOLD ITALIC SMALL ALPHA..MATHEMATICAL BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D750, 0x1D76E }, // L&  [31] MATHEMATICAL BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGA
    Interval{ 0x1D770, 0x1D788 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD SMALL OMEGA
    Interval{ 0x1D78A, 0x1D7A8 }, // L&  [31] MATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D7AA, 0x1D7C2 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D7C4, 0x1D7CB }, // L&   [8] MATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD SMALL DIGAMMA
    Interval{ 0x1DF00, 0x1DF09 }, // L&  [10] LATIN SMALL LETTER FENG DIGRAPH WITH TRILL..LATIN SMALL LETTER T WITH HOOK AND RETROFLEX HOOK
    Interval{ 0x1DF0A, 0x1DF0A }, // Lo       LATIN LETTER RETROFLEX CLICK WITH RETROFLEX HOOK
    Interval{ 0x1DF0B, 0x1DF1E }, // L&  [20] LATIN SMALL LETTER ESH WITH DOUBLE BAR..LATIN SMALL LETTER S WITH CURL
    Interval{ 0x1DF25, 0x1DF2A }, // L&   [6] LATIN SMALL LETTER D WITH MID-HEIGHT LEFT HOOK..LATIN SMALL LETTER T WITH MID-HEIGHT LEFT HOOK
    Interval{ 0x1E030, 0x1E06D }, // Lm  [62] MODIFIER LETTER CYRILLIC SMALL A..MODIFIER LETTER CYRILLIC SMALL STRAIGHT U WITH STROKE
    Interval{ 0x1E100, 0x1E12C }, // Lo  [45] NYIAKENG PUACHUE HMONG LETTER MA..NYIAKENG PUACHUE HMONG LETTER W
    Interval{ 0x1E137, 0x1E13D }, // Lm   [7] NYIAKENG PUACHUE HMONG SIGN FOR PERSON..NYIAKENG PUACHUE HMONG SYLLABLE LENGTHENER
    Interval{ 0x1E14E, 0x1E14E }, // Lo       NYIAKENG PUACHUE HMONG LOGOGRAM NYAJ
    Interval{ 0x1E290, 0x1E2AD }, // Lo  [30] TOTO LETTER PA..TOTO LETTER A
    Interval{ 0x1E2C0, 0x1E2EB }, // Lo  [44] WANCHO LETTER AA..WANCHO LETTER YIH
    Interval{ 0x1E4D0, 0x1E4EA }, // Lo  [27] NAG MUNDARI LETTER O..NAG MUNDARI LETTER ELL
    Interval{ 0x1E4EB, 0x1E4EB }, // Lm       NAG MUNDARI SIGN OJOD
    Interval{ 0x1E7E0, 0x1E7E6 }, // Lo   [7] ETHIOPIC SYLLABLE HHYA..ETHIOPIC SYLLABLE HHYO
    Interval{ 0x1E7E8, 0x1E7EB }, // Lo   [4] ETHIOPIC SYLLABLE GURAGE HHWA..ETHIOPIC SYLLABLE HHWE
    Interval{ 0x1E7ED, 0x1E7EE }, // Lo   [2] ETHIOPIC SYLLABLE GURAGE MWI..ETHIOPIC SYLLABLE GURAGE MWEE
    Interval{ 0x1E7F0, 0x1E7FE }, // Lo  [15] ETHIOPIC SYLLABLE GURAGE QWI..ETHIOPIC SYLLABLE GURAGE PWEE
    Interval{ 0x1E800, 0x1E8C4 }, // Lo [197] MENDE KIKAKUI SYLLABLE M001 KI..MENDE KIKAKUI SYLLABLE M060 NYON
    Interval{ 0x1E900, 0x1E943 }, // L&  [68] ADLAM CAPITAL LETTER ALIF..ADLAM SMALL LETTER SHA
    Interval{ 0x1E94B, 0x1E94B }, // Lm       ADLAM NASALIZATION MARK
    Interval{ 0x1EE00, 0x1EE03 }, // Lo   [4] ARABIC MATHEMATICAL ALEF..ARABIC MATHEMATICAL DAL
    Interval{ 0x1EE05, 0x1EE1F }, // Lo  [27] ARABIC MATHEMATICAL WAW..ARABIC MATHEMATICAL DOTLESS QAF
    Interval{ 0x1EE21, 0x1EE22 }, // Lo   [2] ARABIC MATHEMATICAL INITIAL BEH..ARABIC MATHEMATICAL INITIAL JEEM
    Interval{ 0x1EE24, 0x1EE24 }, // Lo       ARABIC MATHEMATICAL INITIAL HEH
    Interval{ 0x1EE27, 0x1EE27 }, // Lo       ARABIC MATHEMATICAL INITIAL HAH
    Interval{ 0x1EE29, 0x1EE32 }, // Lo  [10] ARABIC MATHEMATICAL INITIAL YEH..ARABIC MATHEMATICAL INITIAL QAF
    Interval{ 0x1EE34, 0x1EE37 }, // Lo   [4] ARABIC MATHEMATICAL INITIAL SHEEN..ARABIC MATHEMATICAL INITIAL KHAH
    Interval{ 0x1EE39, 0x1EE39 }, // Lo       ARABIC MATHEMATICAL INITIAL DAD
    Interval{ 0x1EE3B, 0x1EE3B }, // Lo       ARABIC MATHEMATICAL INITIAL GHAIN
    Interval{ 0x1EE42, 0x1EE42 }, // Lo       ARABIC MATHEMATICAL TAILED JEEM
    Interval{ 0x1EE47, 0x1EE47 }, // Lo       ARABIC MATHEMATICAL TAILED HAH
    Interval{ 0x1EE49, 0x1EE49 }, // Lo       ARABIC MATHEMATICAL TAILED YEH
    Interval{ 0x1EE4B, 0x1EE4B }, // Lo       ARABIC MATHEMATICAL TAILED LAM
    Interval{ 0x1EE4D, 0x1EE4F }, // Lo   [3] ARABIC MATHEMATICAL TAILED NOON..ARABIC MATHEMATICAL TAILED AIN
    Interval{ 0x1EE51, 0x1EE52 }, // Lo   [2] ARABIC MATHEMATICAL TAILED SAD..ARABIC MATHEMATICAL TAILED QAF
    Interval{ 0x1EE54, 0x1EE54 }, // Lo       ARABIC MATHEMATICAL TAILED SHEEN
    Interval{ 0x1EE57, 0x1EE57 }, // Lo       ARABIC MATHEMATICAL TAILED KHAH
    Interval{ 0x1EE59, 0x1EE59 }, // Lo       ARABIC MATHEMATICAL TAILED DAD
    Interval{ 0x1EE5B, 0x1EE5B }, // Lo       ARABIC MATHEMATICAL TAILED GHAIN
    Interval{ 0x1EE5D, 0x1EE5D }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS NOON
    Interval{ 0x1EE5F, 0x1EE5F }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS QAF
    Interval{ 0x1EE61, 0x1EE62 }, // Lo   [2] ARABIC MATHEMATICAL STRETCHED BEH..ARABIC MATHEMATICAL STRETCHED JEEM
    Interval{ 0x1EE64, 0x1EE64 }, // Lo       ARABIC MATHEMATICAL STRETCHED HEH
    Interval{ 0x1EE67, 0x1EE6A }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED HAH..ARABIC MATHEMATICAL STRETCHED KAF
    Interval{ 0x1EE6C, 0x1EE72 }, // Lo   [7] ARABIC MATHEMATICAL STRETCHED MEEM..ARABIC MATHEMATICAL STRETCHED QAF
    Interval{ 0x1EE74, 0x1EE77 }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED SHEEN..ARABIC MATHEMATICAL STRETCHED KHAH
    Interval{ 0x1EE79, 0x1EE7C }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED DAD..ARABIC MATHEMATICAL STRETCHED DOTLESS BEH
    Interval{ 0x1EE7E, 0x1EE7E }, // Lo       ARABIC MATHEMATICAL STRETCHED DOTLESS FEH
    Interval{ 0x1EE80, 0x1EE89 }, // Lo  [10] ARABIC MATHEMATICAL LOOPED ALEF..ARABIC MATHEMATICAL LOOPED YEH
    Interval{ 0x1EE8B, 0x1EE9B }, // Lo  [17] ARABIC MATHEMATICAL LOOPED LAM..ARABIC MATHEMATICAL LOOPED GHAIN
    Interval{ 0x1EEA1, 0x1EEA3 }, // Lo   [3] ARABIC MATHEMATICAL DOUBLE-STRUCK BEH..ARABIC MATHEMATICAL DOUBLE-STRUCK DAL
    Interval{ 0x1EEA5, 0x1EEA9 }, // Lo   [5] ARABIC MATHEMATICAL DOUBLE-STRUCK WAW..ARABIC MATHEMATICAL DOUBLE-STRUCK YEH
    Interval{ 0x1EEAB, 0x1EEBB }, // Lo  [17] ARABIC MATHEMATICAL DOUBLE-STRUCK LAM..ARABIC MATHEMATICAL DOUBLE-STRUCK GHAIN
    Interval{ 0x20000, 0x2A6DF }, // Lo [42720] CJK UNIFIED IDEOGRAPH-20000..CJK UNIFIED IDEOGRAPH-2A6DF
    Interval{ 0x2A700, 0x2B739 }, // Lo [4154] CJK UNIFIED IDEOGRAPH-2A700..CJK UNIFIED IDEOGRAPH-2B739
    Interval{ 0x2B740, 0x2B81D }, // Lo [222] CJK UNIFIED IDEOGRAPH-2B740..CJK UNIFIED IDEOGRAPH-2B81D
    Interval{ 0x2B820, 0x2CEA1 }, // Lo [5762] CJK UNIFIED IDEOGRAPH-2B820..CJK UNIFIED IDEOGRAPH-2CEA1
    Interval{ 0x2CEB0, 0x2EBE0 }, // Lo [7473] CJK UNIFIED IDEOGRAPH-2CEB0..CJK UNIFIED IDEOGRAPH-2EBE0
    Interval{ 0x2F800, 0x2FA1D }, // Lo [542] CJK COMPATIBILITY IDEOGRAPH-2F800..CJK COMPATIBILITY IDEOGRAPH-2FA1D
    Interval{ 0x30000, 0x3134A }, // Lo [4939] CJK UNIFIED IDEOGRAPH-30000..CJK UNIFIED IDEOGRAPH-3134A
    Interval{ 0x31350, 0x323AF }, // Lo [4192] CJK UNIFIED IDEOGRAPH-31350..CJK UNIFIED IDEOGRAPH-323AF
}; // }}}
auto static const Lowercase = std::array<Interval, 686>{ // {{{
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00AA, 0x00AA }, // Lo       FEMININE ORDINAL INDICATOR
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00BA, 0x00BA }, // Lo       MASCULINE ORDINAL INDICATOR
    Interval{ 0x00DF, 0x00F6 }, // L&  [24] LATIN SMALL LETTER SHARP S..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x00FF }, // L&   [8] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER Y WITH DIAERESIS
    Interval{ 0x0101, 0x0101 }, // L&       LATIN SMALL LETTER A WITH MACRON
    Interval{ 0x0103, 0x0103 }, // L&       LATIN SMALL LETTER A WITH BREVE
    Interval{ 0x0105, 0x0105 }, // L&       LATIN SMALL LETTER A WITH OGONEK
    Interval{ 0x0107, 0x0107 }, // L&       LATIN SMALL LETTER C WITH ACUTE
    Interval{ 0x0109, 0x0109 }, // L&       LATIN SMALL LETTER C WITH CIRCUMFLEX
    Interval{ 0x010B, 0x010B }, // L&       LATIN SMALL LETTER C WITH DOT ABOVE
    Interval{ 0x010D, 0x010D }, // L&       LATIN SMALL LETTER C WITH CARON
    Interval{ 0x010F, 0x010F }, // L&       LATIN SMALL LETTER D WITH CARON
    Interval{ 0x0111, 0x0111 }, // L&       LATIN SMALL LETTER D WITH STROKE
    Interval{ 0x0113, 0x0113 }, // L&       LATIN SMALL LETTER E WITH MACRON
    Interval{ 0x0115, 0x0115 }, // L&       LATIN SMALL LETTER E WITH BREVE
    Interval{ 0x0117, 0x0117 }, // L&       LATIN SMALL LETTER E WITH DOT ABOVE
    Interval{ 0x0119, 0x0119 }, // L&       LATIN SMALL LETTER E WITH OGONEK
    Interval{ 0x011B, 0x011B }, // L&       LATIN SMALL LETTER E WITH CARON
    Interval{ 0x011D, 0x011D }, // L&       LATIN SMALL LETTER G WITH CIRCUMFLEX
    Interval{ 0x011F, 0x011F }, // L&       LATIN SMALL LETTER G WITH BREVE
    Interval{ 0x0121, 0x0121 }, // L&       LATIN SMALL LETTER G WITH DOT ABOVE
    Interval{ 0x0123, 0x0123 }, // L&       LATIN SMALL LETTER G WITH CEDILLA
    Interval{ 0x0125, 0x0125 }, // L&       LATIN SMALL LETTER H WITH CIRCUMFLEX
    Interval{ 0x0127, 0x0127 }, // L&       LATIN SMALL LETTER H WITH STROKE
    Interval{ 0x0129, 0x0129 }, // L&       LATIN SMALL LETTER I WITH TILDE
    Interval{ 0x012B, 0x012B }, // L&       LATIN SMALL LETTER I WITH MACRON
    Interval{ 0x012D, 0x012D }, // L&       LATIN SMALL LETTER I WITH BREVE
    Interval{ 0x012F, 0x012F }, // L&       LATIN SMALL LETTER I WITH OGONEK
    Interval{ 0x0131, 0x0131 }, // L&       LATIN SMALL LETTER DOTLESS I
    Interval{ 0x0133, 0x0133 }, // L&       LATIN SMALL LIGATURE IJ
    Interval{ 0x0135, 0x0135 }, // L&       LATIN SMALL LETTER J WITH CIRCUMFLEX
    Interval{ 0x0137, 0x0138 }, // L&   [2] LATIN SMALL LETTER K WITH CEDILLA..LATIN SMALL LETTER KRA
    Interval{ 0x013A, 0x013A }, // L&       LATIN SMALL LETTER L WITH ACUTE
    Interval{ 0x013C, 0x013C }, // L&       LATIN SMALL LETTER L WITH CEDILLA
    Interval{ 0x013E, 0x013E }, // L&       LATIN SMALL LETTER L WITH CARON
    Interval{ 0x0140, 0x0140 }, // L&       LATIN SMALL LETTER L WITH MIDDLE DOT
    Interval{ 0x0142, 0x0142 }, // L&       LATIN SMALL LETTER L WITH STROKE
    Interval{ 0x0144, 0x0144 }, // L&       LATIN SMALL LETTER N WITH ACUTE
    Interval{ 0x0146, 0x0146 }, // L&       LATIN SMALL LETTER N WITH CEDILLA
    Interval{ 0x0148, 0x0149 }, // L&   [2] LATIN SMALL LETTER N WITH CARON..LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
    Interval{ 0x014B, 0x014B }, // L&       LATIN SMALL LETTER ENG
    Interval{ 0x014D, 0x014D }, // L&       LATIN SMALL LETTER O WITH MACRON
    Interval{ 0x014F, 0x014F }, // L&       LATIN SMALL LETTER O WITH BREVE
    Interval{ 0x0151, 0x0151 }, // L&       LATIN SMALL LETTER O WITH DOUBLE ACUTE
    Interval{ 0x0153, 0x0153 }, // L&       LATIN SMALL LIGATURE OE
    Interval{ 0x0155, 0x0155 }, // L&       LATIN SMALL LETTER R WITH ACUTE
    Interval{ 0x0157, 0x0157 }, // L&       LATIN SMALL LETTER R WITH CEDILLA
    Interval{ 0x0159, 0x0159 }, // L&       LATIN SMALL LETTER R WITH CARON
    Interval{ 0x015B, 0x015B }, // L&       LATIN SMALL LETTER S WITH ACUTE
    Interval{ 0x015D, 0x015D }, // L&       LATIN SMALL LETTER S WITH CIRCUMFLEX
    Interval{ 0x015F, 0x015F }, // L&       LATIN SMALL LETTER S WITH CEDILLA
    Interval{ 0x0161, 0x0161 }, // L&       LATIN SMALL LETTER S WITH CARON
    Interval{ 0x0163, 0x0163 }, // L&       LATIN SMALL LETTER T WITH CEDILLA
    Interval{ 0x0165, 0x0165 }, // L&       LATIN SMALL LETTER T WITH CARON
    Interval{ 0x0167, 0x0167 }, // L&       LATIN SMALL LETTER T WITH STROKE
    Interval{ 0x0169, 0x0169 }, // L&       LATIN SMALL LETTER U WITH TILDE
    Interval{ 0x016B, 0x016B }, // L&       LATIN SMALL LETTER U WITH MACRON
    Interval{ 0x016D, 0x016D }, // L&       LATIN SMALL LETTER U WITH BREVE
    Interval{ 0x016F, 0x016F }, // L&       LATIN SMALL LETTER U WITH RING ABOVE
    Interval{ 0x0171, 0x0171 }, // L&       LATIN SMALL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x0173, 0x0173 }, // L&       LATIN SMALL LETTER U WITH OGONEK
    Interval{ 0x0175, 0x0175 }, // L&       LATIN SMALL LETTER W WITH CIRCUMFLEX
    Interval{ 0x0177, 0x0177 }, // L&       LATIN SMALL LETTER Y WITH CIRCUMFLEX
    Interval{ 0x017A, 0x017A }, // L&       LATIN SMALL LETTER Z WITH ACUTE
    Interval{ 0x017C, 0x017C }, // L&       LATIN SMALL LETTER Z WITH DOT ABOVE
    Interval{ 0x017E, 0x0180 }, // L&   [3] LATIN SMALL LETTER Z WITH CARON..LATIN SMALL LETTER B WITH STROKE
    Interval{ 0x0183, 0x0183 }, // L&       LATIN SMALL LETTER B WITH TOPBAR
    Interval{ 0x0185, 0x0185 }, // L&       LATIN SMALL LETTER TONE SIX
    Interval{ 0x0188, 0x0188 }, // L&       LATIN SMALL LETTER C WITH HOOK
    Interval{ 0x018C, 0x018D }, // L&   [2] LATIN SMALL LETTER D WITH TOPBAR..LATIN SMALL LETTER TURNED DELTA
    Interval{ 0x0192, 0x0192 }, // L&       LATIN SMALL LETTER F WITH HOOK
    Interval{ 0x0195, 0x0195 }, // L&       LATIN SMALL LETTER HV
    Interval{ 0x0199, 0x019B }, // L&   [3] LATIN SMALL LETTER K WITH HOOK..LATIN SMALL LETTER LAMBDA WITH STROKE
    Interval{ 0x019E, 0x019E }, // L&       LATIN SMALL LETTER N WITH LONG RIGHT LEG
    Interval{ 0x01A1, 0x01A1 }, // L&       LATIN SMALL LETTER O WITH HORN
    Interval{ 0x01A3, 0x01A3 }, // L&       LATIN SMALL LETTER OI
    Interval{ 0x01A5, 0x01A5 }, // L&       LATIN SMALL LETTER P WITH HOOK
    Interval{ 0x01A8, 0x01A8 }, // L&       LATIN SMALL LETTER TONE TWO
    Interval{ 0x01AA, 0x01AB }, // L&   [2] LATIN LETTER REVERSED ESH LOOP..LATIN SMALL LETTER T WITH PALATAL HOOK
    Interval{ 0x01AD, 0x01AD }, // L&       LATIN SMALL LETTER T WITH HOOK
    Interval{ 0x01B0, 0x01B0 }, // L&       LATIN SMALL LETTER U WITH HORN
    Interval{ 0x01B4, 0x01B4 }, // L&       LATIN SMALL LETTER Y WITH HOOK
    Interval{ 0x01B6, 0x01B6 }, // L&       LATIN SMALL LETTER Z WITH STROKE
    Interval{ 0x01B9, 0x01BA }, // L&   [2] LATIN SMALL LETTER EZH REVERSED..LATIN SMALL LETTER EZH WITH TAIL
    Interval{ 0x01BD, 0x01BF }, // L&   [3] LATIN SMALL LETTER TONE FIVE..LATIN LETTER WYNN
    Interval{ 0x01C6, 0x01C6 }, // L&       LATIN SMALL LETTER DZ WITH CARON
    Interval{ 0x01C9, 0x01C9 }, // L&       LATIN SMALL LETTER LJ
    Interval{ 0x01CC, 0x01CC }, // L&       LATIN SMALL LETTER NJ
    Interval{ 0x01CE, 0x01CE }, // L&       LATIN SMALL LETTER A WITH CARON
    Interval{ 0x01D0, 0x01D0 }, // L&       LATIN SMALL LETTER I WITH CARON
    Interval{ 0x01D2, 0x01D2 }, // L&       LATIN SMALL LETTER O WITH CARON
    Interval{ 0x01D4, 0x01D4 }, // L&       LATIN SMALL LETTER U WITH CARON
    Interval{ 0x01D6, 0x01D6 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
    Interval{ 0x01D8, 0x01D8 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
    Interval{ 0x01DA, 0x01DA }, // L&       LATIN SMALL LETTER U WITH DIAERESIS AND CARON
    Interval{ 0x01DC, 0x01DD }, // L&   [2] LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE..LATIN SMALL LETTER TURNED E
    Interval{ 0x01DF, 0x01DF }, // L&       LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
    Interval{ 0x01E1, 0x01E1 }, // L&       LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
    Interval{ 0x01E3, 0x01E3 }, // L&       LATIN SMALL LETTER AE WITH MACRON
    Interval{ 0x01E5, 0x01E5 }, // L&       LATIN SMALL LETTER G WITH STROKE
    Interval{ 0x01E7, 0x01E7 }, // L&       LATIN SMALL LETTER G WITH CARON
    Interval{ 0x01E9, 0x01E9 }, // L&       LATIN SMALL LETTER K WITH CARON
    Interval{ 0x01EB, 0x01EB }, // L&       LATIN SMALL LETTER O WITH OGONEK
    Interval{ 0x01ED, 0x01ED }, // L&       LATIN SMALL LETTER O WITH OGONEK AND MACRON
    Interval{ 0x01EF, 0x01F0 }, // L&   [2] LATIN SMALL LETTER EZH WITH CARON..LATIN SMALL LETTER J WITH CARON
    Interval{ 0x01F3, 0x01F3 }, // L&       LATIN SMALL LETTER DZ
    Interval{ 0x01F5, 0x01F5 }, // L&       LATIN SMALL LETTER G WITH ACUTE
    Interval{ 0x01F9, 0x01F9 }, // L&       LATIN SMALL LETTER N WITH GRAVE
    Interval{ 0x01FB, 0x01FB }, // L&       LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
    Interval{ 0x01FD, 0x01FD }, // L&       LATIN SMALL LETTER AE WITH ACUTE
    Interval{ 0x01FF, 0x01FF }, // L&       LATIN SMALL LETTER O WITH STROKE AND ACUTE
    Interval{ 0x0201, 0x0201 }, // L&       LATIN SMALL LETTER A WITH DOUBLE GRAVE
    Interval{ 0x0203, 0x0203 }, // L&       LATIN SMALL LETTER A WITH INVERTED BREVE
    Interval{ 0x0205, 0x0205 }, // L&       LATIN SMALL LETTER E WITH DOUBLE GRAVE
    Interval{ 0x0207, 0x0207 }, // L&       LATIN SMALL LETTER E WITH INVERTED BREVE
    Interval{ 0x0209, 0x0209 }, // L&       LATIN SMALL LETTER I WITH DOUBLE GRAVE
    Interval{ 0x020B, 0x020B }, // L&       LATIN SMALL LETTER I WITH INVERTED BREVE
    Interval{ 0x020D, 0x020D }, // L&       LATIN SMALL LETTER O WITH DOUBLE GRAVE
    Interval{ 0x020F, 0x020F }, // L&       LATIN SMALL LETTER O WITH INVERTED BREVE
    Interval{ 0x0211, 0x0211 }, // L&       LATIN SMALL LETTER R WITH DOUBLE GRAVE
    Interval{ 0x0213, 0x0213 }, // L&       LATIN SMALL LETTER R WITH INVERTED BREVE
    Interval{ 0x0215, 0x0215 }, // L&       LATIN SMALL LETTER U WITH DOUBLE GRAVE
    Interval{ 0x0217, 0x0217 }, // L&       LATIN SMALL LETTER U WITH INVERTED BREVE
    Interval{ 0x0219, 0x0219 }, // L&       LATIN SMALL LETTER S WITH COMMA BELOW
    Interval{ 0x021B, 0x021B }, // L&       LATIN SMALL LETTER T WITH COMMA BELOW
    Interval{ 0x021D, 0x021D }, // L&       LATIN SMALL LETTER YOGH
    Interval{ 0x021F, 0x021F }, // L&       LATIN SMALL LETTER H WITH CARON
    Interval{ 0x0221, 0x0221 }, // L&       LATIN SMALL LETTER D WITH CURL
    Interval{ 0x0223, 0x0223 }, // L&       LATIN SMALL LETTER OU
    Interval{ 0x0225, 0x0225 }, // L&       LATIN SMALL LETTER Z WITH HOOK
    Interval{ 0x0227, 0x0227 }, // L&       LATIN SMALL LETTER A WITH DOT ABOVE
    Interval{ 0x0229, 0x0229 }, // L&       LATIN SMALL LETTER E WITH CEDILLA
    Interval{ 0x022B, 0x022B }, // L&       LATIN SMALL LETTER O WITH DIAERESIS AND MACRON
    Interval{ 0x022D, 0x022D }, // L&       LATIN SMALL LETTER O WITH TILDE AND MACRON
    Interval{ 0x022F, 0x022F }, // L&       LATIN SMALL LETTER O WITH DOT ABOVE
    Interval{ 0x0231, 0x0231 }, // L&       LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON
    Interval{ 0x0233, 0x0239 }, // L&   [7] LATIN SMALL LETTER Y WITH MACRON..LATIN SMALL LETTER QP DIGRAPH
    Interval{ 0x023C, 0x023C }, // L&       LATIN SMALL LETTER C WITH STROKE
    Interval{ 0x023F, 0x0240 }, // L&   [2] LATIN SMALL LETTER S WITH SWASH TAIL..LATIN SMALL LETTER Z WITH SWASH TAIL
    Interval{ 0x0242, 0x0242 }, // L&       LATIN SMALL LETTER GLOTTAL STOP
    Interval{ 0x0247, 0x0247 }, // L&       LATIN SMALL LETTER E WITH STROKE
    Interval{ 0x0249, 0x0249 }, // L&       LATIN SMALL LETTER J WITH STROKE
    Interval{ 0x024B, 0x024B }, // L&       LATIN SMALL LETTER Q WITH HOOK TAIL
    Interval{ 0x024D, 0x024D }, // L&       LATIN SMALL LETTER R WITH STROKE
    Interval{ 0x024F, 0x0293 }, // L&  [69] LATIN SMALL LETTER Y WITH STROKE..LATIN SMALL LETTER EZH WITH CURL
    Interval{ 0x0295, 0x02AF }, // L&  [27] LATIN LETTER PHARYNGEAL VOICED FRICATIVE..LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    Interval{ 0x02B0, 0x02B8 }, // Lm   [9] MODIFIER LETTER SMALL H..MODIFIER LETTER SMALL Y
    Interval{ 0x02C0, 0x02C1 }, // Lm   [2] MODIFIER LETTER GLOTTAL STOP..MODIFIER LETTER REVERSED GLOTTAL STOP
    Interval{ 0x02E0, 0x02E4 }, // Lm   [5] MODIFIER LETTER SMALL GAMMA..MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    Interval{ 0x0345, 0x0345 }, // Mn       COMBINING GREEK YPOGEGRAMMENI
    Interval{ 0x0371, 0x0371 }, // L&       GREEK SMALL LETTER HETA
    Interval{ 0x0373, 0x0373 }, // L&       GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0377, 0x0377 }, // L&       GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037A, 0x037A }, // Lm       GREEK YPOGEGRAMMENI
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x0390, 0x0390 }, // L&       GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
    Interval{ 0x03AC, 0x03CE }, // L&  [35] GREEK SMALL LETTER ALPHA WITH TONOS..GREEK SMALL LETTER OMEGA WITH TONOS
    Interval{ 0x03D0, 0x03D1 }, // L&   [2] GREEK BETA SYMBOL..GREEK THETA SYMBOL
    Interval{ 0x03D5, 0x03D7 }, // L&   [3] GREEK PHI SYMBOL..GREEK KAI SYMBOL
    Interval{ 0x03D9, 0x03D9 }, // L&       GREEK SMALL LETTER ARCHAIC KOPPA
    Interval{ 0x03DB, 0x03DB }, // L&       GREEK SMALL LETTER STIGMA
    Interval{ 0x03DD, 0x03DD }, // L&       GREEK SMALL LETTER DIGAMMA
    Interval{ 0x03DF, 0x03DF }, // L&       GREEK SMALL LETTER KOPPA
    Interval{ 0x03E1, 0x03E1 }, // L&       GREEK SMALL LETTER SAMPI
    Interval{ 0x03E3, 0x03E3 }, // L&       COPTIC SMALL LETTER SHEI
    Interval{ 0x03E5, 0x03E5 }, // L&       COPTIC SMALL LETTER FEI
    Interval{ 0x03E7, 0x03E7 }, // L&       COPTIC SMALL LETTER KHEI
    Interval{ 0x03E9, 0x03E9 }, // L&       COPTIC SMALL LETTER HORI
    Interval{ 0x03EB, 0x03EB }, // L&       COPTIC SMALL LETTER GANGIA
    Interval{ 0x03ED, 0x03ED }, // L&       COPTIC SMALL LETTER SHIMA
    Interval{ 0x03EF, 0x03F3 }, // L&   [5] COPTIC SMALL LETTER DEI..GREEK LETTER YOT
    Interval{ 0x03F5, 0x03F5 }, // L&       GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F8, 0x03F8 }, // L&       GREEK SMALL LETTER SHO
    Interval{ 0x03FB, 0x03FC }, // L&   [2] GREEK SMALL LETTER SAN..GREEK RHO WITH STROKE SYMBOL
    Interval{ 0x0430, 0x045F }, // L&  [48] CYRILLIC SMALL LETTER A..CYRILLIC SMALL LETTER DZHE
    Interval{ 0x0461, 0x0461 }, // L&       CYRILLIC SMALL LETTER OMEGA
    Interval{ 0x0463, 0x0463 }, // L&       CYRILLIC SMALL LETTER YAT
    Interval{ 0x0465, 0x0465 }, // L&       CYRILLIC SMALL LETTER IOTIFIED E
    Interval{ 0x0467, 0x0467 }, // L&       CYRILLIC SMALL LETTER LITTLE YUS
    Interval{ 0x0469, 0x0469 }, // L&       CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS
    Interval{ 0x046B, 0x046B }, // L&       CYRILLIC SMALL LETTER BIG YUS
    Interval{ 0x046D, 0x046D }, // L&       CYRILLIC SMALL LETTER IOTIFIED BIG YUS
    Interval{ 0x046F, 0x046F }, // L&       CYRILLIC SMALL LETTER KSI
    Interval{ 0x0471, 0x0471 }, // L&       CYRILLIC SMALL LETTER PSI
    Interval{ 0x0473, 0x0473 }, // L&       CYRILLIC SMALL LETTER FITA
    Interval{ 0x0475, 0x0475 }, // L&       CYRILLIC SMALL LETTER IZHITSA
    Interval{ 0x0477, 0x0477 }, // L&       CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
    Interval{ 0x0479, 0x0479 }, // L&       CYRILLIC SMALL LETTER UK
    Interval{ 0x047B, 0x047B }, // L&       CYRILLIC SMALL LETTER ROUND OMEGA
    Interval{ 0x047D, 0x047D }, // L&       CYRILLIC SMALL LETTER OMEGA WITH TITLO
    Interval{ 0x047F, 0x047F }, // L&       CYRILLIC SMALL LETTER OT
    Interval{ 0x0481, 0x0481 }, // L&       CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x048B, 0x048B }, // L&       CYRILLIC SMALL LETTER SHORT I WITH TAIL
    Interval{ 0x048D, 0x048D }, // L&       CYRILLIC SMALL LETTER SEMISOFT SIGN
    Interval{ 0x048F, 0x048F }, // L&       CYRILLIC SMALL LETTER ER WITH TICK
    Interval{ 0x0491, 0x0491 }, // L&       CYRILLIC SMALL LETTER GHE WITH UPTURN
    Interval{ 0x0493, 0x0493 }, // L&       CYRILLIC SMALL LETTER GHE WITH STROKE
    Interval{ 0x0495, 0x0495 }, // L&       CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
    Interval{ 0x0497, 0x0497 }, // L&       CYRILLIC SMALL LETTER ZHE WITH DESCENDER
    Interval{ 0x0499, 0x0499 }, // L&       CYRILLIC SMALL LETTER ZE WITH DESCENDER
    Interval{ 0x049B, 0x049B }, // L&       CYRILLIC SMALL LETTER KA WITH DESCENDER
    Interval{ 0x049D, 0x049D }, // L&       CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
    Interval{ 0x049F, 0x049F }, // L&       CYRILLIC SMALL LETTER KA WITH STROKE
    Interval{ 0x04A1, 0x04A1 }, // L&       CYRILLIC SMALL LETTER BASHKIR KA
    Interval{ 0x04A3, 0x04A3 }, // L&       CYRILLIC SMALL LETTER EN WITH DESCENDER
    Interval{ 0x04A5, 0x04A5 }, // L&       CYRILLIC SMALL LIGATURE EN GHE
    Interval{ 0x04A7, 0x04A7 }, // L&       CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
    Interval{ 0x04A9, 0x04A9 }, // L&       CYRILLIC SMALL LETTER ABKHASIAN HA
    Interval{ 0x04AB, 0x04AB }, // L&       CYRILLIC SMALL LETTER ES WITH DESCENDER
    Interval{ 0x04AD, 0x04AD }, // L&       CYRILLIC SMALL LETTER TE WITH DESCENDER
    Interval{ 0x04AF, 0x04AF }, // L&       CYRILLIC SMALL LETTER STRAIGHT U
    Interval{ 0x04B1, 0x04B1 }, // L&       CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
    Interval{ 0x04B3, 0x04B3 }, // L&       CYRILLIC SMALL LETTER HA WITH DESCENDER
    Interval{ 0x04B5, 0x04B5 }, // L&       CYRILLIC SMALL LIGATURE TE TSE
    Interval{ 0x04B7, 0x04B7 }, // L&       CYRILLIC SMALL LETTER CHE WITH DESCENDER
    Interval{ 0x04B9, 0x04B9 }, // L&       CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
    Interval{ 0x04BB, 0x04BB }, // L&       CYRILLIC SMALL LETTER SHHA
    Interval{ 0x04BD, 0x04BD }, // L&       CYRILLIC SMALL LETTER ABKHASIAN CHE
    Interval{ 0x04BF, 0x04BF }, // L&       CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
    Interval{ 0x04C2, 0x04C2 }, // L&       CYRILLIC SMALL LETTER ZHE WITH BREVE
    Interval{ 0x04C4, 0x04C4 }, // L&       CYRILLIC SMALL LETTER KA WITH HOOK
    Interval{ 0x04C6, 0x04C6 }, // L&       CYRILLIC SMALL LETTER EL WITH TAIL
    Interval{ 0x04C8, 0x04C8 }, // L&       CYRILLIC SMALL LETTER EN WITH HOOK
    Interval{ 0x04CA, 0x04CA }, // L&       CYRILLIC SMALL LETTER EN WITH TAIL
    Interval{ 0x04CC, 0x04CC }, // L&       CYRILLIC SMALL LETTER KHAKASSIAN CHE
    Interval{ 0x04CE, 0x04CF }, // L&   [2] CYRILLIC SMALL LETTER EM WITH TAIL..CYRILLIC SMALL LETTER PALOCHKA
    Interval{ 0x04D1, 0x04D1 }, // L&       CYRILLIC SMALL LETTER A WITH BREVE
    Interval{ 0x04D3, 0x04D3 }, // L&       CYRILLIC SMALL LETTER A WITH DIAERESIS
    Interval{ 0x04D5, 0x04D5 }, // L&       CYRILLIC SMALL LIGATURE A IE
    Interval{ 0x04D7, 0x04D7 }, // L&       CYRILLIC SMALL LETTER IE WITH BREVE
    Interval{ 0x04D9, 0x04D9 }, // L&       CYRILLIC SMALL LETTER SCHWA
    Interval{ 0x04DB, 0x04DB }, // L&       CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
    Interval{ 0x04DD, 0x04DD }, // L&       CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
    Interval{ 0x04DF, 0x04DF }, // L&       CYRILLIC SMALL LETTER ZE WITH DIAERESIS
    Interval{ 0x04E1, 0x04E1 }, // L&       CYRILLIC SMALL LETTER ABKHASIAN DZE
    Interval{ 0x04E3, 0x04E3 }, // L&       CYRILLIC SMALL LETTER I WITH MACRON
    Interval{ 0x04E5, 0x04E5 }, // L&       CYRILLIC SMALL LETTER I WITH DIAERESIS
    Interval{ 0x04E7, 0x04E7 }, // L&       CYRILLIC SMALL LETTER O WITH DIAERESIS
    Interval{ 0x04E9, 0x04E9 }, // L&       CYRILLIC SMALL LETTER BARRED O
    Interval{ 0x04EB, 0x04EB }, // L&       CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
    Interval{ 0x04ED, 0x04ED }, // L&       CYRILLIC SMALL LETTER E WITH DIAERESIS
    Interval{ 0x04EF, 0x04EF }, // L&       CYRILLIC SMALL LETTER U WITH MACRON
    Interval{ 0x04F1, 0x04F1 }, // L&       CYRILLIC SMALL LETTER U WITH DIAERESIS
    Interval{ 0x04F3, 0x04F3 }, // L&       CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x04F5, 0x04F5 }, // L&       CYRILLIC SMALL LETTER CHE WITH DIAERESIS
    Interval{ 0x04F7, 0x04F7 }, // L&       CYRILLIC SMALL LETTER GHE WITH DESCENDER
    Interval{ 0x04F9, 0x04F9 }, // L&       CYRILLIC SMALL LETTER YERU WITH DIAERESIS
    Interval{ 0x04FB, 0x04FB }, // L&       CYRILLIC SMALL LETTER GHE WITH STROKE AND HOOK
    Interval{ 0x04FD, 0x04FD }, // L&       CYRILLIC SMALL LETTER HA WITH HOOK
    Interval{ 0x04FF, 0x04FF }, // L&       CYRILLIC SMALL LETTER HA WITH STROKE
    Interval{ 0x0501, 0x0501 }, // L&       CYRILLIC SMALL LETTER KOMI DE
    Interval{ 0x0503, 0x0503 }, // L&       CYRILLIC SMALL LETTER KOMI DJE
    Interval{ 0x0505, 0x0505 }, // L&       CYRILLIC SMALL LETTER KOMI ZJE
    Interval{ 0x0507, 0x0507 }, // L&       CYRILLIC SMALL LETTER KOMI DZJE
    Interval{ 0x0509, 0x0509 }, // L&       CYRILLIC SMALL LETTER KOMI LJE
    Interval{ 0x050B, 0x050B }, // L&       CYRILLIC SMALL LETTER KOMI NJE
    Interval{ 0x050D, 0x050D }, // L&       CYRILLIC SMALL LETTER KOMI SJE
    Interval{ 0x050F, 0x050F }, // L&       CYRILLIC SMALL LETTER KOMI TJE
    Interval{ 0x0511, 0x0511 }, // L&       CYRILLIC SMALL LETTER REVERSED ZE
    Interval{ 0x0513, 0x0513 }, // L&       CYRILLIC SMALL LETTER EL WITH HOOK
    Interval{ 0x0515, 0x0515 }, // L&       CYRILLIC SMALL LETTER LHA
    Interval{ 0x0517, 0x0517 }, // L&       CYRILLIC SMALL LETTER RHA
    Interval{ 0x0519, 0x0519 }, // L&       CYRILLIC SMALL LETTER YAE
    Interval{ 0x051B, 0x051B }, // L&       CYRILLIC SMALL LETTER QA
    Interval{ 0x051D, 0x051D }, // L&       CYRILLIC SMALL LETTER WE
    Interval{ 0x051F, 0x051F }, // L&       CYRILLIC SMALL LETTER ALEUT KA
    Interval{ 0x0521, 0x0521 }, // L&       CYRILLIC SMALL LETTER EL WITH MIDDLE HOOK
    Interval{ 0x0523, 0x0523 }, // L&       CYRILLIC SMALL LETTER EN WITH MIDDLE HOOK
    Interval{ 0x0525, 0x0525 }, // L&       CYRILLIC SMALL LETTER PE WITH DESCENDER
    Interval{ 0x0527, 0x0527 }, // L&       CYRILLIC SMALL LETTER SHHA WITH DESCENDER
    Interval{ 0x0529, 0x0529 }, // L&       CYRILLIC SMALL LETTER EN WITH LEFT HOOK
    Interval{ 0x052B, 0x052B }, // L&       CYRILLIC SMALL LETTER DZZHE
    Interval{ 0x052D, 0x052D }, // L&       CYRILLIC SMALL LETTER DCHE
    Interval{ 0x052F, 0x052F }, // L&       CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0560, 0x0588 }, // L&  [41] ARMENIAN SMALL LETTER TURNED AYB..ARMENIAN SMALL LETTER YI WITH STROKE
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FC, 0x10FC }, // Lm       MODIFIER LETTER GEORGIAN NAR
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1D00, 0x1D2B }, // L&  [44] LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTER SMALL CAPITAL EL
    Interval{ 0x1D2C, 0x1D6A }, // Lm  [63] MODIFIER LETTER CAPITAL A..GREEK SUBSCRIPT SMALL LETTER CHI
    Interval{ 0x1D6B, 0x1D77 }, // L&  [13] LATIN SMALL LETTER UE..LATIN SMALL LETTER TURNED G
    Interval{ 0x1D78, 0x1D78 }, // Lm       MODIFIER LETTER CYRILLIC EN
    Interval{ 0x1D79, 0x1D9A }, // L&  [34] LATIN SMALL LETTER INSULAR G..LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
    Interval{ 0x1D9B, 0x1DBF }, // Lm  [37] MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER LETTER SMALL THETA
    Interval{ 0x1E01, 0x1E01 }, // L&       LATIN SMALL LETTER A WITH RING BELOW
    Interval{ 0x1E03, 0x1E03 }, // L&       LATIN SMALL LETTER B WITH DOT ABOVE
    Interval{ 0x1E05, 0x1E05 }, // L&       LATIN SMALL LETTER B WITH DOT BELOW
    Interval{ 0x1E07, 0x1E07 }, // L&       LATIN SMALL LETTER B WITH LINE BELOW
    Interval{ 0x1E09, 0x1E09 }, // L&       LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
    Interval{ 0x1E0B, 0x1E0B }, // L&       LATIN SMALL LETTER D WITH DOT ABOVE
    Interval{ 0x1E0D, 0x1E0D }, // L&       LATIN SMALL LETTER D WITH DOT BELOW
    Interval{ 0x1E0F, 0x1E0F }, // L&       LATIN SMALL LETTER D WITH LINE BELOW
    Interval{ 0x1E11, 0x1E11 }, // L&       LATIN SMALL LETTER D WITH CEDILLA
    Interval{ 0x1E13, 0x1E13 }, // L&       LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
    Interval{ 0x1E15, 0x1E15 }, // L&       LATIN SMALL LETTER E WITH MACRON AND GRAVE
    Interval{ 0x1E17, 0x1E17 }, // L&       LATIN SMALL LETTER E WITH MACRON AND ACUTE
    Interval{ 0x1E19, 0x1E19 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
    Interval{ 0x1E1B, 0x1E1B }, // L&       LATIN SMALL LETTER E WITH TILDE BELOW
    Interval{ 0x1E1D, 0x1E1D }, // L&       LATIN SMALL LETTER E WITH CEDILLA AND BREVE
    Interval{ 0x1E1F, 0x1E1F }, // L&       LATIN SMALL LETTER F WITH DOT ABOVE
    Interval{ 0x1E21, 0x1E21 }, // L&       LATIN SMALL LETTER G WITH MACRON
    Interval{ 0x1E23, 0x1E23 }, // L&       LATIN SMALL LETTER H WITH DOT ABOVE
    Interval{ 0x1E25, 0x1E25 }, // L&       LATIN SMALL LETTER H WITH DOT BELOW
    Interval{ 0x1E27, 0x1E27 }, // L&       LATIN SMALL LETTER H WITH DIAERESIS
    Interval{ 0x1E29, 0x1E29 }, // L&       LATIN SMALL LETTER H WITH CEDILLA
    Interval{ 0x1E2B, 0x1E2B }, // L&       LATIN SMALL LETTER H WITH BREVE BELOW
    Interval{ 0x1E2D, 0x1E2D }, // L&       LATIN SMALL LETTER I WITH TILDE BELOW
    Interval{ 0x1E2F, 0x1E2F }, // L&       LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
    Interval{ 0x1E31, 0x1E31 }, // L&       LATIN SMALL LETTER K WITH ACUTE
    Interval{ 0x1E33, 0x1E33 }, // L&       LATIN SMALL LETTER K WITH DOT BELOW
    Interval{ 0x1E35, 0x1E35 }, // L&       LATIN SMALL LETTER K WITH LINE BELOW
    Interval{ 0x1E37, 0x1E37 }, // L&       LATIN SMALL LETTER L WITH DOT BELOW
    Interval{ 0x1E39, 0x1E39 }, // L&       LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
    Interval{ 0x1E3B, 0x1E3B }, // L&       LATIN SMALL LETTER L WITH LINE BELOW
    Interval{ 0x1E3D, 0x1E3D }, // L&       LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
    Interval{ 0x1E3F, 0x1E3F }, // L&       LATIN SMALL LETTER M WITH ACUTE
    Interval{ 0x1E41, 0x1E41 }, // L&       LATIN SMALL LETTER M WITH DOT ABOVE
    Interval{ 0x1E43, 0x1E43 }, // L&       LATIN SMALL LETTER M WITH DOT BELOW
    Interval{ 0x1E45, 0x1E45 }, // L&       LATIN SMALL LETTER N WITH DOT ABOVE
    Interval{ 0x1E47, 0x1E47 }, // L&       LATIN SMALL LETTER N WITH DOT BELOW
    Interval{ 0x1E49, 0x1E49 }, // L&       LATIN SMALL LETTER N WITH LINE BELOW
    Interval{ 0x1E4B, 0x1E4B }, // L&       LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
    Interval{ 0x1E4D, 0x1E4D }, // L&       LATIN SMALL LETTER O WITH TILDE AND ACUTE
    Interval{ 0x1E4F, 0x1E4F }, // L&       LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
    Interval{ 0x1E51, 0x1E51 }, // L&       LATIN SMALL LETTER O WITH MACRON AND GRAVE
    Interval{ 0x1E53, 0x1E53 }, // L&       LATIN SMALL LETTER O WITH MACRON AND ACUTE
    Interval{ 0x1E55, 0x1E55 }, // L&       LATIN SMALL LETTER P WITH ACUTE
    Interval{ 0x1E57, 0x1E57 }, // L&       LATIN SMALL LETTER P WITH DOT ABOVE
    Interval{ 0x1E59, 0x1E59 }, // L&       LATIN SMALL LETTER R WITH DOT ABOVE
    Interval{ 0x1E5B, 0x1E5B }, // L&       LATIN SMALL LETTER R WITH DOT BELOW
    Interval{ 0x1E5D, 0x1E5D }, // L&       LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
    Interval{ 0x1E5F, 0x1E5F }, // L&       LATIN SMALL LETTER R WITH LINE BELOW
    Interval{ 0x1E61, 0x1E61 }, // L&       LATIN SMALL LETTER S WITH DOT ABOVE
    Interval{ 0x1E63, 0x1E63 }, // L&       LATIN SMALL LETTER S WITH DOT BELOW
    Interval{ 0x1E65, 0x1E65 }, // L&       LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
    Interval{ 0x1E67, 0x1E67 }, // L&       LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
    Interval{ 0x1E69, 0x1E69 }, // L&       LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
    Interval{ 0x1E6B, 0x1E6B }, // L&       LATIN SMALL LETTER T WITH DOT ABOVE
    Interval{ 0x1E6D, 0x1E6D }, // L&       LATIN SMALL LETTER T WITH DOT BELOW
    Interval{ 0x1E6F, 0x1E6F }, // L&       LATIN SMALL LETTER T WITH LINE BELOW
    Interval{ 0x1E71, 0x1E71 }, // L&       LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
    Interval{ 0x1E73, 0x1E73 }, // L&       LATIN SMALL LETTER U WITH DIAERESIS BELOW
    Interval{ 0x1E75, 0x1E75 }, // L&       LATIN SMALL LETTER U WITH TILDE BELOW
    Interval{ 0x1E77, 0x1E77 }, // L&       LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
    Interval{ 0x1E79, 0x1E79 }, // L&       LATIN SMALL LETTER U WITH TILDE AND ACUTE
    Interval{ 0x1E7B, 0x1E7B }, // L&       LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
    Interval{ 0x1E7D, 0x1E7D }, // L&       LATIN SMALL LETTER V WITH TILDE
    Interval{ 0x1E7F, 0x1E7F }, // L&       LATIN SMALL LETTER V WITH DOT BELOW
    Interval{ 0x1E81, 0x1E81 }, // L&       LATIN SMALL LETTER W WITH GRAVE
    Interval{ 0x1E83, 0x1E83 }, // L&       LATIN SMALL LETTER W WITH ACUTE
    Interval{ 0x1E85, 0x1E85 }, // L&       LATIN SMALL LETTER W WITH DIAERESIS
    Interval{ 0x1E87, 0x1E87 }, // L&       LATIN SMALL LETTER W WITH DOT ABOVE
    Interval{ 0x1E89, 0x1E89 }, // L&       LATIN SMALL LETTER W WITH DOT BELOW
    Interval{ 0x1E8B, 0x1E8B }, // L&       LATIN SMALL LETTER X WITH DOT ABOVE
    Interval{ 0x1E8D, 0x1E8D }, // L&       LATIN SMALL LETTER X WITH DIAERESIS
    Interval{ 0x1E8F, 0x1E8F }, // L&       LATIN SMALL LETTER Y WITH DOT ABOVE
    Interval{ 0x1E91, 0x1E91 }, // L&       LATIN SMALL LETTER Z WITH CIRCUMFLEX
    Interval{ 0x1E93, 0x1E93 }, // L&       LATIN SMALL LETTER Z WITH DOT BELOW
    Interval{ 0x1E95, 0x1E9D }, // L&   [9] LATIN SMALL LETTER Z WITH LINE BELOW..LATIN SMALL LETTER LONG S WITH HIGH STROKE
    Interval{ 0x1E9F, 0x1E9F }, // L&       LATIN SMALL LETTER DELTA
    Interval{ 0x1EA1, 0x1EA1 }, // L&       LATIN SMALL LETTER A WITH DOT BELOW
    Interval{ 0x1EA3, 0x1EA3 }, // L&       LATIN SMALL LETTER A WITH HOOK ABOVE
    Interval{ 0x1EA5, 0x1EA5 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EA7, 0x1EA7 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EA9, 0x1EA9 }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EAB, 0x1EAB }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EAD, 0x1EAD }, // L&       LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EAF, 0x1EAF }, // L&       LATIN SMALL LETTER A WITH BREVE AND ACUTE
    Interval{ 0x1EB1, 0x1EB1 }, // L&       LATIN SMALL LETTER A WITH BREVE AND GRAVE
    Interval{ 0x1EB3, 0x1EB3 }, // L&       LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
    Interval{ 0x1EB5, 0x1EB5 }, // L&       LATIN SMALL LETTER A WITH BREVE AND TILDE
    Interval{ 0x1EB7, 0x1EB7 }, // L&       LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
    Interval{ 0x1EB9, 0x1EB9 }, // L&       LATIN SMALL LETTER E WITH DOT BELOW
    Interval{ 0x1EBB, 0x1EBB }, // L&       LATIN SMALL LETTER E WITH HOOK ABOVE
    Interval{ 0x1EBD, 0x1EBD }, // L&       LATIN SMALL LETTER E WITH TILDE
    Interval{ 0x1EBF, 0x1EBF }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EC1, 0x1EC1 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EC3, 0x1EC3 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EC5, 0x1EC5 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EC7, 0x1EC7 }, // L&       LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EC9, 0x1EC9 }, // L&       LATIN SMALL LETTER I WITH HOOK ABOVE
    Interval{ 0x1ECB, 0x1ECB }, // L&       LATIN SMALL LETTER I WITH DOT BELOW
    Interval{ 0x1ECD, 0x1ECD }, // L&       LATIN SMALL LETTER O WITH DOT BELOW
    Interval{ 0x1ECF, 0x1ECF }, // L&       LATIN SMALL LETTER O WITH HOOK ABOVE
    Interval{ 0x1ED1, 0x1ED1 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1ED3, 0x1ED3 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1ED5, 0x1ED5 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1ED7, 0x1ED7 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1ED9, 0x1ED9 }, // L&       LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EDB, 0x1EDB }, // L&       LATIN SMALL LETTER O WITH HORN AND ACUTE
    Interval{ 0x1EDD, 0x1EDD }, // L&       LATIN SMALL LETTER O WITH HORN AND GRAVE
    Interval{ 0x1EDF, 0x1EDF }, // L&       LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
    Interval{ 0x1EE1, 0x1EE1 }, // L&       LATIN SMALL LETTER O WITH HORN AND TILDE
    Interval{ 0x1EE3, 0x1EE3 }, // L&       LATIN SMALL LETTER O WITH HORN AND DOT BELOW
    Interval{ 0x1EE5, 0x1EE5 }, // L&       LATIN SMALL LETTER U WITH DOT BELOW
    Interval{ 0x1EE7, 0x1EE7 }, // L&       LATIN SMALL LETTER U WITH HOOK ABOVE
    Interval{ 0x1EE9, 0x1EE9 }, // L&       LATIN SMALL LETTER U WITH HORN AND ACUTE
    Interval{ 0x1EEB, 0x1EEB }, // L&       LATIN SMALL LETTER U WITH HORN AND GRAVE
    Interval{ 0x1EED, 0x1EED }, // L&       LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
    Interval{ 0x1EEF, 0x1EEF }, // L&       LATIN SMALL LETTER U WITH HORN AND TILDE
    Interval{ 0x1EF1, 0x1EF1 }, // L&       LATIN SMALL LETTER U WITH HORN AND DOT BELOW
    Interval{ 0x1EF3, 0x1EF3 }, // L&       LATIN SMALL LETTER Y WITH GRAVE
    Interval{ 0x1EF5, 0x1EF5 }, // L&       LATIN SMALL LETTER Y WITH DOT BELOW
    Interval{ 0x1EF7, 0x1EF7 }, // L&       LATIN SMALL LETTER Y WITH HOOK ABOVE
    Interval{ 0x1EF9, 0x1EF9 }, // L&       LATIN SMALL LETTER Y WITH TILDE
    Interval{ 0x1EFB, 0x1EFB }, // L&       LATIN SMALL LETTER MIDDLE-WELSH LL
    Interval{ 0x1EFD, 0x1EFD }, // L&       LATIN SMALL LETTER MIDDLE-WELSH V
    Interval{ 0x1EFF, 0x1F07 }, // L&   [9] LATIN SMALL LETTER Y WITH LOOP..GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F10, 0x1F15 }, // L&   [6] GREEK SMALL LETTER EPSILON WITH PSILI..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F27 }, // L&   [8] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F30, 0x1F37 }, // L&   [8] GREEK SMALL LETTER IOTA WITH PSILI..GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F40, 0x1F45 }, // L&   [6] GREEK SMALL LETTER OMICRON WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F60, 0x1F67 }, // L&   [8] GREEK SMALL LETTER OMEGA WITH PSILI..GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F70, 0x1F7D }, // L&  [14] GREEK SMALL LETTER ALPHA WITH VARIA..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1F87 }, // L&   [8] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1F90, 0x1F97 }, // L&   [8] GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FA0, 0x1FA7 }, // L&   [8] GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FB0, 0x1FB4 }, // L&   [5] GREEK SMALL LETTER ALPHA WITH VRACHY..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FB7 }, // L&   [2] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FC7 }, // L&   [2] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FD7 }, // L&   [2] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
    Interval{ 0x1FE0, 0x1FE7 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FF7 }, // L&   [2] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
    Interval{ 0x2071, 0x2071 }, // Lm       SUPERSCRIPT LATIN SMALL LETTER I
    Interval{ 0x207F, 0x207F }, // Lm       SUPERSCRIPT LATIN SMALL LETTER N
    Interval{ 0x2090, 0x209C }, // Lm  [13] LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCRIPT SMALL LETTER T
    Interval{ 0x210A, 0x210A }, // L&       SCRIPT SMALL G
    Interval{ 0x210E, 0x210F }, // L&   [2] PLANCK CONSTANT..PLANCK CONSTANT OVER TWO PI
    Interval{ 0x2113, 0x2113 }, // L&       SCRIPT SMALL L
    Interval{ 0x212F, 0x212F }, // L&       SCRIPT SMALL E
    Interval{ 0x2134, 0x2134 }, // L&       SCRIPT SMALL O
    Interval{ 0x2139, 0x2139 }, // L&       INFORMATION SOURCE
    Interval{ 0x213C, 0x213D }, // L&   [2] DOUBLE-STRUCK SMALL PI..DOUBLE-STRUCK SMALL GAMMA
    Interval{ 0x2146, 0x2149 }, // L&   [4] DOUBLE-STRUCK ITALIC SMALL D..DOUBLE-STRUCK ITALIC SMALL J
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2170, 0x217F }, // Nl  [16] SMALL ROMAN NUMERAL ONE..SMALL ROMAN NUMERAL ONE THOUSAND
    Interval{ 0x2184, 0x2184 }, // L&       LATIN SMALL LETTER REVERSED C
    Interval{ 0x24D0, 0x24E9 }, // So  [26] CIRCLED LATIN SMALL LETTER A..CIRCLED LATIN SMALL LETTER Z
    Interval{ 0x2C30, 0x2C5F }, // L&  [48] GLAGOLITIC SMALL LETTER AZU..GLAGOLITIC SMALL LETTER CAUDATE CHRIVI
    Interval{ 0x2C61, 0x2C61 }, // L&       LATIN SMALL LETTER L WITH DOUBLE BAR
    Interval{ 0x2C65, 0x2C66 }, // L&   [2] LATIN SMALL LETTER A WITH STROKE..LATIN SMALL LETTER T WITH DIAGONAL STROKE
    Interval{ 0x2C68, 0x2C68 }, // L&       LATIN SMALL LETTER H WITH DESCENDER
    Interval{ 0x2C6A, 0x2C6A }, // L&       LATIN SMALL LETTER K WITH DESCENDER
    Interval{ 0x2C6C, 0x2C6C }, // L&       LATIN SMALL LETTER Z WITH DESCENDER
    Interval{ 0x2C71, 0x2C71 }, // L&       LATIN SMALL LETTER V WITH RIGHT HOOK
    Interval{ 0x2C73, 0x2C74 }, // L&   [2] LATIN SMALL LETTER W WITH HOOK..LATIN SMALL LETTER V WITH CURL
    Interval{ 0x2C76, 0x2C7B }, // L&   [6] LATIN SMALL LETTER HALF H..LATIN LETTER SMALL CAPITAL TURNED E
    Interval{ 0x2C7C, 0x2C7D }, // Lm   [2] LATIN SUBSCRIPT SMALL LETTER J..MODIFIER LETTER CAPITAL V
    Interval{ 0x2C81, 0x2C81 }, // L&       COPTIC SMALL LETTER ALFA
    Interval{ 0x2C83, 0x2C83 }, // L&       COPTIC SMALL LETTER VIDA
    Interval{ 0x2C85, 0x2C85 }, // L&       COPTIC SMALL LETTER GAMMA
    Interval{ 0x2C87, 0x2C87 }, // L&       COPTIC SMALL LETTER DALDA
    Interval{ 0x2C89, 0x2C89 }, // L&       COPTIC SMALL LETTER EIE
    Interval{ 0x2C8B, 0x2C8B }, // L&       COPTIC SMALL LETTER SOU
    Interval{ 0x2C8D, 0x2C8D }, // L&       COPTIC SMALL LETTER ZATA
    Interval{ 0x2C8F, 0x2C8F }, // L&       COPTIC SMALL LETTER HATE
    Interval{ 0x2C91, 0x2C91 }, // L&       COPTIC SMALL LETTER THETHE
    Interval{ 0x2C93, 0x2C93 }, // L&       COPTIC SMALL LETTER IAUDA
    Interval{ 0x2C95, 0x2C95 }, // L&       COPTIC SMALL LETTER KAPA
    Interval{ 0x2C97, 0x2C97 }, // L&       COPTIC SMALL LETTER LAULA
    Interval{ 0x2C99, 0x2C99 }, // L&       COPTIC SMALL LETTER MI
    Interval{ 0x2C9B, 0x2C9B }, // L&       COPTIC SMALL LETTER NI
    Interval{ 0x2C9D, 0x2C9D }, // L&       COPTIC SMALL LETTER KSI
    Interval{ 0x2C9F, 0x2C9F }, // L&       COPTIC SMALL LETTER O
    Interval{ 0x2CA1, 0x2CA1 }, // L&       COPTIC SMALL LETTER PI
    Interval{ 0x2CA3, 0x2CA3 }, // L&       COPTIC SMALL LETTER RO
    Interval{ 0x2CA5, 0x2CA5 }, // L&       COPTIC SMALL LETTER SIMA
    Interval{ 0x2CA7, 0x2CA7 }, // L&       COPTIC SMALL LETTER TAU
    Interval{ 0x2CA9, 0x2CA9 }, // L&       COPTIC SMALL LETTER UA
    Interval{ 0x2CAB, 0x2CAB }, // L&       COPTIC SMALL LETTER FI
    Interval{ 0x2CAD, 0x2CAD }, // L&       COPTIC SMALL LETTER KHI
    Interval{ 0x2CAF, 0x2CAF }, // L&       COPTIC SMALL LETTER PSI
    Interval{ 0x2CB1, 0x2CB1 }, // L&       COPTIC SMALL LETTER OOU
    Interval{ 0x2CB3, 0x2CB3 }, // L&       COPTIC SMALL LETTER DIALECT-P ALEF
    Interval{ 0x2CB5, 0x2CB5 }, // L&       COPTIC SMALL LETTER OLD COPTIC AIN
    Interval{ 0x2CB7, 0x2CB7 }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC EIE
    Interval{ 0x2CB9, 0x2CB9 }, // L&       COPTIC SMALL LETTER DIALECT-P KAPA
    Interval{ 0x2CBB, 0x2CBB }, // L&       COPTIC SMALL LETTER DIALECT-P NI
    Interval{ 0x2CBD, 0x2CBD }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC NI
    Interval{ 0x2CBF, 0x2CBF }, // L&       COPTIC SMALL LETTER OLD COPTIC OOU
    Interval{ 0x2CC1, 0x2CC1 }, // L&       COPTIC SMALL LETTER SAMPI
    Interval{ 0x2CC3, 0x2CC3 }, // L&       COPTIC SMALL LETTER CROSSED SHEI
    Interval{ 0x2CC5, 0x2CC5 }, // L&       COPTIC SMALL LETTER OLD COPTIC SHEI
    Interval{ 0x2CC7, 0x2CC7 }, // L&       COPTIC SMALL LETTER OLD COPTIC ESH
    Interval{ 0x2CC9, 0x2CC9 }, // L&       COPTIC SMALL LETTER AKHMIMIC KHEI
    Interval{ 0x2CCB, 0x2CCB }, // L&       COPTIC SMALL LETTER DIALECT-P HORI
    Interval{ 0x2CCD, 0x2CCD }, // L&       COPTIC SMALL LETTER OLD COPTIC HORI
    Interval{ 0x2CCF, 0x2CCF }, // L&       COPTIC SMALL LETTER OLD COPTIC HA
    Interval{ 0x2CD1, 0x2CD1 }, // L&       COPTIC SMALL LETTER L-SHAPED HA
    Interval{ 0x2CD3, 0x2CD3 }, // L&       COPTIC SMALL LETTER OLD COPTIC HEI
    Interval{ 0x2CD5, 0x2CD5 }, // L&       COPTIC SMALL LETTER OLD COPTIC HAT
    Interval{ 0x2CD7, 0x2CD7 }, // L&       COPTIC SMALL LETTER OLD COPTIC GANGIA
    Interval{ 0x2CD9, 0x2CD9 }, // L&       COPTIC SMALL LETTER OLD COPTIC DJA
    Interval{ 0x2CDB, 0x2CDB }, // L&       COPTIC SMALL LETTER OLD COPTIC SHIMA
    Interval{ 0x2CDD, 0x2CDD }, // L&       COPTIC SMALL LETTER OLD NUBIAN SHIMA
    Interval{ 0x2CDF, 0x2CDF }, // L&       COPTIC SMALL LETTER OLD NUBIAN NGI
    Interval{ 0x2CE1, 0x2CE1 }, // L&       COPTIC SMALL LETTER OLD NUBIAN NYI
    Interval{ 0x2CE3, 0x2CE4 }, // L&   [2] COPTIC SMALL LETTER OLD NUBIAN WAU..COPTIC SYMBOL KAI
    Interval{ 0x2CEC, 0x2CEC }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC SHEI
    Interval{ 0x2CEE, 0x2CEE }, // L&       COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF3, 0x2CF3 }, // L&       COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0xA641, 0xA641 }, // L&       CYRILLIC SMALL LETTER ZEMLYA
    Interval{ 0xA643, 0xA643 }, // L&       CYRILLIC SMALL LETTER DZELO
    Interval{ 0xA645, 0xA645 }, // L&       CYRILLIC SMALL LETTER REVERSED DZE
    Interval{ 0xA647, 0xA647 }, // L&       CYRILLIC SMALL LETTER IOTA
    Interval{ 0xA649, 0xA649 }, // L&       CYRILLIC SMALL LETTER DJERV
    Interval{ 0xA64B, 0xA64B }, // L&       CYRILLIC SMALL LETTER MONOGRAPH UK
    Interval{ 0xA64D, 0xA64D }, // L&       CYRILLIC SMALL LETTER BROAD OMEGA
    Interval{ 0xA64F, 0xA64F }, // L&       CYRILLIC SMALL LETTER NEUTRAL YER
    Interval{ 0xA651, 0xA651 }, // L&       CYRILLIC SMALL LETTER YERU WITH BACK YER
    Interval{ 0xA653, 0xA653 }, // L&       CYRILLIC SMALL LETTER IOTIFIED YAT
    Interval{ 0xA655, 0xA655 }, // L&       CYRILLIC SMALL LETTER REVERSED YU
    Interval{ 0xA657, 0xA657 }, // L&       CYRILLIC SMALL LETTER IOTIFIED A
    Interval{ 0xA659, 0xA659 }, // L&       CYRILLIC SMALL LETTER CLOSED LITTLE YUS
    Interval{ 0xA65B, 0xA65B }, // L&       CYRILLIC SMALL LETTER BLENDED YUS
    Interval{ 0xA65D, 0xA65D }, // L&       CYRILLIC SMALL LETTER IOTIFIED CLOSED LITTLE YUS
    Interval{ 0xA65F, 0xA65F }, // L&       CYRILLIC SMALL LETTER YN
    Interval{ 0xA661, 0xA661 }, // L&       CYRILLIC SMALL LETTER REVERSED TSE
    Interval{ 0xA663, 0xA663 }, // L&       CYRILLIC SMALL LETTER SOFT DE
    Interval{ 0xA665, 0xA665 }, // L&       CYRILLIC SMALL LETTER SOFT EL
    Interval{ 0xA667, 0xA667 }, // L&       CYRILLIC SMALL LETTER SOFT EM
    Interval{ 0xA669, 0xA669 }, // L&       CYRILLIC SMALL LETTER MONOCULAR O
    Interval{ 0xA66B, 0xA66B }, // L&       CYRILLIC SMALL LETTER BINOCULAR O
    Interval{ 0xA66D, 0xA66D }, // L&       CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA681, 0xA681 }, // L&       CYRILLIC SMALL LETTER DWE
    Interval{ 0xA683, 0xA683 }, // L&       CYRILLIC SMALL LETTER DZWE
    Interval{ 0xA685, 0xA685 }, // L&       CYRILLIC SMALL LETTER ZHWE
    Interval{ 0xA687, 0xA687 }, // L&       CYRILLIC SMALL LETTER CCHE
    Interval{ 0xA689, 0xA689 }, // L&       CYRILLIC SMALL LETTER DZZE
    Interval{ 0xA68B, 0xA68B }, // L&       CYRILLIC SMALL LETTER TE WITH MIDDLE HOOK
    Interval{ 0xA68D, 0xA68D }, // L&       CYRILLIC SMALL LETTER TWE
    Interval{ 0xA68F, 0xA68F }, // L&       CYRILLIC SMALL LETTER TSWE
    Interval{ 0xA691, 0xA691 }, // L&       CYRILLIC SMALL LETTER TSSE
    Interval{ 0xA693, 0xA693 }, // L&       CYRILLIC SMALL LETTER TCHE
    Interval{ 0xA695, 0xA695 }, // L&       CYRILLIC SMALL LETTER HWE
    Interval{ 0xA697, 0xA697 }, // L&       CYRILLIC SMALL LETTER SHWE
    Interval{ 0xA699, 0xA699 }, // L&       CYRILLIC SMALL LETTER DOUBLE O
    Interval{ 0xA69B, 0xA69B }, // L&       CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA69C, 0xA69D }, // Lm   [2] MODIFIER LETTER CYRILLIC HARD SIGN..MODIFIER LETTER CYRILLIC SOFT SIGN
    Interval{ 0xA723, 0xA723 }, // L&       LATIN SMALL LETTER EGYPTOLOGICAL ALEF
    Interval{ 0xA725, 0xA725 }, // L&       LATIN SMALL LETTER EGYPTOLOGICAL AIN
    Interval{ 0xA727, 0xA727 }, // L&       LATIN SMALL LETTER HENG
    Interval{ 0xA729, 0xA729 }, // L&       LATIN SMALL LETTER TZ
    Interval{ 0xA72B, 0xA72B }, // L&       LATIN SMALL LETTER TRESILLO
    Interval{ 0xA72D, 0xA72D }, // L&       LATIN SMALL LETTER CUATRILLO
    Interval{ 0xA72F, 0xA731 }, // L&   [3] LATIN SMALL LETTER CUATRILLO WITH COMMA..LATIN LETTER SMALL CAPITAL S
    Interval{ 0xA733, 0xA733 }, // L&       LATIN SMALL LETTER AA
    Interval{ 0xA735, 0xA735 }, // L&       LATIN SMALL LETTER AO
    Interval{ 0xA737, 0xA737 }, // L&       LATIN SMALL LETTER AU
    Interval{ 0xA739, 0xA739 }, // L&       LATIN SMALL LETTER AV
    Interval{ 0xA73B, 0xA73B }, // L&       LATIN SMALL LETTER AV WITH HORIZONTAL BAR
    Interval{ 0xA73D, 0xA73D }, // L&       LATIN SMALL LETTER AY
    Interval{ 0xA73F, 0xA73F }, // L&       LATIN SMALL LETTER REVERSED C WITH DOT
    Interval{ 0xA741, 0xA741 }, // L&       LATIN SMALL LETTER K WITH STROKE
    Interval{ 0xA743, 0xA743 }, // L&       LATIN SMALL LETTER K WITH DIAGONAL STROKE
    Interval{ 0xA745, 0xA745 }, // L&       LATIN SMALL LETTER K WITH STROKE AND DIAGONAL STROKE
    Interval{ 0xA747, 0xA747 }, // L&       LATIN SMALL LETTER BROKEN L
    Interval{ 0xA749, 0xA749 }, // L&       LATIN SMALL LETTER L WITH HIGH STROKE
    Interval{ 0xA74B, 0xA74B }, // L&       LATIN SMALL LETTER O WITH LONG STROKE OVERLAY
    Interval{ 0xA74D, 0xA74D }, // L&       LATIN SMALL LETTER O WITH LOOP
    Interval{ 0xA74F, 0xA74F }, // L&       LATIN SMALL LETTER OO
    Interval{ 0xA751, 0xA751 }, // L&       LATIN SMALL LETTER P WITH STROKE THROUGH DESCENDER
    Interval{ 0xA753, 0xA753 }, // L&       LATIN SMALL LETTER P WITH FLOURISH
    Interval{ 0xA755, 0xA755 }, // L&       LATIN SMALL LETTER P WITH SQUIRREL TAIL
    Interval{ 0xA757, 0xA757 }, // L&       LATIN SMALL LETTER Q WITH STROKE THROUGH DESCENDER
    Interval{ 0xA759, 0xA759 }, // L&       LATIN SMALL LETTER Q WITH DIAGONAL STROKE
    Interval{ 0xA75B, 0xA75B }, // L&       LATIN SMALL LETTER R ROTUNDA
    Interval{ 0xA75D, 0xA75D }, // L&       LATIN SMALL LETTER RUM ROTUNDA
    Interval{ 0xA75F, 0xA75F }, // L&       LATIN SMALL LETTER V WITH DIAGONAL STROKE
    Interval{ 0xA761, 0xA761 }, // L&       LATIN SMALL LETTER VY
    Interval{ 0xA763, 0xA763 }, // L&       LATIN SMALL LETTER VISIGOTHIC Z
    Interval{ 0xA765, 0xA765 }, // L&       LATIN SMALL LETTER THORN WITH STROKE
    Interval{ 0xA767, 0xA767 }, // L&       LATIN SMALL LETTER THORN WITH STROKE THROUGH DESCENDER
    Interval{ 0xA769, 0xA769 }, // L&       LATIN SMALL LETTER VEND
    Interval{ 0xA76B, 0xA76B }, // L&       LATIN SMALL LETTER ET
    Interval{ 0xA76D, 0xA76D }, // L&       LATIN SMALL LETTER IS
    Interval{ 0xA76F, 0xA76F }, // L&       LATIN SMALL LETTER CON
    Interval{ 0xA770, 0xA770 }, // Lm       MODIFIER LETTER US
    Interval{ 0xA771, 0xA778 }, // L&   [8] LATIN SMALL LETTER DUM..LATIN SMALL LETTER UM
    Interval{ 0xA77A, 0xA77A }, // L&       LATIN SMALL LETTER INSULAR D
    Interval{ 0xA77C, 0xA77C }, // L&       LATIN SMALL LETTER INSULAR F
    Interval{ 0xA77F, 0xA77F }, // L&       LATIN SMALL LETTER TURNED INSULAR G
    Interval{ 0xA781, 0xA781 }, // L&       LATIN SMALL LETTER TURNED L
    Interval{ 0xA783, 0xA783 }, // L&       LATIN SMALL LETTER INSULAR R
    Interval{ 0xA785, 0xA785 }, // L&       LATIN SMALL LETTER INSULAR S
    Interval{ 0xA787, 0xA787 }, // L&       LATIN SMALL LETTER INSULAR T
    Interval{ 0xA78C, 0xA78C }, // L&       LATIN SMALL LETTER SALTILLO
    Interval{ 0xA78E, 0xA78E }, // L&       LATIN SMALL LETTER L WITH RETROFLEX HOOK AND BELT
    Interval{ 0xA791, 0xA791 }, // L&       LATIN SMALL LETTER N WITH DESCENDER
    Interval{ 0xA793, 0xA795 }, // L&   [3] LATIN SMALL LETTER C WITH BAR..LATIN SMALL LETTER H WITH PALATAL HOOK
    Interval{ 0xA797, 0xA797 }, // L&       LATIN SMALL LETTER B WITH FLOURISH
    Interval{ 0xA799, 0xA799 }, // L&       LATIN SMALL LETTER F WITH STROKE
    Interval{ 0xA79B, 0xA79B }, // L&       LATIN SMALL LETTER VOLAPUK AE
    Interval{ 0xA79D, 0xA79D }, // L&       LATIN SMALL LETTER VOLAPUK OE
    Interval{ 0xA79F, 0xA79F }, // L&       LATIN SMALL LETTER VOLAPUK UE
    Interval{ 0xA7A1, 0xA7A1 }, // L&       LATIN SMALL LETTER G WITH OBLIQUE STROKE
    Interval{ 0xA7A3, 0xA7A3 }, // L&       LATIN SMALL LETTER K WITH OBLIQUE STROKE
    Interval{ 0xA7A5, 0xA7A5 }, // L&       LATIN SMALL LETTER N WITH OBLIQUE STROKE
    Interval{ 0xA7A7, 0xA7A7 }, // L&       LATIN SMALL LETTER R WITH OBLIQUE STROKE
    Interval{ 0xA7A9, 0xA7A9 }, // L&       LATIN SMALL LETTER S WITH OBLIQUE STROKE
    Interval{ 0xA7AF, 0xA7AF }, // L&       LATIN LETTER SMALL CAPITAL Q
    Interval{ 0xA7B5, 0xA7B5 }, // L&       LATIN SMALL LETTER BETA
    Interval{ 0xA7B7, 0xA7B7 }, // L&       LATIN SMALL LETTER OMEGA
    Interval{ 0xA7B9, 0xA7B9 }, // L&       LATIN SMALL LETTER U WITH STROKE
    Interval{ 0xA7BB, 0xA7BB }, // L&       LATIN SMALL LETTER GLOTTAL A
    Interval{ 0xA7BD, 0xA7BD }, // L&       LATIN SMALL LETTER GLOTTAL I
    Interval{ 0xA7BF, 0xA7BF }, // L&       LATIN SMALL LETTER GLOTTAL U
    Interval{ 0xA7C1, 0xA7C1 }, // L&       LATIN SMALL LETTER OLD POLISH O
    Interval{ 0xA7C3, 0xA7C3 }, // L&       LATIN SMALL LETTER ANGLICANA W
    Interval{ 0xA7C8, 0xA7C8 }, // L&       LATIN SMALL LETTER D WITH SHORT STROKE OVERLAY
    Interval{ 0xA7CA, 0xA7CA }, // L&       LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D1, 0xA7D1 }, // L&       LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D3, 0xA7D3 }, // L&       LATIN SMALL LETTER DOUBLE THORN
    Interval{ 0xA7D5, 0xA7D5 }, // L&       LATIN SMALL LETTER DOUBLE WYNN
    Interval{ 0xA7D7, 0xA7D7 }, // L&       LATIN SMALL LETTER MIDDLE SCOTS S
    Interval{ 0xA7D9, 0xA7D9 }, // L&       LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F2, 0xA7F4 }, // Lm   [3] MODIFIER LETTER CAPITAL C..MODIFIER LETTER CAPITAL Q
    Interval{ 0xA7F6, 0xA7F6 }, // L&       LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xA7F8, 0xA7F9 }, // Lm   [2] MODIFIER LETTER CAPITAL H WITH STROKE..MODIFIER LETTER SMALL LIGATURE OE
    Interval{ 0xA7FA, 0xA7FA }, // L&       LATIN LETTER SMALL CAPITAL TURNED M
    Interval{ 0xAB30, 0xAB5A }, // L&  [43] LATIN SMALL LETTER BARRED ALPHA..LATIN SMALL LETTER Y WITH SHORT RIGHT LEG
    Interval{ 0xAB5C, 0xAB5F }, // Lm   [4] MODIFIER LETTER SMALL HENG..MODIFIER LETTER SMALL U WITH LEFT HOOK
    Interval{ 0xAB60, 0xAB68 }, // L&   [9] LATIN SMALL LETTER SAKHA YAT..LATIN SMALL LETTER TURNED R WITH MIDDLE TILDE
    Interval{ 0xAB69, 0xAB69 }, // Lm       MODIFIER LETTER SMALL TURNED W
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0x10428, 0x1044F }, // L&  [40] DESERET SMALL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10780, 0x10780 }, // Lm       MODIFIER LETTER SMALL CAPITAL AA
    Interval{ 0x10783, 0x10785 }, // Lm   [3] MODIFIER LETTER SMALL AE..MODIFIER LETTER SMALL B WITH HOOK
    Interval{ 0x10787, 0x107B0 }, // Lm  [42] MODIFIER LETTER SMALL DZ DIGRAPH..MODIFIER LETTER SMALL V WITH RIGHT HOOK
    Interval{ 0x107B2, 0x107BA }, // Lm   [9] MODIFIER LETTER SMALL CAPITAL Y..MODIFIER LETTER SMALL S WITH CURL
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x118C0, 0x118DF }, // L&  [32] WARANG CITI SMALL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x16E60, 0x16E7F }, // L&  [32] MEDEFAIDRIN SMALL LETTER M..MEDEFAIDRIN SMALL LETTER Y
    Interval{ 0x1D41A, 0x1D433 }, // L&  [26] MATHEMATICAL BOLD SMALL A..MATHEMATICAL BOLD SMALL Z
    Interval{ 0x1D44E, 0x1D454 }, // L&   [7] MATHEMATICAL ITALIC SMALL A..MATHEMATICAL ITALIC SMALL G
    Interval{ 0x1D456, 0x1D467 }, // L&  [18] MATHEMATICAL ITALIC SMALL I..MATHEMATICAL ITALIC SMALL Z
    Interval{ 0x1D482, 0x1D49B }, // L&  [26] MATHEMATICAL BOLD ITALIC SMALL A..MATHEMATICAL BOLD ITALIC SMALL Z
    Interval{ 0x1D4B6, 0x1D4B9 }, // L&   [4] MATHEMATICAL SCRIPT SMALL A..MATHEMATICAL SCRIPT SMALL D
    Interval{ 0x1D4BB, 0x1D4BB }, // L&       MATHEMATICAL SCRIPT SMALL F
    Interval{ 0x1D4BD, 0x1D4C3 }, // L&   [7] MATHEMATICAL SCRIPT SMALL H..MATHEMATICAL SCRIPT SMALL N
    Interval{ 0x1D4C5, 0x1D4CF }, // L&  [11] MATHEMATICAL SCRIPT SMALL P..MATHEMATICAL SCRIPT SMALL Z
    Interval{ 0x1D4EA, 0x1D503 }, // L&  [26] MATHEMATICAL BOLD SCRIPT SMALL A..MATHEMATICAL BOLD SCRIPT SMALL Z
    Interval{ 0x1D51E, 0x1D537 }, // L&  [26] MATHEMATICAL FRAKTUR SMALL A..MATHEMATICAL FRAKTUR SMALL Z
    Interval{ 0x1D552, 0x1D56B }, // L&  [26] MATHEMATICAL DOUBLE-STRUCK SMALL A..MATHEMATICAL DOUBLE-STRUCK SMALL Z
    Interval{ 0x1D586, 0x1D59F }, // L&  [26] MATHEMATICAL BOLD FRAKTUR SMALL A..MATHEMATICAL BOLD FRAKTUR SMALL Z
    Interval{ 0x1D5BA, 0x1D5D3 }, // L&  [26] MATHEMATICAL SANS-SERIF SMALL A..MATHEMATICAL SANS-SERIF SMALL Z
    Interval{ 0x1D5EE, 0x1D607 }, // L&  [26] MATHEMATICAL SANS-SERIF BOLD SMALL A..MATHEMATICAL SANS-SERIF BOLD SMALL Z
    Interval{ 0x1D622, 0x1D63B }, // L&  [26] MATHEMATICAL SANS-SERIF ITALIC SMALL A..MATHEMATICAL SANS-SERIF ITALIC SMALL Z
    Interval{ 0x1D656, 0x1D66F }, // L&  [26] MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL A..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL Z
    Interval{ 0x1D68A, 0x1D6A5 }, // L&  [28] MATHEMATICAL MONOSPACE SMALL A..MATHEMATICAL ITALIC SMALL DOTLESS J
    Interval{ 0x1D6C2, 0x1D6DA }, // L&  [25] MATHEMATICAL BOLD SMALL ALPHA..MATHEMATICAL BOLD SMALL OMEGA
    Interval{ 0x1D6DC, 0x1D6E1 }, // L&   [6] MATHEMATICAL BOLD EPSILON SYMBOL..MATHEMATICAL BOLD PI SYMBOL
    Interval{ 0x1D6FC, 0x1D714 }, // L&  [25] MATHEMATICAL ITALIC SMALL ALPHA..MATHEMATICAL ITALIC SMALL OMEGA
    Interval{ 0x1D716, 0x1D71B }, // L&   [6] MATHEMATICAL ITALIC EPSILON SYMBOL..MATHEMATICAL ITALIC PI SYMBOL
    Interval{ 0x1D736, 0x1D74E }, // L&  [25] MATHEMATICAL BOLD ITALIC SMALL ALPHA..MATHEMATICAL BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D750, 0x1D755 }, // L&   [6] MATHEMATICAL BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD ITALIC PI SYMBOL
    Interval{ 0x1D770, 0x1D788 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD SMALL OMEGA
    Interval{ 0x1D78A, 0x1D78F }, // L&   [6] MATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD PI SYMBOL
    Interval{ 0x1D7AA, 0x1D7C2 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D7C4, 0x1D7C9 }, // L&   [6] MATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD ITALIC PI SYMBOL
    Interval{ 0x1D7CB, 0x1D7CB }, // L&       MATHEMATICAL BOLD SMALL DIGAMMA
    Interval{ 0x1DF00, 0x1DF09 }, // L&  [10] LATIN SMALL LETTER FENG DIGRAPH WITH TRILL..LATIN SMALL LETTER T WITH HOOK AND RETROFLEX HOOK
    Interval{ 0x1DF0B, 0x1DF1E }, // L&  [20] LATIN SMALL LETTER ESH WITH DOUBLE BAR..LATIN SMALL LETTER S WITH CURL
    Interval{ 0x1DF25, 0x1DF2A }, // L&   [6] LATIN SMALL LETTER D WITH MID-HEIGHT LEFT HOOK..LATIN SMALL LETTER T WITH MID-HEIGHT LEFT HOOK
    Interval{ 0x1E030, 0x1E06D }, // Lm  [62] MODIFIER LETTER CYRILLIC SMALL A..MODIFIER LETTER CYRILLIC SMALL STRAIGHT U WITH STROKE
    Interval{ 0x1E922, 0x1E943 }, // L&  [34] ADLAM SMALL LETTER ALIF..ADLAM SMALL LETTER SHA
}; // }}}
auto static const Math = std::array<Interval, 240>{ // {{{
    Interval{ 0x002B, 0x002B }, // Sm       PLUS SIGN
    Interval{ 0x003C, 0x003E }, // Sm   [3] LESS-THAN SIGN..GREATER-THAN SIGN
    Interval{ 0x005E, 0x005E }, // Sk       CIRCUMFLEX ACCENT
    Interval{ 0x007C, 0x007C }, // Sm       VERTICAL LINE
    Interval{ 0x007E, 0x007E }, // Sm       TILDE
    Interval{ 0x00AC, 0x00AC }, // Sm       NOT SIGN
    Interval{ 0x00B1, 0x00B1 }, // Sm       PLUS-MINUS SIGN
    Interval{ 0x00D7, 0x00D7 }, // Sm       MULTIPLICATION SIGN
    Interval{ 0x00F7, 0x00F7 }, // Sm       DIVISION SIGN
    Interval{ 0x03D0, 0x03D2 }, // L&   [3] GREEK BETA SYMBOL..GREEK UPSILON WITH HOOK SYMBOL
    Interval{ 0x03D5, 0x03D5 }, // L&       GREEK PHI SYMBOL
    Interval{ 0x03F0, 0x03F1 }, // L&   [2] GREEK KAPPA SYMBOL..GREEK RHO SYMBOL
    Interval{ 0x03F4, 0x03F5 }, // L&   [2] GREEK CAPITAL THETA SYMBOL..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F6, 0x03F6 }, // Sm       GREEK REVERSED LUNATE EPSILON SYMBOL
    Interval{ 0x0606, 0x0608 }, // Sm   [3] ARABIC-INDIC CUBE ROOT..ARABIC RAY
    Interval{ 0x2016, 0x2016 }, // Po       DOUBLE VERTICAL LINE
    Interval{ 0x2032, 0x2034 }, // Po   [3] PRIME..TRIPLE PRIME
    Interval{ 0x2040, 0x2040 }, // Pc       CHARACTER TIE
    Interval{ 0x2044, 0x2044 }, // Sm       FRACTION SLASH
    Interval{ 0x2052, 0x2052 }, // Sm       COMMERCIAL MINUS SIGN
    Interval{ 0x2061, 0x2064 }, // Cf   [4] FUNCTION APPLICATION..INVISIBLE PLUS
    Interval{ 0x207A, 0x207C }, // Sm   [3] SUPERSCRIPT PLUS SIGN..SUPERSCRIPT EQUALS SIGN
    Interval{ 0x207D, 0x207D }, // Ps       SUPERSCRIPT LEFT PARENTHESIS
    Interval{ 0x207E, 0x207E }, // Pe       SUPERSCRIPT RIGHT PARENTHESIS
    Interval{ 0x208A, 0x208C }, // Sm   [3] SUBSCRIPT PLUS SIGN..SUBSCRIPT EQUALS SIGN
    Interval{ 0x208D, 0x208D }, // Ps       SUBSCRIPT LEFT PARENTHESIS
    Interval{ 0x208E, 0x208E }, // Pe       SUBSCRIPT RIGHT PARENTHESIS
    Interval{ 0x20D0, 0x20DC }, // Mn  [13] COMBINING LEFT HARPOON ABOVE..COMBINING FOUR DOTS ABOVE
    Interval{ 0x20E1, 0x20E1 }, // Mn       COMBINING LEFT RIGHT ARROW ABOVE
    Interval{ 0x20E5, 0x20E6 }, // Mn   [2] COMBINING REVERSE SOLIDUS OVERLAY..COMBINING DOUBLE VERTICAL STROKE OVERLAY
    Interval{ 0x20EB, 0x20EF }, // Mn   [5] COMBINING LONG DOUBLE SOLIDUS OVERLAY..COMBINING RIGHT ARROW BELOW
    Interval{ 0x2102, 0x2102 }, // L&       DOUBLE-STRUCK CAPITAL C
    Interval{ 0x2107, 0x2107 }, // L&       EULER CONSTANT
    Interval{ 0x210A, 0x2113 }, // L&  [10] SCRIPT SMALL G..SCRIPT SMALL L
    Interval{ 0x2115, 0x2115 }, // L&       DOUBLE-STRUCK CAPITAL N
    Interval{ 0x2118, 0x2118 }, // Sm       SCRIPT CAPITAL P
    Interval{ 0x2119, 0x211D }, // L&   [5] DOUBLE-STRUCK CAPITAL P..DOUBLE-STRUCK CAPITAL R
    Interval{ 0x2124, 0x2124 }, // L&       DOUBLE-STRUCK CAPITAL Z
    Interval{ 0x2128, 0x2128 }, // L&       BLACK-LETTER CAPITAL Z
    Interval{ 0x2129, 0x2129 }, // So       TURNED GREEK SMALL LETTER IOTA
    Interval{ 0x212C, 0x212D }, // L&   [2] SCRIPT CAPITAL B..BLACK-LETTER CAPITAL C
    Interval{ 0x212F, 0x2131 }, // L&   [3] SCRIPT SMALL E..SCRIPT CAPITAL F
    Interval{ 0x2133, 0x2134 }, // L&   [2] SCRIPT CAPITAL M..SCRIPT SMALL O
    Interval{ 0x2135, 0x2138 }, // Lo   [4] ALEF SYMBOL..DALET SYMBOL
    Interval{ 0x213C, 0x213F }, // L&   [4] DOUBLE-STRUCK SMALL PI..DOUBLE-STRUCK CAPITAL PI
    Interval{ 0x2140, 0x2144 }, // Sm   [5] DOUBLE-STRUCK N-ARY SUMMATION..TURNED SANS-SERIF CAPITAL Y
    Interval{ 0x2145, 0x2149 }, // L&   [5] DOUBLE-STRUCK ITALIC CAPITAL D..DOUBLE-STRUCK ITALIC SMALL J
    Interval{ 0x214B, 0x214B }, // Sm       TURNED AMPERSAND
    Interval{ 0x2190, 0x2194 }, // Sm   [5] LEFTWARDS ARROW..LEFT RIGHT ARROW
    Interval{ 0x2195, 0x2199 }, // So   [5] UP DOWN ARROW..SOUTH WEST ARROW
    Interval{ 0x219A, 0x219B }, // Sm   [2] LEFTWARDS ARROW WITH STROKE..RIGHTWARDS ARROW WITH STROKE
    Interval{ 0x219C, 0x219F }, // So   [4] LEFTWARDS WAVE ARROW..UPWARDS TWO HEADED ARROW
    Interval{ 0x21A0, 0x21A0 }, // Sm       RIGHTWARDS TWO HEADED ARROW
    Interval{ 0x21A1, 0x21A2 }, // So   [2] DOWNWARDS TWO HEADED ARROW..LEFTWARDS ARROW WITH TAIL
    Interval{ 0x21A3, 0x21A3 }, // Sm       RIGHTWARDS ARROW WITH TAIL
    Interval{ 0x21A4, 0x21A5 }, // So   [2] LEFTWARDS ARROW FROM BAR..UPWARDS ARROW FROM BAR
    Interval{ 0x21A6, 0x21A6 }, // Sm       RIGHTWARDS ARROW FROM BAR
    Interval{ 0x21A7, 0x21A7 }, // So       DOWNWARDS ARROW FROM BAR
    Interval{ 0x21A9, 0x21AD }, // So   [5] LEFTWARDS ARROW WITH HOOK..LEFT RIGHT WAVE ARROW
    Interval{ 0x21AE, 0x21AE }, // Sm       LEFT RIGHT ARROW WITH STROKE
    Interval{ 0x21B0, 0x21B1 }, // So   [2] UPWARDS ARROW WITH TIP LEFTWARDS..UPWARDS ARROW WITH TIP RIGHTWARDS
    Interval{ 0x21B6, 0x21B7 }, // So   [2] ANTICLOCKWISE TOP SEMICIRCLE ARROW..CLOCKWISE TOP SEMICIRCLE ARROW
    Interval{ 0x21BC, 0x21CD }, // So  [18] LEFTWARDS HARPOON WITH BARB UPWARDS..LEFTWARDS DOUBLE ARROW WITH STROKE
    Interval{ 0x21CE, 0x21CF }, // Sm   [2] LEFT RIGHT DOUBLE ARROW WITH STROKE..RIGHTWARDS DOUBLE ARROW WITH STROKE
    Interval{ 0x21D0, 0x21D1 }, // So   [2] LEFTWARDS DOUBLE ARROW..UPWARDS DOUBLE ARROW
    Interval{ 0x21D2, 0x21D2 }, // Sm       RIGHTWARDS DOUBLE ARROW
    Interval{ 0x21D3, 0x21D3 }, // So       DOWNWARDS DOUBLE ARROW
    Interval{ 0x21D4, 0x21D4 }, // Sm       LEFT RIGHT DOUBLE ARROW
    Interval{ 0x21D5, 0x21DB }, // So   [7] UP DOWN DOUBLE ARROW..RIGHTWARDS TRIPLE ARROW
    Interval{ 0x21DD, 0x21DD }, // So       RIGHTWARDS SQUIGGLE ARROW
    Interval{ 0x21E4, 0x21E5 }, // So   [2] LEFTWARDS ARROW TO BAR..RIGHTWARDS ARROW TO BAR
    Interval{ 0x21F4, 0x22FF }, // Sm [268] RIGHT ARROW WITH SMALL CIRCLE..Z NOTATION BAG MEMBERSHIP
    Interval{ 0x2308, 0x2308 }, // Ps       LEFT CEILING
    Interval{ 0x2309, 0x2309 }, // Pe       RIGHT CEILING
    Interval{ 0x230A, 0x230A }, // Ps       LEFT FLOOR
    Interval{ 0x230B, 0x230B }, // Pe       RIGHT FLOOR
    Interval{ 0x2320, 0x2321 }, // Sm   [2] TOP HALF INTEGRAL..BOTTOM HALF INTEGRAL
    Interval{ 0x237C, 0x237C }, // Sm       RIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROW
    Interval{ 0x239B, 0x23B3 }, // Sm  [25] LEFT PARENTHESIS UPPER HOOK..SUMMATION BOTTOM
    Interval{ 0x23B4, 0x23B5 }, // So   [2] TOP SQUARE BRACKET..BOTTOM SQUARE BRACKET
    Interval{ 0x23B7, 0x23B7 }, // So       RADICAL SYMBOL BOTTOM
    Interval{ 0x23D0, 0x23D0 }, // So       VERTICAL LINE EXTENSION
    Interval{ 0x23DC, 0x23E1 }, // Sm   [6] TOP PARENTHESIS..BOTTOM TORTOISE SHELL BRACKET
    Interval{ 0x23E2, 0x23E2 }, // So       WHITE TRAPEZIUM
    Interval{ 0x25A0, 0x25A1 }, // So   [2] BLACK SQUARE..WHITE SQUARE
    Interval{ 0x25AE, 0x25B6 }, // So   [9] BLACK VERTICAL RECTANGLE..BLACK RIGHT-POINTING TRIANGLE
    Interval{ 0x25B7, 0x25B7 }, // Sm       WHITE RIGHT-POINTING TRIANGLE
    Interval{ 0x25BC, 0x25C0 }, // So   [5] BLACK DOWN-POINTING TRIANGLE..BLACK LEFT-POINTING TRIANGLE
    Interval{ 0x25C1, 0x25C1 }, // Sm       WHITE LEFT-POINTING TRIANGLE
    Interval{ 0x25C6, 0x25C7 }, // So   [2] BLACK DIAMOND..WHITE DIAMOND
    Interval{ 0x25CA, 0x25CB }, // So   [2] LOZENGE..WHITE CIRCLE
    Interval{ 0x25CF, 0x25D3 }, // So   [5] BLACK CIRCLE..CIRCLE WITH UPPER HALF BLACK
    Interval{ 0x25E2, 0x25E2 }, // So       BLACK LOWER RIGHT TRIANGLE
    Interval{ 0x25E4, 0x25E4 }, // So       BLACK UPPER LEFT TRIANGLE
    Interval{ 0x25E7, 0x25EC }, // So   [6] SQUARE WITH LEFT HALF BLACK..WHITE UP-POINTING TRIANGLE WITH DOT
    Interval{ 0x25F8, 0x25FF }, // Sm   [8] UPPER LEFT TRIANGLE..LOWER RIGHT TRIANGLE
    Interval{ 0x2605, 0x2606 }, // So   [2] BLACK STAR..WHITE STAR
    Interval{ 0x2640, 0x2640 }, // So       FEMALE SIGN
    Interval{ 0x2642, 0x2642 }, // So       MALE SIGN
    Interval{ 0x2660, 0x2663 }, // So   [4] BLACK SPADE SUIT..BLACK CLUB SUIT
    Interval{ 0x266D, 0x266E }, // So   [2] MUSIC FLAT SIGN..MUSIC NATURAL SIGN
    Interval{ 0x266F, 0x266F }, // Sm       MUSIC SHARP SIGN
    Interval{ 0x27C0, 0x27C4 }, // Sm   [5] THREE DIMENSIONAL ANGLE..OPEN SUPERSET
    Interval{ 0x27C5, 0x27C5 }, // Ps       LEFT S-SHAPED BAG DELIMITER
    Interval{ 0x27C6, 0x27C6 }, // Pe       RIGHT S-SHAPED BAG DELIMITER
    Interval{ 0x27C7, 0x27E5 }, // Sm  [31] OR WITH DOT INSIDE..WHITE SQUARE WITH RIGHTWARDS TICK
    Interval{ 0x27E6, 0x27E6 }, // Ps       MATHEMATICAL LEFT WHITE SQUARE BRACKET
    Interval{ 0x27E7, 0x27E7 }, // Pe       MATHEMATICAL RIGHT WHITE SQUARE BRACKET
    Interval{ 0x27E8, 0x27E8 }, // Ps       MATHEMATICAL LEFT ANGLE BRACKET
    Interval{ 0x27E9, 0x27E9 }, // Pe       MATHEMATICAL RIGHT ANGLE BRACKET
    Interval{ 0x27EA, 0x27EA }, // Ps       MATHEMATICAL LEFT DOUBLE ANGLE BRACKET
    Interval{ 0x27EB, 0x27EB }, // Pe       MATHEMATICAL RIGHT DOUBLE ANGLE BRACKET
    Interval{ 0x27EC, 0x27EC }, // Ps       MATHEMATICAL LEFT WHITE TORTOISE SHELL BRACKET
    Interval{ 0x27ED, 0x27ED }, // Pe       MATHEMATICAL RIGHT WHITE TORTOISE SHELL BRACKET
    Interval{ 0x27EE, 0x27EE }, // Ps       MATHEMATICAL LEFT FLATTENED PARENTHESIS
    Interval{ 0x27EF, 0x27EF }, // Pe       MATHEMATICAL RIGHT FLATTENED PARENTHESIS
    Interval{ 0x27F0, 0x27FF }, // Sm  [16] UPWARDS QUADRUPLE ARROW..LONG RIGHTWARDS SQUIGGLE ARROW
    Interval{ 0x2900, 0x2982 }, // Sm [131] RIGHTWARDS TWO-HEADED ARROW WITH VERTICAL STROKE..Z NOTATION TYPE COLON
    Interval{ 0x2983, 0x2983 }, // Ps       LEFT WHITE CURLY BRACKET
    Interval{ 0x2984, 0x2984 }, // Pe       RIGHT WHITE CURLY BRACKET
    Interval{ 0x2985, 0x2985 }, // Ps       LEFT WHITE PARENTHESIS
    Interval{ 0x2986, 0x2986 }, // Pe       RIGHT WHITE PARENTHESIS
    Interval{ 0x2987, 0x2987 }, // Ps       Z NOTATION LEFT IMAGE BRACKET
    Interval{ 0x2988, 0x2988 }, // Pe       Z NOTATION RIGHT IMAGE BRACKET
    Interval{ 0x2989, 0x2989 }, // Ps       Z NOTATION LEFT BINDING BRACKET
    Interval{ 0x298A, 0x298A }, // Pe       Z NOTATION RIGHT BINDING BRACKET
    Interval{ 0x298B, 0x298B }, // Ps       LEFT SQUARE BRACKET WITH UNDERBAR
    Interval{ 0x298C, 0x298C }, // Pe       RIGHT SQUARE BRACKET WITH UNDERBAR
    Interval{ 0x298D, 0x298D }, // Ps       LEFT SQUARE BRACKET WITH TICK IN TOP CORNER
    Interval{ 0x298E, 0x298E }, // Pe       RIGHT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
    Interval{ 0x298F, 0x298F }, // Ps       LEFT SQUARE BRACKET WITH TICK IN BOTTOM CORNER
    Interval{ 0x2990, 0x2990 }, // Pe       RIGHT SQUARE BRACKET WITH TICK IN TOP CORNER
    Interval{ 0x2991, 0x2991 }, // Ps       LEFT ANGLE BRACKET WITH DOT
    Interval{ 0x2992, 0x2992 }, // Pe       RIGHT ANGLE BRACKET WITH DOT
    Interval{ 0x2993, 0x2993 }, // Ps       LEFT ARC LESS-THAN BRACKET
    Interval{ 0x2994, 0x2994 }, // Pe       RIGHT ARC GREATER-THAN BRACKET
    Interval{ 0x2995, 0x2995 }, // Ps       DOUBLE LEFT ARC GREATER-THAN BRACKET
    Interval{ 0x2996, 0x2996 }, // Pe       DOUBLE RIGHT ARC LESS-THAN BRACKET
    Interval{ 0x2997, 0x2997 }, // Ps       LEFT BLACK TORTOISE SHELL BRACKET
    Interval{ 0x2998, 0x2998 }, // Pe       RIGHT BLACK TORTOISE SHELL BRACKET
    Interval{ 0x2999, 0x29D7 }, // Sm  [63] DOTTED FENCE..BLACK HOURGLASS
    Interval{ 0x29D8, 0x29D8 }, // Ps       LEFT WIGGLY FENCE
    Interval{ 0x29D9, 0x29D9 }, // Pe       RIGHT WIGGLY FENCE
    Interval{ 0x29DA, 0x29DA }, // Ps       LEFT DOUBLE WIGGLY FENCE
    Interval{ 0x29DB, 0x29DB }, // Pe       RIGHT DOUBLE WIGGLY FENCE
    Interval{ 0x29DC, 0x29FB }, // Sm  [32] INCOMPLETE INFINITY..TRIPLE PLUS
    Interval{ 0x29FC, 0x29FC }, // Ps       LEFT-POINTING CURVED ANGLE BRACKET
    Interval{ 0x29FD, 0x29FD }, // Pe       RIGHT-POINTING CURVED ANGLE BRACKET
    Interval{ 0x29FE, 0x2AFF }, // Sm [258] TINY..N-ARY WHITE VERTICAL BAR
    Interval{ 0x2B30, 0x2B44 }, // Sm  [21] LEFT ARROW WITH SMALL CIRCLE..RIGHTWARDS ARROW THROUGH SUPERSET
    Interval{ 0x2B47, 0x2B4C }, // Sm   [6] REVERSE TILDE OPERATOR ABOVE RIGHTWARDS ARROW..RIGHTWARDS ARROW ABOVE REVERSE TILDE OPERATOR
    Interval{ 0xFB29, 0xFB29 }, // Sm       HEBREW LETTER ALTERNATIVE PLUS SIGN
    Interval{ 0xFE61, 0xFE61 }, // Po       SMALL ASTERISK
    Interval{ 0xFE62, 0xFE62 }, // Sm       SMALL PLUS SIGN
    Interval{ 0xFE63, 0xFE63 }, // Pd       SMALL HYPHEN-MINUS
    Interval{ 0xFE64, 0xFE66 }, // Sm   [3] SMALL LESS-THAN SIGN..SMALL EQUALS SIGN
    Interval{ 0xFE68, 0xFE68 }, // Po       SMALL REVERSE SOLIDUS
    Interval{ 0xFF0B, 0xFF0B }, // Sm       FULLWIDTH PLUS SIGN
    Interval{ 0xFF1C, 0xFF1E }, // Sm   [3] FULLWIDTH LESS-THAN SIGN..FULLWIDTH GREATER-THAN SIGN
    Interval{ 0xFF3C, 0xFF3C }, // Po       FULLWIDTH REVERSE SOLIDUS
    Interval{ 0xFF3E, 0xFF3E }, // Sk       FULLWIDTH CIRCUMFLEX ACCENT
    Interval{ 0xFF5C, 0xFF5C }, // Sm       FULLWIDTH VERTICAL LINE
    Interval{ 0xFF5E, 0xFF5E }, // Sm       FULLWIDTH TILDE
    Interval{ 0xFFE2, 0xFFE2 }, // Sm       FULLWIDTH NOT SIGN
    Interval{ 0xFFE9, 0xFFEC }, // Sm   [4] HALFWIDTH LEFTWARDS ARROW..HALFWIDTH DOWNWARDS ARROW
    Interval{ 0x1D400, 0x1D454 }, // L&  [85] MATHEMATICAL BOLD CAPITAL A..MATHEMATICAL ITALIC SMALL G
    Interval{ 0x1D456, 0x1D49C }, // L&  [71] MATHEMATICAL ITALIC SMALL I..MATHEMATICAL SCRIPT CAPITAL A
    Interval{ 0x1D49E, 0x1D49F }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL C..MATHEMATICAL SCRIPT CAPITAL D
    Interval{ 0x1D4A2, 0x1D4A2 }, // L&       MATHEMATICAL SCRIPT CAPITAL G
    Interval{ 0x1D4A5, 0x1D4A6 }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL J..MATHEMATICAL SCRIPT CAPITAL K
    Interval{ 0x1D4A9, 0x1D4AC }, // L&   [4] MATHEMATICAL SCRIPT CAPITAL N..MATHEMATICAL SCRIPT CAPITAL Q
    Interval{ 0x1D4AE, 0x1D4B9 }, // L&  [12] MATHEMATICAL SCRIPT CAPITAL S..MATHEMATICAL SCRIPT SMALL D
    Interval{ 0x1D4BB, 0x1D4BB }, // L&       MATHEMATICAL SCRIPT SMALL F
    Interval{ 0x1D4BD, 0x1D4C3 }, // L&   [7] MATHEMATICAL SCRIPT SMALL H..MATHEMATICAL SCRIPT SMALL N
    Interval{ 0x1D4C5, 0x1D505 }, // L&  [65] MATHEMATICAL SCRIPT SMALL P..MATHEMATICAL FRAKTUR CAPITAL B
    Interval{ 0x1D507, 0x1D50A }, // L&   [4] MATHEMATICAL FRAKTUR CAPITAL D..MATHEMATICAL FRAKTUR CAPITAL G
    Interval{ 0x1D50D, 0x1D514 }, // L&   [8] MATHEMATICAL FRAKTUR CAPITAL J..MATHEMATICAL FRAKTUR CAPITAL Q
    Interval{ 0x1D516, 0x1D51C }, // L&   [7] MATHEMATICAL FRAKTUR CAPITAL S..MATHEMATICAL FRAKTUR CAPITAL Y
    Interval{ 0x1D51E, 0x1D539 }, // L&  [28] MATHEMATICAL FRAKTUR SMALL A..MATHEMATICAL DOUBLE-STRUCK CAPITAL B
    Interval{ 0x1D53B, 0x1D53E }, // L&   [4] MATHEMATICAL DOUBLE-STRUCK CAPITAL D..MATHEMATICAL DOUBLE-STRUCK CAPITAL G
    Interval{ 0x1D540, 0x1D544 }, // L&   [5] MATHEMATICAL DOUBLE-STRUCK CAPITAL I..MATHEMATICAL DOUBLE-STRUCK CAPITAL M
    Interval{ 0x1D546, 0x1D546 }, // L&       MATHEMATICAL DOUBLE-STRUCK CAPITAL O
    Interval{ 0x1D54A, 0x1D550 }, // L&   [7] MATHEMATICAL DOUBLE-STRUCK CAPITAL S..MATHEMATICAL DOUBLE-STRUCK CAPITAL Y
    Interval{ 0x1D552, 0x1D6A5 }, // L& [340] MATHEMATICAL DOUBLE-STRUCK SMALL A..MATHEMATICAL ITALIC SMALL DOTLESS J
    Interval{ 0x1D6A8, 0x1D6C0 }, // L&  [25] MATHEMATICAL BOLD CAPITAL ALPHA..MATHEMATICAL BOLD CAPITAL OMEGA
    Interval{ 0x1D6C1, 0x1D6C1 }, // Sm       MATHEMATICAL BOLD NABLA
    Interval{ 0x1D6C2, 0x1D6DA }, // L&  [25] MATHEMATICAL BOLD SMALL ALPHA..MATHEMATICAL BOLD SMALL OMEGA
    Interval{ 0x1D6DB, 0x1D6DB }, // Sm       MATHEMATICAL BOLD PARTIAL DIFFERENTIAL
    Interval{ 0x1D6DC, 0x1D6FA }, // L&  [31] MATHEMATICAL BOLD EPSILON SYMBOL..MATHEMATICAL ITALIC CAPITAL OMEGA
    Interval{ 0x1D6FB, 0x1D6FB }, // Sm       MATHEMATICAL ITALIC NABLA
    Interval{ 0x1D6FC, 0x1D714 }, // L&  [25] MATHEMATICAL ITALIC SMALL ALPHA..MATHEMATICAL ITALIC SMALL OMEGA
    Interval{ 0x1D715, 0x1D715 }, // Sm       MATHEMATICAL ITALIC PARTIAL DIFFERENTIAL
    Interval{ 0x1D716, 0x1D734 }, // L&  [31] MATHEMATICAL ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D735, 0x1D735 }, // Sm       MATHEMATICAL BOLD ITALIC NABLA
    Interval{ 0x1D736, 0x1D74E }, // L&  [25] MATHEMATICAL BOLD ITALIC SMALL ALPHA..MATHEMATICAL BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D74F, 0x1D74F }, // Sm       MATHEMATICAL BOLD ITALIC PARTIAL DIFFERENTIAL
    Interval{ 0x1D750, 0x1D76E }, // L&  [31] MATHEMATICAL BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGA
    Interval{ 0x1D76F, 0x1D76F }, // Sm       MATHEMATICAL SANS-SERIF BOLD NABLA
    Interval{ 0x1D770, 0x1D788 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD SMALL OMEGA
    Interval{ 0x1D789, 0x1D789 }, // Sm       MATHEMATICAL SANS-SERIF BOLD PARTIAL DIFFERENTIAL
    Interval{ 0x1D78A, 0x1D7A8 }, // L&  [31] MATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOL..MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D7A9, 0x1D7A9 }, // Sm       MATHEMATICAL SANS-SERIF BOLD ITALIC NABLA
    Interval{ 0x1D7AA, 0x1D7C2 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHA..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMEGA
    Interval{ 0x1D7C3, 0x1D7C3 }, // Sm       MATHEMATICAL SANS-SERIF BOLD ITALIC PARTIAL DIFFERENTIAL
    Interval{ 0x1D7C4, 0x1D7CB }, // L&   [8] MATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOL..MATHEMATICAL BOLD SMALL DIGAMMA
    Interval{ 0x1D7CE, 0x1D7FF }, // Nd  [50] MATHEMATICAL BOLD DIGIT ZERO..MATHEMATICAL MONOSPACE DIGIT NINE
    Interval{ 0x1EE00, 0x1EE03 }, // Lo   [4] ARABIC MATHEMATICAL ALEF..ARABIC MATHEMATICAL DAL
    Interval{ 0x1EE05, 0x1EE1F }, // Lo  [27] ARABIC MATHEMATICAL WAW..ARABIC MATHEMATICAL DOTLESS QAF
    Interval{ 0x1EE21, 0x1EE22 }, // Lo   [2] ARABIC MATHEMATICAL INITIAL BEH..ARABIC MATHEMATICAL INITIAL JEEM
    Interval{ 0x1EE24, 0x1EE24 }, // Lo       ARABIC MATHEMATICAL INITIAL HEH
    Interval{ 0x1EE27, 0x1EE27 }, // Lo       ARABIC MATHEMATICAL INITIAL HAH
    Interval{ 0x1EE29, 0x1EE32 }, // Lo  [10] ARABIC MATHEMATICAL INITIAL YEH..ARABIC MATHEMATICAL INITIAL QAF
    Interval{ 0x1EE34, 0x1EE37 }, // Lo   [4] ARABIC MATHEMATICAL INITIAL SHEEN..ARABIC MATHEMATICAL INITIAL KHAH
    Interval{ 0x1EE39, 0x1EE39 }, // Lo       ARABIC MATHEMATICAL INITIAL DAD
    Interval{ 0x1EE3B, 0x1EE3B }, // Lo       ARABIC MATHEMATICAL INITIAL GHAIN
    Interval{ 0x1EE42, 0x1EE42 }, // Lo       ARABIC MATHEMATICAL TAILED JEEM
    Interval{ 0x1EE47, 0x1EE47 }, // Lo       ARABIC MATHEMATICAL TAILED HAH
    Interval{ 0x1EE49, 0x1EE49 }, // Lo       ARABIC MATHEMATICAL TAILED YEH
    Interval{ 0x1EE4B, 0x1EE4B }, // Lo       ARABIC MATHEMATICAL TAILED LAM
    Interval{ 0x1EE4D, 0x1EE4F }, // Lo   [3] ARABIC MATHEMATICAL TAILED NOON..ARABIC MATHEMATICAL TAILED AIN
    Interval{ 0x1EE51, 0x1EE52 }, // Lo   [2] ARABIC MATHEMATICAL TAILED SAD..ARABIC MATHEMATICAL TAILED QAF
    Interval{ 0x1EE54, 0x1EE54 }, // Lo       ARABIC MATHEMATICAL TAILED SHEEN
    Interval{ 0x1EE57, 0x1EE57 }, // Lo       ARABIC MATHEMATICAL TAILED KHAH
    Interval{ 0x1EE59, 0x1EE59 }, // Lo       ARABIC MATHEMATICAL TAILED DAD
    Interval{ 0x1EE5B, 0x1EE5B }, // Lo       ARABIC MATHEMATICAL TAILED GHAIN
    Interval{ 0x1EE5D, 0x1EE5D }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS NOON
    Interval{ 0x1EE5F, 0x1EE5F }, // Lo       ARABIC MATHEMATICAL TAILED DOTLESS QAF
    Interval{ 0x1EE61, 0x1EE62 }, // Lo   [2] ARABIC MATHEMATICAL STRETCHED BEH..ARABIC MATHEMATICAL STRETCHED JEEM
    Interval{ 0x1EE64, 0x1EE64 }, // Lo       ARABIC MATHEMATICAL STRETCHED HEH
    Interval{ 0x1EE67, 0x1EE6A }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED HAH..ARABIC MATHEMATICAL STRETCHED KAF
    Interval{ 0x1EE6C, 0x1EE72 }, // Lo   [7] ARABIC MATHEMATICAL STRETCHED MEEM..ARABIC MATHEMATICAL STRETCHED QAF
    Interval{ 0x1EE74, 0x1EE77 }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED SHEEN..ARABIC MATHEMATICAL STRETCHED KHAH
    Interval{ 0x1EE79, 0x1EE7C }, // Lo   [4] ARABIC MATHEMATICAL STRETCHED DAD..ARABIC MATHEMATICAL STRETCHED DOTLESS BEH
    Interval{ 0x1EE7E, 0x1EE7E }, // Lo       ARABIC MATHEMATICAL STRETCHED DOTLESS FEH
    Interval{ 0x1EE80, 0x1EE89 }, // Lo  [10] ARABIC MATHEMATICAL LOOPED ALEF..ARABIC MATHEMATICAL LOOPED YEH
    Interval{ 0x1EE8B, 0x1EE9B }, // Lo  [17] ARABIC MATHEMATICAL LOOPED LAM..ARABIC MATHEMATICAL LOOPED GHAIN
    Interval{ 0x1EEA1, 0x1EEA3 }, // Lo   [3] ARABIC MATHEMATICAL DOUBLE-STRUCK BEH..ARABIC MATHEMATICAL DOUBLE-STRUCK DAL
    Interval{ 0x1EEA5, 0x1EEA9 }, // Lo   [5] ARABIC MATHEMATICAL DOUBLE-STRUCK WAW..ARABIC MATHEMATICAL DOUBLE-STRUCK YEH
    Interval{ 0x1EEAB, 0x1EEBB }, // Lo  [17] ARABIC MATHEMATICAL DOUBLE-STRUCK LAM..ARABIC MATHEMATICAL DOUBLE-STRUCK GHAIN
    Interval{ 0x1EEF0, 0x1EEF1 }, // Sm   [2] ARABIC MATHEMATICAL OPERATOR MEEM WITH HAH WITH TATWEEL..ARABIC MATHEMATICAL OPERATOR HAH WITH DAL
}; // }}}
auto static const Uppercase = std::array<Interval, 651>{ // {{{
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00DE }, // L&   [7] LATIN CAPITAL LETTER O WITH STROKE..LATIN CAPITAL LETTER THORN
    Interval{ 0x0100, 0x0100 }, // L&       LATIN CAPITAL LETTER A WITH MACRON
    Interval{ 0x0102, 0x0102 }, // L&       LATIN CAPITAL LETTER A WITH BREVE
    Interval{ 0x0104, 0x0104 }, // L&       LATIN CAPITAL LETTER A WITH OGONEK
    Interval{ 0x0106, 0x0106 }, // L&       LATIN CAPITAL LETTER C WITH ACUTE
    Interval{ 0x0108, 0x0108 }, // L&       LATIN CAPITAL LETTER C WITH CIRCUMFLEX
    Interval{ 0x010A, 0x010A }, // L&       LATIN CAPITAL LETTER C WITH DOT ABOVE
    Interval{ 0x010C, 0x010C }, // L&       LATIN CAPITAL LETTER C WITH CARON
    Interval{ 0x010E, 0x010E }, // L&       LATIN CAPITAL LETTER D WITH CARON
    Interval{ 0x0110, 0x0110 }, // L&       LATIN CAPITAL LETTER D WITH STROKE
    Interval{ 0x0112, 0x0112 }, // L&       LATIN CAPITAL LETTER E WITH MACRON
    Interval{ 0x0114, 0x0114 }, // L&       LATIN CAPITAL LETTER E WITH BREVE
    Interval{ 0x0116, 0x0116 }, // L&       LATIN CAPITAL LETTER E WITH DOT ABOVE
    Interval{ 0x0118, 0x0118 }, // L&       LATIN CAPITAL LETTER E WITH OGONEK
    Interval{ 0x011A, 0x011A }, // L&       LATIN CAPITAL LETTER E WITH CARON
    Interval{ 0x011C, 0x011C }, // L&       LATIN CAPITAL LETTER G WITH CIRCUMFLEX
    Interval{ 0x011E, 0x011E }, // L&       LATIN CAPITAL LETTER G WITH BREVE
    Interval{ 0x0120, 0x0120 }, // L&       LATIN CAPITAL LETTER G WITH DOT ABOVE
    Interval{ 0x0122, 0x0122 }, // L&       LATIN CAPITAL LETTER G WITH CEDILLA
    Interval{ 0x0124, 0x0124 }, // L&       LATIN CAPITAL LETTER H WITH CIRCUMFLEX
    Interval{ 0x0126, 0x0126 }, // L&       LATIN CAPITAL LETTER H WITH STROKE
    Interval{ 0x0128, 0x0128 }, // L&       LATIN CAPITAL LETTER I WITH TILDE
    Interval{ 0x012A, 0x012A }, // L&       LATIN CAPITAL LETTER I WITH MACRON
    Interval{ 0x012C, 0x012C }, // L&       LATIN CAPITAL LETTER I WITH BREVE
    Interval{ 0x012E, 0x012E }, // L&       LATIN CAPITAL LETTER I WITH OGONEK
    Interval{ 0x0130, 0x0130 }, // L&       LATIN CAPITAL LETTER I WITH DOT ABOVE
    Interval{ 0x0132, 0x0132 }, // L&       LATIN CAPITAL LIGATURE IJ
    Interval{ 0x0134, 0x0134 }, // L&       LATIN CAPITAL LETTER J WITH CIRCUMFLEX
    Interval{ 0x0136, 0x0136 }, // L&       LATIN CAPITAL LETTER K WITH CEDILLA
    Interval{ 0x0139, 0x0139 }, // L&       LATIN CAPITAL LETTER L WITH ACUTE
    Interval{ 0x013B, 0x013B }, // L&       LATIN CAPITAL LETTER L WITH CEDILLA
    Interval{ 0x013D, 0x013D }, // L&       LATIN CAPITAL LETTER L WITH CARON
    Interval{ 0x013F, 0x013F }, // L&       LATIN CAPITAL LETTER L WITH MIDDLE DOT
    Interval{ 0x0141, 0x0141 }, // L&       LATIN CAPITAL LETTER L WITH STROKE
    Interval{ 0x0143, 0x0143 }, // L&       LATIN CAPITAL LETTER N WITH ACUTE
    Interval{ 0x0145, 0x0145 }, // L&       LATIN CAPITAL LETTER N WITH CEDILLA
    Interval{ 0x0147, 0x0147 }, // L&       LATIN CAPITAL LETTER N WITH CARON
    Interval{ 0x014A, 0x014A }, // L&       LATIN CAPITAL LETTER ENG
    Interval{ 0x014C, 0x014C }, // L&       LATIN CAPITAL LETTER O WITH MACRON
    Interval{ 0x014E, 0x014E }, // L&       LATIN CAPITAL LETTER O WITH BREVE
    Interval{ 0x0150, 0x0150 }, // L&       LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
    Interval{ 0x0152, 0x0152 }, // L&       LATIN CAPITAL LIGATURE OE
    Interval{ 0x0154, 0x0154 }, // L&       LATIN CAPITAL LETTER R WITH ACUTE
    Interval{ 0x0156, 0x0156 }, // L&       LATIN CAPITAL LETTER R WITH CEDILLA
    Interval{ 0x0158, 0x0158 }, // L&       LATIN CAPITAL LETTER R WITH CARON
    Interval{ 0x015A, 0x015A }, // L&       LATIN CAPITAL LETTER S WITH ACUTE
    Interval{ 0x015C, 0x015C }, // L&       LATIN CAPITAL LETTER S WITH CIRCUMFLEX
    Interval{ 0x015E, 0x015E }, // L&       LATIN CAPITAL LETTER S WITH CEDILLA
    Interval{ 0x0160, 0x0160 }, // L&       LATIN CAPITAL LETTER S WITH CARON
    Interval{ 0x0162, 0x0162 }, // L&       LATIN CAPITAL LETTER T WITH CEDILLA
    Interval{ 0x0164, 0x0164 }, // L&       LATIN CAPITAL LETTER T WITH CARON
    Interval{ 0x0166, 0x0166 }, // L&       LATIN CAPITAL LETTER T WITH STROKE
    Interval{ 0x0168, 0x0168 }, // L&       LATIN CAPITAL LETTER U WITH TILDE
    Interval{ 0x016A, 0x016A }, // L&       LATIN CAPITAL LETTER U WITH MACRON
    Interval{ 0x016C, 0x016C }, // L&       LATIN CAPITAL LETTER U WITH BREVE
    Interval{ 0x016E, 0x016E }, // L&       LATIN CAPITAL LETTER U WITH RING ABOVE
    Interval{ 0x0170, 0x0170 }, // L&       LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x0172, 0x0172 }, // L&       LATIN CAPITAL LETTER U WITH OGONEK
    Interval{ 0x0174, 0x0174 }, // L&       LATIN CAPITAL LETTER W WITH CIRCUMFLEX
    Interval{ 0x0176, 0x0176 }, // L&       LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
    Interval{ 0x0178, 0x0179 }, // L&   [2] LATIN CAPITAL LETTER Y WITH DIAERESIS..LATIN CAPITAL LETTER Z WITH ACUTE
    Interval{ 0x017B, 0x017B }, // L&       LATIN CAPITAL LETTER Z WITH DOT ABOVE
    Interval{ 0x017D, 0x017D }, // L&       LATIN CAPITAL LETTER Z WITH CARON
    Interval{ 0x0181, 0x0182 }, // L&   [2] LATIN CAPITAL LETTER B WITH HOOK..LATIN CAPITAL LETTER B WITH TOPBAR
    Interval{ 0x0184, 0x0184 }, // L&       LATIN CAPITAL LETTER TONE SIX
    Interval{ 0x0186, 0x0187 }, // L&   [2] LATIN CAPITAL LETTER OPEN O..LATIN CAPITAL LETTER C WITH HOOK
    Interval{ 0x0189, 0x018B }, // L&   [3] LATIN CAPITAL LETTER AFRICAN D..LATIN CAPITAL LETTER D WITH TOPBAR
    Interval{ 0x018E, 0x0191 }, // L&   [4] LATIN CAPITAL LETTER REVERSED E..LATIN CAPITAL LETTER F WITH HOOK
    Interval{ 0x0193, 0x0194 }, // L&   [2] LATIN CAPITAL LETTER G WITH HOOK..LATIN CAPITAL LETTER GAMMA
    Interval{ 0x0196, 0x0198 }, // L&   [3] LATIN CAPITAL LETTER IOTA..LATIN CAPITAL LETTER K WITH HOOK
    Interval{ 0x019C, 0x019D }, // L&   [2] LATIN CAPITAL LETTER TURNED M..LATIN CAPITAL LETTER N WITH LEFT HOOK
    Interval{ 0x019F, 0x01A0 }, // L&   [2] LATIN CAPITAL LETTER O WITH MIDDLE TILDE..LATIN CAPITAL LETTER O WITH HORN
    Interval{ 0x01A2, 0x01A2 }, // L&       LATIN CAPITAL LETTER OI
    Interval{ 0x01A4, 0x01A4 }, // L&       LATIN CAPITAL LETTER P WITH HOOK
    Interval{ 0x01A6, 0x01A7 }, // L&   [2] LATIN LETTER YR..LATIN CAPITAL LETTER TONE TWO
    Interval{ 0x01A9, 0x01A9 }, // L&       LATIN CAPITAL LETTER ESH
    Interval{ 0x01AC, 0x01AC }, // L&       LATIN CAPITAL LETTER T WITH HOOK
    Interval{ 0x01AE, 0x01AF }, // L&   [2] LATIN CAPITAL LETTER T WITH RETROFLEX HOOK..LATIN CAPITAL LETTER U WITH HORN
    Interval{ 0x01B1, 0x01B3 }, // L&   [3] LATIN CAPITAL LETTER UPSILON..LATIN CAPITAL LETTER Y WITH HOOK
    Interval{ 0x01B5, 0x01B5 }, // L&       LATIN CAPITAL LETTER Z WITH STROKE
    Interval{ 0x01B7, 0x01B8 }, // L&   [2] LATIN CAPITAL LETTER EZH..LATIN CAPITAL LETTER EZH REVERSED
    Interval{ 0x01BC, 0x01BC }, // L&       LATIN CAPITAL LETTER TONE FIVE
    Interval{ 0x01C4, 0x01C4 }, // L&       LATIN CAPITAL LETTER DZ WITH CARON
    Interval{ 0x01C7, 0x01C7 }, // L&       LATIN CAPITAL LETTER LJ
    Interval{ 0x01CA, 0x01CA }, // L&       LATIN CAPITAL LETTER NJ
    Interval{ 0x01CD, 0x01CD }, // L&       LATIN CAPITAL LETTER A WITH CARON
    Interval{ 0x01CF, 0x01CF }, // L&       LATIN CAPITAL LETTER I WITH CARON
    Interval{ 0x01D1, 0x01D1 }, // L&       LATIN CAPITAL LETTER O WITH CARON
    Interval{ 0x01D3, 0x01D3 }, // L&       LATIN CAPITAL LETTER U WITH CARON
    Interval{ 0x01D5, 0x01D5 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
    Interval{ 0x01D7, 0x01D7 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
    Interval{ 0x01D9, 0x01D9 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
    Interval{ 0x01DB, 0x01DB }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
    Interval{ 0x01DE, 0x01DE }, // L&       LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
    Interval{ 0x01E0, 0x01E0 }, // L&       LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
    Interval{ 0x01E2, 0x01E2 }, // L&       LATIN CAPITAL LETTER AE WITH MACRON
    Interval{ 0x01E4, 0x01E4 }, // L&       LATIN CAPITAL LETTER G WITH STROKE
    Interval{ 0x01E6, 0x01E6 }, // L&       LATIN CAPITAL LETTER G WITH CARON
    Interval{ 0x01E8, 0x01E8 }, // L&       LATIN CAPITAL LETTER K WITH CARON
    Interval{ 0x01EA, 0x01EA }, // L&       LATIN CAPITAL LETTER O WITH OGONEK
    Interval{ 0x01EC, 0x01EC }, // L&       LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
    Interval{ 0x01EE, 0x01EE }, // L&       LATIN CAPITAL LETTER EZH WITH CARON
    Interval{ 0x01F1, 0x01F1 }, // L&       LATIN CAPITAL LETTER DZ
    Interval{ 0x01F4, 0x01F4 }, // L&       LATIN CAPITAL LETTER G WITH ACUTE
    Interval{ 0x01F6, 0x01F8 }, // L&   [3] LATIN CAPITAL LETTER HWAIR..LATIN CAPITAL LETTER N WITH GRAVE
    Interval{ 0x01FA, 0x01FA }, // L&       LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
    Interval{ 0x01FC, 0x01FC }, // L&       LATIN CAPITAL LETTER AE WITH ACUTE
    Interval{ 0x01FE, 0x01FE }, // L&       LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
    Interval{ 0x0200, 0x0200 }, // L&       LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
    Interval{ 0x0202, 0x0202 }, // L&       LATIN CAPITAL LETTER A WITH INVERTED BREVE
    Interval{ 0x0204, 0x0204 }, // L&       LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
    Interval{ 0x0206, 0x0206 }, // L&       LATIN CAPITAL LETTER E WITH INVERTED BREVE
    Interval{ 0x0208, 0x0208 }, // L&       LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
    Interval{ 0x020A, 0x020A }, // L&       LATIN CAPITAL LETTER I WITH INVERTED BREVE
    Interval{ 0x020C, 0x020C }, // L&       LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
    Interval{ 0x020E, 0x020E }, // L&       LATIN CAPITAL LETTER O WITH INVERTED BREVE
    Interval{ 0x0210, 0x0210 }, // L&       LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
    Interval{ 0x0212, 0x0212 }, // L&       LATIN CAPITAL LETTER R WITH INVERTED BREVE
    Interval{ 0x0214, 0x0214 }, // L&       LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
    Interval{ 0x0216, 0x0216 }, // L&       LATIN CAPITAL LETTER U WITH INVERTED BREVE
    Interval{ 0x0218, 0x0218 }, // L&       LATIN CAPITAL LETTER S WITH COMMA BELOW
    Interval{ 0x021A, 0x021A }, // L&       LATIN CAPITAL LETTER T WITH COMMA BELOW
    Interval{ 0x021C, 0x021C }, // L&       LATIN CAPITAL LETTER YOGH
    Interval{ 0x021E, 0x021E }, // L&       LATIN CAPITAL LETTER H WITH CARON
    Interval{ 0x0220, 0x0220 }, // L&       LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
    Interval{ 0x0222, 0x0222 }, // L&       LATIN CAPITAL LETTER OU
    Interval{ 0x0224, 0x0224 }, // L&       LATIN CAPITAL LETTER Z WITH HOOK
    Interval{ 0x0226, 0x0226 }, // L&       LATIN CAPITAL LETTER A WITH DOT ABOVE
    Interval{ 0x0228, 0x0228 }, // L&       LATIN CAPITAL LETTER E WITH CEDILLA
    Interval{ 0x022A, 0x022A }, // L&       LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
    Interval{ 0x022C, 0x022C }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND MACRON
    Interval{ 0x022E, 0x022E }, // L&       LATIN CAPITAL LETTER O WITH DOT ABOVE
    Interval{ 0x0230, 0x0230 }, // L&       LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
    Interval{ 0x0232, 0x0232 }, // L&       LATIN CAPITAL LETTER Y WITH MACRON
    Interval{ 0x023A, 0x023B }, // L&   [2] LATIN CAPITAL LETTER A WITH STROKE..LATIN CAPITAL LETTER C WITH STROKE
    Interval{ 0x023D, 0x023E }, // L&   [2] LATIN CAPITAL LETTER L WITH BAR..LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
    Interval{ 0x0241, 0x0241 }, // L&       LATIN CAPITAL LETTER GLOTTAL STOP
    Interval{ 0x0243, 0x0246 }, // L&   [4] LATIN CAPITAL LETTER B WITH STROKE..LATIN CAPITAL LETTER E WITH STROKE
    Interval{ 0x0248, 0x0248 }, // L&       LATIN CAPITAL LETTER J WITH STROKE
    Interval{ 0x024A, 0x024A }, // L&       LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
    Interval{ 0x024C, 0x024C }, // L&       LATIN CAPITAL LETTER R WITH STROKE
    Interval{ 0x024E, 0x024E }, // L&       LATIN CAPITAL LETTER Y WITH STROKE
    Interval{ 0x0370, 0x0370 }, // L&       GREEK CAPITAL LETTER HETA
    Interval{ 0x0372, 0x0372 }, // L&       GREEK CAPITAL LETTER ARCHAIC SAMPI
    Interval{ 0x0376, 0x0376 }, // L&       GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x038F }, // L&   [2] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER OMEGA WITH TONOS
    Interval{ 0x0391, 0x03A1 }, // L&  [17] GREEK CAPITAL LETTER ALPHA..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03AB }, // L&   [9] GREEK CAPITAL LETTER SIGMA..GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
    Interval{ 0x03CF, 0x03CF }, // L&       GREEK CAPITAL KAI SYMBOL
    Interval{ 0x03D2, 0x03D4 }, // L&   [3] GREEK UPSILON WITH HOOK SYMBOL..GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL
    Interval{ 0x03D8, 0x03D8 }, // L&       GREEK LETTER ARCHAIC KOPPA
    Interval{ 0x03DA, 0x03DA }, // L&       GREEK LETTER STIGMA
    Interval{ 0x03DC, 0x03DC }, // L&       GREEK LETTER DIGAMMA
    Interval{ 0x03DE, 0x03DE }, // L&       GREEK LETTER KOPPA
    Interval{ 0x03E0, 0x03E0 }, // L&       GREEK LETTER SAMPI
    Interval{ 0x03E2, 0x03E2 }, // L&       COPTIC CAPITAL LETTER SHEI
    Interval{ 0x03E4, 0x03E4 }, // L&       COPTIC CAPITAL LETTER FEI
    Interval{ 0x03E6, 0x03E6 }, // L&       COPTIC CAPITAL LETTER KHEI
    Interval{ 0x03E8, 0x03E8 }, // L&       COPTIC CAPITAL LETTER HORI
    Interval{ 0x03EA, 0x03EA }, // L&       COPTIC CAPITAL LETTER GANGIA
    Interval{ 0x03EC, 0x03EC }, // L&       COPTIC CAPITAL LETTER SHIMA
    Interval{ 0x03EE, 0x03EE }, // L&       COPTIC CAPITAL LETTER DEI
    Interval{ 0x03F4, 0x03F4 }, // L&       GREEK CAPITAL THETA SYMBOL
    Interval{ 0x03F7, 0x03F7 }, // L&       GREEK CAPITAL LETTER SHO
    Interval{ 0x03F9, 0x03FA }, // L&   [2] GREEK CAPITAL LUNATE SIGMA SYMBOL..GREEK CAPITAL LETTER SAN
    Interval{ 0x03FD, 0x042F }, // L&  [51] GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL..CYRILLIC CAPITAL LETTER YA
    Interval{ 0x0460, 0x0460 }, // L&       CYRILLIC CAPITAL LETTER OMEGA
    Interval{ 0x0462, 0x0462 }, // L&       CYRILLIC CAPITAL LETTER YAT
    Interval{ 0x0464, 0x0464 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED E
    Interval{ 0x0466, 0x0466 }, // L&       CYRILLIC CAPITAL LETTER LITTLE YUS
    Interval{ 0x0468, 0x0468 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
    Interval{ 0x046A, 0x046A }, // L&       CYRILLIC CAPITAL LETTER BIG YUS
    Interval{ 0x046C, 0x046C }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
    Interval{ 0x046E, 0x046E }, // L&       CYRILLIC CAPITAL LETTER KSI
    Interval{ 0x0470, 0x0470 }, // L&       CYRILLIC CAPITAL LETTER PSI
    Interval{ 0x0472, 0x0472 }, // L&       CYRILLIC CAPITAL LETTER FITA
    Interval{ 0x0474, 0x0474 }, // L&       CYRILLIC CAPITAL LETTER IZHITSA
    Interval{ 0x0476, 0x0476 }, // L&       CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
    Interval{ 0x0478, 0x0478 }, // L&       CYRILLIC CAPITAL LETTER UK
    Interval{ 0x047A, 0x047A }, // L&       CYRILLIC CAPITAL LETTER ROUND OMEGA
    Interval{ 0x047C, 0x047C }, // L&       CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
    Interval{ 0x047E, 0x047E }, // L&       CYRILLIC CAPITAL LETTER OT
    Interval{ 0x0480, 0x0480 }, // L&       CYRILLIC CAPITAL LETTER KOPPA
    Interval{ 0x048A, 0x048A }, // L&       CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
    Interval{ 0x048C, 0x048C }, // L&       CYRILLIC CAPITAL LETTER SEMISOFT SIGN
    Interval{ 0x048E, 0x048E }, // L&       CYRILLIC CAPITAL LETTER ER WITH TICK
    Interval{ 0x0490, 0x0490 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH UPTURN
    Interval{ 0x0492, 0x0492 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH STROKE
    Interval{ 0x0494, 0x0494 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
    Interval{ 0x0496, 0x0496 }, // L&       CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
    Interval{ 0x0498, 0x0498 }, // L&       CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
    Interval{ 0x049A, 0x049A }, // L&       CYRILLIC CAPITAL LETTER KA WITH DESCENDER
    Interval{ 0x049C, 0x049C }, // L&       CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
    Interval{ 0x049E, 0x049E }, // L&       CYRILLIC CAPITAL LETTER KA WITH STROKE
    Interval{ 0x04A0, 0x04A0 }, // L&       CYRILLIC CAPITAL LETTER BASHKIR KA
    Interval{ 0x04A2, 0x04A2 }, // L&       CYRILLIC CAPITAL LETTER EN WITH DESCENDER
    Interval{ 0x04A4, 0x04A4 }, // L&       CYRILLIC CAPITAL LIGATURE EN GHE
    Interval{ 0x04A6, 0x04A6 }, // L&       CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
    Interval{ 0x04A8, 0x04A8 }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN HA
    Interval{ 0x04AA, 0x04AA }, // L&       CYRILLIC CAPITAL LETTER ES WITH DESCENDER
    Interval{ 0x04AC, 0x04AC }, // L&       CYRILLIC CAPITAL LETTER TE WITH DESCENDER
    Interval{ 0x04AE, 0x04AE }, // L&       CYRILLIC CAPITAL LETTER STRAIGHT U
    Interval{ 0x04B0, 0x04B0 }, // L&       CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
    Interval{ 0x04B2, 0x04B2 }, // L&       CYRILLIC CAPITAL LETTER HA WITH DESCENDER
    Interval{ 0x04B4, 0x04B4 }, // L&       CYRILLIC CAPITAL LIGATURE TE TSE
    Interval{ 0x04B6, 0x04B6 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
    Interval{ 0x04B8, 0x04B8 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
    Interval{ 0x04BA, 0x04BA }, // L&       CYRILLIC CAPITAL LETTER SHHA
    Interval{ 0x04BC, 0x04BC }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN CHE
    Interval{ 0x04BE, 0x04BE }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
    Interval{ 0x04C0, 0x04C1 }, // L&   [2] CYRILLIC LETTER PALOCHKA..CYRILLIC CAPITAL LETTER ZHE WITH BREVE
    Interval{ 0x04C3, 0x04C3 }, // L&       CYRILLIC CAPITAL LETTER KA WITH HOOK
    Interval{ 0x04C5, 0x04C5 }, // L&       CYRILLIC CAPITAL LETTER EL WITH TAIL
    Interval{ 0x04C7, 0x04C7 }, // L&       CYRILLIC CAPITAL LETTER EN WITH HOOK
    Interval{ 0x04C9, 0x04C9 }, // L&       CYRILLIC CAPITAL LETTER EN WITH TAIL
    Interval{ 0x04CB, 0x04CB }, // L&       CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
    Interval{ 0x04CD, 0x04CD }, // L&       CYRILLIC CAPITAL LETTER EM WITH TAIL
    Interval{ 0x04D0, 0x04D0 }, // L&       CYRILLIC CAPITAL LETTER A WITH BREVE
    Interval{ 0x04D2, 0x04D2 }, // L&       CYRILLIC CAPITAL LETTER A WITH DIAERESIS
    Interval{ 0x04D4, 0x04D4 }, // L&       CYRILLIC CAPITAL LIGATURE A IE
    Interval{ 0x04D6, 0x04D6 }, // L&       CYRILLIC CAPITAL LETTER IE WITH BREVE
    Interval{ 0x04D8, 0x04D8 }, // L&       CYRILLIC CAPITAL LETTER SCHWA
    Interval{ 0x04DA, 0x04DA }, // L&       CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
    Interval{ 0x04DC, 0x04DC }, // L&       CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
    Interval{ 0x04DE, 0x04DE }, // L&       CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
    Interval{ 0x04E0, 0x04E0 }, // L&       CYRILLIC CAPITAL LETTER ABKHASIAN DZE
    Interval{ 0x04E2, 0x04E2 }, // L&       CYRILLIC CAPITAL LETTER I WITH MACRON
    Interval{ 0x04E4, 0x04E4 }, // L&       CYRILLIC CAPITAL LETTER I WITH DIAERESIS
    Interval{ 0x04E6, 0x04E6 }, // L&       CYRILLIC CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x04E8, 0x04E8 }, // L&       CYRILLIC CAPITAL LETTER BARRED O
    Interval{ 0x04EA, 0x04EA }, // L&       CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
    Interval{ 0x04EC, 0x04EC }, // L&       CYRILLIC CAPITAL LETTER E WITH DIAERESIS
    Interval{ 0x04EE, 0x04EE }, // L&       CYRILLIC CAPITAL LETTER U WITH MACRON
    Interval{ 0x04F0, 0x04F0 }, // L&       CYRILLIC CAPITAL LETTER U WITH DIAERESIS
    Interval{ 0x04F2, 0x04F2 }, // L&       CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
    Interval{ 0x04F4, 0x04F4 }, // L&       CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
    Interval{ 0x04F6, 0x04F6 }, // L&       CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
    Interval{ 0x04F8, 0x04F8 }, // L&       CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
    Interval{ 0x04FA, 0x04FA }, // L&       CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK
    Interval{ 0x04FC, 0x04FC }, // L&       CYRILLIC CAPITAL LETTER HA WITH HOOK
    Interval{ 0x04FE, 0x04FE }, // L&       CYRILLIC CAPITAL LETTER HA WITH STROKE
    Interval{ 0x0500, 0x0500 }, // L&       CYRILLIC CAPITAL LETTER KOMI DE
    Interval{ 0x0502, 0x0502 }, // L&       CYRILLIC CAPITAL LETTER KOMI DJE
    Interval{ 0x0504, 0x0504 }, // L&       CYRILLIC CAPITAL LETTER KOMI ZJE
    Interval{ 0x0506, 0x0506 }, // L&       CYRILLIC CAPITAL LETTER KOMI DZJE
    Interval{ 0x0508, 0x0508 }, // L&       CYRILLIC CAPITAL LETTER KOMI LJE
    Interval{ 0x050A, 0x050A }, // L&       CYRILLIC CAPITAL LETTER KOMI NJE
    Interval{ 0x050C, 0x050C }, // L&       CYRILLIC CAPITAL LETTER KOMI SJE
    Interval{ 0x050E, 0x050E }, // L&       CYRILLIC CAPITAL LETTER KOMI TJE
    Interval{ 0x0510, 0x0510 }, // L&       CYRILLIC CAPITAL LETTER REVERSED ZE
    Interval{ 0x0512, 0x0512 }, // L&       CYRILLIC CAPITAL LETTER EL WITH HOOK
    Interval{ 0x0514, 0x0514 }, // L&       CYRILLIC CAPITAL LETTER LHA
    Interval{ 0x0516, 0x0516 }, // L&       CYRILLIC CAPITAL LETTER RHA
    Interval{ 0x0518, 0x0518 }, // L&       CYRILLIC CAPITAL LETTER YAE
    Interval{ 0x051A, 0x051A }, // L&       CYRILLIC CAPITAL LETTER QA
    Interval{ 0x051C, 0x051C }, // L&       CYRILLIC CAPITAL LETTER WE
    Interval{ 0x051E, 0x051E }, // L&       CYRILLIC CAPITAL LETTER ALEUT KA
    Interval{ 0x0520, 0x0520 }, // L&       CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
    Interval{ 0x0522, 0x0522 }, // L&       CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
    Interval{ 0x0524, 0x0524 }, // L&       CYRILLIC CAPITAL LETTER PE WITH DESCENDER
    Interval{ 0x0526, 0x0526 }, // L&       CYRILLIC CAPITAL LETTER SHHA WITH DESCENDER
    Interval{ 0x0528, 0x0528 }, // L&       CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK
    Interval{ 0x052A, 0x052A }, // L&       CYRILLIC CAPITAL LETTER DZZHE
    Interval{ 0x052C, 0x052C }, // L&       CYRILLIC CAPITAL LETTER DCHE
    Interval{ 0x052E, 0x052E }, // L&       CYRILLIC CAPITAL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1E00, 0x1E00 }, // L&       LATIN CAPITAL LETTER A WITH RING BELOW
    Interval{ 0x1E02, 0x1E02 }, // L&       LATIN CAPITAL LETTER B WITH DOT ABOVE
    Interval{ 0x1E04, 0x1E04 }, // L&       LATIN CAPITAL LETTER B WITH DOT BELOW
    Interval{ 0x1E06, 0x1E06 }, // L&       LATIN CAPITAL LETTER B WITH LINE BELOW
    Interval{ 0x1E08, 0x1E08 }, // L&       LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
    Interval{ 0x1E0A, 0x1E0A }, // L&       LATIN CAPITAL LETTER D WITH DOT ABOVE
    Interval{ 0x1E0C, 0x1E0C }, // L&       LATIN CAPITAL LETTER D WITH DOT BELOW
    Interval{ 0x1E0E, 0x1E0E }, // L&       LATIN CAPITAL LETTER D WITH LINE BELOW
    Interval{ 0x1E10, 0x1E10 }, // L&       LATIN CAPITAL LETTER D WITH CEDILLA
    Interval{ 0x1E12, 0x1E12 }, // L&       LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
    Interval{ 0x1E14, 0x1E14 }, // L&       LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
    Interval{ 0x1E16, 0x1E16 }, // L&       LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
    Interval{ 0x1E18, 0x1E18 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
    Interval{ 0x1E1A, 0x1E1A }, // L&       LATIN CAPITAL LETTER E WITH TILDE BELOW
    Interval{ 0x1E1C, 0x1E1C }, // L&       LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
    Interval{ 0x1E1E, 0x1E1E }, // L&       LATIN CAPITAL LETTER F WITH DOT ABOVE
    Interval{ 0x1E20, 0x1E20 }, // L&       LATIN CAPITAL LETTER G WITH MACRON
    Interval{ 0x1E22, 0x1E22 }, // L&       LATIN CAPITAL LETTER H WITH DOT ABOVE
    Interval{ 0x1E24, 0x1E24 }, // L&       LATIN CAPITAL LETTER H WITH DOT BELOW
    Interval{ 0x1E26, 0x1E26 }, // L&       LATIN CAPITAL LETTER H WITH DIAERESIS
    Interval{ 0x1E28, 0x1E28 }, // L&       LATIN CAPITAL LETTER H WITH CEDILLA
    Interval{ 0x1E2A, 0x1E2A }, // L&       LATIN CAPITAL LETTER H WITH BREVE BELOW
    Interval{ 0x1E2C, 0x1E2C }, // L&       LATIN CAPITAL LETTER I WITH TILDE BELOW
    Interval{ 0x1E2E, 0x1E2E }, // L&       LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
    Interval{ 0x1E30, 0x1E30 }, // L&       LATIN CAPITAL LETTER K WITH ACUTE
    Interval{ 0x1E32, 0x1E32 }, // L&       LATIN CAPITAL LETTER K WITH DOT BELOW
    Interval{ 0x1E34, 0x1E34 }, // L&       LATIN CAPITAL LETTER K WITH LINE BELOW
    Interval{ 0x1E36, 0x1E36 }, // L&       LATIN CAPITAL LETTER L WITH DOT BELOW
    Interval{ 0x1E38, 0x1E38 }, // L&       LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
    Interval{ 0x1E3A, 0x1E3A }, // L&       LATIN CAPITAL LETTER L WITH LINE BELOW
    Interval{ 0x1E3C, 0x1E3C }, // L&       LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
    Interval{ 0x1E3E, 0x1E3E }, // L&       LATIN CAPITAL LETTER M WITH ACUTE
    Interval{ 0x1E40, 0x1E40 }, // L&       LATIN CAPITAL LETTER M WITH DOT ABOVE
    Interval{ 0x1E42, 0x1E42 }, // L&       LATIN CAPITAL LETTER M WITH DOT BELOW
    Interval{ 0x1E44, 0x1E44 }, // L&       LATIN CAPITAL LETTER N WITH DOT ABOVE
    Interval{ 0x1E46, 0x1E46 }, // L&       LATIN CAPITAL LETTER N WITH DOT BELOW
    Interval{ 0x1E48, 0x1E48 }, // L&       LATIN CAPITAL LETTER N WITH LINE BELOW
    Interval{ 0x1E4A, 0x1E4A }, // L&       LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
    Interval{ 0x1E4C, 0x1E4C }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
    Interval{ 0x1E4E, 0x1E4E }, // L&       LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
    Interval{ 0x1E50, 0x1E50 }, // L&       LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
    Interval{ 0x1E52, 0x1E52 }, // L&       LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
    Interval{ 0x1E54, 0x1E54 }, // L&       LATIN CAPITAL LETTER P WITH ACUTE
    Interval{ 0x1E56, 0x1E56 }, // L&       LATIN CAPITAL LETTER P WITH DOT ABOVE
    Interval{ 0x1E58, 0x1E58 }, // L&       LATIN CAPITAL LETTER R WITH DOT ABOVE
    Interval{ 0x1E5A, 0x1E5A }, // L&       LATIN CAPITAL LETTER R WITH DOT BELOW
    Interval{ 0x1E5C, 0x1E5C }, // L&       LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
    Interval{ 0x1E5E, 0x1E5E }, // L&       LATIN CAPITAL LETTER R WITH LINE BELOW
    Interval{ 0x1E60, 0x1E60 }, // L&       LATIN CAPITAL LETTER S WITH DOT ABOVE
    Interval{ 0x1E62, 0x1E62 }, // L&       LATIN CAPITAL LETTER S WITH DOT BELOW
    Interval{ 0x1E64, 0x1E64 }, // L&       LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
    Interval{ 0x1E66, 0x1E66 }, // L&       LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
    Interval{ 0x1E68, 0x1E68 }, // L&       LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
    Interval{ 0x1E6A, 0x1E6A }, // L&       LATIN CAPITAL LETTER T WITH DOT ABOVE
    Interval{ 0x1E6C, 0x1E6C }, // L&       LATIN CAPITAL LETTER T WITH DOT BELOW
    Interval{ 0x1E6E, 0x1E6E }, // L&       LATIN CAPITAL LETTER T WITH LINE BELOW
    Interval{ 0x1E70, 0x1E70 }, // L&       LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
    Interval{ 0x1E72, 0x1E72 }, // L&       LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
    Interval{ 0x1E74, 0x1E74 }, // L&       LATIN CAPITAL LETTER U WITH TILDE BELOW
    Interval{ 0x1E76, 0x1E76 }, // L&       LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
    Interval{ 0x1E78, 0x1E78 }, // L&       LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
    Interval{ 0x1E7A, 0x1E7A }, // L&       LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
    Interval{ 0x1E7C, 0x1E7C }, // L&       LATIN CAPITAL LETTER V WITH TILDE
    Interval{ 0x1E7E, 0x1E7E }, // L&       LATIN CAPITAL LETTER V WITH DOT BELOW
    Interval{ 0x1E80, 0x1E80 }, // L&       LATIN CAPITAL LETTER W WITH GRAVE
    Interval{ 0x1E82, 0x1E82 }, // L&       LATIN CAPITAL LETTER W WITH ACUTE
    Interval{ 0x1E84, 0x1E84 }, // L&       LATIN CAPITAL LETTER W WITH DIAERESIS
    Interval{ 0x1E86, 0x1E86 }, // L&       LATIN CAPITAL LETTER W WITH DOT ABOVE
    Interval{ 0x1E88, 0x1E88 }, // L&       LATIN CAPITAL LETTER W WITH DOT BELOW
    Interval{ 0x1E8A, 0x1E8A }, // L&       LATIN CAPITAL LETTER X WITH DOT ABOVE
    Interval{ 0x1E8C, 0x1E8C }, // L&       LATIN CAPITAL LETTER X WITH DIAERESIS
    Interval{ 0x1E8E, 0x1E8E }, // L&       LATIN CAPITAL LETTER Y WITH DOT ABOVE
    Interval{ 0x1E90, 0x1E90 }, // L&       LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
    Interval{ 0x1E92, 0x1E92 }, // L&       LATIN CAPITAL LETTER Z WITH DOT BELOW
    Interval{ 0x1E94, 0x1E94 }, // L&       LATIN CAPITAL LETTER Z WITH LINE BELOW
    Interval{ 0x1E9E, 0x1E9E }, // L&       LATIN CAPITAL LETTER SHARP S
    Interval{ 0x1EA0, 0x1EA0 }, // L&       LATIN CAPITAL LETTER A WITH DOT BELOW
    Interval{ 0x1EA2, 0x1EA2 }, // L&       LATIN CAPITAL LETTER A WITH HOOK ABOVE
    Interval{ 0x1EA4, 0x1EA4 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EA6, 0x1EA6 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EA8, 0x1EA8 }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EAA, 0x1EAA }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EAC, 0x1EAC }, // L&       LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EAE, 0x1EAE }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
    Interval{ 0x1EB0, 0x1EB0 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
    Interval{ 0x1EB2, 0x1EB2 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
    Interval{ 0x1EB4, 0x1EB4 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND TILDE
    Interval{ 0x1EB6, 0x1EB6 }, // L&       LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
    Interval{ 0x1EB8, 0x1EB8 }, // L&       LATIN CAPITAL LETTER E WITH DOT BELOW
    Interval{ 0x1EBA, 0x1EBA }, // L&       LATIN CAPITAL LETTER E WITH HOOK ABOVE
    Interval{ 0x1EBC, 0x1EBC }, // L&       LATIN CAPITAL LETTER E WITH TILDE
    Interval{ 0x1EBE, 0x1EBE }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1EC0, 0x1EC0 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1EC2, 0x1EC2 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1EC4, 0x1EC4 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1EC6, 0x1EC6 }, // L&       LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EC8, 0x1EC8 }, // L&       LATIN CAPITAL LETTER I WITH HOOK ABOVE
    Interval{ 0x1ECA, 0x1ECA }, // L&       LATIN CAPITAL LETTER I WITH DOT BELOW
    Interval{ 0x1ECC, 0x1ECC }, // L&       LATIN CAPITAL LETTER O WITH DOT BELOW
    Interval{ 0x1ECE, 0x1ECE }, // L&       LATIN CAPITAL LETTER O WITH HOOK ABOVE
    Interval{ 0x1ED0, 0x1ED0 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
    Interval{ 0x1ED2, 0x1ED2 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
    Interval{ 0x1ED4, 0x1ED4 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    Interval{ 0x1ED6, 0x1ED6 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
    Interval{ 0x1ED8, 0x1ED8 }, // L&       LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    Interval{ 0x1EDA, 0x1EDA }, // L&       LATIN CAPITAL LETTER O WITH HORN AND ACUTE
    Interval{ 0x1EDC, 0x1EDC }, // L&       LATIN CAPITAL LETTER O WITH HORN AND GRAVE
    Interval{ 0x1EDE, 0x1EDE }, // L&       LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
    Interval{ 0x1EE0, 0x1EE0 }, // L&       LATIN CAPITAL LETTER O WITH HORN AND TILDE
    Interval{ 0x1EE2, 0x1EE2 }, // L&       LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
    Interval{ 0x1EE4, 0x1EE4 }, // L&       LATIN CAPITAL LETTER U WITH DOT BELOW
    Interval{ 0x1EE6, 0x1EE6 }, // L&       LATIN CAPITAL LETTER U WITH HOOK ABOVE
    Interval{ 0x1EE8, 0x1EE8 }, // L&       LATIN CAPITAL LETTER U WITH HORN AND ACUTE
    Interval{ 0x1EEA, 0x1EEA }, // L&       LATIN CAPITAL LETTER U WITH HORN AND GRAVE
    Interval{ 0x1EEC, 0x1EEC }, // L&       LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
    Interval{ 0x1EEE, 0x1EEE }, // L&       LATIN CAPITAL LETTER U WITH HORN AND TILDE
    Interval{ 0x1EF0, 0x1EF0 }, // L&       LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
    Interval{ 0x1EF2, 0x1EF2 }, // L&       LATIN CAPITAL LETTER Y WITH GRAVE
    Interval{ 0x1EF4, 0x1EF4 }, // L&       LATIN CAPITAL LETTER Y WITH DOT BELOW
    Interval{ 0x1EF6, 0x1EF6 }, // L&       LATIN CAPITAL LETTER Y WITH HOOK ABOVE
    Interval{ 0x1EF8, 0x1EF8 }, // L&       LATIN CAPITAL LETTER Y WITH TILDE
    Interval{ 0x1EFA, 0x1EFA }, // L&       LATIN CAPITAL LETTER MIDDLE-WELSH LL
    Interval{ 0x1EFC, 0x1EFC }, // L&       LATIN CAPITAL LETTER MIDDLE-WELSH V
    Interval{ 0x1EFE, 0x1EFE }, // L&       LATIN CAPITAL LETTER Y WITH LOOP
    Interval{ 0x1F08, 0x1F0F }, // L&   [8] GREEK CAPITAL LETTER ALPHA WITH PSILI..GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F28, 0x1F2F }, // L&   [8] GREEK CAPITAL LETTER ETA WITH PSILI..GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F38, 0x1F3F }, // L&   [8] GREEK CAPITAL LETTER IOTA WITH PSILI..GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F5F }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F68, 0x1F6F }, // L&   [8] GREEK CAPITAL LETTER OMEGA WITH PSILI..GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
    Interval{ 0x1FB8, 0x1FBB }, // L&   [4] GREEK CAPITAL LETTER ALPHA WITH VRACHY..GREEK CAPITAL LETTER ALPHA WITH OXIA
    Interval{ 0x1FC8, 0x1FCB }, // L&   [4] GREEK CAPITAL LETTER EPSILON WITH VARIA..GREEK CAPITAL LETTER ETA WITH OXIA
    Interval{ 0x1FD8, 0x1FDB }, // L&   [4] GREEK CAPITAL LETTER IOTA WITH VRACHY..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE8, 0x1FEC }, // L&   [5] GREEK CAPITAL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF8, 0x1FFB }, // L&   [4] GREEK CAPITAL LETTER OMICRON WITH VARIA..GREEK CAPITAL LETTER OMEGA WITH OXIA
    Interval{ 0x2102, 0x2102 }, // L&       DOUBLE-STRUCK CAPITAL C
    Interval{ 0x2107, 0x2107 }, // L&       EULER CONSTANT
    Interval{ 0x210B, 0x210D }, // L&   [3] SCRIPT CAPITAL H..DOUBLE-STRUCK CAPITAL H
    Interval{ 0x2110, 0x2112 }, // L&   [3] SCRIPT CAPITAL I..SCRIPT CAPITAL L
    Interval{ 0x2115, 0x2115 }, // L&       DOUBLE-STRUCK CAPITAL N
    Interval{ 0x2119, 0x211D }, // L&   [5] DOUBLE-STRUCK CAPITAL P..DOUBLE-STRUCK CAPITAL R
    Interval{ 0x2124, 0x2124 }, // L&       DOUBLE-STRUCK CAPITAL Z
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x2128, 0x2128 }, // L&       BLACK-LETTER CAPITAL Z
    Interval{ 0x212A, 0x212D }, // L&   [4] KELVIN SIGN..BLACK-LETTER CAPITAL C
    Interval{ 0x2130, 0x2133 }, // L&   [4] SCRIPT CAPITAL E..SCRIPT CAPITAL M
    Interval{ 0x213E, 0x213F }, // L&   [2] DOUBLE-STRUCK CAPITAL GAMMA..DOUBLE-STRUCK CAPITAL PI
    Interval{ 0x2145, 0x2145 }, // L&       DOUBLE-STRUCK ITALIC CAPITAL D
    Interval{ 0x2160, 0x216F }, // Nl  [16] ROMAN NUMERAL ONE..ROMAN NUMERAL ONE THOUSAND
    Interval{ 0x2183, 0x2183 }, // L&       ROMAN NUMERAL REVERSED ONE HUNDRED
    Interval{ 0x24B6, 0x24CF }, // So  [26] CIRCLED LATIN CAPITAL LETTER A..CIRCLED LATIN CAPITAL LETTER Z
    Interval{ 0x2C00, 0x2C2F }, // L&  [48] GLAGOLITIC CAPITAL LETTER AZU..GLAGOLITIC CAPITAL LETTER CAUDATE CHRIVI
    Interval{ 0x2C60, 0x2C60 }, // L&       LATIN CAPITAL LETTER L WITH DOUBLE BAR
    Interval{ 0x2C62, 0x2C64 }, // L&   [3] LATIN CAPITAL LETTER L WITH MIDDLE TILDE..LATIN CAPITAL LETTER R WITH TAIL
    Interval{ 0x2C67, 0x2C67 }, // L&       LATIN CAPITAL LETTER H WITH DESCENDER
    Interval{ 0x2C69, 0x2C69 }, // L&       LATIN CAPITAL LETTER K WITH DESCENDER
    Interval{ 0x2C6B, 0x2C6B }, // L&       LATIN CAPITAL LETTER Z WITH DESCENDER
    Interval{ 0x2C6D, 0x2C70 }, // L&   [4] LATIN CAPITAL LETTER ALPHA..LATIN CAPITAL LETTER TURNED ALPHA
    Interval{ 0x2C72, 0x2C72 }, // L&       LATIN CAPITAL LETTER W WITH HOOK
    Interval{ 0x2C75, 0x2C75 }, // L&       LATIN CAPITAL LETTER HALF H
    Interval{ 0x2C7E, 0x2C80 }, // L&   [3] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC CAPITAL LETTER ALFA
    Interval{ 0x2C82, 0x2C82 }, // L&       COPTIC CAPITAL LETTER VIDA
    Interval{ 0x2C84, 0x2C84 }, // L&       COPTIC CAPITAL LETTER GAMMA
    Interval{ 0x2C86, 0x2C86 }, // L&       COPTIC CAPITAL LETTER DALDA
    Interval{ 0x2C88, 0x2C88 }, // L&       COPTIC CAPITAL LETTER EIE
    Interval{ 0x2C8A, 0x2C8A }, // L&       COPTIC CAPITAL LETTER SOU
    Interval{ 0x2C8C, 0x2C8C }, // L&       COPTIC CAPITAL LETTER ZATA
    Interval{ 0x2C8E, 0x2C8E }, // L&       COPTIC CAPITAL LETTER HATE
    Interval{ 0x2C90, 0x2C90 }, // L&       COPTIC CAPITAL LETTER THETHE
    Interval{ 0x2C92, 0x2C92 }, // L&       COPTIC CAPITAL LETTER IAUDA
    Interval{ 0x2C94, 0x2C94 }, // L&       COPTIC CAPITAL LETTER KAPA
    Interval{ 0x2C96, 0x2C96 }, // L&       COPTIC CAPITAL LETTER LAULA
    Interval{ 0x2C98, 0x2C98 }, // L&       COPTIC CAPITAL LETTER MI
    Interval{ 0x2C9A, 0x2C9A }, // L&       COPTIC CAPITAL LETTER NI
    Interval{ 0x2C9C, 0x2C9C }, // L&       COPTIC CAPITAL LETTER KSI
    Interval{ 0x2C9E, 0x2C9E }, // L&       COPTIC CAPITAL LETTER O
    Interval{ 0x2CA0, 0x2CA0 }, // L&       COPTIC CAPITAL LETTER PI
    Interval{ 0x2CA2, 0x2CA2 }, // L&       COPTIC CAPITAL LETTER RO
    Interval{ 0x2CA4, 0x2CA4 }, // L&       COPTIC CAPITAL LETTER SIMA
    Interval{ 0x2CA6, 0x2CA6 }, // L&       COPTIC CAPITAL LETTER TAU
    Interval{ 0x2CA8, 0x2CA8 }, // L&       COPTIC CAPITAL LETTER UA
    Interval{ 0x2CAA, 0x2CAA }, // L&       COPTIC CAPITAL LETTER FI
    Interval{ 0x2CAC, 0x2CAC }, // L&       COPTIC CAPITAL LETTER KHI
    Interval{ 0x2CAE, 0x2CAE }, // L&       COPTIC CAPITAL LETTER PSI
    Interval{ 0x2CB0, 0x2CB0 }, // L&       COPTIC CAPITAL LETTER OOU
    Interval{ 0x2CB2, 0x2CB2 }, // L&       COPTIC CAPITAL LETTER DIALECT-P ALEF
    Interval{ 0x2CB4, 0x2CB4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC AIN
    Interval{ 0x2CB6, 0x2CB6 }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
    Interval{ 0x2CB8, 0x2CB8 }, // L&       COPTIC CAPITAL LETTER DIALECT-P KAPA
    Interval{ 0x2CBA, 0x2CBA }, // L&       COPTIC CAPITAL LETTER DIALECT-P NI
    Interval{ 0x2CBC, 0x2CBC }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
    Interval{ 0x2CBE, 0x2CBE }, // L&       COPTIC CAPITAL LETTER OLD COPTIC OOU
    Interval{ 0x2CC0, 0x2CC0 }, // L&       COPTIC CAPITAL LETTER SAMPI
    Interval{ 0x2CC2, 0x2CC2 }, // L&       COPTIC CAPITAL LETTER CROSSED SHEI
    Interval{ 0x2CC4, 0x2CC4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC SHEI
    Interval{ 0x2CC6, 0x2CC6 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC ESH
    Interval{ 0x2CC8, 0x2CC8 }, // L&       COPTIC CAPITAL LETTER AKHMIMIC KHEI
    Interval{ 0x2CCA, 0x2CCA }, // L&       COPTIC CAPITAL LETTER DIALECT-P HORI
    Interval{ 0x2CCC, 0x2CCC }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HORI
    Interval{ 0x2CCE, 0x2CCE }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HA
    Interval{ 0x2CD0, 0x2CD0 }, // L&       COPTIC CAPITAL LETTER L-SHAPED HA
    Interval{ 0x2CD2, 0x2CD2 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HEI
    Interval{ 0x2CD4, 0x2CD4 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC HAT
    Interval{ 0x2CD6, 0x2CD6 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC GANGIA
    Interval{ 0x2CD8, 0x2CD8 }, // L&       COPTIC CAPITAL LETTER OLD COPTIC DJA
    Interval{ 0x2CDA, 0x2CDA }, // L&       COPTIC CAPITAL LETTER OLD COPTIC SHIMA
    Interval{ 0x2CDC, 0x2CDC }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
    Interval{ 0x2CDE, 0x2CDE }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN NGI
    Interval{ 0x2CE0, 0x2CE0 }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN NYI
    Interval{ 0x2CE2, 0x2CE2 }, // L&       COPTIC CAPITAL LETTER OLD NUBIAN WAU
    Interval{ 0x2CEB, 0x2CEB }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI
    Interval{ 0x2CED, 0x2CED }, // L&       COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CF2, 0x2CF2 }, // L&       COPTIC CAPITAL LETTER BOHAIRIC KHEI
    Interval{ 0xA640, 0xA640 }, // L&       CYRILLIC CAPITAL LETTER ZEMLYA
    Interval{ 0xA642, 0xA642 }, // L&       CYRILLIC CAPITAL LETTER DZELO
    Interval{ 0xA644, 0xA644 }, // L&       CYRILLIC CAPITAL LETTER REVERSED DZE
    Interval{ 0xA646, 0xA646 }, // L&       CYRILLIC CAPITAL LETTER IOTA
    Interval{ 0xA648, 0xA648 }, // L&       CYRILLIC CAPITAL LETTER DJERV
    Interval{ 0xA64A, 0xA64A }, // L&       CYRILLIC CAPITAL LETTER MONOGRAPH UK
    Interval{ 0xA64C, 0xA64C }, // L&       CYRILLIC CAPITAL LETTER BROAD OMEGA
    Interval{ 0xA64E, 0xA64E }, // L&       CYRILLIC CAPITAL LETTER NEUTRAL YER
    Interval{ 0xA650, 0xA650 }, // L&       CYRILLIC CAPITAL LETTER YERU WITH BACK YER
    Interval{ 0xA652, 0xA652 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED YAT
    Interval{ 0xA654, 0xA654 }, // L&       CYRILLIC CAPITAL LETTER REVERSED YU
    Interval{ 0xA656, 0xA656 }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED A
    Interval{ 0xA658, 0xA658 }, // L&       CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS
    Interval{ 0xA65A, 0xA65A }, // L&       CYRILLIC CAPITAL LETTER BLENDED YUS
    Interval{ 0xA65C, 0xA65C }, // L&       CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUS
    Interval{ 0xA65E, 0xA65E }, // L&       CYRILLIC CAPITAL LETTER YN
    Interval{ 0xA660, 0xA660 }, // L&       CYRILLIC CAPITAL LETTER REVERSED TSE
    Interval{ 0xA662, 0xA662 }, // L&       CYRILLIC CAPITAL LETTER SOFT DE
    Interval{ 0xA664, 0xA664 }, // L&       CYRILLIC CAPITAL LETTER SOFT EL
    Interval{ 0xA666, 0xA666 }, // L&       CYRILLIC CAPITAL LETTER SOFT EM
    Interval{ 0xA668, 0xA668 }, // L&       CYRILLIC CAPITAL LETTER MONOCULAR O
    Interval{ 0xA66A, 0xA66A }, // L&       CYRILLIC CAPITAL LETTER BINOCULAR O
    Interval{ 0xA66C, 0xA66C }, // L&       CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA680, 0xA680 }, // L&       CYRILLIC CAPITAL LETTER DWE
    Interval{ 0xA682, 0xA682 }, // L&       CYRILLIC CAPITAL LETTER DZWE
    Interval{ 0xA684, 0xA684 }, // L&       CYRILLIC CAPITAL LETTER ZHWE
    Interval{ 0xA686, 0xA686 }, // L&       CYRILLIC CAPITAL LETTER CCHE
    Interval{ 0xA688, 0xA688 }, // L&       CYRILLIC CAPITAL LETTER DZZE
    Interval{ 0xA68A, 0xA68A }, // L&       CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK
    Interval{ 0xA68C, 0xA68C }, // L&       CYRILLIC CAPITAL LETTER TWE
    Interval{ 0xA68E, 0xA68E }, // L&       CYRILLIC CAPITAL LETTER TSWE
    Interval{ 0xA690, 0xA690 }, // L&       CYRILLIC CAPITAL LETTER TSSE
    Interval{ 0xA692, 0xA692 }, // L&       CYRILLIC CAPITAL LETTER TCHE
    Interval{ 0xA694, 0xA694 }, // L&       CYRILLIC CAPITAL LETTER HWE
    Interval{ 0xA696, 0xA696 }, // L&       CYRILLIC CAPITAL LETTER SHWE
    Interval{ 0xA698, 0xA698 }, // L&       CYRILLIC CAPITAL LETTER DOUBLE O
    Interval{ 0xA69A, 0xA69A }, // L&       CYRILLIC CAPITAL LETTER CROSSED O
    Interval{ 0xA722, 0xA722 }, // L&       LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF
    Interval{ 0xA724, 0xA724 }, // L&       LATIN CAPITAL LETTER EGYPTOLOGICAL AIN
    Interval{ 0xA726, 0xA726 }, // L&       LATIN CAPITAL LETTER HENG
    Interval{ 0xA728, 0xA728 }, // L&       LATIN CAPITAL LETTER TZ
    Interval{ 0xA72A, 0xA72A }, // L&       LATIN CAPITAL LETTER TRESILLO
    Interval{ 0xA72C, 0xA72C }, // L&       LATIN CAPITAL LETTER CUATRILLO
    Interval{ 0xA72E, 0xA72E }, // L&       LATIN CAPITAL LETTER CUATRILLO WITH COMMA
    Interval{ 0xA732, 0xA732 }, // L&       LATIN CAPITAL LETTER AA
    Interval{ 0xA734, 0xA734 }, // L&       LATIN CAPITAL LETTER AO
    Interval{ 0xA736, 0xA736 }, // L&       LATIN CAPITAL LETTER AU
    Interval{ 0xA738, 0xA738 }, // L&       LATIN CAPITAL LETTER AV
    Interval{ 0xA73A, 0xA73A }, // L&       LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR
    Interval{ 0xA73C, 0xA73C }, // L&       LATIN CAPITAL LETTER AY
    Interval{ 0xA73E, 0xA73E }, // L&       LATIN CAPITAL LETTER REVERSED C WITH DOT
    Interval{ 0xA740, 0xA740 }, // L&       LATIN CAPITAL LETTER K WITH STROKE
    Interval{ 0xA742, 0xA742 }, // L&       LATIN CAPITAL LETTER K WITH DIAGONAL STROKE
    Interval{ 0xA744, 0xA744 }, // L&       LATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKE
    Interval{ 0xA746, 0xA746 }, // L&       LATIN CAPITAL LETTER BROKEN L
    Interval{ 0xA748, 0xA748 }, // L&       LATIN CAPITAL LETTER L WITH HIGH STROKE
    Interval{ 0xA74A, 0xA74A }, // L&       LATIN CAPITAL LETTER O WITH LONG STROKE OVERLAY
    Interval{ 0xA74C, 0xA74C }, // L&       LATIN CAPITAL LETTER O WITH LOOP
    Interval{ 0xA74E, 0xA74E }, // L&       LATIN CAPITAL LETTER OO
    Interval{ 0xA750, 0xA750 }, // L&       LATIN CAPITAL LETTER P WITH STROKE THROUGH DESCENDER
    Interval{ 0xA752, 0xA752 }, // L&       LATIN CAPITAL LETTER P WITH FLOURISH
    Interval{ 0xA754, 0xA754 }, // L&       LATIN CAPITAL LETTER P WITH SQUIRREL TAIL
    Interval{ 0xA756, 0xA756 }, // L&       LATIN CAPITAL LETTER Q WITH STROKE THROUGH DESCENDER
    Interval{ 0xA758, 0xA758 }, // L&       LATIN CAPITAL LETTER Q WITH DIAGONAL STROKE
    Interval{ 0xA75A, 0xA75A }, // L&       LATIN CAPITAL LETTER R ROTUNDA
    Interval{ 0xA75C, 0xA75C }, // L&       LATIN CAPITAL LETTER RUM ROTUNDA
    Interval{ 0xA75E, 0xA75E }, // L&       LATIN CAPITAL LETTER V WITH DIAGONAL STROKE
    Interval{ 0xA760, 0xA760 }, // L&       LATIN CAPITAL LETTER VY
    Interval{ 0xA762, 0xA762 }, // L&       LATIN CAPITAL LETTER VISIGOTHIC Z
    Interval{ 0xA764, 0xA764 }, // L&       LATIN CAPITAL LETTER THORN WITH STROKE
    Interval{ 0xA766, 0xA766 }, // L&       LATIN CAPITAL LETTER THORN WITH STROKE THROUGH DESCENDER
    Interval{ 0xA768, 0xA768 }, // L&       LATIN CAPITAL LETTER VEND
    Interval{ 0xA76A, 0xA76A }, // L&       LATIN CAPITAL LETTER ET
    Interval{ 0xA76C, 0xA76C }, // L&       LATIN CAPITAL LETTER IS
    Interval{ 0xA76E, 0xA76E }, // L&       LATIN CAPITAL LETTER CON
    Interval{ 0xA779, 0xA779 }, // L&       LATIN CAPITAL LETTER INSULAR D
    Interval{ 0xA77B, 0xA77B }, // L&       LATIN CAPITAL LETTER INSULAR F
    Interval{ 0xA77D, 0xA77E }, // L&   [2] LATIN CAPITAL LETTER INSULAR G..LATIN CAPITAL LETTER TURNED INSULAR G
    Interval{ 0xA780, 0xA780 }, // L&       LATIN CAPITAL LETTER TURNED L
    Interval{ 0xA782, 0xA782 }, // L&       LATIN CAPITAL LETTER INSULAR R
    Interval{ 0xA784, 0xA784 }, // L&       LATIN CAPITAL LETTER INSULAR S
    Interval{ 0xA786, 0xA786 }, // L&       LATIN CAPITAL LETTER INSULAR T
    Interval{ 0xA78B, 0xA78B }, // L&       LATIN CAPITAL LETTER SALTILLO
    Interval{ 0xA78D, 0xA78D }, // L&       LATIN CAPITAL LETTER TURNED H
    Interval{ 0xA790, 0xA790 }, // L&       LATIN CAPITAL LETTER N WITH DESCENDER
    Interval{ 0xA792, 0xA792 }, // L&       LATIN CAPITAL LETTER C WITH BAR
    Interval{ 0xA796, 0xA796 }, // L&       LATIN CAPITAL LETTER B WITH FLOURISH
    Interval{ 0xA798, 0xA798 }, // L&       LATIN CAPITAL LETTER F WITH STROKE
    Interval{ 0xA79A, 0xA79A }, // L&       LATIN CAPITAL LETTER VOLAPUK AE
    Interval{ 0xA79C, 0xA79C }, // L&       LATIN CAPITAL LETTER VOLAPUK OE
    Interval{ 0xA79E, 0xA79E }, // L&       LATIN CAPITAL LETTER VOLAPUK UE
    Interval{ 0xA7A0, 0xA7A0 }, // L&       LATIN CAPITAL LETTER G WITH OBLIQUE STROKE
    Interval{ 0xA7A2, 0xA7A2 }, // L&       LATIN CAPITAL LETTER K WITH OBLIQUE STROKE
    Interval{ 0xA7A4, 0xA7A4 }, // L&       LATIN CAPITAL LETTER N WITH OBLIQUE STROKE
    Interval{ 0xA7A6, 0xA7A6 }, // L&       LATIN CAPITAL LETTER R WITH OBLIQUE STROKE
    Interval{ 0xA7A8, 0xA7A8 }, // L&       LATIN CAPITAL LETTER S WITH OBLIQUE STROKE
    Interval{ 0xA7AA, 0xA7AE }, // L&   [5] LATIN CAPITAL LETTER H WITH HOOK..LATIN CAPITAL LETTER SMALL CAPITAL I
    Interval{ 0xA7B0, 0xA7B4 }, // L&   [5] LATIN CAPITAL LETTER TURNED K..LATIN CAPITAL LETTER BETA
    Interval{ 0xA7B6, 0xA7B6 }, // L&       LATIN CAPITAL LETTER OMEGA
    Interval{ 0xA7B8, 0xA7B8 }, // L&       LATIN CAPITAL LETTER U WITH STROKE
    Interval{ 0xA7BA, 0xA7BA }, // L&       LATIN CAPITAL LETTER GLOTTAL A
    Interval{ 0xA7BC, 0xA7BC }, // L&       LATIN CAPITAL LETTER GLOTTAL I
    Interval{ 0xA7BE, 0xA7BE }, // L&       LATIN CAPITAL LETTER GLOTTAL U
    Interval{ 0xA7C0, 0xA7C0 }, // L&       LATIN CAPITAL LETTER OLD POLISH O
    Interval{ 0xA7C2, 0xA7C2 }, // L&       LATIN CAPITAL LETTER ANGLICANA W
    Interval{ 0xA7C4, 0xA7C7 }, // L&   [4] LATIN CAPITAL LETTER C WITH PALATAL HOOK..LATIN CAPITAL LETTER D WITH SHORT STROKE OVERLAY
    Interval{ 0xA7C9, 0xA7C9 }, // L&       LATIN CAPITAL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D0 }, // L&       LATIN CAPITAL LETTER CLOSED INSULAR G
    Interval{ 0xA7D6, 0xA7D6 }, // L&       LATIN CAPITAL LETTER MIDDLE SCOTS S
    Interval{ 0xA7D8, 0xA7D8 }, // L&       LATIN CAPITAL LETTER SIGMOID S
    Interval{ 0xA7F5, 0xA7F5 }, // L&       LATIN CAPITAL LETTER REVERSED HALF H
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0x10400, 0x10427 }, // L&  [40] DESERET CAPITAL LETTER LONG I..DESERET CAPITAL LETTER EW
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x118A0, 0x118BF }, // L&  [32] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI CAPITAL LETTER VIYO
    Interval{ 0x16E40, 0x16E5F }, // L&  [32] MEDEFAIDRIN CAPITAL LETTER M..MEDEFAIDRIN CAPITAL LETTER Y
    Interval{ 0x1D400, 0x1D419 }, // L&  [26] MATHEMATICAL BOLD CAPITAL A..MATHEMATICAL BOLD CAPITAL Z
    Interval{ 0x1D434, 0x1D44D }, // L&  [26] MATHEMATICAL ITALIC CAPITAL A..MATHEMATICAL ITALIC CAPITAL Z
    Interval{ 0x1D468, 0x1D481 }, // L&  [26] MATHEMATICAL BOLD ITALIC CAPITAL A..MATHEMATICAL BOLD ITALIC CAPITAL Z
    Interval{ 0x1D49C, 0x1D49C }, // L&       MATHEMATICAL SCRIPT CAPITAL A
    Interval{ 0x1D49E, 0x1D49F }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL C..MATHEMATICAL SCRIPT CAPITAL D
    Interval{ 0x1D4A2, 0x1D4A2 }, // L&       MATHEMATICAL SCRIPT CAPITAL G
    Interval{ 0x1D4A5, 0x1D4A6 }, // L&   [2] MATHEMATICAL SCRIPT CAPITAL J..MATHEMATICAL SCRIPT CAPITAL K
    Interval{ 0x1D4A9, 0x1D4AC }, // L&   [4] MATHEMATICAL SCRIPT CAPITAL N..MATHEMATICAL SCRIPT CAPITAL Q
    Interval{ 0x1D4AE, 0x1D4B5 }, // L&   [8] MATHEMATICAL SCRIPT CAPITAL S..MATHEMATICAL SCRIPT CAPITAL Z
    Interval{ 0x1D4D0, 0x1D4E9 }, // L&  [26] MATHEMATICAL BOLD SCRIPT CAPITAL A..MATHEMATICAL BOLD SCRIPT CAPITAL Z
    Interval{ 0x1D504, 0x1D505 }, // L&   [2] MATHEMATICAL FRAKTUR CAPITAL A..MATHEMATICAL FRAKTUR CAPITAL B
    Interval{ 0x1D507, 0x1D50A }, // L&   [4] MATHEMATICAL FRAKTUR CAPITAL D..MATHEMATICAL FRAKTUR CAPITAL G
    Interval{ 0x1D50D, 0x1D514 }, // L&   [8] MATHEMATICAL FRAKTUR CAPITAL J..MATHEMATICAL FRAKTUR CAPITAL Q
    Interval{ 0x1D516, 0x1D51C }, // L&   [7] MATHEMATICAL FRAKTUR CAPITAL S..MATHEMATICAL FRAKTUR CAPITAL Y
    Interval{ 0x1D538, 0x1D539 }, // L&   [2] MATHEMATICAL DOUBLE-STRUCK CAPITAL A..MATHEMATICAL DOUBLE-STRUCK CAPITAL B
    Interval{ 0x1D53B, 0x1D53E }, // L&   [4] MATHEMATICAL DOUBLE-STRUCK CAPITAL D..MATHEMATICAL DOUBLE-STRUCK CAPITAL G
    Interval{ 0x1D540, 0x1D544 }, // L&   [5] MATHEMATICAL DOUBLE-STRUCK CAPITAL I..MATHEMATICAL DOUBLE-STRUCK CAPITAL M
    Interval{ 0x1D546, 0x1D546 }, // L&       MATHEMATICAL DOUBLE-STRUCK CAPITAL O
    Interval{ 0x1D54A, 0x1D550 }, // L&   [7] MATHEMATICAL DOUBLE-STRUCK CAPITAL S..MATHEMATICAL DOUBLE-STRUCK CAPITAL Y
    Interval{ 0x1D56C, 0x1D585 }, // L&  [26] MATHEMATICAL BOLD FRAKTUR CAPITAL A..MATHEMATICAL BOLD FRAKTUR CAPITAL Z
    Interval{ 0x1D5A0, 0x1D5B9 }, // L&  [26] MATHEMATICAL SANS-SERIF CAPITAL A..MATHEMATICAL SANS-SERIF CAPITAL Z
    Interval{ 0x1D5D4, 0x1D5ED }, // L&  [26] MATHEMATICAL SANS-SERIF BOLD CAPITAL A..MATHEMATICAL SANS-SERIF BOLD CAPITAL Z
    Interval{ 0x1D608, 0x1D621 }, // L&  [26] MATHEMATICAL SANS-SERIF ITALIC CAPITAL A..MATHEMATICAL SANS-SERIF ITALIC CAPITAL Z
    Interval{ 0x1D63C, 0x1D655 }, // L&  [26] MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL A..MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL Z
    Interval{ 0x1D670, 0x1D689 }, // L&  [26] MATHEMATICAL MONOSPACE CAPITAL A..MATHEMATICAL MONOSPACE CAPITAL Z
    Interval{ 0x1D6A8, 0x1D6C0 }, // L&  [25] MATHEMATICAL BOLD CAPITAL ALPHA..MATHEMATICAL BOLD CAPITAL OMEGA
    Interval{ 0x1D6E2, 0x1D6FA }, // L&  [25] MATHEMATICAL ITALIC CAPITAL ALPHA..MATHEMATICAL ITALIC CAPITAL OMEGA
    Interval{ 0x1D71C, 0x1D734 }, // L&  [25] MATHEMATICAL BOLD ITALIC CAPITAL ALPHA..MATHEMATICAL BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D756, 0x1D76E }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD CAPITAL ALPHA..MATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGA
    Interval{ 0x1D790, 0x1D7A8 }, // L&  [25] MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL ALPHA..MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGA
    Interval{ 0x1D7CA, 0x1D7CA }, // L&       MATHEMATICAL BOLD CAPITAL DIGAMMA
    Interval{ 0x1E900, 0x1E921 }, // L&  [34] ADLAM CAPITAL LETTER ALIF..ADLAM CAPITAL LETTER SHA
    Interval{ 0x1F130, 0x1F149 }, // So  [26] SQUARED LATIN CAPITAL LETTER A..SQUARED LATIN CAPITAL LETTER Z
    Interval{ 0x1F150, 0x1F169 }, // So  [26] NEGATIVE CIRCLED LATIN CAPITAL LETTER A..NEGATIVE CIRCLED LATIN CAPITAL LETTER Z
    Interval{ 0x1F170, 0x1F189 }, // So  [26] NEGATIVE SQUARED LATIN CAPITAL LETTER A..NEGATIVE SQUARED LATIN CAPITAL LETTER Z
}; // }}}
auto static const XID_Continue = std::array<Interval, 1344>{ // {{{
    Interval{ 0x0030, 0x0039 }, // Nd  [10] DIGIT ZERO..DIGIT NINE
    Interval{ 0x0041, 0x005A }, // L&  [26] LATIN CAPITAL LETTER A..LATIN CAPITAL LETTER Z
    Interval{ 0x005F, 0x005F }, // Pc       LOW LINE
    Interval{ 0x0061, 0x007A }, // L&  [26] LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    Interval{ 0x00AA, 0x00AA }, // Lo       FEMININE ORDINAL INDICATOR
    Interval{ 0x00B5, 0x00B5 }, // L&       MICRO SIGN
    Interval{ 0x00B7, 0x00B7 }, // Po       MIDDLE DOT
    Interval{ 0x00BA, 0x00BA }, // Lo       MASCULINE ORDINAL INDICATOR
    Interval{ 0x00C0, 0x00D6 }, // L&  [23] LATIN CAPITAL LETTER A WITH GRAVE..LATIN CAPITAL LETTER O WITH DIAERESIS
    Interval{ 0x00D8, 0x00F6 }, // L&  [31] LATIN CAPITAL LETTER O WITH STROKE..LATIN SMALL LETTER O WITH DIAERESIS
    Interval{ 0x00F8, 0x01BA }, // L& [195] LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER EZH WITH TAIL
    Interval{ 0x01BB, 0x01BB }, // Lo       LATIN LETTER TWO WITH STROKE
    Interval{ 0x01BC, 0x01BF }, // L&   [4] LATIN CAPITAL LETTER TONE FIVE..LATIN LETTER WYNN
    Interval{ 0x01C0, 0x01C3 }, // Lo   [4] LATIN LETTER DENTAL CLICK..LATIN LETTER RETROFLEX CLICK
    Interval{ 0x01C4, 0x0293 }, // L& [208] LATIN CAPITAL LETTER DZ WITH CARON..LATIN SMALL LETTER EZH WITH CURL
    Interval{ 0x0294, 0x0294 }, // Lo       LATIN LETTER GLOTTAL STOP
    Interval{ 0x0295, 0x02AF }, // L&  [27] LATIN LETTER PHARYNGEAL VOICED FRICATIVE..LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    Interval{ 0x02B0, 0x02C1 }, // Lm  [18] MODIFIER LETTER SMALL H..MODIFIER LETTER REVERSED GLOTTAL STOP
    Interval{ 0x02C6, 0x02D1 }, // Lm  [12] MODIFIER LETTER CIRCUMFLEX ACCENT..MODIFIER LETTER HALF TRIANGULAR COLON
    Interval{ 0x02E0, 0x02E4 }, // Lm   [5] MODIFIER LETTER SMALL GAMMA..MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
    Interval{ 0x02EC, 0x02EC }, // Lm       MODIFIER LETTER VOICING
    Interval{ 0x02EE, 0x02EE }, // Lm       MODIFIER LETTER DOUBLE APOSTROPHE
    Interval{ 0x0300, 0x036F }, // Mn [112] COMBINING GRAVE ACCENT..COMBINING LATIN SMALL LETTER X
    Interval{ 0x0370, 0x0373 }, // L&   [4] GREEK CAPITAL LETTER HETA..GREEK SMALL LETTER ARCHAIC SAMPI
    Interval{ 0x0374, 0x0374 }, // Lm       GREEK NUMERAL SIGN
    Interval{ 0x0376, 0x0377 }, // L&   [2] GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA..GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
    Interval{ 0x037B, 0x037D }, // L&   [3] GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
    Interval{ 0x037F, 0x037F }, // L&       GREEK CAPITAL LETTER YOT
    Interval{ 0x0386, 0x0386 }, // L&       GREEK CAPITAL LETTER ALPHA WITH TONOS
    Interval{ 0x0387, 0x0387 }, // Po       GREEK ANO TELEIA
    Interval{ 0x0388, 0x038A }, // L&   [3] GREEK CAPITAL LETTER EPSILON WITH TONOS..GREEK CAPITAL LETTER IOTA WITH TONOS
    Interval{ 0x038C, 0x038C }, // L&       GREEK CAPITAL LETTER OMICRON WITH TONOS
    Interval{ 0x038E, 0x03A1 }, // L&  [20] GREEK CAPITAL LETTER UPSILON WITH TONOS..GREEK CAPITAL LETTER RHO
    Interval{ 0x03A3, 0x03F5 }, // L&  [83] GREEK CAPITAL LETTER SIGMA..GREEK LUNATE EPSILON SYMBOL
    Interval{ 0x03F7, 0x0481 }, // L& [139] GREEK CAPITAL LETTER SHO..CYRILLIC SMALL LETTER KOPPA
    Interval{ 0x0483, 0x0487 }, // Mn   [5] COMBINING CYRILLIC TITLO..COMBINING CYRILLIC POKRYTIE
    Interval{ 0x048A, 0x052F }, // L& [166] CYRILLIC CAPITAL LETTER SHORT I WITH TAIL..CYRILLIC SMALL LETTER EL WITH DESCENDER
    Interval{ 0x0531, 0x0556 }, // L&  [38] ARMENIAN CAPITAL LETTER AYB..ARMENIAN CAPITAL LETTER FEH
    Interval{ 0x0559, 0x0559 }, // Lm       ARMENIAN MODIFIER LETTER LEFT HALF RING
    Interval{ 0x0560, 0x0588 }, // L&  [41] ARMENIAN SMALL LETTER TURNED AYB..ARMENIAN SMALL LETTER YI WITH STROKE
    Interval{ 0x0591, 0x05BD }, // Mn  [45] HEBREW ACCENT ETNAHTA..HEBREW POINT METEG
    Interval{ 0x05BF, 0x05BF }, // Mn       HEBREW POINT RAFE
    Interval{ 0x05C1, 0x05C2 }, // Mn   [2] HEBREW POINT SHIN DOT..HEBREW POINT SIN DOT
    Interval{ 0x05C4, 0x05C5 }, // Mn   [2] HEBREW MARK UPPER DOT..HEBREW MARK LOWER DOT
    Interval{ 0x05C7, 0x05C7 }, // Mn       HEBREW POINT QAMATS QATAN
    Interval{ 0x05D0, 0x05EA }, // Lo  [27] HEBREW LETTER ALEF..HEBREW LETTER TAV
    Interval{ 0x05EF, 0x05F2 }, // Lo   [4] HEBREW YOD TRIANGLE..HEBREW LIGATURE YIDDISH DOUBLE YOD
    Interval{ 0x0610, 0x061A }, // Mn  [11] ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM..ARABIC SMALL KASRA
    Interval{ 0x0620, 0x063F }, // Lo  [32] ARABIC LETTER KASHMIRI YEH..ARABIC LETTER FARSI YEH WITH THREE DOTS ABOVE
    Interval{ 0x0640, 0x0640 }, // Lm       ARABIC TATWEEL
    Interval{ 0x0641, 0x064A }, // Lo  [10] ARABIC LETTER FEH..ARABIC LETTER YEH
    Interval{ 0x064B, 0x065F }, // Mn  [21] ARABIC FATHATAN..ARABIC WAVY HAMZA BELOW
    Interval{ 0x0660, 0x0669 }, // Nd  [10] ARABIC-INDIC DIGIT ZERO..ARABIC-INDIC DIGIT NINE
    Interval{ 0x066E, 0x066F }, // Lo   [2] ARABIC LETTER DOTLESS BEH..ARABIC LETTER DOTLESS QAF
    Interval{ 0x0670, 0x0670 }, // Mn       ARABIC LETTER SUPERSCRIPT ALEF
    Interval{ 0x0671, 0x06D3 }, // Lo  [99] ARABIC LETTER ALEF WASLA..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
    Interval{ 0x06D5, 0x06D5 }, // Lo       ARABIC LETTER AE
    Interval{ 0x06D6, 0x06DC }, // Mn   [7] ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA..ARABIC SMALL HIGH SEEN
    Interval{ 0x06DF, 0x06E4 }, // Mn   [6] ARABIC SMALL HIGH ROUNDED ZERO..ARABIC SMALL HIGH MADDA
    Interval{ 0x06E5, 0x06E6 }, // Lm   [2] ARABIC SMALL WAW..ARABIC SMALL YEH
    Interval{ 0x06E7, 0x06E8 }, // Mn   [2] ARABIC SMALL HIGH YEH..ARABIC SMALL HIGH NOON
    Interval{ 0x06EA, 0x06ED }, // Mn   [4] ARABIC EMPTY CENTRE LOW STOP..ARABIC SMALL LOW MEEM
    Interval{ 0x06EE, 0x06EF }, // Lo   [2] ARABIC LETTER DAL WITH INVERTED V..ARABIC LETTER REH WITH INVERTED V
    Interval{ 0x06F0, 0x06F9 }, // Nd  [10] EXTENDED ARABIC-INDIC DIGIT ZERO..EXTENDED ARABIC-INDIC DIGIT NINE
    Interval{ 0x06FA, 0x06FC }, // Lo   [3] ARABIC LETTER SHEEN WITH DOT BELOW..ARABIC LETTER GHAIN WITH DOT BELOW
    Interval{ 0x06FF, 0x06FF }, // Lo       ARABIC LETTER HEH WITH INVERTED V
    Interval{ 0x0710, 0x0710 }, // Lo       SYRIAC LETTER ALAPH
    Interval{ 0x0711, 0x0711 }, // Mn       SYRIAC LETTER SUPERSCRIPT ALAPH
    Interval{ 0x0712, 0x072F }, // Lo  [30] SYRIAC LETTER BETH..SYRIAC LETTER PERSIAN DHALATH
    Interval{ 0x0730, 0x074A }, // Mn  [27] SYRIAC PTHAHA ABOVE..SYRIAC BARREKH
    Interval{ 0x074D, 0x07A5 }, // Lo  [89] SYRIAC LETTER SOGDIAN ZHAIN..THAANA LETTER WAAVU
    Interval{ 0x07A6, 0x07B0 }, // Mn  [11] THAANA ABAFILI..THAANA SUKUN
    Interval{ 0x07B1, 0x07B1 }, // Lo       THAANA LETTER NAA
    Interval{ 0x07C0, 0x07C9 }, // Nd  [10] NKO DIGIT ZERO..NKO DIGIT NINE
    Interval{ 0x07CA, 0x07EA }, // Lo  [33] NKO LETTER A..NKO LETTER JONA RA
    Interval{ 0x07EB, 0x07F3 }, // Mn   [9] NKO COMBINING SHORT HIGH TONE..NKO COMBINING DOUBLE DOT ABOVE
    Interval{ 0x07F4, 0x07F5 }, // Lm   [2] NKO HIGH TONE APOSTROPHE..NKO LOW TONE APOSTROPHE
    Interval{ 0x07FA, 0x07FA }, // Lm       NKO LAJANYALAN
    Interval{ 0x07FD, 0x07FD }, // Mn       NKO DANTAYALAN
    Interval{ 0x0800, 0x0815 }, // Lo  [22] SAMARITAN LETTER ALAF..SAMARITAN LETTER TAAF
    Interval{ 0x0816, 0x0819 }, // Mn   [4] SAMARITAN MARK IN..SAMARITAN MARK DAGESH
    Interval{ 0x081A, 0x081A }, // Lm       SAMARITAN MODIFIER LETTER EPENTHETIC YUT
    Interval{ 0x081B, 0x0823 }, // Mn   [9] SAMARITAN MARK EPENTHETIC YUT..SAMARITAN VOWEL SIGN A
    Interval{ 0x0824, 0x0824 }, // Lm       SAMARITAN MODIFIER LETTER SHORT A
    Interval{ 0x0825, 0x0827 }, // Mn   [3] SAMARITAN VOWEL SIGN SHORT A..SAMARITAN VOWEL SIGN U
    Interval{ 0x0828, 0x0828 }, // Lm       SAMARITAN MODIFIER LETTER I
    Interval{ 0x0829, 0x082D }, // Mn   [5] SAMARITAN VOWEL SIGN LONG I..SAMARITAN MARK NEQUDAA
    Interval{ 0x0840, 0x0858 }, // Lo  [25] MANDAIC LETTER HALQA..MANDAIC LETTER AIN
    Interval{ 0x0859, 0x085B }, // Mn   [3] MANDAIC AFFRICATION MARK..MANDAIC GEMINATION MARK
    Interval{ 0x0860, 0x086A }, // Lo  [11] SYRIAC LETTER MALAYALAM NGA..SYRIAC LETTER MALAYALAM SSA
    Interval{ 0x0870, 0x0887 }, // Lo  [24] ARABIC LETTER ALEF WITH ATTACHED FATHA..ARABIC BASELINE ROUND DOT
    Interval{ 0x0889, 0x088E }, // Lo   [6] ARABIC LETTER NOON WITH INVERTED SMALL V..ARABIC VERTICAL TAIL
    Interval{ 0x0898, 0x089F }, // Mn   [8] ARABIC SMALL HIGH WORD AL-JUZ..ARABIC HALF MADDA OVER MADDA
    Interval{ 0x08A0, 0x08C8 }, // Lo  [41] ARABIC LETTER BEH WITH SMALL V BELOW..ARABIC LETTER GRAF
    Interval{ 0x08C9, 0x08C9 }, // Lm       ARABIC SMALL FARSI YEH
    Interval{ 0x08CA, 0x08E1 }, // Mn  [24] ARABIC SMALL HIGH FARSI YEH..ARABIC SMALL HIGH SIGN SAFHA
    Interval{ 0x08E3, 0x0902 }, // Mn  [32] ARABIC TURNED DAMMA BELOW..DEVANAGARI SIGN ANUSVARA
    Interval{ 0x0903, 0x0903 }, // Mc       DEVANAGARI SIGN VISARGA
    Interval{ 0x0904, 0x0939 }, // Lo  [54] DEVANAGARI LETTER SHORT A..DEVANAGARI LETTER HA
    Interval{ 0x093A, 0x093A }, // Mn       DEVANAGARI VOWEL SIGN OE
    Interval{ 0x093B, 0x093B }, // Mc       DEVANAGARI VOWEL SIGN OOE
    Interval{ 0x093C, 0x093C }, // Mn       DEVANAGARI SIGN NUKTA
    Interval{ 0x093D, 0x093D }, // Lo       DEVANAGARI SIGN AVAGRAHA
    Interval{ 0x093E, 0x0940 }, // Mc   [3] DEVANAGARI VOWEL SIGN AA..DEVANAGARI VOWEL SIGN II
    Interval{ 0x0941, 0x0948 }, // Mn   [8] DEVANAGARI VOWEL SIGN U..DEVANAGARI VOWEL SIGN AI
    Interval{ 0x0949, 0x094C }, // Mc   [4] DEVANAGARI VOWEL SIGN CANDRA O..DEVANAGARI VOWEL SIGN AU
    Interval{ 0x094D, 0x094D }, // Mn       DEVANAGARI SIGN VIRAMA
    Interval{ 0x094E, 0x094F }, // Mc   [2] DEVANAGARI VOWEL SIGN PRISHTHAMATRA E..DEVANAGARI VOWEL SIGN AW
    Interval{ 0x0950, 0x0950 }, // Lo       DEVANAGARI OM
    Interval{ 0x0951, 0x0957 }, // Mn   [7] DEVANAGARI STRESS SIGN UDATTA..DEVANAGARI VOWEL SIGN UUE
    Interval{ 0x0958, 0x0961 }, // Lo  [10] DEVANAGARI LETTER QA..DEVANAGARI LETTER VOCALIC LL
    Interval{ 0x0962, 0x0963 }, // Mn   [2] DEVANAGARI VOWEL SIGN VOCALIC L..DEVANAGARI VOWEL SIGN VOCALIC LL
    Interval{ 0x0966, 0x096F }, // Nd  [10] DEVANAGARI DIGIT ZERO..DEVANAGARI DIGIT NINE
    Interval{ 0x0971, 0x0971 }, // Lm       DEVANAGARI SIGN HIGH SPACING DOT
    Interval{ 0x0972, 0x0980 }, // Lo  [15] DEVANAGARI LETTER CANDRA A..BENGALI ANJI
    Interval{ 0x0981, 0x0981 }, // Mn       BENGALI SIGN CANDRABINDU
    Interval{ 0x0982, 0x0983 }, // Mc   [2] BENGALI SIGN ANUSVARA..BENGALI SIGN VISARGA
    Interval{ 0x0985, 0x098C }, // Lo   [8] BENGALI LETTER A..BENGALI LETTER VOCALIC L
    Interval{ 0x098F, 0x0990 }, // Lo   [2] BENGALI LETTER E..BENGALI LETTER AI
    Interval{ 0x0993, 0x09A8 }, // Lo  [22] BENGALI LETTER O..BENGALI LETTER NA
    Interval{ 0x09AA, 0x09B0 }, // Lo   [7] BENGALI LETTER PA..BENGALI LETTER RA
    Interval{ 0x09B2, 0x09B2 }, // Lo       BENGALI LETTER LA
    Interval{ 0x09B6, 0x09B9 }, // Lo   [4] BENGALI LETTER SHA..BENGALI LETTER HA
    Interval{ 0x09BC, 0x09BC }, // Mn       BENGALI SIGN NUKTA
    Interval{ 0x09BD, 0x09BD }, // Lo       BENGALI SIGN AVAGRAHA
    Interval{ 0x09BE, 0x09C0 }, // Mc   [3] BENGALI VOWEL SIGN AA..BENGALI VOWEL SIGN II
    Interval{ 0x09C1, 0x09C4 }, // Mn   [4] BENGALI VOWEL SIGN U..BENGALI VOWEL SIGN VOCALIC RR
    Interval{ 0x09C7, 0x09C8 }, // Mc   [2] BENGALI VOWEL SIGN E..BENGALI VOWEL SIGN AI
    Interval{ 0x09CB, 0x09CC }, // Mc   [2] BENGALI VOWEL SIGN O..BENGALI VOWEL SIGN AU
    Interval{ 0x09CD, 0x09CD }, // Mn       BENGALI SIGN VIRAMA
    Interval{ 0x09CE, 0x09CE }, // Lo       BENGALI LETTER KHANDA TA
    Interval{ 0x09D7, 0x09D7 }, // Mc       BENGALI AU LENGTH MARK
    Interval{ 0x09DC, 0x09DD }, // Lo   [2] BENGALI LETTER RRA..BENGALI LETTER RHA
    Interval{ 0x09DF, 0x09E1 }, // Lo   [3] BENGALI LETTER YYA..BENGALI LETTER VOCALIC LL
    Interval{ 0x09E2, 0x09E3 }, // Mn   [2] BENGALI VOWEL SIGN VOCALIC L..BENGALI VOWEL SIGN VOCALIC LL
    Interval{ 0x09E6, 0x09EF }, // Nd  [10] BENGALI DIGIT ZERO..BENGALI DIGIT NINE
    Interval{ 0x09F0, 0x09F1 }, // Lo   [2] BENGALI LETTER RA WITH MIDDLE DIAGONAL..BENGALI LETTER RA WITH LOWER DIAGONAL
    Interval{ 0x09FC, 0x09FC }, // Lo       BENGALI LETTER VEDIC ANUSVARA
    Interval{ 0x09FE, 0x09FE }, // Mn       BENGALI SANDHI MARK
    Interval{ 0x0A01, 0x0A02 }, // Mn   [2] GURMUKHI SIGN ADAK BINDI..GURMUKHI SIGN BINDI
    Interval{ 0x0A03, 0x0A03 }, // Mc       GURMUKHI SIGN VISARGA
    Interval{ 0x0A05, 0x0A0A }, // Lo   [6] GURMUKHI LETTER A..GURMUKHI LETTER UU
    Interval{ 0x0A0F, 0x0A10 }, // Lo   [2] GURMUKHI LETTER EE..GURMUKHI LETTER AI
    Interval{ 0x0A13, 0x0A28 }, // Lo  [22] GURMUKHI LETTER OO..GURMUKHI LETTER NA
    Interval{ 0x0A2A, 0x0A30 }, // Lo   [7] GURMUKHI LETTER PA..GURMUKHI LETTER RA
    Interval{ 0x0A32, 0x0A33 }, // Lo   [2] GURMUKHI LETTER LA..GURMUKHI LETTER LLA
    Interval{ 0x0A35, 0x0A36 }, // Lo   [2] GURMUKHI LETTER VA..GURMUKHI LETTER SHA
    Interval{ 0x0A38, 0x0A39 }, // Lo   [2] GURMUKHI LETTER SA..GURMUKHI LETTER HA
    Interval{ 0x0A3C, 0x0A3C }, // Mn       GURMUKHI SIGN NUKTA
    Interval{ 0x0A3E, 0x0A40 }, // Mc   [3] GURMUKHI VOWEL SIGN AA..GURMUKHI VOWEL SIGN II
    Interval{ 0x0A41, 0x0A42 }, // Mn   [2] GURMUKHI VOWEL SIGN U..GURMUKHI VOWEL SIGN UU
    Interval{ 0x0A47, 0x0A48 }, // Mn   [2] GURMUKHI VOWEL SIGN EE..GURMUKHI VOWEL SIGN AI
    Interval{ 0x0A4B, 0x0A4D }, // Mn   [3] GURMUKHI VOWEL SIGN OO..GURMUKHI SIGN VIRAMA
    Interval{ 0x0A51, 0x0A51 }, // Mn       GURMUKHI SIGN UDAAT
    Interval{ 0x0A59, 0x0A5C }, // Lo   [4] GURMUKHI LETTER KHHA..GURMUKHI LETTER RRA
    Interval{ 0x0A5E, 0x0A5E }, // Lo       GURMUKHI LETTER FA
    Interval{ 0x0A66, 0x0A6F }, // Nd  [10] GURMUKHI DIGIT ZERO..GURMUKHI DIGIT NINE
    Interval{ 0x0A70, 0x0A71 }, // Mn   [2] GURMUKHI TIPPI..GURMUKHI ADDAK
    Interval{ 0x0A72, 0x0A74 }, // Lo   [3] GURMUKHI IRI..GURMUKHI EK ONKAR
    Interval{ 0x0A75, 0x0A75 }, // Mn       GURMUKHI SIGN YAKASH
    Interval{ 0x0A81, 0x0A82 }, // Mn   [2] GUJARATI SIGN CANDRABINDU..GUJARATI SIGN ANUSVARA
    Interval{ 0x0A83, 0x0A83 }, // Mc       GUJARATI SIGN VISARGA
    Interval{ 0x0A85, 0x0A8D }, // Lo   [9] GUJARATI LETTER A..GUJARATI VOWEL CANDRA E
    Interval{ 0x0A8F, 0x0A91 }, // Lo   [3] GUJARATI LETTER E..GUJARATI VOWEL CANDRA O
    Interval{ 0x0A93, 0x0AA8 }, // Lo  [22] GUJARATI LETTER O..GUJARATI LETTER NA
    Interval{ 0x0AAA, 0x0AB0 }, // Lo   [7] GUJARATI LETTER PA..GUJARATI LETTER RA
    Interval{ 0x0AB2, 0x0AB3 }, // Lo   [2] GUJARATI LETTER LA..GUJARATI LETTER LLA
    Interval{ 0x0AB5, 0x0AB9 }, // Lo   [5] GUJARATI LETTER VA..GUJARATI LETTER HA
    Interval{ 0x0ABC, 0x0ABC }, // Mn       GUJARATI SIGN NUKTA
    Interval{ 0x0ABD, 0x0ABD }, // Lo       GUJARATI SIGN AVAGRAHA
    Interval{ 0x0ABE, 0x0AC0 }, // Mc   [3] GUJARATI VOWEL SIGN AA..GUJARATI VOWEL SIGN II
    Interval{ 0x0AC1, 0x0AC5 }, // Mn   [5] GUJARATI VOWEL SIGN U..GUJARATI VOWEL SIGN CANDRA E
    Interval{ 0x0AC7, 0x0AC8 }, // Mn   [2] GUJARATI VOWEL SIGN E..GUJARATI VOWEL SIGN AI
    Interval{ 0x0AC9, 0x0AC9 }, // Mc       GUJARATI VOWEL SIGN CANDRA O
    Interval{ 0x0ACB, 0x0ACC }, // Mc   [2] GUJARATI VOWEL SIGN O..GUJARATI VOWEL SIGN AU
    Interval{ 0x0ACD, 0x0ACD }, // Mn       GUJARATI SIGN VIRAMA
    Interval{ 0x0AD0, 0x0AD0 }, // Lo       GUJARATI OM
    Interval{ 0x0AE0, 0x0AE1 }, // Lo   [2] GUJARATI LETTER VOCALIC RR..GUJARATI LETTER VOCALIC LL
    Interval{ 0x0AE2, 0x0AE3 }, // Mn   [2] GUJARATI VOWEL SIGN VOCALIC L..GUJARATI VOWEL SIGN VOCALIC LL
    Interval{ 0x0AE6, 0x0AEF }, // Nd  [10] GUJARATI DIGIT ZERO..GUJARATI DIGIT NINE
    Interval{ 0x0AF9, 0x0AF9 }, // Lo       GUJARATI LETTER ZHA
    Interval{ 0x0AFA, 0x0AFF }, // Mn   [6] GUJARATI SIGN SUKUN..GUJARATI SIGN TWO-CIRCLE NUKTA ABOVE
    Interval{ 0x0B01, 0x0B01 }, // Mn       ORIYA SIGN CANDRABINDU
    Interval{ 0x0B02, 0x0B03 }, // Mc   [2] ORIYA SIGN ANUSVARA..ORIYA SIGN VISARGA
    Interval{ 0x0B05, 0x0B0C }, // Lo   [8] ORIYA LETTER A..ORIYA LETTER VOCALIC L
    Interval{ 0x0B0F, 0x0B10 }, // Lo   [2] ORIYA LETTER E..ORIYA LETTER AI
    Interval{ 0x0B13, 0x0B28 }, // Lo  [22] ORIYA LETTER O..ORIYA LETTER NA
    Interval{ 0x0B2A, 0x0B30 }, // Lo   [7] ORIYA LETTER PA..ORIYA LETTER RA
    Interval{ 0x0B32, 0x0B33 }, // Lo   [2] ORIYA LETTER LA..ORIYA LETTER LLA
    Interval{ 0x0B35, 0x0B39 }, // Lo   [5] ORIYA LETTER VA..ORIYA LETTER HA
    Interval{ 0x0B3C, 0x0B3C }, // Mn       ORIYA SIGN NUKTA
    Interval{ 0x0B3D, 0x0B3D }, // Lo       ORIYA SIGN AVAGRAHA
    Interval{ 0x0B3E, 0x0B3E }, // Mc       ORIYA VOWEL SIGN AA
    Interval{ 0x0B3F, 0x0B3F }, // Mn       ORIYA VOWEL SIGN I
    Interval{ 0x0B40, 0x0B40 }, // Mc       ORIYA VOWEL SIGN II
    Interval{ 0x0B41, 0x0B44 }, // Mn   [4] ORIYA VOWEL SIGN U..ORIYA VOWEL SIGN VOCALIC RR
    Interval{ 0x0B47, 0x0B48 }, // Mc   [2] ORIYA VOWEL SIGN E..ORIYA VOWEL SIGN AI
    Interval{ 0x0B4B, 0x0B4C }, // Mc   [2] ORIYA VOWEL SIGN O..ORIYA VOWEL SIGN AU
    Interval{ 0x0B4D, 0x0B4D }, // Mn       ORIYA SIGN VIRAMA
    Interval{ 0x0B55, 0x0B56 }, // Mn   [2] ORIYA SIGN OVERLINE..ORIYA AI LENGTH MARK
    Interval{ 0x0B57, 0x0B57 }, // Mc       ORIYA AU LENGTH MARK
    Interval{ 0x0B5C, 0x0B5D }, // Lo   [2] ORIYA LETTER RRA..ORIYA LETTER RHA
    Interval{ 0x0B5F, 0x0B61 }, // Lo   [3] ORIYA LETTER YYA..ORIYA LETTER VOCALIC LL
    Interval{ 0x0B62, 0x0B63 }, // Mn   [2] ORIYA VOWEL SIGN VOCALIC L..ORIYA VOWEL SIGN VOCALIC LL
    Interval{ 0x0B66, 0x0B6F }, // Nd  [10] ORIYA DIGIT ZERO..ORIYA DIGIT NINE
    Interval{ 0x0B71, 0x0B71 }, // Lo       ORIYA LETTER WA
    Interval{ 0x0B82, 0x0B82 }, // Mn       TAMIL SIGN ANUSVARA
    Interval{ 0x0B83, 0x0B83 }, // Lo       TAMIL SIGN VISARGA
    Interval{ 0x0B85, 0x0B8A }, // Lo   [6] TAMIL LETTER A..TAMIL LETTER UU
    Interval{ 0x0B8E, 0x0B90 }, // Lo   [3] TAMIL LETTER E..TAMIL LETTER AI
    Interval{ 0x0B92, 0x0B95 }, // Lo   [4] TAMIL LETTER O..TAMIL LETTER KA
    Interval{ 0x0B99, 0x0B9A }, // Lo   [2] TAMIL LETTER NGA..TAMIL LETTER CA
    Interval{ 0x0B9C, 0x0B9C }, // Lo       TAMIL LETTER JA
    Interval{ 0x0B9E, 0x0B9F }, // Lo   [2] TAMIL LETTER NYA..TAMIL LETTER TTA
    Interval{ 0x0BA3, 0x0BA4 }, // Lo   [2] TAMIL LETTER NNA..TAMIL LETTER TA
    Interval{ 0x0BA8, 0x0BAA }, // Lo   [3] TAMIL LETTER NA..TAMIL LETTER PA
    Interval{ 0x0BAE, 0x0BB9 }, // Lo  [12] TAMIL LETTER MA..TAMIL LETTER HA
    Interval{ 0x0BBE, 0x0BBF }, // Mc   [2] TAMIL VOWEL SIGN AA..TAMIL VOWEL SIGN I
    Interval{ 0x0BC0, 0x0BC0 }, // Mn       TAMIL VOWEL SIGN II
    Interval{ 0x0BC1, 0x0BC2 }, // Mc   [2] TAMIL VOWEL SIGN U..TAMIL VOWEL SIGN UU
    Interval{ 0x0BC6, 0x0BC8 }, // Mc   [3] TAMIL VOWEL SIGN E..TAMIL VOWEL SIGN AI
    Interval{ 0x0BCA, 0x0BCC }, // Mc   [3] TAMIL VOWEL SIGN O..TAMIL VOWEL SIGN AU
    Interval{ 0x0BCD, 0x0BCD }, // Mn       TAMIL SIGN VIRAMA
    Interval{ 0x0BD0, 0x0BD0 }, // Lo       TAMIL OM
    Interval{ 0x0BD7, 0x0BD7 }, // Mc       TAMIL AU LENGTH MARK
    Interval{ 0x0BE6, 0x0BEF }, // Nd  [10] TAMIL DIGIT ZERO..TAMIL DIGIT NINE
    Interval{ 0x0C00, 0x0C00 }, // Mn       TELUGU SIGN COMBINING CANDRABINDU ABOVE
    Interval{ 0x0C01, 0x0C03 }, // Mc   [3] TELUGU SIGN CANDRABINDU..TELUGU SIGN VISARGA
    Interval{ 0x0C04, 0x0C04 }, // Mn       TELUGU SIGN COMBINING ANUSVARA ABOVE
    Interval{ 0x0C05, 0x0C0C }, // Lo   [8] TELUGU LETTER A..TELUGU LETTER VOCALIC L
    Interval{ 0x0C0E, 0x0C10 }, // Lo   [3] TELUGU LETTER E..TELUGU LETTER AI
    Interval{ 0x0C12, 0x0C28 }, // Lo  [23] TELUGU LETTER O..TELUGU LETTER NA
    Interval{ 0x0C2A, 0x0C39 }, // Lo  [16] TELUGU LETTER PA..TELUGU LETTER HA
    Interval{ 0x0C3C, 0x0C3C }, // Mn       TELUGU SIGN NUKTA
    Interval{ 0x0C3D, 0x0C3D }, // Lo       TELUGU SIGN AVAGRAHA
    Interval{ 0x0C3E, 0x0C40 }, // Mn   [3] TELUGU VOWEL SIGN AA..TELUGU VOWEL SIGN II
    Interval{ 0x0C41, 0x0C44 }, // Mc   [4] TELUGU VOWEL SIGN U..TELUGU VOWEL SIGN VOCALIC RR
    Interval{ 0x0C46, 0x0C48 }, // Mn   [3] TELUGU VOWEL SIGN E..TELUGU VOWEL SIGN AI
    Interval{ 0x0C4A, 0x0C4D }, // Mn   [4] TELUGU VOWEL SIGN O..TELUGU SIGN VIRAMA
    Interval{ 0x0C55, 0x0C56 }, // Mn   [2] TELUGU LENGTH MARK..TELUGU AI LENGTH MARK
    Interval{ 0x0C58, 0x0C5A }, // Lo   [3] TELUGU LETTER TSA..TELUGU LETTER RRRA
    Interval{ 0x0C5D, 0x0C5D }, // Lo       TELUGU LETTER NAKAARA POLLU
    Interval{ 0x0C60, 0x0C61 }, // Lo   [2] TELUGU LETTER VOCALIC RR..TELUGU LETTER VOCALIC LL
    Interval{ 0x0C62, 0x0C63 }, // Mn   [2] TELUGU VOWEL SIGN VOCALIC L..TELUGU VOWEL SIGN VOCALIC LL
    Interval{ 0x0C66, 0x0C6F }, // Nd  [10] TELUGU DIGIT ZERO..TELUGU DIGIT NINE
    Interval{ 0x0C80, 0x0C80 }, // Lo       KANNADA SIGN SPACING CANDRABINDU
    Interval{ 0x0C81, 0x0C81 }, // Mn       KANNADA SIGN CANDRABINDU
    Interval{ 0x0C82, 0x0C83 }, // Mc   [2] KANNADA SIGN ANUSVARA..KANNADA SIGN VISARGA
    Interval{ 0x0C85, 0x0C8C }, // Lo   [8] KANNADA LETTER A..KANNADA LETTER VOCALIC L
    Interval{ 0x0C8E, 0x0C90 }, // Lo   [3] KANNADA LETTER E..KANNADA LETTER AI
    Interval{ 0x0C92, 0x0CA8 }, // Lo  [23] KANNADA LETTER O..KANNADA LETTER NA
    Interval{ 0x0CAA, 0x0CB3 }, // Lo  [10] KANNADA LETTER PA..KANNADA LETTER LLA
    Interval{ 0x0CB5, 0x0CB9 }, // Lo   [5] KANNADA LETTER VA..KANNADA LETTER HA
    Interval{ 0x0CBC, 0x0CBC }, // Mn       KANNADA SIGN NUKTA
    Interval{ 0x0CBD, 0x0CBD }, // Lo       KANNADA SIGN AVAGRAHA
    Interval{ 0x0CBE, 0x0CBE }, // Mc       KANNADA VOWEL SIGN AA
    Interval{ 0x0CBF, 0x0CBF }, // Mn       KANNADA VOWEL SIGN I
    Interval{ 0x0CC0, 0x0CC4 }, // Mc   [5] KANNADA VOWEL SIGN II..KANNADA VOWEL SIGN VOCALIC RR
    Interval{ 0x0CC6, 0x0CC6 }, // Mn       KANNADA VOWEL SIGN E
    Interval{ 0x0CC7, 0x0CC8 }, // Mc   [2] KANNADA VOWEL SIGN EE..KANNADA VOWEL SIGN AI
    Interval{ 0x0CCA, 0x0CCB }, // Mc   [2] KANNADA VOWEL SIGN O..KANNADA VOWEL SIGN OO
    Interval{ 0x0CCC, 0x0CCD }, // Mn   [2] KANNADA VOWEL SIGN AU..KANNADA SIGN VIRAMA
    Interval{ 0x0CD5, 0x0CD6 }, // Mc   [2] KANNADA LENGTH MARK..KANNADA AI LENGTH MARK
    Interval{ 0x0CDD, 0x0CDE }, // Lo   [2] KANNADA LETTER NAKAARA POLLU..KANNADA LETTER FA
    Interval{ 0x0CE0, 0x0CE1 }, // Lo   [2] KANNADA LETTER VOCALIC RR..KANNADA LETTER VOCALIC LL
    Interval{ 0x0CE2, 0x0CE3 }, // Mn   [2] KANNADA VOWEL SIGN VOCALIC L..KANNADA VOWEL SIGN VOCALIC LL
    Interval{ 0x0CE6, 0x0CEF }, // Nd  [10] KANNADA DIGIT ZERO..KANNADA DIGIT NINE
    Interval{ 0x0CF1, 0x0CF2 }, // Lo   [2] KANNADA SIGN JIHVAMULIYA..KANNADA SIGN UPADHMANIYA
    Interval{ 0x0CF3, 0x0CF3 }, // Mc       KANNADA SIGN COMBINING ANUSVARA ABOVE RIGHT
    Interval{ 0x0D00, 0x0D01 }, // Mn   [2] MALAYALAM SIGN COMBINING ANUSVARA ABOVE..MALAYALAM SIGN CANDRABINDU
    Interval{ 0x0D02, 0x0D03 }, // Mc   [2] MALAYALAM SIGN ANUSVARA..MALAYALAM SIGN VISARGA
    Interval{ 0x0D04, 0x0D0C }, // Lo   [9] MALAYALAM LETTER VEDIC ANUSVARA..MALAYALAM LETTER VOCALIC L
    Interval{ 0x0D0E, 0x0D10 }, // Lo   [3] MALAYALAM LETTER E..MALAYALAM LETTER AI
    Interval{ 0x0D12, 0x0D3A }, // Lo  [41] MALAYALAM LETTER O..MALAYALAM LETTER TTTA
    Interval{ 0x0D3B, 0x0D3C }, // Mn   [2] MALAYALAM SIGN VERTICAL BAR VIRAMA..MALAYALAM SIGN CIRCULAR VIRAMA
    Interval{ 0x0D3D, 0x0D3D }, // Lo       MALAYALAM SIGN AVAGRAHA
    Interval{ 0x0D3E, 0x0D40 }, // Mc   [3] MALAYALAM VOWEL SIGN AA..MALAYALAM VOWEL SIGN II
    Interval{ 0x0D41, 0x0D44 }, // Mn   [4] MALAYALAM VOWEL SIGN U..MALAYALAM VOWEL SIGN VOCALIC RR
    Interval{ 0x0D46, 0x0D48 }, // Mc   [3] MALAYALAM VOWEL SIGN E..MALAYALAM VOWEL SIGN AI
    Interval{ 0x0D4A, 0x0D4C }, // Mc   [3] MALAYALAM VOWEL SIGN O..MALAYALAM VOWEL SIGN AU
    Interval{ 0x0D4D, 0x0D4D }, // Mn       MALAYALAM SIGN VIRAMA
    Interval{ 0x0D4E, 0x0D4E }, // Lo       MALAYALAM LETTER DOT REPH
    Interval{ 0x0D54, 0x0D56 }, // Lo   [3] MALAYALAM LETTER CHILLU M..MALAYALAM LETTER CHILLU LLL
    Interval{ 0x0D57, 0x0D57 }, // Mc       MALAYALAM AU LENGTH MARK
    Interval{ 0x0D5F, 0x0D61 }, // Lo   [3] MALAYALAM LETTER ARCHAIC II..MALAYALAM LETTER VOCALIC LL
    Interval{ 0x0D62, 0x0D63 }, // Mn   [2] MALAYALAM VOWEL SIGN VOCALIC L..MALAYALAM VOWEL SIGN VOCALIC LL
    Interval{ 0x0D66, 0x0D6F }, // Nd  [10] MALAYALAM DIGIT ZERO..MALAYALAM DIGIT NINE
    Interval{ 0x0D7A, 0x0D7F }, // Lo   [6] MALAYALAM LETTER CHILLU NN..MALAYALAM LETTER CHILLU K
    Interval{ 0x0D81, 0x0D81 }, // Mn       SINHALA SIGN CANDRABINDU
    Interval{ 0x0D82, 0x0D83 }, // Mc   [2] SINHALA SIGN ANUSVARAYA..SINHALA SIGN VISARGAYA
    Interval{ 0x0D85, 0x0D96 }, // Lo  [18] SINHALA LETTER AYANNA..SINHALA LETTER AUYANNA
    Interval{ 0x0D9A, 0x0DB1 }, // Lo  [24] SINHALA LETTER ALPAPRAANA KAYANNA..SINHALA LETTER DANTAJA NAYANNA
    Interval{ 0x0DB3, 0x0DBB }, // Lo   [9] SINHALA LETTER SANYAKA DAYANNA..SINHALA LETTER RAYANNA
    Interval{ 0x0DBD, 0x0DBD }, // Lo       SINHALA LETTER DANTAJA LAYANNA
    Interval{ 0x0DC0, 0x0DC6 }, // Lo   [7] SINHALA LETTER VAYANNA..SINHALA LETTER FAYANNA
    Interval{ 0x0DCA, 0x0DCA }, // Mn       SINHALA SIGN AL-LAKUNA
    Interval{ 0x0DCF, 0x0DD1 }, // Mc   [3] SINHALA VOWEL SIGN AELA-PILLA..SINHALA VOWEL SIGN DIGA AEDA-PILLA
    Interval{ 0x0DD2, 0x0DD4 }, // Mn   [3] SINHALA VOWEL SIGN KETTI IS-PILLA..SINHALA VOWEL SIGN KETTI PAA-PILLA
    Interval{ 0x0DD6, 0x0DD6 }, // Mn       SINHALA VOWEL SIGN DIGA PAA-PILLA
    Interval{ 0x0DD8, 0x0DDF }, // Mc   [8] SINHALA VOWEL SIGN GAETTA-PILLA..SINHALA VOWEL SIGN GAYANUKITTA
    Interval{ 0x0DE6, 0x0DEF }, // Nd  [10] SINHALA LITH DIGIT ZERO..SINHALA LITH DIGIT NINE
    Interval{ 0x0DF2, 0x0DF3 }, // Mc   [2] SINHALA VOWEL SIGN DIGA GAETTA-PILLA..SINHALA VOWEL SIGN DIGA GAYANUKITTA
    Interval{ 0x0E01, 0x0E30 }, // Lo  [48] THAI CHARACTER KO KAI..THAI CHARACTER SARA A
    Interval{ 0x0E31, 0x0E31 }, // Mn       THAI CHARACTER MAI HAN-AKAT
    Interval{ 0x0E32, 0x0E33 }, // Lo   [2] THAI CHARACTER SARA AA..THAI CHARACTER SARA AM
    Interval{ 0x0E34, 0x0E3A }, // Mn   [7] THAI CHARACTER SARA I..THAI CHARACTER PHINTHU
    Interval{ 0x0E40, 0x0E45 }, // Lo   [6] THAI CHARACTER SARA E..THAI CHARACTER LAKKHANGYAO
    Interval{ 0x0E46, 0x0E46 }, // Lm       THAI CHARACTER MAIYAMOK
    Interval{ 0x0E47, 0x0E4E }, // Mn   [8] THAI CHARACTER MAITAIKHU..THAI CHARACTER YAMAKKAN
    Interval{ 0x0E50, 0x0E59 }, // Nd  [10] THAI DIGIT ZERO..THAI DIGIT NINE
    Interval{ 0x0E81, 0x0E82 }, // Lo   [2] LAO LETTER KO..LAO LETTER KHO SUNG
    Interval{ 0x0E84, 0x0E84 }, // Lo       LAO LETTER KHO TAM
    Interval{ 0x0E86, 0x0E8A }, // Lo   [5] LAO LETTER PALI GHA..LAO LETTER SO TAM
    Interval{ 0x0E8C, 0x0EA3 }, // Lo  [24] LAO LETTER PALI JHA..LAO LETTER LO LING
    Interval{ 0x0EA5, 0x0EA5 }, // Lo       LAO LETTER LO LOOT
    Interval{ 0x0EA7, 0x0EB0 }, // Lo  [10] LAO LETTER WO..LAO VOWEL SIGN A
    Interval{ 0x0EB1, 0x0EB1 }, // Mn       LAO VOWEL SIGN MAI KAN
    Interval{ 0x0EB2, 0x0EB3 }, // Lo   [2] LAO VOWEL SIGN AA..LAO VOWEL SIGN AM
    Interval{ 0x0EB4, 0x0EBC }, // Mn   [9] LAO VOWEL SIGN I..LAO SEMIVOWEL SIGN LO
    Interval{ 0x0EBD, 0x0EBD }, // Lo       LAO SEMIVOWEL SIGN NYO
    Interval{ 0x0EC0, 0x0EC4 }, // Lo   [5] LAO VOWEL SIGN E..LAO VOWEL SIGN AI
    Interval{ 0x0EC6, 0x0EC6 }, // Lm       LAO KO LA
    Interval{ 0x0EC8, 0x0ECE }, // Mn   [7] LAO TONE MAI EK..LAO YAMAKKAN
    Interval{ 0x0ED0, 0x0ED9 }, // Nd  [10] LAO DIGIT ZERO..LAO DIGIT NINE
    Interval{ 0x0EDC, 0x0EDF }, // Lo   [4] LAO HO NO..LAO LETTER KHMU NYO
    Interval{ 0x0F00, 0x0F00 }, // Lo       TIBETAN SYLLABLE OM
    Interval{ 0x0F18, 0x0F19 }, // Mn   [2] TIBETAN ASTROLOGICAL SIGN -KHYUD PA..TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS
    Interval{ 0x0F20, 0x0F29 }, // Nd  [10] TIBETAN DIGIT ZERO..TIBETAN DIGIT NINE
    Interval{ 0x0F35, 0x0F35 }, // Mn       TIBETAN MARK NGAS BZUNG NYI ZLA
    Interval{ 0x0F37, 0x0F37 }, // Mn       TIBETAN MARK NGAS BZUNG SGOR RTAGS
    Interval{ 0x0F39, 0x0F39 }, // Mn       TIBETAN MARK TSA -PHRU
    Interval{ 0x0F3E, 0x0F3F }, // Mc   [2] TIBETAN SIGN YAR TSHES..TIBETAN SIGN MAR TSHES
    Interval{ 0x0F40, 0x0F47 }, // Lo   [8] TIBETAN LETTER KA..TIBETAN LETTER JA
    Interval{ 0x0F49, 0x0F6C }, // Lo  [36] TIBETAN LETTER NYA..TIBETAN LETTER RRA
    Interval{ 0x0F71, 0x0F7E }, // Mn  [14] TIBETAN VOWEL SIGN AA..TIBETAN SIGN RJES SU NGA RO
    Interval{ 0x0F7F, 0x0F7F }, // Mc       TIBETAN SIGN RNAM BCAD
    Interval{ 0x0F80, 0x0F84 }, // Mn   [5] TIBETAN VOWEL SIGN REVERSED I..TIBETAN MARK HALANTA
    Interval{ 0x0F86, 0x0F87 }, // Mn   [2] TIBETAN SIGN LCI RTAGS..TIBETAN SIGN YANG RTAGS
    Interval{ 0x0F88, 0x0F8C }, // Lo   [5] TIBETAN SIGN LCE TSA CAN..TIBETAN SIGN INVERTED MCHU CAN
    Interval{ 0x0F8D, 0x0F97 }, // Mn  [11] TIBETAN SUBJOINED SIGN LCE TSA CAN..TIBETAN SUBJOINED LETTER JA
    Interval{ 0x0F99, 0x0FBC }, // Mn  [36] TIBETAN SUBJOINED LETTER NYA..TIBETAN SUBJOINED LETTER FIXED-FORM RA
    Interval{ 0x0FC6, 0x0FC6 }, // Mn       TIBETAN SYMBOL PADMA GDAN
    Interval{ 0x1000, 0x102A }, // Lo  [43] MYANMAR LETTER KA..MYANMAR LETTER AU
    Interval{ 0x102B, 0x102C }, // Mc   [2] MYANMAR VOWEL SIGN TALL AA..MYANMAR VOWEL SIGN AA
    Interval{ 0x102D, 0x1030 }, // Mn   [4] MYANMAR VOWEL SIGN I..MYANMAR VOWEL SIGN UU
    Interval{ 0x1031, 0x1031 }, // Mc       MYANMAR VOWEL SIGN E
    Interval{ 0x1032, 0x1037 }, // Mn   [6] MYANMAR VOWEL SIGN AI..MYANMAR SIGN DOT BELOW
    Interval{ 0x1038, 0x1038 }, // Mc       MYANMAR SIGN VISARGA
    Interval{ 0x1039, 0x103A }, // Mn   [2] MYANMAR SIGN VIRAMA..MYANMAR SIGN ASAT
    Interval{ 0x103B, 0x103C }, // Mc   [2] MYANMAR CONSONANT SIGN MEDIAL YA..MYANMAR CONSONANT SIGN MEDIAL RA
    Interval{ 0x103D, 0x103E }, // Mn   [2] MYANMAR CONSONANT SIGN MEDIAL WA..MYANMAR CONSONANT SIGN MEDIAL HA
    Interval{ 0x103F, 0x103F }, // Lo       MYANMAR LETTER GREAT SA
    Interval{ 0x1040, 0x1049 }, // Nd  [10] MYANMAR DIGIT ZERO..MYANMAR DIGIT NINE
    Interval{ 0x1050, 0x1055 }, // Lo   [6] MYANMAR LETTER SHA..MYANMAR LETTER VOCALIC LL
    Interval{ 0x1056, 0x1057 }, // Mc   [2] MYANMAR VOWEL SIGN VOCALIC R..MYANMAR VOWEL SIGN VOCALIC RR
    Interval{ 0x1058, 0x1059 }, // Mn   [2] MYANMAR VOWEL SIGN VOCALIC L..MYANMAR VOWEL SIGN VOCALIC LL
    Interval{ 0x105A, 0x105D }, // Lo   [4] MYANMAR LETTER MON NGA..MYANMAR LETTER MON BBE
    Interval{ 0x105E, 0x1060 }, // Mn   [3] MYANMAR CONSONANT SIGN MON MEDIAL NA..MYANMAR CONSONANT SIGN MON MEDIAL LA
    Interval{ 0x1061, 0x1061 }, // Lo       MYANMAR LETTER SGAW KAREN SHA
    Interval{ 0x1062, 0x1064 }, // Mc   [3] MYANMAR VOWEL SIGN SGAW KAREN EU..MYANMAR TONE MARK SGAW KAREN KE PHO
    Interval{ 0x1065, 0x1066 }, // Lo   [2] MYANMAR LETTER WESTERN PWO KAREN THA..MYANMAR LETTER WESTERN PWO KAREN PWA
    Interval{ 0x1067, 0x106D }, // Mc   [7] MYANMAR VOWEL SIGN WESTERN PWO KAREN EU..MYANMAR SIGN WESTERN PWO KAREN TONE-5
    Interval{ 0x106E, 0x1070 }, // Lo   [3] MYANMAR LETTER EASTERN PWO KAREN NNA..MYANMAR LETTER EASTERN PWO KAREN GHWA
    Interval{ 0x1071, 0x1074 }, // Mn   [4] MYANMAR VOWEL SIGN GEBA KAREN I..MYANMAR VOWEL SIGN KAYAH EE
    Interval{ 0x1075, 0x1081 }, // Lo  [13] MYANMAR LETTER SHAN KA..MYANMAR LETTER SHAN HA
    Interval{ 0x1082, 0x1082 }, // Mn       MYANMAR CONSONANT SIGN SHAN MEDIAL WA
    Interval{ 0x1083, 0x1084 }, // Mc   [2] MYANMAR VOWEL SIGN SHAN AA..MYANMAR VOWEL SIGN SHAN E
    Interval{ 0x1085, 0x1086 }, // Mn   [2] MYANMAR VOWEL SIGN SHAN E ABOVE..MYANMAR VOWEL SIGN SHAN FINAL Y
    Interval{ 0x1087, 0x108C }, // Mc   [6] MYANMAR SIGN SHAN TONE-2..MYANMAR SIGN SHAN COUNCIL TONE-3
    Interval{ 0x108D, 0x108D }, // Mn       MYANMAR SIGN SHAN COUNCIL EMPHATIC TONE
    Interval{ 0x108E, 0x108E }, // Lo       MYANMAR LETTER RUMAI PALAUNG FA
    Interval{ 0x108F, 0x108F }, // Mc       MYANMAR SIGN RUMAI PALAUNG TONE-5
    Interval{ 0x1090, 0x1099 }, // Nd  [10] MYANMAR SHAN DIGIT ZERO..MYANMAR SHAN DIGIT NINE
    Interval{ 0x109A, 0x109C }, // Mc   [3] MYANMAR SIGN KHAMTI TONE-1..MYANMAR VOWEL SIGN AITON A
    Interval{ 0x109D, 0x109D }, // Mn       MYANMAR VOWEL SIGN AITON AI
    Interval{ 0x10A0, 0x10C5 }, // L&  [38] GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    Interval{ 0x10C7, 0x10C7 }, // L&       GEORGIAN CAPITAL LETTER YN
    Interval{ 0x10CD, 0x10CD }, // L&       GEORGIAN CAPITAL LETTER AEN
    Interval{ 0x10D0, 0x10FA }, // L&  [43] GEORGIAN LETTER AN..GEORGIAN LETTER AIN
    Interval{ 0x10FC, 0x10FC }, // Lm       MODIFIER LETTER GEORGIAN NAR
    Interval{ 0x10FD, 0x10FF }, // L&   [3] GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
    Interval{ 0x1100, 0x1248 }, // Lo [329] HANGUL CHOSEONG KIYEOK..ETHIOPIC SYLLABLE QWA
    Interval{ 0x124A, 0x124D }, // Lo   [4] ETHIOPIC SYLLABLE QWI..ETHIOPIC SYLLABLE QWE
    Interval{ 0x1250, 0x1256 }, // Lo   [7] ETHIOPIC SYLLABLE QHA..ETHIOPIC SYLLABLE QHO
    Interval{ 0x1258, 0x1258 }, // Lo       ETHIOPIC SYLLABLE QHWA
    Interval{ 0x125A, 0x125D }, // Lo   [4] ETHIOPIC SYLLABLE QHWI..ETHIOPIC SYLLABLE QHWE
    Interval{ 0x1260, 0x1288 }, // Lo  [41] ETHIOPIC SYLLABLE BA..ETHIOPIC SYLLABLE XWA
    Interval{ 0x128A, 0x128D }, // Lo   [4] ETHIOPIC SYLLABLE XWI..ETHIOPIC SYLLABLE XWE
    Interval{ 0x1290, 0x12B0 }, // Lo  [33] ETHIOPIC SYLLABLE NA..ETHIOPIC SYLLABLE KWA
    Interval{ 0x12B2, 0x12B5 }, // Lo   [4] ETHIOPIC SYLLABLE KWI..ETHIOPIC SYLLABLE KWE
    Interval{ 0x12B8, 0x12BE }, // Lo   [7] ETHIOPIC SYLLABLE KXA..ETHIOPIC SYLLABLE KXO
    Interval{ 0x12C0, 0x12C0 }, // Lo       ETHIOPIC SYLLABLE KXWA
    Interval{ 0x12C2, 0x12C5 }, // Lo   [4] ETHIOPIC SYLLABLE KXWI..ETHIOPIC SYLLABLE KXWE
    Interval{ 0x12C8, 0x12D6 }, // Lo  [15] ETHIOPIC SYLLABLE WA..ETHIOPIC SYLLABLE PHARYNGEAL O
    Interval{ 0x12D8, 0x1310 }, // Lo  [57] ETHIOPIC SYLLABLE ZA..ETHIOPIC SYLLABLE GWA
    Interval{ 0x1312, 0x1315 }, // Lo   [4] ETHIOPIC SYLLABLE GWI..ETHIOPIC SYLLABLE GWE
    Interval{ 0x1318, 0x135A }, // Lo  [67] ETHIOPIC SYLLABLE GGA..ETHIOPIC SYLLABLE FYA
    Interval{ 0x135D, 0x135F }, // Mn   [3] ETHIOPIC COMBINING GEMINATION AND VOWEL LENGTH MARK..ETHIOPIC COMBINING GEMINATION MARK
    Interval{ 0x1369, 0x1371 }, // No   [9] ETHIOPIC DIGIT ONE..ETHIOPIC DIGIT NINE
    Interval{ 0x1380, 0x138F }, // Lo  [16] ETHIOPIC SYLLABLE SEBATBEIT MWA..ETHIOPIC SYLLABLE PWE
    Interval{ 0x13A0, 0x13F5 }, // L&  [86] CHEROKEE LETTER A..CHEROKEE LETTER MV
    Interval{ 0x13F8, 0x13FD }, // L&   [6] CHEROKEE SMALL LETTER YE..CHEROKEE SMALL LETTER MV
    Interval{ 0x1401, 0x166C }, // Lo [620] CANADIAN SYLLABICS E..CANADIAN SYLLABICS CARRIER TTSA
    Interval{ 0x166F, 0x167F }, // Lo  [17] CANADIAN SYLLABICS QAI..CANADIAN SYLLABICS BLACKFOOT W
    Interval{ 0x1681, 0x169A }, // Lo  [26] OGHAM LETTER BEITH..OGHAM LETTER PEITH
    Interval{ 0x16A0, 0x16EA }, // Lo  [75] RUNIC LETTER FEHU FEOH FE F..RUNIC LETTER X
    Interval{ 0x16EE, 0x16F0 }, // Nl   [3] RUNIC ARLAUG SYMBOL..RUNIC BELGTHOR SYMBOL
    Interval{ 0x16F1, 0x16F8 }, // Lo   [8] RUNIC LETTER K..RUNIC LETTER FRANKS CASKET AESC
    Interval{ 0x1700, 0x1711 }, // Lo  [18] TAGALOG LETTER A..TAGALOG LETTER HA
    Interval{ 0x1712, 0x1714 }, // Mn   [3] TAGALOG VOWEL SIGN I..TAGALOG SIGN VIRAMA
    Interval{ 0x1715, 0x1715 }, // Mc       TAGALOG SIGN PAMUDPOD
    Interval{ 0x171F, 0x1731 }, // Lo  [19] TAGALOG LETTER ARCHAIC RA..HANUNOO LETTER HA
    Interval{ 0x1732, 0x1733 }, // Mn   [2] HANUNOO VOWEL SIGN I..HANUNOO VOWEL SIGN U
    Interval{ 0x1734, 0x1734 }, // Mc       HANUNOO SIGN PAMUDPOD
    Interval{ 0x1740, 0x1751 }, // Lo  [18] BUHID LETTER A..BUHID LETTER HA
    Interval{ 0x1752, 0x1753 }, // Mn   [2] BUHID VOWEL SIGN I..BUHID VOWEL SIGN U
    Interval{ 0x1760, 0x176C }, // Lo  [13] TAGBANWA LETTER A..TAGBANWA LETTER YA
    Interval{ 0x176E, 0x1770 }, // Lo   [3] TAGBANWA LETTER LA..TAGBANWA LETTER SA
    Interval{ 0x1772, 0x1773 }, // Mn   [2] TAGBANWA VOWEL SIGN I..TAGBANWA VOWEL SIGN U
    Interval{ 0x1780, 0x17B3 }, // Lo  [52] KHMER LETTER KA..KHMER INDEPENDENT VOWEL QAU
    Interval{ 0x17B4, 0x17B5 }, // Mn   [2] KHMER VOWEL INHERENT AQ..KHMER VOWEL INHERENT AA
    Interval{ 0x17B6, 0x17B6 }, // Mc       KHMER VOWEL SIGN AA
    Interval{ 0x17B7, 0x17BD }, // Mn   [7] KHMER VOWEL SIGN I..KHMER VOWEL SIGN UA
    Interval{ 0x17BE, 0x17C5 }, // Mc   [8] KHMER VOWEL SIGN OE..KHMER VOWEL SIGN AU
    Interval{ 0x17C6, 0x17C6 }, // Mn       KHMER SIGN NIKAHIT
    Interval{ 0x17C7, 0x17C8 }, // Mc   [2] KHMER SIGN REAHMUK..KHMER SIGN YUUKALEAPINTU
    Interval{ 0x17C9, 0x17D3 }, // Mn  [11] KHMER SIGN MUUSIKATOAN..KHMER SIGN BATHAMASAT
    Interval{ 0x17D7, 0x17D7 }, // Lm       KHMER SIGN LEK TOO
    Interval{ 0x17DC, 0x17DC }, // Lo       KHMER SIGN AVAKRAHASANYA
    Interval{ 0x17DD, 0x17DD }, // Mn       KHMER SIGN ATTHACAN
    Interval{ 0x17E0, 0x17E9 }, // Nd  [10] KHMER DIGIT ZERO..KHMER DIGIT NINE
    Interval{ 0x180B, 0x180D }, // Mn   [3] MONGOLIAN FREE VARIATION SELECTOR ONE..MONGOLIAN FREE VARIATION SELECTOR THREE
    Interval{ 0x180F, 0x180F }, // Mn       MONGOLIAN FREE VARIATION SELECTOR FOUR
    Interval{ 0x1810, 0x1819 }, // Nd  [10] MONGOLIAN DIGIT ZERO..MONGOLIAN DIGIT NINE
    Interval{ 0x1820, 0x1842 }, // Lo  [35] MONGOLIAN LETTER A..MONGOLIAN LETTER CHI
    Interval{ 0x1843, 0x1843 }, // Lm       MONGOLIAN LETTER TODO LONG VOWEL SIGN
    Interval{ 0x1844, 0x1878 }, // Lo  [53] MONGOLIAN LETTER TODO E..MONGOLIAN LETTER CHA WITH TWO DOTS
    Interval{ 0x1880, 0x1884 }, // Lo   [5] MONGOLIAN LETTER ALI GALI ANUSVARA ONE..MONGOLIAN LETTER ALI GALI INVERTED UBADAMA
    Interval{ 0x1885, 0x1886 }, // Mn   [2] MONGOLIAN LETTER ALI GALI BALUDA..MONGOLIAN LETTER ALI GALI THREE BALUDA
    Interval{ 0x1887, 0x18A8 }, // Lo  [34] MONGOLIAN LETTER ALI GALI A..MONGOLIAN LETTER MANCHU ALI GALI BHA
    Interval{ 0x18A9, 0x18A9 }, // Mn       MONGOLIAN LETTER ALI GALI DAGALGA
    Interval{ 0x18AA, 0x18AA }, // Lo       MONGOLIAN LETTER MANCHU ALI GALI LHA
    Interval{ 0x18B0, 0x18F5 }, // Lo  [70] CANADIAN SYLLABICS OY..CANADIAN SYLLABICS CARRIER DENTAL S
    Interval{ 0x1900, 0x191E }, // Lo  [31] LIMBU VOWEL-CARRIER LETTER..LIMBU LETTER TRA
    Interval{ 0x1920, 0x1922 }, // Mn   [3] LIMBU VOWEL SIGN A..LIMBU VOWEL SIGN U
    Interval{ 0x1923, 0x1926 }, // Mc   [4] LIMBU VOWEL SIGN EE..LIMBU VOWEL SIGN AU
    Interval{ 0x1927, 0x1928 }, // Mn   [2] LIMBU VOWEL SIGN E..LIMBU VOWEL SIGN O
    Interval{ 0x1929, 0x192B }, // Mc   [3] LIMBU SUBJOINED LETTER YA..LIMBU SUBJOINED LETTER WA
    Interval{ 0x1930, 0x1931 }, // Mc   [2] LIMBU SMALL LETTER KA..LIMBU SMALL LETTER NGA
    Interval{ 0x1932, 0x1932 }, // Mn       LIMBU SMALL LETTER ANUSVARA
    Interval{ 0x1933, 0x1938 }, // Mc   [6] LIMBU SMALL LETTER TA..LIMBU SMALL LETTER LA
    Interval{ 0x1939, 0x193B }, // Mn   [3] LIMBU SIGN MUKPHRENG..LIMBU SIGN SA-I
    Interval{ 0x1946, 0x194F }, // Nd  [10] LIMBU DIGIT ZERO..LIMBU DIGIT NINE
    Interval{ 0x1950, 0x196D }, // Lo  [30] TAI LE LETTER KA..TAI LE LETTER AI
    Interval{ 0x1970, 0x1974 }, // Lo   [5] TAI LE LETTER TONE-2..TAI LE LETTER TONE-6
    Interval{ 0x1980, 0x19AB }, // Lo  [44] NEW TAI LUE LETTER HIGH QA..NEW TAI LUE LETTER LOW SUA
    Interval{ 0x19B0, 0x19C9 }, // Lo  [26] NEW TAI LUE VOWEL SIGN VOWEL SHORTENER..NEW TAI LUE TONE MARK-2
    Interval{ 0x19D0, 0x19D9 }, // Nd  [10] NEW TAI LUE DIGIT ZERO..NEW TAI LUE DIGIT NINE
    Interval{ 0x19DA, 0x19DA }, // No       NEW TAI LUE THAM DIGIT ONE
    Interval{ 0x1A00, 0x1A16 }, // Lo  [23] BUGINESE LETTER KA..BUGINESE LETTER HA
    Interval{ 0x1A17, 0x1A18 }, // Mn   [2] BUGINESE VOWEL SIGN I..BUGINESE VOWEL SIGN U
    Interval{ 0x1A19, 0x1A1A }, // Mc   [2] BUGINESE VOWEL SIGN E..BUGINESE VOWEL SIGN O
    Interval{ 0x1A1B, 0x1A1B }, // Mn       BUGINESE VOWEL SIGN AE
    Interval{ 0x1A20, 0x1A54 }, // Lo  [53] TAI THAM LETTER HIGH KA..TAI THAM LETTER GREAT SA
    Interval{ 0x1A55, 0x1A55 }, // Mc       TAI THAM CONSONANT SIGN MEDIAL RA
    Interval{ 0x1A56, 0x1A56 }, // Mn       TAI THAM CONSONANT SIGN MEDIAL LA
    Interval{ 0x1A57, 0x1A57 }, // Mc       TAI THAM CONSONANT SIGN LA TANG LAI
    Interval{ 0x1A58, 0x1A5E }, // Mn   [7] TAI THAM SIGN MAI KANG LAI..TAI THAM CONSONANT SIGN SA
    Interval{ 0x1A60, 0x1A60 }, // Mn       TAI THAM SIGN SAKOT
    Interval{ 0x1A61, 0x1A61 }, // Mc       TAI THAM VOWEL SIGN A
    Interval{ 0x1A62, 0x1A62 }, // Mn       TAI THAM VOWEL SIGN MAI SAT
    Interval{ 0x1A63, 0x1A64 }, // Mc   [2] TAI THAM VOWEL SIGN AA..TAI THAM VOWEL SIGN TALL AA
    Interval{ 0x1A65, 0x1A6C }, // Mn   [8] TAI THAM VOWEL SIGN I..TAI THAM VOWEL SIGN OA BELOW
    Interval{ 0x1A6D, 0x1A72 }, // Mc   [6] TAI THAM VOWEL SIGN OY..TAI THAM VOWEL SIGN THAM AI
    Interval{ 0x1A73, 0x1A7C }, // Mn  [10] TAI THAM VOWEL SIGN OA ABOVE..TAI THAM SIGN KHUEN-LUE KARAN
    Interval{ 0x1A7F, 0x1A7F }, // Mn       TAI THAM COMBINING CRYPTOGRAMMIC DOT
    Interval{ 0x1A80, 0x1A89 }, // Nd  [10] TAI THAM HORA DIGIT ZERO..TAI THAM HORA DIGIT NINE
    Interval{ 0x1A90, 0x1A99 }, // Nd  [10] TAI THAM THAM DIGIT ZERO..TAI THAM THAM DIGIT NINE
    Interval{ 0x1AA7, 0x1AA7 }, // Lm       TAI THAM SIGN MAI YAMOK
    Interval{ 0x1AB0, 0x1ABD }, // Mn  [14] COMBINING DOUBLED CIRCUMFLEX ACCENT..COMBINING PARENTHESES BELOW
    Interval{ 0x1ABF, 0x1ACE }, // Mn  [16] COMBINING LATIN SMALL LETTER W BELOW..COMBINING LATIN SMALL LETTER INSULAR T
    Interval{ 0x1B00, 0x1B03 }, // Mn   [4] BALINESE SIGN ULU RICEM..BALINESE SIGN SURANG
    Interval{ 0x1B04, 0x1B04 }, // Mc       BALINESE SIGN BISAH
    Interval{ 0x1B05, 0x1B33 }, // Lo  [47] BALINESE LETTER AKARA..BALINESE LETTER HA
    Interval{ 0x1B34, 0x1B34 }, // Mn       BALINESE SIGN REREKAN
    Interval{ 0x1B35, 0x1B35 }, // Mc       BALINESE VOWEL SIGN TEDUNG
    Interval{ 0x1B36, 0x1B3A }, // Mn   [5] BALINESE VOWEL SIGN ULU..BALINESE VOWEL SIGN RA REPA
    Interval{ 0x1B3B, 0x1B3B }, // Mc       BALINESE VOWEL SIGN RA REPA TEDUNG
    Interval{ 0x1B3C, 0x1B3C }, // Mn       BALINESE VOWEL SIGN LA LENGA
    Interval{ 0x1B3D, 0x1B41 }, // Mc   [5] BALINESE VOWEL SIGN LA LENGA TEDUNG..BALINESE VOWEL SIGN TALING REPA TEDUNG
    Interval{ 0x1B42, 0x1B42 }, // Mn       BALINESE VOWEL SIGN PEPET
    Interval{ 0x1B43, 0x1B44 }, // Mc   [2] BALINESE VOWEL SIGN PEPET TEDUNG..BALINESE ADEG ADEG
    Interval{ 0x1B45, 0x1B4C }, // Lo   [8] BALINESE LETTER KAF SASAK..BALINESE LETTER ARCHAIC JNYA
    Interval{ 0x1B50, 0x1B59 }, // Nd  [10] BALINESE DIGIT ZERO..BALINESE DIGIT NINE
    Interval{ 0x1B6B, 0x1B73 }, // Mn   [9] BALINESE MUSICAL SYMBOL COMBINING TEGEH..BALINESE MUSICAL SYMBOL COMBINING GONG
    Interval{ 0x1B80, 0x1B81 }, // Mn   [2] SUNDANESE SIGN PANYECEK..SUNDANESE SIGN PANGLAYAR
    Interval{ 0x1B82, 0x1B82 }, // Mc       SUNDANESE SIGN PANGWISAD
    Interval{ 0x1B83, 0x1BA0 }, // Lo  [30] SUNDANESE LETTER A..SUNDANESE LETTER HA
    Interval{ 0x1BA1, 0x1BA1 }, // Mc       SUNDANESE CONSONANT SIGN PAMINGKAL
    Interval{ 0x1BA2, 0x1BA5 }, // Mn   [4] SUNDANESE CONSONANT SIGN PANYAKRA..SUNDANESE VOWEL SIGN PANYUKU
    Interval{ 0x1BA6, 0x1BA7 }, // Mc   [2] SUNDANESE VOWEL SIGN PANAELAENG..SUNDANESE VOWEL SIGN PANOLONG
    Interval{ 0x1BA8, 0x1BA9 }, // Mn   [2] SUNDANESE VOWEL SIGN PAMEPET..SUNDANESE VOWEL SIGN PANEULEUNG
    Interval{ 0x1BAA, 0x1BAA }, // Mc       SUNDANESE SIGN PAMAAEH
    Interval{ 0x1BAB, 0x1BAD }, // Mn   [3] SUNDANESE SIGN VIRAMA..SUNDANESE CONSONANT SIGN PASANGAN WA
    Interval{ 0x1BAE, 0x1BAF }, // Lo   [2] SUNDANESE LETTER KHA..SUNDANESE LETTER SYA
    Interval{ 0x1BB0, 0x1BB9 }, // Nd  [10] SUNDANESE DIGIT ZERO..SUNDANESE DIGIT NINE
    Interval{ 0x1BBA, 0x1BE5 }, // Lo  [44] SUNDANESE AVAGRAHA..BATAK LETTER U
    Interval{ 0x1BE6, 0x1BE6 }, // Mn       BATAK SIGN TOMPI
    Interval{ 0x1BE7, 0x1BE7 }, // Mc       BATAK VOWEL SIGN E
    Interval{ 0x1BE8, 0x1BE9 }, // Mn   [2] BATAK VOWEL SIGN PAKPAK E..BATAK VOWEL SIGN EE
    Interval{ 0x1BEA, 0x1BEC }, // Mc   [3] BATAK VOWEL SIGN I..BATAK VOWEL SIGN O
    Interval{ 0x1BED, 0x1BED }, // Mn       BATAK VOWEL SIGN KARO O
    Interval{ 0x1BEE, 0x1BEE }, // Mc       BATAK VOWEL SIGN U
    Interval{ 0x1BEF, 0x1BF1 }, // Mn   [3] BATAK VOWEL SIGN U FOR SIMALUNGUN SA..BATAK CONSONANT SIGN H
    Interval{ 0x1BF2, 0x1BF3 }, // Mc   [2] BATAK PANGOLAT..BATAK PANONGONAN
    Interval{ 0x1C00, 0x1C23 }, // Lo  [36] LEPCHA LETTER KA..LEPCHA LETTER A
    Interval{ 0x1C24, 0x1C2B }, // Mc   [8] LEPCHA SUBJOINED LETTER YA..LEPCHA VOWEL SIGN UU
    Interval{ 0x1C2C, 0x1C33 }, // Mn   [8] LEPCHA VOWEL SIGN E..LEPCHA CONSONANT SIGN T
    Interval{ 0x1C34, 0x1C35 }, // Mc   [2] LEPCHA CONSONANT SIGN NYIN-DO..LEPCHA CONSONANT SIGN KANG
    Interval{ 0x1C36, 0x1C37 }, // Mn   [2] LEPCHA SIGN RAN..LEPCHA SIGN NUKTA
    Interval{ 0x1C40, 0x1C49 }, // Nd  [10] LEPCHA DIGIT ZERO..LEPCHA DIGIT NINE
    Interval{ 0x1C4D, 0x1C4F }, // Lo   [3] LEPCHA LETTER TTA..LEPCHA LETTER DDA
    Interval{ 0x1C50, 0x1C59 }, // Nd  [10] OL CHIKI DIGIT ZERO..OL CHIKI DIGIT NINE
    Interval{ 0x1C5A, 0x1C77 }, // Lo  [30] OL CHIKI LETTER LA..OL CHIKI LETTER OH
    Interval{ 0x1C78, 0x1C7D }, // Lm   [6] OL CHIKI MU TTUDDAG..OL CHIKI AHAD
    Interval{ 0x1C80, 0x1C88 }, // L&   [9] CYRILLIC SMALL LETTER ROUNDED VE..CYRILLIC SMALL LETTER UNBLENDED UK
    Interval{ 0x1C90, 0x1CBA }, // L&  [43] GEORGIAN MTAVRULI CAPITAL LETTER AN..GEORGIAN MTAVRULI CAPITAL LETTER AIN
    Interval{ 0x1CBD, 0x1CBF }, // L&   [3] GEORGIAN MTAVRULI CAPITAL LETTER AEN..GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
    Interval{ 0x1CD0, 0x1CD2 }, // Mn   [3] VEDIC TONE KARSHANA..VEDIC TONE PRENKHA
    Interval{ 0x1CD4, 0x1CE0 }, // Mn  [13] VEDIC SIGN YAJURVEDIC MIDLINE SVARITA..VEDIC TONE RIGVEDIC KASHMIRI INDEPENDENT SVARITA
    Interval{ 0x1CE1, 0x1CE1 }, // Mc       VEDIC TONE ATHARVAVEDIC INDEPENDENT SVARITA
    Interval{ 0x1CE2, 0x1CE8 }, // Mn   [7] VEDIC SIGN VISARGA SVARITA..VEDIC SIGN VISARGA ANUDATTA WITH TAIL
    Interval{ 0x1CE9, 0x1CEC }, // Lo   [4] VEDIC SIGN ANUSVARA ANTARGOMUKHA..VEDIC SIGN ANUSVARA VAMAGOMUKHA WITH TAIL
    Interval{ 0x1CED, 0x1CED }, // Mn       VEDIC SIGN TIRYAK
    Interval{ 0x1CEE, 0x1CF3 }, // Lo   [6] VEDIC SIGN HEXIFORM LONG ANUSVARA..VEDIC SIGN ROTATED ARDHAVISARGA
    Interval{ 0x1CF4, 0x1CF4 }, // Mn       VEDIC TONE CANDRA ABOVE
    Interval{ 0x1CF5, 0x1CF6 }, // Lo   [2] VEDIC SIGN JIHVAMULIYA..VEDIC SIGN UPADHMANIYA
    Interval{ 0x1CF7, 0x1CF7 }, // Mc       VEDIC SIGN ATIKRAMA
    Interval{ 0x1CF8, 0x1CF9 }, // Mn   [2] VEDIC TONE RING ABOVE..VEDIC TONE DOUBLE RING ABOVE
    Interval{ 0x1CFA, 0x1CFA }, // Lo       VEDIC SIGN DOUBLE ANUSVARA ANTARGOMUKHA
    Interval{ 0x1D00, 0x1D2B }, // L&  [44] LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTER SMALL CAPITAL EL
    Interval{ 0x1D2C, 0x1D6A }, // Lm  [63] MODIFIER LETTER CAPITAL A..GREEK SUBSCRIPT SMALL LETTER CHI
    Interval{ 0x1D6B, 0x1D77 }, // L&  [13] LATIN SMALL LETTER UE..LATIN SMALL LETTER TURNED G
    Interval{ 0x1D78, 0x1D78 }, // Lm       MODIFIER LETTER CYRILLIC EN
    Interval{ 0x1D79, 0x1D9A }, // L&  [34] LATIN SMALL LETTER INSULAR G..LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
    Interval{ 0x1D9B, 0x1DBF }, // Lm  [37] MODIFIER LETTER SMALL TURNED ALPHA..MODIFIER LETTER SMALL THETA
    Interval{ 0x1DC0, 0x1DFF }, // Mn  [64] COMBINING DOTTED GRAVE ACCENT..COMBINING RIGHT ARROWHEAD AND DOWN ARROWHEAD BELOW
    Interval{ 0x1E00, 0x1F15 }, // L& [278] LATIN CAPITAL LETTER A WITH RING BELOW..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F18, 0x1F1D }, // L&   [6] GREEK CAPITAL LETTER EPSILON WITH PSILI..GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
    Interval{ 0x1F20, 0x1F45 }, // L&  [38] GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F48, 0x1F4D }, // L&   [6] GREEK CAPITAL LETTER OMICRON WITH PSILI..GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
    Interval{ 0x1F50, 0x1F57 }, // L&   [8] GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    Interval{ 0x1F59, 0x1F59 }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA
    Interval{ 0x1F5B, 0x1F5B }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
    Interval{ 0x1F5D, 0x1F5D }, // L&       GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
    Interval{ 0x1F5F, 0x1F7D }, // L&  [31] GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI..GREEK SMALL LETTER OMEGA WITH OXIA
    Interval{ 0x1F80, 0x1FB4 }, // L&  [53] GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI..GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FB6, 0x1FBC }, // L&   [7] GREEK SMALL LETTER ALPHA WITH PERISPOMENI..GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
    Interval{ 0x1FBE, 0x1FBE }, // L&       GREEK PROSGEGRAMMENI
    Interval{ 0x1FC2, 0x1FC4 }, // L&   [3] GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FC6, 0x1FCC }, // L&   [7] GREEK SMALL LETTER ETA WITH PERISPOMENI..GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
    Interval{ 0x1FD0, 0x1FD3 }, // L&   [4] GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
    Interval{ 0x1FD6, 0x1FDB }, // L&   [6] GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK CAPITAL LETTER IOTA WITH OXIA
    Interval{ 0x1FE0, 0x1FEC }, // L&  [13] GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK CAPITAL LETTER RHO WITH DASIA
    Interval{ 0x1FF2, 0x1FF4 }, // L&   [3] GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI..GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
    Interval{ 0x1FF6, 0x1FFC }, // L&   [7] GREEK SMALL LETTER OMEGA WITH PERISPOMENI..GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
    Interval{ 0x203F, 0x2040 }, // Pc   [2] UNDERTIE..CHARACTER TIE
    Interval{ 0x2054, 0x2054 }, // Pc       INVERTED UNDERTIE
    Interval{ 0x2071, 0x2071 }, // Lm       SUPERSCRIPT LATIN SMALL LETTER I
    Interval{ 0x207F, 0x207F }, // Lm       SUPERSCRIPT LATIN SMALL LETTER N
    Interval{ 0x2090, 0x209C }, // Lm  [13] LATIN SUBSCRIPT SMALL LETTER A..LATIN SUBSCRIPT SMALL LETTER T
    Interval{ 0x20D0, 0x20DC }, // Mn  [13] COMBINING LEFT HARPOON ABOVE..COMBINING FOUR DOTS ABOVE
    Interval{ 0x20E1, 0x20E1 }, // Mn       COMBINING LEFT RIGHT ARROW ABOVE
    Interval{ 0x20E5, 0x20F0 }, // Mn  [12] COMBINING REVERSE SOLIDUS OVERLAY..COMBINING ASTERISK ABOVE
    Interval{ 0x2102, 0x2102 }, // L&       DOUBLE-STRUCK CAPITAL C
    Interval{ 0x2107, 0x2107 }, // L&       EULER CONSTANT
    Interval{ 0x210A, 0x2113 }, // L&  [10] SCRIPT SMALL G..SCRIPT SMALL L
    Interval{ 0x2115, 0x2115 }, // L&       DOUBLE-STRUCK CAPITAL N
    Interval{ 0x2118, 0x2118 }, // Sm       SCRIPT CAPITAL P
    Interval{ 0x2119, 0x211D }, // L&   [5] DOUBLE-STRUCK CAPITAL P..DOUBLE-STRUCK CAPITAL R
    Interval{ 0x2124, 0x2124 }, // L&       DOUBLE-STRUCK CAPITAL Z
    Interval{ 0x2126, 0x2126 }, // L&       OHM SIGN
    Interval{ 0x2128, 0x2128 }, // L&       BLACK-LETTER CAPITAL Z
    Interval{ 0x212A, 0x212D }, // L&   [4] KELVIN SIGN..BLACK-LETTER CAPITAL C
    Interval{ 0x212E, 0x212E }, // So       ESTIMATED SYMBOL
    Interval{ 0x212F, 0x2134 }, // L&   [6] SCRIPT SMALL E..SCRIPT SMALL O
    Interval{ 0x2135, 0x2138 }, // Lo   [4] ALEF SYMBOL..DALET SYMBOL
    Interval{ 0x2139, 0x2139 }, // L&       INFORMATION SOURCE
    Interval{ 0x213C, 0x213F }, // L&   [4] DOUBLE-STRUCK SMALL PI..DOUBLE-STRUCK CAPITAL PI
    Interval{ 0x2145, 0x2149 }, // L&   [5] DOUBLE-STRUCK ITALIC CAPITAL D..DOUBLE-STRUCK ITALIC SMALL J
    Interval{ 0x214E, 0x214E }, // L&       TURNED SMALL F
    Interval{ 0x2160, 0x2182 }, // Nl  [35] ROMAN NUMERAL ONE..ROMAN NUMERAL TEN THOUSAND
    Interval{ 0x2183, 0x2184 }, // L&   [2] ROMAN NUMERAL REVERSED ONE HUNDRED..LATIN SMALL LETTER REVERSED C
    Interval{ 0x2185, 0x2188 }, // Nl   [4] ROMAN NUMERAL SIX LATE FORM..ROMAN NUMERAL ONE HUNDRED THOUSAND
    Interval{ 0x2C00, 0x2C7B }, // L& [124] GLAGOLITIC CAPITAL LETTER AZU..LATIN LETTER SMALL CAPITAL TURNED E
    Interval{ 0x2C7C, 0x2C7D }, // Lm   [2] LATIN SUBSCRIPT SMALL LETTER J..MODIFIER LETTER CAPITAL V
    Interval{ 0x2C7E, 0x2CE4 }, // L& [103] LATIN CAPITAL LETTER S WITH SWASH TAIL..COPTIC SYMBOL KAI
    Interval{ 0x2CEB, 0x2CEE }, // L&   [4] COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI..COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA
    Interval{ 0x2CEF, 0x2CF1 }, // Mn   [3] COPTIC COMBINING NI ABOVE..COPTIC COMBINING SPIRITUS LENIS
    Interval{ 0x2CF2, 0x2CF3 }, // L&   [2] COPTIC CAPITAL LETTER BOHAIRIC KHEI..COPTIC SMALL LETTER BOHAIRIC KHEI
    Interval{ 0x2D00, 0x2D25 }, // L&  [38] GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    Interval{ 0x2D27, 0x2D27 }, // L&       GEORGIAN SMALL LETTER YN
    Interval{ 0x2D2D, 0x2D2D }, // L&       GEORGIAN SMALL LETTER AEN
    Interval{ 0x2D30, 0x2D67 }, // Lo  [56] TIFINAGH LETTER YA..TIFINAGH LETTER YO
    Interval{ 0x2D6F, 0x2D6F }, // Lm       TIFINAGH MODIFIER LETTER LABIALIZATION MARK
    Interval{ 0x2D7F, 0x2D7F }, // Mn       TIFINAGH CONSONANT JOINER
    Interval{ 0x2D80, 0x2D96 }, // Lo  [23] ETHIOPIC SYLLABLE LOA..ETHIOPIC SYLLABLE GGWE
    Interval{ 0x2DA0, 0x2DA6 }, // Lo   [7] ETHIOPIC SYLLABLE SSA..ETHIOPIC SYLLABLE SSO
    Interval{ 0x2DA8, 0x2DAE }, // Lo   [7] ETHIOPIC SYLLABLE CCA..ETHIOPIC SYLLABLE CCO
    Interval{ 0x2DB0, 0x2DB6 }, // Lo   [7] ETHIOPIC SYLLABLE ZZA..ETHIOPIC SYLLABLE ZZO
    Interval{ 0x2DB8, 0x2DBE }, // Lo   [7] ETHIOPIC SYLLABLE CCHA..ETHIOPIC SYLLABLE CCHO
    Interval{ 0x2DC0, 0x2DC6 }, // Lo   [7] ETHIOPIC SYLLABLE QYA..ETHIOPIC SYLLABLE QYO
    Interval{ 0x2DC8, 0x2DCE }, // Lo   [7] ETHIOPIC SYLLABLE KYA..ETHIOPIC SYLLABLE KYO
    Interval{ 0x2DD0, 0x2DD6 }, // Lo   [7] ETHIOPIC SYLLABLE XYA..ETHIOPIC SYLLABLE XYO
    Interval{ 0x2DD8, 0x2DDE }, // Lo   [7] ETHIOPIC SYLLABLE GYA..ETHIOPIC SYLLABLE GYO
    Interval{ 0x2DE0, 0x2DFF }, // Mn  [32] COMBINING CYRILLIC LETTER BE..COMBINING CYRILLIC LETTER IOTIFIED BIG YUS
    Interval{ 0x3005, 0x3005 }, // Lm       IDEOGRAPHIC ITERATION MARK
    Interval{ 0x3006, 0x3006 }, // Lo       IDEOGRAPHIC CLOSING MARK
    Interval{ 0x3007, 0x3007 }, // Nl       IDEOGRAPHIC NUMBER ZERO
    Interval{ 0x3021, 0x3029 }, // Nl   [9] HANGZHOU NUMERAL ONE..HANGZHOU NUMERAL NINE
    Interval{ 0x302A, 0x302D }, // Mn   [4] IDEOGRAPHIC LEVEL TONE MARK..IDEOGRAPHIC ENTERING TONE MARK
    Interval{ 0x302E, 0x302F }, // Mc   [2] HANGUL SINGLE DOT TONE MARK..HANGUL DOUBLE DOT TONE MARK
    Interval{ 0x3031, 0x3035 }, // Lm   [5] VERTICAL KANA REPEAT MARK..VERTICAL KANA REPEAT MARK LOWER HALF
    Interval{ 0x3038, 0x303A }, // Nl   [3] HANGZHOU NUMERAL TEN..HANGZHOU NUMERAL THIRTY
    Interval{ 0x303B, 0x303B }, // Lm       VERTICAL IDEOGRAPHIC ITERATION MARK
    Interval{ 0x303C, 0x303C }, // Lo       MASU MARK
    Interval{ 0x3041, 0x3096 }, // Lo  [86] HIRAGANA LETTER SMALL A..HIRAGANA LETTER SMALL KE
    Interval{ 0x3099, 0x309A }, // Mn   [2] COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK..COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    Interval{ 0x309D, 0x309E }, // Lm   [2] HIRAGANA ITERATION MARK..HIRAGANA VOICED ITERATION MARK
    Interval{ 0x309F, 0x309F }, // Lo       HIRAGANA DIGRAPH YORI
    Interval{ 0x30A1, 0x30FA }, // Lo  [90] KATAKANA LETTER SMALL A..KATAKANA LETTER VO
    Interval{ 0x30FC, 0x30FE }, // Lm   [3] KATAKANA-HIRAGANA PROLONGED SOUND MARK..KATAKANA VOICED ITERATION MARK
    Interval{ 0x30FF, 0x30FF }, // Lo       KATAKANA DIGRAPH KOTO
    Interval{ 0x3105, 0x312F }, // Lo  [43] BOPOMOFO LETTER B..BOPOMOFO LETTER NN
    Interval{ 0x3131, 0x318E }, // Lo  [94] HANGUL LETTER KIYEOK..HANGUL LETTER ARAEAE
    Interval{ 0x31A0, 0x31BF }, // Lo  [32] BOPOMOFO LETTER BU..BOPOMOFO LETTER AH
    Interval{ 0x31F0, 0x31FF }, // Lo  [16] KATAKANA LETTER SMALL KU..KATAKANA LETTER SMALL RO
    Interval{ 0x3400, 0x4DBF }, // Lo [6592] CJK UNIFIED IDEOGRAPH-3400..CJK UNIFIED IDEOGRAPH-4DBF
    Interval{ 0x4E00, 0xA014 }, // Lo [21013] CJK UNIFIED IDEOGRAPH-4E00..YI SYLLABLE E
    Interval{ 0xA015, 0xA015 }, // Lm       YI SYLLABLE WU
    Interval{ 0xA016, 0xA48C }, // Lo [1143] YI SYLLABLE BIT..YI SYLLABLE YYR
    Interval{ 0xA4D0, 0xA4F7 }, // Lo  [40] LISU LETTER BA..LISU LETTER OE
    Interval{ 0xA4F8, 0xA4FD }, // Lm   [6] LISU LETTER TONE MYA TI..LISU LETTER TONE MYA JEU
    Interval{ 0xA500, 0xA60B }, // Lo [268] VAI SYLLABLE EE..VAI SYLLABLE NG
    Interval{ 0xA60C, 0xA60C }, // Lm       VAI SYLLABLE LENGTHENER
    Interval{ 0xA610, 0xA61F }, // Lo  [16] VAI SYLLABLE NDOLE FA..VAI SYMBOL JONG
    Interval{ 0xA620, 0xA629 }, // Nd  [10] VAI DIGIT ZERO..VAI DIGIT NINE
    Interval{ 0xA62A, 0xA62B }, // Lo   [2] VAI SYLLABLE NDOLE MA..VAI SYLLABLE NDOLE DO
    Interval{ 0xA640, 0xA66D }, // L&  [46] CYRILLIC CAPITAL LETTER ZEMLYA..CYRILLIC SMALL LETTER DOUBLE MONOCULAR O
    Interval{ 0xA66E, 0xA66E }, // Lo       CYRILLIC LETTER MULTIOCULAR O
    Interval{ 0xA66F, 0xA66F }, // Mn       COMBINING CYRILLIC VZMET
    Interval{ 0xA674, 0xA67D }, // Mn  [10] COMBINING CYRILLIC LETTER UKRAINIAN IE..COMBINING CYRILLIC PAYEROK
    Interval{ 0xA67F, 0xA67F }, // Lm       CYRILLIC PAYEROK
    Interval{ 0xA680, 0xA69B }, // L&  [28] CYRILLIC CAPITAL LETTER DWE..CYRILLIC SMALL LETTER CROSSED O
    Interval{ 0xA69C, 0xA69D }, // Lm   [2] MODIFIER LETTER CYRILLIC HARD SIGN..MODIFIER LETTER CYRILLIC SOFT SIGN
    Interval{ 0xA69E, 0xA69F }, // Mn   [2] COMBINING CYRILLIC LETTER EF..COMBINING CYRILLIC LETTER IOTIFIED E
    Interval{ 0xA6A0, 0xA6E5 }, // Lo  [70] BAMUM LETTER A..BAMUM LETTER KI
    Interval{ 0xA6E6, 0xA6EF }, // Nl  [10] BAMUM LETTER MO..BAMUM LETTER KOGHOM
    Interval{ 0xA6F0, 0xA6F1 }, // Mn   [2] BAMUM COMBINING MARK KOQNDON..BAMUM COMBINING MARK TUKWENTIS
    Interval{ 0xA717, 0xA71F }, // Lm   [9] MODIFIER LETTER DOT VERTICAL BAR..MODIFIER LETTER LOW INVERTED EXCLAMATION MARK
    Interval{ 0xA722, 0xA76F }, // L&  [78] LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF..LATIN SMALL LETTER CON
    Interval{ 0xA770, 0xA770 }, // Lm       MODIFIER LETTER US
    Interval{ 0xA771, 0xA787 }, // L&  [23] LATIN SMALL LETTER DUM..LATIN SMALL LETTER INSULAR T
    Interval{ 0xA788, 0xA788 }, // Lm       MODIFIER LETTER LOW CIRCUMFLEX ACCENT
    Interval{ 0xA78B, 0xA78E }, // L&   [4] LATIN CAPITAL LETTER SALTILLO..LATIN SMALL LETTER L WITH RETROFLEX HOOK AND BELT
    Interval{ 0xA78F, 0xA78F }, // Lo       LATIN LETTER SINOLOGICAL DOT
    Interval{ 0xA790, 0xA7CA }, // L&  [59] LATIN CAPITAL LETTER N WITH DESCENDER..LATIN SMALL LETTER S WITH SHORT STROKE OVERLAY
    Interval{ 0xA7D0, 0xA7D1 }, // L&   [2] LATIN CAPITAL LETTER CLOSED INSULAR G..LATIN SMALL LETTER CLOSED INSULAR G
    Interval{ 0xA7D3, 0xA7D3 }, // L&       LATIN SMALL LETTER DOUBLE THORN
    Interval{ 0xA7D5, 0xA7D9 }, // L&   [5] LATIN SMALL LETTER DOUBLE WYNN..LATIN SMALL LETTER SIGMOID S
    Interval{ 0xA7F2, 0xA7F4 }, // Lm   [3] MODIFIER LETTER CAPITAL C..MODIFIER LETTER CAPITAL Q
    Interval{ 0xA7F5, 0xA7F6 }, // L&   [2] LATIN CAPITAL LETTER REVERSED HALF H..LATIN SMALL LETTER REVERSED HALF H
    Interval{ 0xA7F7, 0xA7F7 }, // Lo       LATIN EPIGRAPHIC LETTER SIDEWAYS I
    Interval{ 0xA7F8, 0xA7F9 }, // Lm   [2] MODIFIER LETTER CAPITAL H WITH STROKE..MODIFIER LETTER SMALL LIGATURE OE
    Interval{ 0xA7FA, 0xA7FA }, // L&       LATIN LETTER SMALL CAPITAL TURNED M
    Interval{ 0xA7FB, 0xA801 }, // Lo   [7] LATIN EPIGRAPHIC LETTER REVERSED F..SYLOTI NAGRI LETTER I
    Interval{ 0xA802, 0xA802 }, // Mn       SYLOTI NAGRI SIGN DVISVARA
    Interval{ 0xA803, 0xA805 }, // Lo   [3] SYLOTI NAGRI LETTER U..SYLOTI NAGRI LETTER O
    Interval{ 0xA806, 0xA806 }, // Mn       SYLOTI NAGRI SIGN HASANTA
    Interval{ 0xA807, 0xA80A }, // Lo   [4] SYLOTI NAGRI LETTER KO..SYLOTI NAGRI LETTER GHO
    Interval{ 0xA80B, 0xA80B }, // Mn       SYLOTI NAGRI SIGN ANUSVARA
    Interval{ 0xA80C, 0xA822 }, // Lo  [23] SYLOTI NAGRI LETTER CO..SYLOTI NAGRI LETTER HO
    Interval{ 0xA823, 0xA824 }, // Mc   [2] SYLOTI NAGRI VOWEL SIGN A..SYLOTI NAGRI VOWEL SIGN I
    Interval{ 0xA825, 0xA826 }, // Mn   [2] SYLOTI NAGRI VOWEL SIGN U..SYLOTI NAGRI VOWEL SIGN E
    Interval{ 0xA827, 0xA827 }, // Mc       SYLOTI NAGRI VOWEL SIGN OO
    Interval{ 0xA82C, 0xA82C }, // Mn       SYLOTI NAGRI SIGN ALTERNATE HASANTA
    Interval{ 0xA840, 0xA873 }, // Lo  [52] PHAGS-PA LETTER KA..PHAGS-PA LETTER CANDRABINDU
    Interval{ 0xA880, 0xA881 }, // Mc   [2] SAURASHTRA SIGN ANUSVARA..SAURASHTRA SIGN VISARGA
    Interval{ 0xA882, 0xA8B3 }, // Lo  [50] SAURASHTRA LETTER A..SAURASHTRA LETTER LLA
    Interval{ 0xA8B4, 0xA8C3 }, // Mc  [16] SAURASHTRA CONSONANT SIGN HAARU..SAURASHTRA VOWEL SIGN AU
    Interval{ 0xA8C4, 0xA8C5 }, // Mn   [2] SAURASHTRA SIGN VIRAMA..SAURASHTRA SIGN CANDRABINDU
    Interval{ 0xA8D0, 0xA8D9 }, // Nd  [10] SAURASHTRA DIGIT ZERO..SAURASHTRA DIGIT NINE
    Interval{ 0xA8E0, 0xA8F1 }, // Mn  [18] COMBINING DEVANAGARI DIGIT ZERO..COMBINING DEVANAGARI SIGN AVAGRAHA
    Interval{ 0xA8F2, 0xA8F7 }, // Lo   [6] DEVANAGARI SIGN SPACING CANDRABINDU..DEVANAGARI SIGN CANDRABINDU AVAGRAHA
    Interval{ 0xA8FB, 0xA8FB }, // Lo       DEVANAGARI HEADSTROKE
    Interval{ 0xA8FD, 0xA8FE }, // Lo   [2] DEVANAGARI JAIN OM..DEVANAGARI LETTER AY
    Interval{ 0xA8FF, 0xA8FF }, // Mn       DEVANAGARI VOWEL SIGN AY
    Interval{ 0xA900, 0xA909 }, // Nd  [10] KAYAH LI DIGIT ZERO..KAYAH LI DIGIT NINE
    Interval{ 0xA90A, 0xA925 }, // Lo  [28] KAYAH LI LETTER KA..KAYAH LI LETTER OO
    Interval{ 0xA926, 0xA92D }, // Mn   [8] KAYAH LI VOWEL UE..KAYAH LI TONE CALYA PLOPHU
    Interval{ 0xA930, 0xA946 }, // Lo  [23] REJANG LETTER KA..REJANG LETTER A
    Interval{ 0xA947, 0xA951 }, // Mn  [11] REJANG VOWEL SIGN I..REJANG CONSONANT SIGN R
    Interval{ 0xA952, 0xA953 }, // Mc   [2] REJANG CONSONANT SIGN H..REJANG VIRAMA
    Interval{ 0xA960, 0xA97C }, // Lo  [29] HANGUL CHOSEONG TIKEUT-MIEUM..HANGUL CHOSEONG SSANGYEORINHIEUH
    Interval{ 0xA980, 0xA982 }, // Mn   [3] JAVANESE SIGN PANYANGGA..JAVANESE SIGN LAYAR
    Interval{ 0xA983, 0xA983 }, // Mc       JAVANESE SIGN WIGNYAN
    Interval{ 0xA984, 0xA9B2 }, // Lo  [47] JAVANESE LETTER A..JAVANESE LETTER HA
    Interval{ 0xA9B3, 0xA9B3 }, // Mn       JAVANESE SIGN CECAK TELU
    Interval{ 0xA9B4, 0xA9B5 }, // Mc   [2] JAVANESE VOWEL SIGN TARUNG..JAVANESE VOWEL SIGN TOLONG
    Interval{ 0xA9B6, 0xA9B9 }, // Mn   [4] JAVANESE VOWEL SIGN WULU..JAVANESE VOWEL SIGN SUKU MENDUT
    Interval{ 0xA9BA, 0xA9BB }, // Mc   [2] JAVANESE VOWEL SIGN TALING..JAVANESE VOWEL SIGN DIRGA MURE
    Interval{ 0xA9BC, 0xA9BD }, // Mn   [2] JAVANESE VOWEL SIGN PEPET..JAVANESE CONSONANT SIGN KERET
    Interval{ 0xA9BE, 0xA9C0 }, // Mc   [3] JAVANESE CONSONANT SIGN PENGKAL..JAVANESE PANGKON
    Interval{ 0xA9CF, 0xA9CF }, // Lm       JAVANESE PANGRANGKEP
    Interval{ 0xA9D0, 0xA9D9 }, // Nd  [10] JAVANESE DIGIT ZERO..JAVANESE DIGIT NINE
    Interval{ 0xA9E0, 0xA9E4 }, // Lo   [5] MYANMAR LETTER SHAN GHA..MYANMAR LETTER SHAN BHA
    Interval{ 0xA9E5, 0xA9E5 }, // Mn       MYANMAR SIGN SHAN SAW
    Interval{ 0xA9E6, 0xA9E6 }, // Lm       MYANMAR MODIFIER LETTER SHAN REDUPLICATION
    Interval{ 0xA9E7, 0xA9EF }, // Lo   [9] MYANMAR LETTER TAI LAING NYA..MYANMAR LETTER TAI LAING NNA
    Interval{ 0xA9F0, 0xA9F9 }, // Nd  [10] MYANMAR TAI LAING DIGIT ZERO..MYANMAR TAI LAING DIGIT NINE
    Interval{ 0xA9FA, 0xA9FE }, // Lo   [5] MYANMAR LETTER TAI LAING LLA..MYANMAR LETTER TAI LAING BHA
    Interval{ 0xAA00, 0xAA28 }, // Lo  [41] CHAM LETTER A..CHAM LETTER HA
    Interval{ 0xAA29, 0xAA2E }, // Mn   [6] CHAM VOWEL SIGN AA..CHAM VOWEL SIGN OE
    Interval{ 0xAA2F, 0xAA30 }, // Mc   [2] CHAM VOWEL SIGN O..CHAM VOWEL SIGN AI
    Interval{ 0xAA31, 0xAA32 }, // Mn   [2] CHAM VOWEL SIGN AU..CHAM VOWEL SIGN UE
    Interval{ 0xAA33, 0xAA34 }, // Mc   [2] CHAM CONSONANT SIGN YA..CHAM CONSONANT SIGN RA
    Interval{ 0xAA35, 0xAA36 }, // Mn   [2] CHAM CONSONANT SIGN LA..CHAM CONSONANT SIGN WA
    Interval{ 0xAA40, 0xAA42 }, // Lo   [3] CHAM LETTER FINAL K..CHAM LETTER FINAL NG
    Interval{ 0xAA43, 0xAA43 }, // Mn       CHAM CONSONANT SIGN FINAL NG
    Interval{ 0xAA44, 0xAA4B }, // Lo   [8] CHAM LETTER FINAL CH..CHAM LETTER FINAL SS
    Interval{ 0xAA4C, 0xAA4C }, // Mn       CHAM CONSONANT SIGN FINAL M
    Interval{ 0xAA4D, 0xAA4D }, // Mc       CHAM CONSONANT SIGN FINAL H
    Interval{ 0xAA50, 0xAA59 }, // Nd  [10] CHAM DIGIT ZERO..CHAM DIGIT NINE
    Interval{ 0xAA60, 0xAA6F }, // Lo  [16] MYANMAR LETTER KHAMTI GA..MYANMAR LETTER KHAMTI FA
    Interval{ 0xAA70, 0xAA70 }, // Lm       MYANMAR MODIFIER LETTER KHAMTI REDUPLICATION
    Interval{ 0xAA71, 0xAA76 }, // Lo   [6] MYANMAR LETTER KHAMTI XA..MYANMAR LOGOGRAM KHAMTI HM
    Interval{ 0xAA7A, 0xAA7A }, // Lo       MYANMAR LETTER AITON RA
    Interval{ 0xAA7B, 0xAA7B }, // Mc       MYANMAR SIGN PAO KAREN TONE
    Interval{ 0xAA7C, 0xAA7C }, // Mn       MYANMAR SIGN TAI LAING TONE-2
    Interval{ 0xAA7D, 0xAA7D }, // Mc       MYANMAR SIGN TAI LAING TONE-5
    Interval{ 0xAA7E, 0xAAAF }, // Lo  [50] MYANMAR LETTER SHWE PALAUNG CHA..TAI VIET LETTER HIGH O
    Interval{ 0xAAB0, 0xAAB0 }, // Mn       TAI VIET MAI KANG
    Interval{ 0xAAB1, 0xAAB1 }, // Lo       TAI VIET VOWEL AA
    Interval{ 0xAAB2, 0xAAB4 }, // Mn   [3] TAI VIET VOWEL I..TAI VIET VOWEL U
    Interval{ 0xAAB5, 0xAAB6 }, // Lo   [2] TAI VIET VOWEL E..TAI VIET VOWEL O
    Interval{ 0xAAB7, 0xAAB8 }, // Mn   [2] TAI VIET MAI KHIT..TAI VIET VOWEL IA
    Interval{ 0xAAB9, 0xAABD }, // Lo   [5] TAI VIET VOWEL UEA..TAI VIET VOWEL AN
    Interval{ 0xAABE, 0xAABF }, // Mn   [2] TAI VIET VOWEL AM..TAI VIET TONE MAI EK
    Interval{ 0xAAC0, 0xAAC0 }, // Lo       TAI VIET TONE MAI NUENG
    Interval{ 0xAAC1, 0xAAC1 }, // Mn       TAI VIET TONE MAI THO
    Interval{ 0xAAC2, 0xAAC2 }, // Lo       TAI VIET TONE MAI SONG
    Interval{ 0xAADB, 0xAADC }, // Lo   [2] TAI VIET SYMBOL KON..TAI VIET SYMBOL NUENG
    Interval{ 0xAADD, 0xAADD }, // Lm       TAI VIET SYMBOL SAM
    Interval{ 0xAAE0, 0xAAEA }, // Lo  [11] MEETEI MAYEK LETTER E..MEETEI MAYEK LETTER SSA
    Interval{ 0xAAEB, 0xAAEB }, // Mc       MEETEI MAYEK VOWEL SIGN II
    Interval{ 0xAAEC, 0xAAED }, // Mn   [2] MEETEI MAYEK VOWEL SIGN UU..MEETEI MAYEK VOWEL SIGN AAI
    Interval{ 0xAAEE, 0xAAEF }, // Mc   [2] MEETEI MAYEK VOWEL SIGN AU..MEETEI MAYEK VOWEL SIGN AAU
    Interval{ 0xAAF2, 0xAAF2 }, // Lo       MEETEI MAYEK ANJI
    Interval{ 0xAAF3, 0xAAF4 }, // Lm   [2] MEETEI MAYEK SYLLABLE REPETITION MARK..MEETEI MAYEK WORD REPETITION MARK
    Interval{ 0xAAF5, 0xAAF5 }, // Mc       MEETEI MAYEK VOWEL SIGN VISARGA
    Interval{ 0xAAF6, 0xAAF6 }, // Mn       MEETEI MAYEK VIRAMA
    Interval{ 0xAB01, 0xAB06 }, // Lo   [6] ETHIOPIC SYLLABLE TTHU..ETHIOPIC SYLLABLE TTHO
    Interval{ 0xAB09, 0xAB0E }, // Lo   [6] ETHIOPIC SYLLABLE DDHU..ETHIOPIC SYLLABLE DDHO
    Interval{ 0xAB11, 0xAB16 }, // Lo   [6] ETHIOPIC SYLLABLE DZU..ETHIOPIC SYLLABLE DZO
    Interval{ 0xAB20, 0xAB26 }, // Lo   [7] ETHIOPIC SYLLABLE CCHHA..ETHIOPIC SYLLABLE CCHHO
    Interval{ 0xAB28, 0xAB2E }, // Lo   [7] ETHIOPIC SYLLABLE BBA..ETHIOPIC SYLLABLE BBO
    Interval{ 0xAB30, 0xAB5A }, // L&  [43] LATIN SMALL LETTER BARRED ALPHA..LATIN SMALL LETTER Y WITH SHORT RIGHT LEG
    Interval{ 0xAB5C, 0xAB5F }, // Lm   [4] MODIFIER LETTER SMALL HENG..MODIFIER LETTER SMALL U WITH LEFT HOOK
    Interval{ 0xAB60, 0xAB68 }, // L&   [9] LATIN SMALL LETTER SAKHA YAT..LATIN SMALL LETTER TURNED R WITH MIDDLE TILDE
    Interval{ 0xAB69, 0xAB69 }, // Lm       MODIFIER LETTER SMALL TURNED W
    Interval{ 0xAB70, 0xABBF }, // L&  [80] CHEROKEE SMALL LETTER A..CHEROKEE SMALL LETTER YA
    Interval{ 0xABC0, 0xABE2 }, // Lo  [35] MEETEI MAYEK LETTER KOK..MEETEI MAYEK LETTER I LONSUM
    Interval{ 0xABE3, 0xABE4 }, // Mc   [2] MEETEI MAYEK VOWEL SIGN ONAP..MEETEI MAYEK VOWEL SIGN INAP
    Interval{ 0xABE5, 0xABE5 }, // Mn       MEETEI MAYEK VOWEL SIGN ANAP
    Interval{ 0xABE6, 0xABE7 }, // Mc   [2] MEETEI MAYEK VOWEL SIGN YENAP..MEETEI MAYEK VOWEL SIGN SOUNAP
    Interval{ 0xABE8, 0xABE8 }, // Mn       MEETEI MAYEK VOWEL SIGN UNAP
    Interval{ 0xABE9, 0xABEA }, // Mc   [2] MEETEI MAYEK VOWEL SIGN CHEINAP..MEETEI MAYEK VOWEL SIGN NUNG
    Interval{ 0xABEC, 0xABEC }, // Mc       MEETEI MAYEK LUM IYEK
    Interval{ 0xABED, 0xABED }, // Mn       MEETEI MAYEK APUN IYEK
    Interval{ 0xABF0, 0xABF9 }, // Nd  [10] MEETEI MAYEK DIGIT ZERO..MEETEI MAYEK DIGIT NINE
    Interval{ 0xAC00, 0xD7A3 }, // Lo [11172] HANGUL SYLLABLE GA..HANGUL SYLLABLE HIH
    Interval{ 0xD7B0, 0xD7C6 }, // Lo  [23] HANGUL JUNGSEONG O-YEO..HANGUL JUNGSEONG ARAEA-E
    Interval{ 0xD7CB, 0xD7FB }, // Lo  [49] HANGUL JONGSEONG NIEUN-RIEUL..HANGUL JONGSEONG PHIEUPH-THIEUTH
    Interval{ 0xF900, 0xFA6D }, // Lo [366] CJK COMPATIBILITY IDEOGRAPH-F900..CJK COMPATIBILITY IDEOGRAPH-FA6D
    Interval{ 0xFA70, 0xFAD9 }, // Lo [106] CJK COMPATIBILITY IDEOGRAPH-FA70..CJK COMPATIBILITY IDEOGRAPH-FAD9
    Interval{ 0xFB00, 0xFB06 }, // L&   [7] LATIN SMALL LIGATURE FF..LATIN SMALL LIGATURE ST
    Interval{ 0xFB13, 0xFB17 }, // L&   [5] ARMENIAN SMALL LIGATURE MEN NOW..ARMENIAN SMALL LIGATURE MEN XEH
    Interval{ 0xFB1D, 0xFB1D }, // Lo       HEBREW LETTER YOD WITH HIRIQ
    Interval{ 0xFB1E, 0xFB1E }, // Mn       HEBREW POINT JUDEO-SPANISH VARIKA
    Interval{ 0xFB1F, 0xFB28 }, // Lo  [10] HEBREW LIGATURE YIDDISH YOD YOD PATAH..HEBREW LETTER WIDE TAV
    Interval{ 0xFB2A, 0xFB36 }, // Lo  [13] HEBREW LETTER SHIN WITH SHIN DOT..HEBREW LETTER ZAYIN WITH DAGESH
    Interval{ 0xFB38, 0xFB3C }, // Lo   [5] HEBREW LETTER TET WITH DAGESH..HEBREW LETTER LAMED WITH DAGESH
    Interval{ 0xFB3E, 0xFB3E }, // Lo       HEBREW LETTER MEM WITH DAGESH
    Interval{ 0xFB40, 0xFB41 }, // Lo   [2] HEBREW LETTER NUN WITH DAGESH..HEBREW LETTER SAMEKH WITH DAGESH
    Interval{ 0xFB43, 0xFB44 }, // Lo   [2] HEBREW LETTER FINAL PE WITH DAGESH..HEBREW LETTER PE WITH DAGESH
    Interval{ 0xFB46, 0xFBB1 }, // Lo [108] HEBREW LETTER TSADI WITH DAGESH..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
    Interval{ 0xFBD3, 0xFC5D }, // Lo [139] ARABIC LETTER NG ISOLATED FORM..ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF ISOLATED FORM
    Interval{ 0xFC64, 0xFD3D }, // Lo [218] ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH REH FINAL FORM..ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
    Interval{ 0xFD50, 0xFD8F }, // Lo  [64] ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM..ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
    Interval{ 0xFD92, 0xFDC7 }, // Lo  [54] ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM..ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
    Interval{ 0xFDF0, 0xFDF9 }, // Lo  [10] ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM..ARABIC LIGATURE SALLA ISOLATED FORM
    Interval{ 0xFE00, 0xFE0F }, // Mn  [16] VARIATION SELECTOR-1..VARIATION SELECTOR-16
    Interval{ 0xFE20, 0xFE2F }, // Mn  [16] COMBINING LIGATURE LEFT HALF..COMBINING CYRILLIC TITLO RIGHT HALF
    Interval{ 0xFE33, 0xFE34 }, // Pc   [2] PRESENTATION FORM FOR VERTICAL LOW LINE..PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
    Interval{ 0xFE4D, 0xFE4F }, // Pc   [3] DASHED LOW LINE..WAVY LOW LINE
    Interval{ 0xFE71, 0xFE71 }, // Lo       ARABIC TATWEEL WITH FATHATAN ABOVE
    Interval{ 0xFE73, 0xFE73 }, // Lo       ARABIC TAIL FRAGMENT
    Interval{ 0xFE77, 0xFE77 }, // Lo       ARABIC FATHA MEDIAL FORM
    Interval{ 0xFE79, 0xFE79 }, // Lo       ARABIC DAMMA MEDIAL FORM
    Interval{ 0xFE7B, 0xFE7B }, // Lo       ARABIC KASRA MEDIAL FORM
    Interval{ 0xFE7D, 0xFE7D }, // Lo       ARABIC SHADDA MEDIAL FORM
    Interval{ 0xFE7F, 0xFEFC }, // Lo [126] ARABIC SUKUN MEDIAL FORM..ARABIC LIGATURE LAM WITH ALEF FINAL FORM
    Interval{ 0xFF10, 0xFF19 }, // Nd  [10] FULLWIDTH DIGIT ZERO..FULLWIDTH DIGIT NINE
    Interval{ 0xFF21, 0xFF3A }, // L&  [26] FULLWIDTH LATIN CAPITAL LETTER A..FULLWIDTH LATIN CAPITAL LETTER Z
    Interval{ 0xFF3F, 0xFF3F }, // Pc       FULLWIDTH LOW LINE
    Interval{ 0xFF41, 0xFF5A }, // L&  [26] FULLWIDTH LATIN SMALL LETTER A..FULLWIDTH LATIN SMALL LETTER Z
    Interval{ 0xFF66, 0xFF6F }, // Lo  [10] HALFWIDTH KATAKANA LETTER WO..HALFWIDTH KATAKANA LETTER SMALL TU
    Interval{ 0xFF70, 0xFF70 }, // Lm       HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
    Interval{ 0xFF71, 0xFF9D }, // Lo  [45] HALFWIDTH KATAKANA LETTER A..HALFWIDTH KATAKANA LETTER N
    Interval{ 0xFF9E, 0xFF9F }, // Lm   [2] HALFWIDTH KATAKANA VOICED SOUND MARK..HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
    Interval{ 0xFFA0, 0xFFBE }, // Lo  [31] HALFWIDTH HANGUL FILLER..HALFWIDTH HANGUL LETTER HIEUH
    Interval{ 0xFFC2, 0xFFC7 }, // Lo   [6] HALFWIDTH HANGUL LETTER A..HALFWIDTH HANGUL LETTER E
    Interval{ 0xFFCA, 0xFFCF }, // Lo   [6] HALFWIDTH HANGUL LETTER YEO..HALFWIDTH HANGUL LETTER OE
    Interval{ 0xFFD2, 0xFFD7 }, // Lo   [6] HALFWIDTH HANGUL LETTER YO..HALFWIDTH HANGUL LETTER YU
    Interval{ 0xFFDA, 0xFFDC }, // Lo   [3] HALFWIDTH HANGUL LETTER EU..HALFWIDTH HANGUL LETTER I
    Interval{ 0x10000, 0x1000B }, // Lo  [12] LINEAR B SYLLABLE B008 A..LINEAR B SYLLABLE B046 JE
    Interval{ 0x1000D, 0x10026 }, // Lo  [26] LINEAR B SYLLABLE B036 JO..LINEAR B SYLLABLE B032 QO
    Interval{ 0x10028, 0x1003A }, // Lo  [19] LINEAR B SYLLABLE B060 RA..LINEAR B SYLLABLE B042 WO
    Interval{ 0x1003C, 0x1003D }, // Lo   [2] LINEAR B SYLLABLE B017 ZA..LINEAR B SYLLABLE B074 ZE
    Interval{ 0x1003F, 0x1004D }, // Lo  [15] LINEAR B SYLLABLE B020 ZO..LINEAR B SYLLABLE B091 TWO
    Interval{ 0x10050, 0x1005D }, // Lo  [14] LINEAR B SYMBOL B018..LINEAR B SYMBOL B089
    Interval{ 0x10080, 0x100FA }, // Lo [123] LINEAR B IDEOGRAM B100 MAN..LINEAR B IDEOGRAM VESSEL B305
    Interval{ 0x10140, 0x10174 }, // Nl  [53] GREEK ACROPHONIC ATTIC ONE QUARTER..GREEK ACROPHONIC STRATIAN FIFTY MNAS
    Interval{ 0x101FD, 0x101FD }, // Mn       PHAISTOS DISC SIGN COMBINING OBLIQUE STROKE
    Interval{ 0x10280, 0x1029C }, // Lo  [29] LYCIAN LETTER A..LYCIAN LETTER X
    Interval{ 0x102A0, 0x102D0 }, // Lo  [49] CARIAN LETTER A..CARIAN LETTER UUU3
    Interval{ 0x102E0, 0x102E0 }, // Mn       COPTIC EPACT THOUSANDS MARK
    Interval{ 0x10300, 0x1031F }, // Lo  [32] OLD ITALIC LETTER A..OLD ITALIC LETTER ESS
    Interval{ 0x1032D, 0x10340 }, // Lo  [20] OLD ITALIC LETTER YE..GOTHIC LETTER PAIRTHRA
    Interval{ 0x10341, 0x10341 }, // Nl       GOTHIC LETTER NINETY
    Interval{ 0x10342, 0x10349 }, // Lo   [8] GOTHIC LETTER RAIDA..GOTHIC LETTER OTHAL
    Interval{ 0x1034A, 0x1034A }, // Nl       GOTHIC LETTER NINE HUNDRED
    Interval{ 0x10350, 0x10375 }, // Lo  [38] OLD PERMIC LETTER AN..OLD PERMIC LETTER IA
    Interval{ 0x10376, 0x1037A }, // Mn   [5] COMBINING OLD PERMIC LETTER AN..COMBINING OLD PERMIC LETTER SII
    Interval{ 0x10380, 0x1039D }, // Lo  [30] UGARITIC LETTER ALPA..UGARITIC LETTER SSU
    Interval{ 0x103A0, 0x103C3 }, // Lo  [36] OLD PERSIAN SIGN A..OLD PERSIAN SIGN HA
    Interval{ 0x103C8, 0x103CF }, // Lo   [8] OLD PERSIAN SIGN AURAMAZDAA..OLD PERSIAN SIGN BUUMISH
    Interval{ 0x103D1, 0x103D5 }, // Nl   [5] OLD PERSIAN NUMBER ONE..OLD PERSIAN NUMBER HUNDRED
    Interval{ 0x10400, 0x1044F }, // L&  [80] DESERET CAPITAL LETTER LONG I..DESERET SMALL LETTER EW
    Interval{ 0x10450, 0x1049D }, // Lo  [78] SHAVIAN LETTER PEEP..OSMANYA LETTER OO
    Interval{ 0x104A0, 0x104A9 }, // Nd  [10] OSMANYA DIGIT ZERO..OSMANYA DIGIT NINE
    Interval{ 0x104B0, 0x104D3 }, // L&  [36] OSAGE CAPITAL LETTER A..OSAGE CAPITAL LETTER ZHA
    Interval{ 0x104D8, 0x104FB }, // L&  [36] OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    Interval{ 0x10500, 0x10527 }, // Lo  [40] ELBASAN LETTER A..ELBASAN LETTER KHE
    Interval{ 0x10530, 0x10563 }, // Lo  [52] CAUCASIAN ALBANIAN LETTER ALT..CAUCASIAN ALBANIAN LETTER KIW
    Interval{ 0x10570, 0x1057A }, // L&  [11] VITHKUQI CAPITAL LETTER A..VITHKUQI CAPITAL LETTER GA
    Interval{ 0x1057C, 0x1058A }, // L&  [15] VITHKUQI CAPITAL LETTER HA..VITHKUQI CAPITAL LETTER RE
    Interval{ 0x1058C, 0x10592 }, // L&   [7] VITHKUQI CAPITAL LETTER SE..VITHKUQI CAPITAL LETTER XE
    Interval{ 0x10594, 0x10595 }, // L&   [2] VITHKUQI CAPITAL LETTER Y..VITHKUQI CAPITAL LETTER ZE
    Interval{ 0x10597, 0x105A1 }, // L&  [11] VITHKUQI SMALL LETTER A..VITHKUQI SMALL LETTER GA
    Interval{ 0x105A3, 0x105B1 }, // L&  [15] VITHKUQI SMALL LETTER HA..VITHKUQI SMALL LETTER RE
    Interval{ 0x105B3, 0x105B9 }, // L&   [7] VITHKUQI SMALL LETTER SE..VITHKUQI SMALL LETTER XE
    Interval{ 0x105BB, 0x105BC }, // L&   [2] VITHKUQI SMALL LETTER Y..VITHKUQI SMALL LETTER ZE
    Interval{ 0x10600, 0x10736 }, // Lo [311] LINEAR A SIGN AB001..LINEAR A SIGN A664
    Interval{ 0x10740, 0x10755 }, // Lo  [22] LINEAR A SIGN A701 A..LINEAR A SIGN A732 JE
    Interval{ 0x10760, 0x10767 }, // Lo   [8] LINEAR A SIGN A800..LINEAR A SIGN A807
    Interval{ 0x10780, 0x10785 }, // Lm   [6] MODIFIER LETTER SMALL CAPITAL AA..MODIFIER LETTER SMALL B WITH HOOK
    Interval{ 0x10787, 0x107B0 }, // Lm  [42] MODIFIER LETTER SMALL DZ DIGRAPH..MODIFIER LETTER SMALL V WITH RIGHT HOOK
    Interval{ 0x107B2, 0x107BA }, // Lm   [9] MODIFIER LETTER SMALL CAPITAL Y..MODIFIER LETTER SMALL S WITH CURL
    Interval{ 0x10800, 0x10805 }, // Lo   [6] CYPRIOT SYLLABLE A..CYPRIOT SYLLABLE JA
    Interval{ 0x10808, 0x10808 }, // Lo       CYPRIOT SYLLABLE JO
    Interval{ 0x1080A, 0x10835 }, // Lo  [44] CYPRIOT SYLLABLE KA..CYPRIOT SYLLABLE WO
    Interval{ 0x10837, 0x10838 }, // Lo   [2] CYPRIOT SYLLABLE XA..CYPRIOT SYLLABLE XE
    Interval{ 0x1083C, 0x1083C }, // Lo       CYPRIOT SYLLABLE ZA
    Interval{ 0x1083F, 0x10855 }, // Lo  [23] CYPRIOT SYLLABLE ZO..IMPERIAL ARAMAIC LETTER TAW
    Interval{ 0x10860, 0x10876 }, // Lo  [23] PALMYRENE LETTER ALEPH..PALMYRENE LETTER TAW
    Interval{ 0x10880, 0x1089E }, // Lo  [31] NABATAEAN LETTER FINAL ALEPH..NABATAEAN LETTER TAW
    Interval{ 0x108E0, 0x108F2 }, // Lo  [19] HATRAN LETTER ALEPH..HATRAN LETTER QOPH
    Interval{ 0x108F4, 0x108F5 }, // Lo   [2] HATRAN LETTER SHIN..HATRAN LETTER TAW
    Interval{ 0x10900, 0x10915 }, // Lo  [22] PHOENICIAN LETTER ALF..PHOENICIAN LETTER TAU
    Interval{ 0x10920, 0x10939 }, // Lo  [26] LYDIAN LETTER A..LYDIAN LETTER C
    Interval{ 0x10980, 0x109B7 }, // Lo  [56] MEROITIC HIEROGLYPHIC LETTER A..MEROITIC CURSIVE LETTER DA
    Interval{ 0x109BE, 0x109BF }, // Lo   [2] MEROITIC CURSIVE LOGOGRAM RMT..MEROITIC CURSIVE LOGOGRAM IMN
    Interval{ 0x10A00, 0x10A00 }, // Lo       KHAROSHTHI LETTER A
    Interval{ 0x10A01, 0x10A03 }, // Mn   [3] KHAROSHTHI VOWEL SIGN I..KHAROSHTHI VOWEL SIGN VOCALIC R
    Interval{ 0x10A05, 0x10A06 }, // Mn   [2] KHAROSHTHI VOWEL SIGN E..KHAROSHTHI VOWEL SIGN O
    Interval{ 0x10A0C, 0x10A0F }, // Mn   [4] KHAROSHTHI VOWEL LENGTH MARK..KHAROSHTHI SIGN VISARGA
    Interval{ 0x10A10, 0x10A13 }, // Lo   [4] KHAROSHTHI LETTER KA..KHAROSHTHI LETTER GHA
    Interval{ 0x10A15, 0x10A17 }, // Lo   [3] KHAROSHTHI LETTER CA..KHAROSHTHI LETTER JA
    Interval{ 0x10A19, 0x10A35 }, // Lo  [29] KHAROSHTHI LETTER NYA..KHAROSHTHI LETTER VHA
    Interval{ 0x10A38, 0x10A3A }, // Mn   [3] KHAROSHTHI SIGN BAR ABOVE..KHAROSHTHI SIGN DOT BELOW
    Interval{ 0x10A3F, 0x10A3F }, // Mn       KHAROSHTHI VIRAMA
    Interval{ 0x10A60, 0x10A7C }, // Lo  [29] OLD SOUTH ARABIAN LETTER HE..OLD SOUTH ARABIAN LETTER THETH
    Interval{ 0x10A80, 0x10A9C }, // Lo  [29] OLD NORTH ARABIAN LETTER HEH..OLD NORTH ARABIAN LETTER ZAH
    Interval{ 0x10AC0, 0x10AC7 }, // Lo   [8] MANICHAEAN LETTER ALEPH..MANICHAEAN LETTER WAW
    Interval{ 0x10AC9, 0x10AE4 }, // Lo  [28] MANICHAEAN LETTER ZAYIN..MANICHAEAN LETTER TAW
    Interval{ 0x10AE5, 0x10AE6 }, // Mn   [2] MANICHAEAN ABBREVIATION MARK ABOVE..MANICHAEAN ABBREVIATION MARK BELOW
    Interval{ 0x10B00, 0x10B35 }, // Lo  [54] AVESTAN LETTER A..AVESTAN LETTER HE
    Interval{ 0x10B40, 0x10B55 }, // Lo  [22] INSCRIPTIONAL PARTHIAN LETTER ALEPH..INSCRIPTIONAL PARTHIAN LETTER TAW
    Interval{ 0x10B60, 0x10B72 }, // Lo  [19] INSCRIPTIONAL PAHLAVI LETTER ALEPH..INSCRIPTIONAL PAHLAVI LETTER TAW
    Interval{ 0x10B80, 0x10B91 }, // Lo  [18] PSALTER PAHLAVI LETTER ALEPH..PSALTER PAHLAVI LETTER TAW
    Interval{ 0x10C00, 0x10C48 }, // Lo  [73] OLD TURKIC LETTER ORKHON A..OLD TURKIC LETTER ORKHON BASH
    Interval{ 0x10C80, 0x10CB2 }, // L&  [51] OLD HUNGARIAN CAPITAL LETTER A..OLD HUNGARIAN CAPITAL LETTER US
    Interval{ 0x10CC0, 0x10CF2 }, // L&  [51] OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    Interval{ 0x10D00, 0x10D23 }, // Lo  [36] HANIFI ROHINGYA LETTER A..HANIFI ROHINGYA MARK NA KHONNA
    Interval{ 0x10D24, 0x10D27 }, // Mn   [4] HANIFI ROHINGYA SIGN HARBAHAY..HANIFI ROHINGYA SIGN TASSI
    Interval{ 0x10D30, 0x10D39 }, // Nd  [10] HANIFI ROHINGYA DIGIT ZERO..HANIFI ROHINGYA DIGIT NINE
    Interval{ 0x10E80, 0x10EA9 }, // Lo  [42] YEZIDI LETTER ELIF..YEZIDI LETTER ET
    Interval{ 0x10EAB, 0x10EAC }, // Mn   [2] YEZIDI COMBINING HAMZA MARK..YEZIDI COMBINING MADDA MARK
    Interval{ 0x10EB0, 0x10EB1 }, // Lo   [2] YEZIDI LETTER LAM WITH DOT ABOVE..YEZIDI LETTER YOT WITH CIRCUMFLEX ABOVE
    Interval{ 0x10EFD, 0x10EFF }, // Mn   [3] ARABIC SMALL LOW WORD SAKTA..ARABIC SMALL LOW WORD MADDA
    Interval{ 0x10F00, 0x10F1C }, // Lo  [29] OLD SOGDIAN LETTER ALEPH..OLD SOGDIAN LETTER FINAL TAW WITH VERTICAL TAIL
    Interval{ 0x10F27, 0x10F27 }, // Lo       OLD SOGDIAN LIGATURE AYIN-DALETH
    Interval{ 0x10F30, 0x10F45 }, // Lo  [22] SOGDIAN LETTER ALEPH..SOGDIAN INDEPENDENT SHIN
    Interval{ 0x10F46, 0x10F50 }, // Mn  [11] SOGDIAN COMBINING DOT BELOW..SOGDIAN COMBINING STROKE BELOW
    Interval{ 0x10F70, 0x10F81 }, // Lo  [18] OLD UYGHUR LETTER ALEPH..OLD UYGHUR LETTER LESH
    Interval{ 0x10F82, 0x10F85 }, // Mn   [4] OLD UYGHUR COMBINING DOT ABOVE..OLD UYGHUR COMBINING TWO DOTS BELOW
    Interval{ 0x10FB0, 0x10FC4 }, // Lo  [21] CHORASMIAN LETTER ALEPH..CHORASMIAN LETTER TAW
    Interval{ 0x10FE0, 0x10FF6 }, // Lo  [23] ELYMAIC LETTER ALEPH..ELYMAIC LIGATURE ZAYIN-YODH
    Interval{ 0x11000, 0x11000 }, // Mc       BRAHMI SIGN CANDRABINDU
    Interval{ 0x11001, 0x11001 }, // Mn       BRAHMI SIGN ANUSVARA
    Interval{ 0x11002, 0x11002 }, // Mc       BRAHMI SIGN VISARGA
    Interval{ 0x11003, 0x11037 }, // Lo  [53] BRAHMI SIGN JIHVAMULIYA..BRAHMI LETTER OLD TAMIL NNNA
    Interval{ 0x11038, 0x11046 }, // Mn  [15] BRAHMI VOWEL SIGN AA..BRAHMI VIRAMA
    Interval{ 0x11066, 0x1106F }, // Nd  [10] BRAHMI DIGIT ZERO..BRAHMI DIGIT NINE
    Interval{ 0x11070, 0x11070 }, // Mn       BRAHMI SIGN OLD TAMIL VIRAMA
    Interval{ 0x11071, 0x11072 }, // Lo   [2] BRAHMI LETTER OLD TAMIL SHORT E..BRAHMI LETTER OLD TAMIL SHORT O
    Interval{ 0x11073, 0x11074 }, // Mn   [2] BRAHMI VOWEL SIGN OLD TAMIL SHORT E..BRAHMI VOWEL SIGN OLD TAMIL SHORT O
    Interval{ 0x11075, 0x11075 }, // Lo       BRAHMI LETTER OLD TAMIL LLA
    Interval{ 0x1107F, 0x11081 }, // Mn   [3] BRAHMI NUMBER JOINER..KAITHI SIGN ANUSVARA
    Interval{ 0x11082, 0x11082 }, // Mc       KAITHI SIGN VISARGA
    Interval{ 0x11083, 0x110AF }, // Lo  [45] KAITHI LETTER A..KAITHI LETTER HA
    Interval{ 0x110B0, 0x110B2 }, // Mc   [3] KAITHI VOWEL SIGN AA..KAITHI VOWEL SIGN II
    Interval{ 0x110B3, 0x110B6 }, // Mn   [4] KAITHI VOWEL SIGN U..KAITHI VOWEL SIGN AI
    Interval{ 0x110B7, 0x110B8 }, // Mc   [2] KAITHI VOWEL SIGN O..KAITHI VOWEL SIGN AU
    Interval{ 0x110B9, 0x110BA }, // Mn   [2] KAITHI SIGN VIRAMA..KAITHI SIGN NUKTA
    Interval{ 0x110C2, 0x110C2 }, // Mn       KAITHI VOWEL SIGN VOCALIC R
    Interval{ 0x110D0, 0x110E8 }, // Lo  [25] SORA SOMPENG LETTER SAH..SORA SOMPENG LETTER MAE
    Interval{ 0x110F0, 0x110F9 }, // Nd  [10] SORA SOMPENG DIGIT ZERO..SORA SOMPENG DIGIT NINE
    Interval{ 0x11100, 0x11102 }, // Mn   [3] CHAKMA SIGN CANDRABINDU..CHAKMA SIGN VISARGA
    Interval{ 0x11103, 0x11126 }, // Lo  [36] CHAKMA LETTER AA..CHAKMA LETTER HAA
    Interval{ 0x11127, 0x1112B }, // Mn   [5] CHAKMA VOWEL SIGN A..CHAKMA VOWEL SIGN UU
    Interval{ 0x1112C, 0x1112C }, // Mc       CHAKMA VOWEL SIGN E
    Interval{ 0x1112D, 0x11134 }, // Mn   [8] CHAKMA VOWEL SIGN AI..CHAKMA MAAYYAA
    Interval{ 0x11136, 0x1113F }, // Nd  [10] CHAKMA DIGIT ZERO..CHAKMA DIGIT NINE
    Interval{ 0x11144, 0x11144 }, // Lo       CHAKMA LETTER LHAA
    Interval{ 0x11145, 0x11146 }, // Mc   [2] CHAKMA VOWEL SIGN AA..CHAKMA VOWEL SIGN EI
    Interval{ 0x11147, 0x11147 }, // Lo       CHAKMA LETTER VAA
    Interval{ 0x11150, 0x11172 }, // Lo  [35] MAHAJANI LETTER A..MAHAJANI LETTER RRA
    Interval{ 0x11173, 0x11173 }, // Mn       MAHAJANI SIGN NUKTA
    Interval{ 0x11176, 0x11176 }, // Lo       MAHAJANI LIGATURE SHRI
    Interval{ 0x11180, 0x11181 }, // Mn   [2] SHARADA SIGN CANDRABINDU..SHARADA SIGN ANUSVARA
    Interval{ 0x11182, 0x11182 }, // Mc       SHARADA SIGN VISARGA
    Interval{ 0x11183, 0x111B2 }, // Lo  [48] SHARADA LETTER A..SHARADA LETTER HA
    Interval{ 0x111B3, 0x111B5 }, // Mc   [3] SHARADA VOWEL SIGN AA..SHARADA VOWEL SIGN II
    Interval{ 0x111B6, 0x111BE }, // Mn   [9] SHARADA VOWEL SIGN U..SHARADA VOWEL SIGN O
    Interval{ 0x111BF, 0x111C0 }, // Mc   [2] SHARADA VOWEL SIGN AU..SHARADA SIGN VIRAMA
    Interval{ 0x111C1, 0x111C4 }, // Lo   [4] SHARADA SIGN AVAGRAHA..SHARADA OM
    Interval{ 0x111C9, 0x111CC }, // Mn   [4] SHARADA SANDHI MARK..SHARADA EXTRA SHORT VOWEL MARK
    Interval{ 0x111CE, 0x111CE }, // Mc       SHARADA VOWEL SIGN PRISHTHAMATRA E
    Interval{ 0x111CF, 0x111CF }, // Mn       SHARADA SIGN INVERTED CANDRABINDU
    Interval{ 0x111D0, 0x111D9 }, // Nd  [10] SHARADA DIGIT ZERO..SHARADA DIGIT NINE
    Interval{ 0x111DA, 0x111DA }, // Lo       SHARADA EKAM
    Interval{ 0x111DC, 0x111DC }, // Lo       SHARADA HEADSTROKE
    Interval{ 0x11200, 0x11211 }, // Lo  [18] KHOJKI LETTER A..KHOJKI LETTER JJA
    Interval{ 0x11213, 0x1122B }, // Lo  [25] KHOJKI LETTER NYA..KHOJKI LETTER LLA
    Interval{ 0x1122C, 0x1122E }, // Mc   [3] KHOJKI VOWEL SIGN AA..KHOJKI VOWEL SIGN II
    Interval{ 0x1122F, 0x11231 }, // Mn   [3] KHOJKI VOWEL SIGN U..KHOJKI VOWEL SIGN AI
    Interval{ 0x11232, 0x11233 }, // Mc   [2] KHOJKI VOWEL SIGN O..KHOJKI VOWEL SIGN AU
    Interval{ 0x11234, 0x11234 }, // Mn       KHOJKI SIGN ANUSVARA
    Interval{ 0x11235, 0x11235 }, // Mc       KHOJKI SIGN VIRAMA
    Interval{ 0x11236, 0x11237 }, // Mn   [2] KHOJKI SIGN NUKTA..KHOJKI SIGN SHADDA
    Interval{ 0x1123E, 0x1123E }, // Mn       KHOJKI SIGN SUKUN
    Interval{ 0x1123F, 0x11240 }, // Lo   [2] KHOJKI LETTER QA..KHOJKI LETTER SHORT I
    Interval{ 0x11241, 0x11241 }, // Mn       KHOJKI VOWEL SIGN VOCALIC R
    Interval{ 0x11280, 0x11286 }, // Lo   [7] MULTANI LETTER A..MULTANI LETTER GA
    Interval{ 0x11288, 0x11288 }, // Lo       MULTANI LETTER GHA
    Interval{ 0x1128A, 0x1128D }, // Lo   [4] MULTANI LETTER CA..MULTANI LETTER JJA
    Interval{ 0x1128F, 0x1129D }, // Lo  [15] MULTANI LETTER NYA..MULTANI LETTER BA
    Interval{ 0x1129F, 0x112A8 }, // Lo  [10] MULTANI LETTER BHA..MULTANI LETTER RHA
    Interval{ 0x112B0, 0x112DE }, // Lo  [47] KHUDAWADI LETTER A..KHUDAWADI LETTER HA
    Interval{ 0x112DF, 0x112DF }, // Mn       KHUDAWADI SIGN ANUSVARA
    Interval{ 0x112E0, 0x112E2 }, // Mc   [3] KHUDAWADI VOWEL SIGN AA..KHUDAWADI VOWEL SIGN II
    Interval{ 0x112E3, 0x112EA }, // Mn   [8] KHUDAWADI VOWEL SIGN U..KHUDAWADI SIGN VIRAMA
    Interval{ 0x112F0, 0x112F9 }, // Nd  [10] KHUDAWADI DIGIT ZERO..KHUDAWADI DIGIT NINE
    Interval{ 0x11300, 0x11301 }, // Mn   [2] GRANTHA SIGN COMBINING ANUSVARA ABOVE..GRANTHA SIGN CANDRABINDU
    Interval{ 0x11302, 0x11303 }, // Mc   [2] GRANTHA SIGN ANUSVARA..GRANTHA SIGN VISARGA
    Interval{ 0x11305, 0x1130C }, // Lo   [8] GRANTHA LETTER A..GRANTHA LETTER VOCALIC L
    Interval{ 0x1130F, 0x11310 }, // Lo   [2] GRANTHA LETTER EE..GRANTHA LETTER AI
    Interval{ 0x11313, 0x11328 }, // Lo  [22] GRANTHA LETTER OO..GRANTHA LETTER NA
    Interval{ 0x1132A, 0x11330 }, // Lo   [7] GRANTHA LETTER PA..GRANTHA LETTER RA
    Interval{ 0x11332, 0x11333 }, // Lo   [2] GRANTHA LETTER LA..GRANTHA LETTER LLA
    Interval{ 0x11335, 0x11339 }, // Lo   [5] GRANTHA LETTER VA..GRANTHA LETTER HA
    Interval{ 0x1133B, 0x1133C }, // Mn   [2] COMBINING BINDU BELOW..GRANTHA SIGN NUKTA
    Interval{ 0x1133D, 0x1133D }, // Lo       GRANTHA SIGN AVAGRAHA
    Interval{ 0x1133E, 0x1133F }, // Mc   [2] GRANTHA VOWEL SIGN AA..GRANTHA VOWEL SIGN I
    Interval{ 0x11340, 0x11340 }, // Mn       GRANTHA VOWEL SIGN II
    Interval{ 0x11341, 0x11344 }, // Mc   [4] GRANTHA VOWEL SIGN U..GRANTHA VOWEL SIGN VOCALIC RR
    Interval{ 0x11347, 0x11348 }, // Mc   [2] GRANTHA VOWEL SIGN EE..GRANTHA VOWEL SIGN AI
    Interval{ 0x1134B, 0x1134D }, // Mc   [3] GRANTHA VOWEL SIGN OO..GRANTHA SIGN VIRAMA
    Interval{ 0x11350, 0x11350 }, // Lo       GRANTHA OM
    Interval{ 0x11357, 0x11357 }, // Mc       GRANTHA AU LENGTH MARK
    Interval{ 0x1135D, 0x11361 }, // Lo   [5] GRANTHA SIGN PLUTA..GRANTHA LETTER VOCALIC LL
    Interval{ 0x11362, 0x11363 }, // Mc   [2] GRANTHA VOWEL SIGN VOCALIC L..GRANTHA VOWEL SIGN VOCALIC LL
    Interval{ 0x11366, 0x1136C }, // Mn   [7] COMBINING GRANTHA DIGIT ZERO..COMBINING GRANTHA DIGIT SIX
    Interval{ 0x11370, 0x11374 }, // Mn   [5] COMBINING GRANTHA LETTER A..COMBINING GRANTHA LETTER PA
    Interval{ 0x11400, 0x11434 }, // Lo  [53] NEWA LETTER A..NEWA LETTER HA
    Interval{ 0x11435, 0x11437 }, // Mc   [3] NEWA VOWEL SIGN AA..NEWA VOWEL SIGN II
    Interval{ 0x11438, 0x1143F }, // Mn   [8] NEWA VOWEL SIGN U..NEWA VOWEL SIGN AI
    Interval{ 0x11440, 0x11441 }, // Mc   [2] NEWA VOWEL SIGN O..NEWA VOWEL SIGN AU
    Interval{ 0x11442, 0x11444 }, // Mn   [3] NEWA SIGN VIRAMA..NEWA SIGN ANUSVARA
    Interval{ 0x11445, 0x11445 }, // Mc       NEWA SIGN VISARGA
    Interval{ 0x11446, 0x11446 }, // Mn       NEWA SIGN NUKTA
    Interval{ 0x11447, 0x1144A }, // Lo   [4] NEWA SIGN AVAGRAHA..NEWA SIDDHI
    Interval{ 0x11450, 0x11459 }, // Nd  [10] NEWA DIGIT ZERO..NEWA DIGIT NINE
    Interval{ 0x1145E, 0x1145E }, // Mn       NEWA SANDHI MARK
    Interval{ 0x1145F, 0x11461 }, // Lo   [3] NEWA LETTER VEDIC ANUSVARA..NEWA SIGN UPADHMANIYA
    Interval{ 0x11480, 0x114AF }, // Lo  [48] TIRHUTA ANJI..TIRHUTA LETTER HA
    Interval{ 0x114B0, 0x114B2 }, // Mc   [3] TIRHUTA VOWEL SIGN AA..TIRHUTA VOWEL SIGN II
    Interval{ 0x114B3, 0x114B8 }, // Mn   [6] TIRHUTA VOWEL SIGN U..TIRHUTA VOWEL SIGN VOCALIC LL
    Interval{ 0x114B9, 0x114B9 }, // Mc       TIRHUTA VOWEL SIGN E
    Interval{ 0x114BA, 0x114BA }, // Mn       TIRHUTA VOWEL SIGN SHORT E
    Interval{ 0x114BB, 0x114BE }, // Mc   [4] TIRHUTA VOWEL SIGN AI..TIRHUTA VOWEL SIGN AU
    Interval{ 0x114BF, 0x114C0 }, // Mn   [2] TIRHUTA SIGN CANDRABINDU..TIRHUTA SIGN ANUSVARA
    Interval{ 0x114C1, 0x114C1 }, // Mc       TIRHUTA SIGN VISARGA
    Interval{ 0x114C2, 0x114C3 }, // Mn   [2] TIRHUTA SIGN VIRAMA..TIRHUTA SIGN NUKTA
    Interval{ 0x114C4, 0x114C5 }, // Lo   [2] TIRHUTA SIGN AVAGRAHA..TIRHUTA GVANG
    Interval{ 0x114C7, 0x114C7 }, // Lo       TIRHUTA OM
    Interval{ 0x114D0, 0x114D9 }, // Nd  [10] TIRHUTA DIGIT ZERO..TIRHUTA DIGIT NINE
    Interval{ 0x11580, 0x115AE }, // Lo  [47] SIDDHAM LETTER A..SIDDHAM LETTER HA
    Interval{ 0x115AF, 0x115B1 }, // Mc   [3] SIDDHAM VOWEL SIGN AA..SIDDHAM VOWEL SIGN II
    Interval{ 0x115B2, 0x115B5 }, // Mn   [4] SIDDHAM VOWEL SIGN U..SIDDHAM VOWEL SIGN VOCALIC RR
    Interval{ 0x115B8, 0x115BB }, // Mc   [4] SIDDHAM VOWEL SIGN E..SIDDHAM VOWEL SIGN AU
    Interval{ 0x115BC, 0x115BD }, // Mn   [2] SIDDHAM SIGN CANDRABINDU..SIDDHAM SIGN ANUSVARA
    Interval{ 0x115BE, 0x115BE }, // Mc       SIDDHAM SIGN VISARGA
    Interval{ 0x115BF, 0x115C0 }, // Mn   [2] SIDDHAM SIGN VIRAMA..SIDDHAM SIGN NUKTA
    Interval{ 0x115D8, 0x115DB }, // Lo   [4] SIDDHAM LETTER THREE-CIRCLE ALTERNATE I..SIDDHAM LETTER ALTERNATE U
    Interval{ 0x115DC, 0x115DD }, // Mn   [2] SIDDHAM VOWEL SIGN ALTERNATE U..SIDDHAM VOWEL SIGN ALTERNATE UU
    Interval{ 0x11600, 0x1162F }, // Lo  [48] MODI LETTER A..MODI LETTER LLA
    Interval{ 0x11630, 0x11632 }, // Mc   [3] MODI VOWEL SIGN AA..MODI VOWEL SIGN II
    Interval{ 0x11633, 0x1163A }, // Mn   [8] MODI VOWEL SIGN U..MODI VOWEL SIGN AI
    Interval{ 0x1163B, 0x1163C }, // Mc   [2] MODI VOWEL SIGN O..MODI VOWEL SIGN AU
    Interval{ 0x1163D, 0x1163D }, // Mn       MODI SIGN ANUSVARA
    Interval{ 0x1163E, 0x1163E }, // Mc       MODI SIGN VISARGA
    Interval{ 0x1163F, 0x11640 }, // Mn   [2] MODI SIGN VIRAMA..MODI SIGN ARDHACANDRA
    Interval{ 0x11644, 0x11644 }, // Lo       MODI SIGN HUVA
    Interval{ 0x11650, 0x11659 }, // Nd  [10] MODI DIGIT ZERO..MODI DIGIT NINE
    Interval{ 0x11680, 0x116AA }, // Lo  [43] TAKRI LETTER A..TAKRI LETTER RRA
    Interval{ 0x116AB, 0x116AB }, // Mn       TAKRI SIGN ANUSVARA
    Interval{ 0x116AC, 0x116AC }, // Mc       TAKRI SIGN VISARGA
    Interval{ 0x116AD, 0x116AD }, // Mn       TAKRI VOWEL SIGN AA
    Interval{ 0x116AE, 0x116AF }, // Mc   [2] TAKRI VOWEL SIGN I..TAKRI VOWEL SIGN II
    Interval{ 0x116B0, 0x116B5 }, // Mn   [6] TAKRI VOWEL SIGN U..TAKRI VOWEL SIGN AU
    Interval{ 0x116B6, 0x116B6 }, // Mc       TAKRI SIGN VIRAMA
    Interval{ 0x116B7, 0x116B7 }, // Mn       TAKRI SIGN NUKTA
    Interval{ 0x116B8, 0x116B8 }, // Lo       TAKRI LETTER ARCHAIC KHA
    Interval{ 0x116C0, 0x116C9 }, // Nd  [10] TAKRI DIGIT ZERO..TAKRI DIGIT NINE
    Interval{ 0x11700, 0x1171A }, // Lo  [27] AHOM LETTER KA..AHOM LETTER ALTERNATE BA
    Interval{ 0x1171D, 0x1171F }, // Mn   [3] AHOM CONSONANT SIGN MEDIAL LA..AHOM CONSONANT SIGN MEDIAL LIGATING RA
    Interval{ 0x11720, 0x11721 }, // Mc   [2] AHOM VOWEL SIGN A..AHOM VOWEL SIGN AA
    Interval{ 0x11722, 0x11725 }, // Mn   [4] AHOM VOWEL SIGN I..AHOM VOWEL SIGN UU
    Interval{ 0x11726, 0x11726 }, // Mc       AHOM VOWEL SIGN E
    Interval{ 0x11727, 0x1172B }, // Mn   [5] AHOM VOWEL SIGN AW..AHOM SIGN KILLER
    Interval{ 0x11730, 0x11739 }, // Nd  [10] AHOM DIGIT ZERO..AHOM DIGIT NINE
    Interval{ 0x11740, 0x11746 }, // Lo   [7] AHOM LETTER CA..AHOM LETTER LLA
    Interval{ 0x11800, 0x1182B }, // Lo  [44] DOGRA LETTER A..DOGRA LETTER RRA
    Interval{ 0x1182C, 0x1182E }, // Mc   [3] DOGRA VOWEL SIGN AA..DOGRA VOWEL SIGN II
    Interval{ 0x1182F, 0x11837 }, // Mn   [9] DOGRA VOWEL SIGN U..DOGRA SIGN ANUSVARA
    Interval{ 0x11838, 0x11838 }, // Mc       DOGRA SIGN VISARGA
    Interval{ 0x11839, 0x1183A }, // Mn   [2] DOGRA SIGN VIRAMA..DOGRA SIGN NUKTA
    Interval{ 0x118A0, 0x118DF }, // L&  [64] WARANG CITI CAPITAL LETTER NGAA..WARANG CITI SMALL LETTER VIYO
    Interval{ 0x118E0, 0x118E9 }, // Nd  [10] WARANG CITI DIGIT ZERO..WARANG CITI DIGIT NINE
    Interval{ 0x118FF, 0x11906 }, // Lo   [8] WARANG CITI OM..DIVES AKURU LETTER E
    Interval{ 0x11909, 0x11909 }, // Lo       DIVES AKURU LETTER O
    Interval{ 0x1190C, 0x11913 }, // Lo   [8] DIVES AKURU LETTER KA..DIVES AKURU LETTER JA
    Interval{ 0x11915, 0x11916 }, // Lo   [2] DIVES AKURU LETTER NYA..DIVES AKURU LETTER TTA
    Interval{ 0x11918, 0x1192F }, // Lo  [24] DIVES AKURU LETTER DDA..DIVES AKURU LETTER ZA
    Interval{ 0x11930, 0x11935 }, // Mc   [6] DIVES AKURU VOWEL SIGN AA..DIVES AKURU VOWEL SIGN E
    Interval{ 0x11937, 0x11938 }, // Mc   [2] DIVES AKURU VOWEL SIGN AI..DIVES AKURU VOWEL SIGN O
    Interval{ 0x1193B, 0x1193C }, // Mn   [2] DIVES AKURU SIGN ANUSVARA..DIVES AKURU SIGN CANDRABINDU
    Interval{ 0x1193D, 0x1193D }, // Mc       DIVES AKURU SIGN HALANTA
    Interval{ 0x1193E, 0x1193E }, // Mn       DIVES AKURU VIRAMA
    Interval{ 0x1193F, 0x1193F }, // Lo       DIVES AKURU PREFIXED NASAL SIGN
    Interval{ 0x11940, 0x11940 }, // Mc       DIVES AKURU MEDIAL YA
    Interval{ 0x11941, 0x11941 }, // Lo       DIVES AKURU INITIAL RA
    Interval{ 0x11942, 0x11942 }, // Mc       DIVES AKURU MEDIAL RA
    Interval{ 0x11943, 0x11943 }, // Mn       DIVES AKURU SIGN NUKTA
    Interval{ 0x11950, 0x11959 }, // Nd  [10] DIVES AKURU DIGIT ZERO..DIVES AKURU DIGIT NINE
    Interval{ 0x119A0, 0x119A7 }, // Lo   [8] NANDINAGARI LETTER A..NANDINAGARI LETTER VOCALIC RR
    Interval{ 0x119AA, 0x119D0 }, // Lo  [39] NANDINAGARI LETTER E..NANDINAGARI LETTER RRA
    Interval{ 0x119D1, 0x119D3 }, // Mc   [3] NANDINAGARI VOWEL SIGN AA..NANDINAGARI VOWEL SIGN II
    Interval{ 0x119D4, 0x119D7 }, // Mn   [4] NANDINAGARI VOWEL SIGN U..NANDINAGARI VOWEL SIGN VOCALIC RR
    Interval{ 0x119DA, 0x119DB }, // Mn   [2] NANDINAGARI VOWEL SIGN E..NANDINAGARI VOWEL SIGN AI
    Interval{ 0x119DC, 0x119DF }, // Mc   [4] NANDINAGARI VOWEL SIGN O..NANDINAGARI SIGN VISARGA
    Interval{ 0x119E0, 0x119E0 }, // Mn       NANDINAGARI SIGN VIRAMA
    Interval{ 0x119E1, 0x119E1 }, // Lo       NANDINAGARI SIGN AVAGRAHA
    Interval{ 0x119E3, 0x119E3 }, // Lo       NANDINAGARI HEADSTROKE
    Interval{ 0x119E4, 0x119E4 }, // Mc       NANDINAGARI VOWEL SIGN PRISHTHAMATRA E
    Interval{ 0x11A00, 0x11A00 }, // Lo       ZANABAZAR SQUARE LETTER A
    Interval{ 0x11A01, 0x11A0A }, // Mn  [10] ZANABAZAR SQUARE VOWEL SIGN I..ZANABAZAR SQUARE VOWEL LENGTH MARK
    Interval{ 0x11A0B, 0x11A32 }, // Lo  [40] ZANABAZAR SQUARE LETTER KA..ZANABAZAR SQUARE LETTER KSSA
    Interval{ 0x11A33, 0x11A38 }, // Mn   [6] ZANABAZAR SQUARE FINAL CONSONANT MARK..ZANABAZAR SQUARE SIGN ANUSVARA
    Interval{ 0x11A39, 0x11A39 }, // Mc       ZANABAZAR SQUARE SIGN VISARGA
    Interval{ 0x11A3A, 0x11A3A }, // Lo       ZANABAZAR SQUARE CLUSTER-INITIAL LETTER RA
    Interval{ 0x11A3B, 0x11A3E }, // Mn   [4] ZANABAZAR SQUARE CLUSTER-FINAL LETTER YA..ZANABAZAR SQUARE CLUSTER-FINAL LETTER VA
    Interval{ 0x11A47, 0x11A47 }, // Mn       ZANABAZAR SQUARE SUBJOINER
    Interval{ 0x11A50, 0x11A50 }, // Lo       SOYOMBO LETTER A
    Interval{ 0x11A51, 0x11A56 }, // Mn   [6] SOYOMBO VOWEL SIGN I..SOYOMBO VOWEL SIGN OE
    Interval{ 0x11A57, 0x11