import chalk from "chalk";
import { accessSync, constants, copyFileSync, existsSync, mkdirSync } from "fs";
import { createRequire } from "module";
import path from "path";
import { pathToFileURL } from "url";
import { LOGS_FOLDER, TORRENT_CACHE_FOLDER, } from "./constants.js";
import { CrossSeedError } from "./errors.js";
const require = createRequire(import.meta.url);
const packageDotJson = require("../package.json");
export const UNPARSABLE_CONFIG_MESSAGE = `
Your config file is improperly formatted. The location of the error is above, \
but you may have to look backwards to see the root cause.
Make sure that
  - strings (words, URLs, etc) are wrapped in "quotation marks"
  - any arrays (lists of things, even one thing) are wrapped in [square brackets]
  - every entry has a comma after it, including inside arrays
`.trim();
/**
 * Returns the appDir relevant to the OS/Environment. Due to initialization of
 * the SQLiteDB during read of db.ts - will need to create appDir in this function if
 * it does not exist (ENOENT)
 *
 * @return a string representing the absolute path to cross-seed config directory
 */
export function appDir() {
    const appDir = process.env.CONFIG_DIR ||
        (process.platform === "win32"
            ? path.resolve(process.env.LOCALAPPDATA, packageDotJson.name)
            : path.resolve(process.env.HOME, `.${packageDotJson.name}`));
    try {
        accessSync(appDir, constants.R_OK | constants.W_OK);
    }
    catch (e) {
        if (e.code === "ENOENT") {
            mkdirSync(appDir, { recursive: true });
            return appDir;
        }
        const dockerMessage = process.env.DOCKER_ENV === "true"
            ? ` Use chown to set the owner to ${process.getuid()}:${process.getgid()}`
            : "";
        throw new CrossSeedError(`cross-seed does not have R/W permissions on your config directory.${dockerMessage}`);
    }
    return appDir;
}
export function createAppDirHierarchy() {
    const appDirPath = appDir();
    mkdirSync(path.join(appDirPath, TORRENT_CACHE_FOLDER), { recursive: true });
    mkdirSync(path.join(appDirPath, LOGS_FOLDER), { recursive: true });
}
export function generateConfig() {
    createAppDirHierarchy();
    const dest = path.join(appDir(), "config.js");
    const templatePath = path.join("./config.template.cjs");
    if (existsSync(dest)) {
        console.log("Configuration file already exists.");
        return;
    }
    copyFileSync(new URL(templatePath, import.meta.url), dest);
    console.log("Configuration file created at", chalk.yellow.bold(dest));
}
export async function getFileConfig() {
    if (process.env.DOCKER_ENV === "true") {
        generateConfig();
    }
    const configPath = path.join(appDir(), "config.js");
    try {
        return (await import(pathToFileURL(configPath).toString())).default;
    }
    catch (e) {
        if (e.code === "ERR_MODULE_NOT_FOUND") {
            return {};
        }
        else if (e instanceof SyntaxError) {
            const location = e.stack.split("\n").slice(0, 3).join("\n");
            throw new CrossSeedError(`\n${chalk.red(location)}\n\n${UNPARSABLE_CONFIG_MESSAGE}`);
        }
        else {
            throw e;
        }
    }
}
//# sourceMappingURL=configuration.js.map