import ms from "ms";
import { Action } from "./constants.js";
import { cleanupDB, db } from "./db.js";
import { exitOnCrossSeedErrors } from "./errors.js";
import { injectSavedTorrents } from "./inject.js";
import { Label, logger } from "./logger.js";
import { bulkSearch, scanRssFeeds } from "./pipeline.js";
import { getRuntimeConfig } from "./runtimeConfig.js";
import { updateCaps } from "./torznab.js";
import { humanReadableDate, Mutex, withMutex } from "./utils.js";
export var JobName;
(function (JobName) {
    JobName["RSS"] = "rss";
    JobName["SEARCH"] = "search";
    JobName["UPDATE_INDEXER_CAPS"] = "updateIndexerCaps";
    JobName["INJECT"] = "inject";
    JobName["CLEANUP"] = "cleanup";
})(JobName || (JobName = {}));
const jobs = [];
class Job {
    name;
    cadence;
    exec;
    isActive;
    runAheadOfSchedule;
    delayNextRun;
    configOverride;
    constructor(name, cadence, exec) {
        this.name = name;
        this.cadence = cadence;
        this.exec = exec;
        this.isActive = false;
        this.runAheadOfSchedule = false;
        this.delayNextRun = false;
        this.configOverride = {};
    }
    async run() {
        if (this.isActive)
            return false;
        this.isActive = true;
        try {
            logger.info({
                label: Label.SCHEDULER,
                message: `starting job: ${this.name}`,
            });
            if (this.runAheadOfSchedule && this.name === JobName.SEARCH) {
                await bulkSearch({ configOverride: this.configOverride });
            }
            else {
                await this.exec();
            }
        }
        finally {
            this.isActive = false;
            this.runAheadOfSchedule = false;
            this.configOverride = {};
        }
        return true;
    }
}
function createJobs() {
    const { action, rssCadence, searchCadence, torznab } = getRuntimeConfig();
    if (rssCadence) {
        jobs.push(new Job(JobName.RSS, rssCadence, scanRssFeeds));
    }
    if (searchCadence) {
        jobs.push(new Job(JobName.SEARCH, searchCadence, bulkSearch));
    }
    if (torznab.length > 0) {
        jobs.push(new Job(JobName.UPDATE_INDEXER_CAPS, ms("1 day"), updateCaps));
    }
    if (action === Action.INJECT) {
        jobs.push(new Job(JobName.INJECT, ms("1 hour"), injectSavedTorrents));
    }
    jobs.push(new Job(JobName.CLEANUP, ms("1 day"), cleanupDB));
}
export function getJobs() {
    return jobs;
}
function logNextRun(name, cadence, lastRun) {
    const now = Date.now();
    const eligibilityTs = lastRun ? lastRun + cadence : now;
    const lastRunStr = !lastRun
        ? "never"
        : now >= lastRun
            ? `${ms(now - lastRun)} ago`
            : `at ${humanReadableDate(lastRun - cadence)}`;
    const nextRunStr = now >= eligibilityTs ? "now" : `in ${ms(eligibilityTs - now)}`;
    logger.info({
        label: Label.SCHEDULER,
        message: `${name}: last run ${lastRunStr}, next run ${nextRunStr}`,
    });
}
export async function getJobLastRun(name) {
    return (await db("job_log").select("last_run").where({ name }).first())
        ?.last_run;
}
export async function checkJobs(options = { isFirstRun: false, useQueue: false }) {
    return withMutex(Mutex.CHECK_JOBS, { useQueue: options.useQueue }, async () => {
        const now = Date.now();
        for (const job of jobs) {
            const lastRun = await getJobLastRun(job.name);
            const eligibilityTs = lastRun ? lastRun + job.cadence : now;
            if (options.isFirstRun) {
                logNextRun(job.name, job.cadence, lastRun);
            }
            if (!job.runAheadOfSchedule) {
                if (jobs.find((j) => j.name === JobName.RSS)?.isActive) {
                    continue;
                }
                if (job.name === JobName.CLEANUP) {
                    if (jobs.some((j) => j.isActive))
                        continue;
                }
            }
            if (job.runAheadOfSchedule || now >= eligibilityTs) {
                job.run()
                    .then(async (didRun) => {
                    if (!didRun)
                        return; // upon success, update the log
                    const toDelay = job.delayNextRun;
                    job.delayNextRun = false;
                    const last_run = toDelay ? now + job.cadence : now;
                    await db("job_log")
                        .insert({ name: job.name, last_run })
                        .onConflict("name")
                        .merge();
                    const cadence = toDelay
                        ? job.cadence * 2
                        : job.cadence;
                    logNextRun(job.name, cadence, now);
                })
                    .catch(exitOnCrossSeedErrors)
                    .catch((e) => void logger.error(e));
            }
        }
    });
}
export async function jobsLoop() {
    createJobs();
    setInterval(checkJobs, ms("1 minute"));
    await checkJobs({ isFirstRun: true, useQueue: false });
    // jobs take too long to run to completion so let process.exit take care of stopping
    return new Promise(() => { });
}
//# sourceMappingURL=jobs.js.map