import { readdirSync } from "fs";
import ms from "ms";
import path from "path";
import { ABS_WIN_PATH_REGEX, InjectionResult, TORRENT_CATEGORY_SUFFIX, TORRENT_TAG, } from "../constants.js";
import { CrossSeedError } from "../errors.js";
import { Label, logger } from "../logger.js";
import { resultOf, resultOfErr } from "../Result.js";
import { getRuntimeConfig } from "../runtimeConfig.js";
import { loadTorrentDirLight } from "../torrent.js";
import { getMaxRemainingBytes, getResumeStopTime, resumeErrSleepTime, resumeSleepTime, shouldRecheck, validateSavePaths, } from "./TorrentClient.js";
import { extractCredentialsFromUrl, extractInt, getLogString, humanReadableSize, sanitizeInfoHash, wait, } from "../utils.js";
const X_WWW_FORM_URLENCODED = {
    "Content-Type": "application/x-www-form-urlencoded",
};
export default class QBittorrent {
    cookie;
    url;
    version;
    versionMajor;
    type = Label.QBITTORRENT;
    constructor() {
        const { qbittorrentUrl } = getRuntimeConfig();
        this.url = extractCredentialsFromUrl(qbittorrentUrl, "/api/v2").unwrapOrThrow(new CrossSeedError("qBittorrent url must be percent-encoded"));
    }
    async login() {
        let response;
        const { href, username, password } = this.url;
        try {
            response = await fetch(`${href}/auth/login`, {
                method: "POST",
                body: new URLSearchParams({ username, password }),
            });
        }
        catch (e) {
            throw new CrossSeedError(`qBittorrent login failed: ${e.message}`);
        }
        if (response.status !== 200) {
            throw new CrossSeedError(`qBittorrent login failed with code ${response.status}`);
        }
        this.cookie = response.headers.getSetCookie()[0];
        if (!this.cookie) {
            throw new CrossSeedError(`qBittorrent login failed: Invalid username or password`);
        }
        const version = await this.request("/app/version", "", X_WWW_FORM_URLENCODED);
        if (!version) {
            throw new CrossSeedError(`qBittorrent login failed: Unable to retrieve version`);
        }
        this.version = version;
        this.versionMajor = extractInt(this.version);
        logger.info({
            label: Label.QBITTORRENT,
            message: `Logged in to qBittorrent ${this.version}`,
        });
    }
    async validateConfig() {
        const { torrentDir } = getRuntimeConfig();
        await this.login();
        await this.createTag();
        if (!torrentDir)
            return;
        if (!readdirSync(torrentDir).some((f) => f.endsWith(".fastresume"))) {
            throw new CrossSeedError("Invalid torrentDir, if no torrents are in client set to null for now: https://www.cross-seed.org/docs/basics/options#torrentdir");
        }
        const searcheesRes = loadTorrentDirLight(torrentDir);
        const infoHashPathMap = await this.getAllDownloadDirs({
            metas: [], // Don't need to account for subfolder layout
            onlyCompleted: false,
            v1HashOnly: true,
        });
        await validateSavePaths(infoHashPathMap, await searcheesRes);
    }
    async request(path, body, headers = {}, retries = 3) {
        const bodyStr = body instanceof FormData
            ? JSON.stringify(Object.fromEntries(body))
            : JSON.stringify(body).replace(/(?:hashes=)([a-z0-9]{40})/i, (match, hash) => match.replace(hash, sanitizeInfoHash(hash)));
        logger.verbose({
            label: Label.QBITTORRENT,
            message: `Making request (${retries}) to ${path} with body ${bodyStr}`,
        });
        let response;
        try {
            response = await fetch(`${this.url.href}${path}`, {
                method: "post",
                headers: { Cookie: this.cookie, ...headers },
                body,
            });
            if (response.status === 403 && retries > 0) {
                logger.verbose({
                    label: Label.QBITTORRENT,
                    message: "Received 403 from API. Logging in again and retrying",
                });
                await this.login();
                return this.request(path, body, headers, retries - 1);
            }
        }
        catch (e) {
            if (retries > 0) {
                logger.verbose({
                    label: Label.QBITTORRENT,
                    message: `Request failed, ${retries} retries remaining: ${e.message}`,
                });
                return this.request(path, body, headers, retries - 1);
            }
            logger.verbose({
                label: Label.QBITTORRENT,
                message: `Request failed after ${retries} retries: ${e.message}`,
            });
        }
        return response?.text();
    }
    getLayoutForNewTorrent(searchee, searcheeInfo, path) {
        return path
            ? "Original"
            : this.isSubfolderContentLayout(searchee, searcheeInfo)
                ? "Subfolder"
                : "Original";
    }
    async getCategoryForNewTorrent(category, savePath, autoTMM) {
        const { duplicateCategories, linkCategory } = getRuntimeConfig();
        if (!duplicateCategories) {
            return category;
        }
        if (!category.length || category === linkCategory) {
            return category; // Use tags for category duplication if linking
        }
        const dupeCategory = category.endsWith(TORRENT_CATEGORY_SUFFIX)
            ? category
            : `${category}${TORRENT_CATEGORY_SUFFIX}`;
        if (!autoTMM)
            return dupeCategory;
        // savePath is guaranteed to be the base category's save path due to autoTMM
        const categories = await this.getAllCategories();
        const newRes = categories.find((c) => c.name === dupeCategory);
        if (!newRes) {
            await this.createCategory(dupeCategory, savePath);
        }
        else if (newRes.savePath !== savePath) {
            await this.editCategory(dupeCategory, savePath);
        }
        return dupeCategory;
    }
    getTagsForNewTorrent(searcheeInfo, path) {
        const { duplicateCategories, linkCategory } = getRuntimeConfig();
        if (!duplicateCategories || !searcheeInfo || !path) {
            return TORRENT_TAG; // Require path to duplicate category using tags
        }
        const searcheeCategory = searcheeInfo.category;
        if (!searcheeCategory.length || searcheeCategory === linkCategory) {
            return TORRENT_TAG;
        }
        if (searcheeCategory.endsWith(TORRENT_CATEGORY_SUFFIX)) {
            return `${TORRENT_TAG},${searcheeCategory}`;
        }
        return `${TORRENT_TAG},${searcheeCategory}${TORRENT_CATEGORY_SUFFIX}`;
    }
    async createTag() {
        await this.request("/torrents/createTags", `tags=${TORRENT_TAG}`, X_WWW_FORM_URLENCODED);
    }
    async createCategory(category, savePath) {
        await this.request("/torrents/createCategory", `category=${category}&savePath=${savePath}`, X_WWW_FORM_URLENCODED);
    }
    async editCategory(category, savePath) {
        await this.request("/torrents/editCategory", `category=${category}&savePath=${savePath}`, X_WWW_FORM_URLENCODED);
    }
    async getAllCategories() {
        const responseText = await this.request("/torrents/categories", "");
        return responseText ? Object.values(JSON.parse(responseText)) : [];
    }
    async addTorrent(formData) {
        await this.request("/torrents/add", formData);
    }
    async recheckTorrent(infoHash) {
        // Pause first as it may resume after recheck automatically
        await this.request(`/torrents/${this.versionMajor >= 5 ? "stop" : "pause"}`, `hashes=${infoHash}`, X_WWW_FORM_URLENCODED);
        await this.request("/torrents/recheck", `hashes=${infoHash}`, X_WWW_FORM_URLENCODED);
    }
    /*
     * @param searchee the Searchee we are generating off (in client)
     * @return either a string containing the path or a error mesage
     */
    async getDownloadDir(meta, options) {
        try {
            const torrentInfo = await this.getTorrentInfo(meta.infoHash);
            if (!torrentInfo) {
                return resultOfErr("NOT_FOUND");
            }
            if (options.onlyCompleted &&
                !this.isTorrentInfoComplete(torrentInfo)) {
                return resultOfErr("TORRENT_NOT_COMPLETE");
            }
            return resultOf(this.getCorrectSavePath(meta, torrentInfo));
        }
        catch (e) {
            logger.debug(e);
            if (e.message.includes("retrieve")) {
                return resultOfErr("NOT_FOUND");
            }
            return resultOfErr("UNKNOWN_ERROR");
        }
    }
    /*
     * @param metas the Searchees we are generating off (in client)
     * @return a map of infohash to path
     */
    async getAllDownloadDirs(options) {
        const torrents = await this.getAllTorrentInfo();
        const torrentSavePaths = new Map();
        const infoHashMetaMap = options.metas.reduce((acc, meta) => {
            acc.set(meta.infoHash, meta);
            return acc;
        }, new Map());
        for (const torrent of torrents) {
            if (options.onlyCompleted && !this.isTorrentInfoComplete(torrent)) {
                continue;
            }
            const meta = infoHashMetaMap.get(torrent.hash) ||
                (torrent.infohash_v2 &&
                    infoHashMetaMap.get(torrent.infohash_v2)) ||
                (torrent.infohash_v1 &&
                    infoHashMetaMap.get(torrent.infohash_v1)) ||
                undefined;
            const savePath = meta
                ? this.getCorrectSavePath(meta, torrent)
                : torrent.save_path;
            if (torrent.infohash_v1?.length) {
                torrentSavePaths.set(torrent.infohash_v1, savePath);
            }
            if (options.v1HashOnly)
                continue;
            torrentSavePaths.set(torrent.hash, savePath);
            if (torrent.infohash_v2?.length) {
                torrentSavePaths.set(torrent.infohash_v2, savePath);
            }
        }
        return torrentSavePaths;
    }
    /*
     * @param searchee the Searchee we are generating off (in client)
     * @param torrentInfo the torrent info from the searchee
     * @return string absolute location from client with content layout considered
     */
    getCorrectSavePath(data, torrentInfo) {
        const subfolderContentLayout = this.isSubfolderContentLayout(data, torrentInfo);
        if (subfolderContentLayout) {
            return ABS_WIN_PATH_REGEX.test(torrentInfo.content_path)
                ? path.win32.dirname(torrentInfo.content_path)
                : path.posix.dirname(torrentInfo.content_path);
        }
        return torrentInfo.save_path;
    }
    /*
     * @return array of all torrents in the client
     */
    async getAllTorrentInfo() {
        const responseText = await this.request("/torrents/info", "");
        if (!responseText) {
            return [];
        }
        return JSON.parse(responseText);
    }
    /*
     * @param hash the hash of the torrent
     * @return the torrent if it exists
     */
    async getTorrentInfo(hash, retries = 0) {
        if (!hash)
            return undefined;
        for (let i = 0; i <= retries; i++) {
            const responseText = await this.request("/torrents/info", `hashes=${hash}`, X_WWW_FORM_URLENCODED);
            if (responseText) {
                const torrents = JSON.parse(responseText);
                if (torrents.length > 0) {
                    return torrents[0];
                }
            }
            const torrents = await this.getAllTorrentInfo();
            const torrentInfo = torrents.find((torrent) => hash === torrent.hash ||
                hash === torrent.infohash_v1 ||
                hash === torrent.infohash_v2);
            if (torrentInfo) {
                return torrentInfo;
            }
            if (i < retries) {
                await wait(ms("1 second") * 2 ** i);
            }
        }
        return undefined;
    }
    /**
     * @return array of all torrents in the client
     */
    async getAllTorrents() {
        const torrents = await this.getAllTorrentInfo();
        return torrents.map((torrent) => ({
            infoHash: torrent.hash,
            category: torrent.category,
            tags: torrent.tags.length ? torrent.tags.split(",") : [],
            trackers: torrent.tracker.length ? [[torrent.tracker]] : undefined,
        }));
    }
    /**
     * @param infoHash the infohash of the torrent
     * @returns whether the torrent is complete
     */
    async isTorrentComplete(infoHash) {
        const torrentInfo = await this.getTorrentInfo(infoHash);
        if (!torrentInfo) {
            return resultOfErr("NOT_FOUND");
        }
        return resultOf(this.isTorrentInfoComplete(torrentInfo));
    }
    isTorrentInfoComplete(torrentInfo) {
        return [
            "uploading",
            "pausedUP",
            "stoppedUP",
            "queuedUP",
            "stalledUP",
            "checkingUP",
            "forcedUP",
        ].includes(torrentInfo.state);
    }
    isSubfolderContentLayout(data, dataInfo) {
        if (data.files.length > 1)
            return false;
        if (path.dirname(data.files[0].path) !== ".")
            return false;
        let dirname = path.posix.dirname;
        let resolve = path.posix.resolve;
        if (ABS_WIN_PATH_REGEX.test(dataInfo.content_path)) {
            dirname = path.win32.dirname;
            resolve = path.win32.resolve;
        }
        return (resolve(dirname(dataInfo.content_path)) !==
            resolve(dataInfo.save_path));
    }
    async resumeInjection(infoHash, decision, options) {
        let sleepTime = resumeSleepTime;
        const maxRemainingBytes = getMaxRemainingBytes(decision);
        const stopTime = getResumeStopTime();
        let stop = false;
        while (Date.now() < stopTime) {
            if (options.checkOnce) {
                if (stop)
                    return;
                stop = true;
            }
            await wait(sleepTime);
            const torrentInfo = await this.getTorrentInfo(infoHash);
            if (!torrentInfo) {
                sleepTime = resumeErrSleepTime; // Dropping connections or restart
                continue;
            }
            if (["checkingDL", "checkingUP"].includes(torrentInfo.state)) {
                continue;
            }
            const torrentLog = `${torrentInfo.name} [${sanitizeInfoHash(infoHash)}]`;
            if (!["pausedDL", "stoppedDL", "pausedUP", "stoppedUP"].includes(torrentInfo.state)) {
                logger.warn({
                    label: Label.QBITTORRENT,
                    message: `Will not resume ${torrentLog}: state is ${torrentInfo.state}`,
                });
                return;
            }
            if (torrentInfo.amount_left > maxRemainingBytes) {
                logger.warn({
                    label: Label.QBITTORRENT,
                    message: `Will not resume ${torrentLog}: ${humanReadableSize(torrentInfo.amount_left, { binary: true })} remaining > ${humanReadableSize(maxRemainingBytes, { binary: true })}`,
                });
                return;
            }
            logger.info({
                label: Label.QBITTORRENT,
                message: `Resuming ${torrentLog}: ${humanReadableSize(torrentInfo.amount_left, { binary: true })} remaining`,
            });
            await this.request(`/torrents/${this.versionMajor >= 5 ? "start" : "resume"}`, `hashes=${infoHash}`, X_WWW_FORM_URLENCODED);
            return;
        }
        logger.warn({
            label: Label.QBITTORRENT,
            message: `Will not resume torrent ${infoHash}: timeout`,
        });
    }
    async inject(newTorrent, searchee, decision, path) {
        const { linkCategory } = getRuntimeConfig();
        try {
            if (await this.getTorrentInfo(newTorrent.infoHash)) {
                return InjectionResult.ALREADY_EXISTS;
            }
            const searcheeInfo = await this.getTorrentInfo(searchee.infoHash);
            if (!searcheeInfo) {
                if (!path) {
                    // This is never possible, being made explicit here
                    throw new Error(`Searchee torrent may have been deleted: ${getLogString(searchee)}`);
                }
                else if (searchee.infoHash) {
                    logger.warn({
                        label: Label.QBITTORRENT,
                        message: `Searchee torrent may have been deleted, tagging may not meet expectations: ${getLogString(searchee)}`,
                    });
                }
            }
            const { savePath, isComplete, autoTMM, category } = path
                ? {
                    savePath: path,
                    isComplete: true,
                    autoTMM: false,
                    category: linkCategory,
                }
                : {
                    savePath: searcheeInfo.save_path,
                    isComplete: this.isTorrentInfoComplete(searcheeInfo),
                    autoTMM: searcheeInfo.auto_tmm,
                    category: searcheeInfo.category,
                };
            if (!isComplete)
                return InjectionResult.TORRENT_NOT_COMPLETE;
            const filename = `${newTorrent.getFileSystemSafeName()}.${TORRENT_TAG}.torrent`;
            const buffer = new Blob([newTorrent.encode()], {
                type: "application/x-bittorrent",
            });
            const toRecheck = shouldRecheck(searchee, decision);
            // ---------------------- Building form data ----------------------
            const formData = new FormData();
            formData.append("torrents", buffer, filename);
            if (!autoTMM) {
                formData.append("downloadPath", savePath);
                formData.append("savepath", savePath);
            }
            formData.append("autoTMM", autoTMM.toString());
            if (category?.length) {
                formData.append("category", await this.getCategoryForNewTorrent(category, savePath, autoTMM));
            }
            formData.append("tags", this.getTagsForNewTorrent(searcheeInfo, path));
            formData.append("contentLayout", this.getLayoutForNewTorrent(searchee, searcheeInfo, path));
            formData.append("skip_checking", (!toRecheck).toString());
            formData.append(this.versionMajor >= 5 ? "stopped" : "paused", toRecheck.toString());
            // for some reason the parser parses the last kv pair incorrectly
            // it concats the value and the sentinel
            formData.append("foo", "bar");
            try {
                await this.addTorrent(formData);
            }
            catch (e) {
                logger.error({
                    label: Label.QBITTORRENT,
                    message: `Failed to add torrent (polling client to confirm): ${e.message}`,
                });
                logger.debug(e);
            }
            const newInfo = await this.getTorrentInfo(newTorrent.infoHash, 5);
            if (!newInfo) {
                throw new Error(`Failed to retrieve torrent after adding`);
            }
            if (toRecheck) {
                await this.recheckTorrent(newInfo.hash);
                this.resumeInjection(newInfo.hash, decision, {
                    checkOnce: false,
                });
            }
            return InjectionResult.SUCCESS;
        }
        catch (e) {
            logger.error({
                label: Label.QBITTORRENT,
                message: `Injection failed for ${getLogString(newTorrent)}: ${e.message}`,
            });
            logger.debug(e);
            return InjectionResult.FAILURE;
        }
    }
}
//# sourceMappingURL=QBittorrent.js.map