import { readdirSync } from "fs";
import { stat, unlink, writeFile } from "fs/promises";
import { dirname, join, resolve, sep } from "path";
import { inspect } from "util";
import xmlrpc from "xmlrpc";
import { InjectionResult, TORRENT_TAG, } from "../constants.js";
import { CrossSeedError } from "../errors.js";
import { Label, logger } from "../logger.js";
import { Metafile } from "../parseTorrent.js";
import { resultOf, resultOfErr } from "../Result.js";
import { getRuntimeConfig } from "../runtimeConfig.js";
import { extractCredentialsFromUrl, humanReadableSize, isTruthy, sanitizeInfoHash, wait, } from "../utils.js";
import { getMaxRemainingBytes, getResumeStopTime, resumeErrSleepTime, resumeSleepTime, shouldRecheck, validateSavePaths, } from "./TorrentClient.js";
import { loadTorrentDirLight } from "../torrent.js";
const COULD_NOT_FIND_INFO_HASH = "Could not find info-hash.";
async function createLibTorrentResumeTree(meta, basePath) {
    async function getFileResumeData(file) {
        const filePathWithoutFirstSegment = file.path
            .split(sep)
            .slice(1)
            .join(sep);
        const resolvedFilePath = resolve(basePath, filePathWithoutFirstSegment);
        const fileStat = await stat(resolvedFilePath).catch(() => ({ isFile: () => false }));
        if (!fileStat.isFile() || fileStat.size !== file.length) {
            return null;
        }
        return {
            completed: Math.ceil(file.length / meta.pieceLength),
            mtime: Math.trunc(fileStat.mtimeMs / 1000),
            priority: 1,
        };
    }
    const fileResumes = await Promise.all(meta.files.map(getFileResumeData));
    return {
        bitfield: Math.ceil(meta.length / meta.pieceLength),
        files: fileResumes.filter(isTruthy),
    };
}
async function saveWithLibTorrentResume(meta, savePath, basePath) {
    const rawWithLibtorrentResume = {
        ...meta.raw,
        libtorrent_resume: await createLibTorrentResumeTree(meta, basePath),
    };
    await writeFile(savePath, new Metafile(rawWithLibtorrentResume).encode());
}
export default class RTorrent {
    client;
    type = Label.RTORRENT;
    constructor() {
        const { rtorrentRpcUrl } = getRuntimeConfig();
        const { href, username, password } = extractCredentialsFromUrl(rtorrentRpcUrl).unwrapOrThrow(new CrossSeedError("rTorrent url must be percent-encoded"));
        const clientCreator = new URL(href).protocol === "https:"
            ? xmlrpc.createSecureClient
            : xmlrpc.createClient;
        const shouldUseAuth = Boolean(username && password);
        this.client = clientCreator({
            url: href,
            basic_auth: shouldUseAuth
                ? { user: username, pass: password }
                : undefined,
        });
    }
    async methodCallP(method, args) {
        logger.verbose({
            label: Label.RTORRENT,
            message: `Calling method ${method} with params ${inspect(args, {
                depth: null,
                compact: true,
            })}`,
        });
        return new Promise((resolve, reject) => {
            this.client.methodCall(method, args, (err, data) => {
                if (err)
                    return reject(err);
                return resolve(data);
            });
        });
    }
    async checkForInfoHashInClient(infoHash) {
        const downloadList = await this.methodCallP("download_list", []);
        return downloadList.includes(infoHash.toUpperCase());
    }
    async checkOriginalTorrent(infoHash, onlyCompleted) {
        const hash = infoHash.toUpperCase();
        let response;
        try {
            response = await this.methodCallP("system.multicall", [
                [
                    {
                        methodName: "d.name",
                        params: [hash],
                    },
                    {
                        methodName: "d.directory",
                        params: [hash],
                    },
                    {
                        methodName: "d.left_bytes",
                        params: [hash],
                    },
                    {
                        methodName: "d.hashing",
                        params: [hash],
                    },
                    {
                        methodName: "d.complete",
                        params: [hash],
                    },
                    {
                        methodName: "d.is_multi_file",
                        params: [hash],
                    },
                    {
                        methodName: "d.is_active",
                        params: [hash],
                    },
                ],
            ]);
        }
        catch (e) {
            logger.debug(e);
            return resultOfErr("FAILURE");
        }
        function isFault(response) {
            return "faultString" in response[0];
        }
        try {
            if (isFault(response)) {
                if (response[0].faultString === COULD_NOT_FIND_INFO_HASH) {
                    return resultOfErr("NOT_FOUND");
                }
                else {
                    throw new Error("Unknown rTorrent fault while checking original torrent");
                }
            }
            const [[name], [directoryBase], [bytesLeft], [hashing], [isCompleteStr], [isMultiFileStr], [isActive],] = response;
            const isComplete = Boolean(Number(isCompleteStr));
            if (onlyCompleted && !isComplete) {
                return resultOfErr("TORRENT_NOT_COMPLETE");
            }
            return resultOf({
                name,
                directoryBase,
                bytesLeft,
                hashing,
                isMultiFile: Boolean(Number(isMultiFileStr)),
                isActive: Boolean(Number(isActive)),
            });
        }
        catch (e) {
            logger.error(e);
            logger.debug("Failure caused by server response below:");
            logger.debug(inspect(response));
            return resultOfErr("FAILURE");
        }
    }
    async getDownloadLocation(meta, searchee, path) {
        if (path) {
            // resolve to absolute because we send the path to rTorrent
            const basePath = resolve(path, meta.name);
            const directoryBase = meta.isSingleFileTorrent ? path : basePath;
            return resultOf({ downloadDir: path, basePath, directoryBase });
        }
        else {
            const result = await this.checkOriginalTorrent(searchee.infoHash, true);
            return result.mapOk(({ directoryBase }) => ({
                directoryBase,
                downloadDir: meta.isSingleFileTorrent
                    ? directoryBase
                    : dirname(directoryBase),
                basePath: meta.isSingleFileTorrent
                    ? join(directoryBase, searchee.name)
                    : directoryBase,
            }));
        }
    }
    async validateConfig() {
        const { rtorrentRpcUrl, torrentDir } = getRuntimeConfig();
        try {
            await this.methodCallP("download_list", []);
        }
        catch (e) {
            logger.debug(e);
            throw new CrossSeedError(`Failed to reach rTorrent at ${rtorrentRpcUrl}`);
        }
        if (!torrentDir)
            return;
        if (!readdirSync(torrentDir).some((f) => f.endsWith("_resume"))) {
            throw new CrossSeedError("Invalid torrentDir, if no torrents are in client set to null for now: https://www.cross-seed.org/docs/basics/options#torrentdir");
        }
        const searcheesRes = loadTorrentDirLight(torrentDir);
        const allTorrents = await this.getAllTorrents();
        const infoHashPathMap = await this.getAllDownloadDirs({
            metas: allTorrents.map((torrent) => torrent.infoHash),
            onlyCompleted: false,
        });
        await validateSavePaths(infoHashPathMap, await searcheesRes);
    }
    async getDownloadDir(meta, options) {
        const result = await this.checkOriginalTorrent(meta.infoHash, options.onlyCompleted);
        return result
            .mapOk(({ directoryBase, isMultiFile }) => {
            return isMultiFile ? dirname(directoryBase) : directoryBase;
        })
            .mapErr((error) => (error === "FAILURE" ? "UNKNOWN_ERROR" : error));
    }
    async getAllDownloadDirs(options) {
        if (options.metas.length === 0)
            return new Map();
        const infoHashes = typeof options.metas[0] === "string"
            ? options.metas
            : options.metas.map((meta) => meta.infoHash);
        let response;
        try {
            response = await this.methodCallP("system.multicall", [
                infoHashes
                    .map((hash) => [
                    {
                        methodName: "d.directory",
                        params: [hash],
                    },
                    {
                        methodName: "d.is_multi_file",
                        params: [hash],
                    },
                ])
                    .flat(),
            ]);
        }
        catch (e) {
            logger.error({
                Label: Label.RTORRENT,
                message: "Failed to get download directories for all torrents",
            });
            logger.debug(e);
            return new Map();
        }
        function isFault(response) {
            return "faultString" in response[0];
        }
        try {
            if (isFault(response)) {
                logger.error({
                    Label: Label.RTORRENT,
                    message: "Fault while getting download directories for all torrents",
                });
                logger.debug(inspect(response));
                return new Map();
            }
            return new Map(infoHashes.map((hash, index) => {
                const directory = response[index * 2][0];
                const isMultiFile = Boolean(Number(response[index * 2 + 1][0]));
                return [hash, isMultiFile ? dirname(directory) : directory];
            }));
        }
        catch (e) {
            logger.error({
                Label: Label.RTORRENT,
                message: "Error parsing response for all torrents",
            });
            logger.debug(e);
            return new Map();
        }
    }
    async isTorrentComplete(infoHash) {
        try {
            const response = await this.methodCallP("d.complete", [
                infoHash,
            ]);
            if (response.length === 0) {
                return resultOfErr("NOT_FOUND");
            }
            return resultOf(response[0] === "1");
        }
        catch (e) {
            return resultOfErr("NOT_FOUND");
        }
    }
    async getAllTorrents() {
        const infoHashes = await this.methodCallP("download_list", []);
        let response;
        try {
            response = await this.methodCallP("system.multicall", [
                infoHashes.map((hash) => {
                    return {
                        methodName: "d.custom1",
                        params: [hash],
                    };
                }),
            ]);
        }
        catch (e) {
            logger.error({
                Label: Label.RTORRENT,
                message: "Failed to get torrent info for all torrents",
            });
            logger.debug(e);
            return [];
        }
        function isFault(response) {
            return "faultString" in response[0];
        }
        if (isFault(response)) {
            logger.error({
                Label: Label.RTORRENT,
                message: "Fault while getting torrent info for all torrents",
            });
            logger.debug(inspect(response));
            return [];
        }
        try {
            // response: [ [tag1], [tag2], ... ], assuming infoHash order is preserved
            return infoHashes.map((hash, index) => ({
                infoHash: hash.toLowerCase(),
                category: "",
                tags: response[index].length !== 1
                    ? response[index]
                    : response[index][0].length
                        ? decodeURIComponent(response[index][0])
                            .split(",")
                            .map((tag) => tag.trim())
                        : [],
            }));
        }
        catch (e) {
            logger.error({
                Label: Label.RTORRENT,
                message: "Error parsing response for all torrents",
            });
            logger.debug(e);
            return [];
        }
    }
    async recheckTorrent(infoHash) {
        // Pause first as it may resume after recheck automatically
        await this.methodCallP("d.pause", [infoHash]);
        await this.methodCallP("d.check_hash", [infoHash]);
    }
    async resumeInjection(infoHash, decision, options) {
        let sleepTime = resumeSleepTime;
        const maxRemainingBytes = getMaxRemainingBytes(decision);
        const stopTime = getResumeStopTime();
        let stop = false;
        while (Date.now() < stopTime) {
            if (options.checkOnce) {
                if (stop)
                    return;
                stop = true;
            }
            await wait(sleepTime);
            const torrentInfoRes = await this.checkOriginalTorrent(infoHash, false);
            if (torrentInfoRes.isErr()) {
                sleepTime = resumeErrSleepTime; // Dropping connections or restart
                continue;
            }
            const torrentInfo = torrentInfoRes.unwrap();
            if (torrentInfo.hashing) {
                continue;
            }
            const torrentLog = `${torrentInfo.name} [${sanitizeInfoHash(infoHash)}]`;
            if (torrentInfo.isActive) {
                logger.warn({
                    label: Label.RTORRENT,
                    message: `Will not resume torrent ${torrentLog}: active`,
                });
                return;
            }
            if (torrentInfo.bytesLeft > maxRemainingBytes) {
                logger.warn({
                    label: Label.RTORRENT,
                    message: `Will not resume torrent ${torrentLog}: ${humanReadableSize(torrentInfo.bytesLeft, { binary: true })} remaining > ${humanReadableSize(maxRemainingBytes, { binary: true })}`,
                });
                return;
            }
            logger.info({
                label: Label.RTORRENT,
                message: `Resuming torrent ${torrentLog}`,
            });
            await this.methodCallP("d.resume", [infoHash]);
        }
        logger.warn({
            label: Label.RTORRENT,
            message: `Will not resume torrent ${infoHash}: timeout`,
        });
    }
    async inject(meta, searchee, decision, path) {
        const { outputDir } = getRuntimeConfig();
        if (await this.checkForInfoHashInClient(meta.infoHash)) {
            return InjectionResult.ALREADY_EXISTS;
        }
        const result = await this.getDownloadLocation(meta, searchee, path);
        if (result.isErr()) {
            switch (result.unwrapErr()) {
                case "NOT_FOUND":
                    return InjectionResult.FAILURE;
                case "TORRENT_NOT_COMPLETE":
                    return InjectionResult.TORRENT_NOT_COMPLETE;
                case "FAILURE":
                    return InjectionResult.FAILURE;
            }
        }
        const { directoryBase, basePath } = result.unwrap();
        const torrentFilePath = resolve(outputDir, `${meta.name}.tmp.${Date.now()}.torrent`);
        await saveWithLibTorrentResume(meta, torrentFilePath, basePath);
        const toRecheck = shouldRecheck(searchee, decision);
        const loadType = toRecheck ? "load.normal" : "load.start";
        const retries = 5;
        for (let i = 0; i < retries; i++) {
            try {
                await this.methodCallP(loadType, [
                    "",
                    torrentFilePath,
                    `d.directory_base.set="${directoryBase}"`,
                    `d.custom1.set="${TORRENT_TAG}"`,
                    `d.custom.set=addtime,${Math.round(Date.now() / 1000)}`,
                    toRecheck
                        ? `d.check_hash=${meta.infoHash.toUpperCase()}`
                        : null,
                ].filter((e) => e !== null));
                if (toRecheck) {
                    this.resumeInjection(meta.infoHash, decision, {
                        checkOnce: false,
                    });
                }
                break;
            }
            catch (e) {
                logger.verbose({
                    label: Label.RTORRENT,
                    message: `Failed to inject torrent ${meta.name} on attempt ${i + 1}/${retries}`,
                });
                logger.debug(e);
                await wait(1000 * Math.pow(2, i));
            }
        }
        for (let i = 0; i < 5; i++) {
            if (await this.checkForInfoHashInClient(meta.infoHash)) {
                setTimeout(() => unlink(torrentFilePath), 1000);
                return InjectionResult.SUCCESS;
            }
            await wait(100 * Math.pow(2, i));
        }
        setTimeout(() => unlink(torrentFilePath), 1000);
        return InjectionResult.FAILURE;
    }
}
//# sourceMappingURL=RTorrent.js.map