import { join } from "path";
import stripAnsi from "strip-ansi";
import winston from "winston";
import DailyRotateFile from "winston-daily-rotate-file";
import { appDir, createAppDir } from "./configuration.js";
import { getRuntimeConfig } from "./runtimeConfig.js";
export var Label;
(function (Label) {
    Label["QBITTORRENT"] = "qbittorrent";
    Label["RTORRENT"] = "rtorrent";
    Label["TRANSMISSION"] = "transmission";
    Label["DELUGE"] = "deluge";
    Label["DECIDE"] = "decide";
    Label["INJECT"] = "inject";
    Label["PREFILTER"] = "prefilter";
    Label["CONFIGDUMP"] = "configdump";
    Label["TORZNAB"] = "torznab";
    Label["SERVER"] = "server";
    Label["STARTUP"] = "startup";
    Label["SCHEDULER"] = "scheduler";
    Label["SEARCH"] = "search";
    Label["RSS"] = "rss";
    Label["ANNOUNCE"] = "announce";
    Label["WEBHOOK"] = "webhook";
    Label["PERF"] = "perf";
    Label["ARRS"] = "arrs";
    Label["RADARR"] = "radarr";
    Label["SONARR"] = "sonarr";
})(Label || (Label = {}));
export let logger;
const redactionMsg = "[REDACTED]";
function redactUrlPassword(message, urlStr) {
    let url;
    try {
        url = new URL(urlStr);
        if (url.password) {
            const urlDecodedPassword = decodeURIComponent(url.password);
            const urlEncodedPassword = encodeURIComponent(url.password);
            message = message.split(url.password).join(redactionMsg);
            message = message.split(urlDecodedPassword).join(redactionMsg);
            message = message.split(urlEncodedPassword).join(redactionMsg);
        }
    }
    catch (e) {
        // do nothing
    }
    return message;
}
function redactMessage(message, options) {
    if (typeof message !== "string") {
        return message;
    }
    const runtimeConfig = options ?? getRuntimeConfig();
    let ret = message;
    ret = ret.replace(/key=[a-zA-Z0-9]+/g, `key=${redactionMsg}`);
    ret = ret.replace(/pass=[a-zA-Z0-9]+/g, `pass=${redactionMsg}`);
    ret = ret.replace(/(?:(?:auto|download)[./]\d+[./])([a-zA-Z0-9]+)/g, (match, key) => match.replace(key, redactionMsg));
    ret = ret.replace(/(?:\d+[./](?:auto|download)[./])([a-zA-Z0-9]+)/g, (match, key) => match.replace(key, redactionMsg));
    ret = ret.replace(/apiKey: '.+'/g, `apiKey: ${redactionMsg}`);
    ret = ret.replace(/\/notification\/crossSeed\/[a-zA-Z-0-9_-]+/g, `/notification/crossSeed/${redactionMsg}`);
    for (const [key, urlStr] of Object.entries(runtimeConfig)) {
        if (key.endsWith("Url") && urlStr) {
            ret = redactUrlPassword(ret, urlStr);
        }
    }
    return ret;
}
function stripAnsiChars(message) {
    if (typeof message !== "string") {
        return message;
    }
    return stripAnsi(message);
}
const logOnceCache = new Set();
export function logOnce(cacheKey, cb, ttl) {
    if (!logOnceCache.has(cacheKey)) {
        logOnceCache.add(cacheKey);
        cb();
        if (ttl) {
            setTimeout(() => {
                logOnceCache.delete(cacheKey);
            }, ttl).unref();
        }
    }
}
export function initializeLogger(options) {
    createAppDir();
    logger = winston.createLogger({
        level: "info",
        format: winston.format.combine(winston.format.timestamp({
            format: "YYYY-MM-DD HH:mm:ss",
        }), winston.format.errors({ stack: true }), winston.format.splat(), winston.format.printf(({ level, message, label, timestamp, stack, cause }) => {
            const msg = `${message}${stack ? `\n${stack}` : ""}${cause ? `\n${cause}` : ""}`;
            return `${timestamp} ${level}: ${label ? `[${label}] ` : ""}${stripAnsiChars(redactMessage(msg, options))}`;
        })),
        transports: [
            new DailyRotateFile({
                filename: "error.%DATE%.log",
                createSymlink: true,
                symlinkName: "error.current.log",
                dirname: join(appDir(), "logs"),
                maxFiles: "14d",
                level: "error",
            }),
            new DailyRotateFile({
                filename: "info.%DATE%.log",
                createSymlink: true,
                symlinkName: "info.current.log",
                dirname: join(appDir(), "logs"),
                maxFiles: "14d",
            }),
            new DailyRotateFile({
                filename: "verbose.%DATE%.log",
                createSymlink: true,
                symlinkName: "verbose.current.log",
                dirname: join(appDir(), "logs"),
                maxFiles: "14d",
                level: "silly",
            }),
            new winston.transports.Console({
                level: options.verbose ? "silly" : "info",
                format: winston.format.combine(winston.format.errors({ stack: true }), winston.format.splat(), winston.format.colorize(), winston.format.printf(({ level, message, label, timestamp, stack, cause, }) => {
                    const msg = `${message}${stack ? `\n${stack}` : ""}${cause ? `\n${cause}` : ""}`;
                    return `${timestamp} ${level}: ${label ? `[${label}] ` : ""}${redactMessage(msg, options)}`;
                })),
            }),
        ],
    });
}
//# sourceMappingURL=logger.js.map