import { getRuntimeConfig } from "../runtimeConfig.js";
function sanitizeUrl(url) {
    if (typeof url === "string") {
        url = new URL(url);
    }
    return url.origin + url.pathname;
}
function getApikey(url) {
    return new URL(url).searchParams.get("apikey");
}
async function backfill(knex) {
    const torznab = getRuntimeConfig()?.torznab ?? [];
    if (torznab.length > 0) {
        await knex("indexer")
            .insert(torznab.map((url) => ({
            url: sanitizeUrl(url),
            apikey: getApikey(url),
            active: true,
        })))
            .onConflict("url")
            .merge(["active", "apikey"]);
        const timestampRows = await knex
            .select("searchee.id as searchee_id", "indexer.id as indexer_id", "searchee.first_searched as first_searched", "searchee.last_searched as last_searched")
            .from("searchee")
            // @ts-expect-error crossJoin supports string
            .crossJoin("indexer");
        await knex.batchInsert("timestamp", timestampRows, 100);
    }
}
async function up(knex) {
    await knex.schema.createTable("indexer", (table) => {
        table.increments("id").primary();
        table.string("url").unique();
        table.string("apikey");
        table.boolean("active");
    });
    await knex.schema.createTable("timestamp", (table) => {
        table.integer("searchee_id").references("id").inTable("searchee");
        table.integer("indexer_id").references("id").inTable("indexer");
        table.integer("first_searched");
        table.integer("last_searched");
        table.primary(["searchee_id", "indexer_id"]);
    });
    await backfill(knex);
}
async function down(knex) {
    await knex.schema.dropTable("timestamp");
    await knex.schema.dropTable("indexer");
}
export default { name: "02-timestamps", up, down };
//# sourceMappingURL=02-timestamps.js.map